/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/popup/userslayermenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/popup/layermenu',
    'io.ox/office/editframework/view/control/userbadge',
    'gettext!io.ox/office/editframework/main'
], function (Utils, LayerMenu, UserBadge, gt) {

    'use strict';

    // class UsersLayerMenu ===================================================

    /**
     * A layer menu displaying collaborating users.
     *
     * @constructor
     *
     * @extends LayerMenu
     *
     * @param {EditView} docView
     *  The document view containing this menu instance.
     */
    function UsersLayerMenu(docView, initOptions) {

        // self reference
        var self = this;

        // the application instance
        var app = docView.getApp();

        // all existing user badges, mapped by unique user index
        var bagdeGroups = {};

        // whether this control is visible when the app window is hidden
        var visibleState = false;

        // whether to automatically show the menu for at least two active users
        var autoShow = true;

        var showOwnColor = Utils.getBooleanOption(initOptions, 'showOwnColor', false);

        // base constructor ---------------------------------------------------

        LayerMenu.call(this, docView, gt('Collaborators'), {
            tooltip: gt('List of collaborators of this document'),
            autoFocus: false,
            rootContainerNode: app.getWindowNode(),
            backgroundRefresh: false
        });

        // private methods ----------------------------------------------------

        /**
         * Reinitializes the user badges. Generates badges for all new clients,
         * and removes badges of inactive clients.
         */
        function fillMenu() {

            var // a local copy of the map, remaining entries are inactive users that will be hidden
                localGroups = _.clone(bagdeGroups);

            // process all active users (returned array is sorted by unique client index)
            _.each(app.getActiveClients(), function (clientData) {

                var // existing badge group
                    badgeGroup = localGroups[clientData.clientId];

                if (badgeGroup) {
                    badgeGroup.setValue(clientData).show();
                    delete localGroups[clientData.clientId];
                } else {
                    badgeGroup = new UserBadge(docView, clientData, { showOwnColor: showOwnColor });
                    bagdeGroups[clientData.clientId] = badgeGroup;
                    self.addGroup(null, badgeGroup);
                }
            });

            // hide all remaining inactive users
            _.invoke(localGroups, 'hide');

            // refresh layout (size and position)
            self.refresh();
        }

        /**
         * Debounced version of the method UsersLayerMenu.fillMenu().
         */
        var fillMenuDebounced = this.createDebouncedMethod('UsersLayerMenu.fillMenuDebounced', null, fillMenu);

        /**
         * Sets the default position of the collaborators list layer menu
         * (right top corner of the application content area).
         */
        function setDefaultPosition() {
            var pos = Utils.getNodePositionInPage(app.getWindowNode());
            self.setNodePosition(pos.left + pos.width - self.getNode().outerWidth() - 40, pos.top + 60);
        }

        // initialization -----------------------------------------------------

        // lazy initialization of the drop-down menu
        this.one({ 'popup:beforeshow': fillMenu, 'popup:show': setDefaultPosition });

        // disable auto-open functionality if user has closed this pop-up
        this.on('menu:userclose', function () { autoShow = false; });

        // auto show/hide collaborator list depending on user event
        // Listening to 'docs:users', not 'docs:users:selection', so that
        // no update happens, if only the selection has changed (46610)
        this.listenTo(app, 'docs:users', function (activeClients) {

            // redraw user badges
            fillMenuDebounced();

            // auto show once if somebody joins
            if (activeClients.length < 2) {
                self.hide();
            } else if (autoShow) {
                //Bug 36345 window also pops up if user is outside of the document
                if (app.getWindow().state.visible) {
                    self.show();
                } else {
                    visibleState = true;
                }
            }
        });

        // bug 36482: repaint when application state changes, e.g. offline mode
        // -> but not, if 'ready' changes to 'sending' or vice versa or if only 'recalc' is sent
        this.listenTo(app, 'docs:state', function (newState, oldState) {
            var noBadgeGroupUpdate = ((newState === 'ready' || newState === 'recalc') && (!oldState || oldState === 'sending')) || (newState === 'sending' && oldState === 'ready');
            if (!noBadgeGroupUpdate) { fillMenuDebounced(); }
        });

        // bug 35534: restore collaborators pop-up
        this.listenTo(app.getWindow(), 'show', function () {
            if (visibleState && autoShow && app.getActiveClients().length >= 2) {
                self.show();
            }
        });

        // saving the state of the menu
        this.listenTo(app.getWindow(), 'beforehide', function () {
            visibleState = self.isVisible();
        });

        // destroy all class members
        this.registerDestructor(function () {
            self = app = docView = bagdeGroups = null;
        });

    } // class UsersLayerMenu

    // exports ================================================================

    // derive this class from class LayerMenu
    return LayerMenu.extend({ constructor: UsersLayerMenu });

});
