/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/editframework/view/dialog/saveasfiledialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/utils/driveutils',
    'gettext!io.ox/office/editframework/main'
], function (Utils, Dialogs, DriveUtils, gt) {

    'use strict';

    // class SaveAsFileDialog =================================================

    /**
     * A custom dialog to save a file in a Drive folder. This dialog provides a
     * folder picker and a simple text input.
     *
     * @constructor
     *
     * @extends ModalDriveDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDriveDialog. Additionally, the following options are supported:
     *  @param {String} [initOptions.preselect]
     *      Drive-File-Id to preselect a folder
     *  @param {String} [initOptions.value='']
     *      The initial value of the text field.
     *  @param {String} [initOptions.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {Boolean} [initOptions.allowEmpty=false]
     *      If set to true, empty text will be considered valid. Otherwise, the
     *      primary button will be disabled as long as the text input field is
     *      empty.
     *  @param {Object} [initOptions.driveUtil=DriveUtils.FolderPicker]
     *      the assigned DriveUtil from the DriveUtils class
     *      can be: FolderPicker, FlatFolderPicker & FilePicker
     */
    var SaveAsFileDialog = Dialogs.ModalDriveDialog.extend({ constructor: function (docView, initOptions) {

        // self reference
        var self = this;

        // label for the save as input node
        var filenameLabelNode = $('<label for="save-as-filename">').text(gt('File name'));

        // the text field control
        var inputNode = $('<input type="text" id="save-as-filename" class="form-control" tabindex="0">').attr({
            placeholder: Utils.getStringOption(initOptions, 'placeholder', ''),
            value: Utils.getStringOption(initOptions, 'value', '')
        });

        // whether to allow empty text
        var allowEmpty = Utils.getBooleanOption(initOptions, 'allowEmpty', false);

        var folderPicker = Utils.getOption(initOptions, 'folderPicker') || new DriveUtils.FolderPicker(docView, initOptions);
        folderPicker.getNode().css('height', '250px');

        // base constructor ---------------------------------------------------

        Dialogs.ModalDriveDialog.call(this, docView, folderPicker, Utils.extendOptions({
            ok: gt('Save'),
            addClass: 'save-as-folder-list',
            height: 350
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Enables or disables the primary button according to the contents of
         * the text field, and the presence of folder id if folder picker is
         * activated.
         */
        function updateOkButton() {
            var inputValue    = inputNode.val();
            var state         = allowEmpty || ($.trim(inputValue) !== '');
            var selectedFile  = self.getSelectedFile();

            state = state && selectedFile && selectedFile.folder_id;
            self.enableOkButton(state);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the text that is contained in the text input element.
         *
         * @returns {String}
         *  The text that is contained in the text input element.
         */
        this.getText = function () {
            return inputNode.val();
        };

        /**
         * Changes the text contained in the text input element.
         *
         * @param {String} text
         *  The text to be inserted into the text input element.
         *
         * @returns {SaveAsFileDialog}
         *  A reference to this instance.
         */
        this.setText = function (text) {
            inputNode.val(text);
            return this;
        };

        /**
         * @returns {String}
         *  The current folder identifier.
         */
        this.getSelectedFolderId = function () {
            var selectedFile = self.getSelectedFile();
            if (selectedFile) {
                return selectedFile.folder_id;
            }
        };

        // initialization -----------------------------------------------------

        // focus and select the text in the input field
        this.on({
            show: function () {
                // give this dialog a good class name
                self.getPopup().addClass('save-as-dialog');

                // add the folder tree and text field
                self.append(filenameLabelNode, inputNode);

                // enable/disable the primary button
                inputNode.on('input', updateOkButton);
                updateOkButton();

                Utils.setFocus(inputNode).select();
            },
            change: updateOkButton
        });

    } }); // class SaveAsFileDialog

    // exports ================================================================

    return SaveAsFileDialog;

});
