/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/editframework/utils/texture', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/drawinglayer/view/imageutil'
], function (Utils, Canvas, ImageUtil) {

    'use strict';

    var isNumber = _.isNumber;
    // list that contains all unresolved canvas nodes, that are waiting for texture to be loaded in.
    // This ensures that only one img loading for one canvas is done.
    // Must be propery maintained and cleaned up, as it is global -> memleaks
    var pendingCanvasesList = [];

    // private static functions ===============================================

    function calculateOffset(rectAlignment, offsetX, offsetY, widthPx, heightPx, patternWidth, patternHeight, options) {
        var tempX = 0, tempY = 0;
        // if tile is flipped/ mirrored (none, horizontal - x, vertical - y, or both - xy)
        var flipMode = Utils.getBooleanOption(options, 'flipMode', false);
        // whether offset should be explicitely set
        var useOffset = Utils.getBooleanOption(options, 'useOffset', false);
        // used for ODF stretching when image (pattern) size is greater than canvas size, image is anchored to coresponding rectAlignment edge
        var anchorToEdge = Utils.getBooleanOption(options, 'anchorToEdge', false);
        // used for ODF streching, if width of image is greater than canvas width
        var widthDiff = patternWidth > widthPx ? widthPx - patternWidth : 0;
        // used for ODF streching, if height of image is greater than canvas height
        var heightDiff = patternHeight > heightPx ? heightPx - patternHeight : 0;

        switch (rectAlignment) {
            case 'topLeft':
                tempX = 0;
                tempY = 0;
                break;
            case 'top':
                tempX = ((widthPx - patternWidth) / 2) % patternWidth;
                if (flipMode) { tempX += patternWidth / 4; }
                if (anchorToEdge && widthDiff) { tempX = widthDiff / 2; }
                break;
            case 'topRight':
                tempX = widthPx % patternWidth;
                if (flipMode) { tempX += patternWidth / 2; }
                if (anchorToEdge && widthDiff) { tempX = widthDiff; }
                break;
            case 'left':
                tempY = ((heightPx - patternHeight) / 2) % patternHeight;
                if (flipMode) { tempY += patternHeight / 4; }
                if (anchorToEdge && heightDiff) { tempY = heightDiff / 2; }
                break;
            case 'center':
                tempX = ((widthPx - patternWidth) / 2) % patternWidth;
                tempY = ((heightPx - patternHeight) / 2) % patternHeight;
                if (flipMode) {
                    tempX += patternWidth / 4;
                    tempY += patternHeight / 4;
                }
                if (anchorToEdge && widthDiff) { tempX = widthDiff / 2; }
                if (anchorToEdge && heightDiff) { tempY = heightDiff / 2; }
                break;
            case 'right':
                tempX = widthPx % patternWidth;
                tempY = ((heightPx - patternHeight) / 2) % patternHeight;
                if (flipMode) {
                    tempX += patternWidth / 2;
                    tempY += patternHeight / 4;
                }
                if (anchorToEdge && widthDiff) { tempX = widthDiff; }
                if (anchorToEdge && heightDiff) { tempY = heightDiff / 2; }
                break;
            case 'bottomLeft':
                tempY = heightPx % patternHeight;
                if (flipMode) { tempY += patternHeight / 2; }
                if (anchorToEdge && heightDiff) { tempY = heightDiff; }
                break;
            case 'bottom':
                tempX = ((widthPx - patternWidth) / 2) % patternWidth;
                tempY = heightPx % patternHeight;
                if (flipMode) {
                    tempX += patternWidth / 4;
                    tempY += patternHeight / 2;
                }
                if (anchorToEdge && widthDiff) { tempX = widthDiff / 2; }
                if (anchorToEdge && heightDiff) { tempY = heightDiff; }
                break;
            case 'bottomRight':
                tempX = widthPx % patternWidth;
                tempY = heightPx % patternHeight;
                if (flipMode) {
                    tempX += patternWidth / 2;
                    tempY += patternHeight / 2;
                }
                if (anchorToEdge && widthDiff) { tempX = widthDiff; }
                if (anchorToEdge && heightDiff) { tempY = heightDiff; }
                break;
            default:
                Utils.error('Texture.calculateOffset(): rectAlignment is unknown!');
        }
        if (useOffset) {
            offsetX %= patternWidth;
            offsetY %= patternHeight;
            // if the values are positive, we need to shift them top left for the 1 pattern dimensions
            // this way pattern will always cover whole canvas object, and there will be no gaps if offset is positive
            if (offsetX > 0) { offsetX -= patternWidth; }
            if (offsetY > 0) { offsetY -= patternHeight; }
            if (tempX > 0) { tempX -= patternWidth; }
            if (tempY > 0) { tempY -= patternHeight; }
            tempX += offsetX;
            tempY += offsetY;
            tempX %= patternWidth;
            tempY %= patternHeight;
        }

        return { offsetX: tempX, offsetY: tempY };
    }

    // standard streching
    function getStrechedCoords(stretchingObj, cWidth, cHeight) {
        var sLeft = stretchingObj.left || 0;
        var sRight = stretchingObj.right || 0;
        var sTop = stretchingObj.top || 0;
        var sBottom = stretchingObj.bottom || 0;

        var tLeft = cWidth * (sLeft / 100);
        var tWidth = cWidth * ((100 - sRight - sLeft) / 100);
        var tTop = cHeight * (sTop / 100);
        var tHeight = cHeight * ((100 - sBottom - sTop) / 100);

        return { left: Utils.round(tLeft, 1), top: Utils.round(tTop, 1), width: Utils.round(tWidth, 1), height: Utils.round(tHeight, 1) };
    }

    // ODF specific streching
    function getStrechedOdfCoords(odfStrectchingObj, canvasWidth, canvasHeight, imgWidth, imgHeight) {
        var sWidth = odfStrectchingObj.width ? Utils.convertHmmToLength(odfStrectchingObj.width, 'px', 1) : imgWidth;
        var sHeight = odfStrectchingObj.height ? Utils.convertHmmToLength(odfStrectchingObj.height, 'px', 1) : imgHeight;
        var rectAlignment = odfStrectchingObj.rectAlignment;

        var calcOffset = calculateOffset(rectAlignment, 0, 0, canvasWidth, canvasHeight, sWidth, sHeight, { anchorToEdge: true });

        var tLeft = calcOffset.offsetX;
        var tWidth = sWidth;
        var tTop = calcOffset.offsetY;
        var tHeight = sHeight;

        return { left: Utils.round(tLeft, 1), top: Utils.round(tTop, 1), width: Utils.round(tWidth, 1), height: Utils.round(tHeight, 1) };
    }

    // static class Texture ===================================================

    /**
     * Provides canvas patterns representing an image as background, and
     * texture as a background, generated from the image.
     */
    var Texture = {};

    // public methods ---------------------------------------------------------

    /**
     * Creates and returns texture fill for given object, based on bitmap and its attributes.
     *
     * @param  {PresentationModel} docModel
     *         Presentation doc model
     * @param  {ContextWrapper} context
     *         Context wrapper context as provided by the class Canvas.
     * @param  {Object} fillAttrs
     *         Shape fill attributes
     * @param  {Number} widthPx
     *         Canvas width in pixels
     * @param  {Number} heightPx
     *          Canvas height in pixels
     * @param  {Number|null} rotationAngle
     *         If exist, angle of rotation in radians
     * @param  {Object} [options]
     *         Optional set of properties.
     *  @param {Boolean} [options.isSlide=false]
     *         If texture is applied to slide background.
     *  @param {Boolean} [options.noAlpha=false]
     *         If alpha should be ignored.
     *
     * @return {jQuery.Promise}
     *  A promise that will be resolved with the generated canvas pattern.
     */
    Texture.getTextureFill = function (docModel, context, fillAttrs, widthPx, heightPx, rotationAngle, options) {

        if (!(fillAttrs && fillAttrs.bitmap && fillAttrs.bitmap.imageUrl)) { return $.Deferred().reject(); } // 52106, imageUrl must be specified

        var imageUrl = ImageUtil.getFileUrl(docModel.getApp(), fillAttrs.bitmap.imageUrl);
        var canvasNode = context.getCanvas();
        var pendingCanvas = _.find(pendingCanvasesList, function (oneCanvas) { return oneCanvas === canvasNode; });

        if (!pendingCanvas) {
            pendingCanvasesList.push(canvasNode);
        } else {
            return $.Deferred().reject();
        }
        var promise = docModel.getApp().createImageNode(imageUrl);

        promise = promise.then(function (imgNode) {
            // if texture is used as slide or shape background
            var isSlideBackground = Utils.getBooleanOption(options, 'isSlide', false);
            // optional parameter wheter or not to ignore alpha transparency value
            var ignoreAlpha = Utils.getBooleanOption(options, 'noAlpha', false);
            // tiling properties as an object, sent with operation
            var tilingObj = fillAttrs.bitmap.tiling;
            // wheter tiling properties are existing
            var isTiling = !_.isEmpty(tilingObj);
            // stretching properties as an objects, sent with operation
            var stretchingObj = fillAttrs.bitmap.stretching;
            // checking existence of streching properties
            var isStretching = !_.isEmpty(stretchingObj);
            // ODF specific stretching properties
            var odfStrectchingObj = fillAttrs.bitmap.stretchingAligned;
            // check for existence of ODF streching properties
            var isOdfStretching = !_.isEmpty(odfStrectchingObj);
            // canvas repeat property, depending on set tiling
            var isRepeat = isTiling ? 'repeat' : 'no-repeat';
            // wheter to unrotate texture for complementary negative angle or not (slide backgrounds have this always)
            var isRotateWithShape = isSlideBackground ? true : fillAttrs.bitmap.rotateWithShape !== false;
            // streching percentage values for x and y direction
            var stretchX = (isTiling && isNumber(tilingObj.stretchX)) ? tilingObj.stretchX / 100 : 1;
            var stretchY = (isTiling && isNumber(tilingObj.stretchY)) ? tilingObj.stretchY / 100 : 1;
            // offset number values from where to start tiling or image. Can be also negative
            var offsetX = (isTiling && isNumber(tilingObj.offX)) ? Utils.convertHmmToLength(tilingObj.offX, 'px', 1) : 0;
            var offsetY = (isTiling && isNumber(tilingObj.offY)) ? Utils.convertHmmToLength(tilingObj.offY, 'px', 1) : 0;
            // normalized transparency value (canvas uses inverted value of filter stored one)
            var nTransparency = isNumber(fillAttrs.bitmap.transparency) ? (1 - fillAttrs.bitmap.transparency) : 1;
            // rectangle alignment properties
            var rectAlignment = (isTiling && tilingObj.rectAlignment) || null;
            // if tile is flipped/ mirrored (none, horizontal - x, vertical - y, or both - xy)
            var flipMode = (isTiling && tilingObj.flipMode) || null;
            // if flip mode is used
            var isFlipMode = flipMode && flipMode !== 'none';
            var dpiDiff = 96 / 96; // TODO: some images have different dpi, filter needs to send them
            // canvas width in pixels
            var cWidth = isTiling ? Utils.round(imgNode[0].width * stretchX * dpiDiff, 1) : widthPx;
            // canvas height in pixels
            var cHeight = isTiling ? Utils.round(imgNode[0].height * stretchY * dpiDiff, 1) : heightPx;
            // temporary helper canvas to store intermediate state
            var tempCanvas = new Canvas(docModel);
            // another helper canvas to store offset state
            var offsetCanvas = null;

            // remove canvas node from the pending list
            if (_.indexOf(pendingCanvasesList, canvasNode) > -1) {
                pendingCanvasesList.splice(_.indexOf(pendingCanvasesList, canvasNode), 1);
            }

            if (!isTiling && !isRotateWithShape) {
                cWidth = widthPx * Math.abs(Math.cos(rotationAngle)) + heightPx * Math.abs(Math.sin(rotationAngle));
                cHeight = heightPx * Math.abs(Math.cos(rotationAngle)) + widthPx * Math.abs(Math.sin(rotationAngle));
            }

            tempCanvas.initialize({ width: cWidth, height: cHeight });

            // $('body').append(tempCanvas.getNode().css({ zIndex: 99, position: 'fixed' })); // DEBUG

            // scaling of the image
            tempCanvas.render(function (tempCtx) {
                var coordObj; // left, top, width and height coordinate values where to place image on canvas
                if (isStretching) {
                    coordObj = getStrechedCoords(stretchingObj, cWidth, cHeight);
                } else if (isOdfStretching) {
                    coordObj = getStrechedOdfCoords(odfStrectchingObj, cWidth, cHeight, imgNode[0].width, imgNode[0].height);
                } else {
                    coordObj = { left: 0, top: 0, width: cWidth, height: cHeight };
                }

                if (!ignoreAlpha) {
                    tempCtx.setGlobalAlpha(nTransparency);
                }
                tempCtx.drawImage(imgNode[0], coordObj);
            });
            // offset
            if (offsetX || offsetY || rectAlignment || flipMode) {
                offsetCanvas = new Canvas(docModel);
                offsetCanvas.initialize({ width: 2 * cWidth, height: 2 * cHeight });
                // $('body').append(offsetCanvas.getNode().css({ zIndex: 99, position: 'fixed', top: (cHeight + 20) })); // DEBUG

                if (isFlipMode) {
                    switch (flipMode) {
                        case 'x':
                            offsetCanvas.render(function (offsetCtx) {
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: cHeight, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(-1, 1);
                                offsetCtx.translate(-cWidth * 2, 0);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: cHeight, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                            });
                            break;
                        case 'y':
                            offsetCanvas.render(function (offsetCtx) {
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.drawImage(tempCanvas, { left: cWidth, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(1, -1);
                                offsetCtx.translate(0, -cHeight * 2);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.drawImage(tempCanvas, { left: cWidth, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                            });
                            break;
                        case 'xy':
                            offsetCanvas.render(function (offsetCtx) {
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(-1, 1);
                                offsetCtx.translate(-cWidth * 2, 0);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(1, -1);
                                offsetCtx.translate(0, -cHeight * 2);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                                offsetCtx.scale(-1, 1);
                                offsetCtx.translate(-cWidth * 2, 0);
                                //offsetCtx.drawRect(0, 0, 2 * cWidth, 2 * cHeight, 'stroke'); // DEBUG
                                offsetCtx.drawImage(tempCanvas, { left: 0, top: 0, width: cWidth, height: cHeight }, { left: 0, top: 0, width: cWidth, height: cHeight });
                            });
                            break;
                        default:
                            Utils.error('Texture.getTextureFill(): flipMode is unknown!');
                    }

                    offsetCanvas.render(function (offsetCtx) {
                        var calcOffset = calculateOffset(rectAlignment, offsetX, offsetY, widthPx, heightPx, cWidth * 2, cHeight * 2, { flipMode: true, useOffset: true });
                        var pattern = offsetCtx.createPattern(offsetCanvas, 'repeat');
                        offsetCtx.clearRect(0, 0, 2 * cWidth, 2 * cHeight);
                        offsetCtx.translate(calcOffset.offsetX, calcOffset.offsetY);
                        offsetCtx.setFillStyle(pattern).drawRect(0, 0, 2 * cWidth - calcOffset.offsetX, 2 * cHeight - calcOffset.offsetY, 'fill');
                    });
                } else {
                    tempCanvas.render(function (tempCtx) {
                        var calcOffset = calculateOffset(rectAlignment, offsetX, offsetY, widthPx, heightPx, cWidth, cHeight, { useOffset: true });
                        var pattern = tempCtx.createPattern(tempCanvas, 'repeat');
                        tempCtx.clearRect(0, 0, cWidth, cHeight);
                        tempCtx.translate(calcOffset.offsetX, calcOffset.offsetY);
                        tempCtx.setFillStyle(pattern).drawRect(0, 0, cWidth - calcOffset.offsetX, cHeight - calcOffset.offsetY, 'fill');
                    });
                }
            }
            // DEBUG
            // tempCanvas.getNode().remove();
            // if (offsetCanvas) { offsetCanvas.getNode().remove(); }

            return context.createPattern(isFlipMode ? offsetCanvas : tempCanvas, isRepeat);
        });

        promise.fail(function () {
            pendingCanvasesList = [];
            Utils.warn('Texture.getTextureFill(): failed to load shape background image!');
        });

        return promise;
    };

    // exports ================================================================

    return Texture;
});
