/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/color', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/colorutils'
], function (Utils, ColorUtils) {

    'use strict';

    // convenience shortcuts
    var round = Math.round;
    var max = Math.max;
    var minMax = Utils.minMax;

    // helper classes
    var RGBModel = ColorUtils.RGBModel;
    var HSLModel = ColorUtils.HSLModel;
    var ColorDescriptor = ColorUtils.ColorDescriptor;

    // hue channel conversion factor (units per degree)
    var HUE_PER_DEG = 60000;

    // the gamma correction factors (intended to be similar to MSO)
    var INV_GAMMA = 2.27;
    var GAMMA = 1 / INV_GAMMA;

    // maps all OOXML preset color names (camel-case!) to RGB values
    var PRESET_COLOR_MAP = {
        aliceBlue: 'f0f8ff',        antiqueWhite: 'faebd7',         aqua: '00ffff',             aquamarine: '7fffd4',
        azure: 'f0ffff',            beige: 'f5f5dc',                bisque: 'ffe4c4',           black: '000000',
        blanchedAlmond: 'ffebcd',   blue: '0000ff',                 blueViolet: '8a2be2',       brown: 'a52a2a',
        burlyWood: 'deb887',        cadetBlue: '5f9ea0',            chartreuse: '7fff00',       chocolate: 'd2691e',
        coral: 'ff7f50',            cornflowerBlue: '6495ed',       cornsilk: 'fff8dc',         crimson: 'dc143c',
        cyan: '00ffff',             darkBlue: '00008b',             darkCyan: '008b8b',         darkGoldenrod: 'b8860b',
        darkGray: 'a9a9a9',         darkGreen: '006400',            darkGrey: 'a9a9a9',         darkKhaki: 'bdb76b',
        darkMagenta: '8b008b',      darkOliveGreen: '556b2f',       darkOrange: 'ff8c00',       darkOrchid: '9932cc',
        darkRed: '8b0000',          darkSalmon: 'e9967a',           darkSeaGreen: '8fbc8f',     darkSlateBlue: '483d8b',
        darkSlateGray: '2f4f4f',    darkSlateGrey: '2f4f4f',        darkTurquoise: '00ced1',    darkViolet: '9400d3',
        deepPink: 'ff1493',         deepSkyBlue: '00bfff',          dimGray: '696969',          dimGrey: '696969',
        dkBlue: '00008b',           dkCyan: '008b8b',               dkGoldenrod: 'b8860b',      dkGray: 'a9a9a9',
        dkGreen: '006400',          dkGrey: 'a9a9a9',               dkKhaki: 'bdb76b',          dkMagenta: '8b008b',
        dkOliveGreen: '556b2f',     dkOrange: 'ff8c00',             dkOrchid: '9932cc',         dkRed: '8b0000',
        dkSalmon: 'e9967a',         dkSeaGreen: '8fbc8b',           dkSlateBlue: '483d8b',      dkSlateGray: '2f4f4f',
        dkSlateGrey: '2f4f4f',      dkTurquoise: '00ced1',          dkViolet: '9400d3',         dodgerBlue: '1e90ff',
        firebrick: 'b22222',        floralWhite: 'fffaf0',          forestGreen: '228b22',      fuchsia: 'ff00ff',
        gainsboro: 'dcdcdc',        ghostWhite: 'f8f8ff',           gold: 'ffd700',             goldenrod: 'daa520',
        gray: '808080',             green: '008000',                greenYellow: 'adff2f',      grey: '808080',
        honeydew: 'f0fff0',         hotPink: 'ff69b4',              indianRed: 'cd5c5c',        indigo: '4b0082',
        ivory: 'fffff0',            khaki: 'f0e68c',                lavender: 'e6e6fa',         lavenderBlush: 'fff0f5',
        lawnGreen: '7cfc00',        lemonChiffon: 'fffacd',         lightBlue: 'add8e6',        lightCoral: 'f08080',
        lightCyan: 'e0ffff',        lightGoldenrodYellow: 'fafad2', lightGray: 'd3d3d3',        lightGreen: '90ee90',
        lightGrey: 'd3d3d3',        lightPink: 'ffb6c1',            lightSalmon: 'ffa07a',      lightSeaGreen: '20b2aa',
        lightSkyBlue: '87cefa',     lightSlateGray: '778899',       lightSlateGrey: '778899',   lightSteelBlue: 'b0c4de',
        lightYellow: 'ffffe0',      lime: '00ff00',                 limeGreen: '32cd32',        linen: 'faf0e6',
        ltBlue: 'add8e6',           ltCoral: 'f08080',              ltCyan: 'e0ffff',           ltGoldenrodYellow: 'fafad2',
        ltGray: 'd3d3d3',           ltGreen: '90ee90',              ltGrey: 'd3d3d3',           ltPink: 'ffb6c1',
        ltSalmon: 'ffa07a',         ltSeaGreen: '20b2aa',           ltSkyBlue: '87cefa',        ltSlateGray: '778899',
        ltSlateGrey: '778899',      ltSteelBlue: 'b0c4de',          ltYellow: 'ffffe0',         magenta: 'ff00ff',
        maroon: '800000',           medAquamarine: '66cdaa',        medBlue: '0000cd',          mediumAquamarine: '66cdaa',
        mediumBlue: '0000cd',       mediumOrchid: 'ba55d3',         mediumPurple: '9370db',     mediumSeaGreen: '3cb371',
        mediumSlateBlue: '7b68ee',  mediumSpringGreen: '00fa9a',    mediumTurquoise: '48d1cc',  mediumVioletRed: 'c71585',
        medOrchid: 'ba55d3',        medPurple: '9370db',            medSeaGreen: '3cb371',      medSlateBlue: '7b68ee',
        medSpringGreen: '00fa9a',   medTurquoise: '48d1cc',         medVioletRed: 'c71585',     midnightBlue: '191970',
        mintCream: 'f5fffa',        mistyRose: 'ffe4e1',            moccasin: 'ffe4b5',         navajoWhite: 'ffdead',
        navy: '000080',             oldLace: 'fdf5e6',              olive: '808000',            oliveDrab: '6b8e23',
        orange: 'ffa500',           orangeRed: 'ff4500',            orchid: 'da70d6',           paleGoldenrod: 'eee8aa',
        paleGreen: '98fb98',        paleTurquoise: 'afeeee',        paleVioletRed: 'db7093',    papayaWhip: 'ffefd5',
        peachPuff: 'ffdab9',        peru: 'cd853f',                 pink: 'ffc0cb',             plum: 'dda0dd',
        powderBlue: 'b0e0e6',       purple: '800080',               red: 'ff0000',              rosyBrown: 'bc8f8f',
        royalBlue: '4169e1',        saddleBrown: '8b4513',          salmon: 'fa8072',           sandyBrown: 'f4a460',
        seaGreen: '2e8b57',         seaShell: 'fff5ee',             sienna: 'a0522d',           silver: 'c0c0c0',
        skyBlue: '87ceeb',          slateBlue: '6a5acd',            slateGray: '708090',        slateGrey: '708090',
        snow: 'fffafa',             springGreen: '00ff7f',          steelBlue: '4682b4',        tan: 'd2b48c',
        teal: '008080',             thistle: 'd8bfd8',              tomato: 'ff6347',           turquoise: '40e0d0',
        violet: 'ee82ee',           wheat: 'f5deb3',                white: 'ffffff',            whiteSmoke: 'f5f5f5',
        yellow: 'ffff00',           yellowGreen: '9acd32'
    };

    // maps all OOXML system color names (camel-case!) to CSS system color names
    var SYSTEM_COLOR_MAP = {
        '3dDkShadow': 'threeddarkshadow',           '3dLight': 'threedhighlight',               activeBorder: 'activeborder',
        activeCaption: 'activecaption',             appWorkspace: 'appworkspace',               background: 'background',
        btnFace: 'buttonface',                      btnHighlight: 'buttonhighlight',            btnShadow: 'buttonshadow',
        btnText: 'buttontext',                      captionText: 'captiontext',                 gradientActiveCaption: 'activecaption',
        gradientInactiveCaption: 'inactivecaption', grayText: 'graytext',                       highlight: 'highlight',
        highlightText: 'highlighttext',             hotLight: 'highlight',                      inactiveBorder: 'inactiveborder',
        inactiveCaption: 'inactivecaption',         inactiveCaptionText: 'inactivecaptiontext', infoBk: 'infobackground',
        infoText: 'infotext',                       menu: 'menu',                               menuBar: 'menu',
        menuHighlight: 'highlight',                 menuText: 'menutext',                       scrollBar: 'scrollbar',
        window: 'window',                           windowFrame: 'windowframe',                 windowText: 'windowtext'
    };

    // private static functions ===============================================

    /**
     * Returns the specified integer property from a JSON object, normalizes
     * the number by the specified scaling factor, but does not restrict it any
     * further.
     *
     * @param {Object} json
     *  An arbitrary JSON object.
     *
     * @param {String} prop
     *  The name of an integer property to be extracted from the JSON object.
     *  If the property does not exist, this function returns zero.
     *
     * @param {Number} scale
     *  The scaling factor the number will be divided by. MUST be positive.
     *
     * @returns {Number}
     *  The normalized value of the specified property.
     */
    function getScaledNum(json, prop, scale) {
        return Utils.getIntegerOption(json, prop, 0) / scale;
    }

    /**
     * Returns the specified integer property from a JSON object, normalizes
     * the number, but does not restrict it any further.
     *
     * @param {Object} json
     *  An arbitrary JSON object.
     *
     * @param {String} prop
     *  The name of an integer property to be extracted from the JSON object.
     *  If the property does not exist, this function returns zero.
     *
     * @returns {Number}
     *  The normalized value of the specified property.
     */
    function getNum(json, prop) {
        return getScaledNum(json, prop, 100000);
    }

    /**
     * Returns the specified integer property from a JSON object, normalizes
     * the number, and restricts it to non-negative numbers.
     *
     * @param {Object} json
     *  An arbitrary JSON object.
     *
     * @param {String} prop
     *  The name of an integer property to be extracted from the JSON object.
     *  If the property does not exist, this function returns zero.
     *
     * @returns {Number}
     *  The normalized value of the specified property, restricted to
     *  non-negative numbers.
     */
    function getPosNum(json, prop) {
        return max(getScaledNum(json, prop, 100000), 0);
    }

    /**
     * Returns the specified integer property from a JSON object, normalizes
     * the number, and restricts it to the interval [0;1].
     *
     * @param {Object} json
     *  An arbitrary JSON object.
     *
     * @param {String} prop
     *  The name of an integer property to be extracted from the JSON object.
     *  If the property does not exist, this function returns zero.
     *
     * @returns {Number}
     *  The normalized value of the specified property, restricted to the
     *  interval [0;1].
     */
    function getLimNum(json, prop) {
        return minMax(getScaledNum(json, prop, 100000), 0, 1);
    }

    // color transformations ==================================================

    // all color transformation handlers, mapped by transformation name
    var TRANSFORMATIONS = {

        alpha:    function (model, transform) { return model.set('a', getNum(transform, 'value')); },
        alphaMod: function (model, transform) { return model.modulate('a', getPosNum(transform, 'value')); },
        alphaOff: function (model, transform) { return model.offset('a', getNum(transform, 'value')); },

        red:      function (model, transform) { return model.toRGB().setGamma('r', getNum(transform, 'value'), GAMMA); },
        redMod:   function (model, transform) { return model.toRGB().modulateGamma('r', getPosNum(transform, 'value'), GAMMA); },
        redOff:   function (model, transform) { return model.toRGB().offsetGamma('r', getNum(transform, 'value'), GAMMA); },

        green:    function (model, transform) { return model.toRGB().setGamma('g', getNum(transform, 'value'), GAMMA); },
        greenMod: function (model, transform) { return model.toRGB().modulateGamma('g', getPosNum(transform, 'value'), GAMMA); },
        greenOff: function (model, transform) { return model.toRGB().offsetGamma('g', getNum(transform, 'value'), GAMMA); },

        blue:     function (model, transform) { return model.toRGB().setGamma('b', getNum(transform, 'value'), GAMMA); },
        blueMod:  function (model, transform) { return model.toRGB().modulateGamma('b', getPosNum(transform, 'value'), GAMMA); },
        blueOff:  function (model, transform) { return model.toRGB().offsetGamma('b', getNum(transform, 'value'), GAMMA); },

        hue:      function (model, transform) { return model.toHSL().setH(getScaledNum(transform, 'value', HUE_PER_DEG)); },
        hueMod:   function (model, transform) { return model.toHSL().modulateH(getPosNum(transform, 'value')); },
        hueOff:   function (model, transform) { return model.toHSL().offsetH(getScaledNum(transform, 'value', HUE_PER_DEG)); },

        sat:      function (model, transform) { return model.toHSL().set('s', getNum(transform, 'value')); },
        satMod:   function (model, transform) { return model.toHSL().modulate('s', getPosNum(transform, 'value')); },
        satOff:   function (model, transform) { return model.toHSL().offset('s', getNum(transform, 'value')); },

        lum:      function (model, transform) { return model.toHSL().set('l', getNum(transform, 'value')); },
        lumMod:   function (model, transform) { return model.toHSL().modulate('l', getPosNum(transform, 'value')); },
        lumOff:   function (model, transform) { return model.toHSL().offset('l', getNum(transform, 'value')); },

        shade:    function (model, transform) { return model.toRGB().darkenGamma(1 - getPosNum(transform, 'value'), GAMMA); },
        tint:     function (model, transform) { return model.toRGB().lightenGamma(1 - getPosNum(transform, 'value'), GAMMA); },

        gamma:    function (model) { return model.toRGB().gamma(GAMMA); },
        invGamma: function (model) { return model.toRGB().gamma(INV_GAMMA); },

        comp:     function (model) { return model.toHSL().comp(); },
        inv:      function (model) { return model.toRGB().invGamma(GAMMA); },
        gray:     function (model) { return model.toRGB().gray(); }
    };

    // class Color ============================================================

    /**
     * Runtime representation of a color of any type supported in document
     * operations.
     *
     * @constructor
     *
     * @param {String} type
     *  The type of the color. Supports all types also supported in document
     *  operations:
     *  - 'auto': The automatic color.
     *  - 'system': Specific colors depending on the operating system.
     *  - 'preset': Predefined colors from a fixed color palette.
     *  - 'crgb': CRGB color model (color channels as percentage, no gamma).
     *  - 'rgb': RGB color model (color channels as 6-digit hexadecimal value).
     *  - 'hsl': HSL color model (hue, saturation, luminance).
     *  - 'scheme': Scheme color from current document theme.
     *
     * @param {String|Object} [value]
     *  The value of the color. Expected data type depends on the color type:
     *  - 'auto': Color value will be ignored and can be omitted.
     *  - 'system', 'preset', 'scheme': A string with the name of the color.
     *  - 'crgb': An object with the integer properties 'r', 'g', and 'b', each
     *      of them in the interval [0;100000] (representing 0% to 100%).
     *  - 'rgb': A string with a 6-digit hexadecimal character (RRGGBB).
     *  - 'hsl': An object with the integer properties 'h', 's', and 'l'. The
     *      property 'h' is expected to be in the interval [0;21599999]
     *      (representing 1/60000 of degrees), the other properties have to be
     *      in the interval [0;100000] (representing 0% to 100%).
     *
     * @param {Array<Object>} [transforms]
     *  An array with color transformations to be applied to the base color
     *  described by the parameters 'type' and 'value'. Each array element must
     *  be an object with the string property 'type', and the optional integer
     *  property 'value'. Supports all color transformations that are supported
     *  in document operations.
     *
     * @param {String} [defRGB]
     *  A fall-back RGB color value (6-digit hexadecimal number as string). May
     *  be provided by document operations, and can be used in case the color
     *  cannot be resolved correctly.
     */
    function Color(type, value, transforms, defRGB) {

        this._type = type;

        // create copies of input values to prevent modification from outside
        if (_.isString(value)) {
            this._value = (type === 'rgb') ? value.toLowerCase() : value;
        } else if (_.isObject(value)) {
            this._value = _.clone(value);
        } else {
            this._value = null;
        }

        this._transforms = _.isArray(transforms) ? transforms.slice() : null;
        this._defRGB = defRGB;

    } // class Color

    // public methods ---------------------------------------------------------

    /**
     * Returns the type of this color.
     *
     * @returns {String}
     *  The type of this color. See constructor parameter 'type' for details.
     */
    Color.prototype.getType = function () {
        return this._type;
    };

    /**
     * Returns whether this color is the automatic color.
     *
     * @returns {Boolean}
     *  Whether this color is the automatic color.
     */
    Color.prototype.isAuto = function () {
        return this._type === 'auto';
    };

    /**
     * Returns whether this color is equal to the passed color.
     *
     * @param {Color} color
     *  The color to be compared with this color.
     *
     * @returns {Boolean}
     *  Whether this color is equal to the passed color.
     */
    Color.prototype.equals = function (color) {
        var jsonColor1 = this.toJSON(), jsonColor2 = color.toJSON();
        // ignore unused properties, e.g. 'fallbackValue'
        return (jsonColor1.type === jsonColor2.type) &&
            _.isEqual(jsonColor1.value, jsonColor2.value) &&
            _.isEqual(jsonColor1.transformations, jsonColor2.transformations);
    };

    /**
     * Returns a deep clone of this color.
     *
     * @returns {Color}
     *  A deep clone of this color.
     */
    Color.prototype.clone = function () {
        return new Color(this._type, this._value, this._transforms, this._defRGB);
    };

    /**
     * Appends one or more color transformations to this color.
     *
     * @param {String|Object|Array} tType
     *  The name of a color transformation, a complete color transformation
     *  object, or an array of transformation objects (see constructor
     *  parameter 'transforms' for details).
     *
     * @param {Number} [tValue]
     *  The value for the color transformation. Can be omitted, if the
     *  specified transformations does not need a value (e.g. 'gray'). Will be
     *  ignored, if the first parameter is not a string.
     *
     * @returns {Color}
     *  A reference to this instance.
     */
    Color.prototype.transform = function (tType, tValue) {
        if (_.isString(tType)) {
            var transform = { type: tType };
            if (_.isNumber(tValue)) { transform.value = tValue; }
            (this._transforms || (this._transforms = [])).push(transform);
        } else if (_.isArray(tType)) {
            this._transforms = this._transforms ? this._transforms.concat(tType) : tType.slice();
        } else if (_.isObject(tType)) {
            (this._transforms || (this._transforms = [])).push(tType);
        }
        return this;
    };

    /**
     * Returns a color descriptor with details about the color represented by
     * this instance.
     *
     * @param {String|Color} auto
     *  Additional information needed to resolve the automatic color. Can be a
     *  string for predefined automatic color types, or an instance of the
     *  class Color with a type other than the automatic color. The following
     *  color types are supported:
     *  - 'text': for text colors (mapped to black),
     *  - 'line': for line colors (mapped to black),
     *  - 'fill': for fill colors (mapped to full transparency).
     *
     * @param {ThemeModel} [themeModel]
     *  The model of a document theme used to map scheme color names to color
     *  values. If missing, theme colors will not be resolved correctly.
     *
     * @returns {ColorDescriptor}
     *  A descriptor for the resulting color.
     */
    Color.prototype.resolve = function (auto, themeModel) {

        // shortcut to the current value
        var value = this._value;
        // shortcut to the default RGB value
        var defRGB = this._defRGB;
        // the color model containing the resolved color
        var colorModel = null;
        // the original color type to be passed to the color descriptor
        var descType = 'rgb';
        // whether to apply the color transformations
        var applyTransforms = true;

        function createHexModel(hex) {
            var rgbModel = _.isString(hex) ? RGBModel.parseHex(hex) : null;
            return rgbModel || new RGBModel(0, 0, 0, 1);
        }

        function createDefaultModel() {
            applyTransforms = false;
            return createHexModel(defRGB);
        }

        // resolve automatic color with explicit color passed in 'auto' parameter
        if ((this._type === 'auto') && (auto instanceof Color)) {
            if (this._transforms) { auto = auto.clone().transform(this._transforms); }
            return auto.resolve('fill', themeModel);
        }

        // create a color model for all supported color types
        switch (this._type) {

            // predefined colors with fixed RGB values
            case 'preset':
                var presetColor = _.isString(value) ? PRESET_COLOR_MAP[value] : null;
                colorModel = createHexModel(presetColor);
                descType = presetColor ? 'preset' : 'rgb';
                break;

            // predefined system UI colors without fixed RGB values
            case 'system':
                var systemColor = _.isString(value) ? SYSTEM_COLOR_MAP[value] : null;
                colorModel = createHexModel(systemColor ? ColorUtils.getSystemColor(systemColor) : null);
                descType = systemColor ? 'system' : 'rgb';
                break;

            // CRGB color model: each channel in the interval [0;100000]
            case 'crgb':
                colorModel = new RGBModel(getLimNum(value, 'r'), getLimNum(value, 'g'), getLimNum(value, 'b'), 1);
                colorModel.gamma(GAMMA);
                break;

            // HSL color model (hue, saturation, luminance)
            case 'hsl':
                colorModel = new HSLModel(getScaledNum(value, 'h', HUE_PER_DEG), getLimNum(value, 's'), getLimNum(value, 'l'), 1);
                descType = 'hsl';
                break;

            // simple hexadecimal RGB
            case 'rgb':
                colorModel = createHexModel(value);
                break;

            // scheme color
            case 'scheme':
                if (themeModel && _.isString(value)) {
                    if (value === 'style') {
                        Utils.warn('Color.resolve(): scheme color type "style" not implemented');
                        colorModel = createDefaultModel();
                    } else {
                        colorModel = createHexModel(themeModel.getSchemeColor(value, '000000'));
                        descType = 'scheme';
                    }
                } else {
                    colorModel = createDefaultModel();
                }
                break;

            // automatic color, effective color depends on passed 'auto' parameter
            case 'auto':
                // resolve automatic color by color type name
                switch (auto) {
                    case 'text':
                    case 'line':
                        colorModel = new RGBModel(0, 0, 0, 1); // black
                        break;
                    case 'fill':
                        colorModel = new RGBModel(0, 0, 0, 0); // full transparency
                        descType = 'transparent';
                        applyTransforms = false;
                        break;
                    default:
                        Utils.error('Color.resolve(): unknown color type "' + auto + '" for auto color');
                        colorModel = createDefaultModel();
                }
                break;

            default:
                Utils.error('Color.resolve(): unknown color type "' + this._type + '"');
                colorModel = createDefaultModel();
        }

        // apply all transformations
        if (applyTransforms && this._transforms) {
            this._transforms.forEach(function (transform) {
                var tType = Utils.getStringOption(transform, 'type', '');
                var handler = TRANSFORMATIONS[tType];
                if (handler) {
                    colorModel = handler(colorModel, transform);
                } else {
                    Utils.warn('Color.resolve(): unknown color transformation "' + tType + '"');
                }
            });
        }

        // create the color descriptor for the resulting color model
        return new ColorDescriptor(colorModel, descType);
    };

    /**
     * Returns a color descriptor with details about the color represented by
     * this instance, assuming it to be a text color on a specific background
     * style. The automatic color will be resolved to the color black or white,
     * depending on the lightness of the passed background colors. All other
     * colors will be resolved according to the method Color.resolve().
     *
     * @param {Array<Color>} fillColors
     *  The source fill colors, from outermost to innermost level. The last
     *  explicit (non-automatic) fill color in the array will be used as
     *  effective fill color.
     *
     * @param {ThemeModel} [themeModel]
     *  The model of a document theme used to map scheme color names to color
     *  values. If missing, theme colors will not be resolved correctly.
     *
     * @returns {ColorDescriptor}
     *  A descriptor for the resulting color.
     */
    Color.prototype.resolveText = function (fillColors, themeModel) {

        // the resulting text color
        var textColor = this;
        // the effective fill color
        var fillColor = null;

        // resolve explicit colors directly
        if (this.isAuto()) {

            // find last non-transparent fill color in the passed array
            // TODO: merge semi-transparent colors?
            fillColors.forEach(function (currFillColor) {
                if (!currFillColor.isAuto()) { fillColor = currFillColor; }
            });

            // resolve to black or white according to lightness of fill color (assume white for missing fill color)
            textColor = new Color('preset', (fillColor && fillColor.resolve('fill', themeModel).dark) ? 'white' : 'black');
        }

        return textColor.resolve('text', themeModel);
    };

    /**
     * Returns a color descriptor with details for a color that is a mix of
     * this color instance, and the passed color instance.
     *
     * @param {Color} color
     *  The other color to be mixed with this color.
     *
     * @param {Number} weighting
     *  Specifies how much of the other color will be mixed into this color.
     *  Values equal to or less than zero will result in an exact copy of THIS
     *  color. Values equal to or greater than one will result in an exact copy
     *  of the OTHER color. Values between zero and one result in a mixed color
     *  between this and the other color, where values close to zero result in
     *  colors similar to THIS color, and values close to one result in colors
     *  similar to the OTHER color.
     *
     * @param {String|Color} auto
     *  Additional information needed to resolve the automatic color. See
     *  method Color.resolve() for details.
     *
     * @param {ThemeModel} [themeModel]
     *  The model of a document theme used to map scheme color names to color
     *  values. If missing, theme colors will not be resolved correctly.
     *
     * @returns {ColorDescriptor}
     *  A descriptor for the resulting color.
     */
    Color.prototype.resolveMixed = function (color, weighting, auto, themeModel) {

        // resolve this color and the other color
        var desc1 = this.resolve(auto, themeModel);
        var desc2 = color.resolve(auto, themeModel);

        // early exit if passed weithing is out of the open interval (0,1)
        if (weighting <= 0) { return desc1; }
        if (weighting >= 1) { return desc2; }

        // use the RGBA color model to calculate a mixed color
        var r = desc1.rgb.r + (desc2.rgb.r - desc1.rgb.r) * weighting;
        var g = desc1.rgb.g + (desc2.rgb.g - desc1.rgb.g) * weighting;
        var b = desc1.rgb.b + (desc2.rgb.b - desc1.rgb.b) * weighting;
        var a = desc1.rgb.a + (desc2.rgb.a - desc1.rgb.a) * weighting;

        return new ColorDescriptor(new RGBModel(r, g, b, a), 'rgb');
    };

    /**
     * Returns the JSON representation of this color, as used in document
     * operations.
     *
     * @returns {Object}
     *  The JSON representation of this color.
     */
    Color.prototype.toJSON = function () {
        var json = { type: this._type };
        if (this._type !== 'auto') { json.value = this._value; }
        if (this._transforms && (this._transforms.length > 0)) { json.transformations = this._transforms.slice(); }
        if (_.isString(this._defRGB)) { json.fallbackValue = this._defRGB; }
        return json;
    };

    // static methods ---------------------------------------------------------

    /**
     * Parses the passed JSON representation of a color, and returns a new
     * instance of the class Color.
     *
     * @param {Any} json
     *  The JSON data to be parsed. Should be an object with the properties
     *  supported in document operations.
     *
     * @returns {Color}
     *  A new color instance.
     */
    Color.parseJSON = function (json) {
        return new Color(
            Utils.getStringOption(json, 'type', 'auto'),
            Utils.getOption(json, 'value', null),
            Utils.getArrayOption(json, 'transformations', null),
            Utils.getStringOption(json, 'fallbackValue', null)
        );
    };

    /**
     * Converts the passed CSS color string to an instance of the class Color.
     *
     * @param {String} [value]
     *  The CSS color value. The following CSS colors will be accepted:
     *  - Predefined color names, e.g. 'red'.
     *  - Hexadecimal colors (6-digit, and 3-digit), with leading hash sign.
     *  - RGB colors in function style, using the rgb() or rgba() function.
     *  - HSL colors in function style, using the hsl() or hsla() function.
     *  If missing, this method will return the value null.
     *
     * @param {Boolean} [fill=false]
     *  If set to true, the color is intended to be used as fill color. If the
     *  color is fully transparent, the automatic color will be returned,
     *  instead of an RGB color with the alpha channel set to 0.
     *
     * @returns {Color|Null}
     *  A new color instance, if the passed string could be converted to a
     *  color, otherwise null.
     */
    Color.parseCSS = function (value, fill) {

        // parse the passed CSS color
        var colorDesc = _.isString(value) ? ColorUtils.parseColor(value) : null;
        if (!colorDesc) { return null; }

        // value of the alpha channel, in the interval [0;100000]
        var alpha = round(colorDesc.a * 100000);

        // return auto color for fill mode will full transparency
        if (fill && (alpha === 0)) {
            return fill ? new Color('auto') : new Color('rgb', '000000').transform('alpha', 0);
        }

        // create a new color object according to the type of the passed CSS color
        var color = (function () {
            switch (colorDesc.type) {
                case 'preset':
                    return new Color('preset', value);
                case 'system':
                    return new Color('system', value);
                case 'hsl':
                    var hsl = colorDesc.hsl;
                    return new Color('hsl', { h: round(hsl.h * HUE_PER_DEG), s: round(hsl.s * 100000), l: round(hsl.l * 100000) });
            }
            return new Color('rgb', colorDesc.hex);
        }());

        // add the alpha channel
        if (alpha < 100000) { color.transform('alpha', alpha); }

        return color;
    };

    /**
     * Returns whether the passed JSON representations of two colors are equal.
     *
     * @param {Object} jsonColor1
     * The JSON representation of the first color to be compared to the other
     * color.
     *
     * @param {Object} jsonColor2
     * The JSON representation of the second color to be compared to the other
     * color.
     *
     * @returns {Boolean}
     *  Whether the JSON representations of two colors are equal.
     */
    Color.isEqual = function (jsonColor1, jsonColor2) {
        // Convert to Color instance, and use its euqals() method. This normalizes
        // missing properties (e.g. missing 'type' property defaults to type 'auto'),
        // and ignores unused properties (e.g. the 'fallbackValue' property).
        return Color.parseJSON(jsonColor1).equals(Color.parseJSON(jsonColor2));
    };

    // constants --------------------------------------------------------------

    /**
     * The JSON representation of the automatic color.
     *
     * @constant
     */
    Color.AUTO = { type: 'auto' };

    /**
     * The JSON representation of the hyperlink scheme color.
     *
     * @constant
     */
    Color.HYPERLINK = { type: 'scheme', value: 'hyperlink' };

    /**
     * The JSON representation of the placeholder color.
     *
     * @constant
     */
    Color.PLACEHOLDER = { type: 'scheme', value: 'phClr' };

    /**
     * The JSON representation of the color 'black'.
     *
     * @constant
     */
    Color.BLACK = { type: 'rgb', value: PRESET_COLOR_MAP.black };

    /**
     * The JSON representation of the color 'white'.
     *
     * @constant
     */
    Color.WHITE = { type: 'rgb', value: PRESET_COLOR_MAP.white };

    // exports ================================================================

    return Color;

});
