/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author York Richter <york.richtert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/control/shapetypepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/view/drawingframe',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, Rectangle, RadioList, Color, DrawingUtils, Labels, DrawingFrame, gt) {

    'use strict';

    var ICON_SIZE = Utils.SMALL_DEVICE ? 21 : 25;

    var ICON_CONTAINER_SIZE = Utils.SMALL_DEVICE ? 35 : 39;

    var SHAPES_PER_ROW = Math.min(12, (Utils.getInnerWindowWidth() / ICON_CONTAINER_SIZE) - 1);

    // attribute set used to render the shape icons
    var RENDER_ATTR_SET = {
        fill: {
            type: 'none'
        },
        line: {
            type: 'solid',
            style: 'solid',
            color: Color.BLACK,
            width: Utils.convertLengthToHmm(1, 'px'),
            headEndLength: 'small',
            headEndWidth: 'small',
            tailEndLength: 'small',
            tailEndWidth: 'small'
        }
    };

    // class ShapeTypePicker ==================================================

    /**
     * @constructor
     *
     * @extends RadioList
     */
    var ShapeTypePicker = RadioList.extend({ constructor: function (docView, initOptions) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, docView, _.extend({
            tooltip: gt('Insert a shape'),
            label: gt('Shape'),
            icon: 'docs-drawing-shape'
        }, initOptions, {
            updateCaptionMode: 'none',
            highlight: _.identity,
            itemDesign: 'grid',
            gridColumns: SHAPES_PER_ROW,
            smallerVersion: { hideLabel: true },
            mruSize: SHAPES_PER_ROW,
            mruLabel: true,
            mruSettingsKey: 'shapetypepicker'
        }));

        // private methods ----------------------------------------------------

        /**
         * hook on create items, only called once per type,
         * sets the button text and fill the chain-data for deferred rendering
         */
        function createItemHandler(event, buttonNode, presetId) {

            // total size of the shape
            var shapeSize = DrawingUtils.createDefaultSizeForShapeId(presetId, ICON_SIZE, ICON_SIZE);
            // extra processing for complex (non-straight) connectors
            var complexConnector = Labels.isPresetConnector(presetId) && !/^(line|straight)/.test(presetId);
            // location of the shape inside the canvas
            var snapRect = complexConnector ? new Rectangle(0, 4, shapeSize.width, shapeSize.height - 8) : null;

            // initialize formatting of the button element
            buttonNode.addClass('mini-caption').css({
                minWidth: ICON_CONTAINER_SIZE,
                maxWidth: ICON_CONTAINER_SIZE,
                maxHeight: ICON_CONTAINER_SIZE,
                padding: 0,
                lineHeight: 'normal'
            });

            // render the shape object
            var options = { attrs: RENDER_ATTR_SET, snapRect: snapRect, drawIconLabel: true, iconLabelFont: '11px OpenSans,sans-serif' };
            var canvasNode = DrawingFrame.drawPreviewShape(docView.getApp(), buttonNode, shapeSize, presetId, options);

            // align the canvas centered in the button element
            canvasNode.css({
                left: Math.floor((ICON_CONTAINER_SIZE - canvasNode[0].width - 6) / 2),
                top: Math.floor((ICON_CONTAINER_SIZE - canvasNode[0].height - 6) / 2)
            });
        }

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('shape-type-picker');

        // add the shape icon as canvas to a new button element
        this.getMenu().on('create:item', createItemHandler);

        // create sections
        _.each(Labels.SHAPE_LABELS_MAP, function (categoryDef, categoryId) {
            self.createMenuSection(categoryId, { label: categoryDef.title });
            // create shape buttons
            _.each(categoryDef.shapes, function (shapeDef, presetId) {
                var tooltip = _.isObject(shapeDef) ? (shapeDef.tooltip || '') : shapeDef;
                self.createOptionButton(presetId, { section: categoryId, tooltip: tooltip });
            });
        });

        // add an activated shape to the MRU list
        this.on('group:change', function (event, presetId) {
            self.getMenu().setMRUValue(presetId);
        });

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = docView = null;
        });

    } }); // class ShapeTypePicker

    // exports ================================================================

    return ShapeTypePicker;

});
