/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/control/arrowstylepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/drawinglayer/view/drawinglabels'
], function (Utils, Forms, Canvas, RadioList, Labels) {

    'use strict';

    // private global functions ===============================================

    /**
     * Creates a bitmap for the specified border line style, and returns its
     * data URL.
     *
     * @param {Number} width
     *  The width of the generated bitmap, in pixels.
     *
     * @param {Number} height
     *  The height of the generated bitmap, in pixels.
     *
     * @param {Object} borderStyle
     *  The style descriptor of a border line, with the following properties:
     *  - {String} borderStyle.style
     *      The effective line style: one of 'solid', 'dashed', 'dotted',
     *      'dashDot', or 'dashDotDot'.
     *  - {Number} borderStyle.width
     *      Width of a (single) line in the border style, in pixels. The value
     *      0 will be interpreted as hair line.
     *  - {String} borderStyle.color
     *      The line color, as CSS color value.
     *  - {Number} [borderStyle.count=1]
     *      The number of parallel lines shown in the border.
     *
     * @returns {String}
     *  The data URL of the generated bitmap.
     */
    var getArrowStyleBitmapUrl = (function () {

        // maps unique bitmap keys values to the data URLs
        var bitmapUrls = {};

        function addArrow(path, options) {

            var arrowType   = Utils.getStringOption(options, 'type', null),
                tail        = Utils.getBooleanOption(options, 'tail', false),
                width       = Utils.getNumberOption(options, 'width', 0),
                height      = Utils.getNumberOption(options, 'height', 0),
                x           = Utils.getNumberOption(options, 'x', 0),
                y           = Utils.getNumberOption(options, 'y', 0),
                rotate      = Utils.getNumberOption(options, 'rotate', 0);

            if (arrowType === 'triangle' || arrowType === 'arrow') {
                x += (tail) ? width : (width * -1);
            }

            switch (arrowType) {
                case 'diamond':
                    path.pushDiamond(x, y, width, height, rotate);
                    break;
                case 'triangle':
                    path.pushTriangle(x, y, width, height, rotate, tail);
                    break;
                case 'arrow':
                    path.pushArrow(x, y, width, height, rotate, tail);
                    break;
                case 'oval':
                    path.pushEllipse(x, y, width, height, rotate);
                    break;
            }
        }

        // the getArrowStyleBitmapUrl() method to be returned from the local scope
        function getArrowStyleBitmapUrl(docView, width, height, arrowStyle) {
            // the canvas element used to generate the bitmaps
            var canvas  = new Canvas(docView),
                arrows  = arrowStyle.style.split(':'),
                head    = arrows[0],
                tail    = arrows[1];
            // unique key of the generated bitmap
            var bitmapKey = width + ',' + height + ',' + arrowStyle.style;

            // return data URL of a bitmap already created
            if (bitmapKey in bitmapUrls) {
                return bitmapUrls[bitmapKey];
            }

            // initialize the canvas with the passed bitmap size (implicitly clears the canvas)
            canvas.initialize({ width: width, height: height });

            canvas.render(function (context) {

                var headPath    = context.createPath(),
                    tailPath    = context.createPath(),
                    path        = context.createPath(),

                    lineWidth   = 1,
                    y           = Math.floor(height / 2) - (lineWidth % 2) / 2,

                    arrowWidth  = (height / 2),
                    arrowHeight = (height / 2);

                context.setLineStyle({ style: arrowStyle.color, width: lineWidth });

                addArrow(headPath, {
                    type:   head,
                    tail:   false,
                    width:  (arrowWidth - 1),
                    height: (arrowHeight - 1),
                    x:      arrowWidth,
                    y:      y
                });

                addArrow(tailPath, {
                    type:   tail,
                    tail:   true,
                    width:  (arrowWidth - 1),
                    height: (arrowHeight - 1),
                    x:      (width - arrowWidth),
                    y:      y
                });

                var lineStartX  = arrowWidth,
                    lineEndX    = (width - arrowWidth);

                if (head === 'none' || head === 'arrow')  { lineStartX  = 1;            }
                if (tail === 'none' || tail === 'arrow')  { lineEndX    = (width - 1);  }

                path.pushLine(lineStartX, y, lineEndX, y);

                context.drawPath(path, 'stroke');
                context.drawPath(headPath, (head === 'arrow') ? 'stroke' : 'fill');
                context.drawPath(tailPath, (tail === 'arrow') ? 'stroke' : 'fill');
            });

            // convert the bitmap to a data URL
            return (bitmapUrls[bitmapKey] = canvas.getDataURL());
        }

        return getArrowStyleBitmapUrl;
    }());

    // class ArrowStylePicker ================================================

    /**
     * A generic drop-down list control for border styles.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Array<Object>} listEntries
     *  An array of descriptors for the entries of the drop-down list. Each
     *  descriptor is an object with the following properties:
     *  - {Any} value
     *      The value associated to the list item.
     *  - {String} label
     *      The text label for the list item.
     *  - {Object} style
     *      The style properties for the generated preview icon, with the
     *      following properties:
     *      - {String} style.style
     *          The line style: one of 'solid', 'dashed', 'dotted', 'dashDot',
     *          or 'dashDotDot'.
     *      - {Number} style.width
     *          Width of a (single) line, in pixels. The value 0 will be
     *          interpreted as hair line.
     *      - {Number} [style.count=1]
     *          Number of parallel lines.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  RadioList.
     */
    function ArrowStylePicker(docView, listEntries, initOptions) {

        // maps control values to border style descriptors
        var arrowStyles = {};

        // base constructor ---------------------------------------------------

        initOptions = Utils.extendOptions({
            icon: 'fa-exchange',
            tooltip: Labels.ARROW_STYLE_LABEL.tooltip,
            updateCaptionMode: 'none'
        }, initOptions);

        RadioList.call(this, docView, Utils.extendOptions({
            dropDownVersion: { label: Utils.getStringOption(initOptions, 'tooltip') }
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Returns the data URL of the preview bitmap for the passed value.
         */
        function getBitmapUrl(value) {
            var arrowStyle = (value && (value in arrowStyles)) ? arrowStyles[value] : { style: 'solid', width: 1, color: '#333333' };
            return getArrowStyleBitmapUrl(docView, 100, 18, arrowStyle);
        }

        /**
         * Creates an image element showing a preview border line of the
         * specified control value.
         */
        function createStyleBox(value) {
            return $('<img class="lineend-style-box" src="' + getBitmapUrl(value) + '">');
        }

        /**
         * Initializes a new list item according to the control value.
         */
        function createItemHandler(event, buttonNode, value) {
            Forms.getCaptionNode(buttonNode).prepend(createStyleBox(value));
        }

        // initialization -----------------------------------------------------

        // register a handler that inserts a border box into each list item
        this.getMenu().on('create:item', createItemHandler);

        // create the bitmaps for the icons, and insert the entries into the drop-down list
        listEntries.forEach(function (entry) {
            arrowStyles[entry.value] = _.extend({ color: '#333333' }, entry.icon);
            this.createOptionButton(entry.value, { });
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = listEntries = initOptions = null;
        });

    } // class ArrowStylePicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: ArrowStylePicker });

});
