/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/utils/scheduler',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/timermixin'
], function (Utils, IteratorUtils, Scheduler, BaseObject, TimerMixin) {

    'use strict';

    // jQuery.Deferred base test ==============================================

    describe('class jQuery.Deferred', function () {
        it('should exist', function () {
            expect($.Deferred).to.be.a('function');
        });
        it('should create a pending deferred object', function () {
            var def = $.Deferred();
            expect(def).to.be.an('object');
            expect(def).to.respondTo('state');
            expect(def).to.respondTo('promise');
            expect(def).to.respondTo('resolve');
            expect(def).to.respondTo('reject');
            expect(def).to.respondTo('done');
            expect(def).to.respondTo('fail');
            expect(def).to.respondTo('then');
        });
        it('should have pending state', function () {
            expect($.Deferred().state()).to.equal('pending');
        });
        it('should resolve immediately', function () {
            var def = $.Deferred();
            expect(def.resolve()).to.equal(def);
            expect(def.state()).to.equal('resolved');
        });
        it('should reject immediately', function () {
            var def = $.Deferred();
            expect(def.reject()).to.equal(def);
            expect(def.state()).to.equal('rejected');
        });
    });

    // mix-in class TimerMixin ================================================

    describe('Toolkit mix-in class TimerMixin', function () {

        it('should exist', function () {
            expect(TimerMixin).to.be.a('function');
        });

        // Scheduler must be empty, otherwise some tests may fail unexpectedly because of other
        // running background timer loops. If this test fails, there may exist another unit test
        // module whare an application has been created globally. Check that the function call
        // AppHelper.createXYZApplication() is located locally inside the callback of the
        // outermost "describe()" of the unit test.
        it('should run with clean timer scheduler', function () {
            expect(Scheduler.empty()).to.equal(true);
        });

        // private helpers ----------------------------------------------------

        function TimerClass() {
            BaseObject.call(this);
            TimerMixin.call(this);
        }

        var clock = null;
        before(function () { clock = sinon.useFakeTimers(); });
        after(function () { clock.restore(); });

        // ensure empty timer queue after each test
        afterEach(function () { clock.tick(10000); });

        // the minimum delay of the timer scheduler
        var INT = Scheduler.INTERVAL;

        // public methods -----------------------------------------------------

        describe('method "executeDelayed"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('executeDelayed');
            });
            it('should invoke callback delayed', function () {
                var spy = sinon.spy();
                var timer = obj.executeDelayed(spy);
                expect(spy.called).to.equal(false);
                expect(timer.state()).to.equal('pending');
                clock.tick(2);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledOnce).to.equal(true);
                expect(spy.calledOn(obj)).to.equal(true);
            });
            it('should invoke callbacks after specified delay time', function () {
                var spy = sinon.spy();
                obj.executeDelayed(spy, { delay: 10 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(spy.called).to.equal(true);
            });
            it('should accept delay time directly as numeric parameter', function () {
                var spy = sinon.spy();
                obj.executeDelayed(spy, 10);
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(spy.called).to.equal(true);
            });
            it('should invoke callbacks in correct order', function () {
                var spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.executeDelayed(spy1, 12); // fires at 12
                obj.executeDelayed(spy2, 10); // fires at 10
                clock.tick(11); // timer fires at 10
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(true);
                // move time to 13
                clock.tick(2); // timer fires at 12
                expect(spy1.called).to.equal(true);
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy();
                var timer = obj.executeDelayed(spy, 10);
                clock.tick(9);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.called).to.equal(false);
            });
            it('should forward the callback result', function () {
                var stub = sinon.stub().returns(42);
                var spy = sinon.spy();
                var timer = obj.executeDelayed(stub, 10).done(spy);
                clock.tick(20);
                expect(stub.called).to.equal(true);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledWith(42)).to.equal(true);
            });
            it('should defer timer promise to callback promise', function () {
                var def = $.Deferred();
                var stub = sinon.stub().returns(def.promise());
                var spy = sinon.spy();
                var timer = obj.executeDelayed(stub, 10).done(spy);
                clock.tick(20);
                expect(stub.called).to.equal(true);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                def.resolve(42);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledWith(42)).to.equal(true);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.executeDelayed(spy, 10);
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy();
                var timer = obj.executeDelayed(spy, 10);
                clock.tick(9);
                obj.destroy();
                clock.tick(1000);
                expect(spy.called).to.equal(false);
                expect(timer.state()).to.equal('rejected');
            });
        });

        describe('method "repeatDelayed"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('repeatDelayed');
            });
            it('should invoke callback delayed and stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub);
                stub.onThirdCall().returns(Utils.BREAK);
                expect(stub.called).to.equal(false);
                clock.tick(1000);
                expect(stub.calledThrice).to.equal(true);
                expect(stub.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass iteration index to callback', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(1000);
                expect(stub.getCall(0).calledWith(0)).to.equal(true);
                expect(stub.getCall(1).calledWith(1)).to.equal(true);
                expect(stub.getCall(2).calledWith(2)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should invoke callbacks after specified delay time', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, { delay: 100 }); // fires at 100, 200, 300, ...
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(99);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(98);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(2);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(98);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should accept delay time directly as numeric parameter', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 100);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(99);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(100);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(100);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var stub = sinon.stub();
                stub.onThirdCall().returns(Utils.BREAK);
                var timer = obj.repeatDelayed(stub, { delay: 'immediate' });
                expect(stub.calledOnce).to.equal(true);
                clock.tick(10000);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use different repetition delay time', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, { delay: 10, repeatDelay: 100 });
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(9);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(98);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(2);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(98);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified number of cycles', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, { delay: 100, cycles: 3 });
                clock.tick(99);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(100);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(100);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy();
                var timer = obj.repeatDelayed(spy, 100);
                clock.tick(250);
                expect(spy.calledTwice).to.equal(true);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.calledTwice).to.equal(true);
            });
            it('should defer next iteration to callback promise', function () {
                var stub = sinon.stub();
                var def = $.Deferred();
                var timer = obj.repeatDelayed(stub, 100);
                stub.onSecondCall().returns(def);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(150);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(100);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(200);
                expect(stub.calledTwice).to.equal(true);
                def.resolve();
                clock.tick(99);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 100);
                stub.onThirdCall().returns($.Deferred().reject());
                clock.tick(250);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(100);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('rejected');
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.repeatDelayed(spy, 100);
                clock.tick(99);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should reduce delay time after a pending promise in fastAsync mode', function () {
                var stub = sinon.stub();
                var def1 = $.Deferred();
                var def2 = $.Deferred();
                var timer = obj.repeatDelayed(stub, { delay: 100, fastAsync: true });
                stub.onFirstCall().returns(def1);
                stub.onSecondCall().returns(def2);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(110);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(40);
                def1.resolve();
                clock.tick(40);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(20);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(200);
                expect(stub.calledTwice).to.equal(true);
                def2.resolve();
                clock.tick(1);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy();
                var timer = obj.repeatDelayed(spy, 100);
                clock.tick(250);
                expect(spy.calledTwice).to.equal(true);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.calledTwice).to.equal(true);
            });
        });

        describe('method "repeatSliced"', function () {

            // Creates a callback spy that simulates to execute for 150ms, returns 'ret'
            // for the first four invocations, and Utils.BREAK for the fifth invocation.
            function makeSpy(ret) {
                return sinon.spy(function (i) {
                    clock.tick(150);
                    return (i === 4) ? Utils.BREAK : ret;
                });
            }

            var obj = new TimerClass();

            it('should exist', function () {
                expect(obj).to.respondTo('repeatSliced');
            });
            it('should invoke callback in time slices and stop on Utils.BREAK', function () {
                var spy = makeSpy(null);
                var timer = obj.repeatSliced(spy);
                expect(spy.called).to.equal(false);
                // tick once to trigger first slice; spy will run twice (150 each)
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                // clock is now 1 after slice#1, next cycle starts in INT-1
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                // clock is now 1 before slice#2
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                // clock is now after slice#2, next cycle starts in INT
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                // clock is now 1 before slice#3
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass iteration index to callback', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy);
                // run all time slices at once
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(0)).to.equal(true);
                expect(spy.getCall(1).calledWith(1)).to.equal(true);
                expect(spy.getCall(2).calledWith(2)).to.equal(true);
                expect(spy.getCall(3).calledWith(3)).to.equal(true);
                expect(spy.getCall(4).calledWith(4)).to.equal(true);
            });
            it('should invoke callbacks after specified delay/slice/interval times', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, { delay: 10, slice: 200, interval: 100 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                // clock is now at 9, tick once to trigger the first cycle
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                // clock is now after first slice, the next cycle starts in 100
                clock.tick(99);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                // clock is now after slice#2, the next cycle starts in 100
                clock.tick(100);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified number of cycles', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, { cycles: 3, interval: 100 });
                // tick once to trigger the first slice; spy will run twice (150 each)
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                // clock is now 1 after slice#1, next time slice starts in 99
                clock.tick(98);
                expect(spy.callCount).to.equal(2);
                // clock is now 1 before slice#2
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, { delay: 'immediate' });
                expect(spy.callCount).to.equal(2);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy);
                // tick once to trigger the first cycle; spy will run twice (150 each)
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.repeatSliced(spy);
                // tick once to trigger the first cycle; spy will run once (150 each) and return a pending promise
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                // loop will wait for the promise
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                // resolve the deferred object to let the loop continue
                def.resolve();
                expect(timer.state()).to.equal('pending');
                expect(spy.callCount).to.equal(1);
                // trigger the next cycles, spy will run twice because it returns a resolved promise
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(1000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.repeatSliced(spy);
                // tick once to trigger the first cycle; spy will run once (150 each) and return a pending promise
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                // clock is now 1 after first cycle, loop will wait for the promise
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                // reject the deferred object
                def.reject();
                expect(spy.callCount).to.equal(1);
                expect(timer.state()).to.equal('rejected');
                // loop must not continue after a rejected promise
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.repeatSliced(spy, { delay: 10 });
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy);
                // tick once to trigger the first cycle; spy will run twice (150 each)
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                // destroy the object
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                // loop must not continue after destruction
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "iterateSliced"', function () {

            // See test cases for TimerMixin.repeatSliced() above for details about how
            // these tests work.

            // Creates a callback spy that simulates to execute for 150ms, and returns 'ret'.
            function makeSpy(ret) {
                return sinon.spy(function () {
                    clock.tick(150);
                    return ret;
                });
            }

            var obj = new TimerClass();

            it('should exist', function () {
                expect(obj).to.respondTo('iterateSliced');
            });
            it('should invoke callback in time slices', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), stub);
                expect(stub.called).to.equal(false);
                clock.tick(1);
                expect(stub.callCount).to.equal(2);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass iterator results to callback', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy);
                clock.tick(1000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(0, { done: false, value: 0 })).to.equal(true);
                expect(spy.getCall(1).calledWith(1, { done: false, value: 1 })).to.equal(true);
                expect(spy.getCall(2).calledWith(2, { done: false, value: 2 })).to.equal(true);
                expect(spy.getCall(3).calledWith(3, { done: false, value: 3 })).to.equal(true);
                expect(spy.getCall(4).calledWith(4, { done: false, value: 4 })).to.equal(true);
            });
            it('should invoke callbacks after specified delay/slice/interval time', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, { delay: 10, slice: 200, interval: 100 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(100);
                expect(spy.callCount).to.equal(4);
                clock.tick(100);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, { delay: 'immediate' });
                expect(spy.callCount).to.equal(2);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                expect(timer.state()).to.equal('pending');
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(5);
                // last array element visited, but iterator did not return the final {done:true} yet
                expect(timer.state()).to.equal('pending');
                clock.tick(INT);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, { delay: 10 });
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "reduceSliced"', function () {

            // See test cases for TimerMixin.repeatSliced() above for details about how
            // these tests work.

            // Creates a callback spy that simulates to execute for 150ms, and returns 'ret'.
            function makeSpy(ret) {
                return sinon.spy(function () {
                    clock.tick(150);
                    return ret;
                });
            }

            // Creates a callback spy that simulates to execute for 150ms, and returns the sum of its arguments
            function makeSumSpy() {
                return sinon.spy(function (a, b) {
                    clock.tick(150);
                    return a + b;
                });
            }

            var obj = new TimerClass();

            it('should exist', function () {
                expect(obj).to.respondTo('iterateSliced');
            });
            it('should invoke callback in time slices', function () {
                var spy = makeSpy();
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass correct values to callback function', function (done) {
                var spy = makeSumSpy();
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(1, 2, { done: false, value: 2 })).to.equal(true);
                expect(spy.getCall(1).calledWith(3, 3, { done: false, value: 3 })).to.equal(true);
                expect(spy.getCall(2).calledWith(6, 4, { done: false, value: 4 })).to.equal(true);
                expect(spy.getCall(3).calledWith(10, 5, { done: false, value: 5 })).to.equal(true);
                expect(spy.getCall(4).calledWith(15, 6, { done: false, value: 6 })).to.equal(true);
                timer.done(function (result) { expect(result).to.equal(21); done(); });
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                expect(timer.state()).to.equal('pending');
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(5);
                // last array element visited, but iterator did not return the final {done:true} yet
                expect(timer.state()).to.equal('pending');
                clock.tick(INT);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "iterateArraySliced"', function () {

            // See test cases for TimerMixin.repeatSliced() above for details about how
            // these tests work.

            // Creates a callback spy that simulates to execute for 150ms, and returns 'ret'.
            function makeSpy(ret) {
                return sinon.spy(function () {
                    clock.tick(150);
                    return ret;
                });
            }

            var ARRAY = ['a', 'b', 'c', 'd', 'e'];
            var obj = new TimerClass();

            it('should exist', function () {
                expect(obj).to.respondTo('iterateArraySliced');
            });
            it('should return a resolved promise, if an empty array is given', function () {
                var spy = sinon.spy();
                var timer = obj.iterateArraySliced([]);
                expect(timer).to.be.an('object');
                expect(timer.state()).to.equal('resolved');
                expect(spy.callCount).to.equal(0);
            });
            it('should invoke callback in time slices', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                var timer = obj.iterateArraySliced(ARRAY, stub);
                expect(stub.called).to.equal(false);
                clock.tick(1);
                expect(stub.callCount).to.equal(2);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass array elements to callback', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('a', 0, ARRAY)).to.equal(true);
                expect(spy.getCall(1).calledWith('b', 1, ARRAY)).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, ARRAY)).to.equal(true);
                expect(spy.getCall(3).calledWith('d', 3, ARRAY)).to.equal(true);
                expect(spy.getCall(4).calledWith('e', 4, ARRAY)).to.equal(true);
            });
            it('should process array elements in reversed order', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, { reverse: true });
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('e', 4, ARRAY)).to.equal(true);
                expect(spy.getCall(1).calledWith('d', 3, ARRAY)).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, ARRAY)).to.equal(true);
                expect(spy.getCall(3).calledWith('b', 1, ARRAY)).to.equal(true);
                expect(spy.getCall(4).calledWith('a', 0, ARRAY)).to.equal(true);
            });
            it('should work with array-like objects', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced('abcde', spy);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('a', 0, 'abcde')).to.equal(true);
                expect(spy.getCall(1).calledWith('b', 1, 'abcde')).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, 'abcde')).to.equal(true);
                expect(spy.getCall(3).calledWith('d', 3, 'abcde')).to.equal(true);
                expect(spy.getCall(4).calledWith('e', 4, 'abcde')).to.equal(true);
            });
            it('should invoke callbacks after specified delay/slice/interval time', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, { delay: 10, slice: 200, interval: 100 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(100);
                expect(spy.callCount).to.equal(4);
                clock.tick(100);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, { delay: 'immediate' });
                expect(spy.callCount).to.equal(2);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                expect(timer.state()).to.equal('pending');
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.iterateArraySliced(ARRAY, spy, { delay: 10 });
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "iterateObjectSliced"', function () {

            // See test cases for TimerMixin.repeatSliced() above for details about how
            // these tests work.

            // Creates a callback spy that simulates to execute for 150ms, and returns 'ret'.
            function makeSpy(ret) {
                return sinon.spy(function () {
                    clock.tick(150);
                    return ret;
                });
            }

            var OBJECT = { a: 1, b: 2, c: 3, d: 4, e: 5 };
            var obj = new TimerClass();

            it('should exist', function () {
                expect(obj).to.respondTo('iterateObjectSliced');
            });
            it('should invoke callback in time slices', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                var timer = obj.iterateObjectSliced(OBJECT, stub);
                expect(stub.called).to.equal(false);
                clock.tick(1);
                expect(stub.callCount).to.equal(2);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass object properties to callback', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(1, 'a', OBJECT)).to.equal(true);
                expect(spy.getCall(1).calledWith(2, 'b', OBJECT)).to.equal(true);
                expect(spy.getCall(2).calledWith(3, 'c', OBJECT)).to.equal(true);
                expect(spy.getCall(3).calledWith(4, 'd', OBJECT)).to.equal(true);
                expect(spy.getCall(4).calledWith(5, 'e', OBJECT)).to.equal(true);
            });
            it('should invoke callbacks after specified delay/slice/interval time', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy, { delay: 10, slice: 200, interval: 100 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(100);
                expect(spy.callCount).to.equal(4);
                clock.tick(100);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy, { delay: 'immediate' });
                expect(spy.callCount).to.equal(2);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                expect(timer.state()).to.equal('pending');
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.iterateObjectSliced(OBJECT, spy, { delay: 10 });
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "createSynchronizedMethod"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('createSynchronizedMethod');
            });
            it('should invoke callback serialized', function () {
                var defs = _.times(3, function () { return $.Deferred(); });
                var stub = sinon.stub();
                var method = obj.createSynchronizedMethod(stub);
                defs.forEach(function (def, index) { stub.onCall(index).returns(def); });
                expect(stub.callCount).to.equal(0);
                var promise0 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise0.state()).to.equal('pending');
                var promise1 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise1.state()).to.equal('pending');
                var promise2 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise2.state()).to.equal('pending');
                clock.tick(100);
                defs[0].resolve();
                expect(stub.callCount).to.equal(1);
                clock.tick(100);
                expect(stub.callCount).to.equal(2);
                expect(promise0.state()).to.equal('resolved');
                expect(promise1.state()).to.equal('pending');
                clock.tick(100);
                defs[2].reject();
                expect(stub.callCount).to.equal(2);
                expect(promise1.state()).to.equal('pending');
                expect(promise2.state()).to.equal('pending');
                clock.tick(100);
                defs[1].resolve();
                expect(stub.callCount).to.equal(2);
                clock.tick(100);
                expect(stub.callCount).to.equal(3);
                expect(promise1.state()).to.equal('resolved');
                expect(promise2.state()).to.equal('rejected');
            });
            it('should pass parameters to callback', function () {
                var defs = _.times(3, function () { return $.Deferred(); });
                var stub = sinon.stub();
                var method = obj.createSynchronizedMethod(stub);
                defs.forEach(function (def, index) { stub.onCall(index).returns(def); });
                expect(stub.callCount).to.equal(0);
                method(1, 'a');
                method(2, 'b');
                method(3, 'c');
                clock.tick(20);
                defs[0].resolve();
                clock.tick(100);
                defs[1].resolve();
                clock.tick(100);
                defs[2].resolve();
                clock.tick(100);
                expect(stub.callCount).to.equal(3);
                expect(stub.getCall(0).calledWith(1, 'a')).to.equal(true);
                expect(stub.getCall(1).calledWith(2, 'b')).to.equal(true);
                expect(stub.getCall(2).calledWith(3, 'c')).to.equal(true);
            });
            it('should invoke callback with current context', function () {
                var spy = sinon.spy();
                var method = obj.createSynchronizedMethod(spy);
                method.call(obj);
                expect(spy.called).to.equal(true);
                expect(spy.calledOn(obj)).to.equal(true);
            });
            it('should not invoke callbacks after destruction', function () {
                var defs = _.times(3, function () { return $.Deferred(); });
                var stub = sinon.stub();
                var method = obj.createSynchronizedMethod(stub);
                _.each(defs, function (def, index) { stub.onCall(index).returns(def); });
                var promises = _.times(defs.length, method);
                clock.tick(100);
                defs[0].resolve();
                clock.tick(100);
                expect(stub.callCount).to.equal(2);
                expect(promises[0].state()).to.equal('resolved');
                clock.tick(100);
                expect(promises[1].state()).to.equal('pending');
                expect(promises[2].state()).to.equal('pending');
                obj.destroy();
                defs[1].resolve();
                clock.tick(1000);
                defs[2].resolve();
                expect(stub.callCount).to.equal(2);
                expect(promises[1].state()).to.equal('rejected');
                expect(promises[2].state()).to.equal('rejected');
            });
        });

        describe('method "createDebouncedMethod"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('createDebouncedMethod');
            });
            it('should invoke callback debounced', function () {
                var spy1 = sinon.spy(), spy2 = sinon.spy();
                var method = obj.createDebouncedMethod(spy1, spy2);
                expect(spy1.callCount).to.equal(0);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(1);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(2);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(3);
                expect(spy2.callCount).to.equal(0);
                clock.tick(10);
                expect(spy1.callCount).to.equal(3);
                expect(spy2.callCount).to.equal(1);
            });
            it('should pass parameters to callback', function () {
                var spy1 = sinon.spy(), spy2 = sinon.spy();
                var method = obj.createDebouncedMethod(spy1, spy2);
                method(1, 'a');
                method(2, 'b');
                method(3, 'c');
                clock.tick(10);
                expect(spy1.callCount).to.equal(3);
                expect(spy1.getCall(0).calledWith(1, 'a')).to.equal(true);
                expect(spy1.getCall(1).calledWith(2, 'b')).to.equal(true);
                expect(spy1.getCall(2).calledWith(3, 'c')).to.equal(true);
                expect(spy2.callCount).to.equal(1);
                expect(spy2.firstCall.args.length).to.equal(0);
            });
            it('should invoke callbacks with current context', function () {
                var spy = sinon.spy();
                var method = obj.createDebouncedMethod(spy, spy);
                method.call(obj);
                clock.tick(10);
                expect(spy.calledTwice).to.equal(true);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
            });
            it('should use specified delay time', function () {
                var spy = sinon.spy();
                var method = obj.createDebouncedMethod(_.noop, spy, { delay: 10 });
                method();
                clock.tick(2);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(11);
                expect(spy.callCount).to.equal(1);
            });
            it('should use specified maximum delay time', function () {
                var spy = sinon.spy();
                var method = obj.createDebouncedMethod(_.noop, spy, { delay: 100, maxDelay: 300 });
                method();
                clock.tick(99);
                method();
                clock.tick(99);
                method();
                clock.tick(99);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(2);
                expect(spy.callCount).to.equal(0);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                method();
                clock.tick(99);
                expect(spy.callCount).to.equal(1);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
            });
            it('should not invoke callback after destruction', function () {
                var spy = sinon.spy();
                var method = obj.createDebouncedMethod(_.noop, spy);
                method();
                obj.destroy();
                clock.tick(1000);
                expect(spy.callCount).to.equal(0);
            });
        });
    });

    // ========================================================================
});
