/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/render/font',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/locale/formatter'
], function (Font, Parser, Formatter) {

    'use strict';

    // class Formatter ========================================================

    describe('Toolkit class Formatter', function () {

        it('should exist', function () {
            expect(Formatter).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var formatter = null;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('creates a new instance', function () {
                formatter = new Formatter();
                expect(formatter).to.be.an('object');
            });
        });

        // public methods -----------------------------------------------------

        describe('method "configure"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('configure');
            });
        });

        describe('method "isValidDate"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('isValidDate');
            });
        });

        describe('method "convertNumberToDate"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('convertNumberToDate');
            });
        });

        describe('method "convertDateToNumber"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('convertDateToNumber');
            });
        });

        describe('method "parseLeadingDate"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('parseLeadingDate');
            });
        });

        describe('method "parseLeadingTime"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('parseLeadingTime');
            });
        });

        describe('method "formatRoman"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('formatRoman');
            });
            it('should return an empty string for zero', function () {
                expect(formatter.formatRoman(0, 0)).to.equal('');
                expect(formatter.formatRoman(0, 1)).to.equal('');
                expect(formatter.formatRoman(0, 2)).to.equal('');
                expect(formatter.formatRoman(0, 3)).to.equal('');
                expect(formatter.formatRoman(0, 4)).to.equal('');
            });
            it('should return roman numbers for decimal numbers 1 to 9', function () {
                expect(formatter.formatRoman(1, 0)).to.equal('I');
                expect(formatter.formatRoman(1, 1)).to.equal('I');
                expect(formatter.formatRoman(1, 2)).to.equal('I');
                expect(formatter.formatRoman(1, 3)).to.equal('I');
                expect(formatter.formatRoman(1, 4)).to.equal('I');
                expect(formatter.formatRoman(2, 0)).to.equal('II');
                expect(formatter.formatRoman(3, 0)).to.equal('III');
                expect(formatter.formatRoman(4, 0)).to.equal('IV');
                expect(formatter.formatRoman(4, 1)).to.equal('IV');
                expect(formatter.formatRoman(4, 2)).to.equal('IV');
                expect(formatter.formatRoman(4, 3)).to.equal('IV');
                expect(formatter.formatRoman(4, 4)).to.equal('IV');
                expect(formatter.formatRoman(5, 0)).to.equal('V');
                expect(formatter.formatRoman(6, 0)).to.equal('VI');
                expect(formatter.formatRoman(7, 0)).to.equal('VII');
                expect(formatter.formatRoman(8, 0)).to.equal('VIII');
                expect(formatter.formatRoman(9, 0)).to.equal('IX');
                expect(formatter.formatRoman(9, 1)).to.equal('IX');
                expect(formatter.formatRoman(9, 2)).to.equal('IX');
                expect(formatter.formatRoman(9, 3)).to.equal('IX');
                expect(formatter.formatRoman(9, 4)).to.equal('IX');
            });
            it('should return roman numbers for decimal numbers 10 to 99', function () {
                expect(formatter.formatRoman(10, 0)).to.equal('X');
                expect(formatter.formatRoman(19, 0)).to.equal('XIX');
                expect(formatter.formatRoman(19, 4)).to.equal('XIX');
                expect(formatter.formatRoman(20, 0)).to.equal('XX');
                expect(formatter.formatRoman(35, 0)).to.equal('XXXV');
                expect(formatter.formatRoman(35, 4)).to.equal('XXXV');
                expect(formatter.formatRoman(39, 0)).to.equal('XXXIX');
                expect(formatter.formatRoman(39, 4)).to.equal('XXXIX');
                expect(formatter.formatRoman(40, 0)).to.equal('XL');
                expect(formatter.formatRoman(40, 4)).to.equal('XL');
                expect(formatter.formatRoman(44, 0)).to.equal('XLIV');
                expect(formatter.formatRoman(44, 4)).to.equal('XLIV');
                expect(formatter.formatRoman(45, 0)).to.equal('XLV');
                expect(formatter.formatRoman(45, 1)).to.equal('VL');
                expect(formatter.formatRoman(45, 2)).to.equal('VL');
                expect(formatter.formatRoman(45, 3)).to.equal('VL');
                expect(formatter.formatRoman(45, 4)).to.equal('VL');
                expect(formatter.formatRoman(48, 0)).to.equal('XLVIII');
                expect(formatter.formatRoman(48, 1)).to.equal('VLIII');
                expect(formatter.formatRoman(49, 0)).to.equal('XLIX');
                expect(formatter.formatRoman(49, 1)).to.equal('VLIV');
                expect(formatter.formatRoman(49, 2)).to.equal('IL');
                expect(formatter.formatRoman(49, 3)).to.equal('IL');
                expect(formatter.formatRoman(49, 4)).to.equal('IL');
                expect(formatter.formatRoman(50, 0)).to.equal('L');
                expect(formatter.formatRoman(50, 4)).to.equal('L');
                expect(formatter.formatRoman(94, 0)).to.equal('XCIV');
                expect(formatter.formatRoman(94, 4)).to.equal('XCIV');
                expect(formatter.formatRoman(95, 0)).to.equal('XCV');
                expect(formatter.formatRoman(95, 1)).to.equal('VC');
                expect(formatter.formatRoman(95, 4)).to.equal('VC');
                expect(formatter.formatRoman(99, 0)).to.equal('XCIX');
                expect(formatter.formatRoman(99, 1)).to.equal('VCIV');
                expect(formatter.formatRoman(99, 2)).to.equal('IC');
                expect(formatter.formatRoman(99, 3)).to.equal('IC');
                expect(formatter.formatRoman(99, 4)).to.equal('IC');
            });
            it('should return roman numbers for decimal numbers 100 to 999', function () {
                expect(formatter.formatRoman(100, 0)).to.equal('C');
                expect(formatter.formatRoman(100, 4)).to.equal('C');
                expect(formatter.formatRoman(199, 0)).to.equal('CXCIX');
                expect(formatter.formatRoman(199, 1)).to.equal('CVCIV');
                expect(formatter.formatRoman(199, 2)).to.equal('CIC');
                expect(formatter.formatRoman(199, 3)).to.equal('CIC');
                expect(formatter.formatRoman(199, 4)).to.equal('CIC');
                expect(formatter.formatRoman(300, 0)).to.equal('CCC');
                expect(formatter.formatRoman(300, 4)).to.equal('CCC');
                expect(formatter.formatRoman(400, 0)).to.equal('CD');
                expect(formatter.formatRoman(400, 4)).to.equal('CD');
                expect(formatter.formatRoman(450, 0)).to.equal('CDL');
                expect(formatter.formatRoman(450, 1)).to.equal('LD');
                expect(formatter.formatRoman(450, 4)).to.equal('LD');
                expect(formatter.formatRoman(490, 0)).to.equal('CDXC');
                expect(formatter.formatRoman(490, 1)).to.equal('LDXL');
                expect(formatter.formatRoman(490, 2)).to.equal('XD');
                expect(formatter.formatRoman(490, 3)).to.equal('XD');
                expect(formatter.formatRoman(490, 4)).to.equal('XD');
                expect(formatter.formatRoman(495, 0)).to.equal('CDXCV');
                expect(formatter.formatRoman(495, 1)).to.equal('LDVL');
                expect(formatter.formatRoman(495, 2)).to.equal('XDV');
                expect(formatter.formatRoman(495, 3)).to.equal('VD');
                expect(formatter.formatRoman(495, 4)).to.equal('VD');
                expect(formatter.formatRoman(499, 0)).to.equal('CDXCIX');
                expect(formatter.formatRoman(499, 1)).to.equal('LDVLIV');
                expect(formatter.formatRoman(499, 2)).to.equal('XDIX');
                expect(formatter.formatRoman(499, 3)).to.equal('VDIV');
                expect(formatter.formatRoman(499, 4)).to.equal('ID');
                expect(formatter.formatRoman(500, 0)).to.equal('D');
                expect(formatter.formatRoman(500, 4)).to.equal('D');
                expect(formatter.formatRoman(888, 0)).to.equal('DCCCLXXXVIII');
                expect(formatter.formatRoman(888, 4)).to.equal('DCCCLXXXVIII');
                expect(formatter.formatRoman(999, 0)).to.equal('CMXCIX');
                expect(formatter.formatRoman(999, 1)).to.equal('LMVLIV');
                expect(formatter.formatRoman(999, 2)).to.equal('XMIX');
                expect(formatter.formatRoman(999, 3)).to.equal('VMIV');
                expect(formatter.formatRoman(999, 4)).to.equal('IM');
            });
            it('should return roman numbers for decimal numbers 1000 to 3999', function () {
                expect(formatter.formatRoman(1000, 0)).to.equal('M');
                expect(formatter.formatRoman(1000, 4)).to.equal('M');
                expect(formatter.formatRoman(3888, 0)).to.equal('MMMDCCCLXXXVIII');
                expect(formatter.formatRoman(3888, 4)).to.equal('MMMDCCCLXXXVIII');
                expect(formatter.formatRoman(3999, 0)).to.equal('MMMCMXCIX');
                expect(formatter.formatRoman(3999, 1)).to.equal('MMMLMVLIV');
                expect(formatter.formatRoman(3999, 2)).to.equal('MMMXMIX');
                expect(formatter.formatRoman(3999, 3)).to.equal('MMMVMIV');
                expect(formatter.formatRoman(3999, 4)).to.equal('MMMIM');
            });
            it('should return null for invalid numbers', function () {
                expect(formatter.formatRoman(-1)).to.equal(null);
                expect(formatter.formatRoman(4000)).to.equal(null);
            });
        });

        describe('function "parseRoman"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('parseRoman');
            });
            it('should convert roman to decimal', function () {
                expect(formatter.parseRoman('')).to.equal(0);
                expect(formatter.parseRoman('I')).to.equal(1);
                expect(formatter.parseRoman('V')).to.equal(5);
                expect(formatter.parseRoman('X')).to.equal(10);
                expect(formatter.parseRoman('L')).to.equal(50);
                expect(formatter.parseRoman('C')).to.equal(100);
                expect(formatter.parseRoman('D')).to.equal(500);
                expect(formatter.parseRoman('M')).to.equal(1000);
                expect(formatter.parseRoman('II')).to.equal(2);
                expect(formatter.parseRoman('III')).to.equal(3);
                expect(formatter.parseRoman('IIII')).to.equal(4);
                expect(formatter.parseRoman('IIIII')).to.equal(5);
                expect(formatter.parseRoman('MMMDDDCCCLLLXXXVVVIII')).to.equal(4998);
                expect(formatter.parseRoman('IV')).to.equal(4);
                expect(formatter.parseRoman('IX')).to.equal(9);
                expect(formatter.parseRoman('XXXIX')).to.equal(39);
                expect(formatter.parseRoman('IM')).to.equal(999);
                expect(formatter.parseRoman('VMIV')).to.equal(999);
                expect(formatter.parseRoman('XMIX')).to.equal(999);
                expect(formatter.parseRoman('LMVLIV')).to.equal(999);
                expect(formatter.parseRoman('CMXCIX')).to.equal(999);
                expect(formatter.parseRoman('DMCDLCXLVXIV')).to.equal(999);
                expect(formatter.parseRoman('MIM')).to.equal(1999);
                expect(formatter.parseRoman('MMMIM')).to.equal(3999);
                expect(formatter.parseRoman('MMMIM')).to.equal(3999);
                expect(formatter.parseRoman('IIV')).to.equal(3);
                expect(formatter.parseRoman('IIIV')).to.equal(2);
                expect(formatter.parseRoman('IIIIV')).to.equal(1);
                expect(formatter.parseRoman('IIIIIV')).to.equal(0);
                expect(formatter.parseRoman('IIIIIIV')).to.equal(-1);
                expect(formatter.parseRoman('IMIVIX')).to.equal(1002);
                expect(formatter.parseRoman('MDCLXVI')).to.equal(1666);
                expect(formatter.parseRoman('IVXLCDM')).to.equal(334);
            });
            it('should accept lower-case strings', function () {
                expect(formatter.parseRoman('mdclxvi')).to.equal(1666);
                expect(formatter.parseRoman('mDcLxVi')).to.equal(1666);
                expect(formatter.parseRoman('MdClXvI')).to.equal(1666);
            });
            it('should return null for invalid parameter', function () {
                expect(formatter.parseRoman('abc')).to.equal(null);
                expect(formatter.parseRoman('I I')).to.equal(null);
                expect(formatter.parseRoman('42')).to.equal(null);
            });
        });

        describe('method "formatStandardNumber"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('formatStandardNumber');
            });
            it('should format numbers to decimal notation', function () {
                expect(formatter.formatStandardNumber(0, 7)).to.equal('0');
                expect(formatter.formatStandardNumber(1, 7)).to.equal('1');
                expect(formatter.formatStandardNumber(-1, 7)).to.equal('-1');
                expect(formatter.formatStandardNumber(12, 7)).to.equal('12');
                expect(formatter.formatStandardNumber(-12, 7)).to.equal('-12');
                expect(formatter.formatStandardNumber(123, 7)).to.equal('123');
                expect(formatter.formatStandardNumber(-123, 7)).to.equal('-123');
                expect(formatter.formatStandardNumber(1230, 7)).to.equal('1230');
                expect(formatter.formatStandardNumber(-1230, 7)).to.equal('-1230');
                expect(formatter.formatStandardNumber(12300, 7)).to.equal('12300');
                expect(formatter.formatStandardNumber(-12300, 7)).to.equal('-12300');
                expect(formatter.formatStandardNumber(123000, 7)).to.equal('123000');
                expect(formatter.formatStandardNumber(-123000, 7)).to.equal('-123000');
                expect(formatter.formatStandardNumber(1230000, 7)).to.equal('1230000');
                expect(formatter.formatStandardNumber(-1230000, 7)).to.equal('-1230000');
                expect(formatter.formatStandardNumber(1e10, 20)).to.equal('10000000000');
                expect(formatter.formatStandardNumber(-1e10, 20)).to.equal('-10000000000');
                expect(formatter.formatStandardNumber(9.87654321e307, 310)).to.match(/^9876543210{299}$/);
                expect(formatter.formatStandardNumber(-9.87654321e307, 310)).to.match(/^-9876543210{299}$/);
                expect(formatter.formatStandardNumber(12345.6, 7)).to.equal('12345,6');
                expect(formatter.formatStandardNumber(-12345.6, 7)).to.equal('-12345,6');
                expect(formatter.formatStandardNumber(1234.56, 7)).to.equal('1234,56');
                expect(formatter.formatStandardNumber(-1234.56, 7)).to.equal('-1234,56');
                expect(formatter.formatStandardNumber(123.456, 7)).to.equal('123,456');
                expect(formatter.formatStandardNumber(-123.456, 7)).to.equal('-123,456');
                expect(formatter.formatStandardNumber(12.3456, 7)).to.equal('12,3456');
                expect(formatter.formatStandardNumber(-12.3456, 7)).to.equal('-12,3456');
                expect(formatter.formatStandardNumber(1.23456, 7)).to.equal('1,23456');
                expect(formatter.formatStandardNumber(-1.23456, 7)).to.equal('-1,23456');
                expect(formatter.formatStandardNumber(0.123, 7)).to.equal('0,123');
                expect(formatter.formatStandardNumber(-0.123, 7)).to.equal('-0,123');
                expect(formatter.formatStandardNumber(0.0123, 7)).to.equal('0,0123');
                expect(formatter.formatStandardNumber(-0.0123, 7)).to.equal('-0,0123');
                expect(formatter.formatStandardNumber(0.00123, 7)).to.equal('0,00123');
                expect(formatter.formatStandardNumber(-0.00123, 7)).to.equal('-0,00123');
            });
            it('should round fractional part in decimal notation', function () {
                expect(formatter.formatStandardNumber(0.12345, 6)).to.equal('0,1235');
                expect(formatter.formatStandardNumber(-0.12345, 6)).to.equal('-0,1235');
                expect(formatter.formatStandardNumber(0.12345, 5)).to.equal('0,123');
                expect(formatter.formatStandardNumber(-0.12345, 5)).to.equal('-0,123');
                expect(formatter.formatStandardNumber(0.12345, 4)).to.equal('0,12');
                expect(formatter.formatStandardNumber(-0.12345, 4)).to.equal('-0,12');
                expect(formatter.formatStandardNumber(0.12345, 3)).to.equal('0,1');
                expect(formatter.formatStandardNumber(-0.12345, 3)).to.equal('-0,1');
                expect(formatter.formatStandardNumber(0.12345, 2)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.12345, 2)).to.equal('-0');
                expect(formatter.formatStandardNumber(0.12345, 1)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.12345, 1)).to.equal('-0');
                expect(formatter.formatStandardNumber(123.456, 6)).to.equal('123,46');
                expect(formatter.formatStandardNumber(-123.456, 6)).to.equal('-123,46');
                expect(formatter.formatStandardNumber(123.456, 5)).to.equal('123,5');
                expect(formatter.formatStandardNumber(-123.456, 5)).to.equal('-123,5');
                expect(formatter.formatStandardNumber(1234.56, 5)).to.equal('1235');
                expect(formatter.formatStandardNumber(-1234.56, 5)).to.equal('-1235');
                expect(formatter.formatStandardNumber(1234.56, 4)).to.equal('1235');
                expect(formatter.formatStandardNumber(-1234.56, 4)).to.equal('-1235');
                expect(formatter.formatStandardNumber(9999.99, 6)).to.equal('10000');
                expect(formatter.formatStandardNumber(-9999.99, 6)).to.equal('-10000');
                expect(formatter.formatStandardNumber(8.9999, 1)).to.equal('9');
                expect(formatter.formatStandardNumber(-8.9999, 1)).to.equal('-9');
                expect(formatter.formatStandardNumber(0.9999, 1)).to.equal('1');
                expect(formatter.formatStandardNumber(-0.9999, 1)).to.equal('-1');
                expect(formatter.formatStandardNumber(0.09999, 1)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.09999, 1)).to.equal('-0');
            });
            it('should format numbers to scientific notation', function () {
                expect(formatter.formatStandardNumber(3450000000, 10)).to.equal('3450000000');
                expect(formatter.formatStandardNumber(-3450000000, 10)).to.equal('-3450000000');
                expect(formatter.formatStandardNumber(3450000000, 9)).to.equal('3,45E+09');
                expect(formatter.formatStandardNumber(-3450000000, 9)).to.equal('-3,45E+09');
                expect(formatter.formatStandardNumber(3450000000, 8)).to.equal('3,45E+09');
                expect(formatter.formatStandardNumber(-3450000000, 8)).to.equal('-3,45E+09');
                expect(formatter.formatStandardNumber(3450000000, 7)).to.equal('3,5E+09');
                expect(formatter.formatStandardNumber(-3450000000, 7)).to.equal('-3,5E+09');
                expect(formatter.formatStandardNumber(3450000000, 6)).to.equal('3E+09');
                expect(formatter.formatStandardNumber(-3450000000, 6)).to.equal('-3E+09');
                expect(formatter.formatStandardNumber(3450000000, 5)).to.equal('3E+09');
                expect(formatter.formatStandardNumber(-3450000000, 5)).to.equal('-3E+09');
                expect(formatter.formatStandardNumber(9876000000, 5)).to.equal('1E+10');
                expect(formatter.formatStandardNumber(-9876000000, 5)).to.equal('-1E+10');
                expect(formatter.formatStandardNumber(0.00003456, 10)).to.equal('0,00003456');
                expect(formatter.formatStandardNumber(-0.00003456, 10)).to.equal('-0,00003456');
                expect(formatter.formatStandardNumber(0.00003456, 9)).to.equal('3,456E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 9)).to.equal('-3,456E-05');
                expect(formatter.formatStandardNumber(0.00003456, 8)).to.equal('3,46E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 8)).to.equal('-3,46E-05');
                expect(formatter.formatStandardNumber(0.00003456, 7)).to.equal('3,5E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 7)).to.equal('-3,5E-05');
                expect(formatter.formatStandardNumber(0.00003456, 6)).to.equal('3E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 6)).to.equal('-3E-05');
                expect(formatter.formatStandardNumber(0.00003456, 5)).to.equal('3E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 5)).to.equal('-3E-05');
                expect(formatter.formatStandardNumber(0.00003456, 4)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.00003456, 4)).to.equal('-0');
                expect(formatter.formatStandardNumber(0.00003456, 1)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.00003456, 1)).to.equal('-0');
                expect(formatter.formatStandardNumber(1e-100, 5)).to.equal('0');
                expect(formatter.formatStandardNumber(-1e-100, 5)).to.equal('-0');
            });
            it('should be able to round up to an exponent with more digits', function () {
                expect(formatter.formatStandardNumber(9.99e99, 8)).to.equal('9,99E+99');
                expect(formatter.formatStandardNumber(9.99e99, 7)).to.equal('1E+100');
            });
            it('should prefer decimal notation over scientific notation', function () {
                expect(formatter.formatStandardNumber(0.001, 5)).to.equal('0,001');
                expect(formatter.formatStandardNumber(10000, 5)).to.equal('10000');
            });
            it('should fail for large integral parts', function () {
                expect(formatter.formatStandardNumber(12345, 4)).to.equal(null);
                expect(formatter.formatStandardNumber(1234, 3)).to.equal(null);
                expect(formatter.formatStandardNumber(123, 2)).to.equal(null);
                expect(formatter.formatStandardNumber(12, 1)).to.equal(null);
                expect(formatter.formatStandardNumber(999.5, 3)).to.equal(null);
                expect(formatter.formatStandardNumber(99.5, 2)).to.equal(null);
                expect(formatter.formatStandardNumber(9.5, 1)).to.equal(null);
                expect(formatter.formatStandardNumber(9.49999999999, 1)).to.equal('9');
            });
            it('should format decimal notation for all powers of 10', function () {
                for (var i = 1; i <= 307; i += 1) {
                    expect(formatter.formatStandardNumber(Number('1e' + i), 310)).to.match(new RegExp('^10{' + i + '}$'));
                    expect(formatter.formatStandardNumber(Number('1e-' + i), 310)).to.match(new RegExp('^0,0{' + (i - 1) + '}1$'));
                }
            });
            it('should format scientific notation for all powers of 10', function () {
                for (var i = 21; i <= 307; i += 1) {
                    expect(formatter.formatStandardNumber(Number('1e' + i), 21)).to.match(new RegExp('^1E\\+' + i + '$'));
                    expect(formatter.formatStandardNumber(Number('1e-' + i), 21)).to.match(new RegExp('^1E-' + i + '$'));
                }
            });
            it('should format with a custom exponent length', function () {
                expect(formatter.formatStandardNumber(1e9, 8, { expLength: 1 })).to.equal('1E+9');
                expect(formatter.formatStandardNumber(1e9, 8, { expLength: 2 })).to.equal('1E+09');
                expect(formatter.formatStandardNumber(1e9, 8, { expLength: 3 })).to.equal('1E+009');
                expect(formatter.formatStandardNumber(1e9, 8, { expLength: 4 })).to.equal('1E+0009');
                expect(formatter.formatStandardNumber(1e19, 8, { expLength: 1 })).to.equal('1E+19');
                expect(formatter.formatStandardNumber(1e19, 8, { expLength: 2 })).to.equal('1E+19');
                expect(formatter.formatStandardNumber(1e19, 8, { expLength: 3 })).to.equal('1E+019');
                expect(formatter.formatStandardNumber(1e19, 8, { expLength: 4 })).to.equal('1E+0019');
                expect(formatter.formatStandardNumber(1e199, 8, { expLength: 1 })).to.equal('1E+199');
                expect(formatter.formatStandardNumber(1e199, 8, { expLength: 2 })).to.equal('1E+199');
                expect(formatter.formatStandardNumber(1e199, 8, { expLength: 3 })).to.equal('1E+199');
                expect(formatter.formatStandardNumber(1e199, 8, { expLength: 4 })).to.equal('1E+0199');
            });
            it('should use scientific notation for specific edge cases', function () {
                expect(formatter.formatStandardNumber(5e-100, 6)).to.equal('5E-100');
                expect(formatter.formatStandardNumber(5e-100, 5)).to.equal('1E-99');
                expect(formatter.formatStandardNumber(5e-10, 5)).to.equal('5E-10');
                expect(formatter.formatStandardNumber(5e-10, 4)).to.equal('0');
                expect(formatter.formatStandardNumber(5e-10, 4, { expLength: 1 })).to.equal('1E-9');
                expect(formatter.formatStandardNumber(5e-10, 3, { expLength: 1 })).to.equal('0');
            });
        });

        describe('method "formatStandardNumberToWidth"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('formatStandardNumberToWidth');
            });
            var font = new Font('times', 10, false, false);
            it('should format positive numbers in decimal notation and round to pixel width', function () {
                var w1 = font.getTextWidth('0,12345'), w2 = font.getTextWidth('0,1235'), w3 = font.getTextWidth('0,123'),
                    w4 = font.getTextWidth('0,12'), w5 = font.getTextWidth('0,1'), w6 = font.getTextWidth('0'),
                    n = 0.12345;
                expect(formatter.formatStandardNumberToWidth(0, 10, font, w1)).to.deep.equal({ text: '0', mant: '0', exp: null, width: w6, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1)).to.deep.equal({ text: '0,12345', mant: '0,12345', exp: null, width: w1, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1 - 0.1)).to.deep.equal({ text: '0,1235', mant: '0,1235', exp: null, width: w2, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2)).to.have.a.property('text', '0,1235');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2 - 0.1)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3 - 0.1)).to.have.a.property('text', '0,12');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4)).to.have.a.property('text', '0,12');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4 - 0.1)).to.have.a.property('text', '0,1');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5)).to.have.a.property('text', '0,1');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5 - 0.1)).to.have.a.property('text', '0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w6)).to.have.a.property('text', '0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w6 - 0.1)).to.equal(null);
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w1)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w2)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w3)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w4)).to.have.a.property('text', '0,12');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w5)).to.have.a.property('text', '0,1');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w6)).to.have.a.property('text', '0');
            });
            it('should format negative numbers in decimal notation and round to pixel width', function () {
                var w1 = font.getTextWidth('-0,12345'), w2 = font.getTextWidth('-0,1235'), w3 = font.getTextWidth('-0,123'),
                    w4 = font.getTextWidth('-0,12'), w5 = font.getTextWidth('-0,1'), w6 = font.getTextWidth('-0'),
                    n = -0.12345;
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1)).to.deep.equal({ text: '-0,12345', mant: '-0,12345', exp: null, width: w1, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1 - 0.1)).to.deep.equal({ text: '-0,1235', mant: '-0,1235', exp: null, width: w2, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2)).to.have.a.property('text', '-0,1235');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2 - 0.1)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3 - 0.1)).to.have.a.property('text', '-0,12');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4)).to.have.a.property('text', '-0,12');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4 - 0.1)).to.have.a.property('text', '-0,1');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5)).to.have.a.property('text', '-0,1');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5 - 0.1)).to.have.a.property('text', '-0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w6)).to.have.a.property('text', '-0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w6 - 0.1)).to.equal(null);
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w1)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w2)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w3)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w4)).to.have.a.property('text', '-0,12');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w5)).to.have.a.property('text', '-0,1');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w6)).to.have.a.property('text', '-0');
            });
            it('should format positive numbers in scientific notation and round to pixel width', function () {
                var w1 = font.getTextWidth('0,00000345'), w2 = font.getTextWidth('3,45E-06'),
                    w3 = font.getTextWidth('3,5E-06'), w4 = font.getTextWidth('3E-06'), w5 = font.getTextWidth('0'),
                    n = 0.00000345;
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1)).to.deep.equal({ text: '0,00000345', mant: '0,00000345', exp: null, width: w1, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1 - 0.1)).to.deep.equal({ text: '3,45E-06', mant: '3,45', exp: '-06', width: w2, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2)).to.have.a.property('text', '3,45E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2 - 0.1)).to.have.a.property('text', '3,5E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3)).to.have.a.property('text', '3,5E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3 - 0.1)).to.have.a.property('text', '3E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4)).to.have.a.property('text', '3E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4 - 0.1)).to.deep.equal({ text: '0', mant: '0', exp: null, width: w5, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5)).to.have.a.property('text', '0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5 - 0.1)).to.equal(null);
            });
            it('should format negative numbers in scientific notation and round to pixel width', function () {
                var w1 = font.getTextWidth('-0,00000345'), w2 = font.getTextWidth('-3,45E-06'),
                    w3 = font.getTextWidth('-3,5E-06'), w4 = font.getTextWidth('-3E-06'), w5 = font.getTextWidth('-0'),
                    n = -0.00000345;
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1)).to.deep.equal({ text: '-0,00000345', mant: '-0,00000345', exp: null, width: w1, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1 - 0.1)).to.deep.equal({ text: '-3,45E-06', mant: '-3,45', exp: '-06', width: w2, portions: null });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2)).to.have.a.property('text', '-3,45E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2 - 0.1)).to.have.a.property('text', '-3,5E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3)).to.have.a.property('text', '-3,5E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3 - 0.1)).to.have.a.property('text', '-3E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4)).to.have.a.property('text', '-3E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4 - 0.1)).to.have.a.property('text', '-0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5)).to.have.a.property('text', '-0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5 - 0.1)).to.equal(null);
            });
            it('should prefer scientific notation for large numbers not fitting into the pixel width', function () {
                var w1 = font.getTextWidth('10000000000'), w2 = font.getTextWidth('1E+10');
                expect(formatter.formatStandardNumberToWidth(1e10, 11, font, w1)).to.have.a.property('text', '10000000000');
                expect(formatter.formatStandardNumberToWidth(1e10, 11, font, w1 - 0.1)).to.have.a.property('text', '1E+10');
                expect(formatter.formatStandardNumberToWidth(1e10, 11, font, w2)).to.have.a.property('text', '1E+10');
                expect(formatter.formatStandardNumberToWidth(1e10, 11, font, w2 - 0.1)).to.equal(null);
            });
            it('should use scientific notation for specific edge cases', function () {
                expect(formatter.formatStandardNumberToWidth(5e-100, 11, font, font.getTextWidth('5E-100'))).to.have.a.property('text', '5E-100');
                expect(formatter.formatStandardNumberToWidth(5e-100, 11, font, font.getTextWidth('1E-99'))).to.have.a.property('text', '1E-99');
                expect(formatter.formatStandardNumberToWidth(5e-100, 11, font, font.getTextWidth('1E-99') - 0.1)).to.have.a.property('text', '0');
                expect(formatter.formatStandardNumberToWidth(5e-10, 11, font, font.getTextWidth('1E-9'), { expLength: 1 })).to.have.a.property('text', '1E-9');
                expect(formatter.formatStandardNumberToWidth(5e-10, 11, font, font.getTextWidth('1E-9'))).to.have.a.property('text', '0');
            });
        });

        describe('method "formatValue"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('formatValue');
            });

            function expectFormattedValue(formatCode, value, expOOX, expODF) {
                var parsedFormatOOX = Parser.parseFormatCode('ooxml', 'op', formatCode);
                expect(formatter.formatValue(parsedFormatOOX, value)).to.equal(expOOX);
                var parsedFormatODF = Parser.parseFormatCode('odf', 'op', formatCode);
                expect(formatter.formatValue(parsedFormatODF, value)).to.equal(_.isUndefined(expODF) ? expOOX : expODF);
            }

            // bug 48011: special LCIDs for system date and time formats
            it('should use the current long system formats for special LCIDs', function () {
                expectFormattedValue('[$-F800]M/D/Y', 36617.25, 'Samstag, 1. April 2000');
                expectFormattedValue('[$-F400]h:m:s', 36617.25, '06:00:00');
            });

            it('should use the special number format: ???-??-#-0', function () {
                expectFormattedValue('???-??-#-0', 123456789, '12345-67-8-9');
                expectFormattedValue('???-??-#-0', 123450000, '12345-00-0-0');
// TODO!
//                expectFormattedValue('???-??-#-0', 1, '     -  --0');
            });
            it('should use a special telephone number format', function () {
                expectFormattedValue('[<=9999999]###-####;(###) ###-####', 1234567, '123-4567');
                expectFormattedValue('[<=9999999]###-####;(###) ###-####', 1234567890123, '(123456) 789-0123');
            });

            // special ODF format codes
            it('should use quarter date format', function () {
                expectFormattedValue('Q', 36617, 'Q', 'Q2');
                expectFormattedValue('QQ', 36617, 'QQ', '2. Quartal');
                expectFormattedValue('QQQ', 36617, 'QQQ', '2. QuartalQ2');
            });
            it('should use weekday format for ODF', function () {
                expectFormattedValue('N', 36617, null, 'N');
                expectFormattedValue('NN', 36617, null, 'Sa');
                expectFormattedValue('NNN', 36617, null, 'Samstag');
// TODO!
//                expectFormattedValue('NNNN', 36617, 'NNNN', 'Saturday, ');
//                expectFormattedValue('NNNNN', 36617, 'NNNN', 'Saturday, N');
            });
            it('should use week number format for ODF', function () {
                expectFormattedValue('W', 36547, 'W', 'W');
                expectFormattedValue('WW', 36547, 'WW', '4');
                expectFormattedValue('WWW', 36547, 'WWW', '4W');
            });

            it('should handle number formats correctly', function () {
                expectFormattedValue('0.00', 9.99999999, '10,00');// Bug 41434

            });

            it('should handle fraction formats correctly', function () {

                function expectFormattedFraction(number, exp1, exp2, exp3) {
                    expectFormattedValue('# ?/?', number, exp1);
                    expectFormattedValue('# ??/??', number, exp2);
                    expectFormattedValue('# ???/???', number, exp3);
                }

                expectFormattedFraction(0,    '0    ', '0      ', '0        ');
                expectFormattedFraction(0.01, '0    ', '0      ', '   1/100');
                expectFormattedFraction(0.98, '1    ', ' 49/50',  '  49/50 ');
                expectFormattedFraction(0.99, '1    ', ' 98/99',  '  99/100');
                expectFormattedFraction(1.01, '1    ', '1      ', '1   1/100');
                expectFormattedFraction(1.02, '1    ', '1  1/50', '1   1/50 ');
                expectFormattedFraction(1.99, '2    ', '1 98/99', '1  99/100');
                expectFormattedFraction(2.75, '2 3/4', '2  3/4 ', '2   3/4  ');
            });
        });
    });

    // ========================================================================
});
