/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/model/clipboardhandlermixin',
    'io.ox/office/textframework/utils/position'
], function (AppHelper, ClipBoardHandlerMixin, Position) {

    'use strict';

    // class ClipBoardHandlerMixin =============================================

    describe('TextFramework class ClipBoardHandlerMixin', function () {

        // private helpers ----------------------------------------------------

        var firstParagraph = null,
            secondParagraph = null,
            firstParagraphInComment = null,
            commentLayer = null,
            firstParaTextBefore = 'Hello new world',
            firstParaTextAfter = 'Hello 12345 world',
            secondParaText = 'More text in paragraph',
            secondParaTextAfter = 'More 111222 behind in paragraph',
            textInComment = 'Text in comment',
            textInCommentAfter = 'Text 12345 comment',
            textInCommentAfter2 = 'Text 111222 behind comment',
            pasteText = '12345',
            clipboardOperations = null,
            clipboardOperations1 = [{ name: 'insertText', start: [0, 0], text: pasteText }],
            clipboardOperations2 = [
                { name: 'insertText', start: [0, 0], text: '111' },
                { name: 'insertRange', start: [0, 3], id: 'cmt0', type: 'comment', position: 'start' },
                { name: 'insertText', start: [0, 4], text: '222' },
                { name: 'insertRange', start: [0, 7], id: 'cmt0', type: 'comment', position: 'end' },
                { name: 'insertComment', start: [0, 8], id: 'cmt0', author: 'author1', uid: '', date: '2016-01-08T12:13:00Z' }, // defining uid causes problems
                { name: 'insertParagraph', attrs: { paragraph: { marginBottom: 0 } }, start: [0], target: 'cmt0' },
                { name: 'insertText', start: [0, 0], text: 'abcdef', target: 'cmt0' },
                { name: 'insertText', start: [0, 9], text: ' behind' }
            ],

            // the operations to be applied by the document model
            OPERATIONS = [
                { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 }, character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' }, paragraph: { lineHeight: { type: 'percent', value: 115 }, marginBottom: 352 } } },
                { name: 'insertParagraph', start: [0] },
                { name: 'insertText', text: firstParaTextBefore, start: [0, 0] },
                { name: 'insertParagraph', start: [1] },
                { name: 'insertText', text: secondParaText, start: [1, 0] }
            ];

        var model;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            model.getClipboardOperations = function () { return clipboardOperations; };
            model.setClipboardOperations = function (ops) { clipboardOperations = ops; };
            model.hasInternalClipboard = _.constant(true);
            model.checkSetClipboardPasteInProgress = _.constant(false);
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(ClipBoardHandlerMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "pasteInternalClipboard"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('pasteInternalClipboard');
            });
            it('should contain the correct text in the first paragraph before pasting', function () {
                firstParagraph = Position.getParagraphElement(model.getNode(), [0]);
                expect($(firstParagraph).text()).to.equal(firstParaTextBefore);
                expect(Position.getParagraphNodeLength(firstParagraph)).to.equal(firstParaTextBefore.length);
            });
            it('should contain the correct text in the first paragraph after pasting', function (done) {
                model.getSelection().setTextSelection([0, 6], [0, 9]); // selecting the word 'new'
                clipboardOperations = clipboardOperations1;
                model.pasteInternalClipboard().done(function () {
                    expect($(firstParagraph).text()).to.equal(firstParaTextAfter);
                    expect(Position.getParagraphNodeLength(firstParagraph)).to.equal(firstParaTextAfter.length);
                    done();
                });
            });
            it('should contain the correct text in the first paragraph of a comment node after pasting', function (done) {
                model.getSelection().setTextSelection([0, 6], [0, 11]); // selecting the word '12345'
                commentLayer = model.getCommentLayer();
                commentLayer.insertComment();
                expect(Position.getParagraphNodeLength(firstParagraph)).to.equal(firstParaTextAfter.length + 3); // 3 additional place holder and range marker nodes
                expect(commentLayer.getComments().length).to.equal(1);
                expect(model.getActiveTarget()).to.equal('cmt0');
                model.insertText(textInComment, [0, 0]);
                firstParagraphInComment = Position.getParagraphElement(model.getCurrentRootNode(model.getActiveTarget()), [0]);
                expect($(firstParagraphInComment).text()).to.equal(textInComment);
                model.getSelection().setTextSelection([0, 5], [0, 7]); // selecting the word 'in'
                clipboardOperations = clipboardOperations1;
                model.pasteInternalClipboard().done(function () {
                    expect($(firstParagraphInComment).text()).to.equal(textInCommentAfter);
                    done();
                });
            });
            it('should paste comment place holders and range markers into the main document', function (done) {
                commentLayer.deActivateCommentNode('cmt0');
                model.getSelection().setTextSelection([1, 5], [1, 9]); // selecting the word 'text' in second paragraph
                secondParagraph = Position.getParagraphElement(model.getNode(), [1]);
                expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParaText.length);
                clipboardOperations = clipboardOperations2;
                model.pasteInternalClipboard().done(function () {
                    expect($(secondParagraph).text()).to.equal(secondParaTextAfter);
                    expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParaTextAfter.length + 3); // 3 additional place holder and range marker nodes
                    expect(commentLayer.getComments().length).to.equal(2);
                    done();
                });
            });
            it('should have pasted a new comment node', function () {
                commentLayer.selectNextComment();
                expect(model.getActiveTarget()).to.equal('cmt0');
                commentLayer.selectNextComment();
                expect(model.getActiveTarget()).to.equal('cmt1');
                commentLayer.selectNextComment();
                expect(model.getActiveTarget()).to.equal('cmt0');
            });
            it('should not paste comment place holders and range markers into a comment', function (done) {
                model.getSelection().setTextSelection([0, 5], [0, 10]); // selecting the word '12345' in first paragraph of comment node
                clipboardOperations = clipboardOperations2;
                model.pasteInternalClipboard().done(function () {
                    expect(commentLayer.getComments().length).to.equal(2); // no new comment inserted by pasting
                    expect($(firstParagraphInComment).text()).to.equal(textInCommentAfter2);
                    expect(Position.getParagraphNodeLength(firstParagraphInComment)).to.equal(textInCommentAfter2.length); // no 3 additional place holder and range marker nodes
                    done();
                });
            });
        });

    });

    // ========================================================================
});
