/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/components/field/fieldmanager'
], function (AppHelper, FieldManager) {

    'use strict';

    // class FieldManager ===========================================================

    describe('Text class FieldManager', function () {

        it('should exist', function () {
            expect(FieldManager).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var FILENAMECONST = 'unnamed (1).docx';

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 } } },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertRange', start: [0, 0], type: 'field', position: 'start' },
            { name: 'insertComplexField', start: [0, 1], instruction: 'FILENAME' },
            { name: 'insertText', start: [0, 2], text: FILENAMECONST },
            { name: 'insertRange', start: [0, 18], type: 'field', position: 'end' }
        ];

        // the ODF operations to be applied by the document model
        var ODF_OPERATIONS = [
            { name: 'insertParagraph', start: [0] },
            { name: 'insertField', start: [0, 0], type: 'date', representation: '1/12/2016', attrs: { character: { field: { dateFormat: 'M/D/YYYY' } } } },
            { name: 'updateField', start: [0, 0], type: 'date', representation: '1/12/2016', attrs: { character: { field: { dateFormat: 'M/D/YYYY', 'text:fixed': 'true' } } } }
        ];

        var model, selection, fieldManager;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            app.getFullFileName = function () { return FILENAMECONST; };
            model = app.getModel();
            selection = model.getSelection();
            fieldManager = model.getFieldManager();
        });

        var odfModel, odfSelection, odfFieldManager;
        AppHelper.createTextApp('odf', ODF_OPERATIONS).done(function (app) {
            app.getFullFileName = function () { return FILENAMECONST; };
            odfModel = app.getModel();
            odfSelection = odfModel.getSelection();
            odfFieldManager = odfModel.getFieldManager();
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a FieldManager class instance', function () {
                expect(fieldManager).to.be.an['instanceof'](FieldManager);
                expect(odfFieldManager).to.be.an['instanceof'](FieldManager);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "fieldsAreInDocument"', function () {
            it('should exist', function () {
                expect(fieldManager).to.respondTo('fieldsAreInDocument');
            });
            it('should return that there are fields in document', function () {
                expect(fieldManager.fieldsAreInDocument()).to.equal(true);
            });

            it('should exist odf', function () {
                expect(odfFieldManager).to.respondTo('fieldsAreInDocument');
            });
            it('should return that there are odf fields in document', function () {
                expect(odfFieldManager.fieldsAreInDocument()).to.equal(true);
            });
        });

        describe('method "isComplexFieldId"', function () {
            it('should exist', function () {
                expect(fieldManager).to.respondTo('isComplexFieldId');
            });
            it('should return that asked id is id of complex field', function () {
                expect(fieldManager.isComplexFieldId('fld0')).to.equal(true);
            });
        });

        describe('method "getComplexField"', function () {
            it('should exist', function () {
                expect(fieldManager).to.respondTo('getComplexField');
            });
            it('should return complex field node', function () {
                expect(fieldManager.getComplexField('fld0')).to.be.an['instanceof']($)
                    .and.to.have.className('complexfield');
            });
        });

        describe('method "getSimpleField"', function () {
            it('should exist', function () {
                expect(odfFieldManager).to.respondTo('getSimpleField');
            });
            it('should return simple odf field node', function () {
                expect(odfFieldManager.getSimpleField('sf0')).to.be.an['instanceof']($)
                    .and.to.have.className('field');
            });
        });

        describe('method "isAutomaticDateField" for odf', function () {
            it('should return simple odf field node', function () {
                expect(odfFieldManager.isFixedSimpleField(odfFieldManager.getSimpleField('sf0'), true)).to.equal(true);
            });
        });

        describe('testing changind date field to automatic from fixed, for odf', function () {
            before(function () {
                odfSelection.setTextSelection(odfSelection.getFirstDocumentPosition());
                odfFieldManager.setDateFieldToAutoUpdate(null, true);
            });
            it('should return that odf field node is not fixed anymore', function () {
                expect(odfFieldManager.isFixedSimpleField(odfFieldManager.getSimpleField('sf0'), true)).to.equal(false);
            });
        });

        describe('method "isHighlightState"', function () {
            before(function () {
                selection.setTextSelection(selection.getFirstDocumentPosition());
            });
            it('should exist', function () {
                expect(fieldManager).to.respondTo('isHighlightState');
            });
            it('should return that complex field is highlighted', function () {
                expect(fieldManager.isHighlightState()).to.equal(true);
            });
        });

        describe('method "getSelectedFieldFormat"', function () {
            before(function () {
                selection.setTextSelection(selection.getFirstDocumentPosition());
            });
            it('should exist', function () {
                expect(fieldManager).to.respondTo('getSelectedFieldFormat');
            });
            it('should return default complex field format of Filename field', function () {
                expect(fieldManager.getSelectedFieldFormat()).to.equal('default');
            });
        });

        describe('method "updateFieldFormatting"', function () {
            before(function () {
                selection.setTextSelection(selection.getFirstDocumentPosition());
                fieldManager.updateFieldFormatting('Upper');
                selection.setTextSelection(selection.getFirstDocumentPosition());
            });
            it('should exist', function () {
                expect(fieldManager).to.respondTo('updateFieldFormatting');
            });
            it('should return new, changed complex field format', function () {
                expect(fieldManager.getSelectedFieldFormat()).to.equal('Upper');
            });
        });

        describe('method "dispatchInsertField"', function () {
            before(function () {
                selection.selectAll();
                model.deleteSelected();
                selection.setTextSelection(selection.getFirstDocumentPosition());
                fieldManager.dispatchInsertField('pagenumber', 'default');
                selection.setTextSelection(selection.getFirstDocumentPosition());
            });
            it('should exist', function () {
                expect(fieldManager).to.respondTo('dispatchInsertField');
            });
            it('should return newly inserted complex field with id fld1', function () {
                expect(fieldManager.getSelectedFieldFormat()).to.equal('default');
                expect(fieldManager.getSelectedFieldNode()).to.be.an['instanceof']($)
                    .and.to.have.className('complexfield');
                expect(fieldManager.getSelectedFieldNode().attr('data-field-id')).to.equal('fld1');
            });
        });

    });

    // ========================================================================
});
