/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/spreadsheet/model/numberformatter'
], function (AppHelper, LocaleData, NumberFormatter) {

    'use strict';

    // class NumberFormatter ==================================================

    describe('Spreadsheet class NumberFormatter', function () {

        it('should exist', function () {
            expect(NumberFormatter).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var docModel = null, numberFormatter = null;
        AppHelper.createSpreadsheetApp('ooxml').done(function (app) {
            docModel = app.getModel();
            numberFormatter = docModel.getNumberFormatter();
        });

        // public methods -----------------------------------------------------

        describe('method "parseFormattedValue"', function () {
            it('should exist', function () {
                expect(numberFormatter).to.respondTo('parseFormattedValue');
            });

            it('should return object with value and format identifier', function () {
                expect(numberFormatter.parseFormattedValue('hallo')).to.deep.equal({ value: 'hallo', format: 0 });
            });

            // NUMBERS --------------------------------
            it('should be able to detect normal numbers', function () {
                expect(numberFormatter.parseFormattedValue('1')).to.deep.equal({ value: 1, format: 0 });
                expect(numberFormatter.parseFormattedValue('1000')).to.deep.equal({ value: 1000, format: 0 });
                expect(numberFormatter.parseFormattedValue('1000' + LocaleData.DEC + '99')).to.deep.equal({ value: 1000.99, format: 0 });

                expect(numberFormatter.parseFormattedValue('1' + LocaleData.GROUP + '000')).to.deep.equal({ value: 1000, format: 3 });
                expect(numberFormatter.parseFormattedValue('1' + LocaleData.GROUP + '000' + LocaleData.GROUP + '000')).to.deep.equal({ value: 1000000, format: 3 });
                expect(numberFormatter.parseFormattedValue('10' + LocaleData.GROUP + '000')).to.deep.equal({ value: 10000, format: 3 });

                expect(numberFormatter.parseFormattedValue('1' + LocaleData.GROUP + '000' + LocaleData.DEC + '99')).to.deep.equal({ value: 1000.99, format: 4 });
                expect(numberFormatter.parseFormattedValue('10' + LocaleData.GROUP + '000' + LocaleData.DEC + '99')).to.deep.equal({ value: 10000.99, format: 4 });

                expect(numberFormatter.parseFormattedValue('(10' + LocaleData.GROUP + '000' + LocaleData.DEC + '99)')).to.deep.equal({ value: -10000.99, format: 4 });
                expect(numberFormatter.parseFormattedValue('(10000)')).to.deep.equal({ value: -10000, format: 0 });

                // Wrong format(s)
                expect(numberFormatter.parseFormattedValue('10' + LocaleData.DEC + '000' + LocaleData.GROUP + '99')).to.deep.equal({ value: '10,000.99', format: 0 });
            });

            // DATE -----------------------------------
            var date_1 = '2016-12-31',
                date_2 = moment().format(LocaleData.SHORT_DATE);

            it('should be able to detect date strings', function () {
                var text = date_1,
                    formattedValue = numberFormatter.parseFormattedValue(text);
                expect(formattedValue.format).to.equal(14);
            });
            it('should be able to detect date strings', function () {
                var text = date_2,
                    formattedValue = numberFormatter.parseFormattedValue(text);
                expect(formattedValue.format).to.equal(14);
            });

            // TIME -----------------------------------
            var time_1 = moment().format(LocaleData.SHORT_TIME),
                time_2 = moment().format(LocaleData.LONG_TIME),
                time_3 = '01:30 am',
                time_4 = '3 pm',
                time_5 = '3 p',
                time_6 = '12 am',
                time_7 = '12 a';

            it('should be able to detect short time strings', function () {
                var text = time_1,
                    formattedValue = numberFormatter.parseFormattedValue(text);
                expect(formattedValue.format).to.equal(20);
            });
            it('should be able to detect long time strings', function () {
                var text = time_2,
                    formattedValue = numberFormatter.parseFormattedValue(text);
                expect(formattedValue.format).to.equal(21);
            });
            it('should be able to detect am/pm time strings', function () {
                var formattedValue3 = numberFormatter.parseFormattedValue(time_3),
                    formattedValue4 = numberFormatter.parseFormattedValue(time_4),
                    formattedValue5 = numberFormatter.parseFormattedValue(time_5),
                    formattedValue6 = numberFormatter.parseFormattedValue(time_6),
                    formattedValue7 = numberFormatter.parseFormattedValue(time_7);
                expect(formattedValue3.format).to.equal(18);
                expect(formattedValue4.format).to.equal(18);
                expect(formattedValue5.format).to.equal(18);
                expect(formattedValue6.format).to.equal(18);
                expect(formattedValue7.format).to.equal(18);
            });

            // DATE / TIME -----------------------------------
            var date_time_1 = moment().format(LocaleData.SHORT_DATE + ' ' + LocaleData.SHORT_TIME);

            it('should be able to detect date/time strings', function () {
                var text = date_time_1,
                    formattedValue = numberFormatter.parseFormattedValue(text);
                expect(formattedValue.format).to.equal(22);
            });

            // CURRENCY -----------------------------------

            var CURRENCY_INT_CODE = '#,##0 [$\u20ac-407];[RED]-#,##0 [$\u20ac-407]';
            var CURRENCY_FLOAT_CODE = '#,##0.00 [$\u20ac-407];[RED]-#,##0.00 [$\u20ac-407]';

            it('should be able to detect normal currency strings', function () {
                expect(numberFormatter.parseFormattedValue('8 ' + LocaleData.CURRENCY)).to.deep.equal({ value: 8, format: CURRENCY_INT_CODE });
                expect(numberFormatter.parseFormattedValue('(8) ' + LocaleData.CURRENCY)).to.deep.equal({ value: -8, format: CURRENCY_INT_CODE });
                expect(numberFormatter.parseFormattedValue('(8 ' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: -8, format: CURRENCY_INT_CODE });

                // Formel-Parser ???
                // expect(numberFormatter.parseFormattedValue('-(8 ' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: -8, format: 42 });
                // expect(numberFormatter.parseFormattedValue('+(8 ' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: 8, format: 42 });
            });
            it('should be able to detect floating currency strings', function () {
                expect(numberFormatter.parseFormattedValue('8' + LocaleData.DEC + '46 ' + LocaleData.CURRENCY)).to.deep.equal({ value: 8.46, format: CURRENCY_FLOAT_CODE });
                expect(numberFormatter.parseFormattedValue('(8' + LocaleData.DEC + '46) ' + LocaleData.CURRENCY)).to.deep.equal({ value: -8.46, format: CURRENCY_FLOAT_CODE });

                expect(numberFormatter.parseFormattedValue('(8' + LocaleData.DEC + '46 ' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: -8.46, format: CURRENCY_FLOAT_CODE });

                // Formel-Parser ???
                // expect(numberFormatter.parseFormattedValue('-(8' + LocaleData.DEC + '46) ' + LocaleData.CURRENCY)).to.deep.equal({ value: -8.46, format: 44 });
                // expect(numberFormatter.parseFormattedValue('+(8' + LocaleData.DEC + '46) ' + LocaleData.CURRENCY)).to.deep.equal({ value: 8.46, format: 44 });
                // expect(numberFormatter.parseFormattedValue('-(8' + LocaleData.DEC + '46 ' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: -8.46, format: 44 });
                // expect(numberFormatter.parseFormattedValue('+(8' + LocaleData.DEC + '46 ' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: 8.46, format: 44 });
            });
            it('should be able to detect currency strings with leading symbol', function () {
                expect(numberFormatter.parseFormattedValue(LocaleData.CURRENCY + '8')).to.deep.equal({ value: 8, format: CURRENCY_INT_CODE });
                expect(numberFormatter.parseFormattedValue(LocaleData.CURRENCY + '(8)')).to.deep.equal({ value: -8, format: CURRENCY_INT_CODE });

                expect(numberFormatter.parseFormattedValue('(' + LocaleData.CURRENCY + '8)')).to.deep.equal({ value: -8, format: CURRENCY_INT_CODE });

                expect(numberFormatter.parseFormattedValue('(' + LocaleData.CURRENCY + '8' + LocaleData.DEC + '46)')).to.deep.equal({ value: -8.46, format: CURRENCY_FLOAT_CODE });

                expect(numberFormatter.parseFormattedValue(LocaleData.CURRENCY + '8')).to.deep.equal({ value: 8, format: CURRENCY_INT_CODE });

                // Formel-Parser ???
                // expect(numberFormatter.parseFormattedValue(LocaleData.CURRENCY + '-(8)')).to.deep.equal({ value: -8, format: 42 });
                // expect(numberFormatter.parseFormattedValue(LocaleData.CURRENCY + '+(8)')).to.deep.equal({ value: 8, format: 42 });
                // expect(numberFormatter.parseFormattedValue('-(' + LocaleData.CURRENCY + ' 8)')).to.deep.equal({ value: -8, format: 42 });
                // expect(numberFormatter.parseFormattedValue('+(' + LocaleData.CURRENCY + ' 8)')).to.deep.equal({ value: 8, format: 42 });
                // expect(numberFormatter.parseFormattedValue('-(' + LocaleData.CURRENCY + ' 8' + LocaleData.DEC + '46)')).to.deep.equal({ value: -8.46, format: 44 });
                // expect(numberFormatter.parseFormattedValue('+(' + LocaleData.CURRENCY + ' 8' + LocaleData.DEC + '46)')).to.deep.equal({ value: 8.46, format: 44 });
            });
            it('should be able to detect floating currency strings with leading currency symbol', function () {
                expect(numberFormatter.parseFormattedValue(LocaleData.CURRENCY + '8' + LocaleData.DEC + '46')).to.deep.equal({ value: 8.46, format: CURRENCY_FLOAT_CODE });
            });
            it('should be able to detect currency strings with leading ISO currency code', function () {
                expect(numberFormatter.parseFormattedValue(LocaleData.ISO_CURRENCY + '8')).to.deep.equal({ value: 8, format: CURRENCY_INT_CODE });
                expect(numberFormatter.parseFormattedValue(LocaleData.ISO_CURRENCY + ' 8' + LocaleData.DEC + '46')).to.deep.equal({ value: 8.46, format: CURRENCY_FLOAT_CODE });
                expect(numberFormatter.parseFormattedValue(' 8' + LocaleData.ISO_CURRENCY)).to.deep.equal({ value: 8, format: CURRENCY_INT_CODE });
                expect(numberFormatter.parseFormattedValue(' 8' + LocaleData.DEC + '46' + LocaleData.ISO_CURRENCY)).to.deep.equal({ value: 8.46, format: CURRENCY_FLOAT_CODE });
            });

            // PERCENT -----------------------------------
            it('should be able to detect normal currency strings', function () {
                expect(numberFormatter.parseFormattedValue('8 %')).to.deep.equal({ value: 0.08, format: 9 });
                expect(numberFormatter.parseFormattedValue('(8) %')).to.deep.equal({ value: -0.08, format: 9 });
                expect(numberFormatter.parseFormattedValue('(8 %)')).to.deep.equal({ value: -0.08, format: 9 });
                expect(numberFormatter.parseFormattedValue('% 8')).to.deep.equal({ value: 0.08, format: 9 });
                expect(numberFormatter.parseFormattedValue('% (8)')).to.deep.equal({ value: -0.08, format: 9 });
                expect(numberFormatter.parseFormattedValue('(% 8)')).to.deep.equal({ value: -0.08, format: 9 });

                expect(numberFormatter.parseFormattedValue('%-(8)')).to.deep.equal({ value: '%-(8)', format: 0 });
                expect(numberFormatter.parseFormattedValue('%+(8)')).to.deep.equal({ value: '%+(8)', format: 0 });

                // Formel-Parser ???
                // expect(numberFormatter.parseFormattedValue('-(8 %)')).to.deep.equal({ value: -0.08, format: 9 });
                // expect(numberFormatter.parseFormattedValue('+(8 %)')).to.deep.equal({ value: 0.08, format: 9 });
                // expect(numberFormatter.parseFormattedValue('-(% 8)')).to.deep.equal({ value: -0.08, format: 9 });
                // expect(numberFormatter.parseFormattedValue('+(% 8)')).to.deep.equal({ value: 0.08, format: 9 });
            });

            // FRACTIONS -----------------------------------
            it('should be able to detect fractions', function () {
                expect(numberFormatter.parseFormattedValue('1 1/2')).to.deep.equal({ value: 1.5, format: 12 });
                expect(numberFormatter.parseFormattedValue('(1 1/2)')).to.deep.equal({ value: -1.5, format: 12 });

                expect(numberFormatter.parseFormattedValue('(-1 1/2)')).to.deep.equal({ value: '(-1 1/2)', format: 0 });
                expect(numberFormatter.parseFormattedValue('-(1/2)')).to.deep.equal({ value: '-(1/2)', format: 0 });
                expect(numberFormatter.parseFormattedValue('(-1/2)')).to.deep.equal({ value: '(-1/2)', format: 0 });

                expect(numberFormatter.parseFormattedValue('1 1/2 %')).to.deep.equal({ value: 0.015, format: 10 });
                expect(numberFormatter.parseFormattedValue('(1 1/2) %')).to.deep.equal({ value: -0.015, format: 10 });
                expect(numberFormatter.parseFormattedValue('(1 1/2%)')).to.deep.equal({ value: -0.015, format: 10 });

                expect(numberFormatter.parseFormattedValue('1 1/2 ' + LocaleData.CURRENCY)).to.deep.equal({ value: 1.5, format: CURRENCY_FLOAT_CODE });
                expect(numberFormatter.parseFormattedValue('(1 1/2) ' + LocaleData.CURRENCY)).to.deep.equal({ value: -1.5, format: CURRENCY_FLOAT_CODE });
                expect(numberFormatter.parseFormattedValue('(1 1/2' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: -1.5, format: CURRENCY_FLOAT_CODE });

                // Formel-Parser ???
                // expect(numberFormatter.parseFormattedValue('-(1 1/2)')).to.deep.equal({ value: -1.5, format: 12 });
                // expect(numberFormatter.parseFormattedValue('-(1 1/2) %')).to.deep.equal({ value: -0.015, format: 10 });
                // expect(numberFormatter.parseFormattedValue('+(1 1/2) %')).to.deep.equal({ value: 0.015, format: 10 });
                // expect(numberFormatter.parseFormattedValue('-(1 1/2%)')).to.deep.equal({ value: -0.015, format: 10 });
                // expect(numberFormatter.parseFormattedValue('+(1 1/2%)')).to.deep.equal({ value: 0.015, format: 10 });
                // expect(numberFormatter.parseFormattedValue('-(1 1/2) ' + LocaleData.CURRENCY)).to.deep.equal({ value: -1.5, format: 44 });
                // expect(numberFormatter.parseFormattedValue('+(1 1/2) ' + LocaleData.CURRENCY)).to.deep.equal({ value: 1.5, format: 44 });
                // expect(numberFormatter.parseFormattedValue('-(1 1/2' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: -1.5, format: 44 });
                // expect(numberFormatter.parseFormattedValue('+(1 1/2' + LocaleData.CURRENCY + ')')).to.deep.equal({ value: 1.5, format: 44 });
            });

            // SIENTIFIC --------------
            it('should be able to detect fractions', function () {
                expect(numberFormatter.parseFormattedValue('42,5e-03')).to.deep.equal({ value: 0.0425, format: 11 });
                expect(numberFormatter.parseFormattedValue('(42,5e-03)')).to.deep.equal({ value: -0.0425, format: 11 });
                expect(numberFormatter.parseFormattedValue('(42,5e-01)%')).to.deep.equal({ value: -0.0425, format: 10 });
                expect(numberFormatter.parseFormattedValue('(42,5e-03)\u20ac')).to.deep.equal({ value: -0.0425, format: CURRENCY_FLOAT_CODE });

                // Formel-Parser ???
                // expect(numberFormatter.parseFormattedValue('-(42,5e-03)')).to.deep.equal({ value: -0.0425, format: 11 });
                // expect(numberFormatter.parseFormattedValue('+(42,5e-03)')).to.deep.equal({ value: 0.0425, format: 11 });
            });
        });

        describe('method "formatValue"', function () {
            it('should exist', function () {
                expect(numberFormatter).to.respondTo('formatValue');
            });
        });
    });

    // ========================================================================
});
