/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/operationcontext',
    'io.ox/office/spreadsheet/model/hyperlinkcollection'
], function (AppHelper, SheetHelper, RangeArray, OperationContext, HyperlinkCollection) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;
    var ra = SheetHelper.ra;

    // class HyperlinkCollection ==============================================

    describe('Spreadsheet class HyperlinkCollection', function () {

        it('should exist', function () {
            expect(HyperlinkCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' }
        ];

        // initialize test document
        var docModel = null, sheetModel = null;
        var hyperlinkCollection = null, cellCollection = null, mergeCollection = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel = docModel.getSheetModel(0);
            hyperlinkCollection = sheetModel.getHyperlinkCollection();
            cellCollection = sheetModel.getCellCollection();
            mergeCollection = sheetModel.getMergeCollection();
        });

        // creates a new operation context
        function op(json) {
            return new OperationContext(docModel, json);
        }

        function linkOp(range, url) {
            var json = range.toJSON();
            json.url = url;
            return op(json);
        }

        function getLinkRanges(collection) {
            var linkModels = collection._getCloneData().linkModels;
            var ranges = RangeArray.map(linkModels, function (linkModel) {
                var range = linkModel.range.clone();
                range.url = linkModel.url;
                return range;
            });
            return ranges.sort();
        }

        // private methods ----------------------------------------------------

        describe('method "_getCloneData"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('_getCloneData');
            });
            it('should return the current link models', function () {
                var cloneData = hyperlinkCollection._getCloneData();
                expect(cloneData).to.have.a.property('linkModels');
                expect(cloneData.linkModels.length).to.equal(0);
            });
        });

        // operation implementations ------------------------------------------

        describe('method "applyInsertHyperlinkOperation"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('applyInsertHyperlinkOperation');
            });
            it('should execute an "insertHyperlink" operation', function () {
                var collection = new HyperlinkCollection(sheetModel);
                var context = linkOp(r('A1:C3'), 'http://www.example.org');
                collection.applyInsertHyperlinkOperation(context);
                var ranges = getLinkRanges(collection);
                expect(ranges).to.stringifyTo('A1:C3');
                expect(ranges[0].url).to.equal('http://www.example.org');
                collection.destroy();
            });
        });

        describe('method "applyDeleteHyperlinkOperation"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('applyDeleteHyperlinkOperation');
            });
            it('should execute a "deleteHyperlink" operation', function () {
                var collection = new HyperlinkCollection(sheetModel);
                var context = linkOp(r('A1:C3'), 'http://www.example.org');
                expect(getLinkRanges(collection).length).to.equal(0);
                collection.applyInsertHyperlinkOperation(context);
                expect(getLinkRanges(collection).length).to.equal(1);
                collection.applyDeleteHyperlinkOperation(context);
                expect(getLinkRanges(collection).length).to.equal(0);
                collection.destroy();
            });
        });

        describe('method "applyCopySheetOperation"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('applyCopySheetOperation');
            });
            it('should clone the existing hyperlinks from the given collection', function () {
                var collection1 = new HyperlinkCollection(sheetModel);
                var context = linkOp(r('A1:C3'), 'http://www.example.org');
                collection1.applyInsertHyperlinkOperation(context);
                var sheetModel2 = docModel.getSheetModel(1);
                var collection2 = new HyperlinkCollection(sheetModel2);
                collection2.applyCopySheetOperation(op({}), collection1);
                var ranges = getLinkRanges(collection2);
                expect(ranges).to.stringifyTo('A1:C3');
                expect(ranges[0].url).to.equal('http://www.example.org');
                collection2.destroy();
                collection1.destroy();
            });
        });

        // public methods -----------------------------------------------------

        describe('method "rangeOverlapsHyperlinks"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('rangeOverlapsHyperlinks');
            });
            it('should return true, if the given range overlaps a hyperlink range', function () {
                var collection = new HyperlinkCollection(sheetModel);
                var context = linkOp(r('A1:C3'), 'http://www.example.org');
                expect(collection.rangeOverlapsHyperlinks(r('B2:F9'))).to.equal(false);
                collection.applyInsertHyperlinkOperation(context);
                expect(collection.rangeOverlapsHyperlinks(r('B2:F9'))).to.equal(true);
                collection.applyDeleteHyperlinkOperation(context);
                expect(collection.rangeOverlapsHyperlinks(r('B2:F9'))).to.equal(false);
                collection.destroy();
            });
        });

        describe('method "getCellURL"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('getCellURL');
            });
            it('should return the correct url on the basis of an address', function () {
                var collection = new HyperlinkCollection(sheetModel);
                expect(collection.getCellURL(a('B3'))).to.equal(null);
                var context = linkOp(r('A1:C3'), 'http://www.example.org');
                collection.applyInsertHyperlinkOperation(context);
                expect(collection.getCellURL(a('B3'))).to.equal('http://www.example.org');
                expect(collection.getCellURL(a('Z99'))).to.equal(null);
            });
        });

        // operation generators -----------------------------------------------

        describe('method "generateHyperlinkOperations"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('generateHyperlinkOperations');
            });

            it('should insert hyperlinks with correct splitting', function (done) {
                var generator1 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise1 = hyperlinkCollection.generateHyperlinkOperations(generator1, ra('A1:C3 E1:G4'), 'http://www.example.org');
                promise1.always(function () {
                    expect(promise1.state()).to.equal('resolved');

                    var ranges1 = getLinkRanges(hyperlinkCollection);
                    expect(ranges1).to.stringifyTo('A1:C3,E1:G4');
                    expect(ranges1[0].url).to.equal('http://www.example.org');
                    expect(ranges1[1].url).to.equal('http://www.example.org');

                    var generator2 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = hyperlinkCollection.generateHyperlinkOperations(generator2, r('B2:F6'), 'http://www.test.de');
                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        var ranges2 = getLinkRanges(hyperlinkCollection);
                        expect(ranges2).to.stringifyTo('A1:C1,E1:G1,A2:A3,B2:F6,G2:G4');
                        expect(ranges2[0].url).to.equal('http://www.example.org');
                        expect(ranges2[1].url).to.equal('http://www.example.org');
                        expect(ranges2[2].url).to.equal('http://www.example.org');
                        expect(ranges2[3].url).to.equal('http://www.test.de');
                        expect(ranges2[4].url).to.equal('http://www.example.org');

                        expect(docModel.applyOperations(generator2, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection).length).to.equal(2);
                        expect(docModel.applyOperations(generator1, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection).length).to.equal(0);
                        done();
                    });
                });
            });
        });

        describe('method "generateAutoFillOperations"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('generateAutoFillOperations');
            });

            it('should create the correct hyperlinks on autofill', function (done) {
                var generator1 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise1 = hyperlinkCollection.generateHyperlinkOperations(generator1, r('A1:B2'), 'http://www.example.org');
                promise1.always(function () {
                    expect(promise1.state()).to.equal('resolved');

                    var generator2 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = hyperlinkCollection.generateAutoFillOperations(generator2, r('A1:B4'), r('A5:B10'), true, false);
                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        var ranges = getLinkRanges(hyperlinkCollection);
                        expect(ranges).to.stringifyTo('A1:B2,A5:B6,A9:B10');
                        expect(ranges[0].url).to.equal('http://www.example.org');
                        expect(ranges[1].url).to.equal('http://www.example.org');
                        expect(ranges[2].url).to.equal('http://www.example.org');

                        expect(docModel.applyOperations(generator2, { undo: true })).to.equal(true);
                        expect(docModel.applyOperations(generator1, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection).length).to.equal(0);
                        done();
                    });
                });
            });
        });

        describe('method "generateMoveCellsOperations"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('generateMoveCellsOperations');
            });

            it('should generate correct moveCells operations (insert column)', function (done) {
                var generator1 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise1 = hyperlinkCollection.generateHyperlinkOperations(generator1, ra('A1:B2 A5:A6 C5:D6'), 'http://www.example.org');
                promise1.always(function () {
                    expect(promise1.state()).to.equal('resolved');

                    var generator2 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = cellCollection.generateMoveCellsOperations(generator2, r('B1:B1048576'), 'right');
                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:C2,A5:A6,D5:E6');

                        expect(docModel.applyOperations(generator2, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:B2,A5:A6,C5:D6');

                        expect(docModel.applyOperations(generator1, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection).length).to.equal(0);
                        done();
                    });
                });
            });

            it('should generate correct moveCells operations (delete column)', function (done) {
                var generator1 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise1 = hyperlinkCollection.generateHyperlinkOperations(generator1, ra('A1:B2 B5:B6 D5:E6'), 'http://www.example.org');
                promise1.always(function () {
                    expect(promise1.state()).to.equal('resolved');

                    var generator2 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = cellCollection.generateMoveCellsOperations(generator2, r('B1:B1048576'), 'left');
                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:A2,C5:D6');

                        expect(docModel.applyOperations(generator2, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:B2,B5:B6,D5:E6');

                        expect(docModel.applyOperations(generator1, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection).length).to.equal(0);
                        done();
                    });
                });
            });
        });

        describe('method "generateMergeCellsOperations"', function () {
            it('should exist', function () {
                expect(hyperlinkCollection).to.respondTo('generateMergeCellsOperations');
            });

            it('should generate correct merge cells operations', function (done) {
                var generator1 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise1 = hyperlinkCollection.generateHyperlinkOperations(generator1, ra('A1:B2 A5:B6 A9:B10'), 'http://www.example.org');
                promise1.always(function () {
                    expect(promise1.state()).to.equal('resolved');

                    var generator2 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = mergeCollection.generateMergeCellsOperations(generator2, r('B2:F10'), 'merge');
                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:B1,A2:A2,B2:F10,A5:A6,A9:A10');

                        expect(docModel.applyOperations(generator2, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:B2,A5:B6,A9:B10');

                        expect(docModel.applyOperations(generator1, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection).length).to.equal(0);
                        done();
                    });
                });
            });

            it('should generate correct merge cells operations #2', function (done) {
                var generator1 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise1 = hyperlinkCollection.generateHyperlinkOperations(generator1, ra('A1:B2 A5:B6 A9:B10'), 'http://www.example.org');
                promise1.always(function () {
                    expect(promise1.state()).to.equal('resolved');

                    var generator2 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = mergeCollection.generateMergeCellsOperations(generator2, r('A2:F10'), 'merge');
                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:B1,A2:F10');

                        expect(docModel.applyOperations(generator2, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:B2,A5:B6,A9:B10');

                        expect(docModel.applyOperations(generator1, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection).length).to.equal(0);
                        done();
                    });
                });
            });

            it('should generate correct merge cells operations #3', function (done) {
                var generator1 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise1 = hyperlinkCollection.generateHyperlinkOperations(generator1, ra('A1:B2 A5:B6 A9:B10'), 'http://www.example.org');
                promise1.always(function () {
                    expect(promise1.state()).to.equal('resolved');

                    var generator2 = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = mergeCollection.generateMergeCellsOperations(generator2, r('B3:F10'), 'merge');
                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:B2,A5:A6,A9:A10');

                        expect(docModel.applyOperations(generator2, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection)).to.stringifyTo('A1:B2,A5:B6,A9:B10');

                        expect(docModel.applyOperations(generator1, { undo: true })).to.equal(true);
                        expect(getLinkRanges(hyperlinkCollection).length).to.equal(0);
                        done();
                    });
                });
            });
        });
    });

    // ========================================================================
});
