/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/model/formula/parser/desccollection'
], function (DescriptorCollection) {

    'use strict';

    // class DescriptorCollection =============================================

    describe('Spreadsheet class DescriptorCollection', function () {

        it('should exist', function () {
            expect(DescriptorCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var sourceMap = { key1: 'name1', key2: 'name3', key3: 'name2', key4: 'name4' };
        var collection = null;

        function createDescriptor(name, key) {
            if (key === 'key4') { return null; }
            var descriptor = DescriptorCollection.newDescriptor(key, 'native.' + name, 'local.' + name);
            if (key === 'key3') {
                descriptor.altNativeNames.push('alt.' + descriptor.nativeName);
                descriptor.altLocalNames.push('alt.' + descriptor.localName);
                descriptor.property = 'custom';
            }
            return descriptor;
        }

        // static methods -----------------------------------------------------

        describe('method "newDescriptor"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).itself.to.respondTo('newDescriptor');
            });
            it('should create a new descriptor', function () {
                var descriptor = DescriptorCollection.newDescriptor('key', 'native', 'local');
                expect(descriptor).to.be.an('object');
                expect(descriptor).to.have.a.property('key', 'key');
                expect(descriptor).to.have.a.property('nativeName', 'native');
                expect(descriptor).to.have.a.property('localName', 'local');
                expect(descriptor).to.have.a.property('altNativeNames').that.is.an('array').and.is.empty;
                expect(descriptor).to.have.a.property('altLocalNames').that.is.an('array').and.is.empty;
            });
        });

        describe('method "create"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).itself.to.respondTo('create');
            });
            it('should create a new collection', function () {
                var spy = sinon.spy(createDescriptor), context = {};
                collection = DescriptorCollection.create(sourceMap, spy, context);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 4);
                sinon.assert.calledWithExactly(spy, 'name1', 'key1', sourceMap);
                sinon.assert.calledWithExactly(spy, 'name2', 'key3', sourceMap);
                sinon.assert.calledWithExactly(spy, 'name3', 'key2', sourceMap);
                sinon.assert.calledWithExactly(spy, 'name4', 'key4', sourceMap);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "get"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).to.respondTo('get');
            });
            it('should return existing descriptors', function () {
                expect(collection.get('key1')).to.be.an('object').with.property('key', 'key1');
                expect(collection.get('key2')).to.be.an('object').with.property('key', 'key2');
                expect(collection.get('key3')).to.be.an('object').with.property('key', 'key3');
            });
            it('should return null for invalid keys', function () {
                expect(collection.get('')).to.equal(null);
                expect(collection.get('__invalid__')).to.equal(null);
                expect(collection.get('KEY1')).to.equal(null);
                expect(collection.get('key4')).to.equal(null);
            });
        });

        describe('method "getProperty"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).to.respondTo('getProperty');
            });
            it('should return existing properties', function () {
                expect(collection.getProperty('key1', 'key', 42)).to.equal('key1');
                expect(collection.getProperty('key2', 'nativeName', 42)).to.equal('native.name3');
                expect(collection.getProperty('key3', 'localName', 42)).to.equal('local.name2');
                expect(collection.getProperty('key3', 'property', 42)).to.equal('custom');
            });
            it('should return default value for missing properties', function () {
                expect(collection.getProperty('key1', 'property', 42)).to.equal(42);
                expect(collection.getProperty('__invalid__', 'key', 42)).to.equal(42);
                expect(collection.getProperty('KEY1', 'key', 42)).to.equal(42);
                expect(collection.getProperty('key4', 'key', 42)).to.equal(42);
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).to.respondTo('forEach');
            });
            it('should visit all descriptors', function () {
                var spy = sinon.spy(), context = {};
                expect(collection.forEach(spy, context)).to.equal(collection);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 3);
                sinon.assert.calledWithExactly(spy, collection.get('key1'), 'key1', collection);
                sinon.assert.calledWithExactly(spy, collection.get('key2'), 'key2', collection);
                sinon.assert.calledWithExactly(spy, collection.get('key3'), 'key3', collection);
            });
        });

        describe('method "getName"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).to.respondTo('getName');
            });
            it('should return native names', function () {
                expect(collection.getName('key1', false)).to.equal('native.name1');
                expect(collection.getName('key2', false)).to.equal('native.name3');
                expect(collection.getName('key3', false)).to.equal('native.name2');
            });
            it('should return localized names', function () {
                expect(collection.getName('key1', true)).to.equal('local.name1');
                expect(collection.getName('key2', true)).to.equal('local.name3');
                expect(collection.getName('key3', true)).to.equal('local.name2');
            });
            it('should return null for invalid keys', function () {
                expect(collection.getName('__invalid__', false)).to.equal(null);
                expect(collection.getName('__invalid__', true)).to.equal(null);
                expect(collection.getName('KEY1', false)).to.equal(null);
                expect(collection.getName('key4', true)).to.equal(null);
                expect(collection.getName('native.name1', false)).to.equal(null);
                expect(collection.getName('native.name1', true)).to.equal(null);
                expect(collection.getName('local.name1', false)).to.equal(null);
                expect(collection.getName('local.name1', true)).to.equal(null);
            });
        });

        describe('method "getKey"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).to.respondTo('getKey');
            });
            it('should return keys for native names', function () {
                expect(collection.getKey('native.name1', false)).to.equal('key1');
                expect(collection.getKey('NATIVE.NAME1', false)).to.equal('key1');
                expect(collection.getKey('native.name2', false)).to.equal('key3');
                expect(collection.getKey('native.name3', false)).to.equal('key2');
                expect(collection.getKey('alt.native.name2', false)).to.equal('key3');
                expect(collection.getKey('ALT.NATIVE.NAME2', false)).to.equal('key3');
            });
            it('should return keys for localized names', function () {
                expect(collection.getKey('local.name1', true)).to.equal('key1');
                expect(collection.getKey('LOCAL.NAME1', true)).to.equal('key1');
                expect(collection.getKey('local.name2', true)).to.equal('key3');
                expect(collection.getKey('local.name3', true)).to.equal('key2');
                expect(collection.getKey('alt.local.name2', true)).to.equal('key3');
                expect(collection.getKey('ALT.LOCAL.NAME2', true)).to.equal('key3');
            });
            it('should return null for invalid keys', function () {
                expect(collection.getKey('__invalid__', false)).to.equal(null);
                expect(collection.getKey('__invalid__', true)).to.equal(null);
                expect(collection.getKey('key1', false)).to.equal(null);
                expect(collection.getKey('key4', true)).to.equal(null);
                expect(collection.getKey('local.name1', false)).to.equal(null);
                expect(collection.getKey('native.name1', true)).to.equal(null);
            });
        });

        describe('method "getAllNames"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).to.respondTo('getAllNames');
            });
            it('should return native names', function () {
                expect(collection.getAllNames(false)).to.deep.equal(['alt.native.name2', 'native.name1', 'native.name2', 'native.name3']);
            });
            it('should return localized names', function () {
                expect(collection.getAllNames(true)).to.deep.equal(['alt.local.name2', 'local.name1', 'local.name2', 'local.name3']);
            });
        });

        describe('method "getAllPattern"', function () {
            it('should exist', function () {
                expect(DescriptorCollection).to.respondTo('getAllPattern');
            });
            it('should return pattern for native names', function () {
                expect(collection.getAllPattern(false)).to.equal('(alt\\.native\\.name2|native\\.name1|native\\.name2|native\\.name3)');
            });
            it('should return pattern for localized names', function () {
                expect(collection.getAllPattern(true)).to.equal('(alt\\.local\\.name2|local\\.name1|local\\.name2|local\\.name3)');
            });
        });
    });

    // ========================================================================
});
