/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/referencefuncs'
], function (AppHelper, SheetHelper, ReferenceFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var r3d = SheetHelper.r3d;
    var r3da = SheetHelper.r3da;
    var mat = SheetHelper.mat;

    // module ReferenceFuncs ==================================================

    describe('Spreadsheet module ReferenceFuncs', function () {

        // the operations to be applied at the test document
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'A2', contents: [
                [2, { v: 3, f: 'SUM(1,2)' }, 'abc'],
                [{ v: 2, f: 'A2', si: 0, sr: 'A3:B3' }, { v: 3, si: 0 }],
                [{ v: 4, f: 'A3:B3*2', mr: 'A4:B4' }, 6]
            ] },

            //horizontal data
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeCells', sheet: 1, start: 'A1', contents: [
                [4.14, 4.19, 5.17, 5.77, 6.39],
                [],
                ['Rot', 'Orange', 'Gelb', 'Gruen', 'Blau'],
                [],
                [4.14, null, null, 4.19, 5.17, 5.77, 6.39],
                [],
                ['Rot', null, null, 'Orange', 'Gelb', 'Gruen', 'Blau']
            ] },

            //vertical data
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'changeCells', sheet: 2, start: 'A1', contents: [
                [4.14, null, 'Rot'],
                [4.19, null, 'Orange'],
                [5.17, null, 'Gelb'],
                [5.77, null, 'Gruen'],
                [6.39, null, 'Blau']
            ] },

            //indirect data
            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet4' },
            { name: 'insertName', formula: 'Sheet4!$C$2:$D$3', label: 'name1' },
            { name: 'changeCells', sheet: 3, start: 'A1', contents: [
                ['B1',     null, 1333],
                ['B2',     null, 45],
                ['George', null, 10],
                [5,        null, 62]
            ] },

            //unsorted match
            { name: 'insertSheet', sheet: 4, sheetName: 'Sheet5' },
            { name: 'changeCells', sheet: 4, start: 'A1', contents: [
                ['a'],
                ['c'],
                ['b'],
                ['d'],
                [null],
                [null],
                [null],
                [null]
            ] }
        ];

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml', OPERATIONS);
        var moduleTester = SheetHelper.createFunctionModuleTester(ReferenceFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('CHOOSE', { toOperand: [1, 2, 3, 4, 5, 6, 7] }, function (CHOOSE) {
            var mat1 = mat([[1]]);
            var ranges = r3da('0:0!A1:A1');
            it('should return selected parameter', function () {
                expect(CHOOSE(1, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(2);
                expect(CHOOSE(2, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal('a');
                expect(CHOOSE(3, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(false);
                expect(CHOOSE(4, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(ErrorCode.DIV0);
                expect(CHOOSE(5, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(null);
                expect(CHOOSE(6, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.deep.equal(mat1);
                expect(CHOOSE(7, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(ranges);
            });
            it('should return #VALUE! for invalid index', function () {
                expect(CHOOSE(-1, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(ErrorCode.VALUE);
                expect(CHOOSE(0, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(ErrorCode.VALUE);
                expect(CHOOSE(8, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(ErrorCode.VALUE);
                expect(CHOOSE(9, 2, 'a', false, ErrorCode.DIV0, null, mat1, ranges)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('COLUMN', { targetAddress: 'E6' }, function (COLUMN) {
            it('should return the column index', function () {
                expect(COLUMN(r3d('0:0!B3:D5'))).to.equal(2);
                expect(COLUMN()).to.equal(5);
            });
        });

        moduleTester.testFunction('COLUMNS', { toOperand: 0 }, function (COLUMNS) {
            it('should return the width of the operand', function () {
                expect(COLUMNS(42)).to.equal(1);
                expect(COLUMNS(mat([[1, 2, 3], [4, 5, 6]]))).to.equal(3);
                expect(COLUMNS(r3d('0:0!B2:C4'))).to.equal(2);
            });
            it('should throw errors for complex ranges', function () {
                expect(COLUMNS(r3d('0:1!B2:C4'))).to.equal(ErrorCode.VALUE);
                expect(COLUMNS(r3da('0:0!B2:C4 0:0!B2:C4'))).to.equal(ErrorCode.REF);
            });
        });

        moduleTester.testFunction('FORMULATEXT', function (FORMULATEXT) {
            it('should return the formula expression for normal formula cells', function () {
                expect(FORMULATEXT(r3d('0:0!B2:B2'))).to.equal('=SUMME(1;2)');
            });
            it('should return the formula expression for shared formulas', function () {
                expect(FORMULATEXT(r3d('0:0!A3:A3'))).to.equal('=A2');
                expect(FORMULATEXT(r3d('0:0!B3:B3'))).to.equal('=B2');
            });
            it('should return the formula expression for matrix formulas', function () {
                expect(FORMULATEXT(r3d('0:0!A4:A4'))).to.equal('{=A3:B3*2}');
                expect(FORMULATEXT(r3d('0:0!B4:B4'))).to.equal('{=A3:B3*2}');
            });
            it('should return #N/A for simple cells', function () {
                expect(FORMULATEXT(r3d('0:0!A2:A2'))).to.equal(ErrorCode.NA);
                expect(FORMULATEXT(r3d('0:0!C2:C2'))).to.equal(ErrorCode.NA);
                expect(FORMULATEXT(r3d('0:0!IV99:IV99'))).to.equal(ErrorCode.NA);
            });
            it('should use the start address of a range', function () {
                expect(FORMULATEXT(r3d('0:0!B2:C3'))).to.equal('=SUMME(1;2)');
            });
        });

        moduleTester.testFunction('HLOOKUP', { refSheet: 1, targetAddress: 'A4', toOperand: [1] }, function (HLOOKUP) {

            var stringMatrix = mat([['Achsen', 'Getriebe', 'Schrauben'], [4, 4, 9], [5, 7, 10], [6, 8, 11]]);
            var numberMatrix = mat([[1, 2, 3], ['a', 'b', 'c'], ['d', 'e', 'f']]);

            it('should return the result of the horizontal search', function () {
                expect(HLOOKUP('Achsen', stringMatrix, 2, true)).to.equal(4);
                expect(HLOOKUP('Getriebe', stringMatrix, 3, false)).to.equal(7);
                expect(HLOOKUP('B', stringMatrix, 3, true)).to.equal(5);
                expect(HLOOKUP('Schrauben', stringMatrix, 4)).to.equal(11);

                expect(HLOOKUP(3, numberMatrix, 2, true)).to.equal('c');
                expect(HLOOKUP(2.5, numberMatrix, 2, true)).to.equal('b');
                expect(HLOOKUP(4, numberMatrix, 2, true)).to.equal('c');

                expect(HLOOKUP('A', stringMatrix, 3, true)).to.equal(ErrorCode.NA);
                expect(HLOOKUP('Achsen', stringMatrix, 0, true)).to.equal(ErrorCode.VALUE);
                expect(HLOOKUP('Achsen', stringMatrix, 100, true)).to.equal(ErrorCode.REF);
            });
            it('should return the result of the horizontal search with wildcards', function () {
                expect(HLOOKUP('Schra?ben', stringMatrix, 4)).to.equal(8);
                expect(HLOOKUP('Schra?ben', stringMatrix, 4, false)).to.equal(11);
                expect(HLOOKUP('G*e', stringMatrix, 4, false)).to.equal(8);
                expect(HLOOKUP('G?e', stringMatrix, 4, false)).to.equal(ErrorCode.NA);
            });

            it('should return the result of the horizontal search with references', function () {
                expect(HLOOKUP(4.19, r3d('1:1!A1:E3'), 3)).to.equal('Orange');
                expect(HLOOKUP(5.75, r3d('1:1!A1:E3'), 3)).to.equal('Gelb');
                expect(HLOOKUP(6.00, r3d('1:1!A1:E3'), 3)).to.equal('Gruen');
                expect(HLOOKUP(7.66, r3d('1:1!A1:E3'), 3)).to.equal('Blau');
            });
        });

        moduleTester.testFunction('INDEX', { refSheet: 1, targetAddress: 'Z26', toOperand: [0] }, function (INDEX) {
            it('should return the result of the INDEX on arrays/matrices', function () {
                expect(INDEX(mat([[1, 2], [3, 4]]), 0, 2)).to.deep.equal(mat([[2], [4]]));
            });
            it('should return the result of the INDEX on refs', function () {
                expect(INDEX(r3d('1:1!A1:B2'), 2, 2)).to.deep.equal(r3d('1:1!B2:B2'));
                expect(INDEX(r3d('1:1!A1:B2'), 2, 1)).to.deep.equal(r3d('1:1!A2:A2'));
            });
            it('should return the result of the INDEX on refs', function () {
                expect(INDEX(r3d('1:1!A1:C5'), 2, 3)).to.deep.equal(r3d('1:1!C2:C2'));
                expect(INDEX(r3d('1:1!A1:C5'), 5, 2)).to.deep.equal(r3d('1:1!B5:B5'));
                expect(INDEX(r3da('1:1!A1:C5 1:1!A7:C10'), 2, 2, 2)).to.deep.equal(r3d('1:1!B8:B8'));
                expect(INDEX(r3d('1:1!A1:C10'), 0, 3, 1)).to.deep.equal(r3d('1:1!C1:C10'));
            });
            it('should return the expected Error of the INDEX function', function () {
                expect(INDEX(mat([[1, 2], [3, 4]]), 0, 2, 2)).to.equal(ErrorCode.REF);
                expect(INDEX(1, 0, 2)).to.equal(ErrorCode.VALUE);

                expect(INDEX(r3da('1:1!A1:B2 1:1!A7:B8'), 3, 2, 2)).to.equal(ErrorCode.REF);
                expect(INDEX(r3da('1:1!A1:B2 1:1!A7:B8'), 2, 3, 2)).to.equal(ErrorCode.REF);
                expect(INDEX(r3da('1:1!A1:B2 1:1!A7:B8'), 2, 2, 3)).to.equal(ErrorCode.REF);
            });
        });

        moduleTester.testFunction('INDIRECT', { refSheet: 3, targetAddress: 'B2' }, function (INDIRECT) {
            it('should return the result for A1 notation', function () {
                expect(INDIRECT('A1')).to.deep.equal(r3d('3:3!A1:A1'));
                expect(INDIRECT('A2', true)).to.deep.equal(r3d('3:3!A2:A2'));
                expect(INDIRECT('A:A')).to.deep.equal(r3d('3:3!A1:A1048576'));
                expect(INDIRECT('1:1')).to.deep.equal(r3d('3:3!A1:XFD1'));
                expect(INDIRECT('Sheet2!A3')).to.deep.equal(r3d('1:1!A3:A3'));
                expect(INDIRECT('A2:B3')).to.deep.equal(r3d('3:3!A2:B3'));
                expect(INDIRECT('name1')).to.deep.equal(r3d('3:3!C2:D3'));
            });
            it('should return the result for R1C1 notation', function () {
                expect(INDIRECT('Z2S1', false)).to.deep.equal(r3d('3:3!A2:A2'));
                expect(INDIRECT('ZS', false)).to.deep.equal(r3d('3:3!B2:B2'));
                expect(INDIRECT('Z[1]S[-1]', false)).to.deep.equal(r3d('3:3!A3:A3'));
                expect(INDIRECT('Z1:Z2', false)).to.deep.equal(r3d('3:3!A1:XFD2'));
                expect(INDIRECT('Z', false)).to.deep.equal(r3d('3:3!A2:XFD2'));
                expect(INDIRECT('S1:S2', false)).to.deep.equal(r3d('3:3!A1:B1048576'));
                expect(INDIRECT('S', false)).to.deep.equal(r3d('3:3!B1:B1048576'));
                expect(INDIRECT('Sheet2!Z3S1', false)).to.deep.equal(r3d('1:1!A3:A3'));
                expect(INDIRECT('Z2S1:Z3S2', false)).to.deep.equal(r3d('3:3!A2:B3'));
                expect(INDIRECT('name1', false)).to.deep.equal(r3d('3:3!C2:D3'));
            });
            it('should throw #REF! error for invalid parameter', function () {
                expect(INDIRECT('1', true)).to.equal(ErrorCode.REF);
                expect(INDIRECT('1', false)).to.equal(ErrorCode.REF);
                expect(INDIRECT('A', true)).to.equal(ErrorCode.REF);
                expect(INDIRECT('A', false)).to.equal(ErrorCode.REF);
                expect(INDIRECT('', true)).to.equal(ErrorCode.REF);
                expect(INDIRECT('', false)).to.equal(ErrorCode.REF);
                expect(INDIRECT('Z2S1', true)).to.equal(ErrorCode.REF);
                expect(INDIRECT('A1', false)).to.equal(ErrorCode.REF);
                expect(INDIRECT('InvalidSheet!A1')).to.equal(ErrorCode.REF);
                expect(INDIRECT('Sheet1:Sheet2!A1')).to.equal(ErrorCode.REF);
            });
        });

        moduleTester.testFunction('LOOKUP', { refSheet: 1, targetAddress: 'A4', toOperand: [1, 2] }, function (LOOKUP) {

            var rateData = [4.14, 4.19, 5.17, 5.77, 6.39];
            var colorData = ['Rot', 'Orange', 'Gelb', 'Gruen', 'Blau'];

            var rateMatrix = mat([rateData]);
            var colorMatrix = mat([colorData]);
            var matrix = mat([rateData, colorData]);

            it('should return the result of the search', function () {
                expect(LOOKUP(4.19, rateMatrix, colorMatrix)).to.equal('Orange');
                expect(LOOKUP(5.75, rateMatrix, colorMatrix)).to.equal('Gelb');
                expect(LOOKUP(7.66, rateMatrix, colorMatrix)).to.equal('Blau');
                expect(LOOKUP(4.19, matrix)).to.equal('Orange');
                expect(LOOKUP(0, rateMatrix, colorMatrix)).to.equal(ErrorCode.NA);
            });

            it('should return the result of the search in unsorted matrix', function () {
                var unsorted = mat([[4.14, 4.19, 15.17, 5.77, 6.39], colorData]);
                expect(LOOKUP(7.66, unsorted)).to.equal('Orange');
            });

            var rateMatrix2 = mat([['rate', null, 4.14, 4.19, 5.17]]);
            var colorMatrix2 = mat([['color', null, 'Rot', 'Orange', 'Gelb']]);

            it('should return the result of the search with confusing types', function () {
                expect(LOOKUP(4.19, rateMatrix2, colorMatrix2)).to.equal('Orange');
                expect(LOOKUP(5.75, rateMatrix2, colorMatrix2)).to.equal('Gelb');
                expect(LOOKUP(7.66, rateMatrix2, colorMatrix2)).to.equal('Gelb');

                expect(LOOKUP(4.19, mat([['rate', null, 4.14, 4.19]]), mat([['color', null, 'Rot', 'Orange']]))).to.equal('Orange');
                expect(LOOKUP(4.19, mat([['rate', null, 4.14]]), mat([['color', null, 'Rot']]))).to.equal('Rot');

                expect(LOOKUP(4.19, mat([['rate', null]]), mat([['color', null]]))).to.equal(ErrorCode.NA);
                expect(LOOKUP(4.19, mat([['rate']]), mat([['color']]))).to.equal(ErrorCode.NA);
            });

            it('should return the result of the search with references', function () {
                expect(LOOKUP(4.19, r3d('1:1!A1:E1'), r3d('1:1!A3:E3'))).to.equal('Orange');
                expect(LOOKUP(5.75, r3d('1:1!A1:E1'), r3d('1:1!A3:E3'))).to.equal('Gelb');
                expect(LOOKUP(7.66, r3d('1:1!A1:E1'), r3d('1:1!A3:E3'))).to.equal('Blau');

                expect(LOOKUP(5.75, r3d('1:1!A1:E3'))).to.equal('Gelb');
            });

            it('should return the result of the search with INCOMPLETE references', function () {
                expect(LOOKUP(4.19, r3d('1:1!A1:E1'), r3d('1:1!A3:C3'))).to.equal('Orange');
                expect(LOOKUP(5.75, r3d('1:1!A1:E1'), r3d('1:1!A3:C3'))).to.equal('Gelb');
                expect(LOOKUP(7.66, r3d('1:1!A1:E1'), r3d('1:1!A3:C3'))).to.equal('Blau');

                expect(LOOKUP(4.19, r3d('1:1!A1:E1'), r3d('1:1!A3:A3'))).to.equal('Orange');
                expect(LOOKUP(5.75, r3d('1:1!A1:E1'), r3d('1:1!A3:A3'))).to.equal('Gelb');
                expect(LOOKUP(6.00, r3d('1:1!A1:E1'), r3d('1:1!A3:A3'))).to.equal('Gruen');
                expect(LOOKUP(7.66, r3d('1:1!A1:E1'), r3d('1:1!A3:A3'))).to.equal('Blau');
            });

            it('should return the result of the search with GAPS in the references', function () {
                expect(LOOKUP(4.19, r3d('1:1!A5:E5'), r3d('1:1!A7:C7'))).to.equal('Orange');
                expect(LOOKUP(5.75, r3d('1:1!A5:E5'), r3d('1:1!A7:C7'))).to.equal('Gelb');

                // expect(LOOKUP(7.66, r3d('1:1!A5:E5'), r3d('1:1!A7:C7'))).to.equal('Blau');
            });
        });

        moduleTester.testFunction('MATCH', { toOperand: [1] }, function (MATCH) {
            var count = mat([[25, 38, 40, 41]]);
            var countRevers = mat([[41, 40, 38, 25]]);

            it('should return the result of the match', function () {
                expect(MATCH(39, count, 1)).to.equal(2);
                expect(MATCH(41, count, 0)).to.equal(4);
                expect(MATCH(41, count, 1)).to.equal(4);
            });
            it('should throw for invalid parameters', function () {
                expect(MATCH(39, count, 0)).to.equal(ErrorCode.NA);
                expect(MATCH(39, count, -1)).to.equal(ErrorCode.NA);
                expect(MATCH(41, count, -1)).to.equal(ErrorCode.NA);
            });

            it('should return the result of the match with reverse order', function () {
                expect(MATCH(39, countRevers, -1)).to.equal(2);
                expect(MATCH(41, countRevers, 0)).to.equal(1);
                expect(MATCH(41, countRevers, -1)).to.equal(1);
            });

            it('should return the result of the last entry without match', function () {
                expect(MATCH('z', mat([['a'], ['c'], ['b'], ['d']]))).to.equal(4);

                expect(MATCH('z', r3d('4:4!A1:A4'))).to.equal(4);
                expect(MATCH('z', r3d('4:4!A1:A50'))).to.equal(4);
                expect(MATCH('z', r3d('4:4!A1:A50'), 1)).to.equal(4);

                expect(MATCH('zzzzzzzzzzzzzzzzzzzz', r3d('4:4!A1:A50'), 1)).to.equal(4);
            });

            it('match without solid order', function () {
                expect(MATCH('Bagels', mat([['[  BREAKFAST ]', 'haha', 'Bagels', 'Breakfast Burrito', 'Cold cereal']]))).to.equal(3);
                expect(MATCH('zzzzzzzzzzzzzzzzzzzz', mat([['[  BREAKFAST ]', null, 'Bagels', 'Breakfast Burrito', 'Cold cereal', 'Cream of wheat', 'Eggs and', 'French Toast', 'Oatmeal', 'Omelettes', 'Pancakes', 'Waffles', null, null, null]]))).to.equal(12);

                expect(MATCH('zzzzzzzzzzzzzzzzzzzz', mat([['[  BREAKFAST ]', null, 'Bagels', 'Breakfast Burrito', 'Cold cereal', null, null, 'Cream of wheat', 'Eggs and', 'French Toast', 'Oatmeal', 'Omelettes', 'Pancakes', 'Waffles', null, null, null]]))).to.equal(14);
            });

            it('match with alphabetic order', function () {
                expect(MATCH('Pants', mat([[0, 'Sweater', 'Jacket', 'Pants']]))).to.equal(4);
                expect(MATCH('Med', mat([[0, 'Small', 'Med', 'Large']]))).to.equal(3);

            });

        });

        moduleTester.testFunction('ROW', { targetAddress: 'E6' }, function (ROW) {
            it('should return the row index', function () {
                expect(ROW(r3d('0:0!B3:D5'))).to.equal(3);
                expect(ROW()).to.equal(6);
            });
        });

        moduleTester.testFunction('ROWS', { toOperand: 0 }, function (ROWS) {
            it('should return the width of the operand', function () {
                expect(ROWS(42)).to.equal(1);
                expect(ROWS(mat([[1, 2, 3], [4, 5, 6]]))).to.equal(2);
                expect(ROWS(r3d('0:0!B2:C4'))).to.equal(3);
            });
            it('should throw errors for complex ranges', function () {
                expect(ROWS(r3d('0:1!B2:C4'))).to.equal(ErrorCode.VALUE);
                expect(ROWS(r3da('0:0!B2:C4 0:0!B2:C4'))).to.equal(ErrorCode.REF);
            });
        });

        moduleTester.testFunction('SWITCH', { toOperand: [2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15] }, function (SWITCH) {
            var mat1 = mat([[1]]);
            var ranges = r3da('0:0!A1:A1');
            it('should return selected parameter', function () {
                expect(SWITCH(42, 42, 2, 43, 'a', 44, false, 45, ErrorCode.DIV0, 46, null, 47, mat1, 41, ranges)).to.equal(2);
                expect(SWITCH(43, 42, 2, 43, 'a', 44, false, 45, ErrorCode.DIV0, 46, null, 47, mat1, 41, ranges)).to.equal('a');
                expect(SWITCH(44, 42, 2, 43, 'a', 44, false, 45, ErrorCode.DIV0, 46, null, 47, mat1, 41, ranges)).to.equal(false);
                expect(SWITCH(45, 42, 2, 43, 'a', 44, false, 45, ErrorCode.DIV0, 46, null, 47, mat1, 41, ranges)).to.equal(ErrorCode.DIV0);
                expect(SWITCH(46, 42, 2, 43, 'a', 44, false, 45, ErrorCode.DIV0, 46, null, 47, mat1, 41, ranges)).to.equal(null);
                expect(SWITCH(47, 42, 2, 43, 'a', 44, false, 45, ErrorCode.DIV0, 46, null, 47, mat1, 41, ranges)).to.deep.equal(mat1);
                expect(SWITCH(41, 42, 2, 43, 'a', 44, false, 45, ErrorCode.DIV0, 46, null, 47, mat1, 41, ranges)).to.equal(ranges);
            });
            it('should match different data types', function () {
                expect(SWITCH('b', 'a', 2, 'b', ranges, true, mat1)).to.equal(ranges);
                expect(SWITCH('B', 'a', 2, 'b', ranges, true, mat1)).to.equal(ranges);
                expect(SWITCH(true, 'a', 2, 'b', ranges, true, mat1)).to.deep.equal(mat1);
            });
            it('should return default value', function () {
                expect(SWITCH('a', 1, 2, ranges)).to.equal(ranges);
            });
            it('should return #N/A for invalid index without default value', function () {
                expect(SWITCH(40, 42, 2, 43, 'a', 44, false, 45, ErrorCode.DIV0, 46, null, 47, mat1, 41, ranges)).to.equal(ErrorCode.NA);
                expect(SWITCH('1', 1, 2)).to.equal(ErrorCode.NA);
                expect(SWITCH(true, 1, 2)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('VLOOKUP', { refSheet: 2, targetAddress: 'D1', toOperand: [1] }, function (VLOOKUP) {

            var stringMatrix = mat([['Achsen', 4, 5, 6], ['Getriebe', 4, 7, 8], ['Schrauben', 9, 10, 11]]);
            var numberMatrix = mat([[1, 'a', 'd'], [2, 'b', 'e'], [3, 'c', 'f']]);

            it('should return the result of the vertical search', function () {
                expect(VLOOKUP('Achsen', stringMatrix, 2, true)).to.equal(4);
                expect(VLOOKUP('Getriebe', stringMatrix, 3, false)).to.equal(7);
                expect(VLOOKUP('B', stringMatrix, 3, true)).to.equal(5);
                expect(VLOOKUP('Schrauben', stringMatrix, 4)).to.equal(11);

                expect(VLOOKUP(3, numberMatrix, 2, true)).to.equal('c');
                expect(VLOOKUP(2.5, numberMatrix, 2, true)).to.equal('b');
                expect(VLOOKUP(4, numberMatrix, 2, true)).to.equal('c');

                expect(VLOOKUP('A', stringMatrix, 3, true)).to.equal(ErrorCode.NA);
                expect(VLOOKUP('Achsen', stringMatrix, 0, true)).to.equal(ErrorCode.VALUE);
                expect(VLOOKUP('Achsen', stringMatrix, 100, true)).to.equal(ErrorCode.REF);
            });

            it('should return the result of the vertical search with wildcards', function () {
                expect(VLOOKUP('Schra?ben', stringMatrix, 4)).to.equal(8);
                expect(VLOOKUP('Schra?ben', stringMatrix, 4, false)).to.equal(11);
                expect(VLOOKUP('G*e', stringMatrix, 4, false)).to.equal(8);
                expect(VLOOKUP('G?e', stringMatrix, 4, false)).to.equal(ErrorCode.NA);
            });

            it('should return the result of the vertical search with references', function () {
                expect(VLOOKUP(4.19, r3d('2:2!A1:C5'), 3)).to.equal('Orange');
                expect(VLOOKUP(5.75, r3d('2:2!A1:C5'), 3)).to.equal('Gelb');
                expect(VLOOKUP(6.00, r3d('2:2!A1:C5'), 3)).to.equal('Gruen');
                expect(VLOOKUP(7.66, r3d('2:2!A1:C5'), 3)).to.equal('Blau');
            });
        });
    });

    // ========================================================================
});
