/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/formatter'
], function (AppHelper, SheetHelper, Utils, Formatter) {

    'use strict';

    // convenience shortcuts
    var i = SheetHelper.i;
    var a = SheetHelper.a;
    var r = SheetHelper.r;

    // class CellCollection ===================================================

    describe('Spreadsheet class CellCollection', function () {

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a1', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: true, number: true, protect: true }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 14, fillColor: { type: 'auto' }, borderLeft: { style: 'none' }, borderRight: { style: 'none' }, borderTop: { style: 'none' }, borderBottom: { style: 'none' }, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a2', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: true, number: true, protect: true }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 20, fillColor: { type: 'auto' }, borderLeft: { style: 'none' }, borderRight: { style: 'none' }, borderTop: { style: 'none' }, borderBottom: { style: 'none' }, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a3', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: true, number: true, protect: true }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 22, fillColor: { type: 'auto' }, borderLeft: { style: 'none' }, borderRight: { style: 'none' }, borderTop: { style: 'none' }, borderBottom: { style: 'none' }, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a4', attrs: { character: { fontName: 'Arial',   fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1'  } }, apply: { font: true, fill: true, border: true, align: true, number: true, protect: true }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: 'FFFF00' }, borderLeft: { style: 'none' }, borderRight: { style: 'none' }, borderTop: { style: 'none' }, borderBottom: { style: 'none' }, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a5', attrs: { character: { fontName: 'Calibri', fontSize: 12, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'rgb',    value: '000000' } }, apply: { font: true, fill: true, border: true, align: true, number: true, protect: true }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: 'FFFF00' }, borderLeft: { style: 'none' }, borderRight: { style: 'none' }, borderTop: { style: 'none' }, borderBottom: { style: 'none' }, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a6', attrs: { character: { fontName: 'Arial',   fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1'  } }, apply: { font: true, fill: true, border: true, align: true, number: true, protect: true }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, fillColor: { type: 'rgb', value: '92D050' }, borderLeft: { style: 'none' }, borderRight: { style: 'none' }, borderTop: { style: 'none' }, borderBottom: { style: 'none' }, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a7',  attrs: { cell: { formatId: 0, fillColor: { type: 'auto' }, borderLeft: { style: 'none' }, borderTop: { style: 'none' }, borderRight: { style: 'none' }, borderBottom: { style: 'none' }, unlocked: false }, character: { fontName: 'Arial', fontSize: 11, color: { type: 'scheme', value: 'dark1' } }, apply: { border: false, align: false, number: false, protect: false, font: false, fill: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a8',  attrs: { cell: { formatId: 0, fillColor: { type: 'auto' },                 borderLeft: { style: 'none' }, borderTop: { style: 'none' }, borderRight: { style: 'none' }, borderBottom: { style: 'none' }, unlocked: false }, character: { fontName: 'Arial', fontSize: 11, color: { type: 'scheme', value: 'dark1' } }, apply: { border: false, align: false, number: false, protect: false, font: false, fill: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a9',  attrs: { cell: { formatId: 0, fillColor: { type: 'rgb', value: '92D050' }, borderLeft: { style: 'none' }, borderTop: { style: 'none' }, borderRight: { style: 'none' }, borderBottom: { style: 'none' }, alignHor: 'auto', alignVert: 'bottom', wrapText: false }, character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, strike: 'none', color: { type: 'scheme', value: 'dark1' }, underline: false }, apply: { border: false, align: false, number: false, protect: false, font: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a10', attrs: { cell: { formatId: 0, fillColor: { type: 'auto' },                 borderLeft: { style: 'none' }, borderTop: { style: 'none' }, borderRight: { style: 'none' }, borderBottom: { style: 'none' }, alignHor: 'auto', alignVert: 'bottom', wrapText: false }, character: { fontName: 'Arial', fontSize: 11, bold: true,  italic: false, strike: 'none', color: { type: 'scheme', value: 'dark1' }, underline: false }, apply: { border: false, align: false, number: false, protect: false, fill: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a11', attrs: { cell: { formatId: 0, fillColor: { type: 'rgb', value: '92D050' }, borderLeft: { style: 'none' }, borderTop: { style: 'none' }, borderRight: { style: 'none' }, borderBottom: { style: 'none' }, alignHor: 'auto', alignVert: 'bottom', wrapText: false }, character: { fontName: 'Arial', fontSize: 11, bold: true,  italic: false, strike: 'none', color: { type: 'scheme', value: 'dark1' }, underline: false }, apply: { border: false, align: false, number: false, protect: false }, styleId: 'Standard' } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a12', attrs: { cell: { formatId: 0, fillColor: { type: 'rgb', value: 'FF0000' }, borderLeft: { style: 'none' }, borderTop: { style: 'none' }, borderRight: { style: 'none' }, borderBottom: { style: 'none' }, alignHor: 'auto', alignVert: 'bottom', wrapText: false, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: false, fill: true, border: false, align: false, number: false, protect: false }, styleId: 'Standard' } },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: [0, 0], contents: [
                { c: [{ v: 36526, s: 'a1' }, { v: 36892, s: 'a1' }, { v: 36526, s: 'a1' }, { v: 0.6666666666666666, s: 'a2' }, { v: 36526.6666666666666666, s: 'a3' }, { v: 36526, s: 'a1' }] },
                { c: [{ v: 36528, s: 'a1' }, { v: 36923, s: 'a1' }, { v: 36892, s: 'a1' }, { v: 0.7083333333333334, s: 'a2' }, { v: 36526.7083333333333334, s: 'a3' }, { v: 36558, s: 'a1' }] },
                { c: [{ v: 36530, s: 'a1' }, { v: 36951, s: 'a1' }, { v: 37257, s: 'a1' }, { v: 0.75, s: 'a2' }, { v: 36526.75, s: 'a3' }, { v: 36588, s: 'a1' }] },
                { c: [{ v: 36532, s: 'a1' }, { v: 36982, s: 'a1' }, { v: 37622, s: 'a1' }, { v: 0.7916666666666666, s: 'a2' }, { v: 36526.7916666666666666, s: 'a3' }, { v: 36620, s: 'a1' }] },
                { c: [{ v: 36534, s: 'a1' }, { v: 37012, s: 'a1' }, { v: 37987, s: 'a1' }, { v: 0.8333333333333334, s: 'a2' }, { v: 36526.8333333333333334, s: 'a3' }, { v: 36651, s: 'a1' }] }
            ] },

            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeCells', sheet: 1, start: [0, 0], contents: [
                { c: [{ v: 'step 1', r: 5 }, { v: 1, s: 'a4', r: 2 }, { v: 'Montag'       }, { v: 'step 1'   }, { v: '1. Step'  }] },
                { c: [{ v: 'step 2'       }, { v: 'step 7'         }, { v: 'step 4', r: 2 }, { v: 'hier'     }, { v: 2, s: 'a4' }, { v: 5, s: 'a4' }, { v: 'Dienstag'     }, { v: '12 \xc4pfel' }, { v: '2. Step' }] },
                { c: [{ v: 'string', r: 2 }, { v: 'step 77'        }, { v: 'step 7'       }, { v: 'step 2'   }, { v: 'string'   }, { v: 6, s: 'a5' }, { v: 'string', r: 2 }, { v: '3. Step'  }] },
                { c: [{ v: 'step 3', r: 2 }, { v: 'string',  r: 2  }, { v: 'ein'          }, { v: 3, s: 'a6' }, { v: 'string'   }, { v: 'Mittwoch' }, { v: 22, s: 'a5'    }, { v: 'hallo'    }] },
                { c: [{ v: 'step 4'       }, { v: 'step 99'        }, { v: 'step 8'       }, { v: 'bla'      }, { v: 'step 3'   }, { v: 4, s: 'a6' }, { v: 99, s: 'a4'    }, { v: 'string'   }, { v: 'Montag' }, { v: '100. Step' }] }
            ] },

            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'changeCells', sheet: 2, start: [0, 0],  contents: [
                { c: [{ v: 1,    s: 'a7' }, { r: 1 }, { f: 'SUM(A1:A3)', v: 21, s: 'a7' }, { f: 'SUM($A$1:A3)', v: 21, s: 'a7' }, { f: 'SUM(A1:A3)', v: 21, s: 'a7' }, { v: 1, s: 'a7' }] },
                { c: [{ v: 5,    s: 'a7' }, { r: 1 }, { f: 'SUM(A2:A4)', v: 50, s: 'a7' }, { f: 'SUM($A$1:A4)', v: 51, s: 'a7' }, { f: 'SUM(A2:A4)', v: 50, s: 'a7' }, { f: '(F1+1)', v: 2, s: 'a7' }] },
                { c: [{ v: 15,   s: 'a7' }, { r: 1 }, { f: 'SUM(A3:A5)', v: 115, s: 'a7' }, { f: 'SUM($A$1:A5)', v: 121, s: 'a7' }, { f: 'SUM(A3:A5)', v: 115, s: 'a7' }] },
                { c: [{ v: 30,   s: 'a7' }, { r: 1 }] },
                { c: [{ v: 70,   s: 'a7' }, { r: 1 }, { r: 4 }, { f: '(G6+1)', v: 2, s: 'a7' }] },
                { c: [{ v: 100,  s: 'a7' }, { r: 1 }, { r: 4 }, { v: 1, s: 'a7' }] },
                { c: [{ v: 200,  s: 'a7' }, { r: 1 }] },
                { c: [{ v: 800,  s: 'a7' }, { r: 1 }] },
                { c: [{ v: 1000, s: 'a7' }, { r: 1 }] },
                { c: [{ v: 3000, s: 'a7' }, { r: 1 }] },
                { c: [{ r: 5 }, { v: 1,    s: 'a7' }, { f: '(F11+1)', v: 2, s: 'a7' }] },
                { c: [{ r: 5 }, { f: '(G12+1)', v: 2, s: 'a7' }, { v: 1, s: 'a7' }] }
            ] },

            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet4' },
            { name: 'changeCells', sheet: 3, start: [2, 4], contents: [
                { c: [{ v: 1, s: 'a9'  }, { v: 4, s: 'a8'  }, { v: 'hier',   s: 'a8'  }] },
                { c: [{ v: 2, s: 'a10' }, { v: 5, s: 'a11' }, { v: 'ein',    s: 'a10' }] },
                { c: [{ v: 3, s: 'a8'  }, { v: 6, s: 'a8'  }, { v: 'string', s: 'a9'  }] }
            ] },

            { name: 'insertSheet', sheet: 4, sheetName: 'ROW_Ops' },
            { name: 'changeRows', sheet: 4, start: 5, s: 'a12', attrs: { row: { customFormat: true } } },
            { name: 'changeCells', sheet: 4, start: [0, 4], contents: [
                { c: [{ v: 1 }, { v: 4 }, { v: 'hier' }] },
                { c: [{ v: 2 }, { v: 5 }, { v: 'ein' }] },
                { c: [{ v: 3 }, { v: 6 }, { v: 'string' }] }
            ] },

            { name: 'insertSheet', sheet: 5, sheetName: 'COL_Ops' },
            { name: 'changeCells', sheet: 5, start: [0, 4], contents: [
                { c: [{ v: 1 }, { v: 4 }, { v: 'hier' }] },
                { c: [{ v: 2 }, { v: 5 }, { v: 'ein' }] },
                { c: [{ v: 3 }, { v: 6 }, { v: 'string' }] }
            ] }
        ];

        // initialize test document
        var docModel = null, sheetModel = null, cellCollection = null, formatter = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel = docModel.getSheetModel(0);
            cellCollection = sheetModel.getCellCollection();
            formatter = new Formatter({ dec: ',', group: '.' });
        });

        // helper methods -----------------------------------------------------
        function prepareDate(options) {
            var dateString = Utils.getStringOption(options, 'dateString', null),
                timeString = Utils.getStringOption(options, 'timeString', null),

                dateParts   = null,
                timeParts   = null,
                day         = 30,
                month       = 11,
                year        = 1899,
                hour        = 0,
                minute      = 0,
                second      = 0,
                milli       = 0;

            if (dateString) {
                dateParts   = dateString.split('.');
                day         = parseInt(dateParts[0], 10);
                month       = (parseInt(dateParts[1], 10) - 1);
                year        = parseInt(dateParts[2], 10);
            }

            if (timeString) {
                timeParts   = timeString.split(':');
                hour        = (timeParts.length >= 2) ? parseInt(timeParts[0], 10) : 0;
                minute      = (timeParts.length >= 2) ? parseInt(timeParts[1], 10) : 0;
                second      = (timeParts.length >= 3) ? parseInt(timeParts[2], 10) : 0;
                milli       = (timeParts.length >= 4) ? parseInt(timeParts[3], 10) : 0;
            }

            // console.log(new Date(year, month, day, hour, minute, second, milli));
            // console.log(new Date(Date.UTC(year, month, day, hour, minute, second, milli)));

            return formatter.convertDateToNumber(new Date(Date.UTC(year, month, day, hour, minute, second, milli)));
        }

        // public methods -----------------------------------------------------

        describe('method "generateAutoFillOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateAutoFillOperations');
            });

            // numbers
            it('should generate correct autofill operations (simple linear numbers -> increase)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col = 'N',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 1,
                    val_cell_2 = 2,
                    val_cell_3 = 3,
                    val_cell_4 = 4,
                    val_cell_5 = 5;

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('should generate correct autofill operations (non-linear numbers -> increase)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col = 'O',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 1,
                    val_cell_2 = 3,
                    val_cell_3 = 4,
                    val_cell_4 = 5.666666666666666,
                    val_cell_5 = 7.166666666666666;

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('INVERTED should generate correct autofill operations (simple linear numbers -> copy)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = true,

                    col = 'N',

                    cell_1 = col + '11',
                    cell_2 = col + '12',
                    cell_3 = col + '13',
                    cell_4 = col + '14',
                    cell_5 = col + '15',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 1,
                    val_cell_2 = 2,
                    val_cell_3 = 3,
                    val_cell_4 = 1,
                    val_cell_5 = 2;

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('TOP should generate correct autofill operations (simple linear numbers -> increase)', function (done) {
                var sheet = 2,
                    count = 5,
                    border = 'top',
                    invertType = false,

                    col = 'N',

                    cell_1 = col + '100',
                    cell_2 = col + '101',
                    cell_3 = col + '102',
                    cell_4 = col + '103',
                    cell_5 = col + '104',
                    cell_6 = col + '105',
                    cell_7 = col + '106',
                    cell_8 = col + '107',
                    cell_9 = col + '108',
                    cell_10 = col + '109',

                    sourceRange = r(cell_6 + ':' + cell_10),

                    val_cell_1 = 1,
                    val_cell_2 = 2,
                    val_cell_3 = 3,
                    val_cell_4 = 4,
                    val_cell_5 = 5,
                    val_cell_6 = 6,
                    val_cell_7 = 7,
                    val_cell_8 = 8,
                    val_cell_9 = 9,
                    val_cell_10 = 10;

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_6), { v: val_cell_6 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_7), { v: val_cell_7 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_8), { v: val_cell_8 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_9), { v: val_cell_9 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_10), { v: val_cell_10 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_6))).to.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.equal(val_cell_10);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5);
                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                        expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                        expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);

                        done();
                    });
                });
            });
            it('LEFT should generate correct autofill operations (simple linear numbers -> copy)', function (done) {
                var sheet = 2,
                    count = 2,
                    border = 'left',
                    invertType = false,

                    col_from = 'H',
                    col_to = 'G',
                    col_to2 = 'F',

                    cell_1 = col_from + '100',
                    cell_2 = col_from + '101',
                    cell_3 = col_from + '102',
                    cell_4 = col_from + '103',
                    cell_5 = col_from + '104',

                    cell_6 = col_to + '100',
                    cell_7 = col_to + '101',
                    cell_8 = col_to + '102',
                    cell_9 = col_to + '103',
                    cell_10 = col_to + '104',

                    cell_11 = col_to2 + '100',
                    cell_12 = col_to2 + '101',
                    cell_13 = col_to2 + '102',
                    cell_14 = col_to2 + '103',
                    cell_15 = col_to2 + '104',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 1,
                    val_cell_2 = 2,
                    val_cell_3 = 3,
                    val_cell_4 = 4,
                    val_cell_5 = 5,

                    val_cell_6 = 1,
                    val_cell_7 = 2,
                    val_cell_8 = 3,
                    val_cell_9 = 4,
                    val_cell_10 = 5,

                    val_cell_11 = 1,
                    val_cell_12 = 2,
                    val_cell_13 = 3,
                    val_cell_14 = 4,
                    val_cell_15 = 5;

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);
                expect(cellCollection.getValue(a(cell_11))).to.equal(null);
                expect(cellCollection.getValue(a(cell_12))).to.equal(null);
                expect(cellCollection.getValue(a(cell_13))).to.equal(null);
                expect(cellCollection.getValue(a(cell_14))).to.equal(null);
                expect(cellCollection.getValue(a(cell_15))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_4), { v: val_cell_4 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_5), { v: val_cell_5 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5);

                    expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_10))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_11))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_12))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_13))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_14))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_15))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_6))).to.equal(val_cell_6);
                        expect(cellCollection.getValue(a(cell_7))).to.equal(val_cell_7);
                        expect(cellCollection.getValue(a(cell_8))).to.equal(val_cell_8);
                        expect(cellCollection.getValue(a(cell_9))).to.equal(val_cell_9);
                        expect(cellCollection.getValue(a(cell_10))).to.equal(val_cell_10);
                        expect(cellCollection.getValue(a(cell_11))).to.equal(val_cell_11);
                        expect(cellCollection.getValue(a(cell_12))).to.equal(val_cell_12);
                        expect(cellCollection.getValue(a(cell_13))).to.equal(val_cell_13);
                        expect(cellCollection.getValue(a(cell_14))).to.equal(val_cell_14);
                        expect(cellCollection.getValue(a(cell_15))).to.equal(val_cell_15);

                        done();
                    });
                });
            });
            it('INVERTED LEFT should generate correct autofill operations (simple linear numbers -> increase)', function (done) {
                var sheet = 2,
                    count = 2,
                    border = 'left',
                    invertType = true,

                    col_from = 'Z',
                    col_to = 'Y',
                    col_to2 = 'X',

                    cell_1 = col_from + '100',
                    cell_2 = col_from + '101',
                    cell_3 = col_from + '102',
                    cell_4 = col_from + '103',
                    cell_5 = col_from + '104',

                    cell_6 = col_to + '100',
                    cell_7 = col_to + '101',
                    cell_8 = col_to + '102',
                    cell_9 = col_to + '103',
                    cell_10 = col_to + '104',

                    cell_11 = col_to2 + '100',
                    cell_12 = col_to2 + '101',
                    cell_13 = col_to2 + '102',
                    cell_14 = col_to2 + '103',
                    cell_15 = col_to2 + '104',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 1,
                    val_cell_2 = 2,
                    val_cell_3 = 3,
                    val_cell_4 = 4,
                    val_cell_5 = 5,

                    val_cell_6 = 0,
                    val_cell_7 = 1,
                    val_cell_8 = 2,
                    val_cell_9 = 3,
                    val_cell_10 = 4,

                    val_cell_11 = -1,
                    val_cell_12 = 0,
                    val_cell_13 = 1,
                    val_cell_14 = 2,
                    val_cell_15 = 3;

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);
                expect(cellCollection.getValue(a(cell_11))).to.equal(null);
                expect(cellCollection.getValue(a(cell_12))).to.equal(null);
                expect(cellCollection.getValue(a(cell_13))).to.equal(null);
                expect(cellCollection.getValue(a(cell_14))).to.equal(null);
                expect(cellCollection.getValue(a(cell_15))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_4), { v: val_cell_4 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_5), { v: val_cell_5 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5);

                    expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_10))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_11))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_12))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_13))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_14))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_15))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_6))).to.equal(val_cell_6);
                        expect(cellCollection.getValue(a(cell_7))).to.equal(val_cell_7);
                        expect(cellCollection.getValue(a(cell_8))).to.equal(val_cell_8);
                        expect(cellCollection.getValue(a(cell_9))).to.equal(val_cell_9);
                        expect(cellCollection.getValue(a(cell_10))).to.equal(val_cell_10);
                        expect(cellCollection.getValue(a(cell_11))).to.equal(val_cell_11);
                        expect(cellCollection.getValue(a(cell_12))).to.equal(val_cell_12);
                        expect(cellCollection.getValue(a(cell_13))).to.equal(val_cell_13);
                        expect(cellCollection.getValue(a(cell_14))).to.equal(val_cell_14);
                        expect(cellCollection.getValue(a(cell_15))).to.equal(val_cell_15);

                        done();
                    });
                });
            });

            // strings
            it('should generate correct autofill operations (simple strings -> copy)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col = 'P',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 'Hallo',
                    val_cell_2 = 'sch\xf6ne',
                    val_cell_3 = 'Welt',
                    val_cell_4 = 'Hallo',
                    val_cell_5 = 'sch\xf6ne';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('MULTI RANGE should generate correct autofill operations (simple strings -> copy)', function (done) {
                var sheet = 2,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col_start = 'K',
                    col_over = 'L',
                    col_end = 'M',

                    cell_10 = col_start + '16',
                    cell_11 = col_start + '17',
                    cell_12 = col_start + '18',
                    cell_13 = col_start + '19',
                    cell_14 = col_start + '20',

                    cell_20 = col_over + '16',
                    cell_21 = col_over + '17',
                    cell_22 = col_over + '18',
                    cell_23 = col_over + '19',
                    cell_24 = col_over + '20',

                    cell_30 = col_end + '16',
                    cell_31 = col_end + '17',
                    cell_32 = col_end + '18',
                    cell_33 = col_end + '19',
                    cell_34 = col_end + '20',

                    sourceRange = r(cell_10 + ':' + cell_32),

                    val_cell_10 = 'String',
                    val_cell_11 = 'Nummer',
                    val_cell_12 = 'eins',
                    val_cell_13 = 'String',
                    val_cell_14 = 'Nummer',

                    val_cell_20 = 'Noch',
                    val_cell_21 = 'ein',
                    val_cell_22 = 'String',
                    val_cell_23 = 'Noch',
                    val_cell_24 = 'ein',

                    val_cell_30 = 'Und',
                    val_cell_31 = 'noch',
                    val_cell_32 = 'einer',
                    val_cell_33 = 'Und',
                    val_cell_34 = 'noch';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_10))).to.equal(null);
                expect(cellCollection.getValue(a(cell_11))).to.equal(null);
                expect(cellCollection.getValue(a(cell_12))).to.equal(null);
                expect(cellCollection.getValue(a(cell_13))).to.equal(null);
                expect(cellCollection.getValue(a(cell_14))).to.equal(null);

                expect(cellCollection.getValue(a(cell_20))).to.equal(null);
                expect(cellCollection.getValue(a(cell_21))).to.equal(null);
                expect(cellCollection.getValue(a(cell_22))).to.equal(null);
                expect(cellCollection.getValue(a(cell_23))).to.equal(null);
                expect(cellCollection.getValue(a(cell_24))).to.equal(null);

                expect(cellCollection.getValue(a(cell_30))).to.equal(null);
                expect(cellCollection.getValue(a(cell_31))).to.equal(null);
                expect(cellCollection.getValue(a(cell_32))).to.equal(null);
                expect(cellCollection.getValue(a(cell_33))).to.equal(null);
                expect(cellCollection.getValue(a(cell_34))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_10), { v: val_cell_10 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_11), { v: val_cell_11 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_12), { v: val_cell_12 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_20), { v: val_cell_20 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_21), { v: val_cell_21 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_22), { v: val_cell_22 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_30), { v: val_cell_30 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_31), { v: val_cell_31 });
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_32), { v: val_cell_32 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_13))).to.equal(val_cell_13);
                        expect(cellCollection.getValue(a(cell_14))).to.equal(val_cell_14);

                        expect(cellCollection.getValue(a(cell_23))).to.equal(val_cell_23);
                        expect(cellCollection.getValue(a(cell_24))).to.equal(val_cell_24);

                        expect(cellCollection.getValue(a(cell_33))).to.equal(val_cell_33);
                        expect(cellCollection.getValue(a(cell_34))).to.equal(val_cell_34);

                        done();
                    });
                });
            });

            // strings + numbers
            it('should generate correct autofill operations (strings + number linear -> increase)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col = 'Q',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 'Step 1',
                    val_cell_2 = 'Step 2',
                    val_cell_3 = 'Step 3',
                    val_cell_4 = 'Step 4',
                    val_cell_5 = 'Step 5';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('should generate correct autofill operations (string + non-linear number -> copy)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col = 'R',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 'Step 1',
                    val_cell_2 = 'step 3',
                    val_cell_3 = 'Step 99',
                    val_cell_4 = 'Step 1',
                    val_cell_5 = 'step 3';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('INVERTED should generate correct autofill operations (strings + number linear -> copy)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = true,

                    col = 'Q',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 'Step 1',
                    val_cell_2 = 'Step 2',
                    val_cell_3 = 'Step 3',
                    val_cell_4 = 'Step 1',
                    val_cell_5 = 'Step 2';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('RIGHT INVERTED should generate correct autofill operations (strings + number linear -> copy)', function (done) {
                var sheet = 0,
                    count = 1,
                    border = 'right',
                    invertType = true,

                    col_from = 'S',
                    col_to = 'T',

                    cell_1 = col_from + '100',
                    cell_2 = col_from + '101',
                    cell_3 = col_from + '102',

                    cell_4 = col_to + '100',
                    cell_5 = col_to + '101',
                    cell_6 = col_to + '102',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 'Step 1',
                    val_cell_2 = 'Step 2',
                    val_cell_3 = 'Step 3',
                    val_cell_4 = 'Step 1',
                    val_cell_5 = 'Step 2',
                    val_cell_6 = 'Step 3';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_6))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5);
                        expect(cellCollection.getValue(a(cell_6))).to.equal(val_cell_6);

                        done();
                    });
                });
            });

            // numbers + strings
            it('should generate correct autofill operations (number linear + string -> increase)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col = 'S',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = '1. Step',
                    val_cell_2 = '2. step',
                    val_cell_3 = '3. Step',
                    val_cell_4 = '4. Step',
                    val_cell_5 = '5. Step';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('should generate correct autofill operations (number non-linear + string -> copy)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col = 'T',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = '1. Step',
                    val_cell_2 = '3. step',
                    val_cell_3 = '99. Step',
                    val_cell_4 = '1. Step',
                    val_cell_5 = '3. step';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });
            it('INVERTED should generate correct autofill operations (number linear + string -> copy)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = true,

                    col = 'S',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = '1. Step',
                    val_cell_2 = '2. step',
                    val_cell_3 = '3. Step',
                    val_cell_4 = '1. Step',
                    val_cell_5 = '2. step';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });

            // date
            it('should generate correct autofill operations (date day linear -> increase)', function (done) {
                var sheet = 0,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'A',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = prepareDate({ dateString: '01.01.2000' }), // 36526
                    val_cell_2 = prepareDate({ dateString: '03.01.2000' }), // 36528
                    val_cell_3 = prepareDate({ dateString: '05.01.2000' }), // 36530
                    val_cell_4 = prepareDate({ dateString: '07.01.2000' }), // 36532
                    val_cell_5 = prepareDate({ dateString: '09.01.2000' }), // 36534
                    val_cell_6 = prepareDate({ dateString: '11.01.2000' }), // 36536
                    val_cell_7 = prepareDate({ dateString: '13.01.2000' }), // 36538
                    val_cell_8 = prepareDate({ dateString: '15.01.2000' }); // 36540

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_1)))).to.deep.equal(formatter.convertNumberToDate(val_cell_1));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_2)))).to.deep.equal(formatter.convertNumberToDate(val_cell_2));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_3)))).to.deep.equal(formatter.convertNumberToDate(val_cell_3));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_4)))).to.deep.equal(formatter.convertNumberToDate(val_cell_4));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_5)))).to.deep.equal(formatter.convertNumberToDate(val_cell_5));
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_6)))).to.deep.equal(formatter.convertNumberToDate(val_cell_6));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_7)))).to.deep.equal(formatter.convertNumberToDate(val_cell_7));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_8)))).to.deep.equal(formatter.convertNumberToDate(val_cell_8));

                    done();
                });
            });
            it('should generate correct autofill operations (date month linear -> increase)', function (done) {
                var sheet = 0,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'B',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = prepareDate({ dateString: '01.01.2001' }), // 36892
                    val_cell_2 = prepareDate({ dateString: '01.02.2001' }), // 36923
                    val_cell_3 = prepareDate({ dateString: '01.03.2001' }), // 36951
                    val_cell_4 = prepareDate({ dateString: '01.04.2001' }), // 36982
                    val_cell_5 = prepareDate({ dateString: '01.05.2001' }), // 37012
                    val_cell_6 = prepareDate({ dateString: '01.06.2001' }), // 37043
                    val_cell_7 = prepareDate({ dateString: '01.07.2001' }), // 37073
                    val_cell_8 = prepareDate({ dateString: '01.08.2001' }); // 37104

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_1)))).to.deep.equal(formatter.convertNumberToDate(val_cell_1));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_2)))).to.deep.equal(formatter.convertNumberToDate(val_cell_2));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_3)))).to.deep.equal(formatter.convertNumberToDate(val_cell_3));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_4)))).to.deep.equal(formatter.convertNumberToDate(val_cell_4));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_5)))).to.deep.equal(formatter.convertNumberToDate(val_cell_5));
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_6)))).to.deep.equal(formatter.convertNumberToDate(val_cell_6));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_7)))).to.deep.equal(formatter.convertNumberToDate(val_cell_7));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_8)))).to.deep.equal(formatter.convertNumberToDate(val_cell_8));

                    done();
                });
            });
            it('should generate correct autofill operations (date year linear -> increase)', function (done) {
                var sheet = 0,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'C',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = prepareDate({ dateString: '01.01.2000' }),
                    val_cell_2 = prepareDate({ dateString: '01.01.2001' }),
                    val_cell_3 = prepareDate({ dateString: '01.01.2002' }),
                    val_cell_4 = prepareDate({ dateString: '01.01.2003' }),
                    val_cell_5 = prepareDate({ dateString: '01.01.2004' }),
                    val_cell_6 = prepareDate({ dateString: '01.01.2005' }),
                    val_cell_7 = prepareDate({ dateString: '01.01.2006' }),
                    val_cell_8 = prepareDate({ dateString: '01.01.2007' });

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_1)))).to.deep.equal(formatter.convertNumberToDate(val_cell_1));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_2)))).to.deep.equal(formatter.convertNumberToDate(val_cell_2));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_3)))).to.deep.equal(formatter.convertNumberToDate(val_cell_3));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_4)))).to.deep.equal(formatter.convertNumberToDate(val_cell_4));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_5)))).to.deep.equal(formatter.convertNumberToDate(val_cell_5));
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_6)))).to.deep.equal(formatter.convertNumberToDate(val_cell_6));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_7)))).to.deep.equal(formatter.convertNumberToDate(val_cell_7));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_8)))).to.deep.equal(formatter.convertNumberToDate(val_cell_8));

                    done();
                });
            });
            it('should generate correct autofill operations (date non-linear -> copy)', function (done) {
                var sheet = 0,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'F',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = prepareDate({ dateString: '01.01.2000' }),
                    val_cell_2 = prepareDate({ dateString: '02.02.2000' }),
                    val_cell_3 = prepareDate({ dateString: '03.03.2000' }),
                    val_cell_4 = prepareDate({ dateString: '04.04.2000' }),
                    val_cell_5 = prepareDate({ dateString: '05.05.2000' }),
                    val_cell_6 = prepareDate({ dateString: '01.01.2000' }),
                    val_cell_7 = prepareDate({ dateString: '02.02.2000' }),
                    val_cell_8 = prepareDate({ dateString: '03.03.2000' });

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_1)))).to.deep.equal(formatter.convertNumberToDate(val_cell_1));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_2)))).to.deep.equal(formatter.convertNumberToDate(val_cell_2));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_3)))).to.deep.equal(formatter.convertNumberToDate(val_cell_3));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_4)))).to.deep.equal(formatter.convertNumberToDate(val_cell_4));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_5)))).to.deep.equal(formatter.convertNumberToDate(val_cell_5));
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_6)))).to.deep.equal(formatter.convertNumberToDate(val_cell_6));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_7)))).to.deep.equal(formatter.convertNumberToDate(val_cell_7));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_8)))).to.deep.equal(formatter.convertNumberToDate(val_cell_8));

                    done();
                });
            });
            it('INVERTED should generate correct autofill operations (date day linear -> copy)', function (done) {
                var sheet = 0,
                    count = 3,
                    border = 'bottom',
                    invertType = true,

                    col = 'A',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = prepareDate({ dateString: '01.01.2000' }), // 36526
                    val_cell_2 = prepareDate({ dateString: '03.01.2000' }), // 36528
                    val_cell_3 = prepareDate({ dateString: '05.01.2000' }), // 36530
                    val_cell_4 = prepareDate({ dateString: '07.01.2000' }), // 36532
                    val_cell_5 = prepareDate({ dateString: '09.01.2000' }), // 36534
                    val_cell_6 = prepareDate({ dateString: '01.01.2000' }), // 36536
                    val_cell_7 = prepareDate({ dateString: '03.01.2000' }), // 36538
                    val_cell_8 = prepareDate({ dateString: '05.01.2000' }); // 36540

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_1)))).to.deep.equal(formatter.convertNumberToDate(val_cell_1));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_2)))).to.deep.equal(formatter.convertNumberToDate(val_cell_2));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_3)))).to.deep.equal(formatter.convertNumberToDate(val_cell_3));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_4)))).to.deep.equal(formatter.convertNumberToDate(val_cell_4));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_5)))).to.deep.equal(formatter.convertNumberToDate(val_cell_5));

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_6)))).to.deep.equal(formatter.convertNumberToDate(val_cell_6));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_7)))).to.deep.equal(formatter.convertNumberToDate(val_cell_7));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_8)))).to.deep.equal(formatter.convertNumberToDate(val_cell_8));

                    done();
                });
            });
            it('RIGHT should generate correct autofill operations (date non-linear -> increae)', function (done) {
                var sheet  = 0,
                    count  = 1,
                    border = 'right',
                    invertType = false,

                    col_from = 'F',
                    col_to   = 'G',

                    cell_1 = col_from + '1',
                    cell_2 = col_from + '2',
                    cell_3 = col_from + '3',
                    cell_4 = col_from + '4',
                    cell_5 = col_from + '5',

                    cell_6 = col_to + '1',
                    cell_7 = col_to + '2',
                    cell_8 = col_to + '3',
                    cell_9 = col_to + '4',
                    cell_10 = col_to + '5',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = prepareDate({ dateString: '01.01.2000' }),
                    val_cell_2 = prepareDate({ dateString: '02.02.2000' }),
                    val_cell_3 = prepareDate({ dateString: '03.03.2000' }),
                    val_cell_4 = prepareDate({ dateString: '04.04.2000' }),
                    val_cell_5 = prepareDate({ dateString: '05.05.2000' }),

                    val_cell_6 = prepareDate({ dateString: '02.01.2000' }),
                    val_cell_7 = prepareDate({ dateString: '03.02.2000' }),
                    val_cell_8 = prepareDate({ dateString: '04.03.2000' }),
                    val_cell_9 = prepareDate({ dateString: '05.04.2000' }),
                    val_cell_10 = prepareDate({ dateString: '06.05.2000' });

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_1)))).to.deep.equal(formatter.convertNumberToDate(val_cell_1));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_2)))).to.deep.equal(formatter.convertNumberToDate(val_cell_2));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_3)))).to.deep.equal(formatter.convertNumberToDate(val_cell_3));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_4)))).to.deep.equal(formatter.convertNumberToDate(val_cell_4));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_5)))).to.deep.equal(formatter.convertNumberToDate(val_cell_5));
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_6)))).to.deep.equal(formatter.convertNumberToDate(val_cell_6));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_7)))).to.deep.equal(formatter.convertNumberToDate(val_cell_7));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_8)))).to.deep.equal(formatter.convertNumberToDate(val_cell_8));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_9)))).to.deep.equal(formatter.convertNumberToDate(val_cell_9));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_10)))).to.deep.equal(formatter.convertNumberToDate(val_cell_10));

                    done();
                });
            });
            // -> missing:
            //      - 12 day interval linear
            //      - linear on hour base (but without hours in numberformat)

            // time
            it('should generate correct autofill operations (date day linear -> increase)', function (done) {
                var sheet = 0,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'D',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = prepareDate({ timeString: '16:00' }), // 36526
                    val_cell_2 = prepareDate({ timeString: '17:00' }), // 36528
                    val_cell_3 = prepareDate({ timeString: '18:00' }), // 36530
                    val_cell_4 = prepareDate({ timeString: '19:00' }), // 36532
                    val_cell_5 = prepareDate({ timeString: '20:00' }), // 36534
                    val_cell_6 = prepareDate({ timeString: '21:00' }), // 36536
                    val_cell_7 = prepareDate({ timeString: '22:00' }), // 36538
                    val_cell_8 = prepareDate({ timeString: '23:00' }); // 36540

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_1)))).to.deep.equal(formatter.convertNumberToDate(val_cell_1));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_2)))).to.deep.equal(formatter.convertNumberToDate(val_cell_2));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_3)))).to.deep.equal(formatter.convertNumberToDate(val_cell_3));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_4)))).to.deep.equal(formatter.convertNumberToDate(val_cell_4));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_5)))).to.deep.equal(formatter.convertNumberToDate(val_cell_5));
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_6)))).to.deep.equal(formatter.convertNumberToDate(val_cell_6));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_7)))).to.deep.equal(formatter.convertNumberToDate(val_cell_7));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_8)))).to.deep.equal(formatter.convertNumberToDate(val_cell_8));

                    done();
                });
            });
            it('should generate correct autofill operations (simple non-linear -> copy)', function (done) {
                var sheet = 0,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col = 'Z',

                    cell_1 = col + '101',
                    cell_2 = col + '102',
                    cell_3 = col + '103',
                    cell_4 = col + '104',
                    cell_5 = col + '105',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = '10:01',
                    val_cell_2 = '10:04',
                    val_cell_3 = '10:55',
                    val_cell_4 = '10:01',
                    val_cell_5 = '10:04';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getValue(a(cell_1))).to.equal(null);
                expect(cellCollection.getValue(a(cell_2))).to.equal(null);
                expect(cellCollection.getValue(a(cell_3))).to.equal(null);
                expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateCellContentOperations(generator, a(cell_1), { v: val_cell_1 });

                promise = promise.then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_2), { v: val_cell_2 }); // small first letter here on purpose
                }).then(function () {
                    return cellCollection.generateCellContentOperations(generator, a(cell_3), { v: val_cell_3 });
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                    expect(cellCollection.getValue(a(cell_2))).to.equal(val_cell_2);
                    expect(cellCollection.getValue(a(cell_3))).to.equal(val_cell_3);
                    expect(cellCollection.getValue(a(cell_4))).to.equal(null);
                    expect(cellCollection.getValue(a(cell_5))).to.equal(null);

                    var promise2 = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');

                        expect(cellCollection.getValue(a(cell_4))).to.equal(val_cell_4);
                        expect(cellCollection.getValue(a(cell_5))).to.equal(val_cell_5); // needs to have the first letter big

                        done();
                    });
                });
            });

            // date + time
            it('should generate correct autofill operations (date day + time linear -> increase)', function (done) {
                var sheet = 0,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'E',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = prepareDate({ dateString: '01.01.2000', timeString: '16:00' }), // 36526
                    val_cell_2 = prepareDate({ dateString: '01.01.2000', timeString: '17:00' }), // 36528
                    val_cell_3 = prepareDate({ dateString: '01.01.2000', timeString: '18:00' }), // 36530
                    val_cell_4 = prepareDate({ dateString: '01.01.2000', timeString: '19:00' }), // 36532
                    val_cell_5 = prepareDate({ dateString: '01.01.2000', timeString: '20:00' }), // 36534
                    val_cell_6 = prepareDate({ dateString: '01.01.2000', timeString: '21:00' }), // 36536
                    val_cell_7 = prepareDate({ dateString: '01.01.2000', timeString: '22:00' }), // 36538
                    val_cell_8 = prepareDate({ dateString: '01.01.2000', timeString: '23:00' }); // 36540

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_1)))).to.deep.equal(formatter.convertNumberToDate(val_cell_1));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_2)))).to.deep.equal(formatter.convertNumberToDate(val_cell_2));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_3)))).to.deep.equal(formatter.convertNumberToDate(val_cell_3));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_4)))).to.deep.equal(formatter.convertNumberToDate(val_cell_4));
                expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_5)))).to.deep.equal(formatter.convertNumberToDate(val_cell_5));
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_6)))).to.deep.equal(formatter.convertNumberToDate(val_cell_6));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_7)))).to.deep.equal(formatter.convertNumberToDate(val_cell_7));
                    expect(formatter.convertNumberToDate(cellCollection.getValue(a(cell_8)))).to.deep.equal(formatter.convertNumberToDate(val_cell_8));

                    done();
                });
            });

            // formulas
            it('should generate correct autofill operations (formulas #1 -> increase)', function (done) {
                var sheet = 2,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'C',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 'SUM(A1:A3)',
                    val_cell_2 = 'SUM(A2:A4)',
                    val_cell_3 = 'SUM(A3:A5)',
                    val_cell_4 = 'SUM(A4:A6)',
                    val_cell_5 = 'SUM(A5:A7)',
                    val_cell_6 = 'SUM(A6:A8)';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getFormula(a(cell_1), 'op')).to.deep.equal(val_cell_1);
                expect(cellCollection.getFormula(a(cell_2), 'op')).to.deep.equal(val_cell_2);
                expect(cellCollection.getFormula(a(cell_3), 'op')).to.deep.equal(val_cell_3);
                expect(cellCollection.getFormula(a(cell_4), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_5), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_6), 'op')).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getFormula(a(cell_4), 'op')).to.deep.equal(val_cell_4);
                    expect(cellCollection.getFormula(a(cell_5), 'op')).to.deep.equal(val_cell_5);
                    expect(cellCollection.getFormula(a(cell_6), 'op')).to.deep.equal(val_cell_6);

                    done();
                });
            });
            it('should generate correct autofill operations (formulas #2 -> increase)', function (done) {
                var sheet = 2,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'D',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 'SUM($A$1:A3)',
                    val_cell_2 = 'SUM($A$1:A4)',
                    val_cell_3 = 'SUM($A$1:A5)',
                    val_cell_4 = 'SUM($A$1:A6)',
                    val_cell_5 = 'SUM($A$1:A7)',
                    val_cell_6 = 'SUM($A$1:A8)';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getFormula(a(cell_1), 'op')).to.deep.equal(val_cell_1);
                expect(cellCollection.getFormula(a(cell_2), 'op')).to.deep.equal(val_cell_2);
                expect(cellCollection.getFormula(a(cell_3), 'op')).to.deep.equal(val_cell_3);
                expect(cellCollection.getFormula(a(cell_4), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_5), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_6), 'op')).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getFormula(a(cell_4), 'op')).to.deep.equal(val_cell_4);
                    expect(cellCollection.getFormula(a(cell_5), 'op')).to.deep.equal(val_cell_5);
                    expect(cellCollection.getFormula(a(cell_6), 'op')).to.deep.equal(val_cell_6);

                    done();
                });
            });
            it('INVERTED should generate correct autofill operations (formulas #3 -> copy)', function (done) {
                var sheet = 2,
                    count = 3,
                    border = 'bottom',
                    invertType = true,

                    col = 'E',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',

                    sourceRange = r(cell_1 + ':' + cell_3),

                    val_cell_1 = 'SUM(A1:A3)',
                    val_cell_2 = 'SUM(A2:A4)',
                    val_cell_3 = 'SUM(A3:A5)',
                    val_cell_4 = 'SUM(A1:A3)',
                    val_cell_5 = 'SUM(A2:A4)',
                    val_cell_6 = 'SUM(A3:A5)';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getFormula(a(cell_1), 'op')).to.deep.equal(val_cell_1);
                expect(cellCollection.getFormula(a(cell_2), 'op')).to.deep.equal(val_cell_2);
                expect(cellCollection.getFormula(a(cell_3), 'op')).to.deep.equal(val_cell_3);
                expect(cellCollection.getFormula(a(cell_4), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_5), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_6), 'op')).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getFormula(a(cell_4), 'op')).to.deep.equal(val_cell_4);
                    expect(cellCollection.getFormula(a(cell_5), 'op')).to.deep.equal(val_cell_5);
                    expect(cellCollection.getFormula(a(cell_6), 'op')).to.deep.equal(val_cell_6);

                    done();
                });
            });
            it('BOTTOM should generate correct autofill operations (formulas #4 -> increase)', function (done) {
                var sheet = 2,
                    count = 3,
                    border = 'bottom',
                    invertType = false,

                    col = 'F',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',

                    sourceRange = r(cell_2 + ':' + cell_2),

                    val_cell_1 = 1,
                    val_cell_2 = '(F1+1)',
                    val_cell_3 = '(F2+1)',
                    val_cell_4 = '(F3+1)',
                    val_cell_5 = '(F4+1)';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                expect(cellCollection.getFormula(a(cell_2), 'op')).to.deep.equal(val_cell_2);
                expect(cellCollection.getFormula(a(cell_3), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_4), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_5), 'op')).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getFormula(a(cell_3), 'op')).to.deep.equal(val_cell_3);
                    expect(cellCollection.getFormula(a(cell_4), 'op')).to.deep.equal(val_cell_4);
                    expect(cellCollection.getFormula(a(cell_5), 'op')).to.deep.equal(val_cell_5);

                    done();
                });
            });
            it('TOP should generate correct autofill operations (formulas #5 -> increase)', function (done) {
                var sheet = 2,
                    count = 3,
                    border = 'top',
                    invertType = false,

                    col = 'G',

                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',

                    sourceRange = r(cell_5 + ':' + cell_5),

                    val_cell_2 = '(G3+1)',
                    val_cell_3 = '(G4+1)',
                    val_cell_4 = '(G5+1)',
                    val_cell_5 = '(G6+1)',
                    val_cell_6 = 1;

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_6))).to.equal(val_cell_6);
                expect(cellCollection.getFormula(a(cell_5), 'op')).to.deep.equal(val_cell_5);
                expect(cellCollection.getFormula(a(cell_4), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_3), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_2), 'op')).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getFormula(a(cell_4), 'op')).to.deep.equal(val_cell_4);
                    expect(cellCollection.getFormula(a(cell_3), 'op')).to.deep.equal(val_cell_3);
                    expect(cellCollection.getFormula(a(cell_2), 'op')).to.deep.equal(val_cell_2);

                    done();
                });
            });
            it('RIGHT should generate correct autofill operations (formulas #6 -> increase)', function (done) {
                var sheet = 2,
                    count = 3,
                    border = 'right',
                    invertType = false,

                    row = '11',

                    cell_1 = 'F' + row,
                    cell_2 = 'G' + row,
                    cell_3 = 'H' + row,
                    cell_4 = 'I' + row,
                    cell_5 = 'J' + row,

                    sourceRange = r(cell_2 + ':' + cell_2),

                    val_cell_1 = 1,
                    val_cell_2 = '(F11+1)',
                    val_cell_3 = '(G11+1)',
                    val_cell_4 = '(H11+1)',
                    val_cell_5 = '(I11+1)';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                expect(cellCollection.getFormula(a(cell_2), 'op')).to.deep.equal(val_cell_2);
                expect(cellCollection.getFormula(a(cell_3), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_4), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_5), 'op')).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getFormula(a(cell_3), 'op')).to.deep.equal(val_cell_3);
                    expect(cellCollection.getFormula(a(cell_4), 'op')).to.deep.equal(val_cell_4);
                    expect(cellCollection.getFormula(a(cell_5), 'op')).to.deep.equal(val_cell_5);

                    done();
                });
            });
            it('LEFT should generate correct autofill operations (formulas #7 -> increase)', function (done) {
                var sheet = 2,
                    count = 3,
                    border = 'left',
                    invertType = false,

                    row = '12',

                    cell_1 = 'G' + row,
                    cell_2 = 'F' + row,
                    cell_3 = 'E' + row,
                    cell_4 = 'D' + row,
                    cell_5 = 'C' + row,

                    sourceRange = r(cell_2 + ':' + cell_2),

                    val_cell_1 = 1,
                    val_cell_2 = '(G12+1)',
                    val_cell_3 = '(F12+1)',
                    val_cell_4 = '(E12+1)',
                    val_cell_5 = '(D12+1)';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.equal(val_cell_1);
                expect(cellCollection.getFormula(a(cell_2), 'op')).to.deep.equal(val_cell_2);
                expect(cellCollection.getFormula(a(cell_3), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_4), 'op')).to.equal(null);
                expect(cellCollection.getFormula(a(cell_5), 'op')).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getFormula(a(cell_3), 'op')).to.deep.equal(val_cell_3);
                    expect(cellCollection.getFormula(a(cell_4), 'op')).to.deep.equal(val_cell_4);
                    expect(cellCollection.getFormula(a(cell_5), 'op')).to.deep.equal(val_cell_5);

                    done();
                });
            });
            // -> missing
            //      - calculation of result doesn't work until now. Could be checked too in the future!
            //      - correct update of the reference cell from a formula is not tested until now.

            // groups
            it('should generate correct autofill operations (mixed groups #1 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'A',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 'step 1',
                    val_cell_2 = 'step 2',
                    val_cell_3 = 'string',
                    val_cell_4 = 'step 3',
                    val_cell_5 = 'step 4',
                    val_cell_6 = 'step 3',
                    val_cell_7 = 'step 4',
                    val_cell_8 = 'string',
                    val_cell_9 = 'step 5',
                    val_cell_10 = 'step 6';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #2 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'B',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 'step 1',
                    val_cell_2 = 'step 7',
                    val_cell_3 = 'string',
                    val_cell_4 = 'step 3',
                    val_cell_5 = 'step 99',
                    val_cell_6 = 'step 13',
                    val_cell_7 = 'step 19',
                    val_cell_8 = 'string',
                    val_cell_9 = 'step 195',
                    val_cell_10 = 'step 291';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #3 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'C',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 'step 1',
                    val_cell_2 = 'step 4',
                    val_cell_3 = 'step 77',
                    val_cell_4 = 'string',
                    val_cell_5 = 'step 8',
                    val_cell_6 = 'step 1',
                    val_cell_7 = 'step 4',
                    val_cell_8 = 'step 77',
                    val_cell_9 = 'string',
                    val_cell_10 = 'step 9';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #4 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'D',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 'step 1',
                    val_cell_2 = 'step 4',
                    val_cell_3 = 'step 7',
                    val_cell_4 = 'string',
                    val_cell_5 = 'bla',
                    val_cell_6 = 'step 10',
                    val_cell_7 = 'step 13',
                    val_cell_8 = 'step 16',
                    val_cell_9 = 'string',
                    val_cell_10 = 'bla';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #5 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'E',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 'step 1',
                    val_cell_2 = 'hier',
                    val_cell_3 = 'step 2',
                    val_cell_4 = 'ein',
                    val_cell_5 = 'step 3',
                    val_cell_6 = 'step 2',
                    val_cell_7 = 'hier',
                    val_cell_8 = 'step 3',
                    val_cell_9 = 'ein',
                    val_cell_10 = 'step 4';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #6 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'F',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 1,
                    val_cell_2 = 2,
                    val_cell_3 = 'string',
                    val_cell_4 = 3,
                    val_cell_5 = 4,
                    val_cell_6 = 3,
                    val_cell_7 = 4,
                    val_cell_8 = 'string',
                    val_cell_9 = 5,
                    val_cell_10 = 6;

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #7 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'G',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 1,
                    val_cell_2 = 5,
                    val_cell_3 = 6,
                    val_cell_4 = 'string',
                    val_cell_5 = 99,
                    val_cell_6 = 9,
                    val_cell_7 = 11.5,
                    val_cell_8 = 14,
                    val_cell_9 = 'string',
                    val_cell_10 = 100;

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #8 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'H',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 'Montag',
                    val_cell_2 = 'Dienstag',
                    val_cell_3 = 'string',
                    val_cell_4 = 'Mittwoch',
                    val_cell_5 = 'string',
                    val_cell_6 = 'Mittwoch',
                    val_cell_7 = 'Donnerstag',
                    val_cell_8 = 'string',
                    val_cell_9 = 'Donnerstag',
                    val_cell_10 = 'string';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #9 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'I',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = 'step 1',
                    val_cell_2 = '12 \xc4pfel',
                    val_cell_3 = 'string',
                    val_cell_4 = 22,
                    val_cell_5 = 'Montag',
                    val_cell_6 = 'step 2',
                    val_cell_7 = '13 \xc4pfel',
                    val_cell_8 = 'string',
                    val_cell_9 = 23,
                    val_cell_10 = 'Dienstag';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });
            it('should generate correct autofill operations (mixed groups #10 -> increase + copy)', function (done) {
                var sheet = 1,
                    count = 5,
                    border = 'bottom',
                    invertType = false,

                    col = 'J',

                    cell_1 = col + '1',
                    cell_2 = col + '2',
                    cell_3 = col + '3',
                    cell_4 = col + '4',
                    cell_5 = col + '5',
                    cell_6 = col + '6',
                    cell_7 = col + '7',
                    cell_8 = col + '8',
                    cell_9 = col + '9',
                    cell_10 = col + '10',

                    sourceRange = r(cell_1 + ':' + cell_5),

                    val_cell_1 = '1. Step',
                    val_cell_2 = '2. Step',
                    val_cell_3 = '3. Step',
                    val_cell_4 = 'hallo',
                    val_cell_5 = '100. Step',
                    val_cell_6 = '4. Step',
                    val_cell_7 = '5. Step',
                    val_cell_8 = '6. Step',
                    val_cell_9 = 'hallo',
                    val_cell_10 = '101. Step';

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();
                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                expect(cellCollection.getValue(a(cell_1))).to.deep.equal(val_cell_1);
                expect(cellCollection.getValue(a(cell_2))).to.deep.equal(val_cell_2);
                expect(cellCollection.getValue(a(cell_3))).to.deep.equal(val_cell_3);
                expect(cellCollection.getValue(a(cell_4))).to.deep.equal(val_cell_4);
                expect(cellCollection.getValue(a(cell_5))).to.deep.equal(val_cell_5);
                expect(cellCollection.getValue(a(cell_6))).to.equal(null);
                expect(cellCollection.getValue(a(cell_7))).to.equal(null);
                expect(cellCollection.getValue(a(cell_8))).to.equal(null);
                expect(cellCollection.getValue(a(cell_9))).to.equal(null);
                expect(cellCollection.getValue(a(cell_10))).to.equal(null);

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getValue(a(cell_6))).to.deep.equal(val_cell_6);
                    expect(cellCollection.getValue(a(cell_7))).to.deep.equal(val_cell_7);
                    expect(cellCollection.getValue(a(cell_8))).to.deep.equal(val_cell_8);
                    expect(cellCollection.getValue(a(cell_9))).to.deep.equal(val_cell_9);
                    expect(cellCollection.getValue(a(cell_10))).to.deep.equal(val_cell_10);

                    done();
                });
            });

            // styles copied?
            it('should generate correct autofill operations (styles -> copy)', function (done) {
                var sheet = 3,
                    count = 2,
                    border = 'bottom',
                    invertType = false,

                    col_start = 'C',
                    col_over = 'D',
                    col_end = 'E',

                    cell_10 = col_start + '5',
                    cell_11 = col_start + '6',
                    cell_12 = col_start + '7',
                    cell_13 = col_start + '8',
                    cell_14 = col_start + '9',

                    cell_20 = col_over + '5',
                    cell_21 = col_over + '6',
                    cell_22 = col_over + '7',
                    cell_23 = col_over + '8',
                    cell_24 = col_over + '9',

                    cell_30 = col_end + '5',
                    cell_31 = col_end + '6',
                    cell_32 = col_end + '7',
                    cell_33 = col_end + '8',
                    cell_34 = col_end + '9',

                    sourceRange = r(cell_10 + ':' + cell_32),

                    val_cell_10 = 'a9',
                    val_cell_11 = 'a10',
                    val_cell_12 = 'a8',
                    val_cell_13 = 'a9',
                    val_cell_14 = 'a10',

                    val_cell_20 = 'a8',
                    val_cell_21 = 'a11',
                    val_cell_22 = 'a8',
                    val_cell_23 = 'a8',
                    val_cell_24 = 'a11',

                    val_cell_30 = 'a8',
                    val_cell_31 = 'a10',
                    val_cell_32 = 'a9',
                    val_cell_33 = 'a8',
                    val_cell_34 = 'a10';

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.getStyleId(a(cell_10))).to.equal(val_cell_10);
                expect(cellCollection.getStyleId(a(cell_11))).to.equal(val_cell_11);
                expect(cellCollection.getStyleId(a(cell_12))).to.equal(val_cell_12);
                expect(cellCollection.getStyleId(a(cell_13))).to.equal('');
                expect(cellCollection.getStyleId(a(cell_14))).to.equal('');

                expect(cellCollection.getStyleId(a(cell_20))).to.equal(val_cell_20);
                expect(cellCollection.getStyleId(a(cell_21))).to.equal(val_cell_21);
                expect(cellCollection.getStyleId(a(cell_22))).to.equal(val_cell_22);
                expect(cellCollection.getStyleId(a(cell_23))).to.equal('');
                expect(cellCollection.getStyleId(a(cell_24))).to.equal('');

                expect(cellCollection.getStyleId(a(cell_30))).to.equal(val_cell_30);
                expect(cellCollection.getStyleId(a(cell_31))).to.equal(val_cell_31);
                expect(cellCollection.getStyleId(a(cell_32))).to.equal(val_cell_32);
                expect(cellCollection.getStyleId(a(cell_33))).to.equal('');
                expect(cellCollection.getStyleId(a(cell_34))).to.equal('');

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });

                var promise = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                    invertType: invertType
                });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getStyleId(a(cell_13))).to.equal(val_cell_13);
                    expect(cellCollection.getStyleId(a(cell_14))).to.equal(val_cell_14);

                    expect(cellCollection.getStyleId(a(cell_23))).to.equal(val_cell_23);
                    expect(cellCollection.getStyleId(a(cell_24))).to.equal(val_cell_24);

                    expect(cellCollection.getStyleId(a(cell_33))).to.equal(val_cell_33);
                    expect(cellCollection.getStyleId(a(cell_34))).to.equal(val_cell_34);

                    done();
                });
            });

            // complete row(s)
            it('should generate correct autofill operations for complete rows #1', function (done) {
                var sheet       = 4,
                    count       = 4,
                    border      = 'bottom',
                    invertType  = false,

                    sourceRange = docModel.makeRowRange(i('5:7')),

                    cells = [
                        { adr: a('A5'), val_before: 1,          val_after: 1           },
                        { adr: a('B5'), val_before: 4,          val_after: 4           },
                        { adr: a('C5'), val_before: 'hier',     val_after: 'hier'      },
                        { adr: a('D5'), val_before: null,       val_after: null        },
                        { adr: a('A6'), val_before: 2,          val_after: 2           },
                        { adr: a('B6'), val_before: 5,          val_after: 5           },
                        { adr: a('C6'), val_before: 'ein',      val_after: 'ein'       },
                        { adr: a('D6'), val_before: null,       val_after: null        },
                        { adr: a('A7'), val_before: 3,          val_after: 3           },
                        { adr: a('B7'), val_before: 6,          val_after: 6           },
                        { adr: a('C7'), val_before: 'string',   val_after: 'string'    },
                        { adr: a('D7'), val_before: null,       val_after: null        },
                        { adr: a('A8'), val_before: null,       val_after: 4           },
                        { adr: a('B8'), val_before: null,       val_after: 7           },
                        { adr: a('C8'), val_before: null,       val_after: 'hier'      },
                        { adr: a('D8'), val_before: null,       val_after: null        },
                        { adr: a('A9'), val_before: null,       val_after: 5           },
                        { adr: a('B9'), val_before: null,       val_after: 8           },
                        { adr: a('C9'), val_before: null,       val_after: 'ein'       },
                        { adr: a('D9'), val_before: null,       val_after: null        },
                        { adr: a('A10'), val_before: null,       val_after: 6          },
                        { adr: a('B10'), val_before: null,       val_after: 9          },
                        { adr: a('C10'), val_before: null,       val_after: 'string'   },
                        { adr: a('D10'), val_before: null,       val_after: null       },
                        { adr: a('A11'), val_before: null,       val_after: 7          },
                        { adr: a('B11'), val_before: null,       val_after: 10         },
                        { adr: a('C11'), val_before: null,       val_after: 'hier'     },
                        { adr: a('D11'), val_before: null,       val_after: null       }
                    ];

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true }),
                    promise     = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_after);
                    });

                    done();
                });
            });
            it('should generate correct autofill operations for complete rows #2', function (done) {
                var sheet       = 4,
                    count       = 4,
                    border      = 'top',
                    invertType  = true,

                    sourceRange = docModel.makeRowRange(i('11:12')),

                    cells = [
                        { adr: a('A5'), val_before: 1,          val_after: 1        },
                        { adr: a('B5'), val_before: 4,          val_after: 4        },
                        { adr: a('C5'), val_before: 'hier',     val_after: 'hier'   },
                        { adr: a('A6'), val_before: 2,          val_after: 2        },
                        { adr: a('B6'), val_before: 5,          val_after: 5        },
                        { adr: a('C6'), val_before: 'ein',      val_after: 'ein'    },
                        { adr: a('A7'), val_before: 3,          val_after: 7        },
                        { adr: a('B7'), val_before: 6,          val_after: 10       },
                        { adr: a('C7'), val_before: 'string',   val_after: 'hier'   },
                        { adr: a('A8'), val_before: 4,          val_after: null     },
                        { adr: a('B8'), val_before: 7,          val_after: null     },
                        { adr: a('C8'), val_before: 'hier',     val_after: null     },
                        { adr: a('A9'), val_before: 5,          val_after: 7        },
                        { adr: a('B9'), val_before: 8,          val_after: 10       },
                        { adr: a('C9'), val_before: 'ein',      val_after: 'hier'   },
                        { adr: a('A10'), val_before: 6,         val_after: null     },
                        { adr: a('B10'), val_before: 9,         val_after: null     },
                        { adr: a('C10'), val_before: 'string',  val_after: null     },
                        { adr: a('A11'), val_before: 7,         val_after: 7        },
                        { adr: a('B11'), val_before: 10,        val_after: 10       },
                        { adr: a('C11'), val_before: 'hier',    val_after: 'hier'   }
                    ];

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true }),
                    promise     = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_after);
                    });

                    done();
                });
            });
            it('should generate correct autofill operations for complete rows #3', function (done) {
                var sheet       = 4,
                    count       = 4,
                    border      = 'bottom',
                    invertType  = false,

                    sourceRange = r('C6:D7'),

                    cells = [
                        { adr: a('C6'),  before: { v: 'ein',   s: ''    }, after: { v: 'ein',  s: ''    } },
                        { adr: a('D6'),  before: { v: null,    s: 'a12' }, after: { v: null,   s: 'a12' } },
                        { adr: a('C7'),  before: { v: 'hier',  s: ''    }, after: { v: 'hier', s: ''    } },
                        { adr: a('D7'),  before: { v: null,    s: ''    }, after: { v: null,   s: ''    } },

                        { adr: a('C8'),  before: { v: null,    s: ''    }, after: { v: 'ein',  s: ''    } },
                        { adr: a('D8'),  before: { v: null,    s: ''    }, after: { v: null,   s: 'a12' } },
                        { adr: a('C9'),  before: { v: 'hier',  s: ''    }, after: { v: 'hier', s: ''    } },
                        { adr: a('D9'),  before: { v: null,    s: ''    }, after: { v: null,   s: ''    } },
                        { adr: a('C10'), before: { v: null,    s: ''    }, after: { v: 'ein',  s: ''    } },
                        { adr: a('D10'), before: { v: null,    s: ''    }, after: { v: null,   s: 'a12' } },
                        { adr: a('C11'), before: { v: 'hier',  s: ''    }, after: { v: 'hier', s: ''    } },
                        { adr: a('D11'), before: { v: null,    s: ''    }, after: { v: null,   s: ''    } }
                    ];

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.before.v);
                    expect(cellCollection.getStyleId(a(obj.adr))).to.equal(obj.before.s);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true }),
                    promise     = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.after.v);
                        expect(cellCollection.getStyleId(a(obj.adr))).to.equal(obj.after.s);
                    });

                    done();
                });
            });

            // complete cols(s)
            it('should generate correct autofill operations for complete cols #1', function (done) {
                var sheet       = 5,
                    count       = 2,
                    border      = 'right',
                    invertType  = false,

                    sourceRange = docModel.makeColRange(i('B:D')),

                    cells = [
                        { adr: a('B4'), val_before: null,       val_after: null     },
                        { adr: a('B5'), val_before: 4,          val_after: 4        },
                        { adr: a('B6'), val_before: 5,          val_after: 5        },
                        { adr: a('B7'), val_before: 6,          val_after: 6        },
                        { adr: a('C4'), val_before: null,       val_after: null     },
                        { adr: a('C5'), val_before: 'hier',     val_after: 'hier'   },
                        { adr: a('C6'), val_before: 'ein',      val_after: 'ein'    },
                        { adr: a('C7'), val_before: 'string',   val_after: 'string' },
                        { adr: a('D4'), val_before: null,       val_after: null     },
                        { adr: a('D5'), val_before: null,       val_after: null     },
                        { adr: a('D6'), val_before: null,       val_after: null     },
                        { adr: a('D7'), val_before: null,       val_after: null     },
                        { adr: a('E4'), val_before: null,       val_after: null     },
                        { adr: a('E5'), val_before: null,       val_after: 5        },
                        { adr: a('E6'), val_before: null,       val_after: 6        },
                        { adr: a('E7'), val_before: null,       val_after: 7        },
                        { adr: a('F4'), val_before: null,       val_after: null     },
                        { adr: a('F5'), val_before: null,       val_after: 'hier'   },
                        { adr: a('F6'), val_before: null,       val_after: 'ein'    },
                        { adr: a('F7'), val_before: null,       val_after: 'string' }
                    ];

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true }),
                    promise     = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_after);
                    });

                    done();
                });
            });
            it('should generate correct autofill operations for complete cols #2', function (done) {
                var sheet       = 5,
                    count       = 3,
                    border      = 'left',
                    invertType  = false,

                    sourceRange = docModel.makeColRange(i('D:D')),

                    cells = [
                        { adr: a('B4'), val_before: null,       val_after: null     },
                        { adr: a('B5'), val_before: 4,          val_after: null     },
                        { adr: a('B6'), val_before: 5,          val_after: null     },
                        { adr: a('B7'), val_before: 6,          val_after: null     },
                        { adr: a('C4'), val_before: null,       val_after: null     },
                        { adr: a('C5'), val_before: 'hier',     val_after: null     },
                        { adr: a('C6'), val_before: 'ein',      val_after: null     },
                        { adr: a('C7'), val_before: 'string',   val_after: null     },
                        { adr: a('D4'), val_before: null,       val_after: null     },
                        { adr: a('D5'), val_before: null,       val_after: null     },
                        { adr: a('D6'), val_before: null,       val_after: null     },
                        { adr: a('D7'), val_before: null,       val_after: null     },
                        { adr: a('E4'), val_before: null,       val_after: null     },
                        { adr: a('E5'), val_before: 5,          val_after: 5        },
                        { adr: a('E6'), val_before: 6,          val_after: 6        },
                        { adr: a('E7'), val_before: 7,          val_after: 7        },
                        { adr: a('F4'), val_before: null,       val_after: null     },
                        { adr: a('F5'), val_before: 'hier',     val_after: 'hier'   },
                        { adr: a('F6'), val_before: 'ein',      val_after: 'ein'    },
                        { adr: a('F7'), val_before: 'string',   val_after: 'string' }
                    ];

                sheetModel      = docModel.getSheetModel(sheet);
                cellCollection  = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true }),
                    promise     = cellCollection.generateAutoFillOperations(generator, sourceRange, border, count, {
                        invertType: invertType
                    });

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_after);
                    });

                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(a(obj.adr))).to.equal(obj.val_before);
                    });

                    done();
                });
            });

            // defined lists

            // user defined lists
            //      -> not yet implemented !!!
        });
    });

    // ========================================================================
});
