/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/presentation/model/objectoperationmixin',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/components/table/table',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/tk/utils'
], function (AppHelper, ObjectOperationMixin, DrawingFrame, Table, DOM, AttributeUtils, Utils) {

    'use strict';

    // mix-in class SlideOperationMixin =======================================

    describe('Presentation mix-in class ObjectOperationMixin', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            selection = null,
            layoutId_1 = 'layout1',
            layoutId_2 = 'layout2',
            masterId_1 = 'master1',
            activeSlide = null,
            allDrawings = null,
            drawing1 = null,
            drawing2 = null,
            drawing3 = null,
            drawing4 = null,
            drawing5 = null,
            operationCounter = 0,
            slide_1_id = 'slide_1', // the ID of the first slide in document view
            slide_3_id = 'slide_3', // the ID of the third slide in document view
            text_para1_drawing1 = 'Hello paragraph 1 in drawing 1',
            text_para1_drawing2 = 'Hello paragraph 1 in drawing 2 of phType content body',
            text_para1_drawing3 = 'Hello paragraph 1 in drawing 3 of phType text body',

            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: {
                        page: { width: 33866, height: 19050, orientation: 'landscape' },
                        defaultTextListStyles: {
                            l1: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 0 } },
                            l2: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 1270 } },
                            l3: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 2540 } }
                        }
                    }
                },
                { name: 'insertMasterSlide', id: masterId_1 },

                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1', left: 1905, top: 5918, width: 21590, height: 4083 } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_1, text: 'Mastertitelformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2', left: 3810, top: 10795, width: 17780, height: 4868 } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_1, text: 'Master-Untertitelformat bearbeiten' },

                { name: 'insertLayoutSlide', id: layoutId_2, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_2, type: 'shape', attrs: { presentation: { phType: 'title' }, drawing: { name: 'Titel 1' } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_2 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_2, text: 'Mastertitelformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_2, type: 'shape', attrs: { presentation: { phIndex: 1 }, drawing: { name: 'Drawing 2', left: 1270, top: 4445, width: 11218, height: 12572 } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_2, attrs: { paragraph: { level: 0 } } },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_2, text: 'Mastertextformat bearbeiten' },

                { name: 'insertSlide', start: [0], target: layoutId_1 },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow left', left: 5027, top: 6725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow up', left: 4027, top: 5725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '00FF00' } } } },
                { name: 'insertParagraph', start: [0, 1, 0] },
                { name: 'insertDrawing', start: [0, 2], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow right', left: 6027, top: 6725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '0000FF' } } } },
                { name: 'insertParagraph', start: [0, 2, 0] },
                { name: 'insertDrawing', start: [0, 3], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow down', left: 4027, top: 7725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF00FF' } } } },
                { name: 'insertParagraph', start: [0, 3, 0] },
                { name: 'insertDrawing', start: [0, 4], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Math Multiply', left: 5027, top: 3725, width: 4419, height: 2509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FFFF00' } } } },
                { name: 'insertParagraph', start: [0, 4, 0] },

                { name: 'insertSlide', start: [1], target: layoutId_1 },
                { name: 'insertDrawing', start: [1, 0], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow left', left: 5027, top: 6725, width: 9419, height: 6000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [1, 0, 0] },
                { name: 'insertDrawing', start: [1, 1], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow up', left: 4027, top: 5000, width: 9419, height: 6000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '00FF00' } } } },
                { name: 'insertParagraph', start: [1, 1, 0] },
                { name: 'insertDrawing', start: [1, 2], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow right', left: 6027, top: 10000, width: 9419, height: 5000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '0000FF' } } } },
                { name: 'insertParagraph', start: [1, 2, 0] },
                { name: 'insertDrawing', start: [1, 3], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow down', left: 4027, top: 7725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF00FF' } } } },
                { name: 'insertParagraph', start: [1, 3, 0] },

                { name: 'insertSlide', start: [2], target: layoutId_1 },
                { name: 'insertDrawing', start: [2, 0], type: 'shape', attrs: { presentation: { phType: 'ctrTitle', phIndex: 10 }, drawing: { name: 'Titel 1', left: 1000, top: 2000, width: 4000, height: 2000 } } },
                { name: 'insertParagraph', start: [2, 0, 0] },
                { name: 'insertText', start: [2, 0, 0, 0], text: text_para1_drawing1 },
                { name: 'insertDrawing', start: [2, 1], type: 'shape', attrs: { presentation: { phIndex: 1 }, drawing: { name: 'Content body', left: 1000, top: 5000, width: 5000, height: 3000 } } },
                { name: 'insertParagraph', start: [2, 1, 0] },
                { name: 'insertText', start: [2, 1, 0, 0], text: text_para1_drawing2 },
                { name: 'insertDrawing', start: [2, 2], type: 'shape', attrs: { presentation: { phType: 'body', phIndex: 2 }, drawing: { name: 'Text body', left: 1000, top: 9000, width: 5000, height: 5000 } } },
                { name: 'insertParagraph', start: [2, 2, 0] },
                { name: 'insertText', start: [2, 2, 0, 0], text: text_para1_drawing3 },
                { name: 'insertDrawing', start: [2, 3], type: 'shape', attrs: { presentation: { phType: 'pic', phIndex: 11 }, drawing: { name: 'Picture place holder', left: 8000, top: 5000, width: 5000, height: 4000 } } },
                { name: 'insertDrawing', start: [2, 4], type: 'shape', attrs: { presentation: { phType: 'tbl', phIndex: 12 }, drawing: { name: 'Table place holder', left: 8000, top: 10000, width: 5000, height: 4000 } } },

                { name: 'insertSlide', start: [3], target: layoutId_1 },
                { name: 'insertDrawing', start: [3, 0], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow left', left: 1027, top: 6725, width: 9419, height: 6000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [3, 0, 0] },
                { name: 'insertDrawing', start: [3, 1], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow up', left: 4027, top: 5000, width: 9419, height: 6000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '00FF00' } } } },
                { name: 'insertParagraph', start: [3, 1, 0] },
                { name: 'insertDrawing', start: [3, 2], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow right', left: 6027, top: 10000, width: 9419, height: 5000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '0000FF' } } } },
                { name: 'insertParagraph', start: [3, 2, 0] },
                { name: 'insertDrawing', start: [3, 3], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow down', left: 3027, top: 7725, width: 9419, height: 6000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF00FF' } } } },
                { name: 'insertParagraph', start: [3, 3, 0] }

            ];

        function getDrawingFromSelection(drawingSelection) {
            var drawingNode = selection.getDrawingNodeFromMultiSelection(drawingSelection);
            return AttributeUtils.getExplicitAttributes(drawingNode).drawing;
        }

        function getAllSelectedDrawings() {
            return selection.isMultiSelection() ? selection.getMultiSelection() : selection.generateMultiSelectionObjectFromDrawingSelection({ anyDrawing: true });
        }

        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(ObjectOperationMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "changeDrawingOrder"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('changeDrawingOrder');
            });

            // Check: Starting with cursor selection, single drawing selection and multi drawing selection
            // Check: One step forwards or backwards or completely to foreground or background
            // Check: Keeping the selection after applying the operation
            // Check: Generating only the required operations
            // Check: Undo and redo

            it('should bring the drawing that contains a text selection one step to the top', function () {

                selection.setTextSelection([0, 0, 0, 0]);  // setting the cursor into the first drawing on the first slide

                activeSlide = model.getSlideById(model.getActiveSlideId());

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide

                expect(allDrawings.length).to.equal(5);

                // assigning drawings
                drawing1 = $(allDrawings[0]);
                drawing2 = $(allDrawings[1]);
                drawing3 = $(allDrawings[2]);
                drawing4 = $(allDrawings[3]);
                drawing5 = $(allDrawings[4]);

                // checking order of drawings
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                operationCounter = model.changeDrawingOrder('forward');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // checking merge of text spans
                expect(drawing1.prevAll().length).to.equal(3); // drawing and two text spans

                // checking the new selection after moving the drawing
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 1])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 2])).to.equal(true);

            });

            it('should bring the selected drawing one step to the top', function () {

                // expecting selection [0,1] to [0,2]

                operationCounter = model.changeDrawingOrder('forward');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);

                // checking merge of text spans
                expect(drawing1.prevAll().length).to.equal(5); // two drawings and three text spans

                // checking the new selection after moving the drawing
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 2])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 3])).to.equal(true);

            });

            it('should bring the selected drawing to the top', function () {

                // expecting selection [0,2] to [0,3]

                operationCounter = model.changeDrawingOrder('front');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // checking merge of text spans
                expect(drawing1.prevAll().length).to.equal(9); // four drawings and five text spans
                expect(drawing1.nextAll().length).to.equal(2); // only one text span and the br element

                // checking the new selection after moving the drawing
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 4])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 5])).to.equal(true);

            });

            it('should not create any operation to bring the top drawing further to the top', function () {

                // expecting selection [0,4] to [0,5]

                operationCounter = model.changeDrawingOrder('front');

                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('forward');

                expect(operationCounter).to.equal(0);

            });

            it('should bring the selected drawing to the back', function () {

                operationCounter = model.changeDrawingOrder('back');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);

                // checking merge of text spans
                expect(drawing1.prevAll().length).to.equal(1); // only one text span
                expect(drawing1.nextAll().length).to.equal(10); // four drawings and five text spans and br element

                // checking the new selection after moving the drawing
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 0])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 1])).to.equal(true);

            });

            it('should not create any operation to bring the background drawing further to the background', function () {

                operationCounter = model.changeDrawingOrder('back');

                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('backward');

                expect(operationCounter).to.equal(0);

            });

            it('should undo the operation that brought the upmost drawing to the back', function (done) {

                var // the undo promise
                    promise = null;

                // check that drawing is still a position [0,0]
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 0])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 1])).to.equal(true);
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);

                // bringing drawing1 to the front again
                promise = model.getUndoManager().undo();

                promise.always(function () {

                    // checking move of drawing
                    expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                    // checking merge of text spans
                    expect(drawing1.prevAll().length).to.equal(9); // four drawings and five text spans
                    expect(drawing1.nextAll().length).to.equal(2); // only one text span and the br element

                    // checking the new selection after moving the drawing
                    expect(_.isEqual(model.getSelection().getStartPosition(), [0, 4])).to.equal(true);
                    expect(_.isEqual(model.getSelection().getEndPosition(), [0, 5])).to.equal(true);

                    done();
                });

            });

            it('should redo the operation that brought the background drawing to the top', function (done) {

                var // the redo promise
                    promise = null;

                // check that drawing is still a position [0,0]
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 4])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 5])).to.equal(true);
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // bringing drawing1 to the front again
                promise = model.getUndoManager().redo();

                promise.always(function () {

                    // checking move of drawing
                    expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);

                    // checking merge of text spans
                    expect(drawing1.prevAll().length).to.equal(1); // only one text span

                    // checking the new selection after moving the drawing
                    expect(_.isEqual(model.getSelection().getStartPosition(), [0, 0])).to.equal(true);
                    expect(_.isEqual(model.getSelection().getEndPosition(), [0, 1])).to.equal(true);

                    done();
                });

            });

            // starting with multi selection, the drawings are in the correct order from drawing1 to drawing5

            it('should bring the selected drawings one step to the top with two operations', function () {

                // checking order of drawings
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                expect(selection.isMultiSelection()).to.equal(false);

                selection.setMultiDrawingSelectionByPosition([[0, 0], [0, 2]]);  // selecting drawing1 and drawing3

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(2);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');

                operationCounter = model.changeDrawingOrder('forward');

                expect(operationCounter).to.equal(2);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // checking the new selection after moving the drawing
                expect(selection.getListOfPositions()).to.equal('(0,1),(0,3)');

            });

            it('should undo the operations that brought the two drawing one step to the front', function (done) {

                var // the undo promise
                    promise = null;

                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);

                // bringing drawing1 and drawing 3 one step backwards
                promise = model.getUndoManager().undo();

                promise.always(function () {

                    // checking move of drawing
                    expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                    expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);

                    // checking the new selection after moving the drawing
                    expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');

                    done();
                });

            });

            it('should generate only one operation, if one of the selected drawings is already in the background', function () {

                expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');

                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);

                operationCounter = model.changeDrawingOrder('backward');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // checking the new selection after moving the drawing
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,1)');
            });

            it('should undo the operations that brought the one drawing one step to the back', function (done) {

                var // the undo promise
                    promise = null;

                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);

                // bringing drawing1 and drawing 3 one step backwards
                promise = model.getUndoManager().undo();

                promise.always(function () {

                    // checking move of drawing
                    expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                    expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);

                    // checking the new selection after moving the drawing
                    // expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');

                    done();
                });

            });

            it('should bring both selected drawings to the top', function () {

                selection.setMultiDrawingSelectionByPosition([[0, 0], [0, 2], [0, 3]]);  // selecting drawing1, drawing3 and drawing4

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(3);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,2),(0,3)');

                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);

                operationCounter = model.changeDrawingOrder('front');

                expect(operationCounter).to.equal(3);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);

                // checking the new selection after moving the drawing
                expect(selection.getListOfPositions()).to.equal('(0,2),(0,3),(0,4)');
            });

            it('should not create any operation, if all drawings are selected', function () {

                selection.setTextSelection([0, 0, 0, 0]); // setting the cursor into the first drawing on the first slide
                selection.selectAllDrawingsOnSlide();

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(5);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,1),(0,2),(0,3),(0,4)');

                operationCounter = model.changeDrawingOrder('front');
                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('forward');
                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('back');
                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('backward');
                expect(operationCounter).to.equal(0);

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(5);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,1),(0,2),(0,3),(0,4)');
            });

            it('should not create move operations by changing the alignment', function () {
                // Select the text of the first drawing
                selection.setTextSelection([1, 0, 0, 0]);

                // align the drawing to left
                operationCounter = model.changeAlignment('left');
                expect(operationCounter).to.equal(true);

                var allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(1);
                // drawing left position must be on the left side on the page with left 0
                expect(getDrawingFromSelection(allDrawingSelections[0]).left).to.equal(0);

                // Select the first and second drawing
                selection.setMultiDrawingSelectionByPosition([[1, 0], [1, 1]]);

                // algin the drawings to 'center'
                operationCounter = model.changeAlignment('center');

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(2);

                // check the new left position
                expect(getDrawingFromSelection(allDrawingSelections[0]).left).to.equal(2014);

                expect(getDrawingFromSelection(allDrawingSelections[1]).left).to.equal(2014);

                // algin the objects to top
                operationCounter = model.changeAlignment('top');

                expect(allDrawingSelections.length).to.equal(2);
                // both drawings must have the same top position, it is the top position of the second drawing
                expect(getDrawingFromSelection(allDrawingSelections[0]).top).to.equal(5000);
                expect(getDrawingFromSelection(allDrawingSelections[1]).top).to.equal(5000);

                // Select three drawing
                selection.setMultiDrawingSelectionByPosition([[1, 0], [1, 1], [1, 2]]);

                // algin the drawings to 'center'
                operationCounter = model.changeAlignment('middle');

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(3);
                //check the new top positions
                expect(getDrawingFromSelection(allDrawingSelections[0]).top).to.equal(7000);
                expect(getDrawingFromSelection(allDrawingSelections[1]).top).to.equal(7000);
                expect(getDrawingFromSelection(allDrawingSelections[2]).top).to.equal(7500);

                // Select the last drawing
                selection.setMultiDrawingSelectionByPosition([[1, 3]]);

                // algin the drawings to 'center'
                operationCounter = model.changeAlignment('right');

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(1);
                //check the new top positions
                expect(getDrawingFromSelection(allDrawingSelections[0]).left).to.equal(24447);

               // Select the first and last drawing
                selection.setMultiDrawingSelectionByPosition([[1, 0], [1, 3]]);

                // algin the drawings to 'center'
                operationCounter = model.changeAlignment('bottom');

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(2);
                //check the new top positions
                expect(getDrawingFromSelection(allDrawingSelections[0]).top).to.equal(8234);
                expect(getDrawingFromSelection(allDrawingSelections[1]).top).to.equal(7725);
            });

        });

        describe('method "distributeDrawings"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('distributeDrawings');
            });

            it('should distribute objects properly', function () {
                // Select the text of the first drawing
                selection.setTextSelection([3, 0, 0, 0]);

                // Select the first and second drawing
                selection.setMultiDrawingSelectionByPosition([[3, 0], [3, 1], [3, 2], [3, 3]]);

                // align the drawing to left
                operationCounter = model.distributeDrawings('horzSlide');
                expect(operationCounter).to.equal(true);

                var allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(4);
                // drawing left position must be on the left side on the page with left 0
                expect(getDrawingFromSelection(allDrawingSelections[0]).left).to.equal(0);

                // drawing must be on right of page with right edge aligned to page right edge
                expect(getDrawingFromSelection(allDrawingSelections[2]).left).to.equal(33866 - 9419); // left = page width - last drawing width

                // align the drawing to left
                operationCounter = model.distributeDrawings('horzDist');
                expect(operationCounter).to.equal(true);

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(4);
                // drawing left position must be on the left side on the page with left 0
                expect(getDrawingFromSelection(allDrawingSelections[0]).left).to.equal(0);

                // drawing must be on right of page with right edge aligned to page right edge
                expect(getDrawingFromSelection(allDrawingSelections[2]).left).to.equal(33866 - 9419); // left = page width - last drawing width

                // align the drawing to left
                operationCounter = model.distributeDrawings('vertSlide');
                expect(operationCounter).to.equal(true);

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(4);
                // drawing top position must be on the top side on the page with top 0
                expect(getDrawingFromSelection(allDrawingSelections[1]).top).to.equal(0);

                expect(getDrawingFromSelection(allDrawingSelections[0]).top).to.equal(4683);
                expect(getDrawingFromSelection(allDrawingSelections[3]).top).to.equal(9367);

                // drawing must be on bottom of page with bottom edge aligned to page bottom edge
                expect(getDrawingFromSelection(allDrawingSelections[2]).top).to.equal(19050 - 5000);

                // align the drawing to left
                operationCounter = model.distributeDrawings('vertDist');
                expect(operationCounter).to.equal(true);

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(4);
                // drawing top position must be on the top side on the page with top 0
                expect(getDrawingFromSelection(allDrawingSelections[1]).top).to.equal(0);

                expect(getDrawingFromSelection(allDrawingSelections[0]).top).to.equal(4683);
                expect(getDrawingFromSelection(allDrawingSelections[3]).top).to.equal(9367);

                // drawing must be on bottom of page with bottom edge aligned to page bottom edge
                expect(getDrawingFromSelection(allDrawingSelections[2]).top).to.equal(19050 - 5000);
            });
        });

        // handling insert of images and tables into place holder drawings
        describe('method "model.setActiveSlideId "', function () {

            it('should exist', function () {
                expect(model).to.respondTo('setActiveSlideId');
            });

            it('should change to the slide with the specified ID', function (done) {

                var changeActiveSlideDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('change:activeslide:done', function () {
                    changeActiveSlideDef.resolve();
                });

                model.setActiveSlideId(slide_3_id); // activating the third slide

                changeActiveSlideDef.always(function () {

                    expect(changeActiveSlideDef.state()).to.equal('resolved');
                    expect(model.getActiveSlideIndex()).to.equal(2); // asynchronous slide formatting has to be finished
                    expect(model.getActiveSlideId()).to.equal(slide_3_id); // asynchronous slide formatting has to be finished

                    activeSlide = model.getSlideById(slide_3_id);

                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide

                    expect(allDrawings.length).to.equal(5);

                    // assigning drawings
                    drawing1 = $(allDrawings[0]);
                    drawing2 = $(allDrawings[1]);
                    drawing3 = $(allDrawings[2]);
                    drawing4 = $(allDrawings[3]);
                    drawing5 = $(allDrawings[4]);

                    expect(drawing3.text()).to.equal(text_para1_drawing3);

                    expect(model.isPlaceHolderDrawing(drawing1)).to.equal(true);
                    expect(model.isPlaceHolderDrawing(drawing2)).to.equal(true);
                    expect(model.isPlaceHolderDrawing(drawing3)).to.equal(true);
                    expect(model.isPlaceHolderDrawing(drawing4)).to.equal(true);
                    expect(model.isPlaceHolderDrawing(drawing5)).to.equal(true);

                    expect(model.isEmptyPlaceHolderDrawing(drawing1)).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(drawing2)).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(drawing3)).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(drawing4)).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(drawing5)).to.equal(true);

                    selection.setTextSelection([2, 0, 0, 0]);  // setting the cursor into the first drawing on the third slide
                    done();
                });
            });
        });

        describe('method "model.handleTemplateImageEvent" ', function () {

            var pictureButton = null;
            var tableButton = null;
            var tableNode = null;
            var defaultTableRowCount = 2;
            var defaultTableCellCount = 10;

            it('should exist', function () {
                expect(model).to.respondTo('handleTemplateImageEvent');
            });

            it('should find buttons for inserting images and tables into the place holder drawings', function () {

                pictureButton = drawing4.find(DOM.IMAGE_TEMPLATE_BUTTON_SELECTOR);
                expect(pictureButton.length).to.equal(1);

                tableButton = drawing5.find(DOM.TABLE_TEMPLATE_BUTTON_SELECTOR);
                expect(tableButton.length).to.equal(1);
            });

            it('should insert a table using the table button inside the place holder drawing', function () {

                model.handleTemplateImageEvent(tableButton);

                expect(drawing5.parent().length).to.equal(0); // drawing no longer in DOM

                // refreshing the affected drawing assigning drawings
                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                expect(allDrawings.length).to.equal(5);
                drawing5 = $(allDrawings[4]);

                expect(model.isEmptyPlaceHolderDrawing(drawing5)).to.equal(false);

                // finding the table inside the drawing
                tableNode = drawing5.find(DOM.TABLE_NODE_SELECTOR);
                expect(tableNode.length).to.equal(1);
                expect(DOM.isTableNodeInTableDrawing(tableNode)).to.equal(true);
            });

            it('should be a table with default number of columns and rows', function () {
                expect(DOM.getTableRows(tableNode).length).to.equal(defaultTableRowCount);
                expect(DOM.getTableCells(tableNode).length).to.equal(defaultTableCellCount);
            });

            it('should be an empty table place holder after removing the drawing', function (done) {

                var // the delete promise
                    delPromise = null,
                    // the selected drawing
                    selectedDrawing = null;

                selection.setTextSelection([2, 4], [2, 5]);  // selecting the drawing that contains the table

                expect(selection.isDrawingSelection()).to.equal(true);

                selectedDrawing = selection.getSelectedDrawing();

                expect(model.isPlaceHolderDrawing(selectedDrawing)).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(selectedDrawing)).to.equal(false);
                expect(Utils.getDomNode(selectedDrawing) === Utils.getDomNode(drawing5)).to.equal(true);

                delPromise = model.deleteSelected({ deleteKey: true }); // deleting the table drawing

                delPromise.always(function () {

                    expect(delPromise.state()).to.equal('resolved');

                    // refreshing the affected drawing assigning drawings
                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(5);
                    drawing5 = $(allDrawings[4]);

                    expect(model.isEmptyPlaceHolderDrawing(drawing5)).to.equal(true);

                    done();
                });
            });

            it('should remove an empty place holder drawing', function (done) {

                var // the delete promise
                    delPromise = null,
                    // the undo promise
                    undoPromise = null;

                selection.setTextSelection([2, 4], [2, 5]);  // selecting the drawing that contains the table

                expect(selection.isDrawingSelection()).to.equal(true);
                expect(Utils.getDomNode(selection.getSelectedDrawing()) === Utils.getDomNode(drawing5)).to.equal(true);

                delPromise = model.deleteSelected({ deleteKey: true }); // deleting the table drawing

                delPromise.always(function () {

                    expect(delPromise.state()).to.equal('resolved');

                    // refreshing the affected drawing assigning drawings
                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(4); // the drawing was really removed

                    // restoring the drawings again
                    undoPromise = model.getUndoManager().undo();
                    undoPromise.always(function () {
                        expect(undoPromise.state()).to.equal('resolved');

                        allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                        expect(allDrawings.length).to.equal(5); // the drawing was really removed
                        drawing5 = $(allDrawings[4]);

                        expect(model.isEmptyPlaceHolderDrawing(drawing5)).to.equal(true);
                        tableButton = drawing5.find(DOM.TABLE_TEMPLATE_BUTTON_SELECTOR);
                        expect(tableButton.length).to.equal(1);

                        done();
                    });
                });
            });

        });

        describe('method "model.insertImageURL"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('insertImageURL');
            });

        });

        describe('method "model.insertTextFrame" ', function () {

            var // a table node inside a drawing of type 'table'
                tableNode = null;

            it('should exist', function () {
                expect(model).to.respondTo('insertTextFrame');
            });

            it('should contain three empty place holder drawings after deleting drawing 2 completely', function (done) {

                var // the delete promise
                    delPromise = null;

                selection.setTextSelection([2, 1], [2, 2]);  // selecting the drawing that contains the table

                expect(selection.isDrawingSelection()).to.equal(true);
                expect(Utils.getDomNode(selection.getSelectedDrawing()) === Utils.getDomNode(drawing2)).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(drawing2)).to.equal(false);

                delPromise = model.deleteSelected({ deleteKey: true }); // deleting the table drawing

                delPromise.always(function () {

                    expect(delPromise.state()).to.equal('resolved');
                    expect(drawing2.parent().length).to.equal(0); // drawing no longer in DOM

                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(5); // the drawing was really removed
                    drawing2 = $(allDrawings[3]);

                    expect(model.isEmptyPlaceHolderDrawing(drawing2)).to.equal(true);

                    done();
                });

            });

            it('should contain four empty place holder drawings after removing the text in drawing 3', function (done) {

                var // the delete promise
                    delPromise = null,
                    // the text length
                    textLength = text_para1_drawing3.length;

                selection.setTextSelection([2, 2], [2, 3]);  // selecting the drawing that contains the table

                expect(selection.isDrawingSelection()).to.equal(true);
                expect(Utils.getDomNode(selection.getSelectedDrawing()) === Utils.getDomNode(drawing3)).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(drawing3)).to.equal(false);
                expect(drawing3.text()).to.equal(text_para1_drawing3);

                // selecting the complete text in the place holder drawing
                selection.setTextSelection([2, 2, 0, 0], [2, 2, 0, textLength]);

                delPromise = model.deleteSelected({ deleteKey: true }); // deleting the table drawing

                delPromise.always(function () {

                    expect(delPromise.state()).to.equal('resolved');
                    expect(drawing3.parent().length).to.equal(1); // the drawing is still in the DOM

                    // allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    // expect(allDrawings.length).to.equal(5); // the drawing was really removed
                    // drawing3 = $(allDrawings[4]);

                    expect(model.isEmptyPlaceHolderDrawing(drawing3)).to.equal(true);

                    done();
                });
            });

            // Inserting three tables using the table button in the top bar, not the place holder image.
            // The selection is inside the title drawing node, that is not affected by the table insertion.
            // Used options for inserting from top bar: { insertTable: true, size: { width: 3, height: 5 } }

            it('should have the selection in the title drawing that is not affected by the table insertion', function () {

                selection.setTextSelection([2, 0, 0, 0]);  // setting the selection in the non empty title drawing

                expect(selection.isDrawingSelection()).to.equal(false);
                expect(Utils.getDomNode(selection.getSelectedTextFrameDrawing()) === Utils.getDomNode(drawing1)).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(drawing1)).to.equal(false);
                expect(drawing1.text()).to.equal(text_para1_drawing1);

                expect(model.isEmptyPlaceHolderDrawing(drawing1)).to.equal(false);
                expect(model.isEmptyPlaceHolderDrawing(drawing2)).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(drawing3)).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(drawing4)).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(drawing5)).to.equal(true);
            });

            it('should insert the first table into the most specialized place holder first', function () {

                var // the options send to insertTextFrame, if the top bar is used for inserting a table
                    options = { insertTable: true, size: { width: 3, height: 5 } };

                // drawing 5 is specialized to tables
                expect(model.isEmptyPlaceHolderDrawing(drawing5)).to.equal(true);

                // inserting the table via top bar
                model.insertTextFrame(options);

                expect(drawing5.parent().length).to.equal(0); // drawing no longer in DOM

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR); // getting all drawings on slide
                expect(allDrawings.length).to.equal(5);
                drawing5 = $(allDrawings[4]);

                // finding the table inside the drawing
                tableNode = drawing5.find(DOM.TABLE_NODE_SELECTOR);
                expect(tableNode.length).to.equal(1);
                expect(DOM.isTableNodeInTableDrawing(tableNode)).to.equal(true);

                expect(model.isEmptyPlaceHolderDrawing(allDrawings[0])).to.equal(false);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[1])).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[2])).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[3])).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[4])).to.equal(false);
            });

            it('should insert the second table into the content body place holder', function () {

                var // the options send to insertTextFrame, if the top bar is used for inserting a table
                    options = { insertTable: true, size: { width: 3, height: 5 } };

                // drawing 2 is a content body place holder
                expect(model.isEmptyPlaceHolderDrawing(drawing2)).to.equal(true);

                // inserting the table via top bar
                model.insertTextFrame(options);

                // expect(drawing2.parent().length).to.equal(0); // drawing no longer in DOM

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                expect(allDrawings.length).to.equal(5);
                drawing2 = $(allDrawings[1]);

                // finding the table inside the drawing
                tableNode = drawing2.find(DOM.TABLE_NODE_SELECTOR);
                expect(tableNode.length).to.equal(1);
                expect(DOM.isTableNodeInTableDrawing(tableNode)).to.equal(true);

                expect(model.isEmptyPlaceHolderDrawing(allDrawings[0])).to.equal(false);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[1])).to.equal(false);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[2])).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[3])).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[4])).to.equal(false);
            });

            it('should insert the third table as new drawing into the slide root, not as place holder', function () {

                var // the options send to insertTextFrame, if the top bar is used for inserting a table
                    options = { insertTable: true, size: { width: 3, height: 5 } },
                    // the new inserted drawing
                    drawing6 = null;

                // there are 5 drawings on the slide
                expect(allDrawings.length).to.equal(5);

                // inserting the table via top bar
                model.insertTextFrame(options);

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                expect(allDrawings.length).to.equal(6); // a new drawing was generated
                drawing6 = $(allDrawings[1]);

                // finding the table inside the drawing
                tableNode = drawing6.find(DOM.TABLE_NODE_SELECTOR);
                expect(tableNode.length).to.equal(1);
                expect(DOM.isTableNodeInTableDrawing(tableNode)).to.equal(true);

                expect(model.isEmptyPlaceHolderDrawing(allDrawings[0])).to.equal(false);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[1])).to.equal(false);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[2])).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[3])).to.equal(true);
                expect(model.isEmptyPlaceHolderDrawing(allDrawings[4])).to.equal(false);
                expect(model.isPlaceHolderDrawing(allDrawings[5])).to.equal(false);
            });

            // pressing undo 5 times to delete the three tables and insert the text into the place holder drawings

            it('first undo should remove the table drawing that is no place holder drawing', function (done) {

                var undoPromise = null;

                // restoring the drawings again
                undoPromise = model.getUndoManager().undo();
                undoPromise.always(function () {
                    expect(undoPromise.state()).to.equal('resolved');

                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(5); // the drawing was removed

                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[0])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[1])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[2])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[3])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[4])).to.equal(false);

                    done();
                });

            });

            it('second undo should convert the table place holder drawing to an empty content body place holder drawing', function (done) {

                var undoPromise = null;

                // restoring the drawings again
                undoPromise = model.getUndoManager().undo();
                undoPromise.always(function () {
                    expect(undoPromise.state()).to.equal('resolved');

                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(5); // no drawing was removed

                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[0])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[1])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[2])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[3])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[4])).to.equal(false);

                    done();
                });
            });

            it('third undo should convert the table place holder drawing to an empty table place holder drawing', function (done) {

                var undoPromise = null;

                // restoring the drawings again
                undoPromise = model.getUndoManager().undo();
                undoPromise.always(function () {
                    expect(undoPromise.state()).to.equal('resolved');

                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(5); // no drawing was removed

                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[0])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[1])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[2])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[3])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[4])).to.equal(true);

                    done();
                });
            });

            it('fourth undo should fill the text into the text place holder drawing', function (done) {

                var undoPromise = null;

                // restoring the drawings again
                undoPromise = model.getUndoManager().undo();
                undoPromise.always(function () {
                    expect(undoPromise.state()).to.equal('resolved');

                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(5);

                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[0])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[1])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[2])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[3])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[4])).to.equal(true);

                    expect($(allDrawings[2]).text()).to.equal(text_para1_drawing3);

                    done();
                });
            });

            // filling the text into the content body place holder drawing
            it('fourth undo should fill the text into the content body place holder drawing', function (done) {

                var undoPromise = null;

                // restoring the drawings again
                undoPromise = model.getUndoManager().undo();
                undoPromise.always(function () {
                    expect(undoPromise.state()).to.equal('resolved');

                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(5);

                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[0])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[1])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[2])).to.equal(false);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[3])).to.equal(true);
                    expect(model.isEmptyPlaceHolderDrawing(allDrawings[4])).to.equal(true);

                    expect($(allDrawings[1]).text()).to.equal(text_para1_drawing2);

                    done();
                });
            });

        });

        // Table modifications and table formatting

        describe('method "model.insertTextFrame with table style" ', function () {

            // a table node inside a drawing of type 'table'
            var tableNode = null;

            it('should exist', function () {
                expect(model).to.respondTo('insertTextFrame');
            });

            it('should change to the slide with the specified ID', function (done) {

                var changeActiveSlideDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('change:activeslide:done', function () {
                    changeActiveSlideDef.resolve();
                });

                model.setActiveSlideId(slide_1_id); // activating the third slide

                changeActiveSlideDef.always(function () {

                    expect(changeActiveSlideDef.state()).to.equal('resolved');
                    expect(model.getActiveSlideIndex()).to.equal(0); // asynchronous slide formatting has to be finished
                    expect(model.getActiveSlideId()).to.equal(slide_1_id); // asynchronous slide formatting has to be finished

                    activeSlide = model.getSlideById(slide_1_id);

                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide

                    expect(allDrawings.length).to.equal(5);

                    // assigning drawings
                    drawing1 = $(allDrawings[0]);
                    drawing2 = $(allDrawings[1]);
                    drawing3 = $(allDrawings[2]);
                    drawing4 = $(allDrawings[3]);
                    drawing5 = $(allDrawings[4]);

                    expect(model.isPlaceHolderDrawing(drawing1)).to.equal(false);
                    expect(model.isPlaceHolderDrawing(drawing2)).to.equal(false);
                    expect(model.isPlaceHolderDrawing(drawing3)).to.equal(false);
                    expect(model.isPlaceHolderDrawing(drawing4)).to.equal(false);
                    expect(model.isPlaceHolderDrawing(drawing5)).to.equal(false);

                    selection.setTextSelection([0, 0, 0, 0]);  // setting the cursor into the first drawing on the third slide
                    done();
                });
            });

            it('should insert a table with the default table style', function (done) {

                // the options send to insertTextFrame, if the top bar is used for inserting a table
                var options = { insertTable: true, size: { width: 2, height: 3 } };
                // the ID of the default table style, if no other style is defined via operation
                var defaultTableStyleId = '{5C22544A-7EE6-4342-B048-85BDC9FD1C3A}';
                // a collector for all cells of the table
                var allCells = null;
                // the jQuerified last cell of the table
                var lastCell = null;
                // the table formatting deferred
                var tableFormattingDef = null;

                // inserting the table via top bar
                model.insertTextFrame(options);

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR); // getting all drawings on slide
                expect(allDrawings.length).to.equal(6);
                drawing5 = $(allDrawings[5]);

                // finding the table inside the drawing
                tableNode = drawing5.find(DOM.TABLE_NODE_SELECTOR);
                expect(tableNode.length).to.equal(1);
                expect(DOM.isTableNodeInTableDrawing(tableNode)).to.equal(true);
                expect(AttributeUtils.getExplicitAttributes(drawing5).styleId).to.equal(defaultTableStyleId);

                expect(DOM.getTableRows(drawing5).length).to.equal(3);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(6);
                lastCell = allCells.last();

                tableFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('table:formatting:done', function () {
                    if (lastCell.css('background-color')) { tableFormattingDef.resolve(); }
                });

                // checking the table cell background (using the DOM node)

                tableFormattingDef.always(function () {
                    expect($(allCells[1]).css('background-color')).to.equal('rgb(79, 129, 189)'); // #4f81bd, accent1
                    expect($(allCells[3]).css('background-color')).to.equal('rgb(185, 205, 229)');
                    expect(lastCell.css('background-color')).to.equal('rgb(222, 231, 243)');
                    done();
                });

            });

            // setting the table grid style to the inserted table

            it('should set the table grid table style', function (done) {

                // the ID of the table grid style
                var tableGridStyleId = '{5940675A-B579-460E-94D1-54222C63F5DA}';
                // a collector for all cells of the table
                var allCells = null;
                // the jQuerified last cell of the table
                var lastCell = null;
                // the table formatting deferred
                var tableFormattingDef = $.Deferred();

                model.setAttributes('table', { styleId: tableGridStyleId }, { clear: true });

                // finding the table inside the drawing
                tableNode = drawing5.find(DOM.TABLE_NODE_SELECTOR);
                expect(tableNode.length).to.equal(1);
                expect(DOM.isTableNodeInTableDrawing(tableNode)).to.equal(true);
                expect(AttributeUtils.getExplicitAttributes(drawing5).styleId).to.equal(tableGridStyleId);

                allCells = DOM.getTableCells(drawing5);
                lastCell = allCells.last();

                model.on('table:formatting:done', function () {
                    if (lastCell.css('background-color') === 'transparent') { tableFormattingDef.resolve(); }
                });

                tableFormattingDef.always(function () {
                    expect($(allCells[1]).css('background-color')).to.equal('transparent');
                    expect($(allCells[3]).css('background-color')).to.equal('transparent');
                    expect(lastCell.css('background-color')).to.equal('transparent');
                    done();
                });

            });

            // set a new (complex) table style, before inserting a new row

            it('should set a complex table style to the selected table and format it correctly', function (done) {

                // the ID of the table grid style
                var tableGridStyleId = '{F5AB1C69-6EDB-4FF4-983F-18BD219EF322}';
                // a collector for all cells of the table
                var allCells = null;
                // the jQuerified last cell of the table
                var lastCell = null;
                // the table formatting deferred
                var tableFormattingDef = $.Deferred();

                model.setAttributes('table', { styleId: tableGridStyleId }, { clear: true });

                // finding the table inside the drawing
                tableNode = drawing5.find(DOM.TABLE_NODE_SELECTOR);
                expect(tableNode.length).to.equal(1);
                expect(DOM.isTableNodeInTableDrawing(tableNode)).to.equal(true);
                expect(AttributeUtils.getExplicitAttributes(drawing5).styleId).to.equal(tableGridStyleId);

                allCells = DOM.getTableCells(drawing5);
                lastCell = allCells.last();

                model.on('table:formatting:done', function () {
                    if (lastCell.css('background-color') !== 'transparent') { tableFormattingDef.resolve(); }
                });

                tableFormattingDef.always(function () {
                    expect($(allCells[1]).css('background-color')).to.equal('rgb(155, 187, 89)');
                    expect($(allCells[3]).css('background-color')).to.equal('rgb(216, 229, 190)');
                    expect(lastCell.css('background-color')).to.equal('rgb(236, 242, 223)');
                    done();
                });

            });

            // adding a row, so that new formatting is triggered
            it('should insert a row into the table and force an update of table formatting', function (done) {

                // a collector for all cells of the table
                var allCells = null;
                // the jQuerified last cell of the table
                var lastCell = null;
                // the table formatting deferred
                var tableFormattingDef = $.Deferred();
                // the background color of the last table cell
                var oldLastCellColor = null;

                // setting the cursor into the table
                selection.setTextSelection([0, 5, 0, 0, 0, 0]);  // setting the cursor into the table

                // checking the position
                var enclosingTable = selection.getEnclosingTable();
                var additionallySelectedDrawing = selection.getSelectedTextFrameDrawing();
                var tableNode = additionallySelectedDrawing.find(DOM.TABLE_NODE_SELECTOR);

                expect(tableNode.length).to.equal(1);
                expect(enclosingTable === drawing5[0]).to.equal(true);
                expect(enclosingTable === additionallySelectedDrawing[0]).to.equal(true);

                // inserting a row behind the first row
                model.insertRow();

                expect(DOM.getTableRows(enclosingTable).length).to.equal(4);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(8);

                lastCell = allCells.last();
                var insertedCell = $(allCells[3]); // second row
                oldLastCellColor = insertedCell.css('background-color');

                model.on('table:formatting:done', function () {
                    if (insertedCell.css('background-color') !== oldLastCellColor) { tableFormattingDef.resolve(); }
                });

                tableFormattingDef.always(function () {
                    expect($(allCells[1]).css('background-color')).to.equal('rgb(155, 187, 89)');
                    expect($(allCells[3]).css('background-color')).to.equal('rgb(216, 229, 190)');
                    expect($(allCells[5]).css('background-color')).to.equal('rgb(236, 242, 223)');
                    expect(lastCell.css('background-color')).to.equal('rgb(216, 229, 190)');
                    done();
                });

            });

            // undo must remove the inserted row again
            it('should insert a row into the table and force an update of table formatting', function (done) {

                // a collector for all cells of the table
                var allCells = null;
                // the undo promise
                var undoPromise = null;

                expect(DOM.getTableRows(drawing5).length).to.equal(4);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(8);

                // removing the inserted row again
                undoPromise = model.getUndoManager().undo();

                undoPromise.always(function () {

                    expect(undoPromise.state()).to.equal('resolved');

                    expect(DOM.getTableRows(drawing5).length).to.equal(3);  // 3 rows left
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(6);  // 6 cells left

                    done();
                });

            });

            // adding a column
            it('should insert a column into the table', function () {

                // a collector for all cells of the table
                var allCells = null;

                // checking 3 rows and 6 cells before inserting the column
                expect(DOM.getTableRows(drawing5).length).to.equal(3);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(6);

                // inserting a row behind the first row
                model.insertColumn();

                // checking 3 rows and 9 cells after inserting the column
                expect(DOM.getTableRows(drawing5).length).to.equal(3);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(9);
            });

            // deleting a column
            it('should delete the selected column(s) in the table', function (done) {

                // a collector for all cells of the table
                var allCells = null;
                // the delete promise
                var deletePromise = null;

                // checking 3 rows and 9 cells before inserting the column
                expect(DOM.getTableRows(drawing5).length).to.equal(3);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(9);

                // deleting the selected column
                deletePromise = model.deleteColumns();

                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    // checking 3 rows and 6 cells after deleting the column
                    expect(DOM.getTableRows(drawing5).length).to.equal(3);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(6);

                    done();
                });
            });

            // undo must insert the deleted column again
            it('should insert a column via undo', function (done) {

                // a collector for all cells of the table
                var allCells = null;
                // the undo promise
                var undoPromise = null;

                // checking 3 rows and 6 cells before undo
                expect(DOM.getTableRows(drawing5).length).to.equal(3);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(6);

                // inserting the removed column again
                undoPromise = model.getUndoManager().undo();

                undoPromise.always(function () {

                    expect(undoPromise.state()).to.equal('resolved');

                    expect(DOM.getTableRows(drawing5).length).to.equal(3);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(9);  // 9 cells in table

                    done();
                });

            });

            // insert a new row
            it('should insert a row into the table', function () {

                // a collector for all cells of the table
                var allCells = null;

                // checking 3 rows and 9 cells before inserting the row
                expect(DOM.getTableRows(drawing5).length).to.equal(3);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(9);

                // inserting a row behind the first row (using convenience function)
                model.insertRow();

                // checking 4 rows and 12 cells after inserting the row
                expect(DOM.getTableRows(drawing5).length).to.equal(4);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(12);
            });

            // checking horizontally merged table cells

            // merging two cells horizontally (not possible via GUI)
            it('should merge two selected cells horizontally', function () {

                // a collector for all cells of the table
                var allCells = null;

                // checking 4 rows and 12 cells before inserting the row
                expect(DOM.getTableRows(drawing5).length).to.equal(4);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(12);

                // setting selection into two cells
                selection.setTextSelection([0, 5, 1, 0, 0, 0], [0, 5, 1, 1, 0, 0]);  // selecting two cells in the second row

                // merging the two selected cells (not possible via GUI)
                model.mergeCells();

                // checking 4 rows and 11 cells after merging the two selected cells
                expect(DOM.getTableRows(drawing5).length).to.equal(4);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(11);
            });

            // inserting a new row with horizontally merged cells
            it('should insert a row with horizontally merged cells into the table', function () {

                // a collector for all cells of the table
                var allCells = null;
                var allRows = null;
                var allCellsInOneRow = null;

                // setting selection into merged cell
                selection.setTextSelection([0, 5, 1, 0, 0, 0]);  // setting selection in the merged cell

                // checking 4 rows and 11 cells before inserting the row
                expect(DOM.getTableRows(drawing5).length).to.equal(4);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(11);

                // inserting a row behind the first row (using convenience function)
                model.insertRow();

                // checking 5 rows and 13 cells after inserting the row
                allRows = DOM.getTableRows(drawing5);
                expect(allRows.length).to.equal(5);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(13);

                // counting the number of cells in each row
                expect($(allRows[0]).find('td').length).to.equal(3);
                expect($(allRows[1]).find('td').length).to.equal(2);
                expect($(allRows[2]).find('td').length).to.equal(2);
                expect($(allRows[3]).find('td').length).to.equal(3);
                expect($(allRows[4]).find('td').length).to.equal(3);

                // checking the length of the first cells in row 2 and 3
                allCellsInOneRow = $(allRows[1]).find('td'); // second row
                expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);

                allCellsInOneRow = $(allRows[2]).find('td'); // third row
                expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
            });

            // insertColumn, deleteColumn and undo for both

            // inserting a new column if selection is in a horizontally merged cell
            it('should insert a valid column handling horizontally merged cells correctly', function () {

                // a collector for all cells of the table
                var allCells = null;
                var allRows = null;
                var allCellsInOneRow = null;

                // setting selection into merged cell
                selection.setTextSelection([0, 5, 1, 0, 0, 0]);  // setting selection in the merged cell

                // checking 5 rows and 13 cells before inserting the column
                expect(DOM.getTableRows(drawing5).length).to.equal(5);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(13);

                // inserting a column
                model.insertColumn();

                // checking 5 rows and 18 cells after inserting the column
                allRows = DOM.getTableRows(drawing5);
                expect(allRows.length).to.equal(5);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(18);

                // counting the number of cells in each row
                expect($(allRows[0]).find('td').length).to.equal(4);
                expect($(allRows[1]).find('td').length).to.equal(3);
                expect($(allRows[2]).find('td').length).to.equal(3);
                expect($(allRows[3]).find('td').length).to.equal(4);
                expect($(allRows[4]).find('td').length).to.equal(4);

                // checking the length of the cells in row 2 and 3
                // -> all new inserted cells must have a length of 1
                allCellsInOneRow = $(allRows[1]).find('td'); // second row
                expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                allCellsInOneRow = $(allRows[2]).find('td'); // third row
                expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);
            });

            // undo must remove the inserted column with merged cells again
            it('should delete a column via undo', function (done) {

                // a collector for all cells of the table
                var allCells = null;
                // the undo promise
                var undoPromise = null;

                // checking 5 rows and 18 cells before undo
                expect(DOM.getTableRows(drawing5).length).to.equal(5);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(18);

                // inserting the removed column again
                undoPromise = model.getUndoManager().undo();

                undoPromise.always(function () {

                    var allRows = null;
                    var allCellsInOneRow = null;

                    expect(undoPromise.state()).to.equal('resolved');

                    // checking 5 rows and 13 cells after inserting the row
                    allRows = DOM.getTableRows(drawing5);
                    expect(allRows.length).to.equal(5);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(13);

                    // counting the number of cells in each row
                    expect($(allRows[0]).find('td').length).to.equal(3);
                    expect($(allRows[1]).find('td').length).to.equal(2);
                    expect($(allRows[2]).find('td').length).to.equal(2);
                    expect($(allRows[3]).find('td').length).to.equal(3);
                    expect($(allRows[4]).find('td').length).to.equal(3);

                    // checking the length of the first cells in row 2 and 3
                    allCellsInOneRow = $(allRows[1]).find('td'); // second row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);

                    allCellsInOneRow = $(allRows[2]).find('td'); // third row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);

                    done();
                });

            });

            // inserting the column again via redo (for preparing tests of delete column)
            it('should insert a column via redo', function (done) {

                // a collector for all cells of the table
                var allCells = null;
                // the undo promise
                var redoPromise = null;

                // checking 5 rows and 18 cells before undo
                expect(DOM.getTableRows(drawing5).length).to.equal(5);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(13);

                // inserting the removed column again
                redoPromise = model.getUndoManager().redo();

                redoPromise.always(function () {

                    var allRows = null;
                    var allCellsInOneRow = null;

                    expect(redoPromise.state()).to.equal('resolved');

                    // checking 5 rows and 18 cells after inserting the column
                    allRows = DOM.getTableRows(drawing5);
                    expect(allRows.length).to.equal(5);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(18);

                    // counting the number of cells in each row
                    expect($(allRows[0]).find('td').length).to.equal(4);
                    expect($(allRows[1]).find('td').length).to.equal(3);
                    expect($(allRows[2]).find('td').length).to.equal(3);
                    expect($(allRows[3]).find('td').length).to.equal(4);
                    expect($(allRows[4]).find('td').length).to.equal(4);

                    // checking the length of the cells in row 2 and 3
                    // -> all new inserted cells must have a length of 1
                    allCellsInOneRow = $(allRows[1]).find('td'); // second row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    allCellsInOneRow = $(allRows[2]).find('td'); // third row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    done();
                });

            });

            // Deleting a column, if table contains horizontally merged cells
            // First row contains 4 cells with length 1
            // Second and third row have 3 cells. The first cell has a lenght of 2.
            // 3 x 4 cells and 2 x 3 cells -> 18 cells

            // Setting selection into first cell of first row -> deleting column
            // -> Merged cells in row 2 and row 3 need to be reduced in length

            it('should delete the first column and reduce the length of horizontally merged cells', function (done) {

                var allCells = null;
                var allRows = null;
                var allCellsInOneRow = null;
                var deletePromise = null;

                // setting selection into first cell of first row -> this cell is not horizontally merged
                selection.setTextSelection([0, 5, 0, 0, 0, 0]);  // setting selection in the not merged cell

                // checking 5 rows and 18 cells before deleting the column
                expect(DOM.getTableRows(drawing5).length).to.equal(5);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(18);

                // deleting the first row
                deletePromise = model.deleteColumns();

                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    // checking 5 rows and 15 cells after deleting the column
                    // -> 3 cells were deleted, 2 cells were reduced in length
                    allRows = DOM.getTableRows(drawing5);
                    expect(allRows.length).to.equal(5);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(15);

                    // counting the number of cells in each row
                    expect($(allRows[0]).find('td').length).to.equal(3);
                    expect($(allRows[1]).find('td').length).to.equal(3);
                    expect($(allRows[2]).find('td').length).to.equal(3);
                    expect($(allRows[3]).find('td').length).to.equal(3);
                    expect($(allRows[4]).find('td').length).to.equal(3);

                    // checking the length of the cells in row 2 and 3
                    // -> all cells must have a length of 1
                    allCellsInOneRow = $(allRows[1]).find('td'); // second row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    allCellsInOneRow = $(allRows[2]).find('td'); // third row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    done();
                });
            });

            // restoring the previous state via undo
            // -> inserting the column again
            it('should insert a column via undo and expand horizontally merged cells again', function (done) {

                // a collector for all cells of the table
                var allCells = null;
                // the undo promise
                var undoPromise = null;

                // checking 5 rows and 18 cells before undo
                expect(DOM.getTableRows(drawing5).length).to.equal(5);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(15);

                // inserting the removed column again
                undoPromise = model.getUndoManager().undo();

                undoPromise.always(function () {

                    var allRows = null;
                    var allCellsInOneRow = null;

                    expect(undoPromise.state()).to.equal('resolved');

                    // checking 5 rows and 18 cells after inserting the column
                    // -> 3 cells were inserted, 2 cells were expanded in length
                    allRows = DOM.getTableRows(drawing5);
                    expect(allRows.length).to.equal(5);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(18);

                    // counting the number of cells in each row
                    expect($(allRows[0]).find('td').length).to.equal(4);
                    expect($(allRows[1]).find('td').length).to.equal(3);
                    expect($(allRows[2]).find('td').length).to.equal(3);
                    expect($(allRows[3]).find('td').length).to.equal(4);
                    expect($(allRows[4]).find('td').length).to.equal(4);

                    // checking the length of the cells in row 2 and 3
                    // -> all new inserted cells must have a length of 1
                    allCellsInOneRow = $(allRows[1]).find('td'); // second row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    allCellsInOneRow = $(allRows[2]).find('td'); // third row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    done();
                });

            });

            // Deleting three rows using a selection range. Reducing the size of the table
            // is also useful for the following unit tests.
            it('should delete three of five rows using a selection range', function (done) {

                var allCells = null;
                var allRows = null;
                var allCellsInOneRow = null;
                var deletePromise = null;

                // checking 5 rows and 18 cells before deleting the row (4, 3, 3, 4, 4 cells)
                allRows = DOM.getTableRows(drawing5);
                expect(allRows.length).to.equal(5);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(18);

                // paragraph formatting is required for setting selection (no content inside paragraph)
                model.implParagraphChangedSync($(allRows[2]).find('div.p'));
                model.implParagraphChangedSync($(allRows[4]).find('div.p'));

                // setting selection into row 3 to row 5
                selection.setTextSelection([0, 5, 2, 0, 0, 0], [0, 5, 4, 0, 0, 0]);

                // deleting the first row
                deletePromise = model.deleteRows();

                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    // checking 2 rows and 7 cells after deleting the rows
                    allRows = DOM.getTableRows(drawing5);
                    expect(allRows.length).to.equal(2);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(7);

                    // counting the number of cells in each row
                    expect($(allRows[0]).find('td').length).to.equal(4);
                    expect($(allRows[1]).find('td').length).to.equal(3);

                    // checking the length of the cells in row 2 and 3
                    // -> all cells must have a length of 1
                    allCellsInOneRow = $(allRows[1]).find('td'); // second row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    done();
                });
            });

            // Deleting a column, if table contains horizontally merged cells and the
            // selection is inside a horizontally merged cell.
            // First row contains 4 cells with length 1
            // Second row has 3 cells. The first cell has a lenght of 2
            // -> 7 cells

            // Setting selection into first cell of first row -> deleting column

            it('should delete the first column and remove horizontally merged cells completely', function (done) {

                var allCells = null;
                var allRows = null;
                var allCellsInOneRow = null;
                var deletePromise = null;

                // checking 5 rows and 18 cells before deleting the column
                allRows = DOM.getTableRows(drawing5);
                expect(allRows.length).to.equal(2);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(7);

                // setting selection into first cell of first row -> this cell is not horizontally merged
                model.implParagraphChangedSync($(allRows[1]).find('div.p'));
                selection.setTextSelection([0, 5, 1, 0, 0, 0]);  // setting selection in the merged cell in the second row

                // deleting the first row
                deletePromise = model.deleteColumns();

                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    // checking 5 rows and 15 cells after deleting the column
                    // -> 2 cells were deleted in 3 rows, 1 cell was deleted in two rows -> minus 8 cells
                    allRows = DOM.getTableRows(drawing5);
                    expect(allRows.length).to.equal(2);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(4);

                    // counting the number of cells in each row
                    expect($(allRows[0]).find('td').length).to.equal(2);
                    expect($(allRows[1]).find('td').length).to.equal(2);

                    // checking the length of the cells in row 2 and 3
                    // -> all cells must have a length of 1
                    allCellsInOneRow = $(allRows[1]).find('td'); // second row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);

                    done();
                });
            });

            // restoring the previous state via undo
            // -> inserting the column again
            it('should insert a column via undo and insert horizontally merged cells again', function (done) {

                // a collector for all cells of the table
                var allCells = null;
                // a collector for all rows of the table
                var allRows = null;
                // the undo promise
                var undoPromise = null;

                // checking 2 rows and 4 cells before undo
                allRows = DOM.getTableRows(drawing5);
                expect(allRows.length).to.equal(2);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(4);

                // inserting the removed column again
                undoPromise = model.getUndoManager().undo();

                undoPromise.always(function () {

                    var allCellsInOneRow = null;

                    expect(undoPromise.state()).to.equal('resolved');

                    // checking 2 rows and 7 cells after inserting the column
                    // -> 2 cells with length 1 were inserted, 1 cells with length 2
                    allRows = DOM.getTableRows(drawing5);
                    expect(allRows.length).to.equal(2);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(7);

                    // counting the number of cells in each row
                    expect($(allRows[0]).find('td').length).to.equal(4);
                    expect($(allRows[1]).find('td').length).to.equal(3);

                    // checking the length of the cells in row 2 and 3
                    // -> all new inserted cells must have a length of 1
                    allCellsInOneRow = $(allRows[1]).find('td'); // second row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    done();
                });

            });

            // Deleting several columns, so that the complete table is removed.
            // Selection is inside a horizontally merged cell and the following cells in the row.
            // First row contains 4 cells with length 1
            // Second row has 3 cells. The first cell has a lenght of 2.
            // -> 7 cells
            //
            // Setting selection into all three cells of the second row -> deleting column
            // -> The complete table is removed

            it('should delete the table by selecting three cells in a row (one is a horizontally merged cell)', function (done) {

                var allCells = null;
                var allRows = null;
                var deletePromise = null;

                // checking that the table drawing was deleted
                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                expect(allDrawings.length).to.equal(6);

                // checking 2 rows and 7 cells before deleting the column
                allRows = DOM.getTableRows(drawing5);
                expect(allRows.length).to.equal(2);
                allCells = DOM.getTableCells(drawing5);
                expect(allCells.length).to.equal(7);

                // setting selection into all three cell of second row
                model.implParagraphChangedSync($(allRows[1]).find('div.p'));
                selection.setTextSelection([0, 5, 1, 0, 0, 0], [0, 5, 1, 2, 0, 0]);  // setting selection in the three cells in the second row

                // deleting the first row
                deletePromise = model.deleteColumns();

                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    // checking that the table drawing was deleted
                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(5); // one drawing was removed

                    done();
                });
            });

            // restoring the previous state via undo
            // -> inserting the table again including the horizontally merged cells
            it('should create the table and the horizontally merged cells via undo', function (done) {

                // the undo promise
                var undoPromise = null;

                // 5 drawings on the slide
                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                expect(allDrawings.length).to.equal(5); // one drawing was removed

                // inserting the removed column again
                undoPromise = model.getUndoManager().undo();

                undoPromise.always(function () {

                    var allCells = null;
                    var allRows = null;
                    var allCellsInOneRow = null;

                    expect(undoPromise.state()).to.equal('resolved');

                    // 6 drawings on the slide
                    allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide
                    expect(allDrawings.length).to.equal(6);

                    // getting the table drawing
                    drawing5 = $(allDrawings[5]);

                    // finding the table inside the drawing
                    tableNode = drawing5.find(DOM.TABLE_NODE_SELECTOR);
                    expect(tableNode.length).to.equal(1);
                    expect(DOM.isTableNodeInTableDrawing(tableNode)).to.equal(true);

                    // checking 2 rows and 7 cells after inserting the column
                    // -> 6 cells with length 1 were inserted, 1 cell with length 2
                    allRows = DOM.getTableRows(drawing5);
                    expect(allRows.length).to.equal(2);
                    allCells = DOM.getTableCells(drawing5);
                    expect(allCells.length).to.equal(7);

                    // counting the number of cells in each row
                    expect($(allRows[0]).find('td').length).to.equal(4);
                    expect($(allRows[1]).find('td').length).to.equal(3);

                    // checking the length of the cells in row 2
                    allCellsInOneRow = $(allRows[1]).find('td'); // second row
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[0])).to.equal(2);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[1])).to.equal(1);
                    expect(Table.getHorizontalCellLength(allCellsInOneRow[2])).to.equal(1);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[0])).to.equal(true);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[1])).to.equal(false);
                    expect(Table.isHorizontallyMergedCell(allCellsInOneRow[2])).to.equal(false);

                    done();
                });
            });

            // TODO:
            // checking table styles with different themes
        });

        // handling drawing lines and connectors and arrows

        describe('method "insertShape"', function () {

            var connector_without_arrow = 0;
            var connector_with_arrow = 0;

            it('should exist', function () {
                expect(model).to.respondTo('insertShape');
            });

            // analyzing the current state
            it('should have 6 drawings on the first slide', function () {

                expect(model.getActiveSlideIndex()).to.equal(0);
                expect(model.getActiveSlideId()).to.equal(slide_1_id);

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide

                expect(allDrawings.length).to.equal(6); // 6 drawings on the slide

            });

            it('should insert one connector without arrow', function () {

                var options = { rectangle: { left: 1000, top: 2000, width: 4000, height: 500 }, presetShape: 'straightConnector1', lineAttrs: { type: 'solid', color: { type: 'rgb', value: '00FF00' }, width: 100 } };

                model.insertShape(options);

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide

                expect(allDrawings.length).to.equal(7); // one connector line inserted

                connector_without_arrow = allDrawings.last();
            });

            it('should insert one connector with arrow', function () {

                var options = { rectangle: { left: 1000, top: 4000, width: 4000, height: 500 }, presetShape: 'straightConnector1', lineAttrs: { type: 'solid', color: { type: 'rgb', value: '00FF00' }, width: 100, headEndType: 'arrow', tailEndType: 'arrow' } };

                model.insertShape(options);

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide

                expect(allDrawings.length).to.equal(8); // one connector line with arrows inserted

                connector_with_arrow = allDrawings.last();
            });

            it('should insert an arrow with increased height for the canvas compared with the connector without arrows', function () {

                var canvasWithoutArrow = DrawingFrame.getCanvasNode(connector_without_arrow);
                var canvasWithArrow = DrawingFrame.getCanvasNode(connector_with_arrow);

                expect(canvasWithoutArrow.length).to.equal(1); // one canvas in each drawing
                expect(canvasWithArrow.length).to.equal(1); // one canvas in each drawing

                var canvasHeightWithoutArrow = parseInt(canvasWithoutArrow.css('height'), 10);
                var canvasHeightWithArrow = parseInt(canvasWithArrow.css('height'), 10);

                // the height of the canvas with arrow must be increased (20px in this case)
                expect(canvasHeightWithoutArrow > 0).to.equal(true);
                expect(canvasHeightWithArrow > 0).to.equal(true);
                expect(canvasHeightWithoutArrow < canvasHeightWithArrow).to.equal(true);
                expect(canvasHeightWithArrow - canvasHeightWithoutArrow).to.equal(20);
            });
        });
    });

});
