/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('globals/apphelper', [
    'globals/sheethelper',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/model/editmodel',
    'io.ox/office/editframework/view/editview',
    'io.ox/office/text/model/model',
    'io.ox/office/text/view/view',
    'io.ox/office/spreadsheet/model/model',
    'io.ox/office/spreadsheet/view/view',
    'io.ox/office/presentation/model/model',
    'io.ox/office/presentation/view/view'
], function (SheetHelper, TimerMixin, EditModel, EditView, TextModel, TextView, SpreadsheetModel, SpreadsheetView, PresentationModel, PresentationView) {

    'use strict';

    // helper function that does nothing but returns the calling context
    function noop() { return this; }

    // class MockBaseView =====================================================

    /**
     * A simple mock view class for unit tests. Mimics the common public API of
     * a Documents view instance.
     *
     * @constructor
     */
    function MockBaseView(app, docModel) {

        var rootNode = $('<div>');

        this.on = this.one = this.off = this.trigger = noop;
        this.getDocModel = _.constant(docModel);
        this.getContentRootNode = _.constant(rootNode);
        this.getHiddenRootNode = _.constant(rootNode);
        this.getAppPaneNode = _.constant(rootNode);
        this.isVisible = _.constant(true);
        this.isBusy = _.constant(false);
        this.getToolBarTabs = _.constant({ handleActiveToolbarTab: _.noop });
        this.createClipboardNode = function () { return $('<div class="clipboard">'); };
        this.grabFocus = noop;
        this.getZoomFactor = _.constant(100);
        this.enterBusy = noop;
        this.leaveBusy = noop;
        this.updateBusyProgress = noop;
        this.destroy = noop;

    } // class MockBaseView

    // class MockTextFrameworkView ============================================

    function MockTextFrameworkView(app, docModel) {

        MockBaseView.call(this, app, docModel);

        this.getDisplayDateString = _.constant('A');
        this.recalculateDocumentMargin = noop;
        this.hideFieldFormatPopup = noop;
        this.isSearchActive = _.constant(false);
        this.clearVisibleDrawingAnchor = noop;
        this.setVisibleDrawingAnchor = noop;
        this.getDisplayDateString = _.constant('A');
        this.yell = noop;

    } // class MockTextView

    // class MockTextView =====================================================

    function MockTextView(app, docModel) {

        MockTextFrameworkView.call(this, app, docModel);

        this.recalculateDocumentMargin = noop;
        this.hideFieldFormatPopup = noop;

    } // class MockTextView

    // class MockSpreadsheetView ==============================================

    function MockSpreadsheetView(app, docModel) {

        MockBaseView.call(this, app, docModel);

    } // class MockSpreadsheetView

    // class MockPresentationView =============================================

    function MockPresentationView(app, docModel) {

        MockTextFrameworkView.call(this, app, docModel);

        this.getSlidePane = _.constant({
            setSidePaneWidthDuringLoad: noop,
            getSlidePaneSelection: $.noop,
            getSlidePaneContainer: $.noop // no return value like 'noop'
        });

        this.enableVerticalScrollBar = noop;
        this.setZoomType = noop;
        this.handleSlideBackgroundVisibility = noop;

    } // class MockTextView

    // class MockBaseController ===============================================

    /**
     * A simple test controller class for unit tests. Mimics the common public
     * API of a Documents controller instance.
     *
     * @constructor
     */
    function MockBaseController() {

        this.on = this.one = this.off = this.trigger = noop;

        this.update = noop;

    } // class MockBaseController

    // static class AppHelper =================================================

    var AppHelper = {};

    /**
     * Creates a simple test application class for unit tests, containing a
     * real document model instance. Mimics the common public API of a
     * Documents application instance.
     *
     * @returns {CoreApplication}
     *  The application containing an instance of the passed model class.
     */
    AppHelper.createApp = function (appBaseName, fileFormat, ModelClass, ViewClass, MockViewClass, operations, options) {

        var docModel = null;
        var docView = null;
        var docController = null;
        var initDef = $.Deferred();
        var importDef = $.Deferred();

        var moduleName = 'io.ox/office/' + appBaseName;
        var app = ox.ui.createApp({ name: moduleName });
        var appWindow = ox.ui.createWindow({ name: moduleName, search: false, chromeless: true });
        var winRootNode = appWindow.nodes.outer;

        appWindow.isBusy = _.constant(false);
        app.setWindow(appWindow);

        app.getModel = function () { return docModel; };
        app.getView = function () { return docView; };
        app.getController = function () { return docController; };

        app.getWindowId = _.constant(winRootNode.attr('id'));
        app.isActive = _.constant(true);
        app.getRootNode = _.constant(winRootNode);

        app.registerDestructor = noop;
        app.onInit = function (callback, context) { initDef.done(callback.bind(context)); };
        app.isImportStarted = function () { return initDef.state() !== 'pending'; };
        app.getImportStartPromise = _.constant(initDef.promise());
        app.isImportFinished = function () { return importDef.state() !== 'pending'; };
        app.getImportFinishPromise = _.constant(importDef.promise());
        app.leaveBusyDuringImport = noop;
        app.getFileFormat = _.constant(fileFormat);
        app.isOOXML = _.constant(fileFormat === 'ooxml');
        app.isODF = _.constant(fileFormat === 'odf');
        app.getFullFileName = _.constant('testfile.xyz');
        app.getShortFileName = _.constant('testfile');
        app.getFileExtension = _.constant('xyz');
        app.getFilePath = _.constant(['path', 'to']);
        app.registerVisibleStateHandler = noop;

        app.setState = noop;
        app.isInternalError = _.constant(false);
        app.setInternalError = function () { throw new Error('internal error'); };
        app.isLocallyModified = _.constant(true);
        app.setUserState = noop;
        app.sendLogMessage = noop;

        app.getUserSettingsValue = _.constant(null);
        app.destroyImageNodes = noop;
        app.addAuthors = noop;
        app.getAuthorColorIndex = _.constant(1);
        app.isMultiSelectionApplication = _.constant(appBaseName !== 'text');
        app.getActiveClients = _.constant([]);

        app.createImageNode = function (url) {
            var def = new $.Deferred();
            var imgNode = $('<img>');
            var handlers = {
                load: function () { def.resolve(imgNode); },
                error: function () { def.reject(); }
            };
            imgNode.one(handlers).attr('src', url);
            return def.always(function () { imgNode.off(handlers); }).promise();
        };

        app.createDebouncedActionsMethodFor = _.constant(noop);

        app.quit = function () {
            docView.destroy();
            docModel.destroy();
            docModel = docView = docController = null;
        };

        app.waitForEvent = function (object, event) {
            var deferred = $.Deferred();
            $(object).on(event, function () { deferred.resolve(); });
            return deferred.promise();
        };

        // synchronous launcher
        app.setLauncher(function () {
            TimerMixin.call(app);
            docModel = new ModelClass(app);
            var CurrViewClass = (options && options.realView) ? ViewClass : MockViewClass;
            docView = new CurrViewClass(app, docModel);
            docController = new MockBaseController();
            initDef.resolve();
            if (_.isArray(operations)) {
                var converters = options && options.converters;
                operations.forEach(function (operation, index) {
                    if (converters && (operation.name in converters)) {
                        operation = _.clone(operation);
                        converters[operation.name](operation);
                    }
                    if (!docModel.invokeOperationHandler(operation)) {
                        console.error(appBaseName + ' operation #' + index + ' (' + operation.name + ') failed', operation);
                        throw new Error('operation failed');
                    }
                });
            }

            importDef.resolve();
        });

        return app;
    };

    /**
     * Generic implementation to create a test application. The creation of the
     * test application will be placed in a before() block, and the application
     * will be destroyed in an after() block. This results in creation and
     * destruction of the test application inside the surrounding describe()
     * block.
     *
     * @param {Function} generator
     *  The generator callback function for the test application. MUST return a
     *  promise that resolves with the test application.
     *
     * @returns {jQuery.Promise}
     *  A promise that resolves with the test application.
     */
    function createTestApp(generator) {

        var testApp = null;
        var def = $.Deferred();

        before(function (done) {
            try {
                generator().done(function (app) {
                    testApp = app;
                    def.resolve(app);
                    done();
                }).fail(function () {
                    def.reject();
                    done(new Error('cannot create application'));
                });
            } catch (ex) {
                done(ex);
            }
        });

        after(function () {
            if (testApp) { testApp.quit(); }
            testApp = null;
        });

        return def.promise();
    }

    /**
     * Creates a test application with a real instance of an EditModel, and
     * applies the passed document operations in order top initialize the
     * document model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will resolve with the application instance containing an
     *  edit model.
     */
    AppHelper.createEditApp = function (fileFormat, operations, options) {
        return createTestApp(function () {
            var app = AppHelper.createApp('edit', fileFormat, EditModel, EditView, MockBaseView, operations, options);
            app.launch();
            return $.when(app);
        });
    };

    /**
     * Creates a test application with a real text document model, and applies
     * the passed document operations in order top initialize the document
     * model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will resolve with the application instance containing a
     *  real text model.
     */
    AppHelper.createTextApp = function (fileFormat, operations, options) {
        return createTestApp(function () {

            var app = AppHelper.createApp('text', fileFormat, TextModel, TextView, MockTextView, operations, options);
            app.stopOperationDistribution = function (callback) {
                if (_.isFunction(callback)) {
                    callback.call(app);
                }
            };
            app.isOperationsBlockActive = _.constant(false);
            app.getClientOperationName = _.constant('author');
            app.launch();

            var docModel = app.getModel();

            return docModel.updateDocumentFormatting().then(function () {
                docModel.setEditMode(true);
                docModel.getSelection().setTextSelection([0, 0]);
                docModel.setDefaultPaperFormat();
                return app;
            });
        });
    };

    var SPREADSHEET_DEFAULT_OPERATIONS = [
        { name: 'setDocumentAttributes', attrs: { document: { fileFormat: 'ooxml', cols: 16384, rows: 1048576, activeSheet: 0 } } },
        { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' }
    ];

    var SPREADSHEET_OPERATION_CONVERTERS = (function () {

        function convertAddress(operation, propName) {
            if (typeof operation[propName] === 'string') {
                operation[propName] = SheetHelper.a(operation[propName]).toJSON();
            }
        }

        function convertRanges(operation, propName) {
            if (typeof operation[propName] === 'string') {
                operation[propName] = SheetHelper.ra(operation[propName]).toJSON();
            }
        }

        return {
            changeCells: function (operation) {
                convertAddress(operation, 'start');
                operation.contents = operation.contents.map(function (rowData) {
                    if (_.isArray(rowData)) { rowData = { c: rowData }; }
                    rowData.c = rowData.c.map(function (cellData) {
                        cellData = (cellData === null) ? {} :
                            (cellData instanceof SheetHelper.ErrorCode) ? { e: cellData.toJSON() } :
                            _.isObject(cellData) ? cellData :
                            { v: cellData };
                        return cellData;
                    });
                    return rowData;
                });
            },
            mergeCells: function (operation) {
                convertAddress(operation, 'start');
                convertAddress(operation, 'end');
            },
            insertTable: function (operation) {
                convertAddress(operation, 'start');
                convertAddress(operation, 'end');
            },
            insertValidation: function (operation) {
                convertRanges(operation, 'ranges');
            }
        };
    }());

    /**
     * Creates a test application with a real spreadsheet document model, and
     * applies the passed document operations in order top initialize the
     * document model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will resolve with the application instance containing a
     *  real spreadsheet model.
     */
    AppHelper.createSpreadsheetApp = function (fileFormat, operations, options) {
        return createTestApp(function () {

            operations = operations || SPREADSHEET_DEFAULT_OPERATIONS;
            options = _.extend({}, options, { converters: SPREADSHEET_OPERATION_CONVERTERS });
            var app = AppHelper.createApp('spreadsheet', fileFormat, SpreadsheetModel, SpreadsheetView, MockSpreadsheetView, operations, options);
            app.launch();

            var docModel = app.getModel();
            return docModel.postProcessImport().then(function () {
                docModel.setEditMode(true);
                return app;
            });
        });
    };

    /**
     * Creates a test application with a real presentation document model, and
     * applies the passed document operations in order top initialize the
     * document model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will resolve with the application instance containing a
     *  real presentation model.
     */
    AppHelper.createPresentationApp = function (fileFormat, operations, options) {
        return createTestApp(function () {

            var app = AppHelper.createApp('presentation', fileFormat, PresentationModel, PresentationView, MockPresentationView, operations, options);
            app.isOperationsBlockActive = _.constant(false);
            app.stopOperationDistribution = noop;
            app.launch();

            var docModel = app.getModel();
            return docModel.updateDocumentFormatting().then(function () {
                docModel.setEditMode(true);
                return app;
            });
        });
    };

    // exports ================================================================

    return AppHelper;

});
