/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('intervalset/intervalset1', [
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/object/sortedarray',
    'io.ox/office/spreadsheet/utils/intervalarray'
], function (IteratorUtils, SortedArray, IntervalArray) {

    'use strict';

    var ZEROS = '000000000';

    // private global functions ===============================================

    /**
     * Converts the passed number to a 10-digit string with leading zeros.
     *
     * @param {Number} number
     *  The number to be converted to a string.
     *
     * @returns {String}
     *  The passed number, as string with 10 digits and leading zeros.
     */
    function fillZeros(number) {
        return (ZEROS + number).slice(-10);
    }

    /**
     * Sorter callback for a SortedArray instance that stores index intervals
     * sorted by their first index.
     *
     * @param {Interval} interval
     *  The index interval to calculate the sort index for.
     *
     * @returns {String}
     *  A unique key for the passed interval that causes to sort intervals by
     *  their start position.
     */
    function firstSorter(interval) {
        return fillZeros(interval.first) + fillZeros(interval.last);
    }

    /**
     * Sorter callback for a SortedArray instance that stores index intervals
     * sorted by their last index.
     *
     * @param {Interval} interval
     *  The index interval to calculate the sort index for.
     *
     * @returns {String}
     *  A unique key for the passed interval that causes to sort intervals by
     *  their end position.
     */
    function lastSorter(interval) {
        return fillZeros(interval.last) + fillZeros(interval.first);
    }

    // class IntervalTree =====================================================

    /**
     * Internal implementation helper for the class IntervalSet that provides
     * optimized methods for insertion, deletion, and access of intervals.
     *
     * This class implements a centered interval tree. Each instance contains
     * four data structures:
     * - A subtree (another instance of this class) for all intervals that end
     *      before the center index of the bounding interval.
     * - A subtree (another instance of this class) for all intervals that start
     *      after the center index of the bounding interval.
     * - An array for all intervals that cover the center point, sorted by the
     *      start position of the intervals.
     * - An array for all intervals that cover the center point, sorted by the
     *      end position of the intervals.
     *
     * When inserting a new interval, it will either be inserted into one of
     * the subtrees, or into both sorted arrays.
     *
     * When removing an interval, it will either be removed from one of the
     * subtrees, or from both sorted arrays.
     *
     * When looking up intervals for a target index, one of the subtrees will
     * be asked (according to the position of the index relative to the center
     * point), and one of the sorted arrays will be evaluated.
     *
     * Effectively, all operations have complexity O(log N), with N being the
     * number of intervals contained in this tree.
     *
     * @constructor
     *
     * @param {Number} first
     *  The first index of the bounding interval (the 'universe' of this tree).
     *  The intervals inserted in this tree MUST NOT exceed this limit to the
     *  left.
     *
     * @param {Number} last
     *  The last index of the bounding interval (the 'universe' of this tree).
     *  The intervals inserted in this tree MUST NOT exceed this limit to the
     *  right.
     */
    function IntervalTree(first, last) {

        // the bounding interval of this tree
        this._first = first;
        this._last = last;
        // the center index of the bounding interval of this tree
        this._center = Math.floor((first + last) / 2);

        // sub-tree (instance of this class) containing all intervals ending before the center index
        this._tree1 = null;
        // sub-tree (instance of this class) containing all intervals starting after the center index
        this._tree2 = null;

        // an array of all intervals covering the center point, sorted by start index
        this._sorted1 = null;
        // an array of all intervals covering the center point, sorted by end index
        this._sorted2 = null;

    } // class IntervalTree

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this tree is empty.
     *
     * @returns {Boolean}
     *  Whether this tree is empty.
     */
    IntervalTree.prototype.empty = function () {
        return !this._tree1 && !this._tree2 && !this._sorted1;
    };

    /**
     * Returns a shallow or deep clone of this interval tree.
     *
     * @param {Boolean|Function} [deep=false]
     *  If set to true or to a function, a deep clone will be created (all
     *  intervals in this set will be cloned); otherwise a shallow clone will
     *  be created (the intervals in this set will be inserted by reference
     *  into the clone). If a function has been passed, it will be invoked for
     *  each cloned interval, receiving the new cloned interval inserted into
     *  the cloned set as first parameter, and the old interval of this set as
     *  second parameter.
     *
     * @returns {IntervalTree}
     *  The cloned interval tree.
     */
    IntervalTree.prototype.clone = function (deep) {

        // create an empty interval tree, and clone the subtrees
        var clone = new IntervalTree(this._first, this._last);
        if (this._tree1) { clone._tree1 = this._tree1.clone(deep); }
        if (this._tree2) { clone._tree2 = this._tree2.clone(deep); }

        // clone the sorted arrays
        if (this._sorted1) {

            // clone the first array (deeply if necessary)
            clone._sorted1 = this._sorted1.clone(!!deep);

            // insert the cloned elements into the second array (do NOT clone deeply again!)
            if (deep) {
                var sorted2 = clone._sorted2 = new SortedArray(lastSorter);
                clone._sorted1.forEach(function (interval) { sorted2.insert(interval, true); });
            } else {
                clone._sorted2 = this._sorted2.clone();
            }

            // invoke the callback function ONCE for each interval in the sorted arrays
            if (_.isFunction(deep)) {
                var newIntervals = clone._sorted1.values();
                var oldIntervals = this._sorted1.values();
                newIntervals.forEach(function (newInterval, index) {
                    deep(newInterval, oldIntervals[index]);
                });
            }
        }

        return clone;
    };

    /**
     * Returns whether this tree contains an interval covering the passed
     * index.
     *
     * @param {Number} index
     *  The index used to look for an interval in this tree.
     *
     * @returns {Boolean}
     *  Whether this tree contains an interval covering the passed index.
     */
    IntervalTree.prototype.containsIndex = function (index) {

        var contains = false;
        var interval = null;

        if (index < this._center) {
            interval = this._sorted1 && this._sorted1.first();
            contains = (interval && (interval.first <= index)) || (this._tree1 && this._tree1.containsIndex(index));
        } else if (this._center < index) {
            interval = this._sorted2 && this._sorted2.last();
            contains = (interval && (index <= interval.last)) || (this._tree2 && this._tree2.containsIndex(index));
        } else {
            // passed index is the center point of this tree: non-empty sorted array contains a matching interval
            contains = this._sorted1;
        }

        return !!contains;
    };

    /**
     * Returns all intervals in this tree that cover the passed index.
     *
     * @param {Number} index
     *  The index used to look for the intervals in this tree.
     *
     * @returns {IntervalArray}
     *  An array with all intervals in this tree that cover the passed index,
     *  in no specific order.
     */
    IntervalTree.prototype.findByIndex = function (index) {

        var intervals = new IntervalArray();
        var desc = null;
        var array = null;

        if (index < this._center) {
            if (this._tree1) { intervals.append(this._tree1.findByIndex(index)); }
            if (this._sorted1) {
                desc = this._sorted1.find(fillZeros(index + 1));
                array = this._sorted1.values();
                intervals.append(desc ? array.slice(0, desc._ai) : array);
            }
        } else if (this._center < index) {
            if (this._tree2) { intervals.append(this._tree2.findByIndex(index)); }
            if (this._sorted2) {
                desc = this._sorted2.find(fillZeros(index));
                if (desc) { intervals.append(this._sorted2.values().slice(desc._ai)); }
            }
        } else if (this._sorted1) {
            // passed index is the center point of this tree: return all elements of the sorted array
            intervals.append(this._sorted1.values());
        }

        return intervals;
    };

    /**
     * Inserts a new index interval into this tree.
     *
     * @param {Interval} interval
     *  The index interval to be inserted into this tree.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing interval with equal start and end position
     *  as the passed interval will be replaced with the new interval. By
     *  default, the existing interval will be kept, and the passed interval
     *  will be ignored. May be important in situations where the intervals in
     *  the tree contain additional user settings in arbitrary properties.
     *
     * @returns {Interval}
     *  A reference to the interval contained in this tree. This may not be the
     *  same as the passed interval, e.g. if an equal interval already exists
     *  in this tree, and the flag 'replace' has not been set.
     */
    IntervalTree.prototype.insert = function (interval, replace) {

        var subtree = null;

        if (interval.last < this._center) {
            subtree = this._tree1 || (this._tree1 = new IntervalTree(this._first, this._center - 1));
            interval = subtree.insert(interval, replace);
        } else if (this._center < interval.first) {
            subtree = this._tree2 || (this._tree2 = new IntervalTree(this._center + 1, this._last));
            interval = subtree.insert(interval, replace);
        } else {
            var sorted1 = this._sorted1 || (this._sorted1 = new SortedArray(firstSorter));
            var sorted2 = this._sorted2 || (this._sorted2 = new SortedArray(lastSorter));
            sorted1.insert(interval, replace);
            interval = sorted2.insert(interval, replace);
        }

        return interval;
    };

    /**
     * Removes the specified index interval from this tree.
     *
     * @param {Interval} interval
     *  The index interval to be removed from this tree.
     *
     * @returns {Interval|Null}
     *  A reference to the removed interval; or null, if this tree does not
     *  contain an interval that is equal to the passed interval.
     */
    IntervalTree.prototype.remove = function (interval) {

        if (interval.last < this._center) {
            interval = this._tree1 ? this._tree1.remove(interval) : null;
            if (interval && this._tree1.empty()) { this._tree1 = null; }
        } else if (this._center < interval.first) {
            interval = this._tree2 ? this._tree2.insert(interval) : null;
            if (interval && this._tree2.empty()) { this._tree2 = null; }
        } else if (this._sorted1) {
            this._sorted1.remove(interval);
            interval = this._sorted2.remove(interval);
            if (this._sorted1.empty()) { this._sorted1 = this._sorted2 = null; }
        }

        return interval;
    };

    // class IntervalSet ======================================================

    /**
     * A unique set of index intervals. This class provides optimized methods
     * for insertion, deletion, and access of index intervals, while keeping
     * the set unique (two different interval objects in this set will never
     * have the same start end end index).
     *
     * @constructor
     *
     * @param {Interval} boundary
     *  The bounding interval that specifies the 'universe' of this set. The
     *  intervals inserted in this set MUST be located completely inside this
     *  bounding interval.
     */
    function IntervalSet(boundary) {

        // the bounding interval of this set
        this._boundary = boundary.clone();
        // the centered interval tree storing all intervals
        this._tree = new IntervalTree(boundary.first, boundary.last);
        // a plain map that stores all intervals by their keys, for fast direct access
        this._map = {};

    } // class IntervalSet

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this set is empty.
     *
     * @returns {Boolean}
     *  Whether this set is empty.
     */
    IntervalSet.prototype.empty = function () {
        return _.isEmpty(this._map);
    };

    /**
     * Returns the bounding index interval passed to the constructor of this
     * instance.
     *
     * @returns {Interval}
     *  The bounding index interval passed to the constructor of this instance.
     */
    IntervalSet.prototype.boundary = function () {
        return this._boundary.clone();
    };

    /**
     * Returns all index intervals contained in this set as an array.
     *
     * @returns {IntervalArray}
     *  All index intervals contained in this set as an array, in no specific
     *  order.
     */
    IntervalSet.prototype.values = function () {
        var intervals = new IntervalArray();
        _.each(this._map, function (interval) { intervals.push(interval); });
        return intervals;
    };

    /**
     * Returns a shallow or deep clone of this interval set.
     *
     * @param {Boolean|Function} [deep=false]
     *  If set to true or to a function, a deep clone will be created (all
     *  intervals in this set will be cloned); otherwise a shallow clone will
     *  be created (the intervals in this set will be inserted by reference
     *  into the clone). If a function has been passed, it will be invoked for
     *  each cloned interval, receiving the new cloned interval inserted into
     *  the cloned set as first parameter, and the old interval of this set as
     *  second parameter.
     *
     * @param {Object} [context]
     *  The calling context used if the 'deep' parameter is a callback.
     *
     * @returns {IntervalSet}
     *  The cloned interval set.
     */
    IntervalSet.prototype.clone = function (deep, context) {

        // start with a new empty interval set
        var clone = new IntervalSet(this._boundary);

        // create a shallow or deep clone of the interval tree, and the own map
        if (deep) {
            // deep mode: collect the cloned intervals in the own map while cloning
            var callback = _.isFunction(deep) ? deep : _.noop;
            clone._tree = this._tree.clone(function (newInterval, oldInterval) {
                callback.call(context, newInterval, oldInterval);
                clone._map[newInterval.key()] = newInterval;
            });
        } else {
            // shallow mode: simple clones of the tree, and the map
            clone._tree = this._tree.clone();
            clone._map = _.clone(this._map);
        }

        return clone;
    };

    /**
     * Returns the interval object from this set that has the exact position of
     * the passed index interval.
     *
     * @param {Interval} interval
     *  An interval to search for.
     *
     * @returns {Interval|Null}
     *  An interval from this collection with the same start and end position
     *  as the passed interval; otherwise null.
     */
    IntervalSet.prototype.findInterval = function (interval) {
        return this._map[interval.key()] || null;
    };

    /**
     * Returns whether this set contains an interval covering the passed index.
     *
     * @param {Number} index
     *  The index used to look for an interval in this set.
     *
     * @returns {Boolean}
     *  Whether this set contains an interval covering the passed index.
     */
    IntervalSet.prototype.containsIndex = function (index) {
        return this._tree.containsIndex(index);
    };

    /**
     * Returns all intervals in this set that cover the passed index.
     *
     * @param {Number} index
     *  The index used to look for the intervals in this set.
     *
     * @returns {IntervalArray}
     *  An array with all intervals in this set that cover the passed index, in
     *  no specific order.
     */
    IntervalSet.prototype.findByIndex = function (index) {
        return this._tree.findByIndex(index);
    };

    /**
     * Invokes the passed callback for each index interval in this set.
     *
     * @param {Function} callback
     *  The callback function to be invoked for each index interval. Receives
     *  the current interval as first parameter.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {IntervalSet}
     *  A reference to this instance.
     */
    IntervalSet.prototype.forEach = function (callback, context) {
        _.each(this._map, function (interval) {
            callback.call(context, interval);
        });
        return this;
    };

    /**
     * Creates an iterator for all index intervals in this set.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the index intervals have been visited completely.
     *      No more elements are available; this result object does not contain
     *      any other properties!
     *  - {Interval} value
     *      The index interval currently visited.
     */
    IntervalSet.prototype.iterator = function () {
        return IteratorUtils.createPropertyIterator(this._map);
    };

    /**
     * Inserts a new index interval into this set.
     *
     * @param {Interval} interval
     *  The index interval to be inserted into this set.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing interval with equal start and end position
     *  as the passed interval will be replaced with the new interval. By
     *  default, the existing interval will be kept, and the passed interval
     *  will be ignored. May be important in situations where the intervals in
     *  the set contain additional user settings in arbitrary properties.
     *
     * @returns {Interval}
     *  A reference to the interval contained in this set. This may not be the
     *  same as the passed interval, e.g. if an equal interval already exists
     *  in this set, and the flag 'replace' has not been set.
     */
    IntervalSet.prototype.insert = function (interval, replace) {
        interval = this._tree.insert(interval, replace);
        this._map[interval.key()] = interval;
        return interval;
    };

    /**
     * Removes the specified index interval from this set.
     *
     * @param {Interval} interval
     *  The index interval to be removed from this set.
     *
     * @returns {Interval|Null}
     *  A reference to the removed interval; or null, if this set does not
     *  contain an interval that is equal to the passed interval.
     */
    IntervalSet.prototype.remove = function (interval) {
        var key = interval.key();
        if (!(key in this._map)) { return null; }
        delete this._map[key];
        return this._tree.remove(interval);
    };

    /**
     * Inserts all index intervals of the passed set into this set.
     *
     * @param {IntervalSet} intervalSet
     *  The interval set to be merged into this interval set.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, intervals in the passed interval set will replace equal
     *  intervals in this set. By default, the existing intervals of this set
     *  will be preferred.
     *
     * @returns {IntervalSet}
     *  A reference to this instance.
     */
    IntervalSet.prototype.merge = function (intervalSet, replace) {
        intervalSet.forEach(function (interval) { this.insert(interval, replace); }, this);
        return this;
    };

    // exports ================================================================

    return IntervalSet;

});
