/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

module.exports = function (grunt) {

    // imports ================================================================

    var _ = require('underscore');
    var Utils = require('../utils/gruntutils')(grunt);

    // task configuration =====================================================

    grunt.config('lhlint', {

        options: {
            text: [
                'This work is provided under the terms of the CREATIVE COMMONS PUBLIC',
                'LICENSE. This work is protected by copyright and/or other applicable',
                'law. Any use of the work other than as authorized under this license',
                'or copyright law is prohibited.',
                '',
                'http://creativecommons.org/licenses/by-nc-sa/2.5/',
                '',
                'Copyright (C) 2016 OX Software GmbH',
                'Mail: info@open-xchange.com',
                '',
                { re: /@author .+ <.+@.+>/, repeat: true }
            ]
        },

        all: {
            src: [
                'apps/**/*.{js,less}',
                'lib/appsuite/apps/themes/*.less',
                'spec/**/*_spec.js',
                'spec/globals/**/*.js',
                'spec/lib/**/*_ext.js',
                'perf/**/*.{js,less}',
                'grunt/tasks/{eslint,formulares,iconsets,lhlint,reqlint}.js',
                'grunt/utils/**/*.js'
            ],
            options: {
                header: '/**',
                prefix: ' * ',
                footer: ' */'
            }
        },

        readme: {
            src: [
                'apps/**/README',
                'resource/**/README'
            ],
            options: {
                header: '#',
                prefix: '# ',
                footer: '#'
            }
        }
    });

    // task implementation ====================================================

    function encodePattern(string) {
        return string.replace(/[$^*+?!:=.|(){}[\]\\]/g, function (c) { return ('\\' + c); });
    }

    function createRegExp(pattern) {
        return new RegExp('^(' + pattern.replace(/\s+$/, '') + ')$');
    }

    function getLinePattern(pattern, prefix) {
        var prefixPattern = prefix ? encodePattern(prefix) : '';
        if (typeof pattern === 'string') {
            return { re: createRegExp(prefixPattern + encodePattern(pattern)), repeat: false };
        }
        if (pattern instanceof RegExp) {
            return { re: createRegExp(prefixPattern + pattern.source), repeat: false };
        }
        if (_.isObject(pattern) && (pattern.re instanceof RegExp)) {
            return { re: createRegExp(prefixPattern + pattern.re.source), repeat: !!pattern.repeat };
        }
        grunt.fatal('Error in license header pattern definition.');
    }

    grunt.registerMultiTask('lhlint', 'Checks the license headers in source files.', function () {

        // read task options for current target
        var options = this.options({
            text: [],
            header: null,
            prefix: null,
            footer: null
        });

        // prepare license pattern settings
        if (!_.isArray(options.text)) { grunt.fatal('Error in license header pattern definition.'); }
        var license = [];
        if (options.header !== null) { license.push(getLinePattern(options.header, null)); }
        options.text.forEach(function (pattern) { license.push(getLinePattern(pattern, options.prefix)); });
        if (options.footer !== null) { license.push(getLinePattern(options.footer, null)); }

        // number of errors found so far in all files
        var errors = 0;

        // process all source files
        this.filesSrc.forEach(function (path) {

            var lines = grunt.file.read(path).split(/\n/);
            var index = 0;

            license.some(function (pattern) {

                if (!pattern.re.test(lines[index])) {
                    Utils.printPathMessage(path, 'Invalid license header line');
                    Utils.dumpErrorInLines(lines, index, 0);
                    errors += 1;
                    return true;
                }

                index += 1;
                if (pattern.repeat) {
                    while (pattern.re.test(lines[index])) { index += 1; }
                }
            });
        });

        // show resulting task message
        var resolveTask = this.async();
        if (errors > 0) {
            grunt.log.error(errors + ' license header ' + grunt.util.pluralize(errors, 'error/errors') + ' found.');
            resolveTask(false);
        } else {
            resolveTask(true);
        }
    });
};
