/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/scheduler', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/debugutils'
], function (Config, Utils, DebugUtils) {

    'use strict';

    // the forced interval time between timer callbacks, in milliseconds
    var INTERVAL = Config.AUTOTEST ? 10 : 20;

    // the maximum duration before a forced interval will be added, in milliseconds
    var SLICE = Config.AUTOTEST ? 1000 : 200;

    // whether to bypass the timer scheduler, and use the browser timers directly
    var DEBUG_DIRECT_TIMERS = Config.getDebugUrlFlag('office:direct-timers');

    // class Timer ============================================================

    /**
     * A simple descriptor object for a scheduled timer.
     *
     * @constructor
     *
     * @property {Function} fn
     *  The callback function to be invoked when the timer fires.
     *
     * @property {String} id
     *  The globally unique identifier of the timer.
     *
     * @property {Number} ts
     *  The timestamp for the target time when the timer should fire.
     *
     * @param {BaseApplication|Null} app
     *  is needed to find the correct window-id
     *
     * @param {String|Null} infoString
     *  infoString for better retracing source
     *
     * @param {String|Null} priority
     *  if assigned selenium can handle better if the timer is needed or not
     */
    function Timer(callback, delay, app, infoString, priority) {

        this.id = _.uniqueId('timer');
        this.ts = _.now() + delay;
        this.fn = callback;
        this.app = app;
        this.infoString = infoString;
        this.priority = priority;

    } // class Timer

    // class TimerQueue =======================================================

    /**
     * An instance of this class contains a queue of timer data objects that
     * will be stored sorted by their target timestamps.
     *
     * @constructor
     */
    function TimerQueue() {

        // The queue of all pending timers requested by the scheduler. The
        // queue is sorted ascending by target timestamps, the first element in
        // the array will be fired first.
        this.queue = [];

        // The identifiers of all pending timers, as flag set. When cancelling
        // a timer, its identifier will be removed from this set, but it will
        // not yet be removed from the queue, in order to prevent performing a
        // linear search for the queue entry. When processing the queue, all
        // its entries that are not contained anymore in this set will be
        // ignored.
        this.set = {};

    } // class TimerQueue

    // private methods --------------------------------------------------------

    /**
     * Removes all leading timers from the queue that have been cancelled in
     * the meantime, i.e. that have been removed from the flag set.
     */
    TimerQueue.prototype._validate = function () {
        var timerData = null;
        while ((timerData = this.queue[0]) && !(timerData.id in this.set)) {
            this.queue.shift();
        }
        this.log();
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this queue is empty.
     *
     * @returns {Boolean}
     *  Whether this queue is empty.
     */
    TimerQueue.prototype.empty = function () {
        return this.queue.length === 0;
    };

    /**
     * Returns the first timer in this queue, but does not remove the timer
     * from this queue.
     *
     * @returns {Timer|Null}
     *  The first timer in this queue; or null if the queue is empty.
     */
    TimerQueue.prototype.peek = function () {
        return this.queue[0] || null;
    };

    /**
     * Removes the first timer from this queue.
     *
     * @returns {Timer|Null}
     *  The first timer in this queue; or null if the queue is empty.
     */
    TimerQueue.prototype.pull = function () {

        // pull the first timer from the queue (nothing more to do, if the queue is empty)
        var timer = this.queue.shift();
        if (!timer) { return null; }

        // remove the timer from the flag set
        delete this.set[timer.id];

        // remove all following cancelled timers from the queue, to ensure that always
        // a valid pending timer is in front of the timer queue
        this._validate();

        return timer;
    };

    /**
     * Inserts a new timer into this queue.
     *
     * @param {Timer} timer
     *  The timer to be inserted into this queue.
     *
     * @returns {Boolean}
     *  Whether the timer has been inserted in front of the queue.
     */
    TimerQueue.prototype.insert = function (timer) {

        // find the insertion index (after other pending timers with the same timestamp)
        var index = Utils.findLastIndex(this.queue, function (currData) {
            return currData.ts <= timer.ts;
        }, { sorted: true }) + 1;

        // insert the new timer into the queue, and update the flag set
        this.queue.splice(index, 0, timer);
        this.set[timer.id] = true;

        // return whether the timer has been inserted in front of the queue
        return index === 0;
    };

    /**
     * Removes the specified timer from this queue.
     *
     * @param {String} timerId
     *  The identifier of the timer to be removed from this queue.
     *
     * @returns {Boolean}
     *  Whether the timer has been found and removed from the front of the
     *  queue.
     */
    TimerQueue.prototype.remove = function (timerId) {

        // detect whether the timer is located infront of the queue
        var leading = !this.empty() && (this.queue[0].id === timerId);

        // remove the identifier from the flag set, but not from the queue (no linear search)
        delete this.set[timerId];

        // remove all leading cancelled timers from the queue, to ensure that always
        // a valid pending timer is in front of the timer queue
        if (leading) { this._validate(); }

        // return whether the timer was located in front of the queue
        return leading;
    };

    TimerQueue.prototype.log = function () {
        DebugUtils.logQueue(this.queue);
    };

    // private global functions ===============================================

    /**
     * Creates a system timer for the next pending timer in a timer queue that
     * will invoke the callback function associated to that timer.
     *
     * @param {TimerQueue} queue
     *  The timer queue whose first pending timer will be fired next.
     */
    var restartTimerLoop = (function () {

        // identifier of the system timer that will fire the next pending timer in the queue
        var sysTimerId = null;
        // the end timestamp of the current time slice where timer will be fired immediately
        var currSliceEnd = -1;
        // the end timestamp of the current time slice where timer will be fired immediately
        var lastTimerEnd = -1;
        // true while an external timer callback is running (needed to detect re-entrance)
        var firing = false;

        // implementation of the method restartTimerLoop() returned from local scope
        function restartTimerLoop(queue) {

            // do nothing if a timer is currently firing, i.e. a timer is being created from the
            // callback of another timer (this means that this function is currently running and
            // will create another system timer before leaving)
            if (firing) { return; }

            // cancel a running system timer (may have a wrong delay time, after inserting a short timer)
            if (sysTimerId) {
                window.clearTimeout(sysTimerId);
                sysTimerId = null;
            }

            queue.log();

            // get the first pending timer from the queue, but do not remove it yet
            // (a faster timer may be added before the queued timer fires)
            var timer = queue.peek();
            if (!timer) { return; }

            // calculate the delay time for the next timer according to its desired timestamp
            var t0 = _.now();
            var delay = Math.max(0, timer.ts - t0);

            // if the timer wants to fire in the current synchronous time slice, and the current
            // timestamp is still inside the time slice, the timer will be fired without extra delay
            if ((currSliceEnd <= t0) || (currSliceEnd <= timer.ts)) {
                delay = Math.max(delay, lastTimerEnd + INTERVAL - t0);
            }

            // create the system timer that will fire the next pending timer from the queue
            sysTimerId = window.setTimeout(function () {

                // immediately reset the identifier to indicate that no system timer is running anymore
                sysTimerId = null;

                // extract the leading timer data from the queue, and update the flag set
                timer = queue.pull();

                // start a new synchronous time slice, if the last one has finished
                var t0 = _.now();
                if (currSliceEnd < t0) {
                    currSliceEnd = t0 + SLICE;
                }

                // execute the callback function associated to the timer
                // bug 48203: also clean up after callback has thrown
                try {
                    firing = true;
                    timer.fn();
                } finally {
                    firing = false;
                    lastTimerEnd = _.now();

                    // create a new system timer for the next pending timer in the queue
                    restartTimerLoop(queue);

                    queue.log();
                }

            }, delay);
        }

        return restartTimerLoop;
    }());

    // singletons =============================================================

    /**
     * The queue of all pending timers requested by the scheduler. The queue is
     * sorted ascending by target timestamps, the first element in the array
     * will be fired first.
     */
    var pendingQueue = new TimerQueue();

    // static class Scheduler =================================================

    /**
     * Synchronizes multiple timer requests globally. All timer requests will
     * be inserted into an internal queue. The scheduler will ensure delay
     * intervals between timer callbacks if necessary, in order to allow the
     * browser to process its event queue, rendering queue, and to process
     * network responses.
     */
    var Scheduler = {};

    // constants --------------------------------------------------------------

    /**
     * The forced interval time between timer callbacks, in milliseconds.
     *
     * @constant
     *
     * @type {Number}
     */
    Scheduler.INTERVAL = INTERVAL;

    /**
     * The maximum duration before a forced interval will be added, in
     * milliseconds.
     *
     * @constant
     *
     * @type {Number}
     */
    Scheduler.SLICE = SLICE;

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the scheduler is empty, i.e. whether it does not contain
     * any pending timers.
     *
     * @returns {Boolean}
     *  Whether the scheduler is empty.
     */
    Scheduler.empty = function () {
        return pendingQueue.empty();
    };

    /**
     * This method is the replacement for the method window.setTimeout().
     *
     * @param {Function} callback
     *  The callback function that will be invoked when the timer fires.
     *
     * @param {Number} delay
     *  The delay time, in milliseconds.
     *
     * @param {BaseApplication|Null} app
     *  is needed to find the correct window-id
     *
     * @param {String|Null} infoString
     *  infoString for better retracing source
     *
     * @param {String|Null} priority
     *  if assigned selenium can handle better if the timer is needed or not
     *
     * @returns {String}
     *  A globally unique identifier for the timer. Intended to be used with
     *  the method Scheduler.clearTimeout().
     */
    Scheduler.setTimeout = DEBUG_DIRECT_TIMERS ? window.setTimeout.bind(window) : function (callback, delay, app, infoString, priority) {

        // the data object with all settings of the new timer
        var timer = new Timer(callback, delay, app, infoString, priority);
        // insert the new timer into the queue
        var leading = pendingQueue.insert(timer);

        // if the new timer has been inserted in front of the queue, a system timer needs to
        // be started that will fire the new timer (either the queue was empty before, or a
        // longer-running system timer will be cancelled)
        if (leading) { restartTimerLoop(pendingQueue); }

        pendingQueue.log();

        // return the identifier of the timer to the caller
        return timer.id;
    };

    /**
     * This method is the replacement for the method window.clearTimeout().
     *
     * @param {String} timerId
     *  The identifier of a timer that has been created with the method
     *  Scheduler.setTimeout().
     */
    Scheduler.clearTimeout = DEBUG_DIRECT_TIMERS ? window.clearTimeout.bind(window) : function (timerId) {

        // remove the timer from the queue, detect whether it was the leading entry inthe queue
        var leading = pendingQueue.remove(timerId);

        // create a new system timer that will fire the (new) first available timer request in the queue
        if (leading) { restartTimerLoop(pendingQueue); }

        pendingQueue.log();
    };

    // exports ================================================================

    return Scheduler;

});
