/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/iteratorutils', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // static class IteratorUtils =============================================

    var IteratorUtils = {};

    // constants --------------------------------------------------------------

    /**
     * A dummy iterator that is always done.
     *
     * @constant
     */
    IteratorUtils.ALWAYS_DONE_ITERATOR = { next: _.constant({ done: true }) };

    // static methods ---------------------------------------------------------

    /**
     * Creates a simple iterator that will visit a single result only.
     *
     * @param {Object} result
     *  The result to be returned by the first invocation of the method next()
     *  of the iterator. MUST contain a property 'value'. The property 'done'
     *  with value false will be added to the result object automatically.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns the passed
     *  result object on first call.
     */
    IteratorUtils.createSingleIterator = function (result) {

        // add the 'done' property to the passed result
        result.done = false;

        // return the implementation of the iterator
        return { next: function () {
            this.next = IteratorUtils.ALWAYS_DONE_ITERATOR.next;
            return result;
        } };
    };

    /**
     * Creates an iterator that will generate an increasing (or decreasing)
     * sequence of integer indexes upon calling its next() method.
     *
     * @param {Number} size
     *  The length of the index sequence to be generated. The iterator will
     *  generate all indexes starting from zero, that are less than this value
     *  (unless the option 'offset' has been used, see below).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.offset=0]
     *      If set to any number, it wiull be added to the indexes generated by
     *      this iterator.
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the indexes will be generated in reversed order.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the index sequence has been generated completely.
     *      No more indexes are available; this result object does not contain
     *      any other properties!
     *  - {Number} value
     *      The current index. This property will be omitted, if the iterator
     *      is done (see property 'done').
     */
    IteratorUtils.createIndexIterator = function (size, options) {

        // the offset to be added to the generated indexes
        var offset = Utils.getNumberOption(options, 'offset', 0);
        // whether to iterate in reversed order
        var reverse = Utils.getBooleanOption(options, 'reverse', false);
        // the next index to be visited
        var index = reverse ? (size - 1) : 0;

        // return the implementation of the iterator
        return { next: function () {
            var result = (reverse ? (index >= 0) : (index < size)) ? { done: false, value: index + offset } : { done: true };
            index += reverse ? -1 : 1;
            return result;
        } };
    };

    /**
     * Creates an iterator that will generate an increasing (or decreasing)
     * sequence of integer indexes inside the specified closed interval upon
     * calling its next() method.
     *
     * @param {Any} first
     *  The first index to be generated by the returned iterator.
     *
     * @param {Any} last
     *  The last index to be generated by the returned iterator.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the indexes will be generated in reversed order.
     *      The index passed to the parameter 'last' will be generated first.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the index sequence has been generated completely.
     *      No more indexes are available; this result object does not contain
     *      any other properties!
     *  - {Number} value
     *      The current index. This property will be omitted, if the iterator
     *      is done (see property 'done').
     */
    IteratorUtils.createIntervalIterator = function (first, last, options) {

        // whether to iterate in reversed order
        var reverse = Utils.getBooleanOption(options, 'reverse', false);
        // the next index to be visited
        var index = reverse ? last : first;

        // return the implementation of the iterator
        return { next: function () {
            var result = (reverse ? (index >= first) : (index <= last)) ? { done: false, value: index } : { done: true };
            index += reverse ? -1 : 1;
            return result;
        } };
    };

    /**
     * Creates an iterator that visits the elements of an array, or another
     * array-like object, upon calling its next() method.
     *
     * @param {Array} array
     *  The array, or another array-like object.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the array will be visited in reverse order, from
     *      its last to its first element.
     *  @param {Number} [options.begin]
     *      If specified, only the array elements with an index greater than or
     *      equal to this value (in reversed mode: with an index less than or
     *      equal to this value) will be visited. If omitted, all elements from
     *      the beginning (in reverse mode: from the end) of the array will be
     *      visited.
     *  @param {Number} [options.end]
     *      If specified, only the array elements with an index less than this
     *      value (in reverse mode: with an index greater than this value) will
     *      be visited (half-open interval!). If omitted, all elements to the
     *      end (in reverse mode: to the beginning) of the array will be
     *      visited.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the array has been visited completely. No more
     *      array elements are available; this result object does not contain
     *      any other properties!
     *  - {Any} value
     *      The value of the array element currently visited. This property
     *      will be omitted, if the iterator is done (see property 'done').
     *  - {Number} index
     *      The index of the array element currently visited. This property
     *      will be omitted, if the iterator is done (see property 'done').
     */
    IteratorUtils.createArrayIterator = function (array, options) {

        // whether to iterate in reversed order
        var reverse = Utils.getBooleanOption(options, 'reverse', false);
        // the first array index
        var begin = Utils.getIntegerOption(options, 'begin', reverse ? (array.length - 1) : 0);
        // one beyond the last array index
        var end = Utils.getIntegerOption(options, 'end', reverse ? -1 : array.length);
        // the index of the next array element to be visited
        var index = begin;

        // return the implementation of the iterator
        return { next: function () {
            if (reverse ? (index <= end) : (index >= end)) { return { done: true }; }
            var result = { done: false, value: array[index], index: index };
            index += reverse ? -1 : 1;
            return result;
        } };
    };

    /**
     * Creates an iterator that visits the properties of an object upon calling
     * its next() method.
     *
     * @param {Object} object
     *  The object whose properties will be visited by the iterator.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the object has been visited completely. No more
     *      property values are available; this result object does not contain
     *      any other properties!
     *  - {Any} value
     *      The value of the object property currently visited. This property
     *      will be omitted, if the iterator is done (see property 'done').
     *  - {String} key
     *      The name of the object property currently visited. This property
     *      will be omitted, if the iterator is done (see property 'done').
     */
    IteratorUtils.createObjectIterator = function (object) {

        // the property names of the object, as array
        var keys = _.keys(object);
        // the index of the next key to be visited
        var index = 0;
        // number of object properties
        var length = keys.length;

        // return the implementation of the iterator
        return { next: function () {
            while ((index < length) && !(keys[index] in object)) { index += 1; }
            if (index >= length) { return { done: true }; }
            var result = { done: false, value: object[keys[index]], key: keys[index] };
            index += 1;
            return result;
        } };
    };

    /**
     * Creates a new iterator that provides the results of a generator callback
     * function.
     *
     * @param {Function} generate
     *  A callback function that generates the result objects for the iterator.
     *  Receives the following parameters:
     *  (1) {Number} index
     *      A zero-based invocation counter.
     *  If the return value of the callback function is an object, it will be
     *  set as result of the iterator step. If the object contains a property
     *  'done' set to true, the iterator stops. Otherwise, the property 'done'
     *  set to false will be added to the result object automatically. If the
     *  return value is not an object, the iterator will skip the result, and
     *  will continue to fetch results from the callback function.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Object}
     *  An iterator wrapping the passed generator callback function.
     */
    IteratorUtils.createWhileIterator = function (generate, context) {

        // the index passed to the callback function
        var index = 0;

        // return an anonymous object with next() method
        return { next: function () {

            // loop until the callback returns a result object
            while (true) {

                // fetch a new result from the generator callback function
                var result = generate.call(context, index);
                index += 1;

                // skip non-object results
                if (!_.isObject(result)) { continue; }

                // stop iteration, if the callback function returns an object with 'done' set to true
                if (result.done === true) {
                    this.next = IteratorUtils.ALWAYS_DONE_ITERATOR.next;
                    return this.next();
                }

                // add the 'done' property (callback is allowed to omit it)
                result.done = false;
                return result;
            }
        } };
    };

    /**
     * Creates a new iterator that transforms (and optionally filters) the
     * values of the passed source iterator.
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function|String} transform
     *  A callback function that receives the result objects of the source
     *  iterator, and returns the modified result object, or a new result
     *  object, or a falsy value to filter the result. Alternatively, can be
     *  the name of a property in the result objects returned by the iterator
     *  that will be used as iterator value. A callback function receives the
     *  following parameters:
     *  (1) {Any} value
     *      The current value received from the source iterator.
     *  (2) {Object} result
     *      The current and complete result object of the source iterator, with
     *      the standard properties 'done' (always false) and 'value', and all
     *      other non-standard value properties returned by the source
     *      iterator. Can be modified in-place and returned.
     *  If the return value is an object, it will be set as result of the
     *  iterator step. If the object contains a property 'done' set to true,
     *  the transformation iterator stops regardless if the source iterator is
     *  already done. Otherwise, the property 'done' set to false will be added
     *  to the result object automatically. If the return value is not an
     *  object, the transformation iterator will skip the result.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Object}
     *  An iterator wrapping the passed source iterator, and transforming the
     *  results of the source iterator.
     */
    IteratorUtils.createTransformIterator = function (iterator, transform, context) {

        // convert a property name passed as transformator to a function that replaces the 'value' property
        if (_.isString(transform)) {
            transform = (function (prop) {
                return function (obj, result) {
                    result.value = obj[prop];
                    return result;
                };
            }(transform));
        }

        // create an iterator that loops until a valid result has been created, or the source iterator is done
        return IteratorUtils.createWhileIterator(function () {
            var result = iterator.next();
            return result.done ? result : transform.call(context, result.value, result);
        });
    };

    /**
     * Creates a new iterator that filters the values of the passed source
     * iterator according to a truth test.
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function|String} filter
     *  A callback predicate function that receives the result objects of the
     *  source iterator, and returns whether the resulting filter iterator will
     *  visit that result; or the name of a property in the result objects
     *  returned by the iterator that will be checked for truthness. A callback
     *  function receives the following parameters:
     *  (1) {Any} value
     *      The current value received from the source iterator.
     *  (2) {Object} result
     *      The current and complete result object of the source iterator, with
     *      the standard properties 'done' (always false) and 'value', and all
     *      other non-standard value properties returned by the source
     *      iterator.
     *  If the return value is truthy, the resulting filter iterator will visit
     *  the passed result.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Object}
     *  An iterator wrapping the passed source iterator, and that visits only
     *  the values of the source iterator that pass a truth test. The filter
     *  iterator will return the original result objects of the source iterator
     *  including all non-standard value properties.
     */
    IteratorUtils.createFilterIterator = function (iterator, filter, context) {

        // convert a property name passed as predicate to a function
        if (_.isString(filter)) { filter = _.property(filter); }

        // create an iterator that loops until a matching result has been found, or the source iterator is done
        return IteratorUtils.createWhileIterator(function () {
            var result = iterator.next();
            return result.done ? result : filter.call(context, result.value, result) ? result : null;
        });
    };

    /**
     * Creates a new iterator that tries to reduce the number of steps visited
     * by the source iterator, by combining the result values of consecutive
     * iterator steps.
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function} reduce
     *  A callback function that receives two consecutive result objects of the
     *  source iterator, and returns a new result object, if the source results
     *  can be combined. Receives the following parameters:
     *  (1) {Object} result1
     *      The first result object of the source iterator, with the standard
     *      properties 'done' (always false) and 'value', and all other
     *      non-standard value properties returned by the source iterator.
     *  (2) {Object} result2
     *      The second result object of the source iterator, with the standard
     *      properties 'done' (always false) and 'value', and all other
     *      non-standard value properties returned by the source iterator.
     *  If the return value is an object, it will be used as combined iterator
     *  result, and the reducing iterator continues to try combining subsequent
     *  results of the source iterator. If the returned object contains a
     *  property 'done' set to true, the reducing iterator stops regardless if
     *  the source iterator is already done. Otherwise, the property 'done' set
     *  to false will be added to the result object automatically. If the
     *  return value is not an object, the passed results cannot be combined,
     *  and the reducing iterator visits the first result. Either of the passed
     *  result objects can be modified in-place, and can be returned by the
     *  callback function.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Object}
     *  An iterator wrapping the passed source iterator, and that visits only
     *  the values of the source iterator that pass a truth test. The filter
     *  iterator will return the original result objects of the source iterator
     *  including all non-standard value properties.
     */
    IteratorUtils.createReduceIterator = function (iterator, reduce, context) {

        // cached iterator result that may be combined with its successors
        var pendingResult = iterator.next();

        // replaces the pending result with the passed result, and returns the old pending result
        function replacePendingResult(nextResult) {
            var result = pendingResult;
            pendingResult = nextResult;
            return result;
        }

        // repeat combining results as often as possible
        return IteratorUtils.createWhileIterator(function () {

            // no more data left for the iterator
            if (pendingResult.done) { return pendingResult; }

            // fetch the next result from the iterator; return the cached result from preceding
            // call, if the original iterator is done
            var nextResult = iterator.next();
            if (nextResult.done) { return replacePendingResult(nextResult); }

            // try to combine two consecutive results via the callback function
            var reducedResult = reduce.call(context, pendingResult, nextResult);

            // the results cannot be combined: cache the new result, and return the old cached result
            if (!_.isObject(reducedResult)) {
                return replacePendingResult(nextResult);
            }

            // the callback may stop the iteration by returning a 'done' object
            if (reducedResult.done === true) {
                return replacePendingResult(reducedResult);
            }

            // use the new result as pending result, and continue with the next iterator result
            pendingResult = reducedResult;
        });
    };

    /**
     * Creates a new iterator by nesting two source iterators. For each single
     * step of the outer iterator, a new inner iterator will be created by
     * invoking the passed generator function.
     *
     * @param {Object} outerIterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function} innerGenerator
     *  A callback function that creates and returns an iterator for each step
     *  of the outer iterator. Receives the following parameters:
     *  (1) {Any} outerValue
     *      The current value received from the outer iterator.
     *  (2) {Object} outerResult
     *      The current result object of the outer iterator, with the standard
     *      properties 'done' (always false) and 'value', and all other
     *      non-standard value properties returned by the outer iterator.
     *  Must return an object implementing the standard EcmaScript iterator
     *  protocol, or the value null. The returned iterator may be empty (it may
     *  return an object with the property 'done' set to true on first
     *  invocation). In the latter cases (value null, or empty iterator), the
     *  outer iterator will be stepped again until a non-empty inner iterator
     *  has been found.
     *
     * @param {Function} [transformResults]
     *  A callback function that returns the result object by combining the
     *  results of the outer and inner iterators. If omitted, the results of
     *  the inner iterators will be returned as received. The callback function
     *  receives the following parameters:
     *  (1) {Object} outerResult
     *      The current result object of the outer iterator, with the standard
     *      properties 'done' (always false) and 'value', and all other
     *      non-standard value properties returned by the source iterator. MUST
     *      NOT be modified!
     *  (2) {Object} innerResult
     *      The current result object of the inner iterator, with the standard
     *      properties 'done' (always false) and 'value', and all other
     *      non-standard value properties returned by the source iterator. Can
     *      be modified in-place and returned.
     *  If the return value is an object, it will be set as result of the
     *  iterator step. If the object contains a property 'done' set to true,
     *  the nested iterator stops regardless if any of the source iterators are
     *  already done. Otherwise, the property 'done' set to false will be added
     *  to the result object automatically. If the return value is not an
     *  object, the nested iterator will skip the result.
     *
     * @param {Object} [context]
     *  The calling context for the callback functions.
     *
     * @returns {Object}
     *  An iterator wrapping the passed outer and inner iterators. The result
     *  object will be the object returned by the transformation callback
     *  function passed to this method.
     */
    IteratorUtils.createNestedIterator = function (outerIterator, innerGenerator, transformResults, context) {

        // the current result of the outer iterator (used multiple times during inner iterator)
        var outerResult = outerIterator.next();
        // the current inner iterator
        var innerIterator = null;

        // return an anonymous object with next() method
        return { next: function () {

            // repeat creating a new inner iterator (it may be empty for specific values of the outer iterator)
            while (!outerResult.done) {

                // create a new inner iterator on demand (callback function may return null to signal an empty iterator)
                if (!innerIterator) {
                    innerIterator = innerGenerator.call(context, outerResult.value, outerResult) || IteratorUtils.ALWAYS_DONE_ITERATOR;
                }

                // the result from the inner iterator
                var innerResult = innerIterator.next();

                // step outer iterator, until an inner iterator returns a value
                if (innerResult.done) {
                    outerResult = outerIterator.next();
                    innerIterator = null;
                    continue;
                }

                // create the result for the nested iterator, skip non-object results
                var result = transformResults ? transformResults.call(context, outerResult, innerResult) : innerResult;
                if (!_.isObject(result)) { continue; }

                // the callback may stop the iteration by returning a 'done' object
                if (result.done === true) {
                    outerResult.done = true;
                    break;
                }

                // add the 'done' property (callback is allowed to omit it)
                result.done = false;
                return result;
            }

            // outer iterator reached its end
            return { done: true };
        } };
    };

    /**
     * Creates a combined iterator that visits the results provided by the
     * passed iterators one after the other.
     *
     * @param {Object} [iterator1 ...]
     *  Each parameter MUST be an object implementing the standard EcmaScript
     *  iterator protocol (each iterator must provide a method next() that
     *  returns an object with the properties 'done' and 'value').
     *
     * @returns {Object}
     *  A new iterator that visits the result objects of the passed iterators
     *  in a single run.
     */
    IteratorUtils.createSerialIterator = function () {

        // special handling for less than two source iterators
        switch (arguments.length) {
            case 0: return IteratorUtils.ALWAYS_DONE_ITERATOR;
            case 1: return arguments[0];
        }

        // create a nested iterator that combines the iterators via an array iterator
        var iterator = IteratorUtils.createArrayIterator(arguments);
        return IteratorUtils.createNestedIterator(iterator, _.identity);
    };

    /**
     * Creates an iterator that visits the values provided by the passed
     * iterators in a specific order, according to some offset or sorting index
     * for the values of all iterators.
     *
     * @param {Array<Object>} iterators
     *  An array of objects implementing the standard EcmaScript iterator
     *  protocol (each iterator must provide a method next() that returns an
     *  object with the properties 'done' and 'value').
     *
     * @param {Function|Array<Function>} indexers
     *  The callback functions that convert the results of the source iterators
     *  to an offset or sorting index that will be used to decide which result
     *  of the iterators will be visited next. If set to a single function, it
     *  will be used for all source iterators. Otherwise, the array MUST have
     *  the same length as the array containing the source iterators. Each
     *  callback function receives the following parameters:
     *  (1) {Any} value
     *      The current value of the respective source iterator.
     *  (2) {Object} result
     *      The complete result object of the respective source iterator.
     *  (3) {Number} index
     *      The array index of the source iterator (may be useful for a single
     *      callback function used for all iterators).
     *  The callback functions MUST return a value that can be compared with
     *  the standard JavaScript comparison operators (numbers or strings). The
     *  return values MUST grow strictly for all results per source iterator.
     *
     * @param {Object} [context]
     *  The calling context for the indexer callback functions.
     *
     * @returns {Object}
     *  A new iterator that visits the result objects of the passed iterators.
     *  The iterator will pick the result of the iterator with the smallest
     *  sort index, according to the return values of the indexers. The result
     *  objects of the synchronized iterator will be shallow copies of the
     *  original iterator results, with the following additional properties:
     *  - {Number} iteratorIndex
     *      The array index of the source iterator.
     */
    IteratorUtils.createSynchronizedIterator = function (iterators, indexers, context) {

        // return an empty iterator, if no source iterators are available
        if (iterators.length === 0) {
            return IteratorUtils.ALWAYS_DONE_ITERATOR;
        }

        // the next (not yet visited) result objects of all iterators
        var results = _.invoke(iterators, 'next');

        // returns the indexer callback function for the passed iterator array index
        var getIndexer = _.isFunction(indexers) ? _.constant(indexers) : function (index) { return indexers[index]; };

        // returns the offset of the specified iterator
        function getOffset(index) {
            var result = results[index];
            return result.done ? null : getIndexer(index).call(context, result.value, result, index);
        }

        // the current offsets of all iterator results
        var offsets = _.times(results.length, getOffset);

        // return the synchronized iterator object
        return { next: function () {

            // find the iterator result with the minimum offset
            var minOffset = Number.MAX_VALUE;
            var minIndex = null;
            offsets.forEach(function (offset, index) {
                if ((offset !== null) && (offset < minOffset)) {
                    minOffset = offset;
                    minIndex = index;
                }
            });

            // early exit, if all iterators are done
            if (minIndex === null) { return { done: true }; }

            // get the result object of the iterator, and step that iterator ahead
            var result = _.extend({ iteratorIndex: minIndex }, results[minIndex]);
            results[minIndex] = iterators[minIndex].next();
            offsets[minIndex] = getOffset(minIndex);

            return result;
        } };
    };

    /**
     * Creates an iterator that visits the values provided by the passed
     * iterators simultaneously, according to some offset or sorting index for
     * the values of all iterators.
     *
     * @param {Array<Object>} iterators
     *  An array of objects implementing the standard EcmaScript iterator
     *  protocol (each iterator must provide a method next() that returns an
     *  object with the properties 'done' and 'value').
     *
     * @param {Function|Array<Function>} indexers
     *  The callback functions that convert the results of the source iterators
     *  to an offset or sorting index that will be used to create the combined
     *  results of the new parallel iterator. If set to a single function, it
     *  will be used for all source iterators. Otherwise, the array MUST have
     *  the same length as the array containing the source iterators. Each
     *  callback function receives the following parameters:
     *  (1) {Any} value
     *      The current value of the respective source iterator.
     *  (2) {Object} result
     *      The complete result object of the respective source iterator.
     *  (3) {Number} index
     *      The array index of the source iterator (may be useful for a single
     *      callback function used for all iterators).
     *  The callback functions MUST return a value that can be compared with
     *  the standard JavaScript comparison operators (numbers or strings). The
     *  return values MUST grow strictly for all results per source iterator.
     *
     * @param {Object} [context]
     *  The calling context for the indexer callback functions.
     *
     * @returns {Object}
     *  A new iterator that visits the result objects of the passed iterators
     *  simultaneously. The iterator will provide all results of the source
     *  iterators that have an equal sorting index, and will pass the value
     *  null as result for all iterators that have skipped that sorting index.
     *  The result objects of the parallel iterator will contain the following
     *  properties:
     *  - {Boolean} done
     *      If set to true, the source iterators have been visited completely.
     *      No more data is available; this result object does not contain any
     *      other properties!
     *  - {Array<Any>} value
     *      The values of the source iterators, or null for all iterators that
     *      did not provide a result object for the current sorting index that
     *      is contained in the property 'offset' (see below).
     *  - {Array<Object|Null>} results
     *      The complete result objects of the source iterators, or null for
     *      all iterators that did not provide a result object for the current
     *      sorting index that is contained in the property 'offset' (see
     *      below). At least one element in this array will be a valid result
     *      object. This array MUST NOT be changed!
     *  - {Boolean} complete
     *      Whether all iterators have provided a result object for the current
     *      sorting index (i.e. whether all elements in the property 'results'
     *      are result objects). This property is provided for convenience and
     *      to increase performance.
     *  - {Number} offset
     *      The offset or sorting index of the results of the source iterators,
     *      as provided by the indexer callback functions.
     */
    IteratorUtils.createParallelIterator = function (iterators, indexers, context) {

        // return an empty iterator, if no source iterators are available
        if (iterators.length === 0) {
            return IteratorUtils.ALWAYS_DONE_ITERATOR;
        }

        // the next (not yet visited) result objects of all iterators
        var results = _.invoke(iterators, 'next');

        // returns the indexer callback function for the passed iterator array index
        var getIndexer = _.isFunction(indexers) ? _.constant(indexers) : function (index) { return indexers[index]; };

        // returns the offset of the specified iterator
        function getOffset(index) {
            var result = results[index];
            return result.done ? null : getIndexer(index).call(context, result.value, result, index);
        }

        // returns the minimum of the passed offsets (either parameter can be null)
        function getMinOffset(offset1, offset2) {
            return (offset1 === null) ? offset2 : (offset2 === null) ? offset1 : Math.min(offset1, offset2);
        }

        // the current offsets of all iterator results
        var offsets = _.times(results.length, getOffset);
        // next offset to be visited
        var currOffset = offsets.reduce(getMinOffset, null);

        // return the parallel iterator object
        return { next: function () {

            // early exit if all iterators are done
            if (currOffset === null) {
                this.next = IteratorUtils.ALWAYS_DONE_ITERATOR.next;
                return this.next();
            }

            // the results and values to be inserted into the result object of the parallel iterator
            var iterResults = [];
            var iterValues = [];
            var complete = true;
            // offset to be visited in next iteration step (minimum of all offsets greater than currOffset)
            var nextOffset = null;

            function pushResult(result) {
                iterResults.push(result);
                iterValues.push(result ? result.value : null);
                if (!result) { complete = false; }
            }

            // collect the results into an array, and step ahead all affected iterators
            results.forEach(function (result, index) {

                // current iterator is done
                if (result.done) { return pushResult(null); }

                // the offset is greater than the current offset to be visited
                var offset = offsets[index];
                if (currOffset < offset) {
                    nextOffset = getMinOffset(nextOffset, offset);
                    return pushResult(null);
                }

                // result of the current iterator will be part of the result of the parallel iterator
                pushResult(result);
                results[index] = iterators[index].next();
                offsets[index] = offset = getOffset(index);
                nextOffset = getMinOffset(nextOffset, offset);
            });

            // create and return the result object
            var currResult = { done: false, value: iterValues, results: iterResults, complete: complete, offset: currOffset };
            currOffset = nextOffset;
            return currResult;
        } };
    };

    // converters and loops ---------------------------------------------------

    /**
     * Collects all result values of the passed iterator in a plain JS array.
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @returns {Array}
     *  An array with all result values (the 'value' properties of the result
     *  objects) returned by the iterator.
     */
    IteratorUtils.toArray = function (iterator) {
        var array = [];
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            array.push(result.value);
        }
        return array;
    };

    /**
     * Invokes the passed callback function for all results of the passed
     * iterator.
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each valid result of the
     *  passed iterator. Receives the following parameters:
     *  (1) {Any} value
     *      The 'value' property from the iterator result.
     *  (2) {Object} result
     *      The complete iterator result, that may contain additional
     *      non-standard value properties.
     *  If the callback function returns the Utils.BREAK object, the iteration
     *  process will be stopped immediately.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the callback function has
     *  returned Utils.BREAK to stop the iteration process; otherwise
     *  undefined.
     */
    IteratorUtils.forEach = function (iterator, callback, context) {
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            if (callback.call(context, result.value, result) === Utils.BREAK) {
                return Utils.BREAK;
            }
        }
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for at least one value of the passed iterator.
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each valid
     *  result of the passed iterator until it returns a truthy value. After
     *  that, iteration will be stopped immediately, the remaining values of
     *  the iterator will not be visited anymore. Receives the following
     *  parameters:
     *  (1) {Any} value
     *      The 'value' property from the iterator result.
     *  (2) {Object} result
     *      The complete iterator result, that may contain additional
     *      non-standard value properties.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for at
     *  least one value of the passed iterator.
     */
    IteratorUtils.some = function (iterator, predicate, context) {
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            if (predicate.call(context, result.value, result)) { return true; }
        }
        return false;
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for all values of the passed iterator.
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each valid
     *  result of the passed iterator until it returns a falsy value.  After
     *  that, iteration will be stopped immediately, the remaining values of
     *  the iterator will not be visited anymore. Receives
     *  the following parameters:
     *  (1) {Any} value
     *      The 'value' property from the iterator result.
     *  (2) {Object} result
     *      The complete iterator result, that may contain additional
     *      non-standard value properties.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for all
     *  values of the passed iterator.
     */
    IteratorUtils.every = function (iterator, predicate, context) {
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            if (!predicate.call(context, result.value, result)) { return false; }
        }
        return true;
    };

    /**
     * Invokes the passed callback function for all results of the passed
     * iterator, and returns the return values as plain array.
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each valid result of the
     *  passed iterator. Receives the following parameters:
     *  (1) {Any} value
     *      The 'value' property from the iterator result.
     *  (2) {Object} result
     *      The complete iterator result, that may contain additional
     *      non-standard value properties.
     *  The return value of the callback function will be inserted into the
     *  resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Array<Any>}
     *  The return values of the callback function.
     */
    IteratorUtils.map = function (iterator, callback, context) {
        var array = [];
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            array.push(callback.call(context, result.value, result));
        }
        return array;
    };

    /**
     * Invokes the passed callback function for all results of the passed
     * iterator, and reduces the iterator values to a single result value.
     *
     * @param {Any} initial
     *  The initial result value passed to the first invocation of the callback
     *  function (if the iterator is empty, this value becomes the final result
     *  value immediately).
     *
     * @param {Object} iterator
     *  An object implementing the standard EcmaScript iterator protocol (must
     *  provide a method next() that returns an object with the properties
     *  'done' and 'value').
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each valid result of the
     *  passed iterator. Receives the following parameters:
     *  (1) {Any} prevValue
     *      The result value of the previous invocation of this callback
     *      function (or the initial value on first step of the iterator).
     *  (2) {Any} value
     *      The 'value' property from the iterator result.
     *  (3) {Object} result
     *      The complete iterator result, that may contain additional
     *      non-standard value properties.
     *  The return value of the callback function will become the new result
     *  value that will be passed as 'prevValue' with the next iterator step.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The final result (the last return value of the callback fucntion).
     */
    IteratorUtils.reduce = function (initial, iterator, callback, context) {
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            initial = callback.call(context, initial, result.value, result);
        }
        return initial;
    };

    // exports ================================================================

    return IteratorUtils;

});
