/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/tk/utils/deferredutils', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/debugutils'
], function (Config, Utils, DebugUtils) {

    'use strict';

    // Created deferreds which are pending
    var pendingDeferreds = {};

    // static class DeferredUtils =============================================

    var DeferredUtils = {};

    // static methods ---------------------------------------------------------

    /**
     * Create a new deferred object.
     *
     * @param {BaseApplication} app
     *  The application.
     *
     * @param {String} infoString
     *  If specified, this string is used for logging information about run
     *  times of asynchronous operations. This option is only evaluated, if
     *  debug mode is enabled.
     *
     * @param {String} priority
     *  If specified, this string is used for specifiy the priority of the
     *  deferred. It used by selenium tests to see if all jobs are done.
     *
     * @returns {jQuery.Deferred}
     *  The new deferred object.
     */
    DeferredUtils.createDeferred = function (app, infoString, priority) {
        var defID = _.uniqueId('deferred');

        pendingDeferreds[defID] = { id: defID, app: app, infoString: infoString, priority: priority };

        DebugUtils.logQueue(pendingDeferreds, true);

        return new $.Deferred().always(function () {
            delete pendingDeferreds[defID];
            DebugUtils.logQueue(pendingDeferreds, true);
        });
    };

    /**
     * returns the BaseApplication.
     * if its assigned in options it is directly returned.
     * otherwise if baseobject.getApp() is a function, this app is returned
     * otherwise if baseobject.getNode() is a function,
     * its "window" will be used to find the correct app
     *
     * @param {Baseobject} baseobject
     *
     *  @param {Object|Null} options
     *   @param {BaseApplication) [options.app]
     *      app is needed for correct window-id
     *
     * @returns {BaseApplication}
     */
    DeferredUtils.getApp = Config.AUTOTEST ? function (baseobject, options) {
        var app = Utils.getObjectOption(options, 'app');
        if (!app) {
            if (baseobject.getApp) {
                app = baseobject.getApp();
            } else if (baseobject.getNode) {
                var parent = baseobject.getNode().closest('.window-container');
                var id = parent.attr('id');

                var myWin = _.findWhere(ox.ui.windowManager.getWindows(), { id: id });

                if (myWin) {
                    app = myWin.app;
                } else if (!Config.UNITTEST) {
                    // window.console.error('no app found', id, parent);
                }
            } else if (baseobject.isStandalone) {
                app = baseobject;
            } else if (!Config.UNITTEST) {
                window.console.error('no app found', baseobject);
            }
        }
        return app;
    } : Utils.NOOP;

    // exports ================================================================

    return DeferredUtils;
});
