/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/render/colorutils', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // threshold luma for dark colors
    var Y_DARK = 0.4;

    // maps the names of all CSS preset colors to the RGB color values
    var PRESET_COLOR_MAP = {
        aliceblue: 'f0f8ff',        antiquewhite: 'faebd7',         aqua: '00ffff',             aquamarine: '7fffd4',
        azure: 'f0ffff',            beige: 'f5f5dc',                bisque: 'ffe4c4',           black: '000000',
        blanchedalmond: 'ffebcd',   blue: '0000ff',                 blueviolet: '8a2be2',       brown: 'a52a2a',
        burlywood: 'deb887',        cadetblue: '5f9ea0',            chartreuse: '7fff00',       chocolate: 'd2691e',
        coral: 'ff7f50',            cornflowerblue: '6495ed',       cornsilk: 'fff8dc',         crimson: 'dc143c',
        darkblue: '00008b',         darkcyan: '008b8b',             darkgoldenrod: 'b8860b',    darkgray: 'a9a9a9',
        darkgreen: '006400',        darkgrey: 'a9a9a9',             darkkhaki: 'bdb76b',        darkmagenta: '8b008b',
        darkolivegreen: '556b2f',   darkorange: 'ff8c00',           darkorchid: '9932cc',       darkred: '8b0000',
        darksalmon: 'e9967a',       darkseagreen: '8fbc8f',         darkslateblue: '483d8b',    darkslategray: '2f4f4f',
        darkslategrey: '2f4f4f',    darkturquoise: '00ced1',        darkviolet: '9400d3',       deeppink: 'ff1493',
        deepskyblue: '00bfff',      dimgray: '696969',              dimgrey: '696969',          dodgerblue: '1e90ff',
        firebrick: 'b22222',        floralwhite: 'fffaf0',          forestgreen: '228b22',      fuchsia: 'ff00ff',
        gainsboro: 'dcdcdc',        ghostwhite: 'f8f8ff',           gold: 'ffd700',             goldenrod: 'daa520',
        gray: '808080',             green: '008000',                greenyellow: 'adff2f',      grey: '808080',
        honeydew: 'f0fff0',         hotpink: 'ff69b4',              indianred: 'cd5c5c',        indigo: '4b0082',
        ivory: 'fffff0',            khaki: 'f0e68c',                lavender: 'e6e6fa',         lavenderblush: 'fff0f5',
        lawngreen: '7cfc00',        lemonchiffon: 'fffacd',         lightblue: 'add8e6',        lightcoral: 'f08080',
        lightcyan: 'e0ffff',        lightgoldenrodyellow: 'fafad2', lightgray: 'd3d3d3',        lightgreen: '90ee90',
        lightgrey: 'd3d3d3',        lightpink: 'ffb6c1',            lightsalmon: 'ffa07a',      lightseagreen: '20b2aa',
        lightskyblue: '87cefa',     lightslategray: '778899',       lightslategrey: '778899',   lightsteelblue: 'b0c4de',
        lightyellow: 'ffffe0',      lime: '00ff00',                 limegreen: '32cd32',        linen: 'faf0e6',
        maroon: '800000',           mediumaquamarine: '66cdaa',     mediumblue: '0000cd',       mediumorchid: 'ba55d3',
        mediumpurple: '9370db',     mediumseagreen: '3cb371',       mediumslateblue: '7b68ee',  mediumspringgreen: '00fa9a',
        mediumturquoise: '48d1cc',  mediumvioletred: 'c71585',      midnightblue: '191970',     mintcream: 'f5fffa',
        mistyrose: 'ffe4e1',        moccasin: 'ffe4b5',             navajowhite: 'ffdead',      navy: '000080',
        oldlace: 'fdf5e6',          olive: '808000',                olivedrab: '6b8e23',        orange: 'ffa500',
        orangered: 'ff4500',        orchid: 'da70d6',               palegoldenrod: 'eee8aa',    palegreen: '98fb98',
        paleturquoise: 'afeeee',    palevioletred: 'db7093',        papayawhip: 'ffefd5',       peachpuff: 'ffdab9',
        peru: 'cd853f',             pink: 'ffc0cb',                 plum: 'dda0dd',             powderblue: 'b0e0e6',
        purple: '800080',           rebeccapurple: '663399',        red: 'ff0000',              rosybrown: 'bc8f8f',
        royalblue: '4169e1',        saddlebrown: '8b4513',          salmon: 'fa8072',           sandybrown: 'f4a460',
        seagreen: '2e8b57',         seashell: 'fff5ee',             sienna: 'a0522d',           silver: 'c0c0c0',
        skyblue: '87ceeb',          slateblue: '6a5acd',            slategray: '708090',        slategrey: '708090',
        snow: 'fffafa',             springgreen: '00ff7f',          steelblue: '4682b4',        tan: 'd2b48c',
        teal: '008080',             thistle: 'd8bfd8',              tomato: 'ff6347',           turquoise: '40e0d0',
        violet: 'ee82ee',           wheat: 'f5deb3',                white: 'ffffff',            whitesmoke: 'f5f5f5',
        yellow: 'ffff00',           yellowgreen: '9acd32'
    };

    // maps the names of all CSS system colors to the RGB color values (values will be initialized lazily)
    var SYSTEM_COLOR_MAP = Utils.makeSet([
        'activeborder',     'activecaption',    'appworkspace',         'background',
        'buttonface',       'buttonhighlight',  'buttonshadow',         'buttontext',
        'captiontext',      'graytext',         'highlight',            'highlighttext',
        'inactiveborder',   'inactivecaption',  'inactivecaptiontext',  'infobackground',
        'infotext',         'menu',             'menutext',             'scrollbar',
        'threeddarkshadow', 'threedface',       'threedhighlight',      'threedlightshadow',
        'threedshadow',     'window',           'windowframe',          'windowtext'
    ], null);

    // private global functions ===============================================

    // converts the passed string representing a hexadecimal digit
    function parseHex1(hex) {
        return parseInt(hex, 16) / 15;
    }

    // converts the passed string representing a hexadecimal byte value
    function parseHex2(hex) {
        return parseInt(hex, 16) / 255;
    }

    // converts the passed string representing a decimal value
    function parseDec(dec, quot) {
        return Math.min(1, parseInt(dec, 10) / quot);
    }

    function formatHex(dec, length) {
        return ('0000000' + dec.toString(16)).substr(-length);
    }

    // class AlphaModel ======================================================

    /**
     * Base class for various color models, providing the alpha channel of a
     * color.
     *
     * @constructor
     *
     * @property {Number} a
     *  The value of the alpha channel (opacity), in the interval [0;1]. The
     *  value 0 represents full transparency, the value 1 represents full
     *  opacity.
     */
    var AlphaModel = _.makeExtendable(function (a) {

        // public properties
        this.a = a;

    }); // class AlphaModel

    // public methods ---------------------------------------------------------

    /**
     * Changes the specified color channel to the specified value. This method
     * is intended to be used by subclasses representing different color models
     * to manipulate the various supported color channels.
     *
     * @param {String} channel
     *  The name of the color channel. The supported channel names depend on
     *  the subclass representing a specific color model. The channel name 'a'
     *  will manipulate the alpha channel provided by this base class.
     *
     * @param {Number} value
     *  The new value of the color channel.
     *
     * @param {Number} [cycle]
     *  If specified, the size of the channel interval (the valid numbers for
     *  the channel). The channel value passed in the parameter 'value' will be
     *  adjusted to the channel size by calculating the modulo. If omitted, the
     *  passed value will be reduced to the interval [0,1].
     *
     * @returns {AlphaModel}
     *  A reference to this instance.
     */
    AlphaModel.prototype.set = function (channel, value, cycle) {
        this[channel] = (typeof cycle === 'number') ? Utils.mod(value, cycle) : Utils.minMax(value, 0, 1);
        return this;
    };

    /**
     * Modulates the specified color channel (multiplies the color channel with
     * the specified factor). This method is intended to be used by subclasses
     * representing different color models to manipulate the various supported
     * color channels.
     *
     * @param {String} channel
     *  The name of the color channel. The supported channel names depend on
     *  the subclass representing a specific color model. The channel name 'a'
     *  will manipulate the alpha channel provided by this base class.
     *
     * @param {Number} factor
     *  The modulation factor. MUST be non-negative.
     *
     * @param {Number} [cycle]
     *  If specified, the size of the channel interval (the valid numbers for
     *  the channel). The resulting channel value will be adjusted to the
     *  channel size by calculating the modulo. If omitted, the resulting value
     *  will be reduced to the interval [0,1].
     *
     * @returns {AlphaModel}
     *  A reference to this instance.
     */
    AlphaModel.prototype.modulate = function (channel, factor, cycle) {
        return this.set(channel, this[channel] * factor, cycle);
    };

    /**
     * Shifts the specified color channel (adds the specified value to the
     * color channel). This method is intended to be used by subclasses
     * representing different color models to manipulate the various supported
     * color channels.
     *
     * @param {String} channel
     *  The name of the color channel. The supported channel names depend on
     *  the subclass representing a specific color model. The channel name 'a'
     *  will manipulate the alpha channel provided by this base class.
     *
     * @param {Number} shift
     *  The value to be added to the color channel.
     *
     * @param {Number} [cycle]
     *  If specified, the size of the channel interval (the valid numbers for
     *  the channel). The resulting channel value will be adjusted to the
     *  channel size by calculating the modulo. If omitted, the resulting value
     *  will be reduced to the interval [0,1].
     *
     * @returns {AlphaModel}
     *  A reference to this instance.
     */
    AlphaModel.prototype.offset = function (channel, shift, cycle) {
        return this.set(channel, this[channel] + shift, cycle);
    };

    // class RGBModel =========================================================

    /**
     * Representation of a color using the RGB color model. This class adds the
     * color channels 'r', 'g', and 'b' which can be manipulated by the methods
     * of the base class AlphaModel.
     *
     * @constructor
     *
     * @extends AlphaModel
     *
     * @property {Number} r
     *  The value of the red channel, in the interval [0;1].
     *
     * @property {Number} g
     *  The value of the green channel, in the interval [0;1].
     *
     * @property {Number} b
     *  The value of the blue channel, in the interval [0;1].
     */
    var RGBModel = AlphaModel.extend({ constructor: function (r, g, b, a) {

        // base constructor
        AlphaModel.call(this, a);

        // public properties
        this.r = r;
        this.g = g;
        this.b = b;

    } }); // class RGBModel

    // static methods ---------------------------------------------------------

    /**
     * Parses the passed hexadecimal RGB string and creates an instance of the
     * class RGBModel.
     *
     * @param {String} rgb
     *  The string representation of a 6-digit hexadecimal number (an RGB color
     *  in the format 'RRGGBB').
     *
     * @param {Boolean} [extended=false]
     *  If set to true, the parameter 'rgb' can also be a 3-digit hexadecimal
     *  number in the format 'RGB', or a 4-digit hexadecimal number in the
     *  format 'RGBA', or an 8-digit hexadecimal number in the format
     *  'RRGGBBAA'.
     *
     * @returns {RGBModel|Null}
     *  The new RGB color model created from the passed string; or null, if the
     *  passed string is not a valid hexadecimal number.
     */
    RGBModel.parseHex = function (rgb, extended) {

        if (/^[0-9a-f]+$/i.test(rgb)) {
            switch (rgb.length) {
                case 3: return extended ? new RGBModel(parseHex1(rgb[0]), parseHex1(rgb[1]), parseHex1(rgb[2]), 1) : null;
                case 4: return extended ? new RGBModel(parseHex1(rgb[0]), parseHex1(rgb[1]), parseHex1(rgb[2]), parseHex1(rgb[3])) : null;
                case 6: return new RGBModel(parseHex2(rgb.substr(0, 2)), parseHex2(rgb.substr(2, 2)), parseHex2(rgb.substr(4)), 1);
                case 8: return extended ? new RGBModel(parseHex2(rgb.substr(0, 2)), parseHex2(rgb.substr(2, 2)), parseHex2(rgb.substr(4, 2)), parseHex2(rgb.substr(6))) : null;
            }
        }

        return null;
    };

    /**
     * Creates an instance of the class RGBModel from the passed RGBA byte
     * components.
     *
     * @param {Array<Number>|TypedArray} bytes
     *  The array of bytes representing a pixel. The array MUST contain at
     *  least 4 elements in order RGBA, as used for example by the class
     *  ImageData.
     *
     * @returns {RGBModel}
     *  The new RGB color model created from the passed bytes.
     */
    RGBModel.parseBytes = function (bytes) {
        return new RGBModel(bytes[0] / 255, bytes[1] / 255, bytes[2] / 255, bytes[3] / 255);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this RGB color model.
     *
     * @returns {RGBModel}
     *  A clone of this RGB color model.
     */
    RGBModel.prototype.clone = function () {
        return new RGBModel(this.r, this.g, this.b, this.a);
    };

    /**
     * Returns the luma (the weighted lightness) of this RGB color.
     *
     * @returns {Number}
     *  The luma of this color, in the interval [0;1].
     */
    RGBModel.prototype.getLuma = function () {
        return 0.2126 * this.r + 0.7152 * this.g + 0.0722 * this.b;
    };

    /**
     * Returns a reference to this instance. Used to ease the conversion from
     * different color models to an RGB color model.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.toRGB = function () {
        return this;
    };

    /**
     * Creates an instance of the class HSLModel representing the color of this
     * RGB color model.
     *
     * @returns {HSLModel}
     *  An instance of the class HSLModel representing the color of this RGB
     *  color model.
     */
    RGBModel.prototype.toHSL = function () {

        var max = Math.max(this.r, this.g, this.b);
        var min = Math.min(this.r, this.g, this.b);
        var diff = max - min;
        var sum = max + min;

        var h = (diff === 0) ? 0 : (max === this.r) ? Utils.mod((this.g - this.b) / diff * 60, 360) : (max === this.g) ? ((this.b - this.r) / diff * 60 + 120) : ((this.r - this.g) / diff * 60 + 240);
        var s = ((sum === 0) || (sum === 2)) ? 0 : (sum < 1) ? (diff / sum) : (diff / (2 - sum));
        var l = sum / 2;

        return new HSLModel(h, s, l, this.a);
    };

    /**
     * Returns the byte representation of the color contained in this model.
     *
     * @returns {Array<Number>}
     *  The byte representation of the color contained in this model. The array
     *  will contain 4 integers, in the order RGBA as used for example in
     *  ImageData objects.
     */
    RGBModel.prototype.toBytes = function () {

        // double rounding to workaround rounding errors during RGB/HSL conversion
        var r8 = Math.round(Math.round(this.r * 25500000) / 100000);
        var g8 = Math.round(Math.round(this.g * 25500000) / 100000);
        var b8 = Math.round(Math.round(this.b * 25500000) / 100000);
        var a8 = Math.round(this.a * 255);

        return [r8, g8, b8, a8];
    };

    /**
     * Returns the CSS representation of the color contained in this model.
     *
     * @returns {String}
     *  The CSS representation of the color contained in this model.
     */
    RGBModel.prototype.toCSS = function () {

        // prefer keyword 'transparent' for fully transparent colors
        if (this.a <= 0) { return 'transparent'; }

        // opaque color: return #RRGGBB style; otherwise return rgba() function style
        var bytes = this.toBytes();
        return (this.a >= 1) ? ('#' + formatHex(bytes[0] * 0x10000 + bytes[1] * 0x100 + bytes[2], 6)) :
            ('rgba(' + bytes[0] + ',' + bytes[1] + ',' + bytes[2] + ',' + Utils.round(this.a, 0.001) + ')');
    };

    /**
     * Applies a gamma shift to this color model.
     *
     * @param {Number} gamma
     *  The gamma shift to be applied to the color channels.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.gamma = function (gamma) {
        this.r = Math.pow(this.r, gamma);
        this.g = Math.pow(this.g, gamma);
        this.b = Math.pow(this.b, gamma);
        return this;
    };

    /**
     * Changes this color model to its inverse RGB color.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.inv = function () {
        this.r = 1 - this.r;
        this.g = 1 - this.g;
        this.b = 1 - this.b;
        return this;
    };

    /**
     * Changes this color model to its weighted grayscale.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.gray = function () {
        this.r = this.g = this.b = this.getLuma();
        return this;
    };

    // class HSLModel =========================================================

    /**
     * Representation of a color using the HSL color model. This class adds the
     * color channels 'h', 's', and 'l' which can be manipulated by the methods
     * of the base class AlphaModel.
     *
     * @constructor
     *
     * @extends AlphaModel
     *
     * @property {Number} h
     *  The value of the hue channel, in the half-open interval [0;360). The
     *  value 0 represents red, 120 represents green, and 240 represents blue.
     *
     * @property {Number} s
     *  The value of the saturation channel, in the interval [0;1]. The value 0
     *  represents gray, the value 1 represents the fully saturated color.
     *
     * @property {Number} l
     *  The value of the luminance channel, in the interval [0;1]. The value 0
     *  represents black (regardless of the other values), and the value 1
     *  represents white.
     */
    var HSLModel = AlphaModel.extend({ constructor: function (h, s, l, a) {

        // base constructor
        AlphaModel.call(this, a);

        // public properties
        this.h = h;
        this.s = s;
        this.l = l;

    } }); // class HSLModel

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this HSL color model.
     *
     * @returns {HSLModel}
     *  A clone of this HSL color model.
     */
    HSLModel.prototype.clone = function () {
        return new HSLModel(this.h, this.s, this.l, this.a);
    };

    /**
     * Creates an instance of the class RGBModel representing the color of this
     * HSL color model.
     *
     * @returns {RGBModel}
     *  An instance of the class RGBModel representing the color of this HSL
     *  color model.
     */
    HSLModel.prototype.toRGB = function () {

        var h = this.h / 60;
        var r = (Utils.minMax(Math.abs(h - 3) - 1, 0, 1) - 0.5) * this.s + 0.5;
        var g = (Utils.minMax(2 - Math.abs(2 - h), 0, 1) - 0.5) * this.s + 0.5;
        var b = (Utils.minMax(2 - Math.abs(4 - h), 0, 1) - 0.5) * this.s + 0.5;
        var l = 0;

        if (this.l < 0.5) {
            l = 2 * this.l;
            r *= l;
            g *= l;
            b *= l;
        } else if (this.l > 0.5) {
            l = 2 - 2 * this.l;
            r = 1 - (1 - r) * l;
            g = 1 - (1 - g) * l;
            b = 1 - (1 - b) * l;
        }

        return new RGBModel(r, g, b, this.a);
    };

    /**
     * Returns a reference to this instance. Used to ease the conversion from
     * different color models to an HSL color model.
     *
     * @returns {HSLModel}
     *  A reference to this instance.
     */
    HSLModel.prototype.toHSL = function () {
        return this;
    };

    /**
     * Returns the CSS representation of the color contained in this model.
     *
     * @returns {String}
     *  The CSS representation of the color contained in this model
     */
    HSLModel.prototype.toCSS = function () {

        // prefer keyword 'transparent' for fully transparent colors
        if (this.a <= 0) { return 'transparent'; }

        // opaque color: return hsl() function style; otherwise return hsla() function style
        var hsl = Math.round(this.h) + ',' + Math.round(this.s * 100) + '%,' + Math.round(this.l * 100) + '%';
        return (this.a >= 1) ? ('hsl(' + hsl + ')') : ('hsla(' + hsl + ',' + Utils.round(this.a, 0.001) + ')');
    };

    /**
     * Darkens this color according to the passed value.
     *
     * @param {Number} factor
     *  The factor to darken this color, in the closed interval [0;1]. The
     *  number 0 will not modify this color, the number 1 will change this
     *  color to black, all numbers inside the interval will darken this color
     *  accordingly.
     *
     * @returns {HSLModel}
     *  A reference to this instance.
     */
    HSLModel.prototype.darken = function (factor) {
        this.l *= Utils.minMax(1 - factor, 0, 1);
        return this;
    };

    /**
     * Lightens this color according to the passed value.
     *
     * @param {Number} factor
     *  The factor to lighten this color, in the closed interval [0;1]. The
     *  number 0 will not modify this color, the number 1 will change this
     *  color to white, all numbers inside the interval will lighten this color
     *  accordingly.
     *
     * @returns {HSLModel}
     *  A reference to this instance.
     */
    HSLModel.prototype.lighten = function (factor) {
        this.l += (1 - this.l) * Utils.minMax(factor, 0, 1);
        return this;
    };

    /**
     * Changes this color model to its complement color (rotates the hue by
     * 180 degrees).
     *
     * @returns {HSLModel}
     *  A reference to this instance.
     */
    HSLModel.prototype.comp = function () {
        return this.offset('h', 180, 360);
    };

    // class ColorDescriptor ==================================================

    /**
     * A descriptor object with details about a color.
     *
     * @constructor
     *
     * @property {RGBModel} rgb
     *  The RGB color model of the color.
     *
     * @property {HSLModel} hsl
     *  The HSL color model of the color.
     *
     * @property {Number} a
     *  The value of the alpha channel (0 to 1). This property is provided for
     *  convenience, it is also contained in the RGB color model, and in the
     *  HSL color model.
     *
     * @property {Number} y
     *  The luma (weighted lightness) of the color (0 to 1).
     *
     * @property {Boolean} dark
     *  Whether the color is considered to be dark (the luma is less than a
     *  specific threshold value).
     *
     * @property {Array<Number>} bytes
     *  The RGBA representation of the color, as 8-bit values in an array with
     *  four elements (RGBA order as used in ImageData objects).
     *
     * @property {String} css
     *  The exact CSS representation of the color.
     *
     * @property {String} hex
     *  The upper-case hexadecimal RRGGBB value, ignoring the actual opacity of
     *  the color.
     *
     * @property {String} type
     *  The type of the original color this descriptor instance has been
     *  created from. Will be one of the following values:
     *  - 'transparent' for full transparency (e.g. after parsing the CSS
     *      keyword 'transparent').
     *  - 'rgb' for any generic color in the RGB color model (e.g. after
     *      parsing CSS colors in '#RRGGB' style, or rgb() function style).
     *  - 'hsl' for any generic color in the HSL color model (e.g. after
     *      parsing CSS colors in hsl() function style).
     *  - 'preset' for a preset color (e.g. after parsing CSS color names).
     *  - 'system' for a system color (e.g. after parsing CSS color names).
     */
    function ColorDescriptor(model, type) {

        // both color models
        this.rgb = model.toRGB();
        this.hsl = model.toHSL();

        // alpha channel, and luma
        this.a = model.a;
        this.y = this.rgb.getLuma();
        this.dark = this.y <= Y_DARK;

        // byte representation
        this.bytes = this.rgb.toBytes();

        // CSS color representation
        this.css = model.toCSS();

        // plain hexadecimal RGB representation without alpha channel
        this.hex = formatHex(this.bytes[0] * 0x10000 + this.bytes[1] * 0x100 + this.bytes[2], 6).toUpperCase();

        // add the passed native color type
        this.type = type;

    } // class ColorDescriptor

    // static class ColorUtils ================================================

    var ColorUtils = {
        RGBModel: RGBModel,
        HSLModel: HSLModel,
        ColorDescriptor: ColorDescriptor
    };

    // static methods ---------------------------------------------------------

    /**
     * Returns the RGB value of the specified CSS preset color.
     *
     * @param {String} presetColor
     *  The name of a preset color, as used in CSS. The character case of the
     *  color name will be ignored.
     *
     * @returns {String|Null}
     *  The hexadecimal RGB value of the preset color, as 6-digit string; or
     *  null, if the passed color name is invalid.
     */
    ColorUtils.getPresetColor = function (presetColor) {
        return PRESET_COLOR_MAP[presetColor.toLowerCase()] || null;
    };

    /**
     * Returns the RGB value of the specified CSS system color.
     *
     * @param {String} sysColor
     *  The name of a system color, as used in CSS. The character case of the
     *  color name will be ignored.
     *
     * @returns {String|Null}
     *  The hexadecimal RGB value of the system color, as 6-digit string; or
     *  null, if the passed color name is invalid.
     */
    ColorUtils.getSystemColor = (function () {

        var canvas = $('<canvas width="3" height="3">');
        var context = canvas[0].getContext('2d');

        return function (systemColor) {
            var key = systemColor.toLowerCase();
            var rgb = SYSTEM_COLOR_MAP[key];
            if (rgb === null) {
                context.fillStyle = systemColor;
                context.fillRect(0, 0, 3, 3);
                var pixel = context.getImageData(1, 1, 1, 1).data;
                SYSTEM_COLOR_MAP[key] = rgb = ('00000' + (pixel[0] * 0x10000 + pixel[1] * 0x100 + pixel[2]).toString(16)).substr(-6).toLowerCase();
            }
            return rgb || null;
        };
    }());

    /**
     * Converts the passed CSS color string to a color descriptor.
     *
     * @param {String} value
     *  The CSS color value. The following CSS colors will be accepted:
     *  - The keyword 'transparent'.
     *  - Predefined color names, e.g. 'Red'.
     *  - System color names, e.g. 'ButtonFace'.
     *  - Hexadecimal colors (6-digit, and 3-digit), with leading hash sign.
     *  - Hexadecimal colors with alpha channel (8-digit, and 4-digit), with
     *      leading hash sign.
     *  - RGB colors in function style, using the rgb() or rgba() function.
     *  - HSL colors in function style, using the hsl() or hsla() function.
     *
     * @returns {ColorDescriptor|Null}
     *  A color descriptor, if the passed string could be parsed successfully;
     *  otherwise null.
     */
    ColorUtils.parseColor = function (value) {

        // no valid string passed
        if (value.length === 0) { return null; }

        // CSS keyword 'transparent'
        if (value === 'transparent') {
            return new ColorDescriptor(new RGBModel(0, 0, 0, 0), 'transparent');
        }

        // predefined color names
        var presetColor = ColorUtils.getPresetColor(value);
        if (presetColor) {
            return new ColorDescriptor(RGBModel.parseHex(presetColor), 'preset');
        }

        // system color names
        var systemColor = ColorUtils.getSystemColor(value);
        if (systemColor) {
            return new ColorDescriptor(RGBModel.parseHex(systemColor), 'system');
        }

        // 3-digit hexadecimal color (#RGB), 4-digit hexadecimal color (#RGBA),
        // 6-digit hexadecimal color (#RRGGBB), 8-digit hexadecimal color (#RRGGBBAA)
        if (value[0] === '#') {
            return (function () {
                var model = RGBModel.parseHex(value.substr(1), true);
                return model ? new ColorDescriptor(model, 'rgb') : null;
            }());
        }

        // RGB/RGBA color in function style
        var parts = value.match(/^rgb\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)$/) || value.match(/^rgba\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*(1|0|0?\.\d+)\s*\)$/);
        if (parts) {
            return (function () {
                var r = parseDec(parts[1], 255);
                var g = parseDec(parts[2], 255);
                var b = parseDec(parts[3], 255);
                var a = parts[4] ? parseFloat(parts[4]) : 1;
                return new ColorDescriptor(new RGBModel(r, g, b, a), 'rgb');
            }());
        }

        // HSL/HSLA color in function style
        parts = value.match(/^hsl\(\s*(\d+)\s*,\s*(\d+)%\s*,\s*(\d+)%\s*\)$/) || value.match(/^hsla\(\s*(\d+)\s*,\s*(\d+)%\s*,\s*(\d+)%\s*,\s*(1|0|0?\.\d+)\s*\)$/);
        if (parts) {
            return (function () {
                var h = Utils.mod(parseInt(parts[1], 10), 360);
                var s = parseDec(parts[2], 100);
                var l = parseDec(parts[3], 100);
                var a = parts[4] ? parseFloat(parts[4]) : 1;
                return new ColorDescriptor(new HSLModel(h, s, l, a), 'hsl');
            }());
        }

        return null;
    };

    // exports ================================================================

    return ColorUtils;

});
