/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/io', [
    'io.ox/core/http',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/deferredutils'
], function (CoreHTTP, Utils, DeferredUtils) {

    'use strict';

    // static class IO ========================================================

    /**
     * Provides static methods for server/client communication and host file
     * system access.
     */
    var IO = {};

// constants --------------------------------------------------------------

    /**
     * The name of the document filter server module.
     */
    IO.FILTER_MODULE_NAME = 'oxodocumentfilter';

    /**
     * The name of the document converter server module.
     */
    IO.CONVERTER_MODULE_NAME = 'oxodocumentconverter';

    // public methods ---------------------------------------------------------

    /**
     * Creates a function that returns a promise which will be resolved or
     * rejected depending on the result of the passed filter callback function.
     * Intended to be used in promise chains to modify the state of a promise
     * depending on the result data.
     *
     * @param {Function} filter
     *  A function that receives and filters the response of a resolved promise
     *  as first parameter. If this function returns undefined, the promise
     *  created by the method will be rejected. Otherwise, the promise will be
     *  resolved with the return value of this filter function.
     *
     * @param {Object} [context]
     *  The context object that will be bound to the passed filter function.
     *
     * @returns {Function}
     *  A new function that returns a promise that will be resolved or rejected
     *  depending on the result of the passed filter callback function.
     */
    IO.createDeferredFilter = function (filter, context) {

        // create and return a function that call the filter and returns a resolved/rejected promise
        return function (response) {

            var // the resulting deferred object
                def = DeferredUtils.createDeferred(IO, 'IO: createDeferredFilter'),
                // call passed filter callback
                value = _.isObject(response) ? filter.call(context, response) : undefined;

            // resolve the deferred object, if the value returned by the filter function is not undefined
            return _.isUndefined(value) ? def.reject(response) : def.resolve(value);
        };
    };

    /**
     * Sends a request to the server and returns a promise waiting for the
     * response.
     *
     * @param {String} module
     *  The name of the server module that will receive the request.
     *
     * @param {Object} params
     *  Parameters that will be inserted into the request URL (method GET), or
     *  into the request body (method POST).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.method='GET']
     *      The request method. Must be an upper-case string (for example,
     *      'GET', 'POST', etc.). Defaults to 'GET'.
     *  @param {Function} [options.resultFilter]
     *      A function that will be called if the request returns successfully,
     *      and filters the resulting 'data' object returned by the request.
     *      Receives the 'data' object as first parameter. If this function
     *      returns undefined, the entire request will be rejected. Otherwise,
     *      the request will be resolved with the return value of this function
     *      instead of the complete 'data' object.
     *
     * @returns {jQuery.Promise}
     *  The promise of the request. Will be resolved with the 'data' object
     *  returned by the response, if available; or the valid return value of
     *  the result filter callback function, if specified. Otherwise, the
     *  promise will be rejected. Contains the additional method 'abort()' that
     *  allows to abort the running request which rejects the promise. Calling
     *  this method has no effect, if the request is finished already.
     */
    IO.sendRequest = function (module, params, options) {

        var // extract the request method
            method = Utils.getStringOption(options, 'method', 'GET'),
            // extract the result filter callback
            resultFilter = Utils.getFunctionOption(options, 'resultFilter'),
            // properties passed to the server request
            requestProps = { module: module, params: params },
            // the promise representing the core AJAX request
            ajaxRequest = null,
            // the Promise returned by this method
            promise = null;

        // send the AJAX request
        ajaxRequest = CoreHTTP[method](requestProps);

        // reject, if the response contains 'hasErrors:true'
        promise = ajaxRequest.then(function (response) {
            return Utils.getBooleanOption(response, 'hasErrors', false) ? $.Deferred().reject(response) : response;
        });

        // filter the result of the original request according to the passed filter callback
        if (_.isFunction(resultFilter)) {
            promise = promise.then(IO.createDeferredFilter(resultFilter));
        }

        // add an abort() method, forward invocation to AJAX request
        return _.extend(promise, { abort: function () { ajaxRequest.abort(); } });
    };

    /**
     * Reads the specified file and returns a promise that will be resolved or
     * rejected depending on the result of the read operation. The file will be
     * converted to a data URL containing the file contents as Base-64 encoded
     * data and passed to the resolved promise.
     *
     * @param {File} fileDesc
     *  The descriptor of the file to be loaded.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the result object containing the
     *  data URL, or rejected if the read operation failed. If the file size is
     *  known, the promise will be notified about the progress of the operation
     *  (a floating-point number between 0.0 and 1.0). Contains the additional
     *  method 'abort()' that allows to abort the running file reader which
     *  rejects the promise. Calling this method has no effect, if importing
     *  the file is finished already.
     */
    IO.readClientFileAsDataUrl = function (fileDesc) {

        var // resulting deferred object
            def = DeferredUtils.createDeferred(IO, 'IO: readClientFileAsDataURL'),
            // create a browser file reader instance
            reader = window.FileReader ? new window.FileReader() : null;

        if (reader) {

            // register the load event handler, deferred object will be resolved with data URL
            reader.onload = function (event) {
                if (event && event.target && _.isString(event.target.result)) {
                    def.resolve(event.target.result);
                } else {
                    def.reject();
                }
            };

            // register error event handlers, deferred object will be rejected
            reader.onerror = function () {
                def.reject();
            };

            // register abort event handlers, deferred object will be rejected
            reader.onabort = function () {
                def.reject('abort');
            };

            // register progress handler, deferred object will be notified
            reader.onprogress = function (event) {
                if (event.lengthComputable) {
                    def.notify(event.loaded / event.total);
                }
            };

            // forget reference to the file reader after import
            def.always(function () { reader = null; });

            // read the file and generate a data URL
            reader.readAsDataURL(fileDesc);

        } else {
            // file reader not supported
            def.reject();
        }

        // add an abort() method, forward invocation to AJAX request
        return _.extend(def.promise(), {
            abort: function () {
                if (reader) { reader.abort(); }
            }
        });
    };

    /**
     * Loads the specified JSON source file.
     *
     * @param {String} fileName
     *  The name of the file, relative to the application base path, as used by
     *  RequireJS, without file extension (which is assumed to be 'json').
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the JSON data loaded from the
     *  specified source file, or that will be rejected on any error (e.g.,
     *  source file not found, or invalid JSON data).
     */
    IO.loadJSON = function (fileName) {

        // use the text loader of RequireJS to load the JSON file
        return require(['text!' + fileName + '.json']).then(function (response) {

            // check response
            if (!_.isString(response) || (response.length === 0)) {
                Utils.error('IO.loadJSON(): invalid response');
                return $.Deferred().reject(response);
            }

            // parse JSON string to object
            try {
                return JSON.parse(response);
            } catch (ex) {
                Utils.exception(ex);
                return $.Deferred().reject(ex);
            }
        });
    };

    /**
     * Fetches the source code of the specified application in the background.
     *
     * @param {String} modulePath
     *  The path to the application main module. The implementation will fetch
     *  the file 'main.js' inside this directory.
     */
    IO.prefetchModuleSource = _.memoize(function (modulePath) {

        var // determine the module for server-side initialization
            module = modulePath.substr(modulePath.lastIndexOf('/') + 1);

        // send prefetch request to server
        if (module.length > 0) {
            IO.sendRequest(IO.FILTER_MODULE_NAME, { action: 'prefetchofficeparts', part: module });
        }

        // fetch source code for module
        _.delay(function () {
            Utils.log('IO.prefetchModuleSource(): fetching ' + modulePath);
            require([modulePath + '/main']);
        }, 2500);
    });

    // exports ================================================================

    return IO;

});
