/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/menumixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms'
], function (Utils, KeyCodes, Forms) {

    'use strict';

    var // marker CSS class for elements containing an open drop-down menu
        OPEN_CLASS = 'dropdown-open';

    // class MenuMixin ========================================================

    /**
     * Extends a Group object with a drop-down menu element. Creates a new
     * drop-down button element in the group and extends it with a caret sign.
     * Implements mouse and keyboard event handling for that drop-down button
     * (open and close the drop-down menu, and closing the drop-down menu
     * automatically on focus navigation). Adds new methods to the group
     * instance to control the drop-down button and menu.
     *
     * Note: This is a mix-in class supposed to extend an existing instance of
     * the class Group or one of its derived classes. Expects the symbol 'this'
     * to be bound to an instance of Group.
     *
     * @constructor
     *
     * @param {BaseMenu} menu
     *  The pop-up menu instance that will be bound to the group. Can be the
     *  instance of any subclass of BaseMenu. ATTENTION: This mix-in takes
     *  ownership of the pop-up menu instance and will destroy it by itself.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {HTMLElement|jQuery} [initOptions.button]
     *      The drop-down button element that will toggle the drop-down menu
     *      when clicked. If omitted, the drop-down menu cannot be toggled
     *      using the mouse; the client implementation has to provide its own
     *      logic to show and hide the menu.
     *  @param {HTMLElement|jQuery} [initOptions.ariaOwner]
     *      The contextual parent DOM element that acts as owner of the menu.
     *      Exposes the relationship between the parent and the menu element to
     *      assistive technologies where the DOM hierarchy doesn't represent
     *      the relationship. If omitted, the drop-down button element passed
     *      with the option 'button' will be used instead.
     *  @param {Boolean} [initOptions.caret=true]
     *      If set to true or omitted, a caret symbol will be added to the
     *      right border of the drop-down button specified with the option
     *      'button' (see above).
     *  @param {Boolean} [initOptions.tabNavigate=false]
     *      If set to true, the TAB key will move the browser focus through the
     *      form controls of the menu. By default, the TAB key will leave and
     *      close the menu.
     *  @param {Boolean} [initOptions.autoCloseParent=true]
     *      Whether to trigger a 'group:cancel' event after closing this menu
     *      with a click on the menu button.
     */
    function MenuMixin(menu, initOptions) {

        var // self reference (the Group instance)
            self = this,

            // the DOM node this drop-down menu is anchored to
            groupNode = this.getNode(),

            // the root node of the passed pop-up menu
            menuNode = menu.getNode(),

            // the drop-down button
            menuButton = $(Utils.getOption(initOptions, 'button')),

            // the ARIA owner node
            ariaOwner = $(Utils.getOption(initOptions, 'ariaOwner', menuButton)),

            // whether to use TAB key to navigate inside the menu
            tabNavigate = Utils.getOption(initOptions, 'tabNavigate', false),

            // whether to trigger a 'group:cancel' event when closing with mouse click
            autoCloseParent = Utils.getBooleanOption(initOptions, 'autoCloseParent', true),

            // root button from the whole menu tree
            parentAnchor = null,

            // menu from which the current dialog is opened
            parentMenu = null,

            // a flag whether to use the 'parentAnchor' as an anchor for the menu or not
            isRootAnchoredMenu = Utils.getOption(initOptions, 'isRootAnchoredMenu', false);

        // private methods ----------------------------------------------------

        /**
         * Initializes the drop-down menu after it has been shown.
         */
        function popupShowHandler() {
            // add CSS marker class to the group
            groupNode.addClass(OPEN_CLASS);
            ariaOwner.attr('aria-expanded', true);
        }

        /**
         * Deinitializes the drop-down menu before it will be hidden.
         */
        function popupBeforeHideHandler(event, browserEvent, node) {
            // move focus to drop-down button, if drop-down menu is focused and the user press ESC or F6
            if (menu.hasFocus() && browserEvent && (browserEvent.keyCode === KeyCodes.ESCAPE || browserEvent.keyCode === KeyCodes.F6)) {
                Utils.setFocus(menuButton);
            } else {
                Utils.setFocus(node);
                Utils.trigger('change:focus', node, node, null);
            }

            // remove CSS marker class from the group
            groupNode.removeClass(OPEN_CLASS);
            ariaOwner.attr('aria-expanded', false);
        }

        /**
         * Handles keyboard events from any control in the group object.
         */
        function groupKeyDownHandler(event) {
            if (self.isEnabled() && Forms.isVisibleNode(menuButton)) {
                if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true }) || KeyCodes.matchKeyCode(event, 'UP_ARROW', { alt: true })) {
                    event.target = menuButton[0];
                    Forms.triggerClickForKey(event);
                }
            }
        }

        /**
         * Handler for 'remote' events that can be triggered manually at the
         * root node of the group to control the visibility of the drop-down
         * menu from external code.
         */
        function groupRemoteHandler(event, command) {

            switch (command) {
                case 'show':
                    menu.show();
                    // keep menu open regardless of browser focus
                    if (menu.getAutoCloseMode()) {
                        menu.setAutoCloseMode(false);
                        menu.one('popup:hide', function () { menu.setAutoCloseMode(true); });
                    }
                    break;
                case 'hide':
                    menu.hide();
                    break;
            }
        }

        /**
         * Handles click events from the drop-down button, and toggles the
         * drop-down menu.
         */
        function menuButtonClickHandler(event) {

            // do nothing (but trigger the 'group:cancel' event) if the group is disabled
            if (self.isEnabled()) {

                // US 102078806: when the root anchored version from a menu is used, it
                // sets necessary values in its base menu
                if (Utils.SMALL_DEVICE && isRootAnchoredMenu) {
                    menu.setRootAnchoredMenu(parentAnchor, parentMenu, isRootAnchoredMenu);
                }

                // toggle the drop-down menu, this triggers the appropriate event
                menu.toggle();

                // set focus to a control in the menu, if click was triggered by a key
                if (menu.isVisible() && _.isNumber(event.keyCode)) {
                    menu.grabFocus({ bottom: event.keyCode === KeyCodes.UP_ARROW });
                }
            }

            // trigger 'group:cancel' event, if the menu has been closed with mouse click,
            // or after click on a disabled group, but never after key events
            if (autoCloseParent && !menu.isVisible() && !_.isNumber(event.keyCode)) {
                self.triggerCancel({ sourceEvent: event });
            }
        }

        /**
         * Handles keyboard events in the focused drop-down button.
         */
        function menuButtonKeyDownHandler(event) {
            if ((event.keyCode === KeyCodes.ESCAPE) && menu.isVisible()) {
                menu.hide(event);
                return false;
            }
        }

        /**
         * Handles keyboard events inside the open drop-down menu.
         */
        function menuKeyHandler(event) {

            var // distinguish between event types
                keydown = event.type === 'keydown';

            switch (event.keyCode) {
                case KeyCodes.TAB:
                    if (keydown && KeyCodes.matchModifierKeys(event, { shift: null }) && !tabNavigate) {
                        // To prevent problems with event bubbling (Firefox continues
                        // to bubble to the parent of the menu node, while Chrome
                        // always bubbles from the focused DOM node, in this case
                        // from the menu *button*), stop propagation of the original
                        // event, and focus the next control manually.
                        var nextFocusNode = Utils[event.shiftKey ? 'findPreviousNode' : 'findNextNode'](document, menuButton, Forms.FOCUSABLE_SELECTOR);
                        Utils.setFocus($(nextFocusNode));
                        menu.hide();
                        return false;
                    }
                    break;
                case KeyCodes.ESCAPE:
                    if (keydown) {
                        menu.hide(event);
                    }
                    return false;
                case KeyCodes.F6:
                    // Hide drop-down menu on global F6 focus traveling. This will set
                    // the focus back to the drop-down button, so that F6 will jump to
                    // the next view component. Jumping directly from the drop-down menu
                    // (which is located near the body element in the DOM) would select
                    // the wrong element. Ignore all modifier keys here, even on
                    // non-MacOS systems where F6 traveling is triggered by Ctrl+F6
                    // (browsers will move the focus away anyway).
                    if (keydown) {
                        menu.hide(event);
                    }
                    break;
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the button element that toggles the drop-down menu.
         *
         * @returns {jQuery}
         *  The drop-down button element, as jQuery object.
         */
        this.getMenuButton = function () {
            return menuButton;
        };

        /**
         * Returns the instance of the pop-up menu class representing the
         * drop-down menu of this group.
         *
         * @returns {BaseMenu}
         *  The pop-up menu instance representing the drop-down menu of this
         *  group.
         */
        this.getMenu = function () {
            return menu;
        };

        /**
         * Returns the root DOM node of the drop-down menu, as jQuery object.
         *
         * @returns {jQuery}
         *  The root DOM node of the drop-down menu.
         */
        this.getMenuNode = function () {
            return menuNode;
        };

        /**
         * Returns whether the drop-down menu is currently visible.
         */
        this.isMenuVisible = function () {
            return menu.isVisible();
        };

        /**
         * Shows the drop-down menu.
         *
         * @returns {MenuMixin}
         *  A reference to this instance.
         */
        this.showMenu = function () {
            menu.show();
            return this;
        };

        /**
         * Hides the drop-down menu.
         *
         * @returns {MenuMixin}
         *  A reference to this instance.
         */
        this.hideMenu = function () {
            menu.hide();
            return this;
        };

        /**
         * Removes the content nodes from the drop-down menu.
         *
         * @param {String} [selector]
         *  A CSS selector to remove specific content nodes only. If omitted,
         *  all content nodes will be removed.
         *
         * @returns {MenuMixin}
         *  A reference to this instance.
         */
        this.clearMenu = function (selector) {
            menu.clearContents(selector);
            return this;
        };

        /**
         * Sets a reference to the parent anchor node for this menu.
         *
         * @param {jQuery|Null} node
         *  Anchor node from the root menu, or null when the toolbar element is
         *  not shrunken.
         *
         * @returns {MenuMixin}
         *  A reference to this instance.
         */
        this.setParentAnchorNode = function (node) {
            parentAnchor = node;
            return this;
        };

        /**
         * Sets a reference to the parent menu for this menu.
         *
         * @param {BaseMenu|Null} newParentMenu
         *  Reference to the parent menu, or null when the toolbar element is
         *  not shrunken.
         *
         * @returns {MenuMixin}
         *  A reference to this instance.
         */
        this.setParentMenu = function (newParentMenu) {
            parentMenu = newParentMenu;
            return this;
        };

        // initialization -----------------------------------------------------

        // special CSS marker for all groups with a drop-down button
        groupNode.addClass('dropdown-group');

        // convert ENTER and SPACE keys to click events
        Forms.setButtonKeyHandler(menuButton);

        // add the caret symbol if specified
        if (Utils.getBooleanOption(initOptions, 'caret', true) && (menuButton.length > 0)) {
            menuButton.addClass('caret-button').append(Forms.createCaretMarkup('down'));
        }

        // set the ARIA relation between button and menu with the 'aria-owns' attribute
        // and use the ARIA label of the button also for the menu
        menuNode.attr({ id: menu.getUid(), 'aria-label': ariaOwner.attr('aria-label') || null });
        ariaOwner.attr({ 'aria-owns': menu.getUid(), 'aria-haspopup': true, 'aria-expanded': false });

        // register menu node for focus handling (group remains in focused state while focus is in menu node)
        this.registerFocusableNodes(menuNode);

        // register menu button for focus handling (menu remains visible while button is focused)
        menu.registerFocusableNodes(menuButton);

        // register event handlers for the group
        this.on('group:beforehide', function () { menu.hide(); });
        this.on('group:enable', function (event, state) { if (!state) { menu.hide(); } });
        groupNode.on('keydown', groupKeyDownHandler);
        groupNode.on('remote', groupRemoteHandler);

        // register event handlers for the menu button
        Forms.touchAwareListener({ node: menuButton }, menuButtonClickHandler);
        menuButton.on('keydown', menuButtonKeyDownHandler);

        // register event handlers for the drop-down menu
        menu.on('popup:show', popupShowHandler);
        menu.on('popup:beforehide', popupBeforeHideHandler);
        menuNode.on('keydown keypress keyup', menuKeyHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            menu.destroy(); // this instance owns the passed menu
            menu = initOptions = self = groupNode = menuNode = menuButton = null;
        });

    } // class MenuMixin

    // exports ================================================================

    return _.makeExtendable(MenuMixin);

});
