/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/simplemap', [
    'io.ox/office/tk/container/basemap'
], function (BaseMap) {

    'use strict';

    // convenience shortcuts
    var MapProto = BaseMap.prototype;

    // class SimpleMap ========================================================

    /**
     * A simple helper class that represents an associative map of arbitrary
     * values with a few convenience accessor methods.
     *
     * @constructor
     *
     * @extends BaseMap
     */
    var SimpleMap = BaseMap.extend({ constructor: function () {

        // base constructor
        BaseMap.call(this);

    } }); // class SimpleMap

    // static methods ---------------------------------------------------------

    SimpleMap.from = function (list) {
        var map = new SimpleMap();
        var each = list.forEach ? list.forEach.bind(list) : _.each.bind(_, list);
        each(function (value, key) { map.insert(key, value); });
        return map;
    };

    SimpleMap.pluck = function (list, prop) {
        var map = new SimpleMap();
        var each = list.forEach ? list.forEach.bind(list) : _.each.bind(_, list);
        each(function (elem, key) { map.insert(key, elem[prop]); });
        return map;
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a shallow or deep clone of this map.
     *
     * @param {Function} [generator]
     *  A generator callback function that can be used to deeply clone the
     *  elements of this map. Receives the following parameters:
     *  (1) {Any} value
     *      The map element to be cloned.
     *  (2) {String} key
     *      The key of the map element.
     *  The return value of this function will be inserted as new map element
     *  into the clone. If omitted, the elements will be copied by reference.
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {SimpleMap}
     *  The clone of this map.
     */
    SimpleMap.prototype.clone = function (generator, context) {
        var clone = new SimpleMap();
        MapProto._init.call(clone, this, generator, context);
        return clone;
    };

    /**
     * Replaces an element in this map with a new value, or creates a new
     * element.
     *
     * @param {String} key
     *  The key of the map element to be updated.
     *
     * @param {Any} def
     *  The value passed to the callback function for a missing map element.
     *
     * @param {Function} callback
     *  The callback function that will be invoked with the current value of
     *  the map element (or the specified default value). The return value of
     *  this function will become the new value of the map element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The new value of the map element.
     */
    SimpleMap.prototype.update = function (key, def, callback, context) {
        return this.insert(key, callback.call(context, this.get(key, def)));
    };

    /**
     * Moves an element in this map to a new key.
     *
     * @param {String} key
     *  The key of the map element to be moved.
     *
     * @param {String} newKey
     *  The new key of the map element.
     *
     * @returns {SimpleMap}
     *  A reference to this instance.
     */
    SimpleMap.prototype.move = function (key, newKey) {
        if (this.has(key)) { this.insert(newKey, this.remove(key)); }
        return this;
    };

    /**
     * Inserts all elements of the passed map into this map.
     *
     * @param {Object|BaseMap} source
     *  The map to be inserted into this map. Old elements in this map will be
     *  overwritten by the elements of the passed map.
     *
     * @returns {SimpleMap}
     *  A reference to this instance.
     */
    SimpleMap.prototype.merge = function (source) {
        BaseMap.forEach(source, function (value, key) { this.insert(key, value); }, this);
        return this;
    };

    /**
     * Returns the map element with the specified key. If such an element does
     * not exist, the generator callback function will be invoked to create a
     * new map element.
     *
     * @param {String} key
     *  The key of the map element to be returned.
     *
     * @param {Function} [generator]
     *  A generator callback function for missing map elements. Receives the
     *  following parameters:
     *  (1) {String} key
     *      The key of the map element to be generated.
     *  The return value of this function will be inserted as new map element.
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {Any}
     *  The map element for the specified key.
     */
    SimpleMap.prototype.getOrCreate = function (key, generator, context) {
        return this.has(key) ? this.get(key) : this.insert(key, generator.call(context, key));
    };

    /**
     * Returns the map element with the specified key. If such an element does
     * not exist, a new element will be created with the passed class
     * constructor.
     *
     * @param {String} key
     *  The key of the map element to be returned.
     *
     * @param {Function} Ctor
     *  A class constructor function for missing map elements. Receives all
     *  following parameters passed to this method.
     *
     * @param {Any} [...]
     *  All following parameters will be passed to the class constructor.
     *
     * @returns {Any}
     *  The map element for the specified key.
     */
    SimpleMap.prototype.getOrConstruct = function (key, Ctor) {
        return this.has(key) ? this.get(key) : this.insert(key, (function (args) {

            // performance shortcut for few number of arguments
            switch (args.length) {
                case 2: return new Ctor();
                case 3: return new Ctor(args[2]);
                case 4: return new Ctor(args[2], args[3]);
                case 5: return new Ctor(args[2], args[3], args[4]);
            }

            // construct a new instance with arbitrary number of arguments on the fly
            return (function () {
                function Helper() { return Ctor.apply(this, _.toArray(args).slice(2)); }
                Helper.prototype = Ctor.prototype;
                return new Helper();
            }());
        }(arguments)));
    };

    /**
     * Invokes the passed callback function, if this map contains an element
     * with the specified key.
     *
     * @param {String} key
     *  The key of a map element.
     *
     * @param {Function} callback
     *  The callback function that will be invoked if this map contains an
     *  element with the sepcified key. Receives the following parameters:
     *  (1) {Any} value
     *      The map element with the specified key.
     *
     * @param {Object} context
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The return value of the callback function, if the map element exists
     *  and it has been invoked; otherwise undefined.
     */
    SimpleMap.prototype.with = function (key, callback, context) {
        if (this.has(key)) { return callback.call(context, this.get(key)); }
    };

    /**
     * Returns a new map where the values of this map are the keys, and the
     * keys of this map are the values.
     *
     * @returns {SimpleMap}
     *  A new map with swapped values and keys.
     */
    SimpleMap.prototype.invert = function () {
        var map = new SimpleMap();
        this.forEach(function (value, key) { map.insert(value, key); });
        return map;
    };

    // exports ================================================================

    return SimpleMap;

});
