/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/indexset', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // class IndexSet =========================================================

    /**
     * A helper class that represents a set of integral indexes, and preovides
     * methods to efficiently insert and remove such indexes, and find undused
     * indexes.
     *
     * @constructor
     */
    function IndexSet() {

        this._intervals = [];

    } // class IndexSet

    // static methods ---------------------------------------------------------

    /**
     * Creates an index set from the integers in the specified list.
     *
     * @param {Array|Object|BaseMap} list
     *  A list that provides a forEach() method, or that can be iterated over
     *  with the method _.each(). The elements of the list MUST BE integers.
     *
     * @returns {IndexSet}
     *  A new insex set with all indexes in the passed list.
     */
    IndexSet.from = function (list) {
        var set = new IndexSet();
        var each = list.forEach ? list.forEach.bind(list) : _.each.bind(_, list);
        each(function (index) { set.insert(index); });
        return set;
    };

    /**
     * Creates an index set from a specified integer property of all objects in
     * the specified list.
     *
     * @param {Array<Object>|Object<Object>|BaseMap<Object>} list
     *  A list that provides a forEach() method, or that can be iterated over
     *  with the method _.each(). The elements of the list MUST BE objects, the
     *  specified property values MUST BE integers.
     *
     * @param {String} prop
     *  The name of the property to pluck from the objects in the passed list.
     *
     * @returns {IndexSet}
     *  A new insex set with all indexes plucked from the passed list.
     */
    IndexSet.pluck = function (list, prop) {
        var set = new IndexSet();
        var each = list.forEach ? list.forEach.bind(list) : _.each.bind(_, list);
        each(function (value) { set.insert(value[prop]); });
        return set;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this set is empty.
     *
     * @returns {Boolean}
     *  Whether this set is empty.
     */
    IndexSet.prototype.empty = function () {
        return this._intervals.length === 0;
    };

    /**
     * Removes all indexes from this set.
     *
     * @returns {IndexSet}
     *  A reference to this instance.
     */
    IndexSet.prototype.clear = function () {
        this._intervals = [];
        return this;
    };

    /**
     * Creates a clone of this set.
     *
     * @returns {IndexSet}
     *  The clone of this set.
     */
    IndexSet.prototype.clone = function () {
        var clone = new IndexSet();
        this._intervals.forEach(function (interval) {
            clone._intervals.push({ a: interval.a, b: interval.b });
        });
        return clone;
    };

    /**
     * Returns whether this set contains the specified index.
     *
     * @param {Number} index
     *  The index to be checked.
     *
     * @returns {Boolean}
     *  Whether this set contains the specified index.
     */
    IndexSet.prototype.has = function (index) {
        var interval = Utils.findLast(this._intervals, function (interval) { return interval.a <= index; }, { sorted: true });
        return !!interval && (index <= interval.b);
    };

    /**
     * Inserts an index into this set.
     *
     * @param {Number} index
     *  The index to be inserted.
     *
     * @returns {Number}
     *  The passed index, for convenience.
     */
    IndexSet.prototype.insert = function (index) {

        // array index of the interval that contains or precedes the passed index
        var ai = Utils.findLastIndex(this._intervals, function (interval) { return interval.a <= index; }, { sorted: true });

        // nothing to do, if index is already contained in the interval
        var interval = this._intervals[ai];
        if (interval && (index <= interval.b)) { return index; }

        // extend the interval, if it ends exactly before the index; otherwise create a new interval
        if (interval && (interval.b + 1 === index)) {
            interval.b = index;
        } else {
            this._intervals.splice(ai += 1, 0, interval = { a: index, b: index });
        }

        // try to merge with the next interval in the array
        var nextInterval = this._intervals[ai + 1];
        if (nextInterval && (interval.b + 1 === nextInterval.a)) {
            interval.b = nextInterval.b;
            this._intervals.splice(ai + 1, 1);
        }

        return index;
    };

    /**
     * Removes an index from this set.
     *
     * @param {Number} index
     *  The index to be removed.
     *
     * @returns {Number}
     *  The passed index, for convenience.
     */
    IndexSet.prototype.remove = function (index) {

        // array index of the interval that contains or precedes the passed index
        var ai = Utils.findLastIndex(this._intervals, function (interval) { return interval.a <= index; }, { sorted: true });

        // nothing to do, if index is not contained in the interval
        var interval = this._intervals[ai];
        if (!interval || (interval.b < index)) { return index; }

        // delete a single-index interval; or shorten the interval, if the index is located at beginning or end; or split it
        if (interval.a === interval.b) {
            this._intervals.splice(ai, 1);
        } else if (interval.a === index) {
            interval.a += 1;
        } else if (interval.b === index) {
            interval.b -= 1;
        } else {
            this._intervals.splice(ai + 1, 0, { a: index + 1, b: interval.b });
            interval.b = index - 1;
        }

        return index;
    };

    /**
     * Returns the least index that is not contained in this set.
     *
     * @param {Number} [minIndex=0]
     *  The lower boundary for the returned unused index. If omitted, the
     *  indexes in this set are assumed to be non-negative (lower boundary of
     *  zero).
     *
     * @returns {Number}
     *  The least index that is not contained in this set.
     */
    IndexSet.prototype.firstFree = function (minIndex) {

        // the interval that contains or precedes the passed minimum index
        minIndex = minIndex || 0;
        var interval = Utils.findLast(this._intervals, function (interval) { return interval.a <= minIndex; }, { sorted: true });

        // return passed index, if the interval ends before the index; otherwise return successor of interval
        return (!interval || (interval.b < minIndex)) ? minIndex : (interval.b + 1);
    };

    /**
     * Inserts and returns the least index that is not contained in this set.
     *
     * @param {Number} [minIndex=0]
     *  The lower boundary for the returned unused index. If omitted, the
     *  indexes in this set are assumed to be non-negative (lower boundary of
     *  zero).
     *
     * @returns {Number}
     *  The index that has been inserted into this set.
     */
    IndexSet.prototype.reserve = function (minIndex) {
        return this.insert(this.firstFree(minIndex));
    };

    /**
     * Invokes the passed callback function for all indexes in this set in
     * ascending order.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each index. Receives the
     *  following parameters:
     *  (1) {Number} index
     *      The current index.
     *  (2) {IndexSet} set
     *      A reference to this set instance.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {IndexSet}
     *  A reference to this instance.
     */
    IndexSet.prototype.forEach = function (callback, context) {
        this._intervals.forEach(function (interval) {
            for (var index = interval.a; index <= interval.b; index += 1) {
                callback.call(context, index, this);
            }
        }, this);
        return this;
    };

    // exports ================================================================

    return IndexSet;

});
