/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/utils/snapshot', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/textutils'
], function (DrawingFrame, AttributeUtils, BaseObject, DOM, Utils) {

    'use strict';

    // class SnapShot =====================================================

    /**
     * An instance of this class represents the snapshot handler class that
     * can be used to generate and apply snapshot of the current document.
     *
     * @constructor
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function Snapshot(app) {

        var // self reference
            self = this,
            // the text model object
            model = app.getModel(),
            // the page node of the text document
            pageNode = model.getNode(),
            // the current operation state number
            osn = 0,
            // some application specific data in the model
            modelData = null,
            // the collector for list style assigned to slides and drawings
            slideListStyles = null, drawingListStyles = null,
            // the nodes for page content, comment layer and drawing layer
            contentNode = null, commentLayer = null, drawingLayer = null,
            // the nodes for header wrapper and footer wrapper
            headerWrapper = null, footerWrapper = null,
            // the nodes for master slide layer and layout slide layer
            masterSlideLayer = null, layoutSlideLayer = null,
            // active slide id
            activeSlideId = null,
            // the margin at the page, that is used for centering the page including comment layer
            commentMargin = null,
            // the current root node (in which content is deleted, inserted, or gets new attributes)
            currentRootNode = model.getCurrentRootNode();

        // base constructors --------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Helper function that repaints all canvas elements after the cloned drawing node
         * was inserted into the DOM.
         * For performance reasons only those drawings are repainted, that have explicitely
         * set a line or a fill attribute.
         */
        function redrawAllCanvasDrawings() {

            _.each(pageNode.find('.drawing > .content > canvas'), function (canvasNode) {

                var // one drawing node, must be 'jQuerified'
                    drawing = $(canvasNode.parentNode.parentNode),
                    // the explicit attributes at the drawing
                    explicitAttrs = AttributeUtils.getExplicitAttributes(drawing);

                if (explicitAttrs && ((explicitAttrs.line && explicitAttrs.line.type !== 'none') || (explicitAttrs.fill && explicitAttrs.fill.type !== 'none'))) {
                    // updating only those drawings, that have expicitely set line attributes or fill attributes
                    DrawingFrame.updateFormatting(app, drawing, model.getDrawingStyles().getElementAttributes(drawing));
                }
            });

        }

        /**
         * Creating a snapshot of the current document state.
         */
        var createSnapshot = Utils.profileMethod('Snapshot.createSnapshot(): ', function () {

            var // the comment layer node
                commentLayerNode = DOM.getCommentLayerNode(pageNode),
                // the drawing layer node for absolute positioned drawings
                drawingLayerNode = DOM.getTextDrawingLayerNode(pageNode),
                // the header wrapper node
                headerWrapperNode = null,
                // the footer wrapper node
                footerWrapperNode = null,
                // the master slide layer node
                masterSlideLayerNode = null,
                // the layout slide layer node
                layoutSlideLayerNode = null;

            // saving the operation state number
            osn = model.getOperationStateNumber();

            // adding all slides into the page, before creating snapshot
            if (model.useSlideMode()) { model.appendAllSlidesToDom(); }

            // saving the current content node
            contentNode = DOM.getPageContentNode(pageNode).clone(true, true);

            if (model.useSlideMode()) { // OX Presentation specific nodes
                activeSlideId = model.getActiveSlideId();

                // getting the layer nodes for master and layout slides
                masterSlideLayerNode = model.getMasterSlideLayerNode();
                layoutSlideLayerNode = model.getLayoutSlideLayerNode();

                if (masterSlideLayerNode && masterSlideLayerNode.length > 0) { masterSlideLayer = masterSlideLayerNode.clone(true, true); }
                if (layoutSlideLayerNode && layoutSlideLayerNode.length > 0) { layoutSlideLayer = layoutSlideLayerNode.clone(true, true); }

                // saving the list styles, that are assigned to slides and drawings. These list styles cannot be restored
                // from the slides data-object, because the list styles are not stored there.
                slideListStyles = model.getAllSlideListStyleAttributes();
                drawingListStyles = model.getDrawingStyles().getAllDrawingListStyleAttributes();

                // saving the values for 'mainSlideCounter', 'activeSlideId' and 'isMasterView'
                modelData = model.getModelDataObject();

            } else { // OX Text specific nodes

                // saving the comment layer node
                if (commentLayerNode.length > 0) {
                    commentLayer = commentLayerNode.clone(true, true);
                    // saving the right margin information saved at the page (might be modified during removal of content)
                    if (pageNode.hasClass(DOM.COMMENTMARGIN_CLASS)) { commentMargin = pageNode.data(DOM.COMMENTMARGIN_CLASS); }
                }

                // saving the drawing layer node
                if (drawingLayerNode.length > 0) { drawingLayer = drawingLayerNode.clone(true, true); }

                // evaluating the current root node -> check if the first header or last footer need to be exchanged)
                // -> long running stoppable actions in first header or last footer should never occur
                if (DOM.isHeaderWrapper(currentRootNode.parent())) {
                    // receiving the header wrapper
                    headerWrapperNode = DOM.getHeaderWrapperNode(pageNode);
                    // saving the header wrapper node
                    if (headerWrapperNode.length > 0) { headerWrapper = headerWrapperNode.clone(true, true); }
                }

                if (DOM.isFooterWrapper(currentRootNode.parent())) {
                    // receiving the footer wrapper
                    footerWrapperNode = DOM.getFooterWrapperNode(pageNode);
                    // saving the header wrapper node
                    if (footerWrapperNode.length > 0) { footerWrapper = footerWrapperNode.clone(true, true); }
                }
            }

        });

        // methods ------------------------------------------------------------

        /**
         * Applying the snapshot.
         */
        this.apply = Utils.profileMethod('Snapshot.applySnapshot(): ', function () {

            var // the page content node
                currentContentNode = DOM.getPageContentNode(pageNode);

            // replacing the main content node
            currentContentNode.replaceWith(contentNode);

            // restoring header and footer wrapper as children of the page (should never be necessary)
            if (headerWrapper && DOM.getHeaderWrapperNode(pageNode).length > 0) { DOM.getHeaderWrapperNode(pageNode).replaceWith(headerWrapper); }
            if (footerWrapper && DOM.getFooterWrapperNode(pageNode).length > 0) { DOM.getFooterWrapperNode(pageNode).replaceWith(footerWrapper); }

            // restoring the comment layer node
            if (commentLayer) { model.getCommentLayer().getOrCreateCommentLayerNode().replaceWith(commentLayer); }
            // restoring the drawing layer node
            if (drawingLayer) { model.getDrawingLayer().getDrawingLayerNode().replaceWith(drawingLayer); }

            // but removing comment layer and drawing layer if they were created by a paste operation
            if (!commentLayer && DOM.getCommentLayerNode(pageNode).length > 0) { DOM.getCommentLayerNode(pageNode).remove(); }
            if (!drawingLayer && DOM.getTextDrawingLayerNode(pageNode).length > 0) { DOM.getTextDrawingLayerNode(pageNode).remove(); }

            // restoring the master slide layer and the layout slide layer
            if (masterSlideLayer) { model.getMasterSlideLayerNode().replaceWith(masterSlideLayer); }
            if (layoutSlideLayer) { model.getLayoutSlideLayerNode().replaceWith(layoutSlideLayer); }

            // after all nodes are exchanged, it is necessary to update all models and all cached nodes

            // updating cached page content node in page layout object (updating all cached content nodes)
            model.getPageLayout().refreshPageContentNode();

            // triggering that the model will  be recreated -> the models need to be rebuilt from scratch
            model.trigger('document:reloaded', self);

            // setting the OSN
            model.setOperationStateNumber(osn);

            // setting the complete object of the slide list styles
            if (slideListStyles) { model.setAllSlideListStyleAttributes(slideListStyles); }

            // setting the object for all drawing list styles
            if (drawingListStyles) { model.getDrawingStyles().setAllDrawingListStyleAttributes(drawingListStyles); }

            // setting application specific data into the model
            if (modelData) { model.setModelDataObject(modelData); }

            // if active slide id is stored, set it again
            if (activeSlideId) { model.setActiveSlideId(activeSlideId, { forceUpdate: true }); }

            // triggering to listeners that all models are completely recreated
            model.trigger('document:reloaded:after', self);

            // all canvas elements need to be repainted
            redrawAllCanvasDrawings();
        });

        /**
         * Returns the width of the 'comment margin' at the right side of the page in pixel.
         * Or null, if no such margin is set.
         *
         * @returns {Number|null}
         */
        this.getCommentMargin = function () {
            return commentMargin;
        };

        // initialization -----------------------------------------------------

        /**
         * Destroy all class members on destruction
         */
        this.registerDestructor(function () {
            model = pageNode = null;
        });

        //
        createSnapshot();

    } // class SnapShot

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class TriggerObject
    return BaseObject.extend({ constructor: Snapshot });
});
