/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/textframework/selection/remoteselection', [
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom'
], function (BaseObject, TimerMixin, Config, Utils, Position, DOM) {

    'use strict';

    // class RemoteSelection ==================================================

    /**
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     */
    function RemoteSelection(app, rootNode) {

        var // the local selection instance
            selection = null,

            // the display timer of the collaborative overlay
            overlayTimer = null,

            // list with all drawings nodes with mouseenter and mouseleave handlers with the namespace 'remoteselection'
            drawingNodesWithHandlers = [],

            self = this;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Calculate physical positions from logical selection.
         *
         * @param {DOM.Point} anchorPoint
         *  the selection start point
         *
         * @param {DOM.Point} focusPoint
         *  the selection end point.
         *
         * @returns {Object | null}
         *  An selection object containing physical start and end coordinates.
         *  - {Number} Object.anchor - selection start absolute coordinate in pixels
         *  - {Number} Object.focus -  selection end absolute coordinate in pixels
         */
        function calculatePhysicalPosition(anchorPoint, focusPoint) {

            // quit if no logical selection is available
            if (!anchorPoint || !focusPoint) { return null; }

            var // get logical start and end positions for restoring text selection later
                selectionStart = selection.getStartPosition(),
                selectionEnd = selection.getEndPosition(),
                backwards = selection.isBackwards(),
                zoom = app.getView().getZoomFactor() / 100;

            // use decoy span technique and get absolute physical positions of a selection point
            function getPhysicalPosition(point) {

                // ignore this current user's own selection points of course
                if (!point) { return null; }

                var caretSpan = $('<span>').addClass('collaborative-caret').text('|'),
                    cursorElement = $(point.node.parentNode),
                    cursorElementLineHeight = parseFloat(cursorElement.css('line-height'));

                // break cursor element on the text offset
                if (point.offset === 0) {
                    cursorElement.before(caretSpan);
                } else {
                    DOM.splitTextSpan(cursorElement, point.offset, { append: true });
                    cursorElement.after(caretSpan);
                }

                // create caret overlay and calculate its position
                var caretTop = (caretSpan.offset().top - rootNode.offset().top) / zoom  - cursorElementLineHeight  + caretSpan.outerHeight(),
                    caretLeft = (caretSpan.offset().left - rootNode.offset().left) / zoom;

                // restore original state of document
                caretSpan.remove();

                if (point.offset > 0) {
                    Utils.mergeSiblingTextSpans(cursorElement, true);
                }

                return { top: caretTop, left: caretLeft };
            }

            var position = { anchor: getPhysicalPosition(anchorPoint), focus: getPhysicalPosition(focusPoint) };

            // only reset text selection if the page node is the active element.
            // Otherwise, there is a overlay which gets the focus (BUG #36469)
            if (DOM.isPageNode(Utils.getActiveElement())) {
                // return original position of browser cursor after merge

                // never destroying multiple selection (TODO)
                if (!(selection.isMultiSelectionSupported() && selection.isMultiSelection())) {
                    selection.setTextSelection(selectionStart, selectionEnd, { userDataUpdateHandler: true, backwards: backwards });
                }

            }

            return position;
        }

        /**
         * Draw overlays for a specific remote user's selection debounced.
         */
        var renderCollaborativeSelectionDebounced = this.createDebouncedMethod($.noop, function () {
            if (Config.SHOW_REMOTE_SELECTIONS) {
                // first, clear all old selection nodes from the DOM
                rootNode.find('.collaborative-overlay').children().remove();

                removeDrawingNodesHandler();

                // render all remote selections
                _.each(app.getActiveClients(), renderSelection);
            }
        }, { delay: 100, infoString: 'RemoteSelection renderCollaborativeSelectionDebounced()', app: app });

        /**
         * Create a Overlay with the username to show the remote username for a selection.
         * @param {type} username name of the user
         * @param {type} className classname for the overlay
         * @param {jQuery} UserName Overlay
         */
        function getUserNameOverlay(username, className) {
            var usernameOverlay = (_.browser.IE === 10) ? document.createElementNS('http://www.w3.org/2000/svg', 'svg') : document.createElement('div');

            // draw user name
            usernameOverlay.setAttribute('class', 'collaborative-username ' + className);
            if (_.browser.IE === 10) { // create SVG text node and calculate its width (no text wrap support in SVG)
                var txtElem = document.createElementNS('http://www.w3.org/2000/svg', 'text');
                $(txtElem).attr({ x: 5, y: 13, style: 'font-size:12px;fill:white;' });
                txtElem.appendChild(document.createTextNode(username));
                usernameOverlay.appendChild(txtElem);
                $(usernameOverlay).attr({ width: (username.length * 8) + 'px', style: 'height:2em' });
            } else {
                $(usernameOverlay).text(username);
            }

            return usernameOverlay;
        }

        /**
         * Draw overlays for a specific remote user's selection.
         *
         * @param {Object} client
         *  Descriptor of a client editing this document.
         */
        function renderSelection(client) {

            var // the selection of the client
                userSelections = client.userData.selections,
                userTargetSelection = client.userData.target,
                userTargetIndex = client.userData.targetIndex;

            // helper function to determine if selection is on active slide
            function isSelectionOnActiveSlide(userSelection) {
                var state = false;
                var activeSlideId = docModel.getActiveSlideId();

                if (docModel.isLayoutOrMasterId(activeSlideId)) {
                    state = activeSlideId === client.userData.target;
                } else {
                    state = docModel.getActiveSlideIndex() === _.first(userSelection.start);
                }
                return state;
            }

            // draw nothing and quit if:
            // - user is the current user himself
            // - no user data/selection is available.
            if (!client.remote || !userSelections || userSelections.length === 0) {
                return;
            }

            var username = client.userName,
                overlay = rootNode.find('.collaborative-overlay'),
                className = 'user-' + client.colorIndex,
                svgNamespace = 'http://www.w3.org/2000/svg',
                zoom = app.getView().getZoomFactor() / 100,
                docModel = app.getModel(),
                useSlideMode = docModel.useSlideMode(),
                // skip the render of drawing selections if one drawing is not on the active slide
                skipDrawing = false,
                usernameOverlay;

            // loop through the selections of the user
            _.each(userSelections, function (userSelection) {

                // special handling if selection is a top level cursor position on a slide
                // -> not drawing the remote cursor on the slide.
                if (skipDrawing || (useSlideMode && selection.isTopLevelTextCursor(userSelection))) {
                    // TODO: Draw remote selection (on slide pane?)
                    return;
                }

                // Don't draw selection if is on inactive slide, #46237
                if (useSlideMode && !isSelectionOnActiveSlide(userSelection)) { return; }

                var activeRootNode = docModel.getRootNode(userTargetSelection, userTargetIndex),
                    selectionAnchorPoint = Position.getDOMPosition(activeRootNode, userSelection.start),
                    selectionFocusPoint = Position.getDOMPosition(activeRootNode, userSelection.end);

                if (!selectionAnchorPoint || !selectionFocusPoint) { return; }

                var selectionOverlayGroup = $('<div>').addClass('collaborative-selection-group');

                // render the selection for drawings
                if (userSelection.type === 'drawing') {
                    // Test if the drawing is on the aktive slide (Presentation), if not skip the rendering.
                    if (useSlideMode) {
                        var slideID = docModel.getActiveSlideId();
                        var layoutOrMaster = docModel.isMasterView();
                        if (layoutOrMaster) {
                            if (slideID !== client.userData.target) {
                                skipDrawing = true;
                            }
                            // TODO select slidePane if the target is layoutOrMaster too
                        } else {
                            if (client.userData.target || userSelection.start[0] !== docModel.getActiveSlideIndex()) {
                                skipDrawing = true;
                            }
                            // TODO select slidePane it the target is not layoutOrMaster
                        }
                        if (skipDrawing) {
                            return;
                        }
                    }
                    var drawingNode = $(Position.getDOMPosition(activeRootNode, userSelection.start, true).node);
                    if (drawingNode.length === 0) {
                        return;
                    }
                    var drawingOverlay = (_.browser.IE === 10) ? document.createElementNS(svgNamespace, 'svg') : document.createElement('div'),
                        top = useSlideMode ? parseFloat(drawingNode.css('top')) : (drawingNode.offset().top - rootNode.offset().top) / zoom,
                        left = useSlideMode ? parseFloat(drawingNode.css('left')) : (drawingNode.offset().left - rootNode.offset().left) / zoom;

                    usernameOverlay = getUserNameOverlay(username, className);

                    // Set the selection frame position and size attributes
                    $(selectionOverlayGroup).css({
                        top: top - 2,
                        left: left - 2,
                        width: parseFloat(drawingNode.css('width')) + 4,
                        height: parseFloat(drawingNode.css('height')) + 4,
                        transform: drawingNode.css('transform')
                    });

                    $(usernameOverlay).css({ top: 0, left: 0 });

                    selectionOverlayGroup.append(drawingOverlay);
                    selectionOverlayGroup.append(usernameOverlay);

                    // Add 'hover' handler with namespace to show the Username Overlay.
                    // The namespace is to remove the event handler without the function.
                    drawingNode.on({
                        'mouseenter.remoteselection': function () {
                            $(usernameOverlay).show();
                        },
                        'mouseleave.remoteselection': function () {
                            $(usernameOverlay).hide();
                        }
                    });

                    drawingNodesWithHandlers.push(drawingNode);
                    drawingOverlay.setAttribute('class', 'selection-overlay drawing ' + className);
                } else {
                    var caretHandle = $('<div>').addClass('collaborative-cursor-handle'),
                        caretOverlay = $('<div>').addClass('collaborative-cursor'),
                        startNode = $(selectionAnchorPoint.node.parentNode),
                        endNode = $(selectionFocusPoint.node.parentNode),
                        physicalPosition = calculatePhysicalPosition(selectionAnchorPoint, selectionFocusPoint),
                        isRangeSelection = !_.isEqual(physicalPosition.anchor, physicalPosition.focus);

                    usernameOverlay = getUserNameOverlay(username, className);

                    // draw user cursors and names
                    $(usernameOverlay).css('top', endNode.css('line-height'));
                    caretOverlay.css({ top: physicalPosition.focus.top, left: physicalPosition.focus.left, height: parseFloat(endNode.css('line-height')) }).addClass(className);
                    caretOverlay.append(usernameOverlay, caretHandle);
                    caretHandle.hover(function () { $(usernameOverlay).show(); }, function () { $(usernameOverlay).hide(); });
                    overlay.append(caretOverlay);

                    // if remote selection is in header/footer, fix z-index collision.
                    if (client.editor) {
                        if (userTargetSelection) {
                            overlay.addClass('target-overlay');
                        } else {
                            overlay.removeClass('target-overlay');
                        }
                    }

                    if (isRangeSelection) {

                        // draw collaborative selection area highlighting if we have an 'range' selection

                        var startParentNode = startNode.parent(),
                            endParentNode = endNode.parent(),
                            highlightWidth = Math.max(startParentNode.width(), endParentNode.width()),
                            rootNodePos = rootNode.offset(),
                            pageContentNodePos = DOM.getPageContentNode(rootNode).offset(),
                            startTop = physicalPosition.anchor.top,
                            startLeft = physicalPosition.anchor.left,
                            endTop = physicalPosition.focus.top,
                            endLeft = physicalPosition.focus.left,
                            startNodeCell = startNode.closest('td'),
                            endNodeCell = endNode.closest('td'),
                            isTableSelection = startNodeCell.length && endNodeCell.length && !startNodeCell.is(endNodeCell);

                        // special handling of pure table selection
                        if (isTableSelection) {

                            // handle special mega cool firefox cell selection
                            if (userSelection.type === 'cell') {

                                var ovStartPos = startNodeCell.offset(),
                                    ovEndPos = endNodeCell.offset(),
                                    ovWidth = (ovEndPos.left  - ovStartPos.left) / zoom + endNodeCell.outerWidth(),
                                    ovHeight = (ovEndPos.top  - ovStartPos.top) / zoom + endNodeCell.outerHeight(),
                                    ov = (_.browser.IE === 10) ? document.createElementNS(svgNamespace, 'svg') : document.createElement('div');

                                $(ov).css({
                                    top: (ovStartPos.top - rootNodePos.top) / zoom,
                                    left: (ovStartPos.left - rootNodePos.left) / zoom,
                                    width: ovWidth,
                                    height: ovHeight
                                });

                                selectionOverlayGroup.append(ov);

                            } else { // normal table selection (Chrome, IE): iterate cells between start and end

                                var cells = $(Utils.findFarthest(rootNode, endNodeCell, 'table')).find('td'),
                                    cellsToHighlight = cells.slice(cells.index(startNodeCell), cells.index(endNodeCell) + 1);

                                cellsToHighlight.each(function () {

                                    var cellOv = (_.browser.IE === 10) ? document.createElementNS(svgNamespace, 'svg') : document.createElement('div'),
                                        offset = $(this).offset();

                                    $(cellOv).css({
                                        top: (offset.top - rootNodePos.top) / zoom,
                                        left: (offset.left - rootNodePos.left) / zoom,
                                        width: $(this).outerWidth(),
                                        height: $(this).outerHeight()
                                    });

                                    selectionOverlayGroup.append(cellOv);

                                });
                            }
                        } else { // paragraph / mixed selection

                            var startHeight = parseFloat(startNode.css('line-height')),
                                endHeight = parseFloat(endNode.css('line-height')),
                                startBottom = startTop + startHeight,
                                endBottom = endTop + endHeight,
                                isSingleLineSelection = startBottom === endBottom;

                            // selection area is in a line
                            if (isSingleLineSelection) {

                                var selectionOverlay = (_.browser.IE === 10) ? document.createElementNS(svgNamespace, 'svg') : document.createElement('div');
                                $(selectionOverlay).css({ top: Math.min(startTop, endTop), left: startLeft, width: endLeft - startLeft, height: Math.max(startHeight, endHeight) });
                                selectionOverlayGroup.append(selectionOverlay);

                            } else { // multi line selection area

                                var startPrevNode = $(Utils.findPreviousSiblingNode(startNode)),
                                    endPrevNode = $(Utils.findPreviousSiblingNode(endNode)),
                                    isListSelection = DOM.isListLabelNode(startPrevNode) || DOM.isListLabelNode(endPrevNode);

                                // start and end node are empty lines (paragraphs)
                                if (!highlightWidth) { highlightWidth = rootNode.width(); }

                                var headOverlay = document.createElement('div'),
                                    bodyOverlay = document.createElement('div'),
                                    tailOverlay = document.createElement('div');

                                if (_.browser.IE === 10) {
                                    headOverlay = document.createElementNS(svgNamespace, 'svg');
                                    bodyOverlay = document.createElementNS(svgNamespace, 'svg');
                                    tailOverlay = document.createElementNS(svgNamespace, 'svg');
                                }

                                $(headOverlay).css({
                                    top: startTop,
                                    left: startLeft,
                                    width: isListSelection ? (startParentNode.offset().left - rootNodePos.left) / zoom  + startParentNode.width() - startLeft + startPrevNode.width() :
                                        (startParentNode.offset().left - rootNodePos.left) / zoom  + startParentNode.width() - startLeft,
                                    height: startHeight
                                });

                                $(bodyOverlay).css({
                                    top: startTop + startHeight,
                                    left: (isListSelection ? pageContentNodePos.left - rootNodePos.left : Math.min(startParentNode.offset().left, endParentNode.offset().left) - rootNodePos.left) / zoom,
                                    width: isListSelection ? rootNode.width() : highlightWidth,
                                    height: endTop - startTop - startHeight
                                });

                                $(tailOverlay).css({
                                    top: endTop,
                                    left: isListSelection ? (endNode.offset().left - rootNodePos.left) / zoom - endPrevNode.width() :
                                        (endParentNode.offset().left - rootNodePos.left) / zoom,
                                    width: endLeft - ((endParentNode.offset().left - rootNodePos.left) / zoom),
                                    height: endHeight
                                });

                                selectionOverlayGroup.append(headOverlay, bodyOverlay, tailOverlay);
                            }
                        }

                        _.each(selectionOverlayGroup.children(), function (overlayElement) {
                            overlayElement.setAttribute('class', 'selection-overlay ' + className);
                        });
                    }
                }

                if (!_.isBoolean(client.selectionChanged)) { $(usernameOverlay).show(); }  // 36800

                overlay.append(selectionOverlayGroup);
            });

            // show user name for a second
            if (overlayTimer) { overlayTimer.abort(); }

            overlayTimer = self.executeDelayed(function () {
                overlay.find('.collaborative-username').fadeOut('fast');
            }, { delay: 3000, infoString: 'RemoteSelection: overlayTimer', app: app });
        }

        /**
         * Remove the 'Hover' event handlers to show the username tooltip from the selected drawing layer.
         */
        function removeDrawingNodesHandler() {
            // remove the handlers from the drawing nodes
            drawingNodesWithHandlers.forEach(function (drawingNode) {
                drawingNode.off('mouseenter.remoteselection mouseleave.remoteselection');
            });

            drawingNodesWithHandlers = [];
        }

        // public methods -----------------------------------------------------

        /**
         * Draw overlays for the selections of all remote clients.
         *
         * @returns {RemoteSelection}
         *  A reference to this instance.
         */
        this.renderCollaborativeSelections = function () {
            renderCollaborativeSelectionDebounced();
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members when application is alive
        app.onInit(function () {
            removeDrawingNodesHandler();
            selection = app.getModel().getSelection();
            // render selection if the user change the slide to see the current selection of the new slide
            self.listenTo(app.getModel(), 'change:slide', _.bind(self.renderCollaborativeSelections, self));
            // render selection if drawing attributes will be changed
            self.listenTo(app.getModel(), 'change:drawing', _.bind(self.renderCollaborativeSelections, self));
            // render selection if the list of active clients has changed
            self.listenTo(app, 'docs:users:selection', _.bind(self.renderCollaborativeSelections, self));
        });

        // destroy all class members
        this.registerDestructor(function () {
            app = rootNode = selection = drawingNodesWithHandlers = null;
        });

    } // class RemoteSelection

    // exports ================================================================

    return BaseObject.extend({ constructor: RemoteSelection });

});
