/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/groupoperationmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/textutils'
], function (DrawingFrame, AttributeUtils, DOM, Operations, Position, Utils) {

    'use strict';

    // mix-in class GroupOperationMixin ======================================

    /**
     * A mix-in class for the document model class providing the group
     * operation handling used in a presentation and text document. This
     * includes the grouping and ungrouping of drawings.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The application instance.
     */
    function GroupOperationMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Handler for grouping the specified drawings.
         *
         * @param {Number[]} start
         *  The logical position of the grouped drawing.
         *
         * @param {String} target
         *  The id of the layout slide, that is the base for the specified slide.
         *
         * @param {Number[][]} drawings
         *  The logical positions of all drawings that will be grouped.
         *
         * @param {Object} attrs
         *  The drawing attributes. Especially the position and the size of the
         *  drawing group.
         *
         * @param {Object[]} [childAttrs]
         *  The drawing attributes of the grouped drawings. Especially the 'top'
         *  and 'left' values cannot be calculated for drawing that will be grouped
         *  into a rotated drawing group. Therefore this parameter is only necessary
         *  for rotated drawing groups. And such an operation can only be created
         *  as an undo of an ungrouping of a rotated drawing group.
         *
         * @returns {Boolean}
         *  Whether the slide has been inserted successfully.
         */
        function implGroupDrawings(start, target, drawings, attrs, childAttrs) {

            // grouping the drawings at the logical positions
            // -> inserting the new drawing of type 'group'
            // -> shifting all drawings into the drawing of type 'group'

            // In OX Presentation all drawings are on the same slide. In OX Text the
            // drawings can be positioned anywhere.

            var // text span that will precede the drawing group
                span = null,
                // deep copy of attributes, because they are modified in webkit browsers
                attributes = _.copy(attrs, true),
                // the target node for the logical positions
                targetNode = null,
                // new drawing node
                drawingNode = null,
                // the content node inside the new drawing node
                contentNode = null,
                // the function used to insert the new drawing frame
                insertFunction = 'insertAfter',
                // an array with all jQuerified drawing nodes.
                allChildDrawings = null,
                // the logical slide position
                slidePos = _.initial(start);

            try {
                span = self.prepareTextSpanForInsertion(start, {}, target);
            } catch (ex) {
            }

            if (!span) { return false; }

            // For the user with edit rights, the drawings are already selected. They
            // can directly be taken from the selection. All other clients need to find
            // the drawings with their logical positions.
            // -> but not if undo/redo is running
            if (self.getEditMode() && !self.isUndoRedoRunning()) {
                allChildDrawings = self.getSelection().getArrayOfSelectedDrawingNodes();
            } else {
                // collecting all drawings, before the new group drawing is inserted
                targetNode = self.getRootNode(target);
                allChildDrawings = [];
                _.each(drawings, function (drawingPosition) {
                    // -> collect all drawings in 'allChildDrawings'
                    var oneDrawing = null;
                    var position = _.clone(slidePos);
                    position.push(drawingPosition);
                    oneDrawing = Position.getDOMPosition(targetNode, position, true);
                    if (oneDrawing && oneDrawing.node && DrawingFrame.isDrawingFrame(oneDrawing.node)) {
                        allChildDrawings.push($(oneDrawing.node));
                    } else {
                        return false; // invalid logical position
                    }
                });
            }

            // insert the drawing with default settings between the two text nodes (store original URL for later use)
            drawingNode = DrawingFrame.createDrawingFrame('group')[insertFunction](span);
            contentNode = DrawingFrame.getContentNode(drawingNode);

            // Shifting all children into the group container
            // iterating over all child drawing nodes
            _.each(allChildDrawings, function (drawing) {
                var prevSpan = drawing.prev();
                drawing.addClass(DrawingFrame.GROUPED_NODE_CLASS);
                contentNode.append(drawing);
                Utils.mergeSiblingTextSpans(prevSpan, true); // merge with next span
            });

            // apply the passed drawing attributes
            if (_.isObject(attributes)) {

                // assigning the attributes to the group (no rotation!)
                self.getDrawingStyles().setElementAttributes(drawingNode, attributes);

                if (childAttrs) {
                    _.each(allChildDrawings, function (drawing, counter) {
                        self.getDrawingStyles().setElementAttributes(drawing, childAttrs[counter]);
                    });

                    // updating the new group once more (45853)
                    self.getDrawingStyles().updateElementFormatting(drawingNode);
                }
            }

            return true;
        }

        /**
         * Handler for ungrouping the specified drawing.
         *
         * @param {Number[]} start
         *  The logical position of the drawing.
         *
         * @param {String} target
         *  The id of the layout slide, that is the base for the specified slide.
         *
         * @param {Number[][]} drawings
         *  The (new) logical positions for all drawings that will be ungrouped. The
         *  number of logical positions must be the same as the number of child drawings
         *  in the group.
         *
         * @param {Object[]} groupAttrsContainer
         *  A collector for the drawing attributes of the drawing of type 'group'.
         *
         * @param {Object[]} childAttrsContainer
         *  A collector for the drawing attributes of the grouped drawings. This
         *  is necessary to create the undo operation for a rotated drawing group.
         *
         * @returns {Boolean}
         *  Whether the slide has been inserted successfully.
         */
        function implUngroupDrawings(start, target, drawings, groupAttrsContainer, childAttrsContainer) {

            // the target node for the logical positions
            var targetNode = self.getRootNode(target);
            // the drawing group node
            var drawingGroupNode = Position.getDOMPosition(targetNode, start, true);
            // the content node inside the new drawing group node
            var contentNode = null;
            // all drawing nodes.
            var allChildDrawings = null;
            // the span before the drawing group node
            var prevSpan = null;
            // the drawing styles object
            var drawingStyles = self.getDrawingStyles();
            // a counter
            var counter = 0;
            // a collector for the attributes of the child drawings
            var allNewDrawingAttributes = null;
            // the logical slide position
            var slidePos = _.initial(start);
            // the last value of the logical position of the grouped drawing
            var drawingPos = _.last(start);

            // the specified drawing must be a group drawing node
            if (!(drawingGroupNode && drawingGroupNode.node && DrawingFrame.isGroupDrawingFrame(drawingGroupNode.node))) { return false; }

            drawingGroupNode = $(drawingGroupNode.node);

            contentNode = DrawingFrame.getContentNode(drawingGroupNode);

            allChildDrawings = contentNode.children(DrawingFrame.NODE_SELECTOR);

            // compare the number of drawings with the number of specified logical positions
            if (drawings && drawings.length !== allChildDrawings.length) { return false; }  // invalid operation

            if (!drawings) {
                drawings = [];
                _.each(allChildDrawings, function () {
                    drawings.push(drawingPos + counter);
                    counter++;
                });
            }

            allNewDrawingAttributes = getChildDrawingAttrsAfterUngroup(drawingGroupNode, allChildDrawings, groupAttrsContainer, childAttrsContainer);

            // detaching the group from the DOM
            prevSpan = drawingGroupNode.prev();
            drawingGroupNode.detach();
            Utils.mergeSiblingTextSpans(prevSpan, true); // merge with next span

            // all children need to get new explicit attributes for 'top', 'left', 'width', 'height'
            // and 'rotation' corresponding to the position and size of the drawing of type 'group'.
            assignDrawingAttributesToDrawings(allChildDrawings, allNewDrawingAttributes);

            // iterating over all child drawings
            _.each(allChildDrawings, function (drawingNode, index) {

                var // the logical position to insert the drawing
                    pos = _.clone(slidePos),
                    // the text span, after that the drawing can be inserted
                    span = null;

                pos.push(drawings[index]);
                span = self.prepareTextSpanForInsertion(pos, {}, target);

                if (span) {
                    $(drawingNode).removeClass(DrawingFrame.GROUPED_NODE_CLASS).insertAfter(span);
                    drawingStyles.updateElementFormatting(drawingNode);
                } else {
                    return false;
                }
            });

            // finally the (empty) drawing group node (and its selection) can be removed
            if (drawingGroupNode.data('selection')) { DrawingFrame.removeDrawingSelection(drawingGroupNode); }
            drawingGroupNode.remove();

            return true;
        }

        /**
         * Getting the position attributes for a drawing group, that is specified by all its
         * children. This is used to determine the position of a new created drawing group.
         *
         * @param {jQuery|jQuery[]|Node[]} drawings
         *  A drawing container over that can be iterated.
         *
         * @returns {Object}
         *  An object containing the properties 'left', 'top', 'width' and 'height'.
         */
        function getGroupDrawingPositionAttributes(drawings) {

            var attrs = { left: -1, top: -1, width: -1, height: -1 },
                right = -1, bottom = -1;

            _.each(drawings, function (drawing) {
                // explicit attributes (is this sufficient)
                // -> Place holder drawings cannot be grouped -> no 'inherited' positions!
                var explicitAttrs = AttributeUtils.getExplicitAttributes(drawing);

                if (explicitAttrs.drawing) {

                    explicitAttrs = explicitAttrs.drawing;

                    if (!_.isNumber(explicitAttrs.left)) { explicitAttrs.left = 0; } // setting the default value (47463)
                    if (!_.isNumber(explicitAttrs.top)) { explicitAttrs.top = 0; } // setting the default value

                    if ((_.isNumber(explicitAttrs.left)) && ((attrs.left === -1) || (explicitAttrs.left < attrs.left))) { attrs.left = explicitAttrs.left; }
                    if ((_.isNumber(explicitAttrs.top)) && ((attrs.top === -1) || (explicitAttrs.top < attrs.top))) { attrs.top = explicitAttrs.top; }

                    if (_.isNumber(explicitAttrs.left) && _.isNumber(explicitAttrs.width)) {
                        if (right === -1 || right < (explicitAttrs.left + explicitAttrs.width)) {
                            right = explicitAttrs.left + explicitAttrs.width;
                        }
                    }

                    if (_.isNumber(explicitAttrs.top) && _.isNumber(explicitAttrs.height)) {
                        if (bottom === -1 || bottom < (explicitAttrs.top + explicitAttrs.height)) {
                            bottom = explicitAttrs.top + explicitAttrs.height;
                        }
                    }
                }
            });

            // switching to default value 0
            if (attrs.left === -1) { attrs.left = 0; }
            if (attrs.top === -1) { attrs.top = 0; }
            attrs.width = right - attrs.left + 1;
            attrs.height = bottom - attrs.top + 1;

            return attrs;
        }

        /**
         * Saving the specified drawing attributes in a collector.
         *
         * @param {Object[]} container
         *  A container for the drawing attributes of the grouped drawings and the drawing
         *  group itself. This 'collecting' is necessary to create the undo operation for
         *  a (rotated) drawing group.
         *
         * @param {Object} drawingAttrs
         *  The explicit drawing attributes at a drawing.
         */
        function saveRequiredDrawingAttributes(container, drawingAttrs) {

            // the object for saving the drawing attributes
            var savedDrawingAttrs = {};
            // the properties that need to be saved
            var props = ['left', 'top', 'width', 'height', 'rotation'];

            _.each(props, function (prop) {
                if (_.isNumber(drawingAttrs[prop])) { savedDrawingAttrs[prop] = drawingAttrs[prop]; }
                // rotation must be set explicitely
                if (prop === 'rotation' && !_.isNumber(savedDrawingAttrs[prop])) { savedDrawingAttrs[prop] = 0; }
            });

            container.push({ drawing: savedDrawingAttrs });
        }

        /**
         * A function that gets all the drawing attributes of all child drawings
         * of a specified drawing group. These drawing attributes needs to be
         * assigned to the child drawings, after the drawing group is removed.
         *
         * @param {jQuery} drawingGroupNode
         *  The drawing group node.
         *
         * @param {jQuery} allChildDrawings
         *  The container with all drawing children of the drawing group node.
         *
         * @param {Object[]} childAttrsContainer
         *  A collector for the drawing attributes of the grouped drawings. This
         *  is necessary to create the undo operation for a rotated drawing group.
         *
         * @returns {Object[]}
         *  A container that contains the drawing attributes in the order of the
         *  specified child drawings.
         */
        function getChildDrawingAttrsAfterUngroup(drawingGroupNode, allChildDrawings, groupAttrsContainer, childAttrsContainer) {

            // a collector for all drawing attributes of the child drawings
            var allNewDrawingAttributes = [];
            // an optional rotation angle of the group
            var groupRotationAngle = 0;
            // the jQuery offset of the complete slide
            var slideOffset = drawingGroupNode.closest(DOM.SLIDE_SELECTOR).offset();
            // the current zoom factor
            var zoomFactor = app.getView().getZoomFactor() / 100;
            // the explicit drawing attributes of the group
            var drawingGroupAttrs = null;

            // is the drawing group rotated? -> all children need to get the rotation of the group node
            if (DrawingFrame.isRotatedDrawingNode(drawingGroupNode)) { groupRotationAngle = DrawingFrame.getDrawingRotationAngle(self, drawingGroupNode); }

            // saving the original positions in the gropuAttrsContainer
            drawingGroupAttrs = AttributeUtils.getExplicitAttributes(drawingGroupNode);
            if (drawingGroupAttrs && drawingGroupAttrs.drawing) { saveRequiredDrawingAttributes(groupAttrsContainer, drawingGroupAttrs.drawing); }

            // iterating over all child drawings
            _.each(allChildDrawings, function (drawing) {

                var // the rotation angle of one child drawing
                    drawingAngle = 0,
                    // the negative drawing angle of the group
                    completeDrawingAngle = 0,
                    // the jQuerified drawing
                    $drawing = $(drawing),
                    // the jQuery offset of one drawing
                    drawingOffset = null,
                    // the explicit drawing attributes of the child
                    drawingAttrs = null;

                if (DrawingFrame.isRotatedDrawingNode(drawing)) { drawingAngle = DrawingFrame.getDrawingRotationAngle(self, drawing); }
                if (drawingAngle) { $(drawing).css({ transform: 'rotate(0deg)' }); } // rotating back the child drawing
                if (groupRotationAngle) {
                    completeDrawingAngle = -groupRotationAngle;
                    // assigning the negative group rotation to the child drawing so that there is no remaining rotation
                    if (completeDrawingAngle !== 0) { $(drawing).css({ transform: 'rotate(' + completeDrawingAngle + 'deg)' }); }
                    // saving the original positions in the childAttrsContainer (only for rotated drawings to generate undo operation)
                    drawingAttrs = AttributeUtils.getExplicitAttributes(drawing);
                    // saving the values for 'left', 'top', 'width', 'height' and 'rotation' of the grouped drawing
                    if (drawingAttrs && drawingAttrs.drawing) { saveRequiredDrawingAttributes(childAttrsContainer, drawingAttrs.drawing); }
                }

                // reading value of upper left corner of the unrotated drawing
                drawingOffset = $drawing.offset();

                allNewDrawingAttributes.push({
                    left: Utils.convertLengthToHmm((drawingOffset.left - slideOffset.left) / zoomFactor, 'px'),
                    top: Utils.convertLengthToHmm((drawingOffset.top - slideOffset.top) / zoomFactor, 'px'),
                    width: Utils.convertLengthToHmm($drawing.width(), 'px'),
                    height: Utils.convertLengthToHmm($drawing.height(), 'px'),
                    rotation: groupRotationAngle + drawingAngle
                });
            });

            return allNewDrawingAttributes;
        }

        /**
         * Assigning new drawing attributes to the children of a group, when the group is 'ungrouped'.
         *
         * @param {jQuery} allChildDrawings
         *  The container with all drawing children of the drawing group node.
         *
         * @param {Object[]} drawingAttrs
         *  An object with the drawing attributes for all child drawings. The order must be the same
         *  as in the 'allChildDrawings' container.
         */
        function assignDrawingAttributesToDrawings(allChildDrawings, drawingAttrs) {
            _.each(allChildDrawings, function (drawing, counter) {
                self.getDrawingStyles().setElementAttributes(drawing, { drawing: drawingAttrs[counter] });
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Generating a 'group' operation for the selected drawings.
         */
        this.groupDrawings = function () {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the selection object
                selection = self.getSelection(),
                // a collector for all logical positions of the selected drawings
                allPositions = null,
                // a collector for all drawing indices of the selected drawings
                drawingIndices = null,
                // created operation
                newOperation = null,
                // the drawing attributes
                drawingAttrs = null;

            if (selection.isMultiSelectionSupported() && selection.isMultiSelection()) {

                // collecting the logical positions of all selected drawings
                allPositions = selection.getArrayOfLogicalPositions();

                drawingIndices = _.map(allPositions, function (pos) { return _.last(pos); });

                // calculating the position (left, top, width, height) for the drawing of type 'group' -> this is application specific
                if (self.useSlideMode()) {
                    drawingAttrs = getGroupDrawingPositionAttributes(self.getSelection().getArrayOfSelectedDrawingNodes());
                }

                // creating the operation for grouping the selected drawings
                newOperation = { start: _.clone(allPositions[0]), drawings: drawingIndices };
                if (drawingAttrs) { newOperation.attrs = { drawing: drawingAttrs }; }
                generator.generateOperation(Operations.GROUP, newOperation);

                // applying operation
                this.applyOperations(generator);

                // clearing the existing multi selection
                selection.clearMultiSelection();

                // selecting the grouped drawing
                selection.setTextSelection(allPositions[0], Position.increaseLastIndex(allPositions[0]));
            }
        };

        /**
         * Generating an 'ungroup' operation for the selected drawing(s) of type group.
         */
        this.ungroupDrawings = function () {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // created operation
                newOperation = null,
                // the selection object
                selection = self.getSelection(),
                // the logical position of the selected drawing
                start = selection.isDrawingSelection() ? selection.getStartPosition() : null,
                // a selected drawing group node (drawing selection or additional drawing selection)
                drawingNode = selection.isAnyDrawingSelection() ? selection.getClosestSelectedDrawing() : null,
                // the content node inside the selected drawing(s)
                contentNode = null,
                // the number of drawing children inside a selected group
                childrenCount = 0,
                // the logical positions of the ungrouped drawings
                positions = null,
                // an offset for the logical position caused by previous ungrouping
                offset = 0,
                // the group selections of a multiple selection
                allDrawingGroupSelections = null,
                // collector for all non group drawing selections
                allNonGroupSelections = null,
                // collector for all drawing node / position pairs
                allGroupDrawings = null,
                // a collector for all ungrouped drawings(to create the following selection)
                allUngroupedDrawingNodes = $();

            // getting the selected drawing group
            if (selection.isMultiSelectionSupported() && selection.isMultiSelection()) {
                allDrawingGroupSelections = selection.getAllDrawingGroupsFromMultiSelection();

                if (!allDrawingGroupSelections || allDrawingGroupSelections.length === 0) { return; } // no drawing group selected

                allGroupDrawings = [];

                _.each(allDrawingGroupSelections, function (sel) {
                    drawingNode = selection.getDrawingNodeFromMultiSelection(sel);
                    start = selection.getStartPositionFromMultiSelection(sel);
                    allGroupDrawings.push({ node: drawingNode, start: start });
                });

                // saved all non groups, so that they can be selected again after ungrouping
                allNonGroupSelections = selection.getAllNoneDrawingGroupsFromMultiSelection();
                _.each(allNonGroupSelections, function (sel) { allUngroupedDrawingNodes = allUngroupedDrawingNodes.add(selection.getDrawingNodeFromMultiSelection(sel)); });

            } else if (selection.isDrawingSelection()) {
                drawingNode = selection.getSelectedDrawing();
                start = selection.getStartPosition();
                if (!DrawingFrame.isGroupDrawingFrame(drawingNode)) { return; } // no drawing group selected
                if (self.useSlideMode() && start.length > 2) { start = Position.getOxoPosition(self.getCurrentRootNode(), drawingNode, 0); }
                allGroupDrawings = [{ node: drawingNode, start: start }];
            } else if (selection.isAdditionalTextframeSelection()) {
                drawingNode = selection.getSelectedTextFrameDrawing();
                start = Position.getOxoPosition(self.getCurrentRootNode(), drawingNode, 0);
                if (!DrawingFrame.isGroupDrawingFrame(drawingNode)) { return; } // no drawing group selected
                allGroupDrawings = [{ node: drawingNode, start: start }];
            }

            // Iterating over all selected drawing groups
            _.each(allGroupDrawings, function (groupDrawing) {

                var // the logical position of the group drawing node
                    groupPos = _.clone(groupDrawing.start),
                    // all child drawings in the drawing group
                    allChildDrawings = null;

                if (offset > 0) { groupPos = Position.increaseLastIndex(groupPos, offset); }

                // calculating all new positions (required for undo!)
                positions = [];
                contentNode = DrawingFrame.getContentNode(groupDrawing.node);
                allChildDrawings = contentNode.children(DrawingFrame.NODE_SELECTOR);
                childrenCount = allChildDrawings.length;

                _.each(_.range(childrenCount), function (index) {
                    positions.push(Position.increaseLastIndex(groupPos, index));
                });

                // creating the operation for grouping the selected drawings
                newOperation = { start: _.clone(groupPos), drawings: _.map(positions, function (pos) { return _.last(pos); }) };
                generator.generateOperation(Operations.UNGROUP, newOperation);

                // setting the offset for a following ungroup operation
                offset += (childrenCount - 1);

                allUngroupedDrawingNodes = allUngroupedDrawingNodes.add(allChildDrawings);
            });

            // applying operation
            this.applyOperations(generator);

            // selecting all ungrouped drawings
            selection.setMultiDrawingSelection(allUngroupedDrawingNodes);
        };

        // operation handler --------------------------------------------------

        /**
         * Handler for a 'group' operation.
         */
        this.groupDrawingsHandler = function (operation, external) {

            var // the undo operation for the 'group' operation
                undoOperation = null;

            if (!implGroupDrawings(operation.start, operation.target, operation.drawings, operation.attrs, operation.childAttrs, external)) { return false; }

            if (self.getUndoManager().isUndoEnabled()) {
                undoOperation = { name: Operations.UNGROUP, start: operation.start, target: operation.target, drawings: operation.drawings, attrs: operation.attrs };
                self.getUndoManager().addUndo(undoOperation, operation);
            }

            return true;
        };

        /**
         * Handler for an 'ungroup' operation.
         */
        this.ungroupDrawingsHandler = function (operation, external) {

            var // the undo operation for the 'group' operation
                undoOperation = null,
                // a collector for the drawing positions inside the group
                // -> this needs to be collected for an undo of a rotated drawing group
                drawingChildAttrs = [],
                // a collector for the drawing attributes of the drawing of type group
                drawingGroupAttrs = [];

            if (!implUngroupDrawings(operation.start, operation.target, operation.drawings, drawingGroupAttrs, drawingChildAttrs, external)) { return false; }

            if (self.getUndoManager().isUndoEnabled()) {
                undoOperation = { name: Operations.GROUP, start: operation.start, target: operation.target, drawings: operation.drawings, attrs: (drawingGroupAttrs.length > 0) ? drawingGroupAttrs[0] : null, childAttrs: (drawingChildAttrs.length > 0) ? drawingChildAttrs : null };
                self.getUndoManager().addUndo(undoOperation, operation);
            }

            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class GroupOperationMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return GroupOperationMixin;
});
