/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/components/table/tableresize', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/table/table'
], function (Utils, Tracking, AttributeUtils, Operations, DOM, Position, Table) {

    'use strict';

    var // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel',
        // the names of all tracking events
        ALL_TRACKING_EVENT_NAMES = 'tracking:start tracking:move tracking:scroll tracking:end tracking:cancel';

    // static class TableResize ==================================================

    var TableResize = {};

    // static methods ---------------------------------------------------------

    /**
     * Draws a selection box for the specified resize node and registers
     * mouse handlers for moving.
     *
     * @param {Editor} editor
     *  The editor instance to use.
     *
     * @param {jQuery} officemaindiv
     *  The main window in which the resize node will be displayed temporarely
     *
     * @param {HTMLElement|jQuery} resizeNode
     *  The selected table resize node. If this value is a jQuery
     *  collection, uses the first DOM node it contains.
     */
    TableResize.drawTableCellResizeSelection = function (editor, view, officemaindiv, resizeNode) {

        var // the node containing the scroll bar
            scrollNode = view.getContentRootNode();

        /**
         * Handling all tracking events while resizing a table.
         */
        var tableResizeHandler = (function () {

            var startX = 0,
                startY = 0,
                currentX = 0,
                currentY = 0,
                shiftX = 0,
                shiftY = 0,
                // verticalResize is true, if the column width will be modified
                verticalResize = false,
                // horizontalResize is true, if the row height will be modified
                horizontalResize = false,
                // verticalResize is true, if the column width will be modified
                isLeftResize = false,
                // horizontalResize is true, if the row height will be modified
                isTopResize = false,
                // the container element used to visualize the resizing
                resizeLine = $('<div>').addClass('resizeline'),
                // the distance from body element to 'officemaindiv' in pixel
                topDistance = null,
                // the distance from body element to 'officemaindiv' in pixel
                leftDistance = null,
                // the cell node for the selected resize node
                cellNode = null,
                // the row node for the selected resize node
                rowNode =  null,
                // the table node for the selected resize node
                tableNode = null,
                // the table node attributes object for the selected resize node
                tableNodeAttrs = null,
                // the maximum table width
                maxTableWidth = 0,
                // is the selected cell the last cell in its row
                lastCell = false,
                // logical position of the selected node
                tablePosition = [],
                // table grid before shifting column
                oldTableGrid = [],
                // table width after shifting column
                newTableWidth = 0,
                // table width before shifting column
                oldTableWidth = 'auto',
                // table grid, containing relative widths
                tableGrid = [],
                // table grid, containing calculated pixel widhts
                pixelGrid = [],
                // sum of all grid values, will not be modified
                gridSum = 0,
                // the number of the grid count, that will be modified
                shiftedGrid = 0,
                // maximum shift to the left (-1 means 'no limit')
                maxLeftShift = 0,
                // maximum shift to the right (-1 means 'no limit')
                maxRightShift = 0,
                // maximum shift to the top (-1 means 'no limit')
                maxTopShift = 0,
                // maximum shift to the bottom (-1 means 'no limit')
                maxBottomShift = 0,
                // maximum right value of shift position (-1 means 'no limit')
                maxRightValue = 0,
                // minimum left value of shift position (-1 means 'no limit')
                minLeftValue = 0,
                // minimum width of a column in px
                // Fix for 29678, using min width like in Word
                minColumnWidth = 16,
                // minimum height of a row in px
                minRowHeight = 5,
                //zoom factor in floating point notation
                zoomFactor = view.getZoomFactor() / 100,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the start scroll position
                startScrollX = 0, startScrollY = 0,
                // whether the most right cell can be expanded to the right
                allowLastCellOverflow = false,
                // whether the table is located inside a drawing node
                isTableDrawing = false,
               // maximum height of page content in pixels (without margins)
                pageMaxHeight = editor.getPageLayout().getDefPageActiveHeight({ convertToPixel: true }) - 5; // -5 px because of difference in rounding and offsetHeight, position top in IE, FF

            /**
             * Calculates the required data, when mouse down happens on resize node.
             *
             * @param {Event} event
             *  The event object.
             */
            function startResizeTable(event) {
                // mouse down event handler
                startX = event.pageX;
                startY = event.pageY;

                if (!startX && event.originalEvent.pageX && event.originalEvent.pageY) {
                    startX = event.originalEvent.pageX;
                    startY = event.originalEvent.pageY;
                }

                topDistance = officemaindiv.offset().top;
                leftDistance = officemaindiv.offset().left;

                startX -= leftDistance;
                startY -= topDistance;

                if (resizeNode.is('div.resize.right')) {
                    verticalResize = true;
                } else if (resizeNode.is('div.resize.bottom')) {
                    horizontalResize = true;
                } else if (resizeNode.is('div.resize.left')) {
                    verticalResize = true;
                    isLeftResize = true;
                } else if (resizeNode.is('div.resize.top')) {
                    horizontalResize = true;
                    isTopResize = true;
                }

                // calculating maximum resize values
                cellNode = resizeNode.closest('td, th');
                rowNode =  resizeNode.closest('tr');
                tableNode = resizeNode.closest('table');

                // in drawings of type 'table' the most right cells can be expanded to the right
                isTableDrawing = DOM.isTableNodeInTableDrawing(tableNode);
                allowLastCellOverflow = isTableDrawing;

                if (verticalResize) {
                    $(resizeLine).css({ width: '1px', height: '100%', left: startX, top: '0px' });
                    officemaindiv.append(resizeLine);

                    // calculating maxLeftShift and maxRightShift
                    lastCell = !cellNode[0].nextSibling;
                    tablePosition = Position.getOxoPosition(editor.getCurrentRootNode(), tableNode.get(0), 0);
                    tableNodeAttrs = editor.getStyleCollection('table').getElementAttributes(tableNode).table;
                    oldTableGrid = tableNodeAttrs.tableGrid;
                    oldTableWidth = tableNodeAttrs.width;
                    maxTableWidth = tableNode.parent().width();

                    if (oldTableWidth === 'auto') {
                        oldTableWidth = tableNode.outerWidth();
                    } else {
                        oldTableWidth = Utils.convertHmmToLength(oldTableWidth, 'px', 1);
                    }

                    // converting from relational grid to pixel grid
                    var i = 0;
                    for (i = 0; i < oldTableGrid.length; i++) { gridSum += oldTableGrid[i]; }
                    for (i = 0; i < oldTableGrid.length; i++) { pixelGrid.push(Math.round(oldTableGrid[i] * oldTableWidth / gridSum)); }

                    // which border was shifted?
                    shiftedGrid = Table.getGridPositionFromCellPosition(rowNode, cellNode.prevAll().length).end;

                    // Fix for 30798, cells with lists need an increased minimum width
                    if (cellNode.find(DOM.LIST_LABEL_NODE_SELECTOR).length > 0) { minColumnWidth *= 4; }

                    if (isLeftResize) {
                        maxLeftShift = -1; // no limit
                        maxRightShift = pixelGrid[shiftedGrid] * zoomFactor - resizeNode.position().left; // taking into account, that the resizer is shifted to the right
                        allowLastCellOverflow = false;
                    } else {
                        maxLeftShift = pixelGrid[shiftedGrid] * zoomFactor;
                        if (!lastCell) {
                            maxRightShift = pixelGrid[shiftedGrid + 1] * zoomFactor;
                            allowLastCellOverflow = false;
                        } else {
                            maxRightShift = (maxTableWidth - oldTableWidth) * zoomFactor;
                        }
                    }

                } else if (horizontalResize) {
                    $(resizeLine).css({ width: '100%', height: '1px', left: 0, top: startY });
                    officemaindiv.append(resizeLine);

                    if (isTopResize) {
                        // calculating maxTopShift
                        maxTopShift = -1; // no limit

                        // calculating maxBottomShift - max height that single row can have
                        maxBottomShift = cellNode.outerHeight() * zoomFactor - resizeNode.position().top - minRowHeight;
                    } else {
                        // calculating maxTopShift
                        maxTopShift = cellNode.outerHeight() * zoomFactor;
                        // calculating maxBottomShift - max height that single row can have
                        maxBottomShift = (pageMaxHeight - cellNode.outerHeight()) * zoomFactor;
                    }
                }

                editor.getCurrentRootNode().css('cursor', resizeNode.css('cursor'));  // setting cursor for increasing drawing
                $(resizeLine).css('cursor', resizeNode.css('cursor'));  // setting cursor for increasing drawing

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();
            }

            /**
             * Calculates the required data, when mouse move happens.
             *
             * @param {Event} event
             *  The event object.
             */
            function moveResizeTable(event) {

                // mouse move event handler
                currentX = event.pageX;
                currentY = event.pageY;

                if (!currentX && event.originalEvent.pageX) {
                    currentX = event.originalEvent.pageX;
                    currentY = event.originalEvent.pageY;
                }

                topDistance = officemaindiv.offset().top;
                leftDistance = officemaindiv.offset().left;

                currentX = currentX + scrollX - leftDistance;
                currentY = currentY + scrollY - topDistance;

                if (verticalResize) {

                    maxRightValue = (maxRightShift > -1) ? startX + maxRightShift : -1;
                    minLeftValue = (maxLeftShift > -1) ? (startX - maxLeftShift + (minColumnWidth * zoomFactor)) : -1;

                    if (maxRightValue > -1 && !lastCell) { maxRightValue -= (minColumnWidth * zoomFactor); }

                    shiftX = currentX;
                    shiftY = 0;

                    if (maxRightValue > -1 && shiftX >= maxRightValue && !allowLastCellOverflow) {
                        shiftX = maxRightValue;
                    } else if (minLeftValue > -1 && shiftX <= minLeftValue) {
                        shiftX = minLeftValue;
                    }

                } else if (horizontalResize) {
                    shiftX = 0;
                    shiftY = currentY;

                    if (maxTopShift > -1 && (shiftY - startY) <= -maxTopShift) {
                        shiftY = startY - maxTopShift;
                    }
                    if (maxBottomShift > -1 && (shiftY - startY) >= maxBottomShift) {
                        shiftY = startY + maxBottomShift;
                    }
                }

                if ((_.isNumber(shiftX)) && (_.isNumber(shiftY))) {
                    $(resizeLine).css({ left: shiftX - scrollX, top: shiftY - scrollY });
                }
            }

            /**
             * Updates scroll position according to the passed tracking event.
             *
             * @param {Event} event
             *  The event object.
             */
            function scrollResizeTable(event) {

                // update scrollPosition with suggestion from event
                if (event.scrollX) {
                    scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX);
                }

                if (event.scrollY) {
                    scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY);
                }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            /**
             * Calculates the required data, when mouse up happens and generates
             * operations.
             *
             * @param {Event} event
             *  The event object.
             */
            function stopResizeTable(event) {

                var generator = editor.createOperationsGenerator(),
                    rowPosition = null,
                    rowHeight = 0,
                    newRowHeight = 0, oldRowHeight = 0,
                    // the old attributes
                    oldAttrs = null,
                    // the attributes for the setAttributes operation(s)
                    attrs = null, drawingOperationProperties = null,
                    target = editor.getActiveTarget(),
                    operationProperties = {},
                    // the logical table position
                    tablePos = null,
                    // the shift of the table
                    tableShift = null,
                    // the drawing node of a table drawing
                    drawingNode = null,
                    // the attributes of a table drawing node and row
                    drawingAttrs = null, rowAttrs = null,
                    // a shift of the position of the table drawing node
                    deltaTablePos = 0,
                    // saving the old value for the top and left position in the drawing attributes
                    oldDrawingAttrsTop = 0, oldDrawingAttrsLeft = 0,
                    // the height of the table row in the DOM
                    realRowNodeHeight = 0;

                // helper function for the left resizer
                function handleLeftResize() {

                    // setting the table attributes
                    drawingNode = tableNode.closest('div.drawing');
                    // table drawings always have explicit 'drawing' attributes
                    drawingAttrs = AttributeUtils.getExplicitAttributes(drawingNode, { family: 'drawing' });

                    oldDrawingAttrsLeft = drawingAttrs.left;

                    if (editor.useSlideMode()) {
                        // setting the left value of the drawing by comparing the vertical line with the slide position
                        drawingAttrs.left = Utils.convertLengthToHmm((resizeLine.offset().left - drawingNode.closest('div.slide').offset().left) / zoomFactor);

                        // improving precision -> table width can only be modified like change of left drawing position

                    } else {
                        // TODO
                        // drawingAttrs.left += Utils.convertLengthToHmm(shiftX, 'px'); // setting new left position of the table drawing (OX Text style)
                    }

                    attrs = attrs || {};
                    attrs.drawing = drawingAttrs;
                    attrs.drawing.width = newTableWidth; // modifying the table width

                    // improving precision of shiftX by simply using the shift of the table in horizontal direction
                    deltaTablePos = drawingAttrs.left - oldDrawingAttrsLeft;

                    shiftX = Utils.convertHmmToLength(deltaTablePos, 'px', 1);
                }

                // helper function for the top resizer
                function handleTopResize() {

                    // the upper left position of the table (drawing) was changed
                    tablePos = _.initial(rowPosition);
                    tableShift = Utils.convertLengthToHmm(rowHeight - oldRowHeight, 'px');

                    if (tableShift !== 0 && isTableDrawing) {

                        drawingNode = tableNode.closest('div.drawing');
                        // table drawings always have explicit 'drawing' attributes
                        drawingAttrs = AttributeUtils.getExplicitAttributes(drawingNode, { family: 'drawing' });

                        oldDrawingAttrsTop = drawingAttrs.top;

                        if (editor.useSlideMode()) {
                            // setting the top value of the drawing by comparing the horizontal line with the slide position
                            drawingAttrs.top = Utils.convertLengthToHmm((resizeLine.offset().top - drawingNode.closest('div.slide').offset().top) / zoomFactor);
                        } else {
                            // TODO
                            drawingAttrs.top -= tableShift;
                        }

                        drawingOperationProperties = {
                            attrs: { drawing: drawingAttrs },
                            start: tablePos
                        };

                        if (target) { drawingOperationProperties.target = target; }

                        generator.generateOperation(Operations.SET_ATTRIBUTES, drawingOperationProperties);

                        // shift of table in vertical direction
                        deltaTablePos = oldDrawingAttrsTop - drawingAttrs.top;

                        // the height of the first row can be calculated more precise by using the difference of top position of table
                        rowAttrs = AttributeUtils.getExplicitAttributes(rowNode, { family: 'row' });
                        if (rowAttrs && _.isNumber(rowAttrs.height)) {
                            realRowNodeHeight = Utils.convertLengthToHmm(rowNode.outerHeight(true), 'px');
                            newRowHeight = Math.max(realRowNodeHeight, rowAttrs.height) + deltaTablePos;
                        }
                    }
                }

                // taking care of scroll node
                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;

                // mouse up event handler
                currentX = event.pageX;
                currentY = event.pageY;

                if (!currentX && event.originalEvent.changedTouches[0].pageX) {
                    currentX = event.originalEvent.changedTouches[0].pageX;
                    currentY = event.originalEvent.changedTouches[0].pageY;
                }

                topDistance = officemaindiv.offset().top;
                leftDistance = officemaindiv.offset().left;

                currentX = currentX + scrollX - leftDistance;
                currentY = currentY + scrollY - topDistance;

                if (verticalResize) {

                    if ((_.isNumber(currentX)) && (currentX !== startX)) {

                        maxRightValue = (maxRightShift > -1) ? (startX + maxRightShift) : -1;
                        minLeftValue = (maxLeftShift > -1) ? (startX - maxLeftShift + (minColumnWidth * zoomFactor)) : -1;
                        if (!lastCell && maxRightValue > -1) { maxRightValue -= (minColumnWidth * zoomFactor); }

                        if (maxRightValue > -1 && currentX >= maxRightValue && !allowLastCellOverflow) {
                            currentX = maxRightValue;
                        } else if (minLeftValue > -1 && currentX <= minLeftValue) {
                            currentX = minLeftValue;
                        }

                        shiftX = (currentX - startX) / zoomFactor;

                        // the table (drawing) needs to get a new upper left position
                        if (isLeftResize && shiftX !== 0 && isTableDrawing) { handleLeftResize(); }

                        newTableWidth = isLeftResize ? (oldTableWidth - shiftX) : (lastCell ? (oldTableWidth + shiftX) : oldTableWidth);

                        // -> shifting the border
                        pixelGrid[shiftedGrid] = isLeftResize ? (pixelGrid[shiftedGrid] - shiftX) : (pixelGrid[shiftedGrid] + shiftX);

                        if (!lastCell && !isLeftResize) { pixelGrid[shiftedGrid + 1] -= shiftX; }

                        // converting modified pixel grid to new relation table grid
                        for (var i = 0; i < pixelGrid.length; i++) {
                            tableGrid.push(Math.round(gridSum * pixelGrid[i] / newTableWidth));  // only ints
                        }

                        if (!lastCell && !isLeftResize && (editor.getStyleCollection('table').getElementAttributes(tableNode).table.width === 'auto')) {
                            newTableWidth = 'auto';
                        } else {
                            newTableWidth = Utils.convertLengthToHmm(newTableWidth, 'px');
                        }

                        // the table attributes for the setAttributes operation
                        attrs = attrs || {};
                        if (isTableDrawing) {
                            attrs.table = { tableGrid: tableGrid };
                            if (newTableWidth !== 'auto') {
                                attrs.drawing = attrs.drawing || {};
                                attrs.drawing.width = newTableWidth;
                            } // width must be assigned to table drawing
                        } else {
                            attrs.table = { tableGrid: tableGrid, width: newTableWidth };
                        }

                        if (editor.getChangeTrack().isActiveChangeTracking()) {
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = editor.getChangeTrack().getOldNodeAttributes(tableNode);
                            // adding the old attributes, author and date for change tracking
                            if (oldAttrs) {
                                oldAttrs = _.extend(oldAttrs, editor.getChangeTrack().getChangeTrackInfo());
                                attrs.changes = { modified: oldAttrs };
                            }
                        }
                        operationProperties = {
                            attrs: attrs,
                            start: tablePosition
                        };
                        if (target) {
                            operationProperties.target = target;
                        }

                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

                        // Performance: Mark the table with information about vertical resize, so that no full update of cell
                        // attributes is required, but only a recalculation of tab stops.
                        tableNode.data('tableResize', true);

                        editor.applyOperations(generator);

                        tableNode.removeData('tableResize');
                    }

                } else if (horizontalResize) {

                    if ((_.isNumber(currentY)) && (currentY !== startY)) {

                        oldRowHeight = rowNode.outerHeight();

                        rowHeight = isTopResize ? (oldRowHeight - ((currentY - startY) / zoomFactor)) : (oldRowHeight + ((currentY - startY) / zoomFactor));
                        if (rowHeight < 0) { rowHeight = 0; }
                        if (rowHeight > pageMaxHeight) { rowHeight = pageMaxHeight; }
                        newRowHeight = Utils.convertLengthToHmm(rowHeight, 'px');
                        rowPosition = Position.getOxoPosition(editor.getCurrentRootNode(), rowNode.get(0), 0);

                        if (isTopResize && isTableDrawing) { handleTopResize(); }  // handling a top resize of the table

                        // the row attributes for the setAttributes operation
                        attrs = { row: { height: newRowHeight } };

                        if (editor.getChangeTrack().isActiveChangeTracking()) {
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = editor.getChangeTrack().getOldNodeAttributes(rowNode);
                            // adding the old attributes, author and date for change tracking
                            if (oldAttrs) {
                                oldAttrs = _.extend(oldAttrs, editor.getChangeTrack().getChangeTrackInfo());
                                attrs.changes = { modified: oldAttrs };
                            }
                        }

                        operationProperties = {
                            attrs: attrs,
                            start: rowPosition
                        };
                        if (target) {
                            operationProperties.target = target;
                        }

                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

                        editor.applyOperations(generator);
                    }
                }

            }

            /**
             * Finalizes the tracking of the table resizing process.
             */
            function finalizeResizeTable() {
                // leaveTracking();
                resizeNode.off(ALL_TRACKING_EVENT_NAMES);
                view.scrollToChildNode(resizeNode);

                // removing the resize line
                officemaindiv.children('div.resizeline').remove();

                // Resetting cursor, using css file again
                editor.getCurrentRootNode().css('cursor', '');

                // Resetting variables
                shiftX = shiftY = scrollX = scrollY = 0;

                // removing class
                resizeNode.removeClass('resizeActive');
            }

            // return the actual drawingMoveHandler() method
            return function (event) {

                switch (event.type) {
                    case 'tracking:start':
                        startResizeTable(event);
                        break;
                    case 'tracking:move':
                        moveResizeTable(event);
                        break;
                    case 'tracking:scroll':
                        scrollResizeTable(event);
                        break;
                    case 'tracking:end':
                        stopResizeTable(event);
                        finalizeResizeTable(event);
                        break;
                    case 'tracking:cancel':
                        finalizeResizeTable(event);
                        break;
                }
            };

        }()); // end of tableResizeHandler() local scope

        /**
         * Handler for 'tracking:start' events for table resize nodes. For this
         * node all other tracking events have to be registered.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event, that starts the resizing of the table.
         */
        function trackingStartHandler(event) {
            resizeNode.off(TRACKING_EVENT_NAMES);
            resizeNode.on(TRACKING_EVENT_NAMES, tableResizeHandler);

            tableResizeHandler.call(this, event);
        }

        // ensure to work on a jQuery collection
        resizeNode = $(resizeNode);

        // mark this resize node, that it is already prepared for tracking
        resizeNode.addClass('resizeActive');

        Tracking.enableTracking(resizeNode, {
            autoScroll: true,
            borderNode: scrollNode,
            borderMargin: -30,
            borderSize: 60,
            minSpeed: 10,
            maxSpeed: 250
        });
        resizeNode.on('tracking:start', trackingStartHandler);
    };

    // exports ================================================================

    return TableResize;

});
