/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/components/changetrack/changetrack', [
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/table/table',
    'gettext!io.ox/office/textframework/main'
], function (Utils, DateUtils, TriggerObject, TimerMixin, DrawingFrame, AttributeUtils, Operations, DOM, Position, Table, gt) {

    'use strict';

    // private static functions ===============================================

    var // the allowed changeTrack types (order is relevant)
        CHANGE_TRACK_TYPES = ['inserted', 'removed', 'modified'],
        // the allowed changeTrack information saved at nodes
        CHANGE_TRACK_INFOS = ['author', 'date', 'uid'],
        // clip board operations, that will be expanded with change tracking information (explicitly not setAttributes)
        CHANGETRACK_CLIPBOARD_OPERATIONS = [
            Operations.TEXT_INSERT,
            Operations.DRAWING_INSERT,
            Operations.FIELD_INSERT,
            Operations.TAB_INSERT,
            Operations.HARDBREAK_INSERT,
            Operations.PARA_INSERT,
            Operations.ROWS_INSERT,
            Operations.TABLE_INSERT
        ],
        // whether the resizing of a drawing needs to be change tracked
        CHANGE_TRACK_SUPPORTS_DRAWING_RESIZING = false,
        // whether the moving of a drawing needs to be change tracked
        CHANGE_TRACK_SUPPORTS_DRAWING_MOVE = false,
        // whether the assignment of a position to a drawing needs to be change tracked
        CHANGE_TRACK_SUPPORTS_DRAWING_POSITION = false;

    // class ChangeTrack ========================================================

    /**
     * An instance of this class represents a change track handler in the edited document.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     *
     * @param {TextModel} model
     *  The text model instance.
     */
    function ChangeTrack(app, rootNode, model) {

        var // self reference
            self = this,
            // the current changeTrack date
            changeTrackInfoDate = null,
            // the local user id of the current user
            userid = null,
            // the promise for the repeated date update process
            dateUpdatePromise = null,
            // the period of time, that an update of change track time happens repeatedly (in ms).
            updateDateDelay = 30000,
            // the change track selection object
            // -> activeNode: HTMLElement, the one currently node used by the iterator
            // -> activeNodeType: String, the one currently used node type, one of 'inserted', 'removed', 'modified'
            // -> activeNodeTypesOrdered: String[], ordered list of strings, corresponding to priority of node types
            // -> selectedNodes: HTMLElement[], sorted list of HTMLElements of the group belonging to 'activeNode' and 'activeNodeType
            // -> allChangeTrackNodes: jQuery, optional, cache for performance reasons, contains all change track elements in the document
            changeTrackSelection = null,
            // a collector for additional clipboard operations
            clipboardCollector = [],
            // css class for showing change track nodes
            highlightClassname = 'change-track-highlight',
            // whether change track is supported in the current context (for example not inside comments)
            isChangeTrackSupported = true,
            // whether the handler for the change track side bar are active or not
            sideBarHandlerActive = false,
            // the node for the change track side bar
            sideBarNode = null,
            // the node containing the scroll bar
            scrollNode = null,
            // the node with the page content
            pageContentNode = null,
            // collector for all change tracked nodes (required only for the side bar)
            allNodesSideBarCollector = null,
            // a collector for all markers in the change track side bar
            allSideBarMarker = {};

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns all tracking nodes inside the selection. If the selection
         * has no range, return only the currently selected node.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.minimalSelection=false]
         *      If set to true and the selection is a range, the iteration
         *      stops after finding the first change track node. If the
         *      selection is not a range, this parameter will be ignored.
         *  @param {Boolean} [options.markLastParagraph=false]
         *      If set to true and the selection is a range, the final paragraph
         *      of the selection will be marked, if it is not completely part of
         *      the selection. In this case the 'inserted' and 'removed' change
         *      tracks must be resolved, but not the 'modified' change track.
         *  @param {Boolean} [options.split=false]
         *      If set to true the spans that are only partially covered by
         *      the selection, are split during the iteration. As a result
         *      all nodes in the collector are covered completely.
         *
         * @returns {jQuery}
         *  A list of nodes in a jQuery element. The list is sorted corresponding
         *  to the appearance of the nodes in the DOM.
         */
        function getTrackingNodesInSelection(options) {

            var // the tracking nodes in the current selection
                trackingNodes = $(),
                // whether it is sufficient to find only one change tracking node
                minimalSelection = Utils.getBooleanOption(options, 'minimalSelection', false),
                // whether the final paragraph of a selection shall be marked in that way, that
                // the 'modified' change track attribute shall not be resolved. This is the case
                // if this final paragraph is not completely included into the selection.
                markLastParagraph = Utils.getBooleanOption(options, 'markLastParagraph', false),
                // whether spans that are only partially covered by the selection, need to be split
                split = Utils.getBooleanOption(options, 'split', false),
                // a container with all paragraphs. This is necessary to check, if the last paragraph
                // shall be added to the selection
                paragraphsCollector = [];

            // all tracking nodes that are parents, grandparents, ... need to be added
            // to the list of all tracking nodes. Otherwise only text nodes are collected.
            function collectParagraphs(paraNode) {
                if (_.contains(paragraphsCollector, paraNode)) { return; }  // paragraph already checked
                paragraphsCollector.push(paraNode); // collecting each paragraph only once
            }

            // checking, if the final paragraph needs to be added to the change track collector.
            // This is necessary, if it was 'inserted' or 'removed', and is not completely included
            // into the selection.
            function checkFinalParagraph() {

                if (paragraphsCollector.length === 0) { return; }

                var // the final paragraph
                    lastParagraph = paragraphsCollector[paragraphsCollector.length - 1],
                    // an array with the tracking nodes (used to find and insert the paragraph node
                    trackingNodeArray = trackingNodes.get(),
                    // the change tracked children of the last paragraph
                    paraChildren = null,
                    // the first change tracked child of the last paragraph
                    firstChild = null,
                    // the position of the first child in the list of change tracked nodes
                    firstChildPosition = null;

                // is the last paragraph a change tracked node ('inserted' or 'removed' that is not
                // already part of the collected tracking nodes?
                // -> searching from end, because it probably needs to be added to the end
                if (lastParagraph && (DOM.isChangeTrackInsertNode(lastParagraph) || DOM.isChangeTrackRemoveNode(lastParagraph)) && (trackingNodeArray.lastIndexOf(lastParagraph) === -1)) {
                    // the paragraph must be inserted into the list of tracking nodes
                    // -> at a position before its children or at the end, if none of its children
                    // are part of the list
                    paraChildren = getAllChangeTrackNodes(lastParagraph);

                    if (paraChildren.length > 0) {
                        firstChild = paraChildren[0];
                        firstChildPosition = trackingNodeArray.lastIndexOf(firstChild);
                        if (firstChildPosition === -1) {
                            trackingNodes = trackingNodes.add(lastParagraph);  // Simply adding to the end
                        } else {
                            trackingNodeArray.splice(firstChildPosition, 0, lastParagraph);  // inserting last paragraph before its first child
                            trackingNodes = $(trackingNodeArray); // converting back to jQuery list
                        }

                    } else {
                        trackingNodes = trackingNodes.add(lastParagraph);  // Simply adding to the end
                    }

                    // setting the 'skipModify' data attribute, so that 'modify' will not be resolved/rejected
                    if (markLastParagraph) { $(lastParagraph).data('skipModify', true); }
                }
            }

            if (model.getSelection().hasRange()) {

                // iterating over all nodes in the selection (using option split
                // to split partially covered spans). Using 'shortestPath: true' it is possible to check,
                // if complete paragraphs or tables are part of the selection. In this case the change
                // tracked children can be found with the 'find' function.
                model.getSelection().iterateNodes(function (node) {

                    var // a collector for all change tracked children of a content node
                        changeTrackChildren = null,
                        // the parent node of a text span
                        parentNode = null;

                    // in text component nodes the span has the change track attribute, not the
                    // text component itself. But collector must contain the text component.
                    function getValidChangeTrackNode(node) {
                        return DOM.isTextComponentNode(node.parentNode) ? node.parentNode : node;
                    }

                    // collecting all paragraphs in a collector
                    if (DOM.isTextSpan(node)) {
                        parentNode = Utils.getDomNode(node).parentNode;
                        collectParagraphs(parentNode);
                    }

                    if (DOM.isChangeTrackNode(node)) {
                        trackingNodes = trackingNodes.add(node);
                    }

                    if (minimalSelection && trackingNodes.length > 0) { return Utils.BREAK; }  // exit as early as possible

                    // collecting all children inside the content node
                    if (DOM.isContentNode(node)) {
                        changeTrackChildren = getAllChangeTrackNodes(node);

                        if (changeTrackChildren.length > 0) {
                            _.each(changeTrackChildren, function (node) {
                                trackingNodes = trackingNodes.add(getValidChangeTrackNode(node));
                            });
                        }
                    }

                    if (minimalSelection && trackingNodes.length > 0) { return Utils.BREAK; }  // exit as early as possible

                }, null, { split: split, shortestPath: true });

                // special handling for the final paragraph, that needs to be inserted into the tracking
                // node collection, if it was 'inserted' or 'removed' AND the previous paragraph is also
                // added to the paragraph collector.
                checkFinalParagraph();

                // TODO: grouping of cells to columns? In which scenario is this necessary? Tables are always
                // completely part of the selection or not. Otherwise only paragraphs are completely evaluated.

            } else {
                trackingNodes = findChangeTrackNodeAtCursorPosition();
            }

            return trackingNodes;
        }

        /**
         * Returns the currently selected tracking node at the cursor position or at the
         * start position of a selection range. If there is no change tracking element, an
         * empty jQuery list is returned.
         *
         * @returns {jQuery}
         *  A list of selected nodes in a jQuery element.
         */
        function findChangeTrackNodeAtCursorPosition() {

            var
                // root node holder, either default app root node, or if exists, header/footer root node
                thisRootNode = model.getCurrentRootNode(),
                // the node info for the start position of the current selection
                nodeInfo = Position.getDOMPosition(thisRootNode, model.getSelection().getStartPosition()),
                // the node at the cursor position
                currentNode = nodeInfo ? nodeInfo.node : null,
                // the offset at the cursor position
                offset = nodeInfo ? nodeInfo.offset : null,
                // a helper node for empty text spans
                emptySpan = null,
                // the currently selected tracking node
                trackingNode = null;

            // fast exit, if the node could not be determined
            if (!currentNode) { return; }

            // special handling for drawings
            if (model.isDrawingSelected()) { return model.getSelection().getSelectedDrawing(); }

            // special handling for empty text nodes between inline nodes, for example div elements
            if ((currentNode.nodeType === 3) && currentNode.parentNode && DOM.isEmptySpan(currentNode.parentNode)) {
                emptySpan = currentNode.parentNode;
                // checking for previous or following
                if (emptySpan.previousSibling && DOM.isChangeTrackNode(emptySpan.previousSibling)) {
                    trackingNode = $(emptySpan.previousSibling);
                } else if (emptySpan.nextSibling && DOM.isChangeTrackNode(emptySpan.nextSibling)) {
                    trackingNode = $(emptySpan.nextSibling);
                }
            }

            trackingNode = trackingNode || $(currentNode).closest(DOM.CHANGETRACK_NODE_SELECTOR);  // standard process to find change track node

            if ((trackingNode.length === 0) && (offset === 0) && currentNode.parentNode.previousSibling && DOM.isChangeTrackNode(currentNode.parentNode.previousSibling)) {
                // this happens behind a div.inline component
                trackingNode = $(currentNode.parentNode.previousSibling);
            }

            if ((trackingNode.length === 0) && (DOM.isEmptySpan(currentNode.parentNode) || ((currentNode.nodeType === 3) && (currentNode.length === offset))) && DOM.isChangeTrackNode(currentNode.parentNode.nextSibling)) {
                // this happens before a change tracked component, div.inline or text span (special handling for empty spans between two tabs)
                trackingNode = $(currentNode.parentNode.nextSibling);
            }

            return trackingNode;
        }

        /**
         * Returns for a given list of nodes an expanded list, that contains
         * all the given nodes and additionally all nodes affected by the
         * change track process.
         * Using a jQuery element as collection, keeps all nodes unique.
         *
         * @param {HTMLElement[]|jQuery} nodes
         *  A list of all nodes, for that the adjacent nodes shall be detected
         *
         * @param {String} [type]
         *  The type of resolving ('inserted', 'removed' or 'modified'). I not specified,
         *  all types will be evaluated.
         *
         * @returns {jQuery}
         *  A list of nodes in a jQuery element.
         */
        function groupTrackingNodes(nodes, type) {

            var // the jQuery collector for all grouped nodes
                allNodes = $(),
                // whether all types shall be resolved (if 'type' is not defined)
                allResolveTypes = !type,
                // a list with all resolve types
                resolveTypeList = allResolveTypes ? CHANGE_TRACK_TYPES : [type];

            _.each(nodes, function (node) {
                _.each(resolveTypeList, function (oneType) {
                    var combinedNodes = combineAdjacentNodes(node, { type: oneType });
                    allNodes = allNodes.add(combinedNodes);
                });
            });

            return allNodes;
        }

        /**
         * Starting the process for updating the change track info date
         * regularly. This is needed for performance reasons, so that
         * 'self.updateChangeTrackInfoDate()' is not called for every
         * operation.
         */
        function startDateUpdate() {
            if (!dateUpdatePromise) {
                // immediately start updating
                dateUpdatePromise = self.repeatDelayed(self.updateChangeTrackInfoDate, { delay: 'immediate', repeatDelay: updateDateDelay, infoString: 'Text: ChangeTrack: startDateUpdate', app: app });
            }
        }

        /**
         * Stopping the process for updating the change track info date
         * regularly. This is needed for performance reasons, so that
         * 'self.updateChangeTrackInfoDate()' is not called for every
         * operation.
         */
        function stopDateUpdate() {
            if (dateUpdatePromise) {
                dateUpdatePromise.abort();
                dateUpdatePromise = null;
            }
        }

        /**
         * Collecting all change tracking nodes in the document
         *
         * @param {HTMLElement|jQuery} [node]
         *  The node, in which the change track nodes are searched. If
         *  not specified, the rootNode is used.
         *  If this object is a jQuery collection, uses the first node
         *  it contains.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.multiple=false]
         *      If set to true all change tracked nodes in all headers and
         *      footers will be collected. The default behavior is, that
         *      only the nodes from one repeated header or footer are
         *      collected.
         *      For resolving change tracks, only the first occurrence of
         *      a node is required. For visualization on left side bar, all
         *      nodes are required.
         *
         * @returns {jQuery}
         *  A list of change track nodes in the document.
         */
        function getAllChangeTrackNodes(node, options) {

            var // the node, in which the change track nodes are searched
                searchNode = node || rootNode,
                // the ordered list of all change track nodes (using change tracked place holder nodes instead of
                // absolute positioned drawings because of correct node order, 36318)
                allChangeTrackNodes = $(searchNode).find(DOM.CHANGETRACK_NODE_SELECTOR).not('.drawing.absolute, .list-label > span'),
                // the change track nodes inside the header or footer
                allChangeTrackNodesMarginal = [],
                // collecting all nodes with a specific id
                allHeaderFooterNodes = {},
                // whether all nodes in cloned targets shall be collected (used for identical headers or footers)
                multiple = Utils.getBooleanOption(options, 'multiple', false);

            // whether there is at least one marginal node inside the tracking node collector
            function containsMarginalNode(allNodes) {
                return _.find(allNodes, DOM.isMarginalNode);
            }

            // whether a marginal node was already collected in another header/footer with the same id
            // -> every header/footer node with specific ID must be handled only once
            function alreadyCollectedInAnotherContainer(rootNode, node) {

                var // the container, that contains the specified node
                    containerNode = DOM.getMarginalTargetNode(rootNode, node),
                    // the container id
                    containerId = (containerNode.length > 0) ? DOM.getTargetContainerId(Utils.getDomNode(containerNode)) : null;

                if (_.isString(containerId)) {

                    containerNode = Utils.getDomNode(containerNode);

                    if (allHeaderFooterNodes[containerId]) {
                        // same container id, but different container nodes -> this node must not be collected
                        if (containerNode !== allHeaderFooterNodes[containerId]) {
                            return true;
                        }

                    } else {
                        allHeaderFooterNodes[containerId] = containerNode;
                    }
                }

                return false;
            }

            // 1. keeping correct order for all those nodes with class 'marginal'
            // 2. filtering all those nodes, that are in more than one header or footer
            if (containsMarginalNode(allChangeTrackNodes)) {
                allChangeTrackNodes = _.reject(allChangeTrackNodes, function (node) {
                    if (DOM.isMarginalNode(node)) {
                        // collecting all change track nodes in header or footer, but not in template and only once for each container id
                        // -> collection all nodes in all headers and footers can be enabled with the property 'multiple'
                        if (!DOM.isInsideHeaderFooterTemplateNode(rootNode, node) && (multiple || !alreadyCollectedInAnotherContainer(rootNode, node))) { allChangeTrackNodesMarginal.push(node); }
                        return true;
                    }
                    return false;
                });

                // adding the marginal change tracking nodes to the end of the collector
                if (allChangeTrackNodesMarginal.length > 0) {
                    allChangeTrackNodes = allChangeTrackNodes.concat(allChangeTrackNodesMarginal);
                }

                // needs to be a jQuery collection
                allChangeTrackNodes = $(allChangeTrackNodes);
            }

            return allChangeTrackNodes;
        }

        /**
         * Checking, whether the specified change track selection is a column
         * selection. This is the case if the change track type is 'inserted'
         * and 'removed' and only table cell elements are part of the selection.
         *
         * @param {Object} selection
         *  The change track selection that will be investigated.
         *
         * @returns {Boolean}
         *  Whether the specified change track selection is a column selection.
         */
        function checkColumnSelection(selection) {

            var // a helper node
                noCellNode = null;

            // checking node type
            if (selection.activeNodeType === 'modified') { return false; }

            // checking the selected nodes: A column selection must have 2 cells at least
            if ((!selection.selectedNodes) || (selection.selectedNodes.length < 2)) { return false; }

            // checking, that all members of selected nodes are cells
            noCellNode = _.find(selection.selectedNodes, function (oneNode) {
                return !DOM.isTableCellNode(oneNode);
            });

            if (noCellNode) { return false; }

            return true;
        }

        /**
         * Setting a change track selection. The new change track selection must be
         * given to this function. The text selection is set to the first logical
         * position of the active node of the change track selection. So it must
         * not be possible, that a text selection is visible next to a change track
         * selection.
         * Before setting the new change track selection, an old change track
         * selection is cleared.
         *
         * @param {Object} selection
         *  The change track selection that will be set.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the change track selection.
         *  The following options are supported:
         *  @param {Boolean} [options.makeVisible=true]
         *      Whether the change track selection shall be made visible immediately.
         *  @param {Boolean} [keepChangeTrackPopup=false]
         *      Whether this change of change track selection was triggered by the change
         *      track pop up. In this case the resulting change of text selection must
         *      not lead to an update of the change track pop up.
         */
        function setChangeTrackSelection(selection, options) {

            var // whether the change track selection shall be made visible
                makeVisible = Utils.getBooleanOption(options, 'makeVisible', true),
                // whether the change track pop up triggered this change of selection
                keepChangeTrackPopup = Utils.getBooleanOption(options, 'keepChangeTrackPopup', false),
                // whether this change track selection is a column selection (in this case the text selection must be a cursor selection)
                isColumnSelection = checkColumnSelection(selection),
                // whether this is a selection inside header or footer
                isMarginalSelection = DOM.isMarginalNode(selection.activeNode),
                // the root node for the logical position
                localRootNode = isMarginalSelection ? DOM.getMarginalTargetNode(rootNode, selection.activeNode) : rootNode,
                // the logical start position of the change track selection
                textStartPosition = null,
                // the logical end position of the change track selection
                textEndPosition = null,
                // whether the last position of the node shall be calculated
                useLastPosition = true;

            // if the change track is located inside header or footer, the edit mode must be activated
            if (isMarginalSelection) {
                // Check, if this is the correct header / footer that is activated
                if (model.isHeaderFooterEditState()) {
                    if (Utils.getDomNode(model.getCurrentRootNode()) !== Utils.getDomNode(localRootNode)) {
                        // Switching to another header or footer
                        model.getPageLayout().switchTargetNodeInHeaderMode(localRootNode);
                    }
                } else {
                    model.getPageLayout().enterHeaderFooterEditMode(localRootNode);
                }
            } else {
                if (model.isHeaderFooterEditState()) {
                    model.getPageLayout().leaveHeaderFooterEditMode();
                    model.getSelection().setNewRootNode(rootNode); // restore original rootNode
                }
            }

            // the logical start position of the change track selection
            textStartPosition = Position.getTextLevelPositionInNode(_.first(selection.selectedNodes), localRootNode);
            // the logical end position of the change track selection
            useLastPosition = !DOM.isParagraphNode(_.last(selection.selectedNodes)); // task 40735, not highlighting complete following paragraph
            textEndPosition = isColumnSelection ? null : Position.getTextLevelPositionInNode(_.last(selection.selectedNodes), localRootNode, { lastPosition: useLastPosition });

            // removing color of existing selection
            self.clearChangeTrackSelection();

            // assigning the new selection
            changeTrackSelection = selection;

            // highlight change track nodes
            if (makeVisible) {
                _.each(changeTrackSelection.selectedNodes, function (node) {
                    // highlight table nodes only
                    if (DOM.isTableNode(node) || DOM.isTableRowNode(node) || DOM.isTableCellNode(node)) { $(node).addClass(highlightClassname); }
                });
            }

            // adapting the text selection to the change track selection
            model.getSelection().setTextSelection(textStartPosition, textEndPosition,  { keepChangeTrackPopup: keepChangeTrackPopup });
        }

        /**
         * Finding the next or following change tracking node from the current text selection (not
         * change track selection!).
         *
         * @param {jQuery} allNodes
         *  A jQuery list with all change track nodes in the document.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the search of the change track node.
         *  The following options are supported:
         *  @param {Boolean} [options.next=true]
         *      Whether the following or the previous change track node shall be found.
         *  @param {Boolean} [options.reuseSameNode=false]
         *      Whether the same selected node can be reused. This is useful, if
         *      the same node contains a further change tracked modification and in the
         *      previous step one of these changes was accepted or rejected (41153).
         *  @param {Boolean} [options.useDocumentBorder=false]
         *      Whether the change track node shall be found starting the search from the beginning
         *      (for next = true) or the end (for next = false) of the document.
         *
         * @returns {HTMLElement|Null}
         *  The next or the previous change track node in relation to the current cursor position.
         *  If this element is a span inside an in-line component, the in-line component itself
         *  must be returned.
         */
        function getNextChangeTrackNode(allNodes, options) {

            var // whether the following valid change track shall be found or the previous
                next = Utils.getBooleanOption(options, 'next', true),
                // whether the following valid change track shall be found or the previous
                reuseSameNode = Utils.getBooleanOption(options, 'reuseSameNode', false),
                // whether the search shall be started at the beginning or end of the document
                useDocumentBorder = Utils.getBooleanOption(options, 'useDocumentBorder', false),
                // start searching at the current start position of the selection
                name = next ? 'findNextNode' : 'findPreviousNode',
                // the start position from which the neighboring change track node shall be searched
                startPosition = useDocumentBorder ? (next ? model.getSelection().getFirstDocumentPosition() : model.getSelection().getLastDocumentPosition()) : model.getSelection().getStartPosition(),
                // the root node corresponding to the logical position
                localRootNode = model.getSelection().getRootNode(),
                // node info for the current selection
                nodeInfo = Position.getDOMPosition(localRootNode, startPosition),
                // the node currently selected by the text cursor (or at the beginning of an arbitrary selection range)
                node = nodeInfo ? nodeInfo.node : null,
                // the change tracked node for the selection (must not be a span inside an in-line component
                nextNode = null,
                // whether the search need to be continued, because the found node is inside template node
                repeat = true;

            if (!node) { return null; }

            // replacing text nodes by its parent text span
            if (node.nodeType === 3) {
                node = node.parentNode;
            }

            // handle a change tracked node immediately, if it contains selection (also handling for 41153)
            // -> handling the parent node, because the paragraph might be the change tracked node
            if (reuseSameNode) {
                if (allNodes.is(node)) {
                    repeat = false;
                    nextNode = node;
                }

                if (allNodes.is(node.parentNode)) { // but prefer the parent, because paragraph is in collector before its span
                    repeat = false;
                    nextNode = node.parentNode;
                }
            }

            while (repeat) {
                nextNode = Utils[name](localRootNode, node, DOM.CHANGETRACK_NODE_SELECTOR, null);
                repeat = DOM.isMarginalNode(nextNode) && DOM.isInsideHeaderFooterTemplateNode(rootNode, nextNode);
                if (repeat) { node = nextNode; }
            }

            if (!nextNode && allNodes.length > 0) {
                nextNode = allNodes[0]; // no change track found in the specified direction
            }

            // the next node for the selection must not be the span inside an in-line component. In this
            // case it must be the in-line component itself.
            if (nextNode && nextNode.parentNode && DOM.isTextComponentNode(nextNode.parentNode)) {
                nextNode = nextNode.parentNode;
            }

            return nextNode;
        }

        /**
         * Creating and returning an ordered list of change track types for a specified node.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @returns {Any[]}
         *  The ordered list of change track types for the given node. This can be
         *  an array of strings are an empty array.
         */
        function getOrderedChangeTrackTypes(node) {

            var // the different change track node types ordered by priority
                nodeTypeOrder = [];

            // Info: If a change track is 'inserted' and 'removed', the removal has to happen
            // AFTER the insertion. Therefore removal is always the more relevant tracking
            // type, because it happened after the insertion.

            if (DOM.isChangeTrackRemoveNode(node)) {
                nodeTypeOrder.push('removed');
            }

            if (DOM.isChangeTrackInsertNode(node)) {
                nodeTypeOrder.push('inserted');
            }

            if (DOM.isChangeTrackModifyNode(node)) {
                nodeTypeOrder.push('modified');
            }

            return nodeTypeOrder;
        }

        /**
         * Creating and returning a new selection object describing a change track selection.
         * This object requires the elements 'activeNode', 'activeNodeType', 'activeNodeTypesOrdered',
         * 'selectedNodes' and optionally 'allChangeTrackNodes'.
         *
         * @param {HTMLElement} activeNode
         *  The HTMLElement of the change track selection. Must be the first element of all elements
         *  in the selection.
         *
         * @param {String} activeNodeType
         *  The change track type of the active node. Must be one of the values defined in
         *  CHANGE_TRACK_TYPES.
         *
         * @param {String[]} activeNodeTypesOrdered
         *  An ordered list of strings of node types defined in CHANGE_TRACK_TYPES. The order
         *  corresponds to the priority of the change tracking types for the active node.
         *
         * @param {HTMLElement[]} selectedNodes
         *  An ordered list of all HTML elements, that are combined in the current change track
         *  selection.
         *
         * @param {jQuery} [allChangeTrackNodes]
         *  An optional cache that contains all change track elements in the document. This is
         *  saved in the selection for performance reasons.
         *
         * @returns {Object}
         *  An object, describing the change track selection.
         */
        function createChangeTrackSelection(activeNode, activeNodeType, activeNodeTypesOrdered, selectedNodes, allChangeTrackNodes) {

            var // the new change track selection object
                selection = {};

            selection.activeNode = activeNode;
            selection.activeNodeType = activeNodeType;
            selection.activeNodeTypesOrdered = activeNodeTypesOrdered;
            selection.selectedNodes = selectedNodes;
            selection.allChangeTrackNodes = allChangeTrackNodes;  // this can be 'undefined'

            return selection;
        }

        /**
         * Finding a valid change track selection. This can start with an existing
         * change track selection or without such a selection at any text selection.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the search of a valid change
         *  track selection.
         *  The following options are supported:
         *  @param {Boolean} [options.next=true]
         *      Whether the following or the previous change track selection node shall be found.
         *  @param {Boolean} [options.reuseSameNode=false]
         *      Whether the same selected node can be reused. This is useful, if
         *      the same node contains a further change tracked modification.
         *
         * @returns {Object|Null}
         *  The next or the previous change track selection or Null, if no valid change track selection
         *  can be determined.
         */
        function iterateNextValidChangeTrackSelection(options) {

            var // whether the following valid change track shall be found or the previous
                next = Utils.getBooleanOption(options, 'next', true),
                // whether the following valid change track shall be found or the previous
                reuseSameNode = Utils.getBooleanOption(options, 'reuseSameNode', false),
                // whether the iteration shall be continued
                continueIteration = true,
                // whether a valid change track selection was found
                foundValidChangeTrackSelection = false,
                // the active node of a change track selection
                activeNode = null,
                // an ordered list of change track types for the active node
                activeNodeTypesOrdered = null,
                // the change track type for the active node
                activeNodeType = null,
                // the ordered array of selected nodes neighboring the active node and its change track type
                selectedNodes = null,
                // the list of all change track nodes in a jQuery list
                allChangeTrackNodes = null,
                // a helper object to create a change track selection
                localChangeTrackSelection = null;

            // helper function to find next or previous change track node type in ordered list of change track type.
            // If there is no previous or following change track type in the list, null is returned.
            function getNextNodeType() {

                var index = null,
                    newIndex = null;

                if (next && (_.last(activeNodeTypesOrdered) === activeNodeType)) { return null; }

                if (!next && (_.first(activeNodeTypesOrdered) === activeNodeType)) { return null; }

                index = _.indexOf(activeNodeTypesOrdered, activeNodeType);

                newIndex = next ? (index + 1) : (index - 1);

                return activeNodeTypesOrdered[newIndex];
            }

            // helper function to find next or previous change track node in jQuery list of all change track nodes.
            // If there is no previous or following change track node in the jQuery list, null is returned.
            function getNextChangeTrackingNodeInListOfChangeTrackingNodes() {

                var // the index of the node inside the jQuery collection of all change track nodes
                    index = null,
                    // the modified new index
                    newIndex = null,
                    // a helper node
                    checkNode = activeNode;

                // receiving the node from list of all nodes at specified index
                // -> for text component nodes the 'div' element must be returned, not the span inside
                function getNodeAtIndex(index) {

                    if (DOM.isTextComponentNode(Utils.getDomNode(allChangeTrackNodes[index]).parentNode)) {
                        return Utils.getDomNode(allChangeTrackNodes[index]).parentNode;
                    }
                    return allChangeTrackNodes[index];
                }

                // the active node can be an in-line component, but the list of all change tracks contains
                // the span inside the in-line components.
                if (DOM.isTextComponentNode(checkNode) && !DOM.isDrawingPlaceHolderNode(checkNode)) { checkNode = Utils.getDomNode(checkNode).firstChild; }

                // searching the current node inside the list of all change track nodes.
                index = allChangeTrackNodes.index(checkNode);

                if (next && (index === allChangeTrackNodes.length - 1)) { return getNodeAtIndex(0); }

                if (!next && (index === 0)) { return getNodeAtIndex(allChangeTrackNodes.length - 1); }

                newIndex = next ? (index + 1) : (index - 1);

                return getNodeAtIndex(newIndex);
            }

            if (changeTrackSelection) {

                // there is an existing change track selection -> using this selection as start point for new selection
                if (!changeTrackSelection.allChangeTrackNodes) { changeTrackSelection.allChangeTrackNodes = getAllChangeTrackNodes(); }

                allChangeTrackNodes = changeTrackSelection.allChangeTrackNodes;
                activeNode = changeTrackSelection.activeNode;
                activeNodeType = changeTrackSelection.activeNodeType;
                activeNodeTypesOrdered = changeTrackSelection.activeNodeTypesOrdered;

                // first iteration step for existing change track selection
                activeNodeType = getNextNodeType();
                if (!activeNodeType) {
                    // finding the next change tracking node
                    activeNode = getNextChangeTrackingNodeInListOfChangeTrackingNodes();
                    if (!activeNode) { return null; }
                    activeNodeTypesOrdered = getOrderedChangeTrackTypes(activeNode);
                    activeNodeType = next ? _.first(activeNodeTypesOrdered) : _.last(activeNodeTypesOrdered);
                }

            } else {
                // There is no existing change track selection
                // -> collecting all change tracking nodes in the document
                allChangeTrackNodes = getAllChangeTrackNodes();
                // there is no existing change track selection -> finding the next or previous change track node
                if (allChangeTrackNodes.length > 0) {
                    activeNode = getNextChangeTrackNode(allChangeTrackNodes, { next: next, reuseSameNode: reuseSameNode });
                    if (!activeNode) {
                        // there is the chance to find a change track element, starting from beginning or end of document
                        activeNode = getNextChangeTrackNode(allChangeTrackNodes, { next: next, useDocumentBorder: true });
                        if (!activeNode) { return null; }
                    }
                    activeNodeTypesOrdered = getOrderedChangeTrackTypes(activeNode);
                    activeNodeType = next ? _.first(activeNodeTypesOrdered) : _.last(activeNodeTypesOrdered);
                } else {
                    return null;
                }

            }

            // 1. iterating over all change tracking nodes
            while (activeNode && continueIteration) {

                // 2. iterating over all activeNodesTypesOrdered
                while (activeNodeType && continueIteration) {

                    if (DOM.isFloatingNode(activeNode) || DOM.isDrawingPlaceHolderNode(activeNode)) {
                        // floated nodes (drawings) are never grouped with their neighbors
                        selectedNodes = [activeNode];
                    } else {
                        // finding all adjacent elements for the specified active node and the change track type.
                        // Using 'findTableChildren', so that search in combineAdjacentNodes happens not from span to
                        // paragraph to cell to row to table, but in the opposite direction.
                        selectedNodes = combineAdjacentNodes(activeNode, { type: activeNodeType, findTableChildren: true });
                    }

                    // is the active node type the first element of the selected nodes?
                    if (selectedNodes && (selectedNodes[0] === activeNode)) {
                        continueIteration = false;
                        foundValidChangeTrackSelection = true;
                    }

                    if (!foundValidChangeTrackSelection) {
                        // changing to the next/previous change tracking type
                        activeNodeType = getNextNodeType();
                    }
                }

                if (!foundValidChangeTrackSelection) {
                    // finding the next change tracking node
                    activeNode = getNextChangeTrackingNodeInListOfChangeTrackingNodes();
                    if (!activeNode) { return null; }
                    activeNodeTypesOrdered = getOrderedChangeTrackTypes(activeNode);
                    activeNodeType = next ? _.first(activeNodeTypesOrdered) : _.last(activeNodeTypesOrdered);
                }
            }

            if (foundValidChangeTrackSelection) {
                localChangeTrackSelection = createChangeTrackSelection(activeNode, activeNodeType, activeNodeTypesOrdered, selectedNodes, allChangeTrackNodes);
            }

            return localChangeTrackSelection;
        }

        /**
         * Selecting all nodes with the same tracking type that are direct neighbors.
         *
         * @param {HTMLElement|jQuery} trackingNode
         *  Element that will be evaluated for change tracking.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the preparation of the span.
         *  The following options are supported:
         *  @param {String} [options.type='']
         *   The type of resolving, must be member of CHANGE_TRACK_TYPES.
         *  @param {Boolean} [options.findTableChildren=false]
         *   Whether the group shall be created from table to its children and not
         *   vice versa. Typically the grouping process is based to search from table
         *   cells to table rows to tables. But if a table node is given to this
         *   function as trackingNode and the rows or cells of the same group need
         *   to be defined, 'findTableChildren' must be set to true.
         *
         * @returns {HTMLElement[]}
         *  All nodes with the same tracking type that are direct neighbors within
         *  one array.
         */
        function combineAdjacentNodes(trackingNode, options) {

            var // all adjacent nodes with specified type
                allNodes = [],
                // the tracking type of combined nodes
                type = Utils.getStringOption(options, 'type', '').toLowerCase(),
                // the author, whose tracking nodes will be combined
                author = null,
                // the currently investigated node
                currentNode = Utils.getDomNode(trackingNode),
                // whether the tracking node is a table cell node
                isTableCellTrackingNode = DOM.isTableCellNode(trackingNode),
                // whether a table needs to find affected children
                findTableChildren = Utils.getBooleanOption(options, 'findTableChildren', false);

            // helper function to find a previous or following change track node, that
            // has the same resolveType and the same author (checked by the validator).
            function neighboringChangeTrackNode(node, validator, options) {

                var // the relevant change track node
                    neighborNode = null,
                    // the parent of the current node and its neighbor node
                    parentNode = null, neighborParentNode = null, checkNode = null,
                    // the child of a specified paragraph node
                    childNode = null,
                    // the currently investigated paragraph, cell, row, or table node
                    paraNode = null, cellNode = null, rowNode = null, tableNode = null,
                    // whether the next or previous node shall be found
                    next = Utils.getBooleanOption(options, 'next', true),
                    // whether neighbors with same modification shall be searched
                    modified = Utils.getBooleanOption(options, 'modified', false),
                    // the name of the dom function to be called
                    finder = next ? 'nextSibling' : 'previousSibling';

                if (DOM.isTableNode(node)) {
                    neighborNode = null;  // no additional node need to be collected
                    // -> is it possible, that rows and cells need to be collected, too?
                    // -> neighboring tables will not be combined (like neighboring paragraphs)
                } else if (DOM.isTableRowNode(node)) {

                    rowNode = node;

                    if (rowNode[finder]) {
                        rowNode = rowNode[finder];

                        // skipping page break rows
                        while (rowNode && DOM.isTablePageBreakRowNode(rowNode)) {
                            rowNode = rowNode[finder] || null;
                        }

                        if (rowNode && DOM.isTableRowNode(rowNode) && validator.call(this, rowNode)) {
                            neighborNode = rowNode;
                        }
                    } else {
                        if (next) {
                            // checking also the complete table element, if there is no following row
                            tableNode = rowNode.parentNode.parentNode;

                            if (tableNode && DOM.isTableNode(tableNode) && validator.call(this, tableNode)) {
                                neighborNode = tableNode;
                            }
                        }
                    }

                } else if (DOM.isTableCellNode(node)) {

                    cellNode = node;

                    // 'modified': Finding all single modified cells, also the table (ignoring rows)
                    // 'inserted'/'deleted': All cells in column are found in another process

                    if (!modified) {
                        neighborNode = null;  // accepting insert/delete of one column at a time
                        // -> neighboring columns will not be combined (like neighboring paragraphs)
                        // -> 'inserted' and 'deleted' can be set at cells only for complete columns.
                    } else {

                        // simple case for modifications:
                        // -> accepting/rejecting all modifications in the table in one step
                        // -> this includes modifications in cells and the table itself,
                        // but not rows (example: modified table style)

                        if (cellNode[finder]) {
                            cellNode = cellNode[finder];

                            if (cellNode && DOM.isTableCellNode(cellNode) && validator.call(this, cellNode)) {
                                neighborNode = cellNode;
                            }
                        } else {
                            // no neighboring cell found -> checking cells in next row (but ignoring the row itself)
                            rowNode = cellNode.parentNode;
                            if (rowNode && rowNode[finder]) {
                                rowNode = rowNode[finder];
                                if (rowNode && DOM.isTableRowNode(rowNode)) {
                                    cellNode = next ? rowNode.firstChild : rowNode.lastChild;
                                    if (cellNode && DOM.isTableCellNode(cellNode) && validator.call(this, cellNode)) {
                                        neighborNode = cellNode;
                                    }
                                }
                            } else {
                                if (next) {
                                    // checking also the complete table element, if there is no following row
                                    tableNode = rowNode.parentNode.parentNode;
                                    if (tableNode && DOM.isTableNode(tableNode) && validator.call(this, tableNode)) {
                                        neighborNode = tableNode;
                                    }
                                }
                            }
                        }
                    }

                } else if (DOM.isParagraphNode(node)) {

                    if (modified) {
                        paraNode = node;
                    } else {
                        paraNode = next ? node : node[finder]; // using previous paragraph, when searching to the beginning
                    }

                    // skipping page break nodes
                    while (paraNode && DOM.isPageBreakNode(paraNode)) {
                        paraNode = paraNode[finder] || null;
                    }

                    if (paraNode && DOM.isParagraphNode(paraNode)) {
                        if (modified) {
                            // modifications can be grouped on paragraph level, independent from the content inside the paragraph
                            paraNode = paraNode[finder];
                            if (paraNode && DOM.isParagraphNode(paraNode) && validator.call(this, paraNode)) {
                                neighborNode = paraNode;  // improvement: checking type of modification
                            }
                        } else {
                            childNode = next ? DOM.findFirstPortionSpan(paraNode) : DOM.findLastPortionSpan(paraNode);
                            if (childNode && (DOM.isTextSpan(childNode) || DOM.isTextComponentNode(childNode)) && validator.call(this, childNode)) {
                                neighborNode = childNode;
                            } else if (Position.getParagraphNodeLength(paraNode) === 0) {
                                // checking neighboring paragraph, if it is empty
                                if (next) {
                                    paraNode = paraNode[finder];
                                    // skipping page break nodes
                                    while (paraNode && DOM.isPageBreakNode(paraNode)) {
                                        paraNode = paraNode[finder] || null;
                                    }
                                }

                                if (paraNode && DOM.isParagraphNode(paraNode) && validator.call(this, paraNode)) {
                                    neighborNode = paraNode;  // using the current paragraph
                                }
                            }
                        }
                    }

                } else {  // text level components

                    checkNode = node[finder];  // getting the sibling

                    while (checkNode && !DOM.isInlineComponentNode(checkNode) && !DOM.isTextSpan(checkNode)) {  // but handling only text component siblings
                        checkNode = checkNode[finder] || null;
                    }

                    if (checkNode) {
                        // checking a direct sibling, but ignoring nodes of specified type
                        if (validator.call(this, checkNode)) {
                            neighborNode = checkNode;  // improvement for 'modified': checking type of modification
                        }
                    } else {
                        // checking if a paragraph parent node can be used temporary (only valid for text level nodes)
                        // For modified attributes, no switch from text to para and vice versa
                        if ((!modified) && (DOM.isTextSpan(node) || DOM.isInlineComponentNode(node))) {

                            parentNode = node.parentNode;

                            if (parentNode) {
                                if (!next) {
                                    // searching to the beginning -> adding current paragraph parent
                                    if (DOM.isParagraphNode(parentNode) && validator.call(this, parentNode)) {
                                        neighborNode = parentNode;  // using the current paragraph
                                    }
                                } else {
                                    // searching to the end -> adding the following paragraph parent
                                    neighborParentNode = parentNode[finder];

                                    // skipping page break nodes
                                    while (neighborParentNode && DOM.isPageBreakNode(neighborParentNode)) {
                                        neighborParentNode = neighborParentNode[finder] || null;
                                    }

                                    if (DOM.isParagraphNode(parentNode) && neighborParentNode && DOM.isParagraphNode(neighborParentNode) && validator.call(this, neighborParentNode)) {
                                        neighborNode = neighborParentNode;  // using the previous or following paragraph
                                    }
                                }
                            }
                        }
                    }
                }

                return neighborNode;
            }

            // helper function to collect all neighboring nodes with
            function collectNeighbours(node, validator, options) {

                var // whether the modified neighbors shall be search
                    modified = Utils.getBooleanOption(options, 'modified', false),
                    // the currently investigated neighbor node
                    current = neighboringChangeTrackNode(node, validator, { next: false, modified: modified });

                while (current) {
                    allNodes.unshift(current);
                    current = neighboringChangeTrackNode(current, validator, { next: false, modified: modified });
                }

                current = neighboringChangeTrackNode(node, validator, { next: true, modified: modified });

                while (current) {
                    allNodes.push(current);
                    current = neighboringChangeTrackNode(current, validator, { next: true, modified: modified });
                }

            }

            // helper function to remove empty text spans, range marker nodes or comment
            // place holder nodes, if they are at the beginning or at the end of the nodes
            // collection. This can be more than one on each side, for example caused by
            // floated drawings
            function removeUntrackedNodesAtBorders(allNodes) {

                var // the node to be checked, if it can be removed from selection
                    checkNode = allNodes[allNodes.length - 1];

                while (isChangeTrackSkipNode(checkNode) && !DOM.isChangeTrackNode(checkNode)) {
                    allNodes.pop();
                    checkNode = allNodes[allNodes.length - 1];
                }

                checkNode = allNodes[0];

                while (isChangeTrackSkipNode(checkNode) && !DOM.isChangeTrackNode(checkNode)) {
                    allNodes.shift();
                    checkNode = allNodes[0];
                }

            }

            // Some specified nodes must not split a group of neighboring change track nodes, because
            // they contain meta information, like comments.
            function isChangeTrackSkipNode(node) {
                return DOM.isEmptySpan(node) || DOM.isRangeMarkerNode(node) || DOM.isCommentPlaceHolderNode(node);
            }

            // special handling required for table columns (cells that are inserted or removed)
            if (isTableCellTrackingNode &&
                (((type === 'inserted') && DOM.isChangeTrackInsertNode(currentNode)) || ((type === 'removed') && DOM.isChangeTrackRemoveNode(currentNode)))) {
                // -> collecting all cells of the column and adding it to allNodes
                // -> no further collecting required
                return Table.getAllCellsNodesInColumn(rootNode, currentNode);
            }

            // special handling for table nodes, if this is a 'up-down-selection' (caused by next and forward button)
            // The process in 'collectNeighbours' is a 'down-up' from table cells to rows to tables.
            if (DOM.isTableNode(currentNode) && findTableChildren) {

                // a modified table must be grouped with the modified cells
                if ((type === 'modified') && DOM.isChangeTrackModifyNode(currentNode)) {
                    _.each($(currentNode).find('> tbody > tr > td[data-change-track-modified="true"]'), function (node) {
                        allNodes.push(node);
                    });
                } else if ((type === 'inserted') && DOM.isChangeTrackInsertNode(currentNode)) { // an inserted table must be grouped with all rows
                    _.each($(currentNode).find('> tbody > tr[data-change-track-inserted="true"]'), function (node) {
                        allNodes.push(node);
                    });
                }

                // filtering by author
                if (type) {
                    author = self.getChangeTrackInfoFromNode(currentNode, 'author', type);
                    allNodes = _.filter(allNodes, function (oneNode) {
                        return self.nodeHasChangeTrackAuthor(oneNode, type, author);
                    });
                }

                // finally adding the table node
                // -> in the selection the table must be 'before' its rows, so that resolving of
                // change tracking happens in the correct order (-> unshift instead of push)
                allNodes.unshift(currentNode);

                return allNodes;
            }

            allNodes.push(currentNode);
            author = self.getChangeTrackInfoFromNode(currentNode, 'author', type);

            if ((DOM.isChangeTrackRemoveNode(currentNode)) && (type === 'removed')) {
                collectNeighbours(currentNode, function (node) {
                    return (DOM.isChangeTrackRemoveNode(node) && self.nodeHasChangeTrackAuthor(node, 'removed', author)) || isChangeTrackSkipNode(node);
                });
            } else if ((DOM.isChangeTrackInsertNode(currentNode)) && (type === 'inserted')) {
                collectNeighbours(currentNode, function (node) {
                    return (DOM.isChangeTrackInsertNode(node) && self.nodeHasChangeTrackAuthor(node, 'inserted', author)) || isChangeTrackSkipNode(node);
                });
            } else if (DOM.isChangeTrackModifyNode(currentNode) && (type === 'modified')) {
                collectNeighbours(currentNode, function (node) {
                    return (DOM.isChangeTrackModifyNode(node) && self.nodeHasChangeTrackAuthor(node, 'modified', author)) || isChangeTrackSkipNode(node);
                }, { modified: true });
            }

            // empty spans are only allowed between tracking nodes, if they are not tracked
            removeUntrackedNodesAtBorders(allNodes);

            return allNodes;
        }

        /**
         * Checking whether inside a specified paragraph all elements are change tracked
         * and after resolving all change tracks, the paragraph will be empty.
         *
         * The paragraph will be empty, if:
         * - all children from paragraph node are in the specified list of resolve nodes
         *   and are of type 'inserted' and accepted is false.
         *   or
         * - all children from paragraph node are in the specified list of resolve nodes
         *   and are of type 'removed' and accepted is true.
         *
         * @param {HTMLElement|jQuery} node
         *  The paragraph element whose children will be investigated.
         *
         * @param {jQuery|Object[]} resolveNodes
         *  The list of all nodes, that are resolved in this specific process.
         *
         * @param {Boolean} accepted
         *  Whether the change tracks shall be accepted or rejected.
         *
         * @returns {Boolean}
         *  Whether all sub nodes of the paragraph are removed during accepting
         *  or rejecting the change tracks.
         */
        function paragraphEmptyAfterResolving(node, resolveNodes, accepted) {

            var // the html dom node
                domNode = Utils.getDomNode(node),
                // the child, that is not correctly change tracked
                childNode = null,
                // the change track type that needs to be checked
                type = accepted ? 'removed' : 'inserted';

            // check, if the node is a paragraph
            if (!DOM.isParagraphNode(domNode)) { return false; }

            // iterating over all children. Trying to find one node, that is
            // not an empty text span and that is not of the specific type or
            // that is not in the list of all nodes that are resolved in this process.
            childNode = _.find($(domNode).children(), function (child) {
                return !DOM.isEmptySpan(child) && (!self.nodeHasSpecifiedChangeTrackType(child, type) || !_.contains(resolveNodes, child));
            });

            return !childNode;
        }

        /**
         * Checking, whether a specified node will be removed during current accepting
         * or rejecting of change tracks.
         *
         * @param {HTMLElement|jQuery} node
         *  The node, that will be investigated.
         *
         * @param {jQuery|Object[]} resolveNodes
         *  The list of all nodes, that are resolved in this specific process.
         *
         * @param {Boolean} accepted
         *  Whether the change tracks shall be accepted or rejected.
         *
         * @returns {Boolean}
         *  Whether the node will be removed during current accepting or rejecting of
         *  change tracks.
         */
        function nodeRemovedDuringResolving(node, resolveNodes, accepted) {

            var // whether the node will be removed
                removed = false;

            // helper function
            function nodeRemoved(oneNode) {
                return _.contains(resolveNodes, Utils.getDomNode(oneNode)) && ((DOM.isChangeTrackInsertNode(oneNode) && !accepted) || (DOM.isChangeTrackRemoveNode(oneNode) && accepted));
            }

            // if the node is a table node, all rows need to be checked
            if (DOM.isTableNode(node)) {
                removed = _.all(DOM.getTableRows(node), function (row) {
                    return nodeRemoved(row);
                });
            } else {
                removed = nodeRemoved(node);
            }

            return removed;
        }

        /**
         * Adding change tracking information (change track type object with author and
         * date) to a given operation. The operation is modified directly, no copy is
         * created. This functionality is currently implemented for clipboard operations.
         *
         * @param {Object} operation
         *  The operation that will be expanded for the change track information.
         *
         * @param {Object} [info]
         *  The change tracking information object containing the author and the date.
         *  This parameter is optional. If it is not set, this object is generated.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the addition of change tracking
         *  information.
         *  The following options are supported:
         *  @param {Boolean} [options.clipboard=false]
         *      Whether this function was called from the clip board API.
         */
        function addChangeTrackInfoToOperation(operation, info, options) {

            if (!operation) { return; }

            var // the key for the changes object
                changesKey = null,
                // whether this function was triggered from clip board API
                isClipboard = Utils.getBooleanOption(options, 'clipboard', false);

            // helper function to generate and collect additional setAttributes
            // operations after splitParagraph operations.
            function generateChangeTrackSetAttributesOperation() {
                var paraPos = _.clone(operation.start);
                paraPos.pop();
                paraPos[paraPos.length - 1]++;
                clipboardCollector.push({ name: Operations.SET_ATTRIBUTES, start: paraPos, attrs: { changes: { inserted: info } } });
            }

            // never adding change track info to operations that add content into comments
            if (operation.target && model.getCommentLayer().isCommentTarget(operation.target)) { return; }

            // generating change track info, if not provided
            if (!info) { info = self.getChangeTrackInfo(); }

            // Info: Operations.SET_ATTRIBUTES does not need to be change tracked from clip board operations
            // because during pasting, attributes can also be assigned to inserted content. It is even better
            // not to change track setAttributes (with 'modified' key), because otherwise the user needs to
            // resolve two change tracks.

            if (isClipboard) {
                if (_.contains(CHANGETRACK_CLIPBOARD_OPERATIONS, operation.name)) {
                    changesKey = 'inserted';
                } else if (Operations.PARA_SPLIT === operation.name) {
                    // After splitParagraph an additional setAttributes operations is required,
                    // so that the new paragraph is marked as inserted.
                    generateChangeTrackSetAttributesOperation();
                }
            }

            // Idea: In the future this function can be the central point, where change track information is
            // added to any operation. Not only for clip board generated operations. In this case, the
            // changesKey must be a parameter or it must be determined dynamically.

            if (changesKey) {
                operation.attrs = operation.attrs || {};
                operation.attrs.changes = operation.attrs.changes || {};
                operation.attrs.changes[changesKey] = info;
            }
        }

        /**
         * Checking whether a node will be removed within this change track resolution.
         * In this case, further operations for this node must not be generated, for
         * example additional setAttributes operation, that might cause problems because
         * of the operation order.
         *
         * @param {HTMLElement|jQuery} node
         *  The paragraph element whose children will be investigated.
         *
         * @param {jQuery|Object[]} resolveNodes
         *  The list of all nodes, that are resolved in this specific process.
         *
         * @param {Boolean} accepted
         *  Whether the change tracks shall be accepted or rejected.
         *
         * @returns {Boolean}
         *  Whether the node will be removed within this change track resolution
         *  process.
         */
        function isRemovalNode(node, resolveNodes, accepted) {

            if (!_.contains(resolveNodes, node)) { return false; }

            return ((accepted && DOM.isChangeTrackRemoveNode(node)) || (!accepted && DOM.isChangeTrackInsertNode(node)));
        }

        /**
         * Test function, to guarantee that a setAttributes operation is never executed
         * after a delete operation on the same range. For this it is necessary, that
         * in the specified operations array, the setAttributes operation must not be
         * listed before the delete operation (the order is reversed later).
         * Checked are only operations with logical positions, that are listed in the
         * array 'positions'.
         *
         * @param {Object[]} operations
         *  The operation container.
         *
         * @param {Object} positions
         *  Optionally an array containing objects with the logical positions and target IDs,
         *  that need to be investigated. These objects must have the properties 'pos' and
         *  'targetId'. For comparison of targets it is important, that 'undefined' can be
         *  used.
         */
        function validateDeleteOrderAtPositions(operations, positions) {

            var // the currently investigated position
                position = null, target = null,
                // an array with all indexes of setAttributes operations at specified positions
                setAttributesCollector = [];

            // checking the order of setAttributes and delete operations
            function validateOrder(operation, index) {

                if (operation.name === Operations.SET_ATTRIBUTES && _.isEqual(position, operation.start) && operation.target === target) {
                    setAttributesCollector.push(index); // saving the index for later usage
                }

                if (operation.name === Operations.DELETE && _.isEqual(position, operation.start) && operation.target === target) {

                    if (setAttributesCollector.length > 0) {

                        // reverting the order, so that the index decreases
                        setAttributesCollector.reverse();

                        // iterating over all affected indexes in the operations array
                        _.each(setAttributesCollector, function (oneIndex) {
                            // inserting all set attributes operation behind the delete operation
                            operations.splice(index + 1, 0, operations[oneIndex]);
                            // delete the set attributes operations
                            operations.splice(oneIndex, 1);
                            // reducing the current index of the delete operation
                            index--;
                        });

                    }

                    setAttributesCollector = [];
                    return Utils.BREAK;
                }
            }

            // Iterating over all specified positions
            _.each(positions, function (onePosition) {
                position = onePosition.pos;
                target = onePosition.targetId;  // using undefined as valid value for comparison
                Utils.iterateArray(operations, validateOrder);
            });
        }

        /**
         * Test function, to guarantee that a setAttributes operation is never executed
         * after a delete or a merge operation on the same range. For this it is necessary,
         * that in the specified operations array, the setAttributes operation must not be
         * listed before the delete or merge operation (the order is reverted later).
         *
         * @param {Object[]} operations
         *  The operation container.
         *
         * @param {String} operationName
         *  The name of the operation that is compared with the setAttributes operation.
         *
         * @param {Number} [positionCorrection]
         *  An optional correction value for the logical start position, dependent from
         *  the operation. The 'delete' operation can be used as is, but the 'merge'
         *  operation needs to be increased by '1', because a setAttribute at position
         *  [1] must not be executed after a merge operation at position [0].
         */
        function validateSetAttributesOrder(operations, operationName, positionCorrection) {

            var // whether the check needs to be repeated
                doCheck = true,
                // a collector for all setAttributes operations
                setAttributesCollector = {},
                // a key generated from start and end position
                key = '';

            // helper function to generate keys to recognize start and end position
            function generateKey(posA, posB, target) {
                return posA.join('_') + '__' + posB.join('_') + (_.isString(target) ? ('__' + target) : '');
            }

            // checking the order of setAttributes and delete operations
            function validateOrder(operation, index) {

                var // a logical position that includes optionally the specified corrected position
                    correctedPosition = null;

                if (operation.name === Operations.SET_ATTRIBUTES) {
                    key = generateKey(operation.start, operation.end || operation.start, operation.target);
                    setAttributesCollector[key] = index; // saving the index for later usage
                }

                if (operation.name === operationName) {

                    if (_.isNumber(positionCorrection) && positionCorrection > 0) {
                        correctedPosition = _.clone(operation.start);
                        correctedPosition[correctedPosition.length - 1] += positionCorrection;
                    }

                    key = generateKey(correctedPosition ? correctedPosition : operation.start, operation.end || (correctedPosition ? correctedPosition : operation.start), operation.target);

                    if (_.isNumber(setAttributesCollector[key])) {
                        // inserting the set attributes operation behind the delete operation
                        operations.splice(index + 1, 0, operations[setAttributesCollector[key]]);
                        // delete the set attributes operation
                        operations.splice(setAttributesCollector[key], 1);
                        // repeating the check from the beginning
                        doCheck = true;
                        return Utils.BREAK;
                    }
                }
            }

            // running the test
            while (doCheck) {
                doCheck = false;
                setAttributesCollector = {};
                Utils.iterateArray(operations, validateOrder);
            }
        }

        /**
         * To increase the stability of the operations it is sometimes necessary to
         * check the correct order of the operations. This is especially necessary,
         * if the all change track types are resolved in one step. This happens after
         * pressing button 'Accept all' or 'Reject all' or after accepting or rejecting
         * a selection range.
         *
         * @param {Object[]} operations
         *  The operation container.
         *
         * @param {Object} [allPositions]
         *  Optionally an array containing objects with the logical positions and target IDs,
         *  that need to be investigated. These objects must have the properties 'pos' and
         *  'targetId'. For comparison of targets it is important, that 'undefined' can be
         *  used.
         */
        function checkOperationsIntegrity(operations, allPositions) {

            if (_.isArray(allPositions)) {
                validateDeleteOrderAtPositions(operations, allPositions);
            } else {
                // Check: setAttributes must be positioned before delete (for same range)
                validateSetAttributesOrder(operations, Operations.DELETE);
                // Check: setAttributes must be positioned before merge (for same range)
                validateSetAttributesOrder(operations, Operations.PARA_MERGE, 1);
            }

            // additional checks can follow here
        }

        /**
         * Checking the validity of an existing change track selection. In some cases
         * it might happen, that a selection is no longer valid (36299)
         *
         * @param {Object} selection
         *  The change track selection that will be set.
         */
        function checkSelectionValidity(selection) {

            try {
                // check, if the node is still in the DOM
                Position.getTextLevelPositionInNode(_.first(selection.selectedNodes), rootNode);
            } catch (ex) {
                // deleting an existing change track selection
                self.clearChangeTrackSelection();
            }
        }

        /**
         * removes all change track highlight marker.
         *  Only used for ios-devices.
         */
        function removeHighlighting() {
            rootNode.find('.' + highlightClassname).removeClass(highlightClassname);
        }

        /**
         * After the event 'document:reloaded' was fired by the model, some cached nodes
         * need to be invalidated.
         */
        function handleDocumentReload() {
            // invalidating cached nodes, after the document was reloaded
            sideBarNode = null;
            pageContentNode = null;
        }

        // methods ------------------------------------------------------------

        /**
         * Whether the assignment of a new position (in-line, left floated, ...) to a drawing
         * needs to be change tracked. Currently this is not supported in OOXML format.
         * Therefore the change track information will  not be available in MS Word and will
         * be lost after reloading a document into OX Text. In this case, it will lead to an
         * error, if the change will be rejected in OX Text.
         */
        this.ctSupportsDrawingPosition = function () {
            return CHANGE_TRACK_SUPPORTS_DRAWING_POSITION;
        };

        /**
         * Whether the resizing of a drawing in OX Text needs to be change tracked. Currently
         * this is not supported in OOXML format. Therefore the change track information will
         * not be available in MS Word and will be lost after reloading a document into OX Text.
         * In this case, it will lead to an error, if the change will be rejected in OX Text.
         */
        this.ctSupportsDrawingResizing = function () {
            return CHANGE_TRACK_SUPPORTS_DRAWING_RESIZING;
        };

        /**
         * Whether the moving of a drawing in OX Text needs to be change tracked. Currently
         * this is not supported in OOXML format. Therefore the change track information will
         * not be available in MS Word and will be lost after reloading a document into OX Text.
         * In this case, it will lead to an error, if the change will be rejected in OX Text.
         * On the other hand, OOXML supports change tracking of moving of in-line drawings
         * (from in-line to in-line), what is not supported by OX Text yet.
         */
        this.ctSupportsDrawingMove = function () {
            return CHANGE_TRACK_SUPPORTS_DRAWING_MOVE;
        };

        /**
         * Updates the current date for change track operations on a regular basis.
         * This is needed for performance reasons, so that the function
         * DateUtils.getMSFormatIsoDateString() is not called for every operation.
         */
        this.updateChangeTrackInfoDate = function () {
            changeTrackInfoDate = DateUtils.getMSFormatIsoDateString({ useSeconds: false });
        };

        /**
         * Returns, whether change tracking is activated on the document.
         *
         * @returns {Boolean}
         *  Whether change tracking is activated locally.
         */
        this.isActiveChangeTracking = function () {
            if (app.isODF()) { return false; } // in odf files change tracking is not supported yet
            return (isChangeTrackSupported && model.getDocumentAttribute('changeTracking')) || false;
        };

        /**
         * Returns, whether change tracking is activated in the document attributes. This state is
         * shown in the UI check box. This is not used during working with the document, because the
         * state can be changed temporarily, for example if a comment node is activated.
         *
         * @returns {Boolean}
         *  Whether change tracking is activated locally (independent from the global variable
         *  'isChangeTrackSupported' that can be used to deactivate change tracking temporarily).
         */
        this.isActiveChangeTrackingInDocAttributes = function () {
            if (app.isODF()) { return false; } // in odf files change tracking is not supported yet
            return (model.getDocumentAttribute('changeTracking')) || false;
        };

        /**
         * Sets the state for change tracking to 'true' or 'false'.
         * If set to true, change tracking is activated on the document.
         *
         * @param {Boolean} state
         *  The state for the change tracking. 'true' enables change tracking,
         *  'false' disables' change tracking.
         */
        this.setActiveChangeTracking = function (state) {
            if (typeof state !== 'boolean') { return; }
            model.applyOperations({ name: Operations.SET_DOCUMENT_ATTRIBUTES, attrs: { document: { changeTracking: state } } });
            // activating the change tracking timer update
            if (state) {
                startDateUpdate();
            } else {
                stopDateUpdate();
            }
        };

        /**
         * Returns the current state of the global 'isChangeTrackSupported'. This
         * variable can be used to disable change tracking temporarily, for example
         * inside comments.
         *
         * @returns {Boolean}
         *  Whether change tracking is supported in the current context.
         */
        this.isChangeTrackSupported = function () {
            return isChangeTrackSupported;
        };

        /**
         * Sets the state of the global variable 'isChangeTrackSupported'. This
         * variable can be used to disable change tracking temporarily, for example
         * inside comments.
         *
         * @param {Boolean} state
         *  Setting the state for the change tracking support in the current context.
         */
        this.setChangeTrackSupported = function (state) {
            isChangeTrackSupported = state;
        };

        /**
         * Returning the current author of the application as string
         * with full display name.
         *
         * @returns {String}
         *  The author string for the change track info object.
         */
        this.getChangeTrackInfoAuthor = function () {
            return app.getClientOperationName();
        };

        /**
         * Returning the current date as string in ISO date format.
         * This function calls DateUtils.getMSFormatIsoDateString() to simulate
         * the MS behavior, that the local time is send a UTC time. The
         * ending 'Z' in the string represents UTC time, but MS always
         * uses local time with ending 'Z'.
         *
         * @returns {String}
         *  The date string for the change track info object.
         */
        this.getChangeTrackInfoDate = function () {
            // Performance: changeTrackInfoDate is stored in a variable, needs only to be updated every 30s
            if (!dateUpdatePromise && self.isActiveChangeTracking()) { startDateUpdate(); }
            return changeTrackInfoDate ? changeTrackInfoDate : DateUtils.getMSFormatIsoDateString({ useSeconds: false });
        };

        /**
         * Returning an object used for additional change track information with author
         * and date of tracked change.
         *
         * @returns {Object}
         *  The change track info object containing author and date.
         */
        this.getChangeTrackInfo = function () {
            if (!dateUpdatePromise && self.isActiveChangeTracking()) { startDateUpdate(); }
            if (!userid) { userid = Utils.calculateUserId(ox.user_id); }
            return { author: self.getChangeTrackInfoAuthor(), date: self.getChangeTrackInfoDate(), uid: userid.toString() };
        };

        /**
         * Receiving the author, date and uid of a change track node for a specified type.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @param {String} info
         *  The info that is searched from the node. Allowed values are listed in
         *  'CHANGE_TRACK_INFOS'. 'author', 'date' and 'uid' are currently supported.
         *
         * @param {String} type
         *  The type for which the author is searched. Allowed values are listed in
         *  'CHANGE_TRACK_TYPES'.
         *
         * @returns {Number|String|Null}
         *  The requested information. This can be the 'author', the 'date' or null,
         *  if the information could not be found.
         */
        this.getChangeTrackInfoFromNode = function (node, info, type) {

            if (!node || !_.contains(CHANGE_TRACK_TYPES, type) || !_.contains(CHANGE_TRACK_INFOS, info)) { return null; }

            if (DOM.isTextComponentNode(node) && !DOM.isComplexFieldNode(node)) { node = Utils.getDomNode(node).firstChild; }

            var // the author or date of the change tracked node
                value = null,
                // the explicit attributes set at the specified node
                explicitAttributes = AttributeUtils.getExplicitAttributes(node);

            if (explicitAttributes && explicitAttributes.changes && explicitAttributes.changes[type]) {
                value = explicitAttributes.changes[type][info];
                if (info === 'uid') { value = Utils.resolveUserId(parseInt(value, 10)); }
            }

            return value;
        };

        /**
         * Checking whether a specified node is a change tracked 'inserted' node, that was inserted
         * by the currently editing user.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @returns {Boolean}
         *  Whether a specified node is a change tracked 'inserted' node, that was inserted
         *  by the currently editing user.
         */
        this.isInsertNodeByCurrentAuthor = function (node) {
            return node && DOM.isChangeTrackInsertNode(node) && (self.getChangeTrackInfoFromNode(node, 'author', 'inserted') === self.getChangeTrackInfoAuthor());
        };

        /**
         * Checking whether all specified nodes in the nodes collector are change tracked
         * 'inserted', and this was inserted by the currently editing user.
         *
         * @param {Node[]|jQuery|Null} nodes
         *  The DOM nodes to be checked. This can be an array of nodes or a jQuery collection.
         *  If missing or empty, returns false.
         *
         * @returns {Boolean}
         *  Whether all specified nodes are change tracked 'inserted', and this was inserted
         *  by the currently editing user.
         */
        this.allAreInsertedNodesByCurrentAuthor = function (nodes) {

            var // whether all specified nodes in the collector were inserted by the current author
                allInserted = true;

            if ((!nodes) || (nodes.length === 0)) { return false; }

            Utils.iterateArray(nodes, function (node) {
                if (!self.isInsertNodeByCurrentAuthor(node)) {
                    allInserted = false;
                    return Utils.BREAK;
                }
            });

            return allInserted;
        };

        /**
         * Checking whether a specified node has a specified change track type.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @param {String} type
         *  The type for which the node is checked. Allowed values are listed in
         *  'CHANGE_TRACK_TYPES'.
         *
         * @returns {Boolean}
         *  Whether the specified node has the specified change track type.
         */
        this.nodeHasSpecifiedChangeTrackType = function (node, type) {
            return node &&
                   ((type === 'inserted' && DOM.isChangeTrackInsertNode(node)) ||
                    (type === 'removed' && DOM.isChangeTrackRemoveNode(node)) ||
                    (type === 'modified' && DOM.isChangeTrackModifyNode(node)));
        };

        /**
         * Checking whether a specified node has a specified author for a specified
         * change track type.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @param {String} type
         *  The type for which the author is searched. Allowed values are listed in
         *  'CHANGE_TRACK_TYPES'.
         *
         * @param {String} author
         *  The specified author, who is checked to be the change track owner.
         *
         * @returns {Boolean}
         *  Whether the specified node has the specified author.
         */
        this.nodeHasChangeTrackAuthor = function (node, type, author) {
            return node && self.nodeHasSpecifiedChangeTrackType(node, type) && (self.getChangeTrackInfoFromNode(node, 'author', type) === author);
        };

        /**
         * During pasting content into the document, if change tracking is NOT
         * activated, one additional setAttributes operations is required,
         * directly following the first splitParagraph operation. This
         * setAttributes operation removes change tracking information, from
         * the split paragraph.
         *
         * @param {Object[]} operations
         *  The list of all operations that will be applied during pasting.
         */
        this.removeChangeTrackInfoAfterSplitInPaste = function (operations) {

            var // a counter for the operations
                counter = 0,
                // the first split operation in the operations array
                splitOperation = null;

            // helper function to create a setAttributes operation
            // to remove change tracks family.
            function generateChangeTrackSetAttributesOperation(operation) {
                var paraPos = _.clone(operation.start);
                paraPos.pop();
                paraPos[paraPos.length - 1]++;
                return { name: Operations.SET_ATTRIBUTES, start: paraPos, attrs: { changes: { inserted: null, removed: null, modified: null } } };
            }

            // finding the first splitParagraph operation
            splitOperation = _.find(operations, function (operation) {
                counter++;
                return (Operations.PARA_SPLIT === operation.name);
            });

            // Adding a setAttributes operation behind the first splitParagraph operation
            if (splitOperation) {
                operations.splice(counter, 0, generateChangeTrackSetAttributesOperation(splitOperation));
            }
        };

        /**
         * During pasting content into the document, if change tracking is activated,
         * the change track information need to be added to the operations.
         * Additionally some additional setAttributes operations are required,
         * directly following the splitParagraph operations.
         * If there is content pasted into a comment, this must not be change tracked.
         *
         * @param {Object[]} operations
         *  The list of all operations that will be applied during pasting.
         */
        this.handleChangeTrackingDuringPaste = function (operations) {

            var // the information object for the change tracking
                changeTrackInfo = self.getChangeTrackInfo();

            _(operations).map(function (operation) {
                if (!operation) { return null; }
                addChangeTrackInfoToOperation(operation, changeTrackInfo, { clipboard: true });
                return operation;
            });

            // order the new setAttributes operations into the correct positions
            // in the operations collector
            if (clipboardCollector.length) {
                _.each(operations, function (operation, index) {
                    if (Operations.PARA_SPLIT === operation.name) {
                        var newOperation = clipboardCollector.shift();
                        operations.splice(index + 1, 0, newOperation);
                    }
                });
                // operations = operations.concat(self.getClipboardCollector());
                clipboardCollector = [];
            }
        };

        /**
         * Shows the change track group of a user selection by setting
         * temporary background colors to all change tracking nodes,
         * which belongs to the current change track node of the selection.
         */
        this.showChangeTrackGroup = function () {

            // do nothing with range selections (explicitely also not with drawing selections)
            if (model.getSelection().hasRange()) { return; } // && !model.getSelection().isDrawingFrameSelection()) { return; }

            var // the node currently selected by the text cursor (or at the beginning of an arbitrary selection range)
                node = findChangeTrackNodeAtCursorPosition();

            // quit if a change track node is not found
            if (node.length === 0) { return; }

            var nodeRelevantType = this.getRelevantTrackingTypeByNode(node),
                nodeGroup = groupTrackingNodes(node, nodeRelevantType),
                activeNode = nodeGroup[0],
                activeNodeTypesOrdered = getOrderedChangeTrackTypes(activeNode),
                localChangeTrackSelection = createChangeTrackSelection(activeNode, nodeRelevantType, activeNodeTypesOrdered, nodeGroup);

            if (localChangeTrackSelection) {
                // setting the new change track selection. Because this process also modifies the text
                // selection, this could repaint the visible change track pop up again. To avoid this
                // the flag 'keepChangeTrackPopup' must be transported to the 'change' and 'selection'
                // events, so that the change track pop up is not
                setChangeTrackSelection(localChangeTrackSelection, { keepChangeTrackPopup: true });
            }

        };

        /**
         *  Clears the current change track selection and remove all of their highlights
         */
        this.clearChangeTrackSelection = function () {
            if (!changeTrackSelection) { return; }
            $(changeTrackSelection.selectedNodes).removeClass(highlightClassname);
            changeTrackSelection = null;
        };

        /**
         * Receiving the relevant tracking information of a specified node.
         * Relevant means, that 'inserted' or 'removed' are more important
         * than 'modified'.
         *
         * @param {HTMLElement|jQuery} oneNode
         *  The element whose tracking information is investigated.
         *
         * @returns {Object}
         *  The relevant tracking info of a given node, or null, if the
         *  selection contains no tracking node. The relevant info is an
         *  object containing the properties 'type', 'author' and 'date'.
         */
        this.getRelevantTrackingInfoByNode = function (oneNode) {

            var // the explicit attributes of the node
                explicitAttributes = null,
                // the relevant type of the node
                allTypes = null, type = null,
                // the type of the node affected by change track
                nodeType = null,
                // the author, date and user id of the modified type
                author = null, date = null, uid = null,
                // the jQuerified node
                localNode = oneNode ? $(oneNode) : null;

            // helper function to determine the type of the node
            // that is displayed to the user:
            // Allowed values: text, paragraph, table cell, table row, table, drawing, text frame
            function getTypeOfNode(node) {

                var description = '';

                if (DOM.isTextSpan(node) || DOM.isTextComponentNode(node)) {
                    description = 'text';
                } else if (DOM.isParagraphNode(node)) {
                    description = 'paragraph';
                } else if (DOM.isTableNode(node)) {
                    description = 'table';
                } else if (DOM.isTableRowNode(node)) {
                    description = 'table row';
                } else if (DOM.isTableCellNode(node)) {
                    description = 'table cell';
                } else if (DrawingFrame.isTextFrameShapeDrawingFrame(node)) {
                    description = 'text frame';
                } else if (DOM.isDrawingFrame(node)) {
                    description = 'drawing';
                }

                return description;
            }

            if (localNode && localNode.length > 0) {

                if (DOM.isTextComponentNode(localNode)) { localNode = $(Utils.getDomNode(localNode).firstChild); }

                explicitAttributes = AttributeUtils.getExplicitAttributes(localNode);

                if (explicitAttributes && explicitAttributes.changes) {

                    // finding the relevant change track type
                    allTypes = getOrderedChangeTrackTypes(localNode);
                    type = allTypes.length > 0 ? allTypes[0] : null;

                    if (type && explicitAttributes.changes[type]) {
                        author = explicitAttributes.changes[type].author;
                        date = explicitAttributes.changes[type].date;
                        uid = _.isString(explicitAttributes.changes[type].uid) ? Utils.resolveUserId(parseInt(explicitAttributes.changes[type].uid, 10)) : null;

                        // also returning the node type, that will be affected by this change track:
                        // text, paragraph, table cell, table row, table
                        nodeType = getTypeOfNode(localNode);
                    }
                }

            }

            return type ? { type: type, author: author, date: date, uid: uid, nodeType: nodeType } : null;
        };

        /**
         * Receiving the relevant tracking information of the current selection.
         * If the selection has a range, null is returned. Relevant means,
         * that 'inserted' or 'removed' are more important than 'modified'.
         *
         * @returns {Object|null}
         *  The relevant tracking info of a given node, or null, if the
         *  selection contains no tracking node. The relevant info is an
         *  object containing the properties 'type', 'author' and 'date'.
         */
        this.getRelevantTrackingInfo = function () {

            // no relevant type, if there is a selection range
            // but allowing selection of inline elements (tab, drawing, ...)

            // -> checking inline div elements or (floated) drawings
            if (model.getSelection().hasRange() && !model.getSelection().isInlineComponentSelection() && !model.getSelection().isDrawingFrameSelection()) { return null; }

            var node = findChangeTrackNodeAtCursorPosition();

            if (!node || node.length === 0) { return null; }

            return self.getRelevantTrackingInfoByNode(node);
        };

        /**
         * Receiving the relevant tracking information of a specified node.
         * Relevant tracking type means, that 'inserted' or 'removed' are
         * more important than 'modified'.
         *
         * @param {HTMLElement|jQuery} node
         *  The element whose tracking type is investigated.
         *
         * @returns {String}
         *  The relevant tracking type of a given node, or null, if the
         *  selection contains no tracking node.
         */
        this.getRelevantTrackingTypeByNode = function (node) {
            var info = self.getRelevantTrackingInfoByNode(node);
            return info ? info.type : null;
        };

        /**
         * Receiving the relevant tracking type of the current selection.
         * If the selection has a range, null is returned. Relevant means,
         * that 'inserted' or 'removed' are more important than 'modified'.
         *
         * @returns {String}
         *  The relevant tracking type of a given node, or null, if the
         *  selection contains no tracking node.
         */
        this.getRelevantTrackingType = function () {
            var info = self.getRelevantTrackingInfo();
            return info ? info.type : null;
        };

        /**
         * Checking, if inside an array of operations at least one operation
         * contains change tracking information.
         *
         * @param {Array} operations
         *  The operations to be checked.
         *
         * @returns {Boolean}
         *  Whether at least one operation in the specified array of operations
         *  contains change tracking information.
         */
        this.hasChangeTrackOperation = function (operations) {
            return _.filter(operations, self.isChangeTrackOperation).length > 0;
        };

        /**
         * Checking, whether a specified operation contains change tracking information.
         *
         * @param {Object} operation
         *  The operation to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified operation contains change tracking information.
         */
        this.isChangeTrackOperation = function (operation) {
            return (operation && operation.attrs && operation.attrs.changes &&
                    ((operation.attrs.changes.inserted && operation.attrs.changes.inserted.author) ||
                     (operation.attrs.changes.removed && operation.attrs.changes.removed.author) ||
                     (operation.attrs.changes.modified && operation.attrs.changes.modified.author)));
        };

        /**
         * Saving the state of the change track side bar handler. This is used for
         * performance reasons. Registering handler is only necessary, if they are not
         * already registered. The same for de-registering.
         *
         * @param {Boolean} state
         *  The state of the change track side bar handler.
         */
        this.setSideBarHandlerActive = function (state) {
            sideBarHandlerActive = state;
        };

        /**
         * Returning information about the state of the change track side bar handler.
         *
         * @returns {Boolean}
         *  Whether the handler for the sidebar are activated or not.
         */
        this.isSideBarHandlerActive = function () {
            return sideBarHandlerActive;
        };

        /**
         * Resolving the change tracking for all text spans or only for one
         * selected text span.
         *
         * @param {HTMLElement|jQuery} [trackingNodes]
         *  Optional html element that will be evaluated for change tracking.
         *  If specified, the selection or the parameter 'all' are ignored.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the preparation of the span.
         *  The following options are supported:
         *  @param {String} [options.type='']
         *      The type of resolving ('inserted', 'removed' or 'modified').
         *  @param {Boolean} [options.accepted=true]
         *      If set to true, the change tracking will be resolved as 'accepted'.
         *      Otherwise it will be resolved as 'rejected'.
         *  @param {Boolean} [options.all=false]
         *      If set to true, all text span with change tracking attribute
         *      will be resolved. Otherwise only the currently selected text
         *      span.
         */
        this.resolveChangeTracking = function (trackingNodes, options) {

            var // the operations generator
                generator = model.createOperationsGenerator(),
                // whether all change tracks shall be accepted or rejected
                all = Utils.getBooleanOption(options, 'all', false),
                // saving the start position of an existing range
                savedStartPosition = null,
                // whether a delete operation was used for resolving change tracking
                contentDeleted = false,
                // whether the change shall be accepted or rejected
                accepted = Utils.getBooleanOption(options, 'accepted', true),
                // the type of resolving ('inserted', 'removed' or 'modified')
                resolveType = Utils.getStringOption(options, 'type', ''),
                // a collector for all tracking nodes
                trackingNodesCollector = null,
                // a deferred object for applying the actions
                operationsDef = null,
                // error during resolving change tracks
                resolveError = false,
                // a collector for logical positions of additionally generated delete operations
                additionalDeleteOps = [],
                // a collector for all header or footer nodes
                allContainers = $(),
                // container for all generated operations
                allOperations = null;

            // restoring the old character attribute values
            function receiveOldAttributes(currentAttrs, oldAttrs) {

                var // using old attributes, if available
                    attrs = {},
                    // the attributes, that are currently defined
                    key = null;

                currentAttrs = currentAttrs || {};

                attrs =  _.copy(oldAttrs, true);  // using all old attributes

                // and setting all new attributes, that are not defined in the old attributes, to null.
                for (key in currentAttrs) {
                    if (!(key in oldAttrs)) {
                        attrs[key] = null;
                    }
                }

                return attrs;
            }

            // checking the validity of an existing changeTrack selection
            if (changeTrackSelection && changeTrackSelection.selectedNodes) { checkSelectionValidity(changeTrackSelection); }

            // setting tracking, that will be resolved
            if (changeTrackSelection && changeTrackSelection.selectedNodes && !all) {
                // using selected nodes and resolve type from current change track selection
                trackingNodes = changeTrackSelection.selectedNodes;
                resolveType = changeTrackSelection.activeNodeType;
            } else if (trackingNodes && !all) {
                trackingNodes = $(trackingNodes);
            } else {
                if (all) {
                    // collecting all change track nodes in the document (using from change tracking selection, if available)
                    trackingNodes = (changeTrackSelection && changeTrackSelection.allChangeTrackNodes) ? changeTrackSelection.allChangeTrackNodes : getAllChangeTrackNodes();
                } else {
                    // using the current selection, to resolve the change tracking
                    // -> this can be a selection with or without range.
                    // -> splitting nodes, if they are not completely inside the selection and the selection has a range
                    trackingNodesCollector = getTrackingNodesInSelection({ split: model.getSelection().hasRange(), markLastParagraph: true });

                    // no change tracking nodes were found in the selection
                    if (trackingNodesCollector.length === 0) {
                        return;
                    }

                    // change tracking nodes were found in the selection
                    if (model.getSelection().hasRange()) {

                        // simply using the collected (and already split) nodes
                        // from the selection with range.
                        trackingNodes = trackingNodesCollector;

                        if (model.getSelection().isDrawingFrameSelection()) {
                            // special handling for drawing selections, where the popup box is displayed
                            // and not all change tracks shall be resolved but only the specified type
                            if (!resolveType) { resolveType = self.getRelevantTrackingTypeByNode(trackingNodesCollector); }

                        } else {

                            // after determining the affected nodes, the process of 'all' can be used
                            // to resolve the change tracks. This is not limited to the relevant
                            // resolve type, but includes all resolve types. So inside a given arbitrary
                            // selection, all affected nodes (after grouping maybe even more) need to
                            // be resolved with all resolve types.
                            all = true;
                        }

                    } else {
                        // if resolve type is not specified, the relevant type need to be defined
                        if (!resolveType) {
                            resolveType = self.getRelevantTrackingTypeByNode(trackingNodesCollector);
                        }
                        // -> finding all relevant tracking nodes (keeping each node unique)
                        if (resolveType) {
                            trackingNodes = groupTrackingNodes(trackingNodesCollector, resolveType);
                        }
                    }
                }
            }

            if (trackingNodes && trackingNodes.length > 0) {

                //  check if special complex fields are changetracked, and restore them to original state
                model.trigger('specialFieldsCT:before', trackingNodes);

                // iterating over all tracking nodes
                _.each(trackingNodes, function (trackingNode, nodeIndex) {

                    var // whether the node is located inside header or footer
                        isMarginalNode = DOM.isMarginalNode(trackingNode),
                        // the local root node for the tracking node
                        localRootNode = isMarginalNode ? DOM.getMarginalTargetNode(rootNode, trackingNode) : rootNode,
                        // the logical start and end position of the currently evaluated node (must always be inside one paragraph)
                        changeTrackingRange = Position.getNodeStartAndEndPosition(trackingNode, localRootNode, { fullLogicalPosition: true }),
                        // a neighbor node of the tracking node
                        neighborNode = null,
                        // the following node of the tracking node
                        followingNode = null,
                        // the explicit node attributes
                        explicitAttributes = null,
                        // the original attributes
                        oldCharAttrs = null, oldParaAttrs = null, oldCellAttrs = null, oldRowAttrs = null, oldTableAttrs = null, oldStyleId = null, oldDrawingAttrs = null,
                        // the old attributes to be restored
                        restoredAttrs = null,
                        // whether the node is a text span, a text component (tab, hard break, ...) or a drawing
                        isTextSpan = DOM.isTextSpan(trackingNode) || DOM.isTextComponentNode(trackingNode) || DrawingFrame.isDrawingFrame(trackingNode),
                        // wheter operation creation would lead to an error (empty text spans cannot be addressed, but should not occur here)
                        skipNode = isTextSpan && DOM.isEmptySpan(trackingNode),
                        // whether the node is a paragraph
                        isParagraph = DOM.isParagraphNode(trackingNode),
                        // whether the node is a table
                        isTable = DOM.isTableNode(trackingNode),
                        // whether the node is a table row
                        isTableRow = DOM.isTableRowNode(trackingNode),
                        // whether the node is a table cell
                        isTableCell = DOM.isTableCellNode(trackingNode),
                        // helper variables for column handling
                        tableNode = null, tableGrid = null, gridRange = null, tablePos = null,
                        // whether this is the last node of the trackingNodes collector
                        isLastNode = (nodeIndex === trackingNodes.length - 1),
                        // whether a page break needs to be removed from table
                        removePageBreak = false,
                        // the operations generator for one node (increasing resiliance)
                        localGenerator = model.createOperationsGenerator(),
                        // all operations created for one node
                        oneNodeOperations,
                        // the header or footer container node
                        containerNode = isMarginalNode ? localRootNode : null,
                        // the header or footer target id (if not set, must be 'undefined' for later comparisons)
                        targetId = isMarginalNode ? DOM.getTargetContainerId(Utils.getDomNode(containerNode)) : undefined;

                    // checking for an error in determining the logical positions
                    if (!_.isArray(changeTrackingRange.start) || changeTrackingRange.start.length === 0) { resolveError = true; }

                    // increasing stability, avoiding invalid operations (going to next node)
                    if (resolveError) { return; }

                    // saving start position of the first node
                    savedStartPosition = savedStartPosition || _.clone(changeTrackingRange.start);

                    // increasing stability, avoiding invalid operations (going to next node)
                    if (skipNode) { return; }

                    // creating operation(s) required to accept or reject the change track
                    if ((DOM.isChangeTrackInsertNode(trackingNode)) && ((resolveType === 'inserted') || all)) {
                        if (accepted) {
                            localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { inserted: null } } });
                        } else {
                            if (isTextSpan) {
                                localGenerator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start, end: changeTrackingRange.end });
                                // the 'inserted' is no longer valid at a completely removed node
                                // Info: The order is reversed -> 1. step: Setting attributes, 2. step: Removing all content
                                // Setting attributes, before deleting node. This is (only) necessary if an empty text node remains (for example last node in paragraph)
                                localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { inserted: null } } });
                                contentDeleted = true;
                            } else if (isParagraph) {
                                neighborNode = DOM.getAllowedNeighboringNode(trackingNode, { next: false });
                                followingNode = DOM.getAllowedNeighboringNode(trackingNode, { next: true });
                                // checking, if a following table contains page break attribute
                                if (DOM.isTableNode(followingNode) && DOM.isTableWithManualPageBreak(followingNode) && !isRemovalNode(followingNode, trackingNodes, accepted)) { removePageBreak = true; }

                                // merging the paragraph with its previous sibling, if available
                                if (neighborNode && DOM.isParagraphNode(neighborNode)) {
                                    localGenerator.generateOperation(Operations.PARA_MERGE, { start: Position.increaseLastIndex(changeTrackingRange.start, -1) });
                                    contentDeleted = true;
                                } else if (!neighborNode || DOM.isTableNode(neighborNode)) {
                                    // the paragraph is marked with 'inserted' (maybe even between two tables)
                                    // -> NOT moving text into previous or next table (like MS Office)
                                    // -> delete paragraph, if it is empty or contains only content marked as inserted
                                    // -> merge tables, if they are directly following
                                    if ((Position.getParagraphNodeLength(trackingNode) === 0) || (paragraphEmptyAfterResolving(trackingNode, trackingNodes, accepted))) {
                                        // taking care of operation order: Later reverting of operations -> first delete paragraph,
                                        // then merge tables (if possible). Additionally both tables must still exist, when
                                        // the mergeTable operation is executed (43803).
                                        if (DOM.isTableNode(followingNode) && Table.mergeableTables(neighborNode, followingNode) && !nodeRemovedDuringResolving(neighborNode, trackingNodes, accepted) && !nodeRemovedDuringResolving(followingNode, trackingNodes, accepted)) {
                                            localGenerator.generateOperation(Operations.TABLE_MERGE, { start: Position.getOxoPosition(localRootNode, neighborNode) });
                                        }
                                        // deleting an empty paragraph (or a paragraph containing content completely marked as 'inserted'
                                        localGenerator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                        contentDeleted = true;
                                    }
                                }
                                // removing page break attribute from following table, if required (calculating position before deleting table)
                                if (removePageBreak) {
                                    localGenerator.generateOperation(Operations.SET_ATTRIBUTES, {
                                        attrs: { paragraph: { pageBreakBefore: false } },
                                        start: Position.getFirstPositionInParagraph(localRootNode, Position.getOxoPosition(localRootNode, followingNode)).slice(0, -1)
                                    });
                                }
                                // removing change track attribute from paragraph
                                localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { inserted: null } } });
                            } else if (isTableCell) {
                                // rejecting for all cells in the column
                                localGenerator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                if (isLastNode) {
                                    // Setting new table grid attribute to table
                                    tableNode = $(trackingNode).closest(DOM.TABLE_NODE_SELECTOR);
                                    tableGrid = _.clone(model.getStyleCollection('table').getElementAttributes(tableNode).table.tableGrid);
                                    gridRange = Table.getGridColumnRangeOfCell(trackingNode);
                                    tablePos = Position.getOxoPosition(localRootNode, tableNode, 0);
                                    tableGrid.splice(gridRange.start, gridRange.end - gridRange.start + 1);  // removing column(s) in tableGrid (automatically updated in table node)
                                    localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) });
                                    contentDeleted = true;
                                }
                            } else if (isTable || isTableRow) {

                                // deleting of table needs to be checked and generated before deleting of rows
                                // -> handling the problem, that after reload, table no longer is marked as 'inserted'
                                if (isTableRow && ($(trackingNode).prev().length === 0)) {
                                    tableNode = $(trackingNode).closest(DOM.TABLE_NODE_SELECTOR);
                                    if (tableNode && !DOM.isChangeTrackInsertNode(tableNode)) {
                                        // is there a next sibling, that is not marked as inserted?
                                        // -> comparing the number of next inserted siblings + 1 with the number of rows in the table
                                        if (((($(trackingNode).nextAll(DOM.CHANGETRACK_INSERTED_ROW_SELECTOR)).length) + 1) === DOM.getTableRows(tableNode).length) {
                                            tablePos = Position.getOxoPosition(localRootNode, tableNode, 0);
                                            additionalDeleteOps.push({ pos: tablePos, targetId: targetId }); // Checking later, if the table already got a setAttributes operation (36652)
                                            localGenerator.generateOperation(Operations.DELETE, { start: tablePos });
                                        }
                                    }
                                }

                                localGenerator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                contentDeleted = true;
                            }
                        }
                    }

                    if ((DOM.isChangeTrackRemoveNode(trackingNode)) && ((resolveType === 'removed') || all)) {
                        if (accepted) {
                            if (isTextSpan) {
                                localGenerator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start, end: changeTrackingRange.end });
                                contentDeleted = true;
                            } else if (isParagraph) {
                                neighborNode = DOM.getAllowedNeighboringNode(trackingNode, { next: false });
                                followingNode = DOM.getAllowedNeighboringNode(trackingNode, { next: true });
                                // checking, if the following table contains page break attribute
                                if (DOM.isTableNode(followingNode) && DOM.isTableWithManualPageBreak(followingNode) && !isRemovalNode(followingNode, trackingNodes, accepted)) { removePageBreak = true; }

                                // merging the paragraph with its next sibling, if available
                                if (neighborNode && DOM.isParagraphNode(neighborNode)) {
                                    // merging previous paragraph with marked paragraph
                                    localGenerator.generateOperation(Operations.PARA_MERGE, { start: Position.increaseLastIndex(changeTrackingRange.start, -1) });
                                    contentDeleted = true;
                                } else if (!neighborNode || DOM.isTableNode(neighborNode)) {
                                    // the paragraph is marked with 'deleted' (maybe even between two tables)
                                    // -> delete paragraph, if it is empty or contains only content marked as inserted
                                    // -> merge tables, if they are directly following
                                    if ((Position.getParagraphNodeLength(trackingNode) === 0) || (paragraphEmptyAfterResolving(trackingNode, trackingNodes, accepted))) {
                                        // taking care of operation order: Later reverting of operations -> first delete paragraph,
                                        // then merge tables (if possible). Additionally both tables must still exist, when
                                        // the mergeTable operation is executed (43803).
                                        if (DOM.isTableNode(followingNode) && Table.mergeableTables(neighborNode, followingNode) && !nodeRemovedDuringResolving(neighborNode, trackingNodes, accepted) && !nodeRemovedDuringResolving(followingNode, trackingNodes, accepted)) {
                                            // Checking, if the following table is also change tracked(?)
                                            localGenerator.generateOperation(Operations.TABLE_MERGE, { start: Position.getOxoPosition(localRootNode, neighborNode) });
                                        }
                                        // deleting an empty paragraph (or a paragraph containing content completely marked as 'inserted'
                                        localGenerator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                        contentDeleted = true;
                                    }
                                }
                                // removing page break attribute from second table, if required (calculating position before deleting table)
                                if (removePageBreak) {
                                    localGenerator.generateOperation(Operations.SET_ATTRIBUTES, {
                                        attrs: { paragraph: { pageBreakBefore: false } },
                                        start: Position.getFirstPositionInParagraph(localRootNode, Position.getOxoPosition(localRootNode, followingNode)).slice(0, -1)
                                    });
                                }
                                // removing change track attribute from paragraph
                                localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { removed: null } } });
                            } else if (isTableCell) {
                                // accepting for all cells in the column (only one cell determines complete column)
                                localGenerator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                // Setting new table grid attribute to table
                                if (isLastNode) {
                                    tableNode = $(trackingNode).closest(DOM.TABLE_NODE_SELECTOR);
                                    tableGrid = _.clone(model.getStyleCollection('table').getElementAttributes(tableNode).table.tableGrid);
                                    gridRange = Table.getGridColumnRangeOfCell(trackingNode);
                                    tablePos = Position.getOxoPosition(localRootNode, tableNode, 0);
                                    tableGrid.splice(gridRange.start, gridRange.end - gridRange.start + 1);  // removing column(s) in tableGrid (automatically updated in table node)
                                    localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) });
                                    contentDeleted = true;
                                }
                            } else if (isTable || isTableRow) {

                                // deleting of table needs to be checked and generated before deleting of rows
                                // -> handling the problem, that after reload, table no longer is marked as 'removed'
                                if (isTableRow && ($(trackingNode).prev().length === 0)) {
                                    tableNode = $(trackingNode).closest(DOM.TABLE_NODE_SELECTOR);
                                    if (tableNode && !DOM.isChangeTrackRemoveNode(tableNode)) {
                                        // is there a next sibling, that is not marked as removed?
                                        // -> comparing the number of next removed siblings + 1 with the number of rows in the table
                                        if (((($(trackingNode).nextAll(DOM.CHANGETRACK_REMOVED_ROW_SELECTOR)).length) + 1) === DOM.getTableRows(tableNode).length) {
                                            tablePos = Position.getOxoPosition(localRootNode, tableNode, 0);
                                            additionalDeleteOps.push({ pos: tablePos, targetId: targetId }); // Checking later, if the table already got a setAttributes operation
                                            localGenerator.generateOperation(Operations.DELETE, { start: tablePos });
                                        }
                                    }
                                }

                                localGenerator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                contentDeleted = true;
                            }
                        } else {
                            localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { removed: null } } });
                        }
                    }

                    if ((DOM.isChangeTrackModifyNode(trackingNode)) && ((resolveType === 'modified') || (all && !$(trackingNode).data('skipModify')))) {

                        if (accepted) {
                            localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { modified: null } } });
                        } else {
                            // Restoring the old attribute state
                            explicitAttributes = AttributeUtils.getExplicitAttributes(trackingNode);

                            // comparing 'character' with 'changes.modified.attrs.character' and 'paragraph' with 'changes.modified.attrs.paragraph'
                            oldCharAttrs = receiveOldAttributes(explicitAttributes.character, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.character) || {});
                            oldParaAttrs = receiveOldAttributes(explicitAttributes.paragraph, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.paragraph) || {});
                            oldCellAttrs = receiveOldAttributes(explicitAttributes.cell, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.cell) || {});
                            oldRowAttrs = receiveOldAttributes(explicitAttributes.row, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.row) || {});
                            oldTableAttrs = receiveOldAttributes(explicitAttributes.table, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.table) || {});
                            oldDrawingAttrs = receiveOldAttributes(explicitAttributes.drawing, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.drawing) || {});
                            oldStyleId = (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.styleId) || null;
                            restoredAttrs = { character: oldCharAttrs, paragraph: oldParaAttrs, cell: oldCellAttrs, row: oldRowAttrs, table: oldTableAttrs, drawing: oldDrawingAttrs, changes: { modified: null } };
                            if (oldStyleId) {
                                restoredAttrs.styleId = oldStyleId;
                            } else if (explicitAttributes.styleId) {
                                restoredAttrs.styleId = null;
                            }
                            localGenerator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: restoredAttrs });
                        }
                    }

                    // receiving the new operations for this one node (increasing resiliance)
                    oneNodeOperations = localGenerator.getOperations();

                    // adding target id to operations, if necessary
                    if (isMarginalNode) {
                        _.each(oneNodeOperations, function (operation) {
                            operation.target = targetId;
                        });
                        allContainers = allContainers.add(Utils.getDomNode(containerNode)); // jQuery keeps the collection unique
                    }

                    // checking the new operations for this one node (increasing resiliance)
                    // (alternative solution: special operations generator for delete operations)
                    if (all) { checkOperationsIntegrity(oneNodeOperations); }

                    // adding operations to the global generator
                    generator.appendOperations(oneNodeOperations);

                    // removing data attribute 'skipModify'
                    if ($(trackingNode).data('skipModify')) { $(trackingNode).removeData('skipModify'); }

                });

                // stopping iteration, if error occured -> no operation must be sent
                if (resolveError) {
                    self.clearChangeTrackSelection();
                    return;
                }

                // getting all created operations from the generator
                allOperations = generator.getOperations();

                // checking if additional delete operations were generated (for foreign elements (36652))
                if (additionalDeleteOps) { checkOperationsIntegrity(allOperations, additionalDeleteOps); }

                // operations MUST be executed in reverse order to preserve the positions
                // -> from end of document to beginning of document
                allOperations.reverse();

                // invalidate an existing change track selection
                // -> clearing selection before (!) applying operations, so that
                // the highlighting attributes are not handled inside the operations
                if (changeTrackSelection) { self.clearChangeTrackSelection(); }

                // applying the operations
                model.setGUITriggeredOperation(true);

                // blocking keyboard input during applying of operations
                model.setBlockKeyboardEvent(true);

                // fire apply operations asynchronously
                operationsDef = model.applyOperations(allOperations, { async: true });

                // Wait for a second before showing the progress bar (entering busy mode).
                // Users applying minimal or 'normal' amount of change tracks will not see this progress bar at all.
                app.getView().enterBusy({
                    cancelHandler: function () {
                        if (operationsDef) { operationsDef.abort(); }
                    },
                    delay: 1000,
                    warningLabel: /*#. shown while all recorded changes will be applied to the document */ gt('Applying changes, please wait...')
                });

                // handle the result of change track operations
                operationsDef
                    .progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(progress);
                    })
                    .done(function () {
                        // updating all headers or footers synchronously (required for 'next' or 'prev')
                        if (allContainers.length > 0) { model.trigger('paragraphUpdate:after', allContainers); }
                    })
                    .always(function () {
                        app.getView().leaveBusy();
                        model.setGUITriggeredOperation(false);
                        // allowing keyboard events again
                        model.setBlockKeyboardEvent(false);
                        // after change track is finnished, convert page fields in header to special ones, if they were among change track nodes
                        model.trigger('specialFieldsCT:after');
                    });

                // if content was deleted and there was a range, the new selection must be a text cursor at start position
                if (savedStartPosition && contentDeleted) {
                    model.getSelection().setTextSelection(Position.getFirstPositionInParagraph(rootNode, savedStartPosition));
                }
            }
        };

        /**
         * Updating the change track attributes for a given node.
         *
         * @param {jQuery} node
         *  The node whose character attributes shall be updated, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        this.updateChangeTrackAttributes = function (node, mergedAttributes) {

            var // the explicit attributes of the specified node
                nodeAttrs = null,
                // the place holder node from an absolute positioned drawing
                placeHolderNode = null;

            // helper function to remove all change track attributes
            // from a specified node
            function removeAllChangeTrackAttributesFromNode(oneNode) {
                oneNode.removeAttr('data-change-track-modified');
                oneNode.removeAttr('data-change-track-inserted');
                oneNode.removeAttr('data-change-track-removed');
                oneNode.removeAttr('data-change-track-modified-author');
                oneNode.removeAttr('data-change-track-inserted-author');
                oneNode.removeAttr('data-change-track-removed-author');
            }

            // helper function to copy all change track attributes from
            // one node to another node
            function copyChangeTrackAttributes(from, to) {
                if (from.attr('data-change-track-modified')) { to.attr('data-change-track-modified', from.attr('data-change-track-modified')); }
                if (from.attr('data-change-track-inserted')) { to.attr('data-change-track-inserted', from.attr('data-change-track-inserted')); }
                if (from.attr('data-change-track-removed')) { to.attr('data-change-track-removed', from.attr('data-change-track-removed')); }
                if (from.attr('data-change-track-modified-author')) { to.attr('data-change-track-modified-author', from.attr('data-change-track-modified-author')); }
                if (from.attr('data-change-track-inserted-author')) { to.attr('data-change-track-inserted-author', from.attr('data-change-track-inserted-author')); }
                if (from.attr('data-change-track-removed-author')) { to.attr('data-change-track-removed-author', from.attr('data-change-track-removed-author')); }
            }

            // Setting and deleting changes attributes, but without using attribute from parent!
            if (_.isObject(mergedAttributes.changes)) {

                nodeAttrs = AttributeUtils.getExplicitAttributes(node); // -> no inheritance

                if (_.isObject(nodeAttrs.changes) && (!DOM.isEmptySpan(node) || DOM.isTabSpan(node))) {  // not for empty text spans

                    if (_.isObject(nodeAttrs.changes.modified)) {
                        node.attr('data-change-track-modified', true);  // necessary for visualization of modifications
                        if (nodeAttrs.changes.modified.author) {
                            node.attr('data-change-track-modified-author', app.getAuthorColorIndex(nodeAttrs.changes.modified.author));
                        }
                    } else {
                        node.removeAttr('data-change-track-modified');
                        node.removeAttr('data-change-track-modified-author');
                    }

                    if (_.isObject(nodeAttrs.changes.inserted)) {
                        node.attr('data-change-track-inserted', true);  // necessary for visualization of modifications
                        if (nodeAttrs.changes.inserted.author) {
                            node.attr('data-change-track-inserted-author', app.getAuthorColorIndex(nodeAttrs.changes.inserted.author));
                        }
                    } else {
                        node.removeAttr('data-change-track-inserted');
                        node.removeAttr('data-change-track-inserted-author');
                    }

                    if (_.isObject(nodeAttrs.changes.removed)) {
                        node.attr('data-change-track-removed', true);  // necessary for visualization of modifications
                        if (nodeAttrs.changes.removed.author) {
                            node.attr('data-change-track-removed-author', app.getAuthorColorIndex(nodeAttrs.changes.removed.author));
                        }
                    } else {
                        node.removeAttr('data-change-track-removed');
                        node.removeAttr('data-change-track-removed-author');
                    }
                } else {
                    removeAllChangeTrackAttributesFromNode(node);
                }

            } else {
                removeAllChangeTrackAttributesFromNode(node);
            }

            // setting change track attributes at absolute positioned drawings AND
            // at their place holder nodes. This is important, to keep the correct
            // order of change tracked attributes.
            if (DOM.isDrawingFrame(node) && DOM.isInsideDrawingLayerNode(node)) {
                placeHolderNode = $(DOM.getDrawingPlaceHolderNode(node));
                if (placeHolderNode) {
                    removeAllChangeTrackAttributesFromNode(placeHolderNode);
                    copyChangeTrackAttributes(node, placeHolderNode);
                }
            }

            if (DOM.isParagraphNode(node)) {
                var paraAttributes = model.getParagraphStyles().getElementAttributes(node).paragraph;
                if (paraAttributes.listStyleId !== '') {
                    // Update list styles numbering
                    model.updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: paraAttributes.listStyleId, paraInsert: true });
                }
            }
        };

        /**
         * Receiving the existing explicit attributes for a given node.
         * They are returned for a change track modification operation.
         *
         * @param {jQuery} node
         *  The node whose character attributes have been changed, as
         *  jQuery object.
         *
         * @returns {Object}
         *  An object with all existing explicit node attributes, taking
         *  special care of the change tracking attributes.
         */
        this.getOldNodeAttributes = function (node) {

            var // an object with the old node attributes
                oldAttrs = {};

            // Expanding operation for change tracking with old explicit attributes
            oldAttrs.attrs = AttributeUtils.getExplicitAttributes(node);

            // special handling for multiple modifications
            if (_.isObject(oldAttrs.attrs.changes) && _.isObject(oldAttrs.attrs.changes.modified) && _.isObject(oldAttrs.attrs.changes.modified.attrs)) {
                // Never overwriting existing changes.modified.attrs, because there might be multiple attribute modifications
                oldAttrs = oldAttrs.attrs.changes.modified;  // -> reusing the already registered old attributes
            }

            // old attributes must not contain the information about the changes
            if (oldAttrs && _.isObject(oldAttrs.attrs.changes)) {
                delete oldAttrs.attrs.changes;
            }

            return oldAttrs;
        };

        /**
         * Setting the change track selection, for example after pressing
         * the 'Next' or 'Previous' button in the change track GUI.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the change track selection.
         *  The following options are supported:
         *  @param {Boolean} [options.next=true]
         *      Whether the following (next is true) or the previous change track
         *      node shall be selected.
         *  @param {Boolean} [options.reuseSameNode=false]
         *      Whether the same selected node can be reused. This is useful, if
         *      the same node contains a further change tracked modification.
         */
        this.getNextChangeTrack = function (options) {

            var // the next valid selection
                nextSelection = iterateNextValidChangeTrackSelection(options);

            if (nextSelection) {
                setChangeTrackSelection(nextSelection);  // setting CT selection and making it visible
            } else {
                app.getView().yell({ type: 'info', message: gt('There are no tracked changes in your document.') });
            }

        };

        /**
         * Returning the change track selection.
         *
         * @returns {Object|Null}
         *  An object describing the change track selection or Null, if no change track selection
         *  exists.
         */
        this.getChangeTrackSelection = function () {
            return changeTrackSelection;
        };

        /**
         * Returns whether the current selection contains at least one change track element.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one change track element.
         */
        this.isOnChangeTrackNode = function () {
            // if there were no changeTracked elements, get out here (for performance)
            if (getAllChangeTrackNodes(rootNode).length === 0) { return false; }

            // if there were changeTracked elements and the hole document is selected, get out here (for performance)
            if (model.getSelection().isAllSelected() && getAllChangeTrackNodes(rootNode).length > 0) { return true; }

            var // receiving at least one change track node from the current selection
                changeTrackNodes = getTrackingNodesInSelection({ minimalSelection: true });

            return (changeTrackNodes && changeTrackNodes.length > 0);
        };

        /**
         * Setting, removing or updating the side bar and the markers for the change tracking.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.invalidate=true]
         *      Whether an existing side bar needs to be invalidated.
         *
         * @returns {Boolean}
         *  Whether there is at least one active change track node in the document.
         */
        this.updateSideBar = function (options) {

            var // the first child of the page content node
                firstPageContentNodeChild = null,
                // the top offset of the page content node
                pageContentNodeTopOffset = 0,
                // the top margin of the first child of the content node
                firstChildTopOffset = 0,
                // the minimum and maximum offset, until which the change track divs are created
                minOffset = 0, maxOffset = 0,
                // whether the current marker in the side pane need to be invalidated
                invalidate = Utils.getBooleanOption(options, 'invalidate', true),
                // the current zoom factor of the document
                zoomFactor = 0;

            // creating one marker in the change track side bar
            // for a specified change track node
            function createSideBarMarker(node) {

                var // the vertical offset of the specific node
                    topOffset = Utils.round($(node).offset().top, 1),
                    // the height and offset of the change track marker
                    markerHeight = 0, markerOffset = 0,
                    // a key for each marker
                    markerKey = null,
                    // whether the node is a table cell
                    isCellNode = DOM.isTableCellNode(node),
                    // the target node of a marginal node
                    targetNode = null,
                    // the maximum vertical offset inside a target node
                    maxTargetOffset = 0;

                // Performance: Only insert divs, if the value for offset().top is '> minOffset' and
                // smaller than the height of the scrollNode multiplied with 1.5

                if (topOffset > maxOffset) { return Utils.BREAK; }

                if (topOffset > minOffset) {

                    markerHeight = $(node).is('div.p') ? Utils.convertCssLength($(node.firstChild).css('line-height'), 'px', 1) : (isCellNode ? $(node).outerHeight() : $(node).height());
                    markerOffset = Utils.round(((topOffset + firstChildTopOffset - pageContentNodeTopOffset) / zoomFactor), 1);

                    // handling marginal nodes (39323)
                    if (DOM.isMarginalNode(node)) {

                        targetNode = DOM.getClosestMarginalTargetNode(node);

                        if (targetNode.length > 0) {
                            maxTargetOffset = Utils.round(((targetNode.offset().top + firstChildTopOffset - pageContentNodeTopOffset) / zoomFactor), 1);
                            maxTargetOffset += targetNode.innerHeight();

                            if (markerOffset + markerHeight > maxTargetOffset) {
                                markerHeight -= markerOffset + markerHeight - maxTargetOffset;
                                if (markerHeight < 0) { markerHeight = 0; }
                            }
                        }
                    }

                    markerKey = markerOffset + '_' + markerHeight;

                    // finally inserting the marker for the change tracked element, if it is not already inserted
                    if (!allSideBarMarker[markerKey]) {
                        sideBarNode.append($('<div>').addClass('ctdiv').height(markerHeight).css('top', markerOffset));
                        allSideBarMarker[markerKey] = 1;
                    }
                }

            }

            // not supporting small devices
            if (Utils.SMALL_DEVICE) { return false; }

            // setting scroll node, if not already done
            if (!scrollNode) { scrollNode = app.getView().getContentRootNode(); }

            // setting page content node, if not already done
            if (!pageContentNode) { pageContentNode = rootNode.children(DOM.PAGECONTENT_NODE_SELECTOR); }

            // invalidating an existing side bar
            if (sideBarNode && invalidate) {
                sideBarNode.remove();
                sideBarNode = null;
                allSideBarMarker = {};
            }

            // collecting all nodes with change track information (not necessary after scrolling)
            if (!allNodesSideBarCollector || invalidate) {
                allNodesSideBarCollector = getAllChangeTrackNodes(rootNode, { multiple: true }); // Very fast call, but avoid if possible
            }

            // drawing the side bar, if there is at least one change track
            if (allNodesSideBarCollector.length > 0) {

                // setting the zoom factor
                zoomFactor = app.getView().getZoomFactor() / 100;

                // the top offset of the content node
                pageContentNodeTopOffset = Utils.round(pageContentNode.offset().top, 1);

                // if the first paragraph in the document has a top margin, this also needs to be added
                // into the calculation. Otherwise all markers are shifted by this top margin.
                firstPageContentNodeChild = Utils.getDomNode(pageContentNode).firstChild;
                if (firstPageContentNodeChild && $(firstPageContentNodeChild).css('margin-top')) {
                    firstChildTopOffset = Utils.round(Utils.getElementCssLength(firstPageContentNodeChild, 'margin-top') * zoomFactor, 1);
                }

                // calculating the minimum and maximum offset in that region, for which the
                // side bar marker will be created (for performance reasons)
                maxOffset = 1.5 * (scrollNode.height() + scrollNode.offset().top);  // one and a half screen downwards
                minOffset = -0.3 * maxOffset; // a half screen upwards

                // creating and appending side bar, if it does not exist,
                // and if there is at least one change track
                if (sideBarNode === null) {
                    sideBarNode = $('<div>').addClass('ctsidebar')
                        .css('top',  Utils.getElementCssLength(rootNode, 'padding-top') + model.getPageLayout().getHeightOfFirstHeaderWrapper()) // if header exists, move ctsidebar down for the value of height
                        .css('left', Utils.round(Utils.getElementCssLength(rootNode, 'padding-left') / 2, 1))
                        .height(pageContentNode.height());

                    rootNode.append(sideBarNode);
                }

                // iterating over all collected change tracked nodes
                Utils.iterateArray(allNodesSideBarCollector, function (ctNode) {

                    if (DOM.isTableWithPageBreakRowNode(ctNode)) {
                        // do not mark complete table, because of problem with row inserted for
                        // page break -> mark all rows except the page break row
                        _.each(DOM.getTableRows(ctNode), createSideBarMarker);
                    } else {
                        createSideBarMarker(ctNode);
                    }

                });

            }

            // if there are no change tracks, inform listeners about this
            if (allNodesSideBarCollector.length === 0) {
                model.trigger('changeTrack:stateInfo', { state: false });
            }

            return allNodesSideBarCollector.length > 0;
        };

        /**
         * Sets the highlight marker to the current selected change track group.
         *  But first, it removes all existing highlights.
         *  Only used for ios-devices.
         */
        this.setHighlightToCurrentChangeTrack = function () {
            var node = findChangeTrackNodeAtCursorPosition();

            removeHighlighting();
            node.addClass(highlightClassname);
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {

            // updating the model after the document was reloaded (after cancelling long running actions)
            self.listenTo(model, 'document:reloaded', handleDocumentReload);

        });

    } // class ChangeTrack

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: ChangeTrack });

});
