/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/model/model', [
    'io.ox/office/textframework/model/editor',
    'io.ox/office/text/model/modelattributesmixin',
    'io.ox/office/text/format/listcollection',
    'io.ox/office/text/format/stylesheetmixin',
    'io.ox/office/text/model/pagehandlermixin',
    'io.ox/office/text/model/updatedocumentmixin',
    'io.ox/office/text/model/listhandlermixin',
    'io.ox/office/text/model/updatelistsmixin',
    'io.ox/office/text/components/drawing/drawingresize',
    'io.ox/office/textframework/components/field/fieldmanager',
    'io.ox/office/textframework/model/numberformatter',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/operations'
], function (Editor, ModelAttributesMixin, ListCollection, StylesheetMixin, PageHandlerMixin, UpdateDocumentMixin, ListHandlerMixin, UpdateListsMixin, DrawingResize, FieldManager, NumberFormatter, Utils, Position, Operations) {

    'use strict';

    // class TextModel ================================================

    /**
     * Represents the document model of a text application.
     *
     * @constructor
     *
     * @extends Editor
     * @extends ModelAttributesMixin
     * @extends StylesheetMixin
     * @extends PageHandlerMixin
     * @extends UpdateDocumentMixin
     * @extends ListHandlerMixin
     * @extends UpdateListsMixin
     *
     * @param {TextApplication} app
     *  The application containing this document model.
     */
    function TextModel(app) {

        var // self reference for local functions
            self = this,
            // the root element for the document contents
            pagediv = null;

        // base constructor ---------------------------------------------------

        Editor.call(this, app);
        ModelAttributesMixin.call(this);
        StylesheetMixin.call(this, app);
        PageHandlerMixin.call(this, app);
        UpdateDocumentMixin.call(this, app);
        ListHandlerMixin.call(this);
        UpdateListsMixin.call(this, app);

        // private methods ----------------------------------------------------

        // public methods -----------------------------------------------------

        /**
         * Getting the default text for an empty text frame drawing.
         * This function is application specific.
         * This is not implemented for OX Text yet.
         *
         * @param {HTMLElement|jQuery} drawing
         *  The drawing node.
         *
         * @param {Object} [options]
         *  Some additional options.
         *
         * @returns {String|Null}
         *  The default text for a place holder drawing. Or null, if it cannot be determined.
         */
        this.getDefaultTextForTextFrame = function (/*drawing, options*/) {
            return null;
        };

        /**
         * Handling the property 'autoResizeHeight' of a selected text frame node.
         *
         * @param {Boolean} state
         *  Whether the property 'autoResizeHeight' of a selected text frame shall be
         *  enabled or disabled.
         */
        this.handleTextFrameAutoFit = function (state) {

            var selection = self.getSelection();
            // the operations generator
            var generator = self.createOperationsGenerator();
            // the options for the setAttributes operation
            var operationOptions = {};
            // a selected text frame node or the text frame containing the selection
            var textFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true });

            // collecting the attributes for the operation
            operationOptions.attrs =  {};
            operationOptions.attrs.shape = { autoResizeHeight: state };

            // if auto fit is disabled, the current height must be set explicitely
            if (!state) {
                if (textFrame && textFrame.length > 0) {
                    operationOptions.attrs.drawing = { height: Utils.convertLengthToHmm(textFrame.height(), 'px') };
                }
            }

            if (selection.isAdditionalTextframeSelection()) {
                operationOptions.start = Position.getOxoPosition(self.getCurrentRootNode(), textFrame, 0);
            } else {
                operationOptions.start = selection.getStartPosition();
            }

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            this.applyOperations(generator);
        };

        // initialization -----------------------------------------------------

        pagediv = self.getNode();

        // the textmodel.js has an own drawing style object with own format handler -> that need to be overwritten
        self.setCharacterStyles(self.getStyleCollection('character'));
        self.setParagraphStyles(self.getStyleCollection('paragraph'));
        self.setDrawingStyles(self.getStyleCollection('drawing'));
        self.setTableStyles(self.getStyleCollection('table'));
        self.setTableRowStyles(self.getStyleCollection('row'));
        self.setTableCellStyles(self.getStyleCollection('cell'));
        self.setPageStyles(self.getStyleCollection('page'));

        // setting the handler function for moving and resizing drawings
        self.getSelection().setDrawingResizeHandler(DrawingResize.drawDrawingSelection);

        // setting the handler function for updating lists (this is used during 'updateDocumentFormatting')
        self.setUpdateLists(self.getUpdateListsHandler());

        // setting handler for handling fields in presentation
        self.setFieldManager(new FieldManager(app));

        // setting handler for handling fields in presentation
        self.setNumberFormatter(new NumberFormatter(this));

        // setting the list collection object
        self.setListCollection(new ListCollection(this));

        // setting the handler function for updating lists after the document is imported successfully
        self.waitForImportSuccess(function () {
            self.setUpdateListsDebounced(self.getDebouncedUpdateListsHandler());
        });

        // registering process mouse down handler at the page
        pagediv.on({ 'mousedown touchstart': self.getPageProcessMouseDownHandler() });
        if (self.getListenerList()) { self.getListenerList()['mousedown touchstart'] = self.getPageProcessMouseDownHandler(); }

        // complex fields
        pagediv.on('mouseenter', '.complex-field', self.getFieldManager().createHoverHighlight);
        pagediv.on('mouseleave', '.complex-field', self.getFieldManager().removeHoverHighlight);

    } // class TextModel

    // constants --------------------------------------------------------------

    // exports ================================================================

    // derive this class from class Editor
    return Editor.extend({ constructor: TextModel });

});
