/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/model/listhandlermixin', [
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position'
], function (Utils, DOM, Operations, Position) {

    'use strict';

    // mix-in class ListHandlerMixin ======================================

    /**
     * A mix-in class for the document model class providing functionality
     * for the list handling used in a text document.
     *
     * @constructor
     */
    function ListHandlerMixin() {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Private helper function that generates 'setAttributes' operations with
         * assigned listStyleId and listLevel. Setting styleId to list-paragraph
         * only if current styleId is standard.
         *
         * @param {String} listStyleId
         *  The id of the list style.
         *
         * @param {Number} listLevel
         *  The list level.
         */
        function setListStyle(listStyleId, listLevel) {

            var attrs = self.getAttributes('paragraph'),
                newAttrs = { paragraph: { listStyleId: listStyleId, listLevel: listLevel } };

            //TODO: workaround for broken paint update
            if (_.isUndefined(listLevel) && attrs.paragraph) {
                newAttrs.paragraph.listLevel = attrs.paragraph.listLevel;
            }

            // assigning list style id to paragraph, if it has the default style id (40787)
            if (attrs.styleId === self.getDefaultUIParagraphStylesheet()) {
                // checking, if the paragraph has a style id set
                newAttrs.styleId = self.getDefaultUIParagraphListStylesheet();
            }

            self.setAttributes('paragraph', newAttrs);
        }

        // public methods -----------------------------------------------------

        /**
         * Creates a default list either with decimal numbers or bullets
         *
         * @param {String} type
         *  The list type: 'numbering' or 'bullet'
         */
        this.createDefaultList = function (type) {
            var defListStyleId = self.getListCollection().getDefaultNumId(type);

            if (defListStyleId) {
                var listId = self.getListCollection().getListId(self.getListCollection().getDefaultListDefinition(type), defListStyleId);
                if (listId) {
                    defListStyleId = listId;
                } else {
                    defListStyleId = undefined;
                }
            }

            self.getUndoManager().enterUndoGroup(function () {

                if (defListStyleId === undefined) {
                    var listOperation = self.getListCollection().getDefaultListOperation(type);
                    this.applyOperations(listOperation);
                    defListStyleId = listOperation.listStyleId;
                }
                setListStyle(defListStyleId, 0);
            }, this);
        };

        /**
         * Creates a non-default list either with decimal numbers or bullets that
         * are available at the GUI.
         *
         * @param {String} listStyleId
         *  The id of the list style.
         *
         * @param {Number} listLevel
         *  The list level.
         */
        this.createSelectedListStyle = function (listStyleId, listLevel) {

            self.getUndoManager().enterUndoGroup(function () {
                var // the selection object
                    selection = self.getSelection(),
                    // the page node
                    pageNode = self.getNode(),
                    listOperation,
                    listOperationAndStyle,
                    savedSelection = _.copy(selection.getStartPosition()),
                    _start = selection.getStartPosition(),
                    currentParaPosition,
                    para,
                    paraAttributes,
                    prevParagraph, prevAttributes,
                    nextParagraph, nextAttributes,
                    target = null, // needed for odt handling (44393)
                    fullListStyleId = listStyleId; // needed for odt handling (44393)

                _start.pop();
                para = Position.getParagraphElement(pageNode, _start);
                paraAttributes = self.getParagraphStyles().getElementAttributes(para).paragraph;

                if (!selection.hasRange() && paraAttributes.listStyleId !== '') {
                    //merge level from new listlevel with used list level and apply that new list to all consecutive paragraphs using the old list
                    if (listStyleId !== paraAttributes.listStyleId) {

                        listOperationAndStyle = self.getListCollection().mergeListStyle(paraAttributes.listStyleId, listStyleId, paraAttributes.listLevel);
                        if (listOperationAndStyle.listOperation) {
                            this.applyOperations(listOperationAndStyle.listOperation);
                        }
                        prevParagraph = Utils.findPreviousNode(pageNode, para, DOM.PARAGRAPH_NODE_SELECTOR);
                        while (prevParagraph) {
                            prevAttributes = self.getParagraphStyles().getElementAttributes(prevParagraph).paragraph;
                            if (prevAttributes.listStyleId === paraAttributes.listStyleId) {
                                currentParaPosition = Position.getOxoPosition(pageNode, prevParagraph, 0);
                                currentParaPosition.push(0);
                                selection.setTextSelection(currentParaPosition);
                                setListStyle(listOperationAndStyle.listStyleId);
                                prevParagraph = Utils.findPreviousNode(pageNode, prevParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                            } else {
                                prevParagraph = null;
                            }
                        }

                        nextParagraph = Utils.findNextNode(pageNode, para, DOM.PARAGRAPH_NODE_SELECTOR);
                        while (nextParagraph) {
                            nextAttributes = self.getParagraphStyles().getElementAttributes(nextParagraph).paragraph;
                            if (nextAttributes.listStyleId === paraAttributes.listStyleId) {
                                currentParaPosition = Position.getOxoPosition(pageNode, nextParagraph, 0);
                                currentParaPosition.push(0);
                                selection.setTextSelection(currentParaPosition);
                                setListStyle(listOperationAndStyle.listStyleId);
                                nextParagraph = Utils.findNextNode(pageNode, nextParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                            } else {
                                nextParagraph = null;
                            }
                        }

                        selection.setTextSelection(savedSelection);
                        setListStyle(listOperationAndStyle.listStyleId);
                    }

                } else {
                    var predefinedStyleId = listStyleId;
                    if ((self.getApp().isODF() && self.isHeaderFooterEditState())) { // odt handling (44393)
                        target = self.getActiveTarget();
                        if (target) {
                            fullListStyleId = listStyleId + '_' + target;
                        }
                    } else {
                        var listId = self.getListCollection().getListId(self.getListCollection().getPredefinedListStyles()[listStyleId].definition, listStyleId);
                        if (listId) {
                            listStyleId = listId;
                            fullListStyleId = listStyleId;
                        } else {
                            fullListStyleId = self.getListCollection().getFreeListId();
                            listStyleId = fullListStyleId;
                        }
                    }

                    listOperation = self.getListCollection().getSelectedListStyleOperation(listStyleId, target, predefinedStyleId);

                    if (listOperation) {
                        this.applyOperations(listOperation);
                    }
                    listLevel = Math.max(0, listLevel || 0);  // setting listLevel for paragraphs, that were not in lists before

                    setListStyle(fullListStyleId, listLevel);
                }

            }, this); // enterUndoGroup
        };

        /**
         * Creates a new list after pressing 'Enter'. In this case the list auto detection found a valid structure
         * for a new list style.
         *
         * @param {String} type
         *  The list type: 'numbering' or 'bullet'
         *
         * @param {Object} options
         *  The detected options that are used to create the new list style.
         */
        this.createList = function (type, options) {

            var defListStyleId = (!options || (!options.symbol && !options.listStartValue)) ? self.getListCollection().getDefaultNumId(type) : undefined,
                // the attributes object for the operation
                allAttrs = null,
                // the old paragraph attributes
                oldAttrs = null,
                // the paragraph dom node
                paraNode = null,
                // the operation object
                newOperation = null;

            if (defListStyleId === undefined) {
                var listOperation = self.getListCollection().getDefaultListOperation(type, options);
                self.applyOperations(listOperation);
                defListStyleId = listOperation.listStyleId;
            }
            if (options && options.startPosition) {
                var start = _.clone(options.startPosition),
                    listParaStyleId = this.getDefaultUIParagraphListStylesheet(),
                    insertStyleOperation = null,
                    paragraphStyles = self.getParagraphStyles();

                start.pop();
                // register pending style sheet via 'insertStyleSheet' operation
                if (_.isString(listParaStyleId) && paragraphStyles.isDirty(listParaStyleId)) {

                    var styleSheetAttributes = {
                        attrs: paragraphStyles.getStyleSheetAttributeMap(listParaStyleId),
                        type: 'paragraph',
                        styleId: listParaStyleId,
                        styleName: paragraphStyles.getName(listParaStyleId),
                        parent: paragraphStyles.getParentId(listParaStyleId),
                        uiPriority: paragraphStyles.getUIPriority(listParaStyleId)
                    };

                    // parent is an optional value, should not be send as 'null'
                    if (styleSheetAttributes.parent === null || styleSheetAttributes.parent === '') { delete styleSheetAttributes.parent; }

                    insertStyleOperation = _.extend({ name: Operations.INSERT_STYLESHEET }, styleSheetAttributes);
                    self.applyOperations(insertStyleOperation);

                    // remove the dirty flag
                    paragraphStyles.setDirty(listParaStyleId, false);
                }

                // the attributes object for the operation
                allAttrs = { styleId: listParaStyleId, paragraph: { listStyleId: defListStyleId, listLevel: 0 } };

                // handling change track informations for set attributes operations (36259)
                if (self.getChangeTrack().isActiveChangeTracking()) {
                    // Creating one setAttribute operation for each span inside the paragraph, because of old attributes!
                    oldAttrs = {};
                    paraNode = Position.getParagraphElement(self.getNode(), start);

                    if (DOM.isParagraphNode(paraNode)) {
                        oldAttrs = self.getChangeTrack().getOldNodeAttributes(paraNode);
                    }

                    // adding the old attributes, author and date for change tracking
                    if (oldAttrs) {
                        oldAttrs = _.extend(oldAttrs, self.getChangeTrack().getChangeTrackInfo());
                        allAttrs.changes = { modified: oldAttrs };
                    }
                }

                newOperation = {
                    name: Operations.SET_ATTRIBUTES,
                    attrs: allAttrs,
                    start: start
                };
                self.applyOperations(newOperation);
                newOperation = _.copy(newOperation);
                newOperation.start = Position.increaseLastIndex(newOperation.start);
                self.applyOperations(newOperation);
            } else {
                self.setAttributes('paragraph', { styleId: this.getDefaultUIParagraphListStylesheet(), paragraph: { listStyleId: defListStyleId, listLevel: 0 } });
            }

        };

        /**
         * Removes bullet/numbered list formatting from the selected
         * paragraphs.
         */
        this.removeListAttributes = function () {

            var paragraph = Position.getLastNodeFromPositionByNodeName(self.getCurrentRootNode(), self.getSelection().getStartPosition(), DOM.PARAGRAPH_NODE_SELECTOR),
                prevPara = Utils.findPreviousNode(self.getCurrentRootNode(), paragraph, DOM.PARAGRAPH_NODE_SELECTOR),
                attrs = self.getAttributes('paragraph'),
                newAttrs = { paragraph: { listStyleId: null, listLevel: -1 } };

            if (!attrs.styleId || attrs.styleId === self.getDefaultUIParagraphListStylesheet()) {
                //set list style only, if there is no special para style chosen
                newAttrs.styleId = self.getDefaultUIParagraphStylesheet();
            }

            self.setAttributes('paragraph', newAttrs);
            if (prevPara) {
                self.getParagraphStyles().updateElementFormatting(prevPara);
            }
        };

        /**
         * Returns all predefined bullet list styles.
         *
         * @returns {Object}
         *  A map with list style identifiers as keys, and objects as values
         *  containing the attributes 'definition' with the list style definition,
         *  'listLabel' containing the bullet text, and 'tooltip' containing a GUI
         *  tool tip string for the list style.
         */
        this.getPredefinedBulletListStyles = function () {

            var // the resulting bullet list styles, mapped by list identifier
                bulletListStyles = {},
                // all predefined list styles
                allPredefinedListStyles = self.getListCollection().getPredefinedListStyles();

            _(allPredefinedListStyles).each(function (listStyle, listStyleId) {
                if (listStyle.definition.listLevel0.numberFormat === 'bullet') {
                    bulletListStyles[listStyleId] = listStyle;
                }
            });

            return bulletListStyles;
        };

        /**
         * Returns all predefined numbered list styles.
         *
         * @returns {Object}
         *  A map with list style identifiers as keys, and objects as values
         *  containing the attributes 'definition' with the list style definition,
         *  'listlabel' containing a string with the number 1 formatted according
         *  to the list style, and 'tooltip' containing a GUI tool tip string for
         *  the list style.
         */
        this.getPredefinedNumberedListStyles = function () {

            var // the resulting numbered list styles, mapped by list identifier
                numberedListStyles = {},
                // all predefined list styles
                allPredefinedListStyles = self.getListCollection().getPredefinedListStyles();

            _(allPredefinedListStyles).each(function (listStyle, listStyleId) {
                if (listStyle.definition.listLevel0.numberFormat !== 'bullet') {
                    numberedListStyles[listStyleId] = listStyle;
                }
            });

            return numberedListStyles;
        };

        /**
         * Getter for the default value.
         *
         * @returns {String}
         *  The default constant value
         */
        this.getDefaultListValue = function () {
            return ListHandlerMixin.DEFAULT_VALUE;
        };

        /**
         * Check, whether the specified paragraph attribute set specifies a paragraph with
         * defined bullet type. The bullet can be a character, a number or a bitmap.
         *
         * @param {Object} paragraphAttrs
         *  The paragraph attribute set.
         *
         * @returns {Boolean}
         *  Whether the specified paragraph attribute set specifies a paragraph with defined
         *  list level.
         */
        this.isListParagraph = function (paragraphAttrs) {
            return paragraphAttrs && _.isNumber(paragraphAttrs.listLevel) && paragraphAttrs.listLevel > -1;
        };

        /**
         * Check, whether at a specified position the list auto detection shall be executed. In
         * the text application there are not limitations yet.
         *
         * @returns {Boolean}
         *  Whether the list auto detection can be executed at the current position.
         */
        this.isAutoDetectionPosition = function () {
            return true;
        };

        /**
         * Check, whether a specified string at a specified position in a paragraph with a specified
         * text length can be used to create a list automatically.
         *
         * @param {Number[]} position
         *  The logical position.
         *
         * @param {Number} paragraphLength
         *  The length of the text in the paragraph.
         *
         * @param {String} paraText
         *  The text in the paragraph.
         *
         * @returns {Boolean}
         *  Whether the specified string can be used to create a list automatically.
         */
        this.isListAutoDetectionString = function (position, paragraphLength, paraText) {

            return ((_.last(position) > (paraText.indexOf('. ') + 1)) &&
                    (((paraText.indexOf('. ') >= 0) && (paragraphLength > 3) && (paraText.indexOf('. ') < (paraText.length - 2))) ||
                     ((paraText.indexOf(' ') >= 0) && (paragraphLength > 2) && (paraText.indexOf('* ') === 0 || paraText.indexOf('- ') === 0))));
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class ListHandlerMixin

    // constants --------------------------------------------------------------

    /**
     * A constant that will be used in the GUI to represent the default style
     * of bullet lists or numbered lists.
     */
    ListHandlerMixin.DEFAULT_VALUE = '__DEFAULT__';

    // exports ================================================================

    return ListHandlerMixin;

});
