/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/components/drawing/drawingresize', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position'
], function (Utils, Tracking, AttributeUtils, DrawingFrame, Operations, DOM, Position) {

    'use strict';

    var // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    // static class DrawingResize =============================================

    var DrawingResize = {};

    // static methods ---------------------------------------------------------

    /**
     * Draws a selection box for the specified drawing node and registers
     * mouse handlers for moving and resizing.
     *
     * @param {TextApplication} app
     *  The application instance containing the drawing.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node to be selected, as DOM node or jQuery object.
     */
    DrawingResize.drawDrawingSelection = function (app, drawingNode) {

        var // the text editor
            editor = app.getModel(),
            // the root node of the editor DOM, or if inside header/footer, one of their root nodes
            rootNode = editor.getCurrentRootNode(),
            // the node containing the scroll bar
            scrollNode = app.getView().getContentRootNode(),
            // object containing information about movable and resizable
            options = { movable: true, resizable: true },  // -> parameter?
            // whether the drawing is located in the drawing layer
            isDrawingLayerNode = DOM.isDrawingLayerNode($(drawingNode).parent()),
            // whether the drawing is anchored to the paragraph
            isAbsoluteParagraphNode = !isDrawingLayerNode && DOM.isAbsoluteParagraphDrawing(drawingNode),
            // the container element used to visualize the selection
            selectionBox = null,
            // saving the selected resize node
            resizeNode = null,
            // the container element used to visualize the movement and resizing
            moveBox = null,
            //zoom factor in floating point notation
            zoomFactor,
            // target ID of currently active root node, if existing
            target = editor.getActiveTarget(),
            operationProperties = {};

        // common deinitialization after move or resize tracking
        function leaveTracking() {
            DrawingFrame.toggleTracking(drawingNode, false);
            moveBox.css({ left: '', top: '', right: '', bottom: '', width: '', height: '' });
        }

        /**
         * Handling all tracking events while moving a drawing.
         */
        var drawingMoveHandler = (function () {

            var // size of the drawing
                drawingWidth = 0, drawingHeight = 0,
                // the current position change (in px)
                shiftX = 0, shiftY = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the start scroll position
                startScrollX = 0, startScrollY = 0,
                // left/top distance from drawing to event point (in px)
                leftDrawingDistance = 0, topDrawingDistance = 0,
                // the container node of a page element that contains all top-level content nodes (paragraphs and tables)
                pageContent = isDrawingLayerNode ? editor.getNode() : DOM.getPageContentNode(editor.getNode()),
                // the width of the page content node
                pageContentWidth = isDrawingLayerNode ? Math.round(pageContent.outerWidth(true)) : Math.round(pageContent.width()),
                // the height of the page content node
                pageContentHeight = Math.round(pageContent.outerHeight(true)),
                // whether the mouse was really moved (Fix for 28633)
                mouseMoved = false,
                // the left offset of the page content node -> the drawing must be right of this position
                minLeftPosition = 0,
                // the left offset of the page content node plus its width -> the drawing must be left of this position
                maxRightPosition = 0,
                // the top position of the page content, not the page itself
                minTopPosition = 0,
                // adding the height of the page content (can be only one paragraph) to the top position of the page content
                maxBottomPosition = 0,
                // whether the move of the drawing can be change tracked
                // -> MS Word supports only moving of inline drawings (from inline to inline)
                // -> We do not support 'inline to inline' (in the moment), so that all move operations will not be change tracked yet.
                doChangeTrackMove = editor.getChangeTrack().ctSupportsDrawingMove();

            // initialing the moving of the drawing according to the passed tracking event
            function startMoveDrawing(event) {

                zoomFactor = app.getView().getZoomFactor() / 100;

                // storing old height and width of drawing
                drawingWidth = drawingNode.width();
                drawingHeight = drawingNode.height();

                // updating whether the drawing is located in the drawing layer
                isDrawingLayerNode = DOM.isDrawingLayerNode(drawingNode.parent());

                // updating the width, because outerWidth is dependent from zoom level (also handling comment layer node correctly)
                if (isDrawingLayerNode) { pageContentWidth = Math.round(pageContent.outerWidth(true) - (pageContent.hasClass(DOM.COMMENTMARGIN_CLASS) ? pageContent.data(DOM.COMMENTMARGIN_CLASS) : 0)); }
                pageContentHeight = Math.round(pageContent.outerHeight(true));

                leftDrawingDistance = Math.round(event.pageX - drawingNode.offset().left);
                topDrawingDistance = Math.round(event.pageY - drawingNode.offset().top);

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                // the left offset of the page content node -> the drawing must be right of this position
                minLeftPosition = Math.round(pageContent.offset().left);
                // the top position of the page content, not the page itself
                minTopPosition = editor.isHeaderFooterEditState() ? Math.round(rootNode.offset().top)  : Math.round(pageContent.offset().top);

                if (isDrawingLayerNode) {
                    // Forcing the drawing to stay on the page
                    // the left offset of the page content node plus its width -> the drawing must be left of this position
                    maxRightPosition = minLeftPosition + pageContentWidth - drawingWidth * zoomFactor;
                    // adding the height of the page content (can be only one paragraph) to the top position of the page content
                    maxBottomPosition = minTopPosition + (editor.isHeaderFooterEditState() ? (isDrawingLayerNode ? rootNode.outerHeight(true) : rootNode.height()) :  pageContentHeight) - drawingHeight * zoomFactor;
                } else {
                    // the left offset of the page content node plus its width -> the drawing must be left of this position
                    maxRightPosition = minLeftPosition + (pageContentWidth * zoomFactor);
                    // adding the height of the page content (can be only one paragraph) to the top position of the page content
                    maxBottomPosition = minTopPosition + ((editor.isHeaderFooterEditState() ? (isDrawingLayerNode ? rootNode.outerHeight(true) : rootNode.height()) :  pageContentHeight) * zoomFactor);
                }
            }

            // updating the drawing position according to the passed tracking event
            function updateMove(event) {

                // make move box visible when tracking position has moved
                DrawingFrame.toggleTracking(drawingNode, true);

                // reading scrollPosition again. Maybe it was not updated or not updated completely.
                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;

                shiftX = (event.pageX - event.startX + scrollX) / zoomFactor;
                shiftY = (event.pageY - event.startY + scrollY) / zoomFactor;

                // only move the moveBox, if the mouse was really moved (Fix for 28633)
                mouseMoved = (shiftX !== 0) || (shiftY !== 0);

                if (mouseMoved) {

                    if (event.pageX - leftDrawingDistance < minLeftPosition) { shiftX += (minLeftPosition - event.pageX + leftDrawingDistance) / zoomFactor; }
                    if (event.pageX - leftDrawingDistance > maxRightPosition) { shiftX += (maxRightPosition - event.pageX + leftDrawingDistance) / zoomFactor; }

                    // the upper left corner of the drawing always has to be inside the page content node -> reduce or increase shiftX and shiftY, if necessary
                    if (event.pageY - topDrawingDistance < minTopPosition) { shiftY += (minTopPosition - event.PageY + topDrawingDistance) / zoomFactor; }
                    if (event.pageY - topDrawingDistance > maxBottomPosition) { shiftY += (maxBottomPosition - event.pageY + topDrawingDistance) / zoomFactor; }

                    if (_.isNumber(shiftX) && _.isNumber(shiftY) && ((shiftX !== 0) || (shiftY !== 0))) {
                        moveBox.css({ left: shiftX, top: shiftY, width: drawingWidth, height: drawingHeight });
                    }
                }
            }

            // updates scroll position according to the passed tracking event
            function updateMoveScroll(event) {

                // update scrollPosition with suggestion from event
                if (event.scrollX) {
                    scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX);
                }

                if (event.scrollY) {
                    scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY);
                }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // handling the drawing position, when moving is stopped according to the passed tracking event
            function stopMoveDrawing(event) {

                // mouse up handler
                var generator = editor.createOperationsGenerator(),
                    // the horizontal move of the drawing in 1/100 mm
                    moveX = 0,
                    // the vertical move of the drawing in 1/100 mm
                    moveY = 0,
                    // the logical position where the drawing is located before it needs to be moved
                    updatePosition = null,
                    // the attributes of the drawing node
                    drawingNodeAttrs = null,
                    // the anchorHorOffset property of the drawing
                    anchorHorOffset = 0,
                    // the anchorVertOffset property of the drawing
                    anchorVertOffset = 0,
                    // the anchorHorBase property of the drawing
                    anchorHorBase = 0,
                    // the anchorVertBase property of the drawing
                    anchorVertBase = 0,
                    // the anchorHorAlign property of the drawing
                    anchorHorAlign = 0,
                    // the anchorVertAlign property of the drawing
                    anchorVertAlign = 0,
                    // the saved anchorHorOffset property of the drawing before move
                    oldAnchorHorOffset = 0,
                    // the saved anchorVertOffset property of the drawing before move
                    oldAnchorVertOffset = 0,
                    // paragraph node required for checking implicit paragraphs
                    localParagraph = null,
                    // logical paragraph position required for checking implicit paragraphs
                    localPosition = null,
                    // the logical destination for moved images
                    destPosition = null,
                    // current drawing width, in 1/100 mm
                    drawingWidth = 0,
                    // current drawing height, in 1/100 mm
                    drawingHeight = 0,
                    // the paragraph element containing the drawing node
                    paragraph = null,
                    // total width of the paragraph, in 1/100 mm
                    paraWidth = 0,
                    // is it necessary to move the image?
                    moveImage = false,
                    // whether the drawing was moved or not
                    imageMoved = false,
                    // whether the image was moved downwards caused by the resize operation
                    moveDownwards = false,
                    // position of the mouse up event shifted into the document borders
                    trimmedPosition = null,
                    // the current position sent by the event (in px)
                    currentX = 0, currentY = 0,
                    // the position of the top left corner of the drawing
                    drawingLeftX = 0, drawingTopY = 0,
                    // attributes object for operations
                    allDrawingAttrs = null,
                    // the old drawing attributes
                    oldDrawingAttrs = null,
                    // value of margin top of drawing node
                    drawingMarginTop = 0,
                    // maximum allowed height of page without page margins, in 1/100 mm
                    pageMaxHeight = editor.getPageLayout().getDefPageActiveHeight(),
                    // the start position of the drawing before it was moved relative to the page in pixel
                    startPosition = null,
                    // the maximum vertical offset for drawings at pages (page height minus drawing height)
                    maxVertPageOffset = 0,
                    // the page layout manager
                    pageLayout = null,
                    // the page number
                    pageNumber = 1,
                    // the active root node, might be header or footer
                    activeRootNode = editor.getCurrentRootNode(),
                    // whether the active root node is a header or footer node
                    isHeaderOrFooter = DOM.isHeaderOrFooter(activeRootNode),
                    // an optional footer offset
                    footerOffset = 0,
                    // whether position relative to margin shall be calculated instead of page
                    keepMarginSettings = false;

                function adaptPositionIntoDocument(posX, posY) {

                    // taking care of leftDrawingDistance and topDrawingDistance, so that the upper left corner is inside the pageContent.
                    if (posX < (minLeftPosition + leftDrawingDistance) / zoomFactor) { posX = (minLeftPosition + leftDrawingDistance) / zoomFactor; }
                    if (posX > (maxRightPosition + leftDrawingDistance) / zoomFactor) { posX = (maxRightPosition + leftDrawingDistance) / zoomFactor; }
                    if (posY < (minTopPosition + topDrawingDistance) / zoomFactor) { posY = (minTopPosition + topDrawingDistance) / zoomFactor; }
                    if (posY > (maxBottomPosition + topDrawingDistance) / zoomFactor) { posY = (maxBottomPosition + topDrawingDistance) / zoomFactor; }

                    return { posX: posX, posY: posY };
                }

                // check, whether a specified node contains the pixel position defined by posX and posY
                function isPositionInsideNode(node, posX, posY/*, reverse*/) {

                    function isInside(node, x, y) {

                        return ((Math.round(node.offset().left / zoomFactor) <= x) &&
                                (x <= Math.round((node.offset().left / zoomFactor) + node.outerWidth())) &&
                                (Math.round(node.offset().top  / zoomFactor) <= y) &&
                                (y <= Math.round((node.offset().top / zoomFactor) + node.outerHeight())));

                    }

                    if (!(node instanceof $)) { node = $(node); }

                    return isInside(node, posX, posY);
                }

                function iterateSelectorNodes(topNode, currentNode, posX, posY, selector, skipSelector, options) {

                    var selectorNode = null,
                        reverse = Utils.getBooleanOption(options, 'reverse', false);

                    while (currentNode) {

                        if (isPositionInsideNode(currentNode, posX, posY, reverse)) {
                            selectorNode = currentNode;
                            break;
                        }

                        if (reverse) {
                            currentNode = Utils.findPreviousNode(topNode, currentNode, selector, skipSelector);
                        } else {
                            currentNode = Utils.findNextNode(topNode, currentNode, selector, skipSelector);
                        }
                    }

                    return selectorNode;
                }

                function getParagraphAtPosition(topNode, startNode, shiftX, shiftY, posX, posY) {

                    var searchPrevious = true,
                        searchFollowing = true,
                        paragraph = null,
                        tableCell = null;

                    if ((shiftX > 0) && (shiftY > 0)) { searchPrevious = false; }
                    if ((shiftX < 0) && (shiftY < 0)) { searchFollowing = false; }

                    if (paragraph) { searchFollowing = false; }

                    if (searchFollowing) {
                        paragraph = iterateSelectorNodes(topNode, Utils.getDomNode(startNode), posX, posY, DOM.PARAGRAPH_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR, { reverse: false });
                    }

                    searchPrevious = !paragraph;

                    if (searchPrevious) {
                        paragraph = iterateSelectorNodes(topNode, Utils.getDomNode(startNode), posX, posY, DOM.PARAGRAPH_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR, { reverse: true });
                    }

                    // maybe the paragraph is in a table cell with a cell neighbor that is much higher -> use last paragraph in this cell
                    if (!paragraph) {
                        tableCell = iterateSelectorNodes(topNode, Utils.getDomNode(startNode), posX, posY, DOM.TABLE_CELLNODE_SELECTOR, DrawingFrame.NODE_SELECTOR, { reverse: false });

                        if (!tableCell) {
                            tableCell = iterateSelectorNodes(topNode, Utils.getDomNode(startNode), posX, posY, DOM.TABLE_CELLNODE_SELECTOR, DrawingFrame.NODE_SELECTOR, { reverse: true });
                        }

                        if (tableCell && DOM.isTableCellNode(tableCell) && !DOM.isExceededSizeTableNode($(tableCell).closest(DOM.TABLE_NODE_SELECTOR))) { // check if table is exceeded size for #33275
                            paragraph = DOM.getCellContentNode(tableCell)[0].lastChild;  // the last paragraph of the cell content
                        }
                    }

                    if (paragraph) { paragraph = $(paragraph); }

                    return paragraph;
                }

                // helper function to receive a reasonable value for inserting
                // a moved drawing with activated change tracking.
                function getDrawingInsertPosition() {

                    var // the best possible position to insert the drawing
                        insertPosition = _.clone(updatePosition),  // shifting drawing inside the same paragraph
                        // a local helper node
                        localDrawingNode = drawingNode;

                    // handling drawings in drawing layer
                    if (DOM.isDrawingLayerNode($(localDrawingNode).parent())) {
                        localDrawingNode = $(DOM.getDrawingPlaceHolderNode(localDrawingNode));
                        if (!paragraph) { paragraph = localDrawingNode.parent(); }
                    }

                    insertPosition[insertPosition.length - 1] = Position.getLeadingFloatingDrawingCount(paragraph, anchorVertOffset, localDrawingNode); //#33245 passing optional drawing node, to exclude itself from count

                    return insertPosition;
                }

                // begin of stopMoveDrawing

                if (!mouseMoved) {
                    return;
                }

                zoomFactor = app.getView().getZoomFactor() / 100;

                // setting position of tracking:end event
                currentX = event.pageX / zoomFactor;
                currentY = event.pageY / zoomFactor;

                // scrolling to the correct position -> absolutely necessary!
                scrollNode.scrollLeft(scrollX + startScrollX).scrollTop(scrollY + startScrollY);

                // shifting currentX and currentY to position inside the document
                trimmedPosition = adaptPositionIntoDocument(currentX, currentY);
                currentX = trimmedPosition.posX;
                currentY = trimmedPosition.posY;

                // calculating the real shift of the drawing
                shiftX = currentX - ((event.startX - scrollX) / zoomFactor);
                shiftY = currentY - ((event.startY - scrollY) / zoomFactor);

                // top left corner of the drawing
                drawingLeftX = currentX - (leftDrawingDistance / zoomFactor);
                drawingTopY = currentY - (topDrawingDistance / zoomFactor);

                updatePosition = Position.getOxoPosition(rootNode, drawingNode, 0);

                if ((_.isNumber(shiftX)) && (_.isNumber(shiftY)) && ((shiftX !== 0) || (shiftY !== 0))) {

                    drawingNodeAttrs = editor.getStyleCollection('drawing').getElementAttributes(drawingNode).drawing;

                    if (isDrawingLayerNode) {  // the drawing is anchored to page or margin

                        activeRootNode = editor.getCurrentRootNode();

                        isHeaderOrFooter = DOM.isHeaderOrFooter(activeRootNode);

                        startPosition = Position.getPixelPositionToRootNodeOffset(activeRootNode, drawingNode, zoomFactor * 100);

                        maxVertPageOffset = Utils.convertHmmToLength(app.getModel().getPageLayout().getPageAttribute('height'), 'px', 0.001) - drawingNode.height();

                        // checking the page number of the drawing
                        pageLayout = app.getModel().getPageLayout();

                        // checking the page number of the drawing
                        pageNumber = isHeaderOrFooter ? 1 : pageLayout.getPageNumber(DOM.getDrawingPlaceHolderNode(drawingNode));

                        if (pageNumber > 1) { startPosition.y -= Position.getVerticalPagePixelPosition(editor.getNode(), pageLayout, pageNumber, zoomFactor * 100); }

                        anchorHorOffset = startPosition.x + shiftX;
                        anchorVertOffset = startPosition.y + shiftY;

                        if (DOM.isFooterNode(activeRootNode)) {
                            // if this is a footer, the height between page start and footer need to be added
                            footerOffset = Utils.convertHmmToLength(pageLayout.getPageAttribute('height'), 'px', 1) - Utils.round($(activeRootNode).outerHeight(true), 1);
                            anchorVertOffset += footerOffset;
                        }

                        if (anchorVertOffset < 0) {
                            anchorVertOffset = 0; // forcing on the same page, no shift to page above
                        } else if (anchorVertOffset > maxVertPageOffset) {
                            anchorVertOffset = maxVertPageOffset; // forcing on the same page, no shift to page below
                        }

                        // converting anchorHorOffset and anchorVertOffset to 1/100 mm
                        anchorHorOffset = Utils.convertLengthToHmm(anchorHorOffset, 'px');
                        anchorVertOffset = Utils.convertLengthToHmm(anchorVertOffset, 'px');

                        keepMarginSettings = (drawingNodeAttrs.anchorHorBase === 'margin' && drawingNodeAttrs.anchorVertBase === 'margin');

                        if (keepMarginSettings) {
                            anchorHorOffset -= pageLayout.getPageAttribute('marginLeft');
                            anchorVertOffset -= pageLayout.getPageAttribute('marginTop');
                        }

                        anchorHorAlign = 'offset';
                        anchorHorBase = keepMarginSettings ? 'margin' : 'page';

                        anchorVertAlign = 'offset';
                        anchorVertBase = keepMarginSettings ? 'margin' : 'page';

                    } else {   // the drawing is anchored to paragraph or in-line with text

                        paragraph = drawingNode.parent();
                        // converting to 1/100 mm
                        moveX = Utils.convertLengthToHmm(shiftX, 'px');
                        moveY = Utils.convertLengthToHmm(shiftY, 'px');
                        drawingWidth = Utils.convertLengthToHmm(drawingNode.width(), 'px');
                        drawingHeight = Utils.convertLengthToHmm(drawingNode.height(), 'px');
                        paraWidth = Utils.convertLengthToHmm(paragraph.width(), 'px');
                        // evaluating attributes
                        oldAnchorHorOffset = drawingNodeAttrs.anchorHorOffset;
                        oldAnchorVertOffset = drawingNodeAttrs.anchorVertOffset ? drawingNodeAttrs.anchorVertOffset : 0;
                        anchorHorBase = drawingNodeAttrs.anchorHorBase;
                        anchorVertBase = drawingNodeAttrs.anchorVertBase;
                        anchorHorAlign = drawingNodeAttrs.anchorHorAlign;
                        anchorVertAlign = drawingNodeAttrs.anchorVertAlign;
                        drawingMarginTop = drawingNodeAttrs.marginTop;

                        if ((oldAnchorHorOffset === undefined) || (oldAnchorHorOffset === 0)) {
                            // anchorHorOffset has to be calculated corresponding to the left paragraph border
                            if (anchorHorAlign === 'right') {
                                oldAnchorHorOffset = paraWidth - drawingWidth;
                            } else if (anchorHorAlign === 'center') {
                                oldAnchorHorOffset = Math.round((paraWidth - drawingWidth) / 2);
                            } else {
                                oldAnchorHorOffset = 0;
                            }
                        }

                        anchorHorOffset = oldAnchorHorOffset;
                        anchorVertOffset = oldAnchorVertOffset;

                        // checking position of mouse up event
                        // -> is the top-left corner of the drawing still in the same paragraph?
                        if (isPositionInsideNode(paragraph, drawingLeftX, drawingTopY)) {   // -> new position is in the same paragraph (before the drawing is shifted!)

                            if (moveX !== 0) {

                                // moving an inline drawing needs correction of old horizontal offset
                                if (DOM.isInlineDrawingNode(drawingNode)) { oldAnchorHorOffset = Utils.convertLengthToHmm((drawingNode.offset().left - drawingNode.closest('div.p').offset().left) / zoomFactor, 'px'); }

                                anchorHorOffset = oldAnchorHorOffset + moveX;
                                anchorHorAlign = 'offset';
                                anchorHorBase = 'column';
                                if (anchorHorOffset < 0) {
                                    anchorHorOffset = 0;
                                } else if (anchorHorOffset > (paraWidth - drawingWidth)) {
                                    anchorHorOffset = paraWidth - drawingWidth;
                                }
                            }

                            if (moveY !== 0) {
                                anchorVertAlign = 'offset';
                                anchorVertBase = 'paragraph';

                                anchorVertOffset = Utils.convertLengthToHmm((drawingTopY - (paragraph.offset().top / zoomFactor)), 'px');
                                // anchorVertOffset always has to be >= 0, not leaving the paragraph ('< 0' should never happen here)
                                if (anchorVertOffset < 0) { anchorVertOffset = 0; }

                                // correction for anchorVertOffset, sum of anchorVertOffset and height of drawing (including top margin) cannot be greater than maximum page size
                                if (anchorVertOffset > pageMaxHeight - drawingHeight - drawingMarginTop) {
                                    anchorVertOffset = pageMaxHeight - drawingHeight - drawingMarginTop;
                                }
                            }

                        } else {   // -> new position is in another paragraph or behind the last paragraph (before the drawing is moved!)

                            // paragraph has to be determined from the coordinates of the top left corner of the drawing
                            // -> moving operation for the drawing is always required
                            paragraph = getParagraphAtPosition(rootNode, paragraph, shiftX, shiftY, drawingLeftX, drawingTopY);

                            if (paragraph) {

                                // -> the paragraph must not be the paragraph inside a text frame (not moving inside text frames)
                                if (DOM.isNodeInsideTextFrame(paragraph)) {
                                    paragraph = null;
                                } else {
                                    moveImage = true;
                                }

                            } else {

                                // No paragraph found: Do not call set Attributes and not moveImage
                                moveImage = false;

                                anchorHorOffset = oldAnchorHorOffset + moveX;
                                anchorVertOffset = oldAnchorVertOffset + moveY;

                                anchorVertAlign = 'offset';
                                anchorVertBase = 'paragraph';
                                anchorHorAlign = 'offset';
                                anchorHorBase = 'column';
                            }

                            if (moveImage) {

                                // whether the drawing is a text frame -> it must not be moved into another text frame -> simply do nothing
                                if (DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode) && DOM.isNodeInsideTextFrame(paragraph)) { return; }

                                // updating horizontal and vertical positions
                                anchorVertAlign = 'offset';
                                anchorVertBase = 'paragraph';
                                anchorVertOffset = Utils.convertLengthToHmm((drawingTopY - (paragraph.offset().top / zoomFactor)), 'px');

                                anchorHorAlign = 'offset';
                                anchorHorBase = 'column';
                                anchorHorOffset = Utils.convertLengthToHmm((drawingLeftX - (paragraph.offset().left / zoomFactor)), 'px');

                                destPosition = Position.getOxoPosition(rootNode, paragraph, 0);

                                // moving behind already existing drawings
                                destPosition.push(Position.getLeadingFloatingDrawingCount(paragraph, drawingTopY - (paragraph.offset().top / zoomFactor)));
                            }
                        }
                    }

                    // Generating operations. If change tracking is activated, different operations need
                    // to be created, because the drawing at the old position stays on its position but
                    // needs to be marked as 'deleted'. At the new position a new drawing needs to be
                    // inserted, that is marked as inserted.
                    // If change tracking is NOT active, it is first necessary to switch from in-line mode
                    // to floated mode. Then a move operation might be necessary and finally a further
                    // setAttributes operation, that sets the attributes for the position of the drawing
                    // at its new location.

                    // 1. Switching from in-line to floated mode, if required, using default values for floated mode
                    // -> in the case of activated change tracking this operation is not required. Instead it is
                    // necessary to mark the drawing as 'deleted'.
                    if ((anchorHorOffset !== oldAnchorHorOffset) || (anchorVertOffset !== oldAnchorVertOffset)) {

                        if (editor.getChangeTrack().isActiveChangeTracking() && doChangeTrackMove && !editor.getChangeTrack().isInsertNodeByCurrentAuthor(drawingNode)) {
                            // if change tracking is active, the old drawing position must be marked as 'deleted' ...
                            // -> this is necessary for in-line and non-in-line drawings
                            operationProperties = {
                                attrs: { changes: { removed: editor.getChangeTrack().getChangeTrackInfo() } },
                                start: updatePosition
                            };
                            editor.extendPropertiesWithTarget(operationProperties, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
                        } else if (DOM.isInlineDrawingNode(drawingNode)) {
                            // switching from in-line to floated mode (vertically aligned at paragraph)
                            operationProperties = {
                                attrs: { drawing: { inline: false, anchorHorBase: 'column', anchorVertBase: 'paragraph', anchorHorAlign: 'right', anchorHorOffset: 0, textWrapMode: 'topAndBottom', textWrapSide: null } },
                                start: updatePosition
                            };
                            editor.extendPropertiesWithTarget(operationProperties, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
                        }
                    }

                    // 2. Moving the image, if this is necessary
                    // -> In case of activated change tracking, this is only necessary, if the drawing was inserted
                    //    before by the current author. Never moving drawings located in the drawing layer node.
                    if (moveImage && !isDrawingLayerNode && !_.isEqual(updatePosition, destPosition)) {

                        // moving the drawing, if change tracking is NOT active or it is active
                        // and the current author has inserted the drawing before
                        if (!editor.getChangeTrack().isActiveChangeTracking() || !doChangeTrackMove || (editor.getChangeTrack().isActiveChangeTracking() && editor.getChangeTrack().isInsertNodeByCurrentAuthor(drawingNode))) {

                            // check, if destPosition is located in a non-implicit paragraph -> otherwise create paragraph
                            // similar to editor.handleImplicitParagraph
                            localPosition = _.clone(destPosition);

                            if (localPosition.pop() === 0) {  // is this an empty paragraph?
                                localParagraph = Position.getParagraphElement(rootNode, localPosition);
                                if ((DOM.isImplicitParagraphNode(localParagraph)) && (Position.getParagraphNodeLength(localParagraph) === 0)) {
                                    // removing implicit paragraph node
                                    $(localParagraph).remove();
                                    // creating new paragraph explicitly
                                    operationProperties = { start: localPosition };
                                    editor.extendPropertiesWithTarget(operationProperties, target);
                                    generator.generateOperation(Operations.PARA_INSERT, operationProperties);
                                }
                            }

                            // move operation
                            operationProperties = {
                                start: updatePosition,
                                end: updatePosition,
                                to: destPosition
                            };
                            editor.extendPropertiesWithTarget(operationProperties, target);
                            generator.generateOperation(Operations.MOVE, operationProperties);

                            updatePosition = _.clone(destPosition); // for setting attributes required
                            if (moveDownwards) {
                                updatePosition[updatePosition.length - 1] -= 1;  // moving from [0,0] to [0,2], so that it ends at [0,1]
                            }

                            imageMoved = true;
                        }
                    }

                    // 3. Setting attributes to the drawing at its new location (for the correct positioning)
                    if ((anchorHorOffset !== oldAnchorHorOffset) || (anchorVertOffset !== oldAnchorVertOffset)) {

                        // Inserting a new drawing if change tracking is active and the drawing was not inserted
                        // by the current author before. This is always necessary, if the offset was modified. Independent
                        // whether the drawing was moved before or not.
                        if (editor.getChangeTrack().isActiveChangeTracking() && doChangeTrackMove && !editor.getChangeTrack().isInsertNodeByCurrentAuthor(drawingNode)) {

                            // If change tracking is active and the drawing was not inserted by the current author before, then:
                            // - the drawing needs to be inserted at its new position with an insertDrawing operation.
                            // - if a move operation was required (without change tracking) the new logical position
                            //   is already determined and saved in 'destPosition'.
                            // - if no move operation was required (moving inside the same paragraph), the destPosition needs
                            //   to be determined now using helper function 'getDrawingInsertPosition'.
                            // TODO: Calling getDrawingInsertPosition() is not working for drawings in the drawing layer.
                            // -> But this is no problem, because change tracking does not support move of drawings, that
                            //    are not in-line (behavior of MS Office)
                            if (!destPosition) { destPosition = getDrawingInsertPosition(); }

                            // check, if destPosition is located in a non-implicit paragraph -> otherwise create paragraph
                            // similar to editor.handleImplicitParagraph
                            localPosition = _.clone(destPosition);

                            if (localPosition.pop() === 0) {  // is this an empty paragraph?
                                localParagraph = Position.getParagraphElement(rootNode, localPosition);
                                if ((DOM.isImplicitParagraphNode(localParagraph)) && (Position.getParagraphNodeLength(localParagraph) === 0)) {
                                    // removing implicit paragraph node
                                    $(localParagraph).remove();
                                    // creating new paragraph explicitly
                                    operationProperties = { start: localPosition };
                                    editor.extendPropertiesWithTarget(operationProperties, target);
                                    generator.generateOperation(Operations.PARA_INSERT, operationProperties);
                                }
                            }

                            oldDrawingAttrs = AttributeUtils.getExplicitAttributes(drawingNode);
                            allDrawingAttrs = oldDrawingAttrs;

                            // setting default floated value for the inserted drawing
                            if (DOM.isInlineDrawingNode(drawingNode)) {
                                allDrawingAttrs.drawing.inline = false;  // the drawing is never in-line in new position
                                allDrawingAttrs.drawing.textWrapMode = 'topAndBottom';
                                allDrawingAttrs.drawing.textWrapSide = null;
                            }

                            allDrawingAttrs.changes = { inserted: editor.getChangeTrack().getChangeTrackInfo(), removed: null };

                            operationProperties = {
                                start: destPosition,
                                type: DrawingFrame.getDrawingType(drawingNode),
                                attrs: allDrawingAttrs
                            };
                            editor.extendPropertiesWithTarget(operationProperties, target);
                            generator.generateOperation(Operations.DRAWING_INSERT, operationProperties);

                            updatePosition = _.clone(destPosition); // for setting attributes required
                        }

                        // and finally setting the new position attributes via a setAttributes operation
                        operationProperties = {
                            attrs: { drawing: { anchorHorOffset: anchorHorOffset, anchorVertOffset: anchorVertOffset, anchorHorAlign: anchorHorAlign, anchorVertAlign: anchorVertAlign, anchorHorBase: anchorHorBase, anchorVertBase: anchorVertBase } },
                            start: updatePosition
                        };
                        editor.extendPropertiesWithTarget(operationProperties, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

                    }
                }

                // apply the operations (undo group is created automatically)
                editor.applyOperations(generator);

                if (imageMoved) {
                    // set new text selection after moving the drawing, this will repaint the selection
                    editor.getSelection().setTextSelection(updatePosition, Position.increaseLastIndex(updatePosition));
                    // special handling for text frames, which need, that the frame is repainted after move (Webkit and IE)
                    // -> otherwise setting a selection in the text frame will fail
                    if (DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode) && (_.browser.WebKit || _.browser.IE)) {
                        app.getModel().executeDelayed(function () {
                            DrawingFrame.clearSelection(drawingNode);
                            DrawingResize.drawDrawingSelection(app, drawingNode);
                        }, { infoString: 'DrawingResize: setTextSelectionAfterMove', app: app });
                    }
                }
                if (DOM.isMarginalNode(drawingNode)) {
                    app.getModel().getDrawingLayer().updateCropMarginalParagraphDrawing(drawingNode);
                }
            }

            // finalizes the move tracking
            function finalizeMoveDrawing() {
                leaveTracking();
                drawingNode.off(TRACKING_EVENT_NAMES);

                // Resetting variables for new mouseup events without mousemove
                shiftX = shiftY = scrollX = scrollY = 0;
            }

            // return the actual drawingMoveHandler() method
            return function (event) {

                // handling for text frame drawings, if the click happened inside the internal text frame element
                if ($(event.target).closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0) {

                    Tracking.disableTracking(drawingNode)
                        .off('tracking:start', trackingStartHandler)
                        .off(TRACKING_EVENT_NAMES);

                    return;
                }

                switch (event.type) {
                    case 'tracking:start':
                        startMoveDrawing(event);
                        break;
                    case 'tracking:move':
                        updateMove(event);
                        break;
                    case 'tracking:scroll':
                        updateMoveScroll(event);
                        break;
                    case 'tracking:end':
                        stopMoveDrawing(event);
                        finalizeMoveDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeMoveDrawing(event);
                        break;
                }
            };

        }()); // end of drawingMoveHandler() local scope

        /**
         * Handles all tracking events while resizing a drawing.
         */
        var drawingResizeHandler = (function () {

            var // original size of the drawing
                oldWidth = 0, oldHeight = 0,
                // the size of the resized drawing (in px)
                finalWidth = 0, finalHeight = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the initial scroll position
                startScrollX = 0, startScrollY = 0,
                // whether resizing is available in horizontal/vertical direction
                useX = false, useY = false, topResize = false, leftResize = false, bottomResize = false,
                // correction factor for resizing to the left/top
                scaleX = 0, scaleY = 0,
                // the maximum width of the drawing on the page or in a table cell
                maxDrawingWidth = null,
                // the maximum height of the drawing (sometimes same scaling as width)
                // -> the drawing must not be higher than the page
                maxDrawingHeight = null;

            // initializes resizing the drawing according to the passed tracking event
            function startResizeDrawing(event) {

                var pos = $(event.target).attr('data-pos'),
                    // the maximum height of a drawing related to its width
                    newHeightToWidth = 0,
                    // attributes at page
                    pageAttributes = 0,
                    // the paragraph element containing the drawing node
                    paragraph = null;

                // updating whether the drawing is located in the drawing layer
                isDrawingLayerNode = DOM.isDrawingLayerNode(drawingNode.parent());

                zoomFactor = app.getView().getZoomFactor() / 100;

                // storing old height and width of drawing
                oldWidth = finalWidth = drawingNode.width();
                oldHeight = finalHeight = drawingNode.height();

                // collecting information about the handle node
                useX = /[lr]/.test(pos);
                useY = /[tb]/.test(pos);
                topResize = /[t]/.test(pos);
                bottomResize = /[b]/.test(pos);
                leftResize = /[l]/.test(pos);

                if (/l/.test(pos)) {
                    scaleX = -1;
                    moveBox.css({ left: 'auto', right: 0 });
                } else {
                    scaleX = 1;
                    moveBox.css({ left: 0, right: 'auto' });
                }

                if (/t/.test(pos)) {
                    scaleY = -1;
                    moveBox.css({ top: 'auto', bottom: 0 });
                } else {
                    scaleY = 1;
                    moveBox.css({ top: 0, bottom: 'auto' });
                }

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                // reduce the width of the drawing, if the width is bigger than the width of
                // the page plus the page's right margin or the width of the paragraph in a table cell
                paragraph = drawingNode.parent();
                maxDrawingWidth = paragraph.width();  // using only the paragraph width, no padding or margin (task 28867)

                pageAttributes = editor.getStyleCollection('page').getElementAttributes(editor.getNode());
                // reading page attributes, they are always available -> no need to check existence
                maxDrawingHeight = pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom;
                maxDrawingHeight = Utils.convertHmmToLength(maxDrawingHeight, 'px', 1);

                // in the case of non-deforming resize, there is also a maximum value for the height
                if (useX && useY) {
                    newHeightToWidth = oldHeight * maxDrawingWidth / oldWidth;
                    if ((maxDrawingHeight) && (maxDrawingHeight < newHeightToWidth)) {
                        // reduction of maxDrawingWidth is required
                        maxDrawingWidth *= (maxDrawingHeight / newHeightToWidth);
                    } else {
                        maxDrawingHeight = newHeightToWidth;
                    }
                }
            }

            // updates scroll position according to the passed tracking event
            function updateResizeScroll(event) {

                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // updates resizing the drawing according to the passed tracking event
            function updateResize(event) {

                var deltaX = useX ? ((event.pageX / zoomFactor - event.startX / zoomFactor + scrollX) * scaleX) : 0,
                    deltaY = useY ? ((event.pageY / zoomFactor - event.startY / zoomFactor + scrollY) * scaleY) : 0,
                    // the scaling factor for the width
                    scaleWidth = 1,
                    // the scaling factor for the height
                    scaleHeight = 1;

                // update drawing size
                finalWidth = Math.max(0, oldWidth + deltaX);
                finalHeight = Math.max(0, oldHeight + deltaY);

                // The maximum value for the width is 'maxDrawingWidth' -> avoid endless scrolling
                // There is no limit for the height (only in non-deforming resizing).
                if (finalWidth >= maxDrawingWidth) {
                    finalWidth = maxDrawingWidth;
                }

                if (finalHeight >= maxDrawingHeight) {
                    finalHeight = maxDrawingHeight;
                }

                // use the same scaling factor for vertical and horizontal resizing, if both are enabled
                // -> the larger number wins
                if (useX && useY) {
                    scaleWidth = finalWidth / oldWidth;
                    scaleHeight = finalHeight / oldHeight;

                    if (scaleWidth > scaleHeight) {
                        finalHeight = scaleWidth * oldHeight;
                    } else {
                        finalWidth = scaleHeight * oldWidth;
                    }
                }

                // make move box visible when tracking position has moved
                DrawingFrame.toggleTracking(drawingNode, true);
                moveBox.css({ width: finalWidth, height: finalHeight });
            }

            // resizes the drawing according to the passed tracking event
            function stopResizeDrawing() {

                var generator = editor.createOperationsGenerator(),
                    // all attributes for the operation
                    allAttrs = {},
                    // drawing attributes for the operation
                    attributes = {},
                    // the final scaling of the drawing width and height
                    finalWidthScaling = 1,
                    // the paragraph containing the resized drawing
                    paragraph = drawingNode.parent(),
                    // the logical position where the drawing is located before it needs to be moved
                    updatePosition = Position.getOxoPosition(rootNode, drawingNode, 0),
                    // the attributes of the drawing node
                    drawingNodeAttrs = null,
                    // the saved anchorVertOffset and anchorHorOffset property of the drawing before resize
                    oldAnchorVertOffset = 0, oldAnchorHorOffset = 0,
                    // the old drawing attributes
                    oldAttrs = null,
                    // the start position of the drawing before it was moved relative to the page in pixel
                    startPosition = null,
                    // the page layout manager
                    pageLayout = null,
                    // the affected drawings with auto resize functionality
                    allAutoResizeNodes = null,
                    // the page number
                    pageNumber = 1,
                    // the active root node, this can be a header or footer
                    activeRootNode = null,
                    // an optional offset of the footer node in the page
                    footerOffset = 0,
                    // whether position relative to margin shall be calculated instead of page
                    keepMarginSettings = false;

                // Calculating new width and height

                // calculate width
                if (useX && (finalWidth > 0) && (finalWidth !== oldWidth)) {
                    attributes.width = Utils.convertLengthToHmm(finalWidth, 'px');

                    if (finalWidth > maxDrawingWidth) {
                        finalWidthScaling = maxDrawingWidth / finalWidth;
                        attributes.width = Utils.convertLengthToHmm(maxDrawingWidth, 'px');
                    }
                }

                // calculate height
                if (useY && (finalHeight > 0) && (finalHeight !== oldHeight)) {
                    if (finalWidthScaling !== 1) { finalHeight *= finalWidthScaling; }
                    attributes.height = Utils.convertLengthToHmm(finalHeight, 'px');
                }

                // simplified handling for drawings in drawing layer
                if (isDrawingLayerNode) {

                    drawingNodeAttrs = editor.getStyleCollection('drawing').getElementAttributes(drawingNode).drawing;
                    keepMarginSettings = (drawingNodeAttrs.anchorHorBase === 'margin' && drawingNodeAttrs.anchorVertBase === 'margin');

                    activeRootNode = editor.getCurrentRootNode();

                    zoomFactor = app.getView().getZoomFactor();

                    // is it necessary to update the upper left corner of the drawing?
                    startPosition = Position.getPixelPositionToRootNodeOffset(activeRootNode, drawingNode, zoomFactor);
                    pageLayout = app.getModel().getPageLayout();

                    // checking the page number of the drawing
                    pageNumber = DOM.isHeaderOrFooter(activeRootNode) ? 1 : pageLayout.getPageNumber(DOM.getDrawingPlaceHolderNode(drawingNode));

                    if (pageNumber > 1) { startPosition.y -= Position.getVerticalPagePixelPosition(editor.getNode(), pageLayout, pageNumber, zoomFactor); }

                    if (topResize) {
                        // comparing final height and old height
                        attributes.anchorVertOffset = Math.max(Utils.convertLengthToHmm(startPosition.y, 'px') - Utils.convertLengthToHmm((finalHeight - oldHeight), 'px'), 0);
                    } else {
                        attributes.anchorVertOffset = Utils.convertLengthToHmm(startPosition.y, 'px');
                    }

                    if (DOM.isFooterNode(activeRootNode)) {
                        // if this is a footer, the height between page start and footer need to be added
                        footerOffset = pageLayout.getPageAttribute('height') - Utils.convertLengthToHmm($(activeRootNode).outerHeight(true), 'px');
                        attributes.anchorVertOffset += footerOffset;
                    }

                    if (leftResize) {
                        // comparing final width and old width
                        attributes.anchorHorOffset = Math.max(Utils.convertLengthToHmm(startPosition.x, 'px') - Utils.convertLengthToHmm((finalWidth - oldWidth), 'px'), 0);
                    }

                    // using margin instead of page, if drawing was anchored to margin before
                    if (keepMarginSettings) {
                        if (leftResize) { attributes.anchorHorOffset -= pageLayout.getPageAttribute('marginLeft'); }
                        attributes.anchorVertOffset -= pageLayout.getPageAttribute('marginTop');
                    }

                    // always sending this? Or only, if it was modified? Avoid sending horizontal information, if it was not modified (43996)
                    if (leftResize) {
                        attributes.anchorHorAlign = 'offset';
                        attributes.anchorHorBase = keepMarginSettings ? 'margin' : 'page';
                    }

                    attributes.anchorVertAlign = 'offset';
                    attributes.anchorVertBase = keepMarginSettings ? 'margin' : 'page';

                } else if (isAbsoluteParagraphNode) {

                    // if the drawing was resized to the left or to the top, it might be necessary to update the values
                    // for anchorHorOffset and anchorVertOffset
                    if (topResize || bottomResize) {

                        drawingNodeAttrs = editor.getStyleCollection('drawing').getElementAttributes(drawingNode).drawing;

                        oldAnchorVertOffset = drawingNodeAttrs.anchorVertOffset || 0;

                        if ((oldAnchorVertOffset >= 0 && topResize) || (oldAnchorVertOffset < 0 && bottomResize)) {

                            var deltaHeight = topResize ? (finalHeight - oldHeight) : (oldHeight - finalHeight);

                            attributes.anchorVertOffset = oldAnchorVertOffset - Utils.convertLengthToHmm(deltaHeight, 'px');

                            // simplification -> not allowing to resize drawing above paragraph borders
                            if (topResize && attributes.anchorVertOffset < 0) {
                                attributes.anchorVertOffset = 0;
                            } else if (bottomResize && attributes.anchorVertOffset >= 0) {
                                attributes.anchorVertOffset = -1;
                            }
                        }
                    }

                    // the drawing was resized in left direction
                    if (leftResize) {

                        drawingNodeAttrs = drawingNodeAttrs || editor.getStyleCollection('drawing').getElementAttributes(drawingNode).drawing;

                        oldAnchorHorOffset = drawingNodeAttrs.anchorHorOffset ? drawingNodeAttrs.anchorHorOffset : 0;

                        attributes.anchorHorOffset = oldAnchorHorOffset - Utils.convertLengthToHmm((finalWidth - oldWidth), 'px');
                        if (attributes.anchorHorOffset < 0) {  attributes.anchorHorOffset = 0; }
                    }
                }

                // Generating the setAttribute operation, if necessary
                if (!_.isEmpty(attributes)) {

                    allAttrs = { drawing: attributes };

                    // after resizing an automatic resized text frame in vertical direction by the user,
                    // the automatic resize will be removed
                    if (_.isNumber(attributes.height)) {
                        if (DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                            allAttrs.shape = { autoResizeHeight: false };
                        } else if (DrawingFrame.isGroupDrawingFrame(drawingNode)) {
                            // disable autoResizeHeight for all children inside a resized group
                            allAutoResizeNodes = drawingNode.find(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
                            if (allAutoResizeNodes.length > 0) {
                                _.each(allAutoResizeNodes, function (resizeNode) {
                                    var localDrawingNode = $(resizeNode).parent(),
                                        localDrawingPos = Position.getOxoPosition(rootNode, localDrawingNode, 0);

                                    generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                        start: localDrawingPos,
                                        attrs: { shape: { autoResizeHeight: false } }
                                    });

                                });
                            }
                        }
                    }

                    // Resize of drawing is not supported yet in OOXML file format
                    // -> this code needs to be activated, if OOXML supports change tracking of resizing of drawings
                    // -> in this case ctSupportsDrawingResizing() has to return 'true'
                    if (editor.getChangeTrack().isActiveChangeTracking() && editor.getChangeTrack().ctSupportsDrawingResizing()) {
                        // Expanding operation for change tracking with old explicit attributes
                        oldAttrs = editor.getChangeTrack().getOldNodeAttributes(drawingNode);
                        // adding the old attributes, author and date for change tracking
                        if (oldAttrs) {
                            oldAttrs = _.extend(oldAttrs, editor.getChangeTrack().getChangeTrackInfo());
                            allAttrs.changes = { modified: oldAttrs };
                        }
                    }

                    operationProperties = {
                        start: updatePosition,
                        attrs: allAttrs
                    };

                    editor.extendPropertiesWithTarget(operationProperties, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
                }

                // Applying the operations (undo group is created automatically)
                editor.applyOperations(generator);

                // Triggering text frame border to be repainted
                if (paragraph) { app.getModel().trigger('paragraphUpdate:after', paragraph); }

                // if drawing is inside marginal node, check cropping of drawing
                if (DOM.isMarginalNode(drawingNode)) {
                    app.getModel().getDrawingLayer().updateCropMarginalParagraphDrawing(drawingNode);
                }
            }

            // finalizes the resize tracking
            function finalizeResizeDrawing() {
                leaveTracking();
                resizeNode.off(TRACKING_EVENT_NAMES);
                app.getView().scrollToChildNode(drawingNode);
            }

            // return the actual drawingResizeHandler() method
            return function (event) {

                // Fix for 33987, avoid grabber in IE 11. Also avoiding browser selection in Webkit browsers
                // For Firefox this is useful to avoid the 'mozgrabber' aroung text frames
                // if (_.browser.IE === 11 || _.browser.WebKit) { event.preventDefault(); }
                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startResizeDrawing(event);
                        break;
                    case 'tracking:move':
                        updateResize(event);
                        break;
                    case 'tracking:scroll':
                        updateResizeScroll(event);
                        break;
                    case 'tracking:end':
                        stopResizeDrawing(event);
                        finalizeResizeDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeResizeDrawing(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();
            };

        }()); // end of drawingResizeHandler() local scope

        /**
         * Handler for 'tracking:start' events for 'div.handle' and 'div.move'
         * nodes. This handler has to decide, which node got the event. For this
         * node all other tracking events have to be registered. Additionally the
         * handler is called with the 'tracking:start' event, so that the drawing
         * initialization happens.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event, that starts the moving or the resizing
         *  of a drawing.
         */
        function trackingStartHandler(event) {

            var pos = DrawingFrame.getResizerHandleType(event.target);
            if (_.isString(pos)) {
                resizeNode = $(event.target);
                resizeNode.off(TRACKING_EVENT_NAMES);
                resizeNode.on(TRACKING_EVENT_NAMES, drawingResizeHandler);
                drawingResizeHandler.call(this, event);
            } else {
                drawingNode.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                drawingNode.on(TRACKING_EVENT_NAMES, drawingMoveHandler);
                drawingMoveHandler.call(this, event);
            }
        }

        // starting code of static method drawDrawingSelection()
        drawingNode = $(drawingNode);
        if (drawingNode.length !== 1) {
            Utils.error('DrawingResize.drawDrawingSelection(): single drawing node expected');
            drawingNode = drawingNode.first();
        }

        if (drawingNode.hasClass('horizontal-line')) {
            options.movable = false;
        } else if (drawingNode.hasClass('grouped')) {
            // options.movable = options.resizable = false;
            drawingNode = drawingNode.closest(DrawingFrame.GROUPCONTENT_SELECTOR).parent();
        }

        if (!editor.getEditMode()) {
            options.movable = options.resizable = false;
        }

        options.scaleHandles = 100 / app.getView().getZoomFactor();

        // the container element used to visualize the selection
        selectionBox = DrawingFrame.drawSelection(drawingNode, options);

        // set visible drawing anchor
        app.getView().setVisibleDrawingAnchor(drawingNode, AttributeUtils.getExplicitAttributes(drawingNode));

        // initialize move tracking
        moveBox = selectionBox.children('.tracker');

        if (options.movable) {
            Tracking.enableTracking(drawingNode, {
                autoScroll: true,
                borderNode: scrollNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 10,
                maxSpeed: 250
            });
            drawingNode.on('tracking:start', trackingStartHandler);
        } else {
            Tracking.disableTracking(drawingNode);
            drawingNode.off('tracking:start', trackingStartHandler);
        }

        // initialize resize tracking
        var resizeHandles = selectionBox.find('>.resizers>[data-pos]');
        if (options.resizable) {
            Tracking.enableTracking(resizeHandles, {
                autoScroll: true,
                borderNode: scrollNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 10,
                maxSpeed: 250
            });
            resizeHandles.on('tracking:start', trackingStartHandler);
        } else {
            Tracking.disableTracking(resizeHandles);
            resizeHandles.off('tracking:start', trackingStartHandler);
        }
    };

    // exports ================================================================

    return DrawingResize;

});
