/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 * @author Marko Benigar <marko.benigar@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/view', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/editframework/view/editview',
    'io.ox/office/editframework/view/popup/attributestooltip',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/popup/chartlabelsmenu',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/view/statuspane',
    'io.ox/office/spreadsheet/view/gridpane',
    'io.ox/office/spreadsheet/view/headerpane',
    'io.ox/office/spreadsheet/view/cornerpane',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'io.ox/office/spreadsheet/view/popup/namedrangeslayermenu',
    'io.ox/office/spreadsheet/view/mixin/selectionmixin',
    'io.ox/office/spreadsheet/view/mixin/highlightmixin',
    'io.ox/office/spreadsheet/view/mixin/celleditmixin',
    'io.ox/office/spreadsheet/view/mixin/viewfuncmixin',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/rendercache',
    'gettext!io.ox/office/spreadsheet/main',
    'less!io.ox/office/spreadsheet/view/style'
], function (Utils, KeyCodes, Forms, IteratorUtils, Tracking, SimpleMap, LocaleData, EditView, AttributesToolTip, DrawingFrame, ChartLabelsMenu, SheetUtils, PaneUtils, FormulaUtils, StatusPane, GridPane, HeaderPane, CornerPane, Labels, Controls, DefinedNamesMenu, SelectionMixin, HighlightMixin, CellEditMixin, ViewFuncMixin, RenderUtils, RenderCache, gt) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Interval = SheetUtils.Interval;
    var Range = SheetUtils.Range;
    var IntervalArray = SheetUtils.IntervalArray;
    var RangeArray = SheetUtils.RangeArray;

    // codes for various message texts used in alert boxes
    var RESULT_MESSAGES = {

        'sheet:name:empty':
            gt('The sheet name must not be empty. Please enter a name.'),

        'sheet:name:invalid':
            gt('This sheet name contains invalid characters. Please ensure that the name does not contain / \\ * ? : [ or ], and does neither start nor end with an apostrophe.'),

        'sheet:name:used':
            gt('This sheet name is already used. Please enter another name.'),

        'sheet:locked':
            gt('You cannot use this command on a protected sheet.'),

        'cols:overflow':
            gt('It is not possible to modify more than %1$d columns at the same time.', _.noI18n(SheetUtils.MAX_CHANGE_COLS_COUNT)),

        'rows:overflow':
            gt('It is not possible to modify more than %1$d rows at the same time.', _.noI18n(SheetUtils.MAX_CHANGE_ROWS_COUNT)),

        'cells:locked':
            gt('Protected cells cannot be modified.'),

        'cells:overflow':
            gt('It is not possible to modify more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)),

        'cells:pushoff':
            gt('It is not possible to insert new cells because non-empty cells would be pushed off the end of the sheet.'),

        'merge:overlap':
            //#. Warning text: trying to merge multiple cell ranges which overlap each other
            gt('Overlapping ranges cannot be merged.'),

        'merge:overflow':
            //#. Warning text: trying to merge too many cell ranges at the same time
            //#. %1$d is the maximum number of cell ranges that can be merged at a time
            //#, c-format
            gt('It is not possible to merge more than %1$d ranges at the same time.', _.noI18n(SheetUtils.MAX_MERGED_RANGES_COUNT)),

        'autofill:merge:overlap':
            //#. Warning text: trying to trigger autofill over only a part of at least one merged cell
            gt('Autofill over merged cells is not allowed.'),

        'name:invalid':
            // #. Warning text: the user-defined name of a cell range contains invalid characters
            gt('This name contains invalid characters. Please enter another name.'),

        'name:address':
            // #. Warning text: the user-defined name of a cell range cannot look like a cell, e.g. 'A1', or 'R1C1'
            gt('A cell address cannot be used as name. Please enter another name.'),

        'name:used':
            // #. Warning text: the user-defined name of a cell range is already used
            gt('This name is already used. Please enter another name.'),

        'table:multiselect':
            //#. Warning text: tried to create a table range from a cell multi-selection
            gt('Table cannot be created on multiple cell ranges.'),

        'table:overlap':
            //#. Warning text: tried to create a table range above cells that are already filtered
            gt('Table cannot be created on another filtered range.'),

        'table:move':
            //#. Warning text: tried to move a few cells of a table range to the right, or down, etc.
            gt('Parts of a table cannot be moved.'),

        'table:change':
            //#. Warning text: tried to change (overwrite, merge, etc.) parts of a table range
            gt('Parts of a table cannot be changed.'),

        'table:header:change':
            //#. Warning text: tried to change (overwrite, merge, etc.) the header cells of a table range
            gt('The header cells of a table cannot be changed.'),

        'autofilter:multiselect':
            //#. Warning text: tried to create an auto filter from a cell multi-selection
            gt('Filter cannot be applied to multiple cell ranges.'),

        'autofilter:overlap':
            //#. Warning text: tried to create an auto filter above cells that are already filtered
            gt('Filter cannot be applied to another filtered range.'),

        'autofilter:blank':
            //#. Warning text: tried to filter an empty cell range
            gt('Filtering cannot be performed on blank cells.'),

        'autofilter:move':
            //#. Warning text: tried to move a few cells of an auto filter to the right, or down, etc.
            gt('Parts of the filter cannot be moved.'),

        'sort:multiselect':
            //#. Warning text: tried to sort data of a cell multi-selection
            gt('Sorting cannot be performed on multiple cell ranges.'),

        'sort:blank':
            //#. Warning text: tried to sort an empty cell range
            gt('Sorting cannot be performed on blank cells.'),

        'sort:overflow':
            //#. Warning text: tried to sort too many rows or columns
            gt('Sorting cannot be performed on more than %1$d columns or rows.', _.noI18n(SheetUtils.MAX_SORT_LINES_COUNT)),

        'search:finished':
            //#. Information text: no matching text in document
            gt('No more results for your search.'),

        'replace:nothing':
            //#. Information text: search&replace did not find anything to replace
            gt('Nothing to replace.'),

        'formula:invalid':
            //#. Warning text: a spreadsheet formula entered in a cell or in a dialog text field contains an error
            gt('The formula contains an error.'),

        'formula:matrix:change':
            gt('Parts of a matrix formula cannot be changed.'),

        'formula:matrix:insert':
            gt('It is not possible to insert cells into a matrix formula.'),

        'formula:matrix:delete':
            gt('Parts of a matrix formula cannot be deleted.'),

        'formula:matrix:size':
            //#. Warning text: new matrix formulas must not exceed specific height and width
            gt('It is not possible to insert a matrix formula with more than %1$d elements.', _.noI18n(FormulaUtils.Matrix.MAX_SIZE)),

        'formula:matrix:merged':
            gt('It is not possible to insert a matrix formula over merged cells.'),

        'validation:source':
            //#. Warning text: the values for a drop-down list attached to a spreadsheet cell could not be found
            gt('No source data available for the drop-down list.'),

        'paste:overflow':
            gt('It is not possible to paste more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)),

        'paste:outside':
            gt('It is not possible to paste outside of the valid sheet area.'),

        'paste:locked':
            gt('Pasting into protected cells is not allowed.'),

        'drop:unsupported':
            //#. Warning text: file type dropped from outside into the document not supported
            gt('File type not supported.'),

        'drawing:insert:locked':
            gt('Objects cannot be inserted into a protected sheet.'),

        'drawing:delete:locked':
            gt('Objects on a protected sheet cannot be deleted.'),

        'drawing:change:locked':
            gt('Objects on a protected sheet cannot be changed.')

    };

    // the size of the freeze separator nodes, in pixels
    var FROZEN_SPLIT_SIZE = 1;

    // the size of the split separator nodes, in pixels
    var DYNAMIC_SPLIT_SIZE = 2;

    // the highlighted inner size of the split tracking node, in pixels
    var TRACKING_SPLIT_SIZE = 4;

    // the additional margin of the split tracking nodes, in pixels
    var TRACKING_MARGIN = 3;

    // the position offset of tracking nodes compared to split lines
    var TRACKING_OFFSET = (TRACKING_SPLIT_SIZE - DYNAMIC_SPLIT_SIZE) / 2 + TRACKING_MARGIN;

    // default settings for a pane side
    var DEFAULT_PANE_SIDE_SETTINGS = {
        offset: 0,
        size: 1,
        frozen: false,
        showOppositeScroll: false,
        hiddenSize: 0
    };

    // class SpreadsheetView ==================================================

    /**
     * Represents the entire view of a spreadsheet document. Contains the view
     * panes of the sheet currently shown (the 'active sheet'), which contain
     * the scrollable cell grids (there will be several view panes, if the
     * sheet view is split or frozen); and the selections of all existing
     * sheets.
     *
     * Additionally to the events triggered by the base class EditView, an
     * instance of this class triggers events if global contents of the
     * document, or the contents of the current active sheet have been changed.
     * The following events will be triggered:
     * - 'change:viewattributes'
     *      After the global view attributes of the document model have been
     *      changed.
     * - 'before:activesheet'
     *      Before another sheet will be activated. Event handlers receive the
     *      zero-based index of the old active sheet in the document.
     * - 'change:activesheet'
     *      After another sheet has been activated. Event handlers receive the
     *      zero-based index of the new active sheet in the document, and the
     *      sheet model instance of the new active sheet.
     * - 'preview:activesheet'
     *      Triggered during the import process when the active sheet has been
     *      loaded, and import of all other sheets starts.
     * - 'change:sheets'
     *      After the collection of all sheets in the document has been
     *      changed (after inserting, deleting, moving, renaming, hiding a
     *      visible, or showing a hidden sheet).
     * - 'update:selection:data'
     *      After the selection data (active cell settings, mixed borders,
     *      subtotal results) have been updated.
     * - 'sheet:triggered'
     *      Any event from the active sheet will be forwarded with this event.
     *      Event handlers receive the type name of the event, and all other
     *      parameters sent with the original sheet event.
     * - 'change:sheet:attributes'
     *      After the formatting attributes of the active sheet have been
     *      changed.
     * - 'change:sheet:viewattributes'
     *      After the view attributes of the active sheet have been changed.
     *      Event handlers receive an incomplete attribute map containing all
     *      changed view attributes of the active sheet with their new values.
     * - 'change:selection:prepare'
     *      Will be triggered directly before the 'change:sheet:viewattributes'
     *      event caused by a changed selection. Event handlers receive the new
     *      selection (an instance of the class SheetSelection).
     * - 'change:selection'
     *      Will be triggered directly after the 'change:sheet:viewattributes'
     *      event caused by a changed selection, for convenience. Event
     *      handlers receive the new selection (an instance of the class
     *      SheetSelection).
     * - 'celledit:enter', 'celledit:leave'
     *      After entering or leaving the cell in-place edit mode.
     * - 'celledit:change'
     *      After changing text while cell in-place edit mode is active.
     * - 'refresh:ranges'
     *      Intended to initiate updating the visual representation of the
     *      notified cell ranges in the active sheet.
     * - 'insert:columns', 'delete:columns', 'change:columns'
     *      The respective event forwarded from the column collection of the
     *      active sheet. See class ColRowCollection for details.
     * - 'insert:rows', 'delete:rows', 'change:rows'
     *      The respective event forwarded from the row collection of the
     *      active sheet. See class ColRowCollection for details.
     * - 'insert:merged', 'delete:merged'
     *      The respective event forwarded from the merge collection of the
     *      active sheet. See class MergeCollection for details.
     * - 'change:cells', 'move:cells', 'change:usedarea'
     *      The respective event forwarded from the cell collection of the
     *      active sheet. See class CellCollection for details.
     *   'insert:rule', 'delete:rule', 'change:rule'
     *      The respective events forwarded from the collection of conditional
     *      formattings of the active sheet. See class CondFormatCollection for
     *      details.
     * - 'insert:table', 'change:table', 'delete:table'
     *      The respective event forwarded from the table collection of the
     *      active sheet. See class TableCollection for details.
     * - 'insert:drawing', 'delete:drawing', 'change:drawing'
     *      The respective event forwarded from the drawing collection of the
     *      active sheet. See class SheetDrawingCollection for details.
     * - 'change:highlight:ranges'
     *      After the highlighted ranges in this document view have been
     *      changed. See mix-in class HighlightMixin for more details about
     *      highlighted ranges.
     * - 'render:cellselection'
     *      After the cell selection of the active grid pane has actually been
     *      rendered into the DOM selection layer node (rendering may happen
     *      debounced after several 'change:selection' view events).
     * - 'render:drawingselection'
     *      After the drawing selection of the active grid pane has actually
     *      been rendered into the DOM drawing layer (rendering may happen
     *      debounced after several 'change:selection' view events). Event
     *      handlers receive the DOM drawing frames currently selected in the
     *      active grid pane, as jQuery collection.
     *
     * @constructor
     *
     * @extends EditView
     * @extends SelectionMixin
     * @extends HighlightMixin
     * @extends CellEditMixin
     * @extends ViewFuncMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this spreadsheet view.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model created by the passed application.
     */
    function SpreadsheetView(app, docModel) {

        // self reference
        var self = this;

        // the scrollable root DOM node containing the headers and panes
        var rootNode = $('<div class="abs pane-root">');

        // the top-left corner pane
        var cornerPane = null;

        // settings for all pane sides, mapped by pane side identifier
        var paneSideSettings = {};

        // the row/column header panes, mapped by pane side identifiers
        var headerPaneMap = new SimpleMap();

        // the grid panes, mapped by pane position identifiers
        var gridPaneMap = new SimpleMap();

        // the status pane with the sheet tabs
        var statusPane = null;

        // false while grid panes are not initialized (influences focus handling)
        var panesInitialized = false;

        // the floating menu for defined names
        var definedNamesMenu = null;

        // the floating menu for chart axis and labels formatting
        var chartLabelsMenu = null;

        // rendering cache for cell formatting and text contents
        var renderCache = null;

        // the split line separating the left and right panes
        var colSplitLineNode = $('<div class="split-line columns">');

        // the split line separating the top and bottom panes
        var rowSplitLineNode = $('<div class="split-line rows">');

        // the split tracking node separating left and right panes
        var colSplitTrackingNode = $('<div class="abs split-tracking columns" style="width:' + TRACKING_SPLIT_SIZE + 'px;padding:0 ' + TRACKING_MARGIN + 'px;" tabindex="-1">');

        // the split tracking node separating top and bottom panes
        var rowSplitTrackingNode = $('<div class="abs split-tracking rows" style="height:' + TRACKING_SPLIT_SIZE + 'px;padding:' + TRACKING_MARGIN + 'px 0;" tabindex="-1">');

        // the split tracking node covering the intersection of the other tracking points
        var centerSplitTrackingNode = $('<div class="abs split-tracking columns rows" style="width:' + TRACKING_SPLIT_SIZE + 'px;height:' + TRACKING_SPLIT_SIZE + 'px;padding:' + TRACKING_MARGIN + 'px;" tabindex="-1">');

        // all split tracking nodes, as jQuery collection
        var allSplitTrackingNodes = colSplitTrackingNode.add(rowSplitTrackingNode).add(centerSplitTrackingNode);

        // tracking overlay nodes for column/row resizing
        var resizeOverlayNode = $('<div class="abs resize-tracking"><div class="abs leading"></div><div class="abs trailing"></div></div>');

        // the current status text label
        var statusText = null;

        // storage for last known touch position (to open context menu on the correct position)
        var touchX = null;
        var touchY = null;

        // the collections of visible and hidden sheets
        var visibleSheets = [];
        var hiddenSheets = [];

        // the index and model of the active sheet
        var activeSheet = -1;
        var activeSheetModel = null;

        // collections of the active sheet
        var colCollection = null;
        var rowCollection = null;
        var mergeCollection = null;
        var cellCollection = null;
        var tableCollection = null;
        var validationCollection = null;
        var condFormatCollection = null;
        var drawingCollection = null;

        // base constructors --------------------------------------------------

        EditView.call(this, app, docModel, {
            initHandler: initHandler,
            initDebugHandler: initDebugHandler,
            initGuiHandler: initGuiHandler,
            initDebugGuiHandler: initDebugGuiHandler,
            grabFocusHandler: grabFocusHandler
        });

        // private methods ----------------------------------------------------

        /**
         * Updates the layer intervals in all header panes, passes the changed
         * layer intervals to the rendering cache, and forwards the new layer
         * ranges received from the rendering cache to all grid panes.
         *
         * @param {IntervalArray|Null} colIntervals
         *  An array with column intervals specifying all dirty columns in the
         *  sheet that must be recalculated and repainted (e.g. after one or
         *  more document operations such as inserting or deleting columns).
         *
         * @param {IntervalArray|Null} rowIntervals
         *  An array with row intervals specifying all dirty rows in the sheet
         *  that must be recalculated and repainted (e.g. after one or more
         *  document operations such as inserting or deleting rows).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hide=false]
         *      If set to true, all layers will be explicitly hidden.
         *  @param {Boolean} [options.full=false]
         *      If set to true, all cells in the entire sheet are dirty and
         *      must be recalculated and repainted (e.g. when zooming). This
         *      option will be ignored, if the option 'hide' has been set.
         *  @param {Boolean} [options.force=false]
         *      If set to true, the layer intervals will always be recalculated
         *      regardless of the current scroll position. In difference to the
         *      option 'full', this option does not cause a full repaint.
         *      This option will be ignored, if the option 'hide' has been set.
         */
        var updateRenderingLayers = RenderUtils.profileMethod('SpreadsheetView.updateRenderingLayers()', function (colIntervals, rowIntervals, options) {

            // invalidate the entire sheet, or calculate the dirty ranges from the passed intervals
            var dirtyRanges = new RangeArray();
            if (Utils.getBooleanOption(options, 'full', false)) {
                dirtyRanges.push(docModel.getSheetRange());
            } else {
                if (colIntervals) { dirtyRanges.append(docModel.makeColRanges(colIntervals)); }
                if (rowIntervals) { dirtyRanges.append(docModel.makeRowRanges(rowIntervals)); }
                dirtyRanges = dirtyRanges.merge();
            }

            // force updating the layer intervals, if columns or rows in the sheet are dirty
            if (!dirtyRanges.empty()) {
                RenderUtils.log('dirty ranges: ' + dirtyRanges);
                (options || (options = {})).force = true;
            }

            // update the layer intervals of all header panes
            var headerBoundaryMap = new SimpleMap();
            headerPaneMap.forEach(function (headerPane, paneSide) {
                var boundary = headerPane.updateLayerInterval(options);
                if (boundary) { headerBoundaryMap.insert(paneSide, boundary); }
            });

            // nothing to do without changed intervals
            if (headerBoundaryMap.empty()) { return; }

            // pass all layer boundaries to the rendering cache, result contains all changed layer ranges
            var layerBoundaryMap = renderCache.updateLayerRanges(headerBoundaryMap, dirtyRanges);

            // distribute all changed layer ranges to the grid panes
            layerBoundaryMap.forEach(function (boundary, panePos) {
                gridPaneMap.get(panePos).setLayerRange(boundary, dirtyRanges);
            });
        });

        /**
         * Refreshes the visibility, position, and size of all header panes,
         * grid panes, and tracking nodes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.full=false]
         *      If set to true, all cells in the entire sheet are dirty and
         *      must be recalculated and repainted (e.g. when zooming).
         *  @param {Boolean} [options.force=false]
         *      If set to true, the layer intervals will always be recalculated
         *      regardless of the current scroll position. In difference to the
         *      option 'full', this option does not cause a full repaint.
         */
        var initializePanes = (function () {

            // collect flags from multiple debounced calls
            var cachedOptions = {};

            // collects the options passed to multiple invocations
            function registerInvocation(options) {
                cachedOptions.full = cachedOptions.full || Utils.getBooleanOption(options, 'full', false);
                cachedOptions.force = cachedOptions.force || Utils.getBooleanOption(options, 'force', false);
                // ignore updates of active pane caused by focus changes during pane initialization
                panesInitialized = false;
            }

            // initializes the header and grid panes
            var initializePanes = RenderUtils.profileMethod('SpreadsheetView.initializePanes()', function () {

                // view layout updates may be requested in case the import has failed, and no sheet is available
                if (!activeSheetModel) { return; }

                // whether the focus is located in the grid panes
                var hasFocus = self.hasAppFocus();
                // whether frozen split mode is active
                var frozenSplit = self.hasFrozenSplit();
                // whether dynamic split mode is really active
                var dynamicSplit = !frozenSplit && self.hasSplit();
                // the size of the split lines
                var splitLineSize = frozenSplit ? FROZEN_SPLIT_SIZE : DYNAMIC_SPLIT_SIZE;
                // start position of the split lines
                var splitLineLeft = activeSheetModel.getSplitWidth();
                var splitLineTop = activeSheetModel.getSplitHeight();

                // expand dynamic splits to minimum split size
                if (dynamicSplit) {
                    if (splitLineLeft > 0) { splitLineLeft = Math.max(splitLineLeft, PaneUtils.MIN_PANE_SIZE); }
                    if (splitLineTop > 0) { splitLineTop = Math.max(splitLineTop, PaneUtils.MIN_PANE_SIZE); }
                }

                // whether the left and top panes are visible
                var visibleSides = {};
                visibleSides.left = (dynamicSplit || frozenSplit) && (splitLineLeft > 0);
                visibleSides.top = (dynamicSplit || frozenSplit) && (splitLineTop > 0);

                // calculate current size of header nodes
                // TODO: use current maximum row index visible in the panes
                cornerPane.initializePaneLayout(docModel.getMaxRow());
                var headerWidth = self.getHeaderWidth();
                var headerHeight = self.getHeaderHeight();

                // calculate inner width of left panes
                if (visibleSides.left) {
                    paneSideSettings.left.offset = headerWidth;
                    paneSideSettings.left.size = splitLineLeft;
                    paneSideSettings.left.hiddenSize = frozenSplit ? colCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorLeft')) : 0;
                } else {
                    _.extend(paneSideSettings.left, { offset: 0, size: 0, hiddenSize: 0 });
                }

                // calculate inner height of top panes
                if (visibleSides.top) {
                    paneSideSettings.top.offset = headerHeight;
                    paneSideSettings.top.size = splitLineTop;
                    paneSideSettings.top.hiddenSize = frozenSplit ? rowCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorTop')) : 0;
                } else {
                    _.extend(paneSideSettings.top, { offset: 0, size: 0, hiddenSize: 0 });
                }

                // calculate effective position of split lines
                splitLineLeft = paneSideSettings.left.offset + paneSideSettings.left.size;
                splitLineTop = paneSideSettings.top.offset + paneSideSettings.top.size;

                // determine whether right and bottom panes are visible (must have enough room in split and frozen mode)
                visibleSides.right = !visibleSides.left || (splitLineLeft + splitLineSize + PaneUtils.MIN_PANE_SIZE + Utils.SCROLLBAR_WIDTH <= rootNode.width());
                visibleSides.bottom = !visibleSides.top || (splitLineTop + splitLineSize + PaneUtils.MIN_PANE_SIZE + Utils.SCROLLBAR_HEIGHT <= rootNode.height());

                // visibility of the split lines
                var colSplit = visibleSides.left && visibleSides.right;
                var rowSplit = visibleSides.top && visibleSides.bottom;

                // calculate the resulting grid pane positions and sizes
                if (visibleSides.right) {
                    paneSideSettings.right.offset = colSplit ? (splitLineLeft + splitLineSize) : headerWidth;
                    paneSideSettings.right.size = rootNode.width() - paneSideSettings.right.offset;
                    paneSideSettings.right.hiddenSize = frozenSplit ? (paneSideSettings.left.hiddenSize + paneSideSettings.left.size) : 0;
                } else if (visibleSides.left) {
                    paneSideSettings.left.size = rootNode.width() - headerWidth;
                }
                if (visibleSides.bottom) {
                    paneSideSettings.bottom.offset = rowSplit ? (splitLineTop + splitLineSize) : headerHeight;
                    paneSideSettings.bottom.size = rootNode.height() - paneSideSettings.bottom.offset;
                    paneSideSettings.bottom.hiddenSize = frozenSplit ? (paneSideSettings.top.hiddenSize + paneSideSettings.top.size) : 0;
                } else if (visibleSides.top) {
                    paneSideSettings.top.size = rootNode.height() - headerHeight;
                }

                // set frozen mode (left/top panes are not scrollable in frozen mode in their
                // own direction, e.g. left frozen panes are not scrollable to left/right)
                paneSideSettings.left.frozen = frozenSplit;
                paneSideSettings.right.frozen = false;
                paneSideSettings.top.frozen = frozenSplit;
                paneSideSettings.bottom.frozen = false;

                // set up scroll bar visibility in the opposite direction (e.g. left/right
                // scroll bars of top panes are hidden, if bottom panes are visible)
                paneSideSettings.left.showOppositeScroll = !visibleSides.right;
                paneSideSettings.right.showOppositeScroll = true;
                paneSideSettings.top.showOppositeScroll = !visibleSides.bottom;
                paneSideSettings.bottom.showOppositeScroll = true;

                // initialize the header panes
                headerPaneMap.forEach(function (headerPane, paneSide) {
                    headerPane.initializePaneLayout(paneSideSettings[paneSide]);
                });

                // initialize the grid panes
                gridPaneMap.forEach(function (gridPane, panePos) {
                    gridPane.initializePaneLayout(paneSideSettings[PaneUtils.getColPaneSide(panePos)], paneSideSettings[PaneUtils.getRowPaneSide(panePos)]);
                });

                // update layer intervals and ranges, and the rendering cache
                updateRenderingLayers(null, null, cachedOptions);

                // visibility and position of the split lines
                colSplitLineNode.toggle(colSplit).css({ left: splitLineLeft, width: splitLineSize });
                rowSplitLineNode.toggle(rowSplit).css({ top: splitLineTop, height: splitLineSize });

                // visibility and position of the split tracking nodes
                colSplitTrackingNode.toggle(dynamicSplit && colSplit).css({ left: splitLineLeft - TRACKING_OFFSET });
                rowSplitTrackingNode.toggle(dynamicSplit && rowSplit).css({ top: splitLineTop - TRACKING_OFFSET });
                centerSplitTrackingNode.toggle(dynamicSplit && colSplit && rowSplit).css({ left: splitLineLeft - TRACKING_OFFSET, top: splitLineTop - TRACKING_OFFSET });

                // reset cached settings for next debounced invocations
                cachedOptions = {};
                panesInitialized = true;

                // update focus (active grid pane may have been changed)
                if (hasFocus) { self.grabFocus(); }
            });

            // return a debounced method that waits for document actions/operations before initializing the panes
            return app.createDebouncedActionsMethodFor(self, registerInvocation, initializePanes);
        }());

        /**
         * Triggers all events related to changed view attributes of the active
         * sheet.
         */
        function triggerChangeSheetViewAttributes(newAttributes) {
            var selection = newAttributes.selection;
            if (selection) { self.trigger('change:selection:prepare', selection); }
            self.trigger('change:sheet:viewattributes', newAttributes);
            if (selection) { self.trigger('change:selection', selection); }
        }

        /**
         * Updates the collection of visible sheets.
         */
        function updateVisibleSheets() {

            visibleSheets = [];
            hiddenSheets = [];

            // collect all supported and visible sheets
            var sheetIt = docModel.createSheetIterator();
            IteratorUtils.forEach(sheetIt, function (sheetModel, result) {

                // ignore unsupported sheet types
                if (!self.isSheetTypeSupported(sheetModel.getType())) { return; }

                // collect visible and hidden sheets, count hidden sheets
                var array = sheetModel.getMergedAttributeSet(true).sheet.visible ? visibleSheets : hiddenSheets;
                array.push({ sheet: result.sheet, name: result.name });
            });
        }

        /**
         * Returns the array index of the specified sheet in the collection of
         * all visible sheets.
         */
        function getVisibleIndex(sheet) {
            var arrayIndex = _.sortedIndex(visibleSheets, { sheet: sheet }, 'sheet');
            return ((arrayIndex >= 0) && (arrayIndex < visibleSheets.length) && (visibleSheets[arrayIndex].sheet === sheet)) ? arrayIndex : -1;
        }

        /**
         * Event handler for changed view attributes of the document model.
         * Updates internal settings after changing the active sheet, and
         * triggers 'before:activesheet', 'change:activesheet', and
         * 'change:viewattributes' events.
         */
        function changeViewAttributesHandler(event, changedAttributes) {

            if ('activeSheet' in changedAttributes) {

                // nothing to do if the sheet index is not valid (e.g. initially after import,
                // or during sheet operations such as inserting or deleting a sheet)
                if (activeSheet >= 0) {
                    // leave cell edit mode before switching sheets (drop invalid formulas)
                    self.leaveCellEditMode('force');
                    // hide contents in all header and grid panes immediately
                    updateRenderingLayers(null, null, { hide: true });
                    // notify listeners with the index of the old active sheet
                    self.trigger('before:activesheet', activeSheet);
                }

                // initialize the view according to the new active sheet, and notify listeners
                // (nothing to do when invalidating the active sheet temporarily, e.g. during
                // sheet operations such as inserting or deleting a sheet)
                if (changedAttributes.activeSheet >= 0) {

                    // get model instance and collections of the new active sheet
                    activeSheet = changedAttributes.activeSheet;
                    activeSheetModel = docModel.getSheetModel(activeSheet);
                    colCollection = activeSheetModel.getColCollection();
                    rowCollection = activeSheetModel.getRowCollection();
                    mergeCollection = activeSheetModel.getMergeCollection();
                    cellCollection = activeSheetModel.getCellCollection();
                    tableCollection = activeSheetModel.getTableCollection();
                    validationCollection = activeSheetModel.getValidationCollection();
                    condFormatCollection = activeSheetModel.getCondFormatCollection();
                    drawingCollection = activeSheetModel.getDrawingCollection();

                    // notify listeners
                    self.trigger('change:activesheet', activeSheet, activeSheetModel);
                    triggerChangeSheetViewAttributes(activeSheetModel.getViewAttributes());

                    // refresh the layout of the header and grid panes
                    initializePanes();
                }
            }

            self.trigger('change:viewattributes', changedAttributes);
        }

        /**
         * Prepares insertion of a sheet into the document.
         */
        function beforeInsertSheetHandler(event, sheet) {
            // active sheet will change, if a sheet will be inserted before it
            if (sheet <= activeSheet) {
                docModel.setActiveSheet(-1);
            }
        }

        /**
         * Finalizes insertion of a sheet into the document.
         */
        function afterInsertSheetHandler(event, sheet, sheetModel) {

            // forwards all sheet events to own listeners while the sheet is active
            function forwardActiveSheetEvents(eventTypes, triggerType) {
                self.listenTo(sheetModel, eventTypes, function (event2) {
                    if (sheetModel === activeSheetModel) {
                        var eventType = triggerType || event2.type;
                        self.trigger.apply(self, [eventType].concat(_.toArray(arguments).slice(1)));
                    }
                });
            }

            // registers a callback for a sheet event that will be invoked while the sheet is active
            function handleActiveSheetEvent(eventType, callback) {
                self.listenTo(sheetModel, eventType, function () {
                    if (sheetModel === activeSheetModel) {
                        callback.apply(self, _.toArray(arguments).slice(1));
                    }
                });
            }

            // listen to changes in the sheet, notify listeners of the view
            forwardActiveSheetEvents('change:attributes', 'change:sheet:attributes');
            handleActiveSheetEvent('change:viewattributes', triggerChangeSheetViewAttributes);
            forwardActiveSheetEvents('refresh:ranges');
            forwardActiveSheetEvents('insert:columns delete:columns change:columns');
            forwardActiveSheetEvents('insert:rows delete:rows change:rows');
            forwardActiveSheetEvents('insert:merged delete:merged');
            forwardActiveSheetEvents('change:cells move:cells change:usedarea');
            forwardActiveSheetEvents('insert:rule delete:rule change:rule');
            forwardActiveSheetEvents('insert:table delete:table change:table');
            forwardActiveSheetEvents('insert:drawing delete:drawing change:drawing');

            // a generic 'triggered' event for any event of the active sheet
            forwardActiveSheetEvents('triggered', 'sheet:triggered');

            // update the collection of visible sheets
            updateVisibleSheets();

            // update active sheet index
            if (sheet <= activeSheet) {
                docModel.setActiveSheet(activeSheet + 1);
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Prepares deletion of a sheet from the document.
         */
        function beforeDeleteSheetHandler(event, sheet) {

            // Bug 31772: cancel cell edit mode to prevent generating additional operations
            if (activeSheetModel) {
                self.cancelCellEditMode();
            }

            // active sheet will change, if a sheet before will be deleted, or itself
            if (sheet <= activeSheet) {
                docModel.setActiveSheet(-1);
            }
        }

        /**
         * Finalizes deletion of a sheet from the document.
         */
        function afterDeleteSheetHandler(event, sheet) {

            // update the collection of visible sheets
            updateVisibleSheets();

            // update active sheet index
            if (sheet < activeSheet) {
                // sheet before active sheet deleted: decrease index of active sheet
                docModel.setActiveSheet(activeSheet - 1);
            } else if (sheet === activeSheet) {
                // active sheet deleted: keep index of active sheet (unless it was the last sheet)
                // but use the own 'setActiveSheet' method to look for a visible sheet!
                self.setActiveSheet(activeSheet);
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Prepares moving a sheet inside the document.
         */
        function beforeMoveSheetHandler() {
            // Bug 31772: cancel cell edit mode to prevent generating additional operations
            if (activeSheetModel) {
                self.cancelCellEditMode();
            }
        }

        /**
         * Finalizes moving a sheet inside the document.
         */
        function afterMoveSheetHandler(event, from, to) {

            // update the collection of visible sheets
            updateVisibleSheets();

            // update the active sheet index
            if (activeSheet === from) {
                // active sheet has been moved directly
                docModel.setActiveSheet(to);
            } else if ((from < activeSheet) && (activeSheet <= to)) {
                // active sheet has been moved backwards indirectly
                docModel.setActiveSheet(activeSheet - 1);
            } else if ((to <= activeSheet) && (activeSheet < from)) {
                // active sheet has been moved forwards indirectly
                docModel.setActiveSheet(activeSheet + 1);
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Handles a renamed sheet.
         */
        function renameSheetHandler() {

            // update the collection of visible sheets
            updateVisibleSheets();

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Updates the visible sheets collection, if the visibility of any
         * sheet in the document has been changed.
         */
        function changeSheetAttributesHandler(event, sheet, newAttributes, oldAttributes) {

            // index of the active sheet in the (old) set of visible sheets
            var activeVisibleSheet = self.getActiveSheet({ visible: true });

            // do nothing, if the visibility of the sheet has not been changed
            if (newAttributes.sheet.visible === oldAttributes.sheet.visible) { return; }

            // update the collection of visible sheets
            updateVisibleSheets();

            // activate another sheet, if the active sheet has been hidden
            if ((sheet === activeSheet) && !newAttributes.sheet.visible && (visibleSheets.length > 0)) {
                activeVisibleSheet = Utils.minMax(activeVisibleSheet, 0, visibleSheets.length - 1);
                self.setActiveSheet(activeVisibleSheet, { visible: true });
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Processes column/row operations that have been applied in the active
         * sheet. Collects the dirty column/row indexes in from the operations,
         * and updates the layer intervals of all visible header panes, which
         * will trigger all dependent rendering.
         */
        var changeColRowHandler = (function () {

            // all columns intervals to be refreshed
            var colIntervals = new IntervalArray();
            // all row intervals to be refreshed
            var rowIntervals = new IntervalArray();

            // collects the minimum column/row index changed by an operation
            function registerOperation(event, interval, attributes, styleId, changeFlags) {

                // collect the minimum column/row index changed by an operation
                switch (event.type) {
                    case 'insert:columns':
                    case 'delete:columns':
                        colIntervals.push(new Interval(interval.first, docModel.getMaxCol()));
                        break;

                    case 'change:columns':
                        if (changeFlags.size) {
                            colIntervals.push(new Interval(interval.first, docModel.getMaxCol()));
                            // update size of frozen panes after changing column size or visibility
                            if (self.hasFrozenSplit()) { initializePanes(); }
                        } else {
                            colIntervals.push(interval);
                        }
                        break;

                    case 'insert:rows':
                    case 'delete:rows':
                        rowIntervals.push(new Interval(interval.first, docModel.getMaxRow()));
                        break;

                    case 'change:rows':
                        if (changeFlags.size) {
                            rowIntervals.push(new Interval(interval.first, docModel.getMaxRow()));
                            // update size of frozen panes after changing row size or visibility
                            if (self.hasFrozenSplit()) { initializePanes(); }
                        } else {
                            rowIntervals.push(interval);
                        }
                        break;

                    default:
                        Utils.error('SpreadsheetView.changeColRowHandler(): unknown event "' + event.type + '"');
                        return;
                }
            }

            // bug 36014: debounce if model currently applies more operations (e.g. Undo)
            return app.createDebouncedActionsMethodFor(self, registerOperation, function () {
                updateRenderingLayers(colIntervals, rowIntervals);
                colIntervals.clear();
                rowIntervals.clear();
            });
        }());

        /**
         * Handles changed view attributes in the active sheet. Shows a changed
         * zoom factor in the status label and redraws the view with the new
         * zoom factor, and triggers change events for the attributes.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // hide cell selection while drawings are selected
            if ('selection' in attributes) {
                rootNode.toggleClass('drawing-selection', attributes.selection.drawings.length > 0);
            }

            if ('zoom' in attributes) {
                // zooming: initialize panes (size of column/row headers may change)
                initializePanes({ full: true });
            } else if (Utils.hasProperty(attributes, /^split/)) {
                // split/freeze settings have changed, refresh all panes
                initializePanes({ force: true });
            } else if (Utils.hasProperty(attributes, /^anchor/)) {
                // update visible column/row interval immediately while scrolling
                updateRenderingLayers();
            }
        }

        /**
         * Handles all tracking events for the split separators.
         */
        function splitTrackingHandler(event) {

            // the event source node
            var sourceNode = $(this);
            // whether column split tracking and/or row split tracking is active
            var colSplit = sourceNode.hasClass('columns');
            var rowSplit = sourceNode.hasClass('rows');
            // minimum and maximum position of split lines
            var minLeft = self.getHeaderWidth();
            var maxLeft = rootNode.width() - Utils.SCROLLBAR_WIDTH - DYNAMIC_SPLIT_SIZE;
            var minTop = self.getHeaderHeight();
            var maxTop = rootNode.height() - Utils.SCROLLBAR_HEIGHT - DYNAMIC_SPLIT_SIZE;

            // returns the X position of a split line according to the current event
            function getTrackingSplitLeft() {
                var offset = paneSideSettings.left.offset + paneSideSettings.left.size + (colSplit ? event.offsetX : 0);
                return (offset - PaneUtils.MIN_PANE_SIZE < minLeft) ? minLeft : (offset + PaneUtils.MIN_PANE_SIZE > maxLeft) ? maxLeft : offset;
            }

            // returns the Y position of a split line according to the current event
            function getTrackingSplitTop() {
                var offset = paneSideSettings.top.offset + paneSideSettings.top.size + (rowSplit ? event.offsetY : 0);
                return (offset - PaneUtils.MIN_PANE_SIZE < minTop) ? minTop : (offset + PaneUtils.MIN_PANE_SIZE > maxTop) ? maxTop : offset;
            }

            function updateNodeOffset(node, propName, offset, min, max) {
                node.toggleClass('collapsed', (offset <= min) || (offset >= max))
                    .add(centerSplitTrackingNode)
                    .css(propName, (offset - TRACKING_OFFSET) + 'px');
            }

            function finalizeTracking() {
                allSplitTrackingNodes.removeClass('tracking-active');
                self.grabFocus();
            }

            switch (event.type) {

                case 'tracking:start':
                    colSplitTrackingNode.toggleClass('tracking-active', colSplit);
                    rowSplitTrackingNode.toggleClass('tracking-active', rowSplit);
                    break;

                case 'tracking:move':
                    if (colSplit) { updateNodeOffset(colSplitTrackingNode, 'left', getTrackingSplitLeft(), minLeft, maxLeft); }
                    if (rowSplit) { updateNodeOffset(rowSplitTrackingNode, 'top', getTrackingSplitTop(), minTop, maxTop); }
                    break;

                case 'tracking:end':
                    var splitWidth = null, splitHeight = null;
                    if (colSplit) {
                        splitWidth = getTrackingSplitLeft();
                        splitWidth = ((minLeft < splitWidth) && (splitWidth < maxLeft)) ? activeSheetModel.convertPixelToHmm(splitWidth - self.getHeaderWidth()) : 0;
                    } else {
                        splitWidth = activeSheetModel.getSplitWidthHmm();
                    }
                    if (rowSplit) {
                        splitHeight = getTrackingSplitTop();
                        splitHeight = ((minTop < splitHeight) || (splitHeight < maxTop)) ? activeSheetModel.convertPixelToHmm(splitHeight - self.getHeaderHeight()) : 0;
                    } else {
                        splitHeight = activeSheetModel.getSplitHeightHmm();
                    }
                    activeSheetModel.setDynamicSplit(splitWidth, splitHeight);
                    finalizeTracking();
                    break;

                case 'tracking:cancel':
                    initializePanes();
                    finalizeTracking();
                    break;
            }
        }

        /**
         * Handles 'keydown' events from the view root node.
         */
        function keyDownHandler(event) {

            // do nothing while in-place edit mode is active
            if (self.isCellEditMode()) { return; }

            // special handling for drawing selection
            if (self.hasDrawingSelection()) {

                // delete selected drawings
                if (KeyCodes.matchKeyCode(event, 'DELETE') || KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                    if (self.requireEditMode()) {
                        // bug 39533: delete drawings via controller for busy screen etc.
                        self.executeControllerItem('drawing/delete');
                    }
                    return false;
                }

                return;
            }

            // enter cell edit mode with current cell contents
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                self.enterCellEditMode();
                return false;
            }

            // enter cell edit mode with empty text area
            if (!_.browser.MacOS && KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                self.enterCellEditMode({ text: '' });
                return false;
            }

            // clear contents (not formatting) of all sells in the selection
            if (KeyCodes.matchKeyCode(event, 'DELETE') || (_.browser.MacOS && KeyCodes.matchKeyCode(event, 'BACKSPACE'))) {
                if (self.requireEditMode()) {
                    // bug 39533: clear cell range via controller for busy screen etc.
                    self.executeControllerItem('cell/clear/values');
                }
                return false;
            }
        }

        /**
         * Handles 'keypress' events from the view root node. Starts the cell
         * in-place edit mode on-the-fly for valid Unicode characters.
         */
        function keyPressHandler(event) {

            // the initial text to be inserted into the text area
            var initialText = null;

            // do not handle 'keypress' events bubbled up from the active in-place text
            // area, or if drawings are selected
            if (self.isCellEditMode() || self.hasDrawingSelection()) { return; }

            // ignore key events where either CTRL/META or ALT is pressed, ignore
            // SPACE keys with any control keys (used as shortcuts for column/row selection)
            // - Add OS-switch to allow "alt + l" for "@"-sign typing on osx-devices
            if (((event.charCode > 32) && ((_.browser.MacOS && Utils.boolEq(event.ctrlKey, event.metaKey)) || (!_.browser.MacOS && Utils.boolEq(event.ctrlKey || event.metaKey, event.altKey)))) || ((event.charCode === 32) && !KeyCodes.hasModifierKeys(event))) {

                // build string from character code
                initialText = String.fromCharCode(event.charCode);

                // percentage number format: add percent sign to number-like first character
                if ((self.getNumberFormatCategory() === 'percent') && (/^[-+0-9]$/.test(initialText) || (initialText === LocaleData.DEC))) {
                    initialText += '%';
                }

                // start cell edit mode, and drop the key event
                self.enterCellEditMode({ quick: true, text: initialText, pos: 1 });
                return false;
            }
        }

        function globalContextMenuHandler(event) {

            // quit, if there is no event
            if (_.isUndefined(event)) { return false; }

            // quit if the contextmenu event was triggered on an input-/textfield
            if ($(event.target).is('input, textarea')) { return; }

            var // what was clicked? HeaderPane, StatusPane, GridPane, ...
                target              = event.target,
                // was the event triggered by keyboard?
                isGlobalTrigger     = (_.browser.IE) ? $(event.target).is('.clipboard') : Utils.isContextEventTriggeredByKeyboard(event),

                selectedDrawings    = self.getSelectedDrawings(),
                activeGridPane      = self.getActiveGridPane(),
                layerRootNode       = activeGridPane.getLayerRootNode(),
                scrollNode          = layerRootNode.parent(),
                sheetOffset         = layerRootNode.offset(),
                sheetPosition       = layerRootNode.position(),
                // scroll positions needed for scrolling hacking in IE & Firefox
                scrollLeft          = scrollNode.scrollLeft(),
                scrollTop           = scrollNode.scrollTop(),
                // the node, on which the event should be triggered later
                triggerNode         = null;

            // contextmenu event on the header-pane
            if ($(target).is('.header-pane .cell')) {
                triggerNode = $(target);

            // contextmenu event on the status-pane
            } else if ($(target).is('.status-pane .button, .status-pane .button *') || (isGlobalTrigger && $('.status-pane .active-sheet-group.focused .selected').length > 0)) {
                if (isGlobalTrigger && $('.status-pane .active-sheet-group.focused .selected').length > 0) {
                    triggerNode = $('.status-pane .active-sheet-group.focused .button.selected');
                    event.pageX = (event.pageX - sheetPosition.left + sheetOffset.left);
                    event.pageY = (event.pageY + sheetPosition.top + sheetOffset.top);
                } else {
                    if (event.pageX === 0 || event.pageY === 0 || event.pageX === undefined || event.pageY === undefined) {
                        event.pageX = $(target).offset().left;
                        event.pageY = $(target).offset().top;
                    }
                    triggerNode = $(target);
                }

            // contextmenu event on a drawing
            } else if (selectedDrawings.length > 0) {
                triggerNode = activeGridPane.getDrawingFrame(selectedDrawings[0]);

                // if we have a last known touch position, use it
                if (touchX && touchY) {
                    event.pageX = touchX;
                    event.pageY = touchY;

                // otherwise, locate a meaningful position
                } else if (isGlobalTrigger) {
                    var drawingModel    = self.getDrawingCollection().findModel(selectedDrawings[0]),
                        activeDrawing   = drawingModel.getRectangle();

                    triggerNode = activeGridPane.getDrawingFrame(selectedDrawings[0]);

                    event.target = triggerNode;
                    event.pageX = (activeDrawing.left + (activeDrawing.width / 2) - sheetPosition.left + sheetOffset.left - scrollLeft);
                    event.pageY = (activeDrawing.top + (activeDrawing.height / 2) - sheetPosition.top + sheetOffset.top - scrollTop);
                }

            // contextmenu event on a cell
            } else if ($(target).is('.grid-pane *') || $(target).is('html')) {
                triggerNode = activeGridPane.getNode();

                // if we have a last known touch position, use it
                if (touchX && touchY) {
                    event.pageX = touchX;
                    event.pageY = touchY;

                // otherwise, locate a meaningful position
                } else if (isGlobalTrigger) {
                    var activeCell = self.getActiveCell(),
                        activeCellRect = activeCell ? self.getSheetModel().getCellRectangle(activeCell, { expandMerged: true }) : null;

                    event.pageX = (activeCellRect.left + (activeCellRect.width / 2) - sheetPosition.left + sheetOffset.left - scrollLeft);
                    event.pageY = (activeCellRect.top + (activeCellRect.height / 2) - sheetPosition.top + sheetOffset.top - scrollTop);
                }
            }

            if (!_.isNull(triggerNode)) {
                triggerNode.trigger(new $.Event('documents:contextmenu', { sourceEvent: event }));
            }

            return false;
        }

        /**
         * Handles the right mouseclick
         */
        function mouseDownHandler(event) {

            // bug 41094: user wants to enter text, keyboard shows then hides immediately
            if (Utils.IOS && $(event.target).is('canvas.cell-layer')) {
                event.preventDefault();
            }

            // get out here, if not the right mousebutton was clicked
            if (event.button !== 2) { return; }

            var target = $(event.target);

            if (target.is('img') || target.is('canvas.canvasjs-chart-canvas')) {
                var drawing = DrawingFrame.getDrawingNode(target);
                self.selectDrawing([drawing.index()]);

            } else {
                var clickedCellData = self.getActiveGridPane().getCellDataByOffset(event.pageX, event.pageY);
                if (!self.getSelectedRanges().containsAddress(clickedCellData.address)) {
                    self.selectCell(clickedCellData.address);
                }
            }
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            // handle events of the formula dependency manager (show alerts, etc.)
            var dependencyManager = docModel.getDependencyManager();
            // collect the addresses of all formula cells changed locally (for circular reference alert)
            var changedFormulaMaps = new SimpleMap();

            // insert the root node into the application pane
            self.insertContentNode(rootNode);

            // to save the position from 'touchstart', we have to register some global eventhandler
            // we need this position to open the contextmenu on the correct location
            if (_.browser.iOS || _.browser.Android) {
                app.registerGlobalEventHandler($(document), 'touchstart', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = event.originalEvent.changedTouches[0].pageX;
                    touchY = event.originalEvent.changedTouches[0].pageY;
                });
                app.registerGlobalEventHandler($(document), 'touchend', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    // delay, because some contextmenus opens only on 'touchend' (drawing-context-menus for example)
                    self.executeDelayed(function () { touchX = touchY = null; }, { delay: 100, infoString: 'View: initHandler', app: app });
                });
            }

            // update view when document edit mode changes
            self.listenTo(docModel, 'change:editmode', function (event, editMode) {
                rootNode.toggleClass('edit-mode', editMode);
            });

            // set a custom application state during background formula calculation, and show a busy label
            self.listenTo(dependencyManager, 'recalc:start', function () { app.setUserState('recalc'); });
            self.listenTo(dependencyManager, 'recalc:end recalc:cancel', function () { app.setUserState(null); });
            self.registerStatusCaption('recalc', Labels.CALCULATING_LABEL, { busy: true, delay: 1000 });

            // collect all formula cells that have been changed locally
            self.listenTo(docModel, 'change:cells', function (event, sheet, changeDesc, external) {
                if (!external) {
                    var sheetUid = docModel.getSheetModel(sheet).getUid();
                    var addressMap = changedFormulaMaps.getOrConstruct(sheetUid, SimpleMap);
                    changeDesc.formulaCells.forEach(function (address) {
                        addressMap.insert(address.key(), address);
                    });
                }
            });

            // show a warning alert when reference cycles have been found during calculation
            self.listenTo(dependencyManager, 'recalc:end', function (event, referenceCycles) {

                // check if the reference cycles cover a formula cell changed locally
                var hasCycle = referenceCycles.some(function (addresses) {
                    return addresses.some(function (address) {
                        return changedFormulaMaps.with(address.sheetUid, function (addressMap) {
                            return addressMap.has(address.key());
                        });
                    });
                });
                changedFormulaMaps.clear();

                // show an alert box if a new cycle has been found
                if (hasCycle) {
                    self.yell({
                        type: 'info',
                        //#. title for a warning message: a formula addresses its own cell, e.g. =A1 in cell A1
                        headline: gt('Circular Reference'),
                        message: gt('A reference in the formula is dependent on the result of the formula cell.')
                    });
                }
            });

            // show an error message (without timeout) if the document contains too many formulas
            self.listenOnceTo(dependencyManager, 'recalc:overflow', function () {
                self.yell({
                    type: 'error',
                    //#. The number of spreadsheet formulas is limited in the configuration. If a document exceeds this limit, the user will see this alert.
                    message: gt('This document exceeds the spreadsheet size and complexity limits. Calculation has been stopped. Formula results are not updated anymore.')
                });
            });

            // show warning message if the document contains any formulas with unimplemented functions
            self.listenOnceTo(dependencyManager, 'recalc:end', function () {

                // the formula grammar needed to receive translated function names
                var formulaGrammar = docModel.getFormulaGrammar('ui');
                // the resource keys of all unimplemented functions occurring in the document
                var funcKeys = dependencyManager.getMissingFunctionKeys();
                // the translated function names of all unsupported functions
                var funcNames = funcKeys.map(function (funcKey) {
                    var funcName = formulaGrammar.getFunctionName(funcKey);
                    return funcName ? ('\xa0\u2022\xa0' + funcName) : null;
                }).filter(_.identity).sort();

                // show a warning message if the document contains any unsupported functions
                if (funcNames.length > 0) {
                    self.yell({
                        type: 'warning',
                        //#. A list with all unsupported functions will be shown below this message.
                        message: _.noI18n(gt('This spreadsheet contains formulas with unsupported functions:') + '\n' + funcNames.join('\n')),
                        duration: 30000
                    });
                }
            });

            // show a message if user want to sort over a range which contains merged cells
            self.listenTo(docModel, 'sort:merge:overlap', function () {
                self.yell({
                    type: 'info',
                    //#. Warning text: trying to sort over merged cells is not allowed
                    message: gt('Sorting over merged cells is not allowed.')
                });
            });

            // insert resize overlay node first (will increase its z-index while active)
            rootNode.append(resizeOverlayNode);

            // insert the pane nodes into the DOM
            gridPaneMap.forEach(function (gridPane) { rootNode.append(gridPane.getNode()); });
            headerPaneMap.forEach(function (headerPane) { rootNode.append(headerPane.getNode()); });
            rootNode.append(cornerPane.getNode());

            // append the split lines and split tracking nodes
            rootNode.append(colSplitLineNode, rowSplitLineNode, colSplitTrackingNode, rowSplitTrackingNode, centerSplitTrackingNode);

            // handle sheet collection events
            self.listenTo(docModel, 'change:viewattributes', changeViewAttributesHandler)
                .listenTo(docModel, 'insert:sheet:before', beforeInsertSheetHandler)
                .listenTo(docModel, 'insert:sheet:after', afterInsertSheetHandler)
                .listenTo(docModel, 'delete:sheet:before', beforeDeleteSheetHandler)
                .listenTo(docModel, 'delete:sheet:after', afterDeleteSheetHandler)
                .listenTo(docModel, 'move:sheet:before', beforeMoveSheetHandler)
                .listenTo(docModel, 'move:sheet:after', afterMoveSheetHandler)
                .listenTo(docModel, 'rename:sheet', renameSheetHandler)
                .listenTo(docModel, 'change:sheet:attributes', changeSheetAttributesHandler);

            // register GUI event handlers after successful import
            self.waitForImportSuccess(function () {

                // register the context menu handler
                app.registerGlobalEventHandler($(document), 'contextmenu', globalContextMenuHandler);

                // enable tracking and register tracking event handlers
                Tracking.enableTracking(allSplitTrackingNodes);
                allSplitTrackingNodes.on('tracking:start tracking:move tracking:end tracking:cancel', splitTrackingHandler);

                // handle generic key events
                rootNode.on({ keydown: keyDownHandler, keypress: keyPressHandler, mousedown: mouseDownHandler });
            });

            // hide everything, if import fails
            self.waitForImportFailure(function () {
                rootNode.hide();
                statusPane.hide();
            });

            // activate a sheet in the view after the document has been imported (also if import has failed)
            self.waitForImport(function () {

                // index of the active sheet from document attributes
                var activeSheet = docModel.getDocumentAttribute('activeSheet');
                // array index into the visibleSheets array
                var arrayIndex = Math.max(0, Math.min(visibleSheets.length - 1, _.sortedIndex(visibleSheets, { sheet: activeSheet }, 'sheet')));

                // initialize sheet view settings from the imported sheet attributes
                IteratorUtils.forEach(docModel.createSheetIterator(), function (sheetModel) {
                    sheetModel.initializeViewAttributes();
                });

                // activate a sheet (this causes initial painting)
                self.setActiveSheet(arrayIndex, { visible: true });
            });

            // create the status pane
            self.addPane(statusPane = new StatusPane(self));

            // repaint all grid and header panes on layout changes
            self.on('refresh:layout', function () { initializePanes(); });

            // update locked state of active sheet in the DOM
            self.on('change:activesheet change:sheet:attributes', function () {
                rootNode.toggleClass('sheet-locked', activeSheetModel.isLocked());
            });

            // update header panes after column and/or row operations
            self.on('insert:columns delete:columns change:columns insert:rows delete:rows change:rows', changeColRowHandler);

            // update global settings according to view attributes of active sheet
            self.on('change:sheet:viewattributes', changeSheetViewAttributesHandler);

            // update CSS marker class for cell in-place edit mode
            self.on('celledit:enter celledit:leave', function () {
                rootNode.toggleClass('cell-edit', self.isCellEditMode());
            });

            // process events triggered by header panes
            headerPaneMap.forEach(function (headerPane, paneSide) {

                var columns = PaneUtils.isColumnSide(paneSide);
                var offsetAttr = columns ? 'left' : 'top';
                var sizeAttr = columns ? 'width' : 'height';
                var showResizeTimer = null;

                function showResizerOverlayNode(offset, size) {
                    resizeOverlayNode.attr('data-orientation', columns ? 'columns' : 'rows');
                    updateResizerOverlayNode(offset, size);
                    // Delay displaying the overlay nodes, otherwise they cover
                    // the resizer drag nodes of the header pane which will
                    // interfere with double click detection.
                    showResizeTimer = self.executeDelayed(function () {
                        rootNode.addClass('tracking-active');
                    }, { delay: 200, infoString: 'View: showResizerOverlayNode', app: app });
                }

                function updateResizerOverlayNode(offset, size) {
                    var relativeOffset = paneSideSettings[paneSide].offset + offset - headerPane.getVisiblePosition().offset;
                    resizeOverlayNode.toggleClass('collapsed', size === 0);
                    resizeOverlayNode.find('>.leading').css(sizeAttr, (Math.max(0, relativeOffset) + 10) + 'px');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, (relativeOffset + size) + 'px');
                }

                function hideResizerOverlayNode() {
                    if (showResizeTimer) {
                        showResizeTimer.abort();
                        showResizeTimer = null;
                    }
                    rootNode.removeClass('tracking-active');
                    resizeOverlayNode.find('>.leading').css(sizeAttr, '');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, '');
                }

                // visualize resize tracking from header panes
                headerPane.on({
                    'resize:start': function (event, offset, size) {
                        if (self.leaveCellEditMode()) {
                            showResizerOverlayNode(offset, size);
                        } else {
                            event.preventDefault();
                        }
                    },
                    'resize:move': function (event, offset, size) {
                        updateResizerOverlayNode(offset, size);
                    },
                    'resize:end': function () {
                        hideResizerOverlayNode();
                    }
                });
            });
        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoHeader('selection', 'Current sheet selection')
                .addDebugInfoNode('selection', 'sheet', 'Active (visible) sheet')
                .addDebugInfoNode('selection', 'ranges', 'Selected cell ranges')
                .addDebugInfoNode('selection', 'border', 'Mixed border attributes')
                .addDebugInfoNode('selection', 'draw', 'Selected drawing objects')
                .addDebugInfoNode('selection', 'pane', 'Active (focused) grid pane')
                .addDebugInfoNode('selection', 'auto', 'Auto-fill range')
                .addDebugInfoHeader('active', 'Formatting of the active cell')
                .addDebugInfoNode('active', 'col', 'Explicit attributes of column containing active cell')
                .addDebugInfoNode('active', 'row', 'Explicit attributes of row containing active cell')
                .addDebugInfoNode('active', 'cell', 'Explicit attributes of active cell')
                .addDebugInfoNode('active', 'url', 'Hyperlink URL of active cell');

            // log information about the active sheet
            self.on('change:activesheet change:sheets change:usedarea', function () {
                var used = (cellCollection && (cellCollection.getUsedCols() > 0) && (cellCollection.getUsedRows() > 0)) ? new Range(Address.A1, cellCollection.getLastUsedCell()) : '<empty>';
                operationsPane.setDebugInfoText('selection', 'sheet', 'index=' + activeSheet + ', name="' + self.getSheetName() + '", used=' + used);
            });

            // log all selection events
            self.on('change:sheet:viewattributes', function (event, attributes) {
                if ('selection' in attributes) {
                    var selection = attributes.selection;
                    operationsPane
                        .setDebugInfoText('selection', 'ranges', 'count=' + selection.ranges.length + ', ranges=' + selection.ranges + ', active=' + selection.active + ', cell=' + selection.address + (selection.originCell ? (', origin=' + selection.origin) : ''))
                        .setDebugInfoText('selection', 'draw', 'count=' + selection.drawings.length + ', frames=' + ((selection.drawings.length > 0) ? ('[' + selection.drawings.map(JSON.stringify).join() + ']') : '<none>'));
                }
                if ('activePane' in attributes) {
                    operationsPane.setDebugInfoText('selection', 'pane', attributes.activePane);
                }
                if ('autoFillData' in attributes) {
                    var autoFill = attributes.autoFillData;
                    operationsPane.setDebugInfoText('selection', 'auto', autoFill ? ('border=' + autoFill.border + ', count=' + autoFill.count) : '<none>');
                }
            });

            // log formatting of the active cell, and the entire selection
            self.on('update:selection:data', function () {

                function stringifyAutoStyle(styleId) {
                    var attrs = docModel.getCellAutoStyles().getMergedAttributeSet(styleId);
                    return '<span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>style=' + Utils.escapeHTML(styleId || '<default>') + '</span>';
                }

                function stringifyColRowAttrs(desc, family) {
                    return '<span ' + AttributesToolTip.createAttributeMarkup(Utils.makeSimpleObject(family, desc.merged)) + '>attrs=' + Utils.escapeHTML(Utils.stringifyForDebug(desc.explicit)) + '</span>';
                }

                var activeCell = self.getActiveCell();
                var colDesc = colCollection.getEntry(activeCell[0]);
                var rowDesc = rowCollection.getEntry(activeCell[1]);
                var cellModel = cellCollection.getCellModel(activeCell);

                operationsPane
                    .setDebugInfoMarkup('selection', 'border', '<span ' + AttributesToolTip.createAttributeMarkup({ cell: self.getBorderAttributes() }) + '>mixed-borders</span>')
                    .setDebugInfoMarkup('active', 'col', stringifyAutoStyle(colDesc.style) + ', ' + stringifyColRowAttrs(colDesc, 'column'))
                    .setDebugInfoMarkup('active', 'row', stringifyAutoStyle(rowDesc.style) + ', ' + stringifyColRowAttrs(rowDesc, 'row'))
                    .setDebugInfoMarkup('active', 'cell', 'exists=' + !!cellModel + ', ' + stringifyAutoStyle(cellCollection.getStyleId(activeCell)))
                    .setDebugInfoMarkup('active', 'url', cellCollection.getEffectiveURL(activeCell) || '<none>');
            });

            // handle events triggered by the grid panes
            gridPaneMap.forEach(function (gridPane) {

                // process debug events triggered by grid panes
                gridPane.on('debug:clipboard', function (event, content) {
                    clipboardPane.setClipboardContent(content);
                });

                // forward the events triggered by grid panes to own listeners
                gridPane.on('change:scrollpos', function () {
                    self.trigger('change:scrollpos');
                });
            });

            // register an attribute formatter for number formats
            operationsPane.registerAttributeFormatter({ family: 'cell', name: 'formatId' }, function (formatId, markup) {

                var numberFormatter = docModel.getNumberFormatter();
                var parsedFormat = numberFormatter.getParsedFormat(formatId);
                var SECTION_TITLES = ['Section', 'Format code', 'Category', 'Color', 'Operator', 'Boundary'];
                var SECTION_HEADER_MARKUP = '<tr>' + SECTION_TITLES.map(function (title) { return '<th>' + title + '</th>'; }).join('') + '</tr>';

                function addCellMarkup(value, quote, cols) {
                    markup += '<td' + ((cols > 1) ? ' colspan="' + cols + '"' : '') + '>';
                    quote = quote ? '"' : '';
                    if (value !== null) {
                        switch (typeof value) {
                            case 'number':  markup += value; break;
                            case 'boolean': markup += value; break;
                            case 'string':  markup += Utils.escapeHTML(quote + value + quote); break;
                            case 'object':  markup += Utils.escapeHTML(Utils.stringifyForDebug(value)); break;
                            default: markup += Utils.escapeHTML('<invalid>');
                        }
                    }
                    markup += '</td>';
                }

                function addLineMarkup(label, value, quote) {
                    markup += '<tr><th>' + label + '</th>';
                    addCellMarkup(value, quote, SECTION_TITLES.length - 1);
                    markup += '</tr>';
                }

                function addSectionMarkup(label, section) {
                    markup += '<tr>';
                    addCellMarkup(label);
                    addCellMarkup(section.formatCode, true);
                    addCellMarkup(section.category);
                    addCellMarkup(section.colorName);
                    addCellMarkup(section.operator, true);
                    addCellMarkup(section.operator ? section.boundary : null);
                    markup += '</tr>';
                }

                markup = '<table>';
                addLineMarkup('Format ID', formatId);
                addLineMarkup('Format code', parsedFormat.formatCode, true);
                addLineMarkup('Category', parsedFormat.category);
                addLineMarkup('State', parsedFormat.syntaxError ? 'error' : 'valid');
                markup += SECTION_HEADER_MARKUP;
                parsedFormat.numberSections.forEach(function (section, index) {
                    addSectionMarkup('Num' + (index + 1), section);
                });
                if (parsedFormat.textSection) {
                    addSectionMarkup('Text', parsedFormat.textSection);
                }
                markup += '</table>';
                return markup;
            });
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function initGuiHandler(viewMenuGroup) {

            var
                fontToolBar         = null,
                fontStyleToolBar    = null,
                painterToolBar      = null,
                colorToolBar        = null,
                cellToolBar         = null,
                numberformatToolBar = null,
                cellborderToolBar   = null,
                cellstyleToolBar    = null,
                alignmentToolBar    = null,
                dataToolBar         = null,
                insertToolBar       = null,
                rowsColsToolBar     = null,
                drawingToolBar      = null,
                imageToolBar        = null,
                chartStyleToolBar   = null,
                chartDataToolBar    = null,
                chartLegendToolBar  = null,
                chartSourceToolBar  = null,

                pickFontFamily      = new Controls.FontFamilyPicker(self),
                pickFontSize        = new Controls.FontSizePicker(),
                pickTextColor       = new Controls.TextColorPicker(self),
                pickFillColor       = new Controls.FillColorPicker(self, { icon: 'docs-cell-fill-color' }),
                pickAlignmentHor    = new Controls.AlignmentPicker('horizontal'),
                pickAlignmentVer    = new Controls.AlignmentPicker('vertical'),
                pickMergeCells      = new Controls.MergeCellsPicker(this),
                pickFormatCategory  = new Controls.FormatCategoryPicker(this),
                pickFormatCode      = new Controls.FormatCodePicker(this),
                pickCellBorderMode  = new Controls.CellBorderModePicker(self, { dropDownVersion: { label: Labels.CELL_BORDERS_LABEL } }),
                pickBorderStyle     = new Controls.CellBorderStylePicker(self, { dropDownVersion: { label: /*#. line style (solid, dashed, dotted) of borders in paragraphs and tables cells */ gt.pgettext('borders', 'Border style') } }),
                pickCellBorderColor = new Controls.CellBorderColorPicker(self, { dropDownVersion: { label: Labels.BORDER_COLOR_LABEL } }),

                btnInsertRow        = new Controls.Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }),
                btnInsertColumn     = new Controls.Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }),
                btnDeleteRow        = new Controls.Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }),
                btnDeleteColumn     = new Controls.Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }),
                btnDeleteDrawing    = new Controls.Button(Labels.DELETE_DRAWING_BUTTON_OPTIONS),
                btnBold             = new Controls.Button(Labels.BOLD_BUTTON_OPTIONS),
                btnItalic           = new Controls.Button(Labels.ITALIC_BUTTON_OPTIONS),
                btnUnderline        = new Controls.Button(Labels.UNDERLINE_BUTTON_OPTIONS),
                btnStrike           = new Controls.Button(Labels.STRIKEOUT_BUTTON_OPTIONS),
                btnReset            = new Controls.Button(Labels.CLEAR_FORMAT_BUTTON_OPTIONS),
                btnLinebreak        = new Controls.Button({ icon: 'docs-insert-linebreak', tooltip: /*#. checkbox: automatic word wrapping (multiple text lines) in spreadsheet cells */ gt('Automatic word wrap'), toggle: true, dropDownVersion: { label: gt('Automatic word wrap') } }),
                btnFilter           = new Controls.Button(_.extend({ icon: 'docs-filter', smallerVersion: { hideLabel: true } }, Labels.FILTER_OPTIONS)),
                btnFilterRefresh    = new Controls.Button({ icon: 'fa-refresh', label: /*#. Button label: refresh a filtered cell range */ gt.pgettext('filter', 'Reapply'), tooltip: gt('Reapply the filter in the selected cells'), smallerVersion: { hideLabel: true } }),
                btnSum              = new Controls.Button(_.extend({ icon: 'docs-auto-sum', smallerVersion: { css: { width: 35 }, hideLabel: true } }, Labels.INSERT_SUM_OPTIONS)),
                btnInsertFunction   = new Controls.Button(_.extend({ icon: 'docs-functions', smallerVersion: { css: { width: 35 }, hideLabel: true } }, Labels.INSERT_FUNCTION_OPTIONS)),

                grpFormatCategory   = new Controls.FormatCategoryGroup(this),
                sortMenuBtn         = new Controls.SortMenuButton(this),
                namedRangesMenuBtn  = new Controls.NamedRangesMenuButton(this);

            // -----------------------------------------------------
            // TABS
            //      prepare all tabs (for normal or combined panes)
            // -----------------------------------------------------

            var ROWS_COLUMNS_TITLE_LABEL = /*#. menu title: insert/delete/resize rows and columns in the sheet */ gt.pgettext('menu-title', 'Rows/Columns');
            var DATA_TITLE_LABEL = /*#. tool bar title for sorting, filtering, and other data operations */ gt.pgettext('menu-title', 'Data');

            if (!self.panesCombined()) {
                self.createToolBarTab('format',       { label: Labels.FORMAT_HEADER_LABEL, visibleKey: 'document/editable/cell' });
                self.createToolBarTab('data',         { label: DATA_TITLE_LABEL,           visibleKey: 'document/editable/cell' });
                self.createToolBarTab('insert',       { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable/cell' });
                self.createToolBarTab('rowscols',     { label: ROWS_COLUMNS_TITLE_LABEL,   visibleKey: 'document/editable/cell' });

            } else {
                self.createToolBarTab('format',       { label: gt('Font'),                 visibleKey: 'document/editable/cell' });
                self.createToolBarTab('alignment',    { label: gt('Alignment'),            visibleKey: 'document/editable/cell' });
                self.createToolBarTab('cell',         { label: gt('Cell'),                 visibleKey: 'document/editable/cell' });
                self.createToolBarTab('numberformat', { label: gt('Number format'),        visibleKey: 'document/editable/cell' });
                self.createToolBarTab('data',         { label: DATA_TITLE_LABEL,           visibleKey: 'document/editable/cell' });
                self.createToolBarTab('rowscols',     { label: ROWS_COLUMNS_TITLE_LABEL,   visibleKey: 'document/editable/cell' });
            }

            self.createToolBarTab('drawing', { labelKey: 'drawing/type/label', visibleKey: 'document/editable/drawing' });

            // -----------------------------------------------------
            // TOOLBARS
            //      prepare all toolbars (for normal or combined panes)
            // -----------------------------------------------------

            fontToolBar                 = self.createToolBar('format', { priority: 1, prepareShrink: true, icon: 'fa-font', tooltip: Labels.FONT_STYLES_LABEL });
            fontStyleToolBar            = self.createToolBar('format', { priority: 2, prepareShrink: true, icon: 'docs-font-bold', caretTooltip: gt('More font styles'), splitBtn: { key: 'character/bold', options: Controls.BOLD_BUTTON_OPTIONS } });
            painterToolBar              = self.createToolBar('format', { priority: 3 });
            if (!self.panesCombined()) {
                colorToolBar            = self.createToolBar('format', { priority: 4, prepareShrink: true, icon: 'docs-color', tooltip: gt('Colors') });
                cellToolBar             = self.createToolBar('format', { priority: 5, prepareShrink: true, icon: 'fa-align-center', tooltip: Labels.ALIGNMENT_HEADER_LABEL });
                numberformatToolBar     = self.createToolBar('format', { priority: 6, hideable: true, prepareShrink: true, icon: 'docs-percent', tooltip: Labels.NUMBERFORMAT_HEADER_LABEL });
                cellborderToolBar       = self.createToolBar('format', { priority: 7, hideable: true, prepareShrink: true, icon: 'docs-cell-style', tooltip: Labels.CELL_BORDER_LABEL });
                cellstyleToolBar        = self.createToolBar('format', { priority: 8, hideable: true });
            } else {
                alignmentToolBar        = self.createToolBar('alignment',    { prepareShrink: true, icon: 'docs-cell-h-align-auto', tooltip: gt('Alignment') });
                cellToolBar             = self.createToolBar('cell',         { prepareShrink: true, icon: 'fa-table', tooltip: gt('Cell') });
                numberformatToolBar     = self.createToolBar('numberformat', { prepareShrink: true, icon: 'fa-angle-double-down', tooltip: gt('Number format') });
            }
            dataToolBar                 = self.createToolBar('data');
            if (!self.panesCombined()) {
                insertToolBar           = self.createToolBar('insert');
            }
            rowsColsToolBar             = self.createToolBar('rowscols');
            drawingToolBar              = self.createToolBar('drawing');
            if (app.isOOXML()) {
                imageToolBar            = self.createToolBar('drawing', { visibleKey: 'drawing/image' });
            }
            if (!self.panesCombined()) {
                chartStyleToolBar       = self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 1 });
                chartDataToolBar        = self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 2, hideable: true });
                chartLegendToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 3, hideable: true });
                chartSourceToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 4, hideable: true });
            }

            // -----------------------------------------------------
            // CONTROLS
            //      add all controls
            // -----------------------------------------------------

            // FORMAT ------------------------------------------
            fontToolBar
                .addGroup('character/fontname', pickFontFamily)
                .addGap()
                .addGroup('character/fontsize', pickFontSize);

            fontStyleToolBar
                .addGroup('character/bold', btnBold)
                .addGroup('character/italic', btnItalic)
                .addGroup('character/underline', btnUnderline);

            if (!self.panesCombined()) {
                fontStyleToolBar
                    .addSeparator({ classes: 'noVerticalSeparator' });
            }

            fontStyleToolBar
                .addGroup('character/format', new Controls.CompoundButton(self, { icon: 'docs-font-format', tooltip: gt('More font styles'), dropDownVersion: { visible: false } })
                    .addGroup('character/strike', btnStrike)
                    .addSeparator()
                    .addGroup('cell/attributes/clear', btnReset)
                )
                .addGroup('character/strike', btnStrike.clone({ dropDownVersion: { visible: true } }))
                .addSeparator({ classes: 'hidden' })
                .addGroup('cell/attributes/clear', btnReset.clone({ dropDownVersion: { visible: true } }));

            if (!self.panesCombined()) {
                var btnPainter = new Controls.Button({ icon: 'docs-format-painter', tooltip: Labels.FORMAT_PAINTER_LABEL, toggle: true, dropDownVersion: { label: Labels.FORMAT_PAINTER_LABEL } });

                painterToolBar
                    .addGroup('cell/painter', btnPainter);

                colorToolBar
                    .addGroup('character/color', pickTextColor)
                    .addGroup('cell/fillcolor', pickFillColor);

                cellToolBar
                    .addGroup('cell/alignhor', pickAlignmentHor)
                    .addGroup('cell/alignvert', pickAlignmentVer)
                    .addGroup('cell/linebreak', btnLinebreak)
                    .addGroup('cell/merge', pickMergeCells);

                numberformatToolBar
                    .addGroup('cell/numberformat/category', grpFormatCategory)
                    .addGroup('cell/numberformat/category', pickFormatCategory)
                    .addGroup('cell/numberformat/code', pickFormatCode);
            }

            if (self.panesCombined()) {
                painterToolBar
                    .addGroup('character/color', pickTextColor);
            }

            // ALIGNMENT ------------------------------------------
            if (self.panesCombined()) {
                alignmentToolBar
                    .addGroup('cell/alignhor', pickAlignmentHor)
                    .addGroup('cell/alignvert', pickAlignmentVer)
                    .addGroup('cell/linebreak', btnLinebreak)
                    .addGroup('cell/merge', pickMergeCells);
            }

            // BORDERS ------------------------------------------
            if (!self.panesCombined()) {
                if (app.isOOXML()) {
                    cellborderToolBar
                        .addGroup('cell/border/mode', pickCellBorderMode)
                        .addGroup('cell/border/style/preset', pickBorderStyle)
                        .addGroup('cell/border/color', pickCellBorderColor);

                } else {
                    var pickBorderWidth = new Controls.BorderWidthPicker({ dropDownVersion: { label: Labels.BORDER_WIDTH_LABEL } });

                    cellborderToolBar
                        .addGroup('cell/border/mode', pickCellBorderMode)
                        .addGroup('cell/border/style', pickBorderStyle)
                        .addGroup('cell/border/width', pickBorderWidth)
                        .addGroup('cell/border/color', pickCellBorderColor);
                }
            }

            // CELLS ------------------------------------------
            if (!self.panesCombined()) {
                var pickCellStyle = new Controls.CellStylePicker(self);

                cellstyleToolBar
                    .addGroup('cell/stylesheet', pickCellStyle);
            }

            if (self.panesCombined()) {
                cellToolBar
                    .addGroup('cell/fillcolor', pickFillColor)
                    .addGroup('cell/border/mode', pickCellBorderMode)
                    .addGroup('cell/border/style', pickBorderStyle)
                    .addGroup('cell/border/color', pickCellBorderColor);

                numberformatToolBar
                    .addGroup('cell/numberformat/category', grpFormatCategory)
                    .addGroup('cell/numberformat/category', pickFormatCategory)
                    .addGroup('cell/numberformat/code', pickFormatCode);
            }

            // DATA ------------------------------------------
            if (self.panesCombined()) {
                dataToolBar
                    .addGroup('cell/autoformula', btnSum)
                    .addGroup('function/insert/dialog', btnInsertFunction)
                    .addSeparator();
            }
            dataToolBar
                .addGroup('cell/sort', sortMenuBtn)
                .addSeparator()
                .addGroup('table/filter', btnFilter)
                .addGroup('table/refresh', btnFilterRefresh);

            if (!self.panesCombined()) {
                dataToolBar
                    .addSeparator()
                    .addGroup('names/menu/toggle', namedRangesMenuBtn);
            }

            // INSERT ------------------------------------------
            if (!self.panesCombined()) {
                var btnInsertHyperlink  = new Controls.Button(Labels.HYPERLINK_BUTTON_OPTIONS),
                    btnInsertImage      = new Controls.ImagePicker(self),
                    btnInsertChart      = new Controls.ChartTypePicker(self, Labels.INSERT_CHART_BUTTON_OPTIONS);

                insertToolBar
                    .addGroup('cell/autoformula', btnSum)
                    .addSeparator()
                    .addGroup('function/insert/dialog', btnInsertFunction)
                    .addSeparator()
                    .addGroup('hyperlink/edit/dialog', btnInsertHyperlink)
                    .addSeparator()
                    .addGroup('image/insert/dialog', btnInsertImage);

                if (app.isOOXML()) {
                    insertToolBar
                        .addGap()
                        .addGroup('chart/insert', btnInsertChart);
                }
            }

            rowsColsToolBar
                .addGroup('row/insert', btnInsertRow)
                .addGroup('row/delete', btnDeleteRow);

            if (!self.panesCombined()) {
                var labelRowSize        = new Controls.ColRowSizeLabel(false),
                    fieldRowSize        = new Controls.ColRowSizeField(self, false);

                rowsColsToolBar
                    .addGroup('row/height/active', labelRowSize)
                    .addGroup('row/height/active', fieldRowSize);
            }

            rowsColsToolBar
                .addSeparator()
                .addGroup('column/insert', btnInsertColumn)
                .addGroup('column/delete', btnDeleteColumn);

            if (!self.panesCombined()) {
                var labelColumnSize     = new Controls.ColRowSizeLabel(true),
                    fieldColumnSize     = new Controls.ColRowSizeField(self, true);

                rowsColsToolBar
                    .addGroup('column/width/active', labelColumnSize)
                    .addGroup('column/width/active', fieldColumnSize);
            }

            // DRAWING ------------------------------------------
            drawingToolBar
                .addGroup('drawing/delete', btnDeleteDrawing);

            if (!self.panesCombined()) {
                var pickChartType           = new Controls.ChartTypePicker(self),
                    btnChartLabels          = new Controls.Button(_.extend({ 'aria-owns': chartLabelsMenu.getUid() }, Labels.CHART_LABELS_BUTTON_OPTIONS)),
                    pickChartColorSet       = new Controls.ChartColorSetPicker(self),
                    pickChartStyleSet       = new Controls.ChartStyleSetPicker(),
                    cBoxChartDataLabel      = new Controls.CheckBox(Labels.CHART_SHOW_POINT_LABELS_BUTTON_OPTIONS),
                    cBoxChartVaryColor      = new Controls.CheckBox(Labels.CHART_VARY_POINT_COLORS_BUTTON_OPTIONS),
                    cBoxChartMarkerOnly     = new Controls.CheckBox(Labels.CHART_MARKER_ONLY_BUTTON_OPTIONS),
                    pickChartLegend         = new Controls.ChartLegendPicker(),
                    cBoxChartSource         = new Controls.Button({ icon: 'fa-none', label: /*#. change source data for a chart object in a spreadsheet */ gt.pgettext('chart-source', 'Edit data references') }),
                    btnChartExchange        = new Controls.Button({ icon: 'fa-none', label: /*#. switch orientation of data series in rectangular source data of a chart object */ gt.pgettext('chart-source', 'Switch rows and columns') }),
                    cBoxChartFirstRow       = new Controls.CheckBox({ label: /*#. decide if first row of source data is handled as label(headline) or as contentvalues */ gt.pgettext('chart-source', 'First row as label') }),
                    cBoxChartFirstCol       = new Controls.CheckBox({ label: /*#. decide if first column of source data is handled as label or as contentvalues */ gt.pgettext('chart-source', 'First column as label') });

                if (app.isOOXML()) {
                    var pickBorderPresetStyle   = new Controls.BorderPresetStylePicker(Labels.BORDER_ODF_PRESET_STYLES),
                        pickBorderColor         = new Controls.BorderColorPicker(self, { label: gt('Border color'), smallerVersion: { hideLabel: true } });

                    imageToolBar
                        .addGroup('drawing/border/style', pickBorderPresetStyle)
                        .addGroup('drawing/border/color', pickBorderColor);
                }

                chartStyleToolBar
                    .addGroup('drawing/charttype', pickChartType)
                    .addGap()
                    .addGroup('drawing/chartlabels', btnChartLabels)
                    .addSeparator()
                    .addGroup('drawing/chartcolorset', pickChartColorSet)
                    .addGroup('drawing/chartstyleset', pickChartStyleSet);

                chartDataToolBar
                    .addGroup(null, new Controls.CompoundButton(self, Labels.CHART_DATA_POINTS_BUTTON_OPTIONS)
                        .addGroup('drawing/chartdatalabel', cBoxChartDataLabel)
                        .addGroup('drawing/chartvarycolor', cBoxChartVaryColor)
                        .addGroup('drawing/chartmarkeronly', cBoxChartMarkerOnly)
                    );

                chartLegendToolBar
                    .addGroup('drawing/chartlegend/pos', pickChartLegend);

                chartSourceToolBar
                    .addGroup(null, new Controls.CompoundButton(self, { label: /*#. menu title: options to modify the data source of a chart object */ gt.pgettext('chart-source', 'Data source') })
                        .addGroup('drawing/chartsource', cBoxChartSource)
                        .addGroup('drawing/chartexchange', btnChartExchange)
                        .addGroup('drawing/chartfirstrow', cBoxChartFirstRow)
                        .addGroup('drawing/chartfirstcol', cBoxChartFirstCol)
                    );
            }

            // the 'View' drop-down menu
            viewMenuGroup
                .addSectionLabel(Labels.ZOOM_LABEL)
                .addGroup('view/zoom/dec', new Controls.Button(Labels.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom/inc', new Controls.Button(Labels.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom', new Controls.PercentageLabel(), { inline: true });

            if (app.isOOXML()) {
                viewMenuGroup.addSectionLabel(/*#. menu title: settings for splits and frozen columns/rows in a spreadsheet */ gt.pgettext('sheet-split', 'Split and freeze'))
                    .addGroup('view/split/dynamic', new Controls.DynamicSplitCheckBox())
                    .addGroup('view/split/frozen', new Controls.FrozenSplitCheckBox(self));
            }

            viewMenuGroup.addSectionLabel(Labels.OPTIONS_LABEL)
                // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                .addGroup('view/toolbars/show', new Controls.CheckBox(Labels.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                .addGroup('document/users', new Controls.CheckBox(Labels.SHOW_COLLABORATORS_CHECKBOX_OPTIONS))
                .addGroup('view/grid/show', new Controls.CheckBox({ label: /*#. check box label: show/hide cell grid in the sheet */ gt('Show grid lines'), tooltip: gt('Show or hide the cell grid lines in the current sheet') }))
                .addGroup('view/statuspane/show', new Controls.CheckBox({ label: /*#. check box label: show/hide the sheet tabs at the bottom border */ gt('Show sheet tabs'), tooltip: gt('Show or hide the sheet tabs below the sheet area') }));
        }

        function initDebugGuiHandler(viewMenuGroup) {

            self.createToolBar('debug')
                .addGroup('document/formula/recalcall', new Controls.Button({ icon: 'fa-refresh', label: _.noI18n('Recalculate all'), tooltip: _.noI18n('Recalculate all formulas in the document') }));

            viewMenuGroup
                .addGroup('debug/highlight/formulas', new Controls.CheckBox({ label: _.noI18n('Highlight formula cells') }));
        }

        /**
         * Moves the browser focus into the active sheet pane.
         */
        function grabFocusHandler() {
            // prevent JS errors when importing the document fails, and the view does not have activated a sheet yet
            if (activeSheetModel) {
                self.getActiveGridPane().grabFocus();
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the specified grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getGridPane = function (panePos) {
            return gridPaneMap.get(panePos);
        };

        /**
         * Returns the specified grid pane, if it is visible. Otherwise,
         * returns the nearest visible grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the preferred grid pane. If only one
         *  grid pane is visible, it will be returned regardless of this
         *  parameter. If one of the pane sides contained in the passed pane
         *  position is not visible, returns the grid pane from the other
         *  visible pane side (for example, if requesting the top-left pane
         *  while only bottom panes are visible, returns the bottom-left pane).
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getVisibleGridPane = function (panePos) {

            // jump to other column pane, if the pane side is hidden
            if (paneSideSettings[PaneUtils.getColPaneSide(panePos)].size === 0) {
                panePos = PaneUtils.getNextColPanePos(panePos);
            }

            // jump to other row pane, if the pane side is hidden
            if (paneSideSettings[PaneUtils.getRowPaneSide(panePos)].size === 0) {
                panePos = PaneUtils.getNextRowPanePos(panePos);
            }

            // now, panePos points to a visible grid pane (at least one pane is always visible)
            return gridPaneMap.get(panePos);
        };

        /**
         * Returns the grid pane that is currently focused.
         *
         * @returns {GridPane}
         *  The focused grid pane instance.
         */
        this.getActiveGridPane = function () {
            return this.getVisibleGridPane(activeSheetModel.getViewAttribute('activePane'));
        };

        /**
         * Returns the grid pane that contains the specified target cell. In a
         * frozen split view, only one grid pane can contain the specified
         * cell. In an unfrozen split view, any visible grid pane can contain
         * the cell. In the latter case, the active grid pane will be returned.
         *
         * @param {Address} address
         *  The address of the target cell.
         *
         * @returns {GridPane}
         *  The target grid pane instance.
         */
        this.getTargetGridPane = function (address) {

            // return active grid pane, if the view is not frozen
            if (!this.hasFrozenSplit()) { return this.getActiveGridPane(); }

            // the frozen column and row intervals
            var colInterval = activeSheetModel.getSplitColInterval();
            var rowInterval = activeSheetModel.getSplitRowInterval();
            // preferred pane side identifiers
            var colSide = (_.isObject(colInterval) && (address[0] <= colInterval.last)) ? 'left' : 'right';
            var rowSide = (_.isObject(rowInterval) && (address[1] <= rowInterval.last)) ? 'top' : 'bottom';

            // return the grid pane containing the cell address
            return this.getVisibleGridPane(PaneUtils.getPanePos(colSide, rowSide));
        };

        /**
         * Activates the specified grid pane, unless the view is currently in
         * an internal initialization phase, or the cell edit mode is currently
         * active.
         *
         * @param {String} panePos
         *  The identifier of the grid pane to be activated.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.updateActiveGridPane = function (panePos) {
            // do not change active pane while initializing panes, or while in cell
            // edit mode (range selection mode in formulas)
            if (panesInitialized && !this.isCellEditMode()) {
                this.setSheetViewAttribute('activePane', panePos);
            }
            return this;
        };

        /**
         * Invokes the passed iterator function for all grid panes contained in
         * this view.
         *
         * @param {Function} callback
         *  The callback function invoked for all grid pane instances. Receives
         *  the following parameters:
         *  (1) {GridPane} gridPane
         *      The current grid pane instance.
         *  (2) {String} panePos
         *      The position identifier of the grid pane.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.iterateGridPanes = function (callback, context) {
            gridPaneMap.forEach(callback, context);
            return this;
        };

        /**
         * Returns the specified header pane.
         *
         * @param {String} paneSide
         *  The identifier of the pane side ('left', 'right', 'top', 'bottom').
         *
         * @returns {HeaderPane}
         *  The header pane instance at the specified pane side.
         */
        this.getHeaderPane = function (paneSide) {
            return headerPaneMap.get(paneSide);
        };

        /**
         * Returns the horizontal header pane (left or right) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The horizontal header pane instance at the corresponding pane side.
         */
        this.getColHeaderPane = function (panePos) {
            return headerPaneMap.get(PaneUtils.getColPaneSide(panePos));
        };

        /**
         * Returns the vertical header pane (top or bottom) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The vertical header pane instance at the corresponding pane side.
         */
        this.getRowHeaderPane = function (panePos) {
            return headerPaneMap.get(PaneUtils.getRowPaneSide(panePos));
        };

        /**
         * Returns the first visible header pane in the specified direction
         * (for example, for columns returns the left header pane, if it is
         * visible, otherwise the right header pane).
         *
         * @param {Boolean} columns
         *  If set to true, returns the first visible column header pane (left
         *  or right); otherwise returns the first visible row header pane (top
         *  or bottom).
         *
         * @returns {HeaderPane}
         *  The first visible header pane instance in the specified direction.
         */
        this.getVisibleHeaderPane = function (columns) {
            var paneSide = columns ?
                ((paneSideSettings.left.size > 0) ? 'left' : 'right') :
                ((paneSideSettings.top.size > 0) ? 'top' : 'bottom');
            return headerPaneMap.get(paneSide);
        };

        /**
         * Returns a header pane associated to the grid pane that is currently
         * focused.
         *
         * @param {Boolean} columns
         *  If set to true, returns the column header pane (left or right);
         *  otherwise returns the row header pane (top or bottom).
         *
         * @returns {HeaderPane}
         *  A header pane instance associated to the grid pane that is
         *  currently focused.
         */
        this.getActiveHeaderPane = function (columns) {
            return this.getActiveGridPane().getHeaderPane(columns);
        };

        /**
         * Invokes the passed iterator function for all header panes contained
         * in this view.
         *
         * @param {Function} callback
         *  The callback function invoked for all header pane instances.
         *  Receives the following parameters:
         *  (1) {HeaderPane} headerPane
         *      The current header pane instance.
         *  (2) {String} paneSide
         *      The identifier of the pane side ('left', 'top', etc.).
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.iterateHeaderPanes = function (callback, context) {
            headerPaneMap.forEach(callback, context);
            return this;
        };

        /**
         * Returns the top-left corner pane.
         *
         * @returns {CornerPane}
         *  The corner pane instance.
         */
        this.getCornerPane = function () {
            return cornerPane;
        };

        /**
         * Returns the current width of the row header panes.
         *
         * @returns {Number}
         *  The current width of the row header panes, in pixels.
         */
        this.getHeaderWidth = function () {
            return cornerPane.getNode().outerWidth();
        };

        /**
         * Returns the current height of the column header panes.
         *
         * @returns {Number}
         *  The current height of the column header panes, in pixels.
         */
        this.getHeaderHeight = function () {
            return cornerPane.getNode().outerHeight();
        };

        /**
         * Returns the status pane containing the sheet tabs, and the subtotal
         * label.
         */
        this.getStatusPane = function () {
            return statusPane;
        };

        /**
         * Returns the floating menu containing all defined names.
         *
         * @returns {DefinedNamesMenu}
         *  The floating menu containing all defined names.
         */
        this.getDefinedNamesMenu = function () {
            return definedNamesMenu;
        };

        /**
         * Returns the floating menu for axis and label formatting of chart
         * objects.
         *
         * @returns {ChartLabelsMenu}
         *  The floating menu for axis and label formatting of chart objects.
         */
        this.getChartLabelsMenu = function () {
            return chartLabelsMenu;
        };

        /**
         * Returns the rendering cache containing processed rendering data for
         * the visible columns, rows, and cells.
         *
         * @returns {RenderCache}
         *  The rendering cache of this document.
         */
        this.getRenderCache = function () {
            return renderCache;
        };

        /**
         * Returns the intersection of the passed column/row intervals with the
         * visible intervals of the header panes in the specified direction.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Boolean} columns
         *  Whether to return the column intervals of the left and right header
         *  panes (true), or the row intervals of the top and bottom header
         *  panes (false).
         *
         * @returns {IntervalArray}
         *  The parts of the passed column/row intervals which are covered by
         *  the respective visible header panes.
         */
        this.getVisibleIntervals = function (intervals, columns) {

            // the resulting intervals covered by the header panes
            var resultIntervals = new IntervalArray();

            // convert parameter to an array
            intervals = IntervalArray.get(intervals);

            // pick the intervals of all visible header panes in the correct orientation
            headerPaneMap.forEach(function (headerPane, paneSide) {
                if ((PaneUtils.isColumnSide(paneSide) === columns) && headerPane.isVisible()) {
                    resultIntervals.append(intervals.intersect(headerPane.getRenderInterval()));
                }
            });

            // merge intervals in case the header panes cover the same sheet area
            return resultIntervals.merge();
        };

        /**
         * Shows a notification alert box, if a corresponding message text
         * exists for the passed message code.
         *
         * @param {String} msgCode
         *  The message code to show a notification alert box for.
         *
         * @param {Boolean} [success=false]
         *  If set to true, a success alert box will be shown. By default, an
         *  information alert will be shown.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.yellMessage = function (msgCode, success) {

            // missing edit mode shown via special application method 'rejectEditAttempt()'
            if (msgCode === 'readonly') {
                app.rejectEditAttempt();
            } else if (msgCode === 'image:insert') {
                app.rejectEditAttempt('image');
            } else if (msgCode in RESULT_MESSAGES) {
                this.yell({ type: success ? 'success' : 'info', message: RESULT_MESSAGES[msgCode] });
            } else if (success) {
                this.hideYell(); // bug 47936: hide previously shown warning alert
            }

            return this;
        };

        /**
         * Shows a notification message box, if the passed promise rejects with
         * a message code that has a corresponding message text.
         *
         * @param {jQuery.Promise} promise
         *  A promise representing an asynchronous document model operation. If
         *  the promise will be rejected with an object containing the property
         *  'cause' with a supported message code, an alert box will be shown
         *  to the user.
         *
         * @returns {jQuery.Promise}
         *  The promise passed to this method, for convenience.
         */
        this.yellOnFailure = function (promise) {

            // handle success (e.g., hide previously shown warning alert)
            this.waitForSuccess(promise, function () {
                this.yellMessage('', true);
            });

            // show information alert on error
            this.waitForFailure(promise, function (result) {
                this.yellMessage(Utils.getStringOption(result, 'cause', ''));
            });

            // return the promise for convenience
            return promise;
        };

        /**
         * Shows a notification alert to the user, if the document is in
         * read-only mode.
         *
         * @returns {Boolean}
         *  Whether modifying the document is posssible.
         */
        this.requireEditMode = function () {
            var editMode = docModel.getEditMode();
            if (!editMode) { this.yellMessage('readonly'); }
            return editMode;
        };

        /**
         * Returns the current status text label.
         *
         * @returns {String}
         *  The current status text.
         */
        this.getStatusLabel = function () {
            return statusText;
        };

        /**
         * Sets the passed text as current status text label, shown in the
         * status pane at the bottom border of the application pane.
         *
         * @param {String} text
         *  The new status text.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setStatusLabel = function (text) {
            if (statusText !== text) {
                statusText = text;
                app.getController().update();
            }
            return this;
        };

        /**
         * Returns the zoom factor to be used for the active sheet.
         *
         * @attention
         *  Existence of this method is expected by code in base modules, e.g.
         *  the drawing layer.
         *
         * @returns {Number}
         *  The zoom factor for the active sheet.
         */
        this.getZoomFactor = function () {
            return activeSheetModel ? activeSheetModel.getEffectiveZoom() : 1;
        };

        /**
         * Returns the effective font size for the current zoom factor to be
         * used for the column and row labels in all header panes.
         *
         * @returns {Number}
         *  The effective font size for all header panes, in points.
         */
        this.getHeaderFontSize = function () {
            return Utils.minMax(Utils.round(5 * this.getZoomFactor() + 6, 0.1), 8, Utils.COMPACT_DEVICE ? 12 : 30);
        };

        // active sheet -------------------------------------------------------

        /**
         * Returns the number of visible sheets in this spreadsheet view.
         *
         * @returns {Number}
         *  The number of visible sheets in the document.
         */
        this.getVisibleSheetCount = function () {
            return visibleSheets.length;
        };

        /**
         * Returns the number of hidden sheets in this spreadsheet view that
         * can be made visible.
         *
         * @returns {Number}
         *  The number of hidden sheets in the document. Does NOT include the
         *  sheets with unsupported types that will always be hidden.
         */
        this.getHiddenSheetCount = function () {
            return hiddenSheets.length;
        };

        /**
         * Returns the hidden sheets in this spreadsheet view that
         * can be made visible.
         *
         * @returns {Array<Object>} sheetInfos
         *  sheetInfo.name Name of the sheet
         *  sheetInfo.sheet Index of the sheet
         *  The hidden sheets in the document. Does NOT include the
         *  sheets with unsupported types that will always be hidden.
         */
        this.getHiddenSheets = function () {
            return hiddenSheets;
        };

        /**
         * Invokes the passed iterator function for all visible sheet contained
         * in this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all visible sheets. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The current sheet model instance.
         *  (2) {Number} index
         *      The zero-based sheet index in the collection of all visible
         *      sheets.
         *  (3) {String} sheetName
         *      The name of the sheet.
         *  (4) {Number} sheet
         *      The zero-based sheet index in the document model.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the sheets will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateVisibleSheets = function (iterator, options) {
            return Utils.iterateArray(visibleSheets, function (sheetInfo, index) {
                return iterator.call(this, docModel.getSheetModel(sheetInfo.sheet), index, sheetInfo.name, sheetInfo.sheet);
            }, options);
        };

        /**
         * Returns the zero-based index of the active sheet currently displayed
         * in this spreadsheet view. Optionally, the index of the active sheet
         * inside the collection of all visible sheets can be returned.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the index of the active sheet inside the
         *      collection of all visible sheets will be returned instead of
         *      the physical sheet index.
         *
         * @returns {Number}
         *  The zero-based index of the active sheet, depending on the passed
         *  options. Will return -1, if the index in the collection of visible
         *  sheets is requested, but the active sheet is currently hidden.
         */
        this.getActiveSheet = function (options) {
            // look up the index of the active sheet in the collection of visible sheets,
            // or return the plain sheet index of the active sheet
            return Utils.getBooleanOption(options, 'visible', false) ? getVisibleIndex(activeSheet) : activeSheet;
        };

        /**
         * Activates a sheet in this spreadsheet view instance.
         *
         * @param {Number} index
         *  The zero-based index of the new active sheet. If the specified
         *  sheet is currently hidden, the next visible sheet will be activated
         *  instead.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the passed sheet index will be interpreted as
         *      index into the collection of all visible sheets, instead of the
         *      physical sheet index.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet, options) {

            // fail-safety: nothing useful can be done without a visible sheet
            if (visibleSheets.length === 0) {
                Utils.error('SpredsheetView.setActiveSheet(): no visible sheets available');
                return this;
            }

            // get sheet index from the collection of visible sheets;
            // otherwise adjust the passed sheet index if it points to a hidden sheet
            if (Utils.getBooleanOption(options, 'visible', false)) {
                sheet = visibleSheets[Utils.minMax(sheet, 0, visibleSheets.length - 1)].sheet;
            } else {
                var sheetData = Utils.findFirst(visibleSheets, function (sheetData) { return sheet <= sheetData.sheet; }, { sorted: true });
                sheet = sheetData ? sheetData.sheet : _.last(visibleSheets).sheet;
            }

            // set the new active sheet index (will cause to handle the 'change:viewattributes'
            // event with further processing and initialization for the new active sheet)
            docModel.setActiveSheet(sheet);
            return this;
        };

        /**
         * Activates the preceding visible sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activatePreviousSheet = function () {
            var sheet = this.getActiveSheet({ visible: true });
            return this.setActiveSheet(sheet - 1, { visible: true });
        };

        /**
         * Activates the next visible sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activateNextSheet = function () {
            var sheet = this.getActiveSheet({ visible: true });
            return this.setActiveSheet(sheet + 1, { visible: true });
        };

        /**
         * Activates a sheet for the import preview, while the document import
         * is still running.
         *
         * @internal
         *  Called from the preview handler of the application. MUST NOT be
         *  called directly.
         *
         * @param {Number} index
         *  The zero-based index of the initial active sheet.
         *
         * @returns {Boolean}
         *  Whether a sheet has been activated successfully.
         */
        this.activatePreviewSheet = function (sheet) {

            // the sheet model (no preview possible, if the sheet does not exist)
            var sheetModel = docModel.getSheetModel(sheet);
            if (!sheetModel) { return false; }

            // initialize view settings (scroll position, selection, zoom, split)
            sheetModel.initializeViewAttributes();
            // activate the preview sheet in the view
            this.setActiveSheet(sheet);

            // notify view components, e.g. repaint the sheet tabs once during import
            this.trigger('preview:activesheet');

            // return whether the sheet has been activated successfully
            return activeSheet >= 0;
        };

        /**
         * Returns the model instance of the active sheet.
         *
         * @returns {SheetModel}
         *  The model instance of the active sheet.
         */
        this.getSheetModel = function () {
            return activeSheetModel;
        };

        /**
         * Returns the cell collection instance of the active sheet.
         *
         * @returns {CellCollection}
         *  The cell collection instance of the active sheet.
         */
        this.getCellCollection = function () {
            return cellCollection;
        };

        /**
         * Returns the column collection instance of the active sheet.
         *
         * @returns {ColRowCollection}
         *  The column collection instance of the active sheet.
         */
        this.getColCollection = function () {
            return colCollection;
        };

        /**
         * Returns the row collection instance of the active sheet.
         *
         * @returns {ColRowCollection}
         *  The row collection instance of the active sheet.
         */
        this.getRowCollection = function () {
            return rowCollection;
        };

        /**
         * Returns the merge collection instance of the active sheet.
         *
         * @returns {MergeCollection}
         *  The merge collection instance of the active sheet.
         */
        this.getMergeCollection = function () {
            return mergeCollection;
        };

        /**
         * Returns the table collection instance of the active sheet.
         *
         * @returns {TableCollection}
         *  The table collection instance of the active sheet.
         */
        this.getTableCollection = function () {
            return tableCollection;
        };

        /**
         * Returns the validation collection instance of the active sheet.
         *
         * @returns {ValidationCollection}
         *  The validation collection instance of the active sheet.
         */
        this.getValidationCollection = function () {
            return validationCollection;
        };

        /**
         * Returns the conditional formatting collection of the active sheet.
         *
         * @returns {CondFormatCollection}
         *  The conditional formatting collection of the active sheet.
         */
        this.getCondFormatCollection = function () {
            return condFormatCollection;
        };

        /**
         * Returns the drawing collection instance of the active sheet.
         *
         * @returns {SheetDrawingCollection}
         *  The drawing collection instance of the active sheet.
         */
        this.getDrawingCollection = function () {
            return drawingCollection;
        };

        /**
         * Converts the passed length in pixels to a length in 1/100 of
         * millimeters, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in pixels.
         *
         * @returns {Number}
         *  The converted length in 1/100 of millimeters.
         */
        this.convertPixelToHmm = function (length) {
            return activeSheetModel.convertPixelToHmm(length);
        };

        /**
         * Converts the passed length in 1/100 of millimeters to a length in
         * pixels, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The converted length in pixels.
         */
        this.convertHmmToPixel = function (length) {
            return activeSheetModel.convertHmmToPixel(length);
        };

        /**
         * Returns the absolute position of the passed cell range in the active
         * sheet, in 1/100 of millimeters, as well as in pixels according to
         * the current sheet zoom factor.
         *
         * @param {Range} range
         *  The address of the cell range in the active sheet.
         *
         * @returns {Object}
         *  The absolute position and size of the range in the sheet. See
         *  SheetModel.getRangeRectangle() for details.
         */
        this.getRangeRectangle = function (range) {
            return activeSheetModel.getRangeRectangle(range);
        };

        /**
         * Returns the absolute position of the specified cell in the active
         * sheet, in 1/100 of millimeters, as well as in pixels according to
         * the current sheet zoom factor.
         *
         * @param {Address} address
         *  The address of the cell in the active sheet.
         *
         * @param {Object} [options]
         *  Optional parameters. See method SheetModel.getCellRectangle() for
         *  details.
         *
         * @returns {Object}
         *  The absolute position and size of the cell in the active sheet. See
         *  method SheetModel.getCellRectangle() for details.
         */
        this.getCellRectangle = function (address, options) {
            return activeSheetModel.getCellRectangle(address, options);
        };

        // scrolling ----------------------------------------------------------

        /**
         * Scrolls the active grid pane to the specified cell. In frozen split
         * view, the best fitting grid pane for the passed cell address will be
         * picked instead (see method SpreadsheetView.getTargetGridPane() for
         * details).
         *
         * @param {Address} address
         *  The address of the cell to be scrolled to.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.updateFocus=true]
         *      If set to true or omitted, and if the view is in frozen split
         *      mode, the active grid pane containing the passed cell will be
         *      focused. Otherwise, the grid pane that is currently active will
         *      be scrolled to the passed cell address.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address, options) {

            // the target grid pane to be scrolled (grid pane may change in frozen split mode)
            var gridPane = this.getTargetGridPane(address);

            // set focus to the resulting grid pane (this will update the 'activePane' view attribute)
            if (Utils.getBooleanOption(options, 'updateFocus', true)) {
                gridPane.grabFocus();
            }

            // scroll the resulting grid pane to the passed cell address
            gridPane.scrollToCell(address);
            return this;
        };

        /**
         * Scrolls the active grid pane to the specified drawing frame. In
         * frozen split view, scrolls the bottom-right grid pane instead.
         *
         * @param {Array<Number>} position
         *  The position of the drawing frame to be scrolled to.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position) {

            // the resulting grid pane that will be scrolled
            var gridPane = this.hasFrozenSplit() ? this.getVisibleGridPane('bottomRight') : this.getActiveGridPane();

            gridPane.scrollToDrawingFrame(position);
            return this;
        };

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Forms.addDeviceMarkers(rootNode);

        // create floating menus
        definedNamesMenu = new DefinedNamesMenu(this);
        chartLabelsMenu = new ChartLabelsMenu(this, { anchorBox: function () { return self.getActiveGridPane().getScrollNode(); } });

        // create the rendering cache
        renderCache = new RenderCache(this);

        // create the header pane instances
        PaneUtils.ALL_PANE_SIDES.forEach(function (paneSide) {
            paneSideSettings[paneSide] = _.clone(DEFAULT_PANE_SIDE_SETTINGS);
            headerPaneMap.insert(paneSide, new HeaderPane(this, paneSide));
        }, this);

        // create the grid pane instances
        PaneUtils.ALL_PANE_POSITIONS.forEach(function (panePos) {
            gridPaneMap.insert(panePos, new GridPane(this, panePos));
        }, this);

        // create the corner pane instance
        cornerPane = new CornerPane(this);

        // mix-in classes expecting fully initialized document view
        SelectionMixin.call(this, rootNode);
        HighlightMixin.call(this);
        CellEditMixin.call(this);
        ViewFuncMixin.call(this);

        // destroy all class members on destruction
        this.registerDestructor(function () {

            renderCache.destroy();
            gridPaneMap.forEach(function (gridPane) { gridPane.destroy(); });
            headerPaneMap.forEach(function (headerPane) { headerPane.destroy(); });
            cornerPane.destroy();
            definedNamesMenu.destroy();
            chartLabelsMenu.destroy();

            self = app = docModel = null;
            gridPaneMap = headerPaneMap = cornerPane = statusPane = null;
            definedNamesMenu = chartLabelsMenu = renderCache = null;
            colSplitLineNode = rowSplitLineNode = resizeOverlayNode = null;
            colSplitTrackingNode = rowSplitTrackingNode = centerSplitTrackingNode = allSplitTrackingNodes = null;
            activeSheetModel = colCollection = rowCollection = mergeCollection = cellCollection = null;
            tableCollection = validationCollection = condFormatCollection = drawingCollection = null;
        });

    } // class SpreadsheetView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: SpreadsheetView });

});
