/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/selectionrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, TriggerObject, TimerMixin, Config, SheetUtils, PaneUtils, RenderUtils) {

    'use strict';

    // convenience shortcuts
    var Range = SheetUtils.Range;

    // bug 41205: IE/Edge: navigating via tab key in last row results in rendering errors
    var renderHelperNode = $('<div id="io-ox-office-spreadsheet-ie-render-helper" style="position:absolute;">');

    // private global functions ===============================================

    var setIERenderHelperNode = _.browser.IE ? function () {
        renderHelperNode.remove();
        window.setTimeout(function () {
            $('body').append(renderHelperNode);
        }, 20);
    } : _.noop;

    // class SelectionRenderer ================================================

    /**
     * Renders the own selection, and the selections of remote users into the
     * DOM layers of a single grid pane.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this selection layer renderer.
     */
    function SelectionRenderer(gridPane) {

        // self reference
        var self = this;

        // the spreadsheet model and view
        var docView = gridPane.getDocView();
        var docModel = docView.getDocModel();

        // the remote layer node for displaying selections of other users
        var remoteLayerNode = gridPane.createLayerNode('remote-layer');

        // the selection layer (container for the selected ranges)
        var selectionLayerNode = gridPane.createLayerNode('selection-layer');

        // the cell range and absolute position covered by the layer nodes in the sheet area
        var layerRectangle = null;

        // list of drawing positions which are remotely selected
        var selectedDrawings = [];

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        //Debounced version of the method renderCellSelection().
        var renderCellSelectionDebounced = this.createDebouncedMethod($.noop, renderCellSelection, { delay: 100, infoString: 'SelectionRenderer: renderCellCollection', app: docView.getApp() });

        /**
         * Updates the position of the auto-fill handle and the resize handles
         * for cell selection, when entire columns or rows are selected.
         */
        function updateResizeHandlePositions() {

            // the auto-fill resizer handle node
            var autoFillHandleNode = selectionLayerNode.find('.autofill.resizers>[data-pos]');
            // the range node containing the auto-fill handle
            var autoFillRangeNode = autoFillHandleNode.closest('.range');
            // visible area of the grid pane
            var visibleRect = gridPane.getVisibleRectangle();

            // converts the passed offset in the visible area to an offset relative to 'parentNode'
            function setRelativeOffset(targetNode, offsetName, parentNode, visibleOffset) {
                var offset = visibleRect[offsetName] - layerRectangle[offsetName] - parentNode.position()[offsetName] + visibleOffset;
                targetNode.css(offsetName, offset + 'px');
            }

            // adjust position of the auto-fill handle node for entire column or row selection
            switch (autoFillHandleNode.attr('data-pos')) {
                case 'r':
                    setRelativeOffset(autoFillHandleNode, 'top', autoFillRangeNode, 0);
                    break;
                case 'b':
                    setRelativeOffset(autoFillHandleNode, 'left', autoFillRangeNode, 0);
                    break;
            }

            // adjust position of the selection handle node for entire column or row selection
            selectionLayerNode.find('.select.resizers>[data-pos]').each(function () {

                var handleNode = $(this),
                    rangeNode = handleNode.closest('.range');

                switch (handleNode.attr('data-pos')) {
                    case 'l':
                    case 'r':
                        setRelativeOffset(handleNode, 'top', rangeNode, visibleRect.height / 2);
                        break;
                    case 't':
                    case 'b':
                        setRelativeOffset(handleNode, 'left', rangeNode, visibleRect.width / 2);
                        break;
                }
            });
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderCellSelection() {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) {
                selectionLayerNode.empty();
                return;
            }

            // the entire cell selection
            var selection = docView.getSelection();
            // the range occupied by the active cell (will not be filled)
            var activeRange = null;
            // the position of the active cell, extended to the merged range
            var activeRectangle = null;
            // additional data for active auto-fill tracking
            var autoFillData = docView.getSheetViewAttribute('autoFillData');
            // whether the active cell touches the borders of selection ranges
            var activeBorders = { left: false, top: false, right: false, bottom: false };
            // hide left/top border of selection ranges, if first column/row is hidden
            var firstColHidden = !docView.getColCollection().isEntryVisible(0);
            var firstRowHidden = !docView.getRowCollection().isEntryVisible(0);
            // the HTML mark-up for all selection ranges but the active range
            var rangesMarkup = '';
            // the HTML mark-up for the active selection range
            var activeMarkup = '';

            // returns the mark-up of a resizer handler according to the type of the passed range
            function createResizerMarkup(range, leadingCorner) {
                var hPos = leadingCorner ? 'l' : 'r', vPos = leadingCorner ? 't' : 'b';
                return '<div data-pos="' + (docModel.isColRange(range) ? hPos : docModel.isRowRange(range) ? vPos : (vPos + hPos)) + '"></div>';
            }

            // add active flag to range object
            selection.ranges[selection.active].active = true;

            // adjust single range for auto-fill tracking
            if ((selection.ranges.length === 1) && _.isObject(autoFillData)) {

                // the selected cell range
                var firstRange = selection.ranges.first();
                // whether to expand/shrink the leading or trailing border
                var leading = /^(left|top)$/.test(autoFillData.border);
                // whether to expand/shrink columns or rows
                var columns = /^(left|right)$/.test(autoFillData.border);

                activeRange = firstRange.clone();
                if (autoFillData.count >= 0) {
                    // adjust range for auto-fill mode
                    if (leading) {
                        firstRange.start.move(-autoFillData.count, columns);
                    } else {
                        firstRange.end.move(autoFillData.count, columns);
                    }
                } else {
                    // adjust range for deletion mode
                    firstRange.collapsed = true;
                    if (-autoFillData.count < firstRange.size(columns)) {
                        // range partly covered
                        if (leading) {
                            activeRange.start.move(-autoFillData.count, columns);
                        } else {
                            activeRange.end.move(autoFillData.count, columns);
                        }
                    } else {
                        // special marker for deletion of the entire range
                        activeRange = Utils.BREAK;
                    }
                }

                // add tracking style effect
                firstRange.tracking = true;
            }

            // the range covered by the active cell (or any other range in auto-fill tracking mode)
            activeRange = (activeRange === Utils.BREAK) ? null : activeRange ? activeRange :
                docView.getMergeCollection().expandRangeToMergedRanges(new Range(selection.address));

            // convert active range to a rectangle relative to the layer root node
            if (_.isObject(activeRange)) {
                gridPane.iterateRangesForRendering(activeRange, function (range, rectangle) {
                    activeRectangle = rectangle;
                }, { alignToGrid: true });
            }

            // render all ranges that are visible in this grid pane
            gridPane.iterateRangesForRendering(selection.ranges, function (range, rectangle, index) {

                // hide left/top border of the range, if the first column/row is hidden
                if (firstColHidden && (range.start[0] === 0)) {
                    rectangle.left -= 5;
                    rectangle.width += 5;
                }
                if (firstRowHidden && (range.start[1] === 0)) {
                    rectangle.top -= 5;
                    rectangle.height += 5;
                }

                // generate the HTML mark-up for the range
                var rangeMarkup = '<div class="range';
                if (range.active) { rangeMarkup += ' active'; }
                if (range.tracking) { rangeMarkup += ' tracking-active'; }
                if (range.collapsed) { rangeMarkup += ' collapsed'; }
                rangeMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '" data-index="' + index + '">';

                // insert the semi-transparent fill elements (never cover the active range with the fill elements)
                if (activeRectangle && range.overlaps(activeRange)) {

                    // initialize position of active cell, relative to selection range
                    var relActiveRect = _.clone(activeRectangle);
                    relActiveRect.left -= rectangle.left;
                    relActiveRect.top -= rectangle.top;
                    relActiveRect.right = rectangle.width - relActiveRect.left - relActiveRect.width;
                    relActiveRect.bottom = rectangle.height - relActiveRect.top - relActiveRect.height;

                    // insert fill element above active cell
                    if (relActiveRect.top > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + relActiveRect.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (relActiveRect.left > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;width:' + relActiveRect.left + 'px;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (relActiveRect.right > 0) {
                        rangeMarkup += '<div class="fill" style="left:' + (relActiveRect.left + relActiveRect.width) + 'px;right:0;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (relActiveRect.bottom > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:' + (relActiveRect.top + relActiveRect.height) + 'px;bottom:0;"></div>';
                    }

                    // update border flags for the active cell
                    activeBorders.left = activeBorders.left || (range.start[0] === activeRange.start[0]);
                    activeBorders.top = activeBorders.top || (range.start[1] === activeRange.start[1]);
                    activeBorders.right = activeBorders.right || (range.end[0] === activeRange.end[0]);
                    activeBorders.bottom = activeBorders.bottom || (range.end[1] === activeRange.end[1]);

                } else {
                    rangeMarkup += '<div class="abs fill"></div>';
                }

                // generate the HTML mark-up for the selection range
                rangeMarkup += '<div class="border"></div>';

                // additional mark-up for single-range selection
                if (selection.ranges.length === 1) {
                    if (Utils.TOUCHDEVICE && !_.isObject(autoFillData)) {
                        // add resize handlers for selection on touch devices
                        rangeMarkup += '<div class="select resizers">' + createResizerMarkup(range, true) + createResizerMarkup(range, false) + '</div>';
                    } else {
                        // add the auto-fill handler in the bottom right corner of the selection
                        rangeMarkup += '<div class="autofill resizers">' + createResizerMarkup(range, false) + '</div>';
                    }
                }

                // close the range node
                rangeMarkup += '</div>';

                // and the generated mark-up to the appropriate mark-up list
                if (range.active) { activeMarkup += rangeMarkup; } else { rangesMarkup += rangeMarkup; }
            }, { alignToGrid: true });

            // additions for the active cell (or active range in auto-fill)
            if (activeRectangle) {

                // add thin border for active cell on top of the selection
                rangesMarkup += '<div class="active-cell';
                _.each(activeBorders, function (isBorder, borderName) { if (!isBorder) { rangesMarkup += ' ' + borderName; } });
                rangesMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(activeRectangle) + '"></div>';
            }

            // insert entire HTML mark-up into the selection container node
            selectionLayerNode[0].innerHTML = rangesMarkup + activeMarkup;

            // update the visibility/position of additional DOM elements
            updateResizeHandlePositions();

            // bug 41205: IE/Edge: navigating via tab key in last row results in rendering errors
            setIERenderHelperNode();

            // notify listeners for additional rendering depending on the selection
            self.trigger('render:cellselection');
        }

        /**
         * Renders selections of all remote users which have the same sheet
         * activated.
         */
        var renderRemoteSelections = Config.SHOW_REMOTE_SELECTIONS ? function () {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) {
                remoteLayerNode.empty();
                return;
            }

            // prepare equal mark-up in all selection ranges
            var borderMarkup = '<div class="borders">' + _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('') + '</div>';
            // top-left visible address of the grid pane, to detect if the name-layer outside of the view
            var firstRow = gridPane.getTopLeftAddress()[1];
            // the HTML mark-up for all user selections
            var markup = '';

            // remove the classes from the last remote selected drawings
            selectedDrawings.forEach(function (drawingPos) {
                var drawingFrame = gridPane.getDrawingFrame(drawingPos);
                drawingFrame.removeClass('remoteselection badge-inside');
                drawingFrame.removeAttr('data-username');
            });
            selectedDrawings = [];

            // generate selections of all remote users on the same sheet
            docModel.getViewAttribute('remoteClients').forEach(function (client) {

                // skip users that have another sheet activated
                if (client.userData.sheet !== docModel.getActiveSheet()) { return; }

                // the escaped user name, ready to be inserted into HTML mark-up
                var userName = Utils.escapeHTML(_.noI18n(client.userName));
                // the positions of all selected drawing objects
                var newSelectedDrawings = Utils.getArrayOption(client.userData, 'drawings', null);

                // show drawing selection only, if the selection contains drawings, otherwise show the cell selection
                if (newSelectedDrawings && (newSelectedDrawings.length > 0)) {

                    // render all selected drawings of the user
                    var drawingCollection = docView.getDrawingCollection();
                    newSelectedDrawings.forEach(function (drawingPos) {

                        var drawingModel = drawingCollection.findModel(drawingPos);
                        var drawingFrame = gridPane.getDrawingFrame(drawingPos);
                        if (drawingModel && drawingFrame.length > 0) {

                            var rect = drawingModel.getRectangle();
                            rect = gridPane.convertToLayerRectangle(rect);
                            rect.top -= 2;
                            rect.left -= 2;
                            rect.width += 4;
                            rect.height += 4;

                            // create the mark-up for the range
                            markup += '<div class="border" data-style="' + client.colorIndex + '"';
                            markup += ' style="' + PaneUtils.getRectangleStyleMarkup(rect) + '"></div>';

                            drawingFrame.addClass('remoteselection badge-inside');

                            drawingFrame.attr('data-username', _.noI18n(client.userName));
                            drawingFrame.attr('data-style', client.colorIndex);
                            selectedDrawings.push(drawingPos);
                        }
                    });

                } else { // no drawings, show cell selection

                    // create the array of cell range addresses
                    var ranges = docModel.getFormulaParser().parseRangeList('op', client.userData.ranges);
                    if (!ranges) { return; }

                    // render all ranges of the user
                    gridPane.iterateRangesForRendering(ranges, function (range, rectangle) {

                        // the CSS classes to be set at the range (bug 35604: inside range if on top of the grid pane)
                        var classes = 'range' + ((range.start[1] === firstRow) ? ' badge-inside' : '');

                        // create the mark-up for the range
                        markup += '<div class="' + classes + '" data-style="' + client.colorIndex + '" data-username="' + userName + '"';
                        markup += ' style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '">' + borderMarkup + '</div>';
                    }, { alignToGrid: true });
                }
            });

            // insert entire HTML mark-up into the container node
            remoteLayerNode[0].innerHTML = markup;

        } : $.noop;

        /**
         * Updates the layer nodes after specific document view attributes have
         * been changed.
         */
        function changeViewAttributesHandler(event, attributes) {

            // display remote selections
            if ('remoteClients' in attributes) {
                renderRemoteSelections();
            }
        }

        /**
         * Updates the layer nodes after specific sheet view attributes have
         * been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // render cell selection (triggers a 'render:cellselection' event)
            if (Utils.hasProperty(attributes, /^(selection$|split|autoFillData$)/)) {
                renderCellSelection();
            }

            // trigger a 'render:cellselection' event if active pane changes
            // (no actual rendering needed, border colors will change via CSS)
            if ('activePane' in attributes) {
                self.trigger('render:cellselection');
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('SelectionRenderer.setLayerRange()', function (newLayerRange, newLayerRectangle) {

            // store new layer range and rectangle for convenience
            layerRectangle = newLayerRectangle;

            // render own selection, and selections of all remote editors
            renderCellSelection();
            renderRemoteSelections();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            layerRectangle = null;
            selectionLayerNode.empty();
            remoteLayerNode.empty();
        };

        // initialization -----------------------------------------------------

        // render cell selection again when document edit mode changes
        gridPane.listenToWhenVisible(this, docModel, 'change:editmode', renderCellSelection);

        // render cell selection after merging/unmerging cell ranges (size of active cell may change)
        gridPane.listenToWhenVisible(this, docView, 'insert:merged delete:merged', renderCellSelectionDebounced);

        // update layers according to changed view attributes (only, if this grid pane is visible)
        gridPane.listenToWhenVisible(this, docView, 'change:viewattributes', changeViewAttributesHandler);
        gridPane.listenToWhenVisible(this, docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // post-processing after scrolling, or after selection has been rendered
        this.listenTo(gridPane, 'change:scrollpos', updateResizeHandlePositions);

        // repaint the remote drawing selections after manipulating the drawing collection
        gridPane.listenToWhenVisible(this, docView, 'insert:drawing delete:drawing change:drawing', renderRemoteSelections);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = gridPane = docModel = docView = null;
            remoteLayerNode = selectionLayerNode = null;
        });

    } // class SelectionRenderer

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SelectionRenderer });

});
