/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/viewfuncmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/subtotalresult',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/dialogs',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, IteratorUtils, Color, Border, MixedBorder, HyperlinkUtils, ImageUtils, Config, SheetUtils, PaneUtils, SubtotalResult, FormulaUtils, TokenArray, Labels, Dialogs, gt) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var IntervalArray = SheetUtils.IntervalArray;
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;
    var Scalar = FormulaUtils.Scalar;
    var Matrix = FormulaUtils.Matrix;
    var Dimension = FormulaUtils.Dimension;

    // tolerance for optimal column width, in 1/100 mm
    var SET_COLUMN_WIDTH_TOLERANCE = 50;

    // tolerance for optimal row height (set explicitly), in 1/100 mm
    var SET_ROW_HEIGHT_TOLERANCE = 15;

    // tolerance for optimal row height (updated implicitly), in 1/100 mm
    var UPDATE_ROW_HEIGHT_TOLERANCE = 50;

    // default value for a visible border
    var DEFAULT_SINGLE_BORDER = { style: 'single', width: Utils.convertLengthToHmm(1, 'px'), color: Color.AUTO };

    // maximum size of an image in any direction, in 1/100 mm
    var MAX_IMAGE_SIZE = 21000;

    // mix-in class ViewFuncMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides implementations
     * of complex document functionality and status requests, called from the
     * application controller.
     *
     * @constructor
     */
    function ViewFuncMixin() {

        // self reference (spreadsheet view instance)
        var self = this;

        // the application instance
        var app = this.getApp();

        // the spreadsheet model, and other model objects
        var docModel = this.getDocModel();
        var undoManager = docModel.getUndoManager();
        var numberFormatter = docModel.getNumberFormatter();
        var formulaGrammar = docModel.getFormulaGrammar('ui');
        var autoStyles = docModel.getCellAutoStyles();

        // the active sheet model and index
        var activeSheetModel = null;
        var activeSheet = -1;

        // collections of the active sheet
        var colCollection = null;
        var rowCollection = null;
        var mergeCollection = null;
        var cellCollection = null;
        var hyperlinkCollection = null;
        var tableCollection = null;
        var drawingCollection = null;

        // the contents and formatting of the active cell
        var activeAddress = Address.A1;
        var activeCellValue = null;
        var activeTokenDesc = null;
        var activeCellDisplay = '';
        var activeStyleId = null;
        var activeAttributeSet = null;
        var activeParsedFormat = null;

        // the mixed borders of the current selection
        var selectedMixedBorders = {};

        // a deferred object used to wait for the mixed borders of the current selection
        var mixedBordersDef = this.createDeferredObject({ infoString: 'ViewfuncMixin.mixedBordersDef', app: app });

        // the subtotal data of the current selection
        var selectedSubtotals = new SubtotalResult();

        // private methods ----------------------------------------------------

        /**
         * Updates the cached data of the active cell (contents, formatting),
         * and the entire selection (border settings, subtotals). To improve
         * performance, updating settings of the entire selection (which is a
         * potentially expensive operation) will be done in a sliced background
         * loop.
         *
         * @param {Boolean} forceUpdate
         *  If set to true, current background timers for mixed border settings
         *  and subtotal results will always be aborted. By default, the
         *  background timers will not be aborted if the selected ranges have
         *  not been changed (e.g. while traversing the active cell inside the
         *  selected ranges).
         */
        var updateSelectionSettings = (function () {

            // initial delay before the background loops will be started
            var INITIAL_DELAY_OPTIONS = { delay: 300, infoString: 'ViewFuncMixin.initialDelayTimer', app: app };
            // options for the background timers to collect borders and subtotals in the cell collection
            var BACKGROUND_TIMER_OPTIONS = { slice: 200, interval: 50, app: app };
            // options for nested background timers
            var NESTED_TIMER_OPTIONS = _.extend({ delay: 'immediate' }, BACKGROUND_TIMER_OPTIONS);

            var PROCESS_OUTERBORDER_OPTIONS = _.extend({ infoString: 'ViewFuncMixin: processOuterBorder' }, NESTED_TIMER_OPTIONS);
            var PROCESS_BORDERRANGES_OPTIONS = _.extend({ infoString: 'ViewFuncMixin: processBorderRanges' }, NESTED_TIMER_OPTIONS);
            var GET_MIXEDBORDERS_OPTIONS = _.extend({ infoString: 'ViewFuncMixin: getMixedBorders' }, NESTED_TIMER_OPTIONS);

            var COLLECT_SUBTOTAL_RESULTS_OPTIONS = _.extend({ infoString: 'ViewFuncMixin: collectSubtotalResults' }, BACKGROUND_TIMER_OPTIONS);
            var GET_MIXEDBORDERS_BACKGROUND_OPTIONS = _.extend({ infoString: 'ViewFuncMixin: getMixedBorders.background' }, BACKGROUND_TIMER_OPTIONS);

            // timers for the background loops to collect borders and subtotal results
            var initialDelayTimer = null;
            var mixedBorderTimer = null;
            var subtotalResultTimer = null;
            // the selected cell ranges currently used to collect borders and subtotals
            var selectedRanges = new RangeArray();
            // the new mixed borders, mapped by border keys
            var newMixedBorders = null;

            function getMixedBorder(mixedBorder, border) {
                return mixedBorder ? MixedBorder.mixBorders(mixedBorder, border) : _.clone(border);
            }

            function updateMixedBorder(key, border) {
                var mixedBorder = newMixedBorders[key];
                return (newMixedBorders[key] = getMixedBorder(mixedBorder, border));
            }

            // creates an iterator that provides cell ranges with existing border attributes
            function createBorderAttrIterator(range, visitHor, visitVert, visitDiag) {
                var iterator = cellCollection.createStyleIterator(range, { covered: true });
                return IteratorUtils.createTransformIterator(iterator, function (styleRange, result) {
                    var borderMap = autoStyles.getBorderMap(result.style);
                    var hasHor  = visitHor  && (borderMap.t || borderMap.b);
                    var hasVert = visitVert && (borderMap.l || borderMap.r);
                    var hasDiag = visitDiag && (borderMap.d || borderMap.u);
                    if (!hasHor && !hasVert && !hasDiag) { return null; }
                    styleRange.borderMap = borderMap;
                    return result;
                });
            }

            function processOuterBorder(range, keyH, keyV) {

                var vertical = !keyH;
                var key = vertical ? keyV : keyH;
                var borderLength = 0;

                var iterator = createBorderAttrIterator(range, !vertical, vertical, false);
                var promise = self.iterateSliced(iterator, function (styleRange) {
                    updateMixedBorder(key, styleRange.borderMap[key] || Border.NONE);
                    borderLength += styleRange.size(!vertical);
                }, PROCESS_OUTERBORDER_OPTIONS);

                return promise.then(function () {
                    if (borderLength < range.size(!vertical)) {
                        updateMixedBorder(key, Border.NONE);
                    }
                });
            }

            // Returns the parts of the passed band interval with visible borders, and the resulting mixed border.
            // The resulting interval array contains intervals with additional property 'mixedBorder' that will be
            // null if the entire interval does not contain a visible border (i.e. if the result array is empty).
            function getBorderIntervals(bandInterval, vertical, leading) {

                var key = SheetUtils.getOuterBorderKey(vertical, leading);
                var mixedBorder = null;

                var borderIntervals = IntervalArray.map(bandInterval.ranges, function (attrRange) {
                    var border = attrRange.borderMap[key];
                    if (Border.isVisibleBorder(border)) {
                        mixedBorder = getMixedBorder(mixedBorder, border);
                        var interval = attrRange.interval(!vertical);
                        interval.border = border;
                        return interval;
                    }
                });

                // variable 'mixedBorder' still null: no visible borders in the band
                borderIntervals.mixedBorder = mixedBorder;
                return borderIntervals;
            }

            function processBorderRanges(range, borderRanges, key, vertical) {

                var bandIntervals = vertical ? borderRanges.getColBands({ ranges: true }) : borderRanges.getRowBands({ ranges: true });
                var bandLength = range.size(!vertical);
                var first = range.getStart(vertical);
                var last = range.getEnd(vertical);

                // calculates the mixed borders of the passed adjoining intervals
                function processBorderIntervals(intervals1, intervals2) {

                    // Add the precalculated visible mixed borders stored in the interval arrays.
                    var mixedBorder = null;
                    if (intervals1.mixedBorder) { mixedBorder = updateMixedBorder(key, intervals1.mixedBorder); }
                    if (intervals2 && intervals2.mixedBorder) { mixedBorder = updateMixedBorder(key, intervals2.mixedBorder); }

                    // Mix in an additional invisible border, if the visible border intervals do not cover the entire
                    // band length (but do not calculate the size of all border intervals, if the mixed border already
                    // has 'mixed' state, i.e. it already contains visible and invisible border states).
                    if (!mixedBorder || !mixedBorder.mixed) {
                        // merge the passed interval arrays to determine the intervals covered by either of the  arrays
                        // (these intervals contain a visible border, and the remaining intervals do not have a border)
                        var borderLength = intervals2 ? intervals1.concat(intervals2).merge().size() : intervals1.size();
                        if (borderLength < bandLength) { updateMixedBorder(key, Border.NONE); }
                    }
                }

                // no band intervals available: range does not contain any visible border lines
                if (bandIntervals.empty()) {
                    updateMixedBorder(key, Border.NONE);
                    return $.when();
                }

                // add empty border, if there is a gap with inner borders before the first, after the last, or between band intervals
                if ((first + 1 < bandIntervals.first().first) || (bandIntervals.last().last + 1 < last) || bandIntervals.some(function (bandInterval, index) {
                    return (index > 0) && (bandIntervals[index - 1].last + 2 < bandInterval.first);
                })) {
                    updateMixedBorder(key, Border.NONE);
                }

                return self.iterateArraySliced(bandIntervals, function (bandInterval, index) {

                    // whether to process the borders for the leading boundary of the current band (ignore leading boundary of leading range border)
                    var useLeading = first < bandInterval.first;
                    // whether to process the inner borders in the current band (the band must contain at least two columns/rows)
                    var useInner = bandInterval.size() > 1;
                    // whether to process the borders for the trailing boundary of the current band (ignore trailing boundary of trailing range border)
                    var useTrailing = bandInterval.last < last;

                    // the intervals with visible border attributes at the leading side of the band
                    bandInterval.leading = (useLeading || useInner) ? getBorderIntervals(bandInterval, vertical, true) : null;
                    // the intervals with visible border attributes at the trailing side of the band
                    bandInterval.trailing = (useInner || useTrailing) ? getBorderIntervals(bandInterval, vertical, false) : null;

                    // the preceding band interval, and whether it adjoins to the current band
                    var prevBandInterval = bandIntervals[index - 1];
                    var adjoinsPrevBand = prevBandInterval && (prevBandInterval.last + 1 === bandInterval.first);

                    // the next band interval, and whether it adjoins to the current band
                    var nextBandInterval = bandIntervals[index + 1];
                    var adjoinsNextBand = nextBandInterval && (bandInterval.last + 1 === nextBandInterval.first);

                    // combine the trailing borders of the preceding adjoined band with the leading borders of the current band
                    if (adjoinsPrevBand) { processBorderIntervals(prevBandInterval.trailing, bandInterval.leading); }

                    // process the leading borders of the current band, if the previous band does not adjoin
                    if (useLeading && !adjoinsPrevBand) { processBorderIntervals(bandInterval.leading); }

                    // combine the inner borders in the current band
                    if (useInner) { processBorderIntervals(bandInterval.leading, bandInterval.trailing); }

                    // combine the trailing borders of the current band, if the next band does not adjoin
                    // (adjoining bands will be handled in the next iteration of this loop)
                    if (useTrailing && !adjoinsNextBand) { processBorderIntervals(bandInterval.trailing); }

                }, PROCESS_BORDERRANGES_OPTIONS);
            }

            // process each range in the passed array (exit the loop early, if mixed border becomes ambiguous)
            function getMixedBorders(ranges, keyH, keyV) {
                return self.iterateArraySliced(ranges, function (range) {

                    // early exit, if the collected mixed borders become completely ambiguous
                    var skipH = !keyH || MixedBorder.isFullyAmbiguousBorder(newMixedBorders[keyH]);
                    var skipV = !keyV || MixedBorder.isFullyAmbiguousBorder(newMixedBorders[keyV]);
                    if (skipH && skipV) { return Utils.BREAK; }

                    // property '_outerBorder' used for ranges located at the sheet boundary (do not collect inner borders)
                    if (range._outerBorder) { return processOuterBorder(range, keyH, keyV); }

                    // otherwise, collect the inner border attributes (skip ranges without inner borders)
                    var processH = !skipH && !range.singleRow();
                    var processV = !skipV && !range.singleCol();
                    if (!processH && !processV) { return; }

                    // use a cell style iterator to collect all cell ranges with visible borders into an array
                    var iterator = createBorderAttrIterator(range, processH, processV, false);
                    var promise2 = self.reduceSliced(new RangeArray(), iterator, function (borderRanges, borderRange) {
                        borderRanges.push(borderRange);
                        return borderRanges;
                    }, GET_MIXEDBORDERS_OPTIONS);

                    // process the collected ranges with visible border attributes
                    return promise2.then(function (borderRanges) {
                        return Utils.invokeChainedAsync(
                            processH ? function () { return processBorderRanges(range, borderRanges, keyH, false); } : null,
                            processV ? function () { return processBorderRanges(range, borderRanges, keyV, true); } : null
                        );
                    });
                }, GET_MIXEDBORDERS_BACKGROUND_OPTIONS);
            }

            // collects the mixed border states in the current selection in a background loop
            var collectMixedBorders = SheetUtils.profileAsyncMethod('ViewFuncMixin.collectMixedBorders()', function () {

                var maxCol = docModel.getMaxCol();
                var maxRow = docModel.getMaxRow();

                // Build the iterator ranges for the outer borders of the selected ranges. Use the border column/row inside the range,
                // and the adjacent column/row outside the range, to include border attributes coming from outside cells next to the
                // selected ranges. Range borders located directly at the sheet boundary will not be included here (no cells from
                // outside available). These ranges will be handled below.
                // Example: For the selection range B2:E5, the cells B1:E2 will be used to find the top border settings.
                var borderRangesT = RangeArray.map(selectedRanges, function (range) { return (range.start[1] === 0)    ? null : range.rowRange(-1, 2); });
                var borderRangesB = RangeArray.map(selectedRanges, function (range) { return (range.end[1] === maxRow) ? null : range.rowRange(range.rows() - 1, 2); });
                var borderRangesL = RangeArray.map(selectedRanges, function (range) { return (range.start[0] === 0)    ? null : range.colRange(-1, 2); });
                var borderRangesR = RangeArray.map(selectedRanges, function (range) { return (range.end[0] === maxCol) ? null : range.colRange(range.cols() - 1, 2); });

                // Build the iterator ranges for the outer borders of the selected ranges that are located at the sheet boundary.
                // Example: For the selection range B1:E5, the inner cells B1:E1 will be used to find the top border settings.
                borderRangesT.append(Utils.addProperty(RangeArray.map(selectedRanges, function (range) { return (range.start[1] === 0)    ? range.headerRow()   : null; }), '_outerBorder', true));
                borderRangesB.append(Utils.addProperty(RangeArray.map(selectedRanges, function (range) { return (range.end[1] === maxRow) ? range.footerRow()   : null; }), '_outerBorder', true));
                borderRangesL.append(Utils.addProperty(RangeArray.map(selectedRanges, function (range) { return (range.start[0] === 0)    ? range.leadingCol()  : null; }), '_outerBorder', true));
                borderRangesR.append(Utils.addProperty(RangeArray.map(selectedRanges, function (range) { return (range.end[0] === maxCol) ? range.trailingCol() : null; }), '_outerBorder', true));

                // collect all mixed borders of all outer and inner border positions
                newMixedBorders = {};
                mixedBorderTimer = Utils.invokeChainedAsync(
                    function () { return getMixedBorders(selectedRanges, 'h', 'v'); },
                    function () { return getMixedBorders(borderRangesT, 't', null); },
                    function () { return getMixedBorders(borderRangesB, 'b', null); },
                    function () { return getMixedBorders(borderRangesL, null, 'l'); },
                    function () { return getMixedBorders(borderRangesR, null, 'r'); }
                );

                // resolve the deferred object, provide the new mixed borders as value
                return mixedBorderTimer.done(function () {

                    // add missing entries (if the ranges do not contain any visible border)
                    selectedMixedBorders = {};
                    _.each('tblrhv', function (key) {
                        var borderName = SheetUtils.getBorderName(key);
                        selectedMixedBorders[borderName] = newMixedBorders[key] || _.clone(Border.NONE);
                    });

                    mixedBordersDef.resolve(selectedMixedBorders);
                });
            });

            // callback function for the iterator loop
            function reduceSubtotalResult(subtotals, result) {
                return subtotals.add(result);
            }

            // collects the subtotal results in the current selection in a background loop
            var collectSubtotalResults = SheetUtils.profileAsyncMethod('ViewFuncMixin.collectSubtotalResults()', function () {

                // collect the subtotal results of the selected ranges
                var iterator = cellCollection.createSubtotalIterator(selectedRanges);
                subtotalResultTimer = self.reduceSliced(new SubtotalResult(), iterator, reduceSubtotalResult, COLLECT_SUBTOTAL_RESULTS_OPTIONS);

                // copy the resulting subtotals, and notify all listeners
                return subtotalResultTimer.done(function (subtotals) {
                    selectedSubtotals = subtotals;
                    self.trigger('update:selection:data');
                });
            });

            // the actual implementation of the method updateSelectionSettings()
            return function (forceUpdate) {

                // update settings of the active cell synchronously for fast GUI feedback
                activeAddress = self.getActiveCell();
                activeCellValue = cellCollection.getValue(activeAddress);
                activeTokenDesc = cellCollection.getTokenArray(activeAddress, { fullMatrix: true, grammarId: 'ui' });
                activeCellDisplay = cellCollection.getDisplayString(activeAddress);
                activeStyleId = cellCollection.getStyleId(activeAddress);
                activeAttributeSet = cellCollection.getAttributeSet(activeAddress);
                activeParsedFormat = cellCollection.getParsedFormat(activeAddress);

                // abort running timers, and restart the background loops
                var newSelectedRanges = self.getSelectedRanges();
                if (forceUpdate || !selectedRanges.equals(newSelectedRanges, true)) {

                    // a new pending deferred object that can be used to wait for the border settings
                    if (mixedBordersDef.state() !== 'pending') {
                        mixedBordersDef = self.createDeferredObject({ infoString: 'ViewFuncMixin.mixedBordersDef 2', app: app });
                    }

                    // abort running timers
                    if (initialDelayTimer) { initialDelayTimer.abort(); initialDelayTimer = null; }
                    if (mixedBorderTimer) { mixedBorderTimer.abort(); mixedBorderTimer = null; }
                    if (subtotalResultTimer) { subtotalResultTimer.abort(); subtotalResultTimer = null; }

                    // cache the current selection ranges
                    selectedRanges = newSelectedRanges;

                    // restart the background loops after a delay
                    initialDelayTimer = self.executeDelayed(_.noop, INITIAL_DELAY_OPTIONS);
                    initialDelayTimer.then(collectMixedBorders).then(collectSubtotalResults);
                }
            };
        }());

        /**
         * Returns the column/row intervals of the selected ranges, optionally
         * depending on a specific target column/row.
         *
         * @param {Boolean} columns
         *  Whether to return column intervals (true) or row intervals (false).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.target]
         *      If specified, the zero-based index of a specific target column
         *      or row. If the current selection contains entire column/row
         *      ranges covering this column/row, the respective intervals of
         *      these ranges will be returned instead; otherwise a single
         *      interval containing the target column/row will be returned.
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the returned intervals will be reduced to the
         *      column or row intervals covered by the visible panes.
         *
         * @returns {IntervalArray}
         *  An array of column/row intervals for the current selection, already
         *  merged and sorted.
         */
        function getSelectedIntervals(columns, options) {

            // the selected ranges
            var ranges = self.getSelectedRanges();
            // the target column/row
            var target = Utils.getIntegerOption(options, 'target');
            // the resulting intervals
            var intervals = null;

            // use entire column/row ranges in selection, if target index has been passed
            if (_.isNumber(target)) {
                // filter full column/row ranges in selection
                ranges = ranges.filter(function (range) { return docModel.isFullRange(range, columns); });
                // if the target column/row is not contained in the selection, make an interval for it
                if (!ranges.containsIndex(target, columns)) {
                    intervals = new IntervalArray(new Interval(target));
                }
            }

            // convert selected ranges to column/row intervals
            if (!intervals) {
                intervals = ranges.intervals(columns);
            }

            // reduce the intervals to the header pane intervals
            if (Utils.getBooleanOption(options, 'visible', false)) {
                intervals = self.getVisibleIntervals(intervals, columns);
            }

            // merge and sort the resulting intervals
            return intervals.merge();
        }

        /**
         * Returns the column/row intervals from the current selection, if it
         * contains hidden columns/rows. If the selected columns/rows are
         * completely visible, the selection consists of a single column/row
         * interval, and that interval is located next to a hidden column/row
         * interval (at either side, preferring leading), that hidden interval
         * will be returned instead.
         *
         * @returns {IntervalArray}
         *  The column/row intervals calculated from the current selection,
         *  containing hidden columns/rows. May be an empty array.
         */
        function getSelectedHiddenIntervals(columns) {

            // the entire selected intervals
            var intervals = getSelectedIntervals(columns);

            // safety check, should not happen
            if (intervals.empty()) { return intervals; }

            // the column/row collection
            var collection = columns ? colCollection : rowCollection;
            // the mixed column/row attributes
            var attributes = collection.getMixedAttributes(intervals);

            // if attribute 'visible' is false or null, the intervals contain hidden entries
            if (attributes.visible !== true) { return intervals; }

            // the resulting intervals
            var resultIntervals = new IntervalArray();

            // no special behavior for multiple visible intervals
            if (intervals.length > 1) { return resultIntervals; }

            // try to find preceding hidden interval
            var lastIndex = intervals.first().first - 1;
            if ((lastIndex >= 0) && !collection.isEntryVisible(lastIndex)) {
                var prevEntry = collection.getPrevVisibleEntry(lastIndex);
                resultIntervals.push(new Interval(prevEntry ? (prevEntry.index + 1) : 0, lastIndex));
                return resultIntervals;
            }

            // try to find following hidden interval
            var firstIndex = intervals[0].last + 1;
            if ((firstIndex <= collection.getMaxIndex()) && !collection.isEntryVisible(firstIndex)) {
                var nextEntry = collection.getNextVisibleEntry(firstIndex);
                resultIntervals.push(new Interval(firstIndex, nextEntry ? (nextEntry.index - 1) : collection.getMaxIndex()));
                return resultIntervals;
            }

            // no hidden intervals found
            return resultIntervals;
        }

        /**
         * Sets the optimal column width in the specified column intervals,
         * based on the content of cells.
         *
         * @param {IntervalArray|Interval} colIntervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function setOptimalColumnWidth(colIntervals) {

            // resulting intervals with the new column widths
            var resultIntervals = new IntervalArray();
            // default width for empty columns
            var defWidth = docModel.getDefaultAttributes('column').width;
            // maximum width for columns
            var maxWidth = docModel.getMaxColWidthHmm();
            // the rendering cache of the active sheet
            var renderCache = self.getRenderCache();
            // left and right cell padding, plus one pixel grid line
            var totalPadding = 2 * activeSheetModel.getEffectiveTextPadding() + 1;
            // the column iterator
            var colIt = colCollection.createIterator(colIntervals, { visible: true });

            // process all visible columns
            var promise = self.iterateSliced(colIt, function (colDesc) {

                // the new column width
                var colWidth = 0;

                renderCache.iterateCellsInCol(colDesc.index, function (element) {
                    if (!element.hasMergedColumns() && (element.contentWidth > 0)) {
                        colWidth = Math.max(colWidth, element.contentWidth + totalPadding);
                    }
                });

                // convert to 1/100 mm; or use default column width, if no cells are available
                colWidth = (colWidth === 0) ? defWidth : activeSheetModel.convertPixelToHmm(colWidth);

                // bug 40737: restrict to maximum column width allowed in the UI
                // TODO: use real maximum according to file format: 255 digits in OOXML, 1000mm in ODF
                colWidth = Math.min(colWidth, maxWidth);

                // do not generate operations, if the custom width was not modified before, and the new width is inside the tolerance
                if (!colDesc.merged.customWidth && (Math.abs(colWidth - colDesc.merged.width) < SET_COLUMN_WIDTH_TOLERANCE)) {
                    return;
                }

                // insert the optimal column width into the cache
                var lastInterval = resultIntervals.last();
                if (lastInterval && (lastInterval.last + 1 === colDesc.index) && (lastInterval.fillData.attrs.width === colWidth)) {
                    lastInterval.last += 1;
                } else {
                    lastInterval = new Interval(colDesc.index);
                    lastInterval.fillData = { attrs: { width: colWidth, customWidth: false } };
                    resultIntervals.push(lastInterval);
                }
            }, { infoString: 'ViewFuncMixin: setOptimalColumnWidth', app: app });

            return promise.then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    return colCollection.generateIntervalOperations(generator, resultIntervals);
                }, { storeSelection: true });
            });
        }

        /**
         * Generates row operations to set the optimal row height in the
         * specified row intervals, based on the content of cells.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {IntervalArray|Interval} rowIntervals
         *  An array of row intervals, or a single row interval.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.update=false]
         *      If set to true, an operation has caused updating the automatic row
         *      heights implicitly. Rows with user-defined height (row attribute
         *      'customHeight' set to true) will not be changed in this case. By
         *      default, the optimal height of all rows in the passed intervals
         *      will be calculated (manual row heights will be reset).
         *  @param {RangeArray} [options.changed]
         *      Contains the changed ranges. Useful to recalculate only changed
         *      cells while calculating the optimal row height.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function setOptimalRowHeight(generator, rowIntervals, options) {

            var update  = Utils.getBooleanOption(options, 'update', false);
            var changed = Utils.getOption(options, 'changed', null);

            // resulting intervals with the new row heights
            var resultIntervals = new IntervalArray();
            // maximum height for rows
            var maxHeight = docModel.getMaxRowHeightHmm();
            // tolerance for automatic row height
            var heightTolerance = update ? UPDATE_ROW_HEIGHT_TOLERANCE : SET_ROW_HEIGHT_TOLERANCE;
            // the rendering cache of the active sheet
            var renderCache = self.getRenderCache();
            // the row iterator
            var rowIt = rowCollection.createIterator(rowIntervals, { visible: true });

            // do not change custom row height in update mode
            if (update) {
                rowIt = IteratorUtils.createFilterIterator(rowIt, function (rowDesc) {
                    return !rowDesc.merged.customHeight;
                });
            }

            // process all visible rows
            var promise = self.iterateSliced(rowIt, function (rowDesc) {

                // calculate resulting optimal height of the current row
                var rowHeight = docModel.getRowHeightHmm(rowDesc.attributes.character);

                renderCache.iterateCellsInRow(rowDesc.index, function (element) {

                    // use the renderCache parameters if
                    //    - the current element has no merged rows
                    //      and
                    //    - the current element has a height bigger than "0"
                    //      and
                    //      - we handle no update
                    //        or
                    //      - we handle an update but the current cell was NOT chanced
                    if (!element.hasMergedRows() && (element.contentHeight > 0) && ((update && !changed.containsAddress(element.address)) || !update)) {
                        rowHeight = Math.max(rowHeight, activeSheetModel.convertPixelToHmm(element.contentHeight));
                    }

                    // only calculate with new parameters, if we handle an update AND the current cell was changed
                    if (update && changed.containsAddress(element.address)) {
                        var attrs            = cellCollection.getAttributeSet(element.address),
                            currentRowHeight = docModel.getRowHeightHmm(attrs.character),
                            font             = docModel.getFontCollection().getFont(attrs.character, { zoom: self.getZoom() }),
                            innerRect        = activeSheetModel.getCellRectangle(element.address),
                            availableWidth   = Math.max(2, innerRect.width - 2 * activeSheetModel.getEffectiveTextPadding()),
                            lines            = 0,
                            cellModel        = cellCollection.getCellModel(element.address);

                        // only calculate lines if 'wrapText' is on
                        if (cellModel && cellModel.d) {
                            cellModel.d.split(/\n/).forEach(function (paragraph) {
                                // replace any control characters with space characters
                                paragraph = Utils.cleanString(paragraph);

                                if (attrs.cell.wrapText) {
                                    // split paragraph to text lines
                                    lines += font.getTextLines(paragraph, availableWidth).length;
                                } else {
                                    lines++;
                                }
                            });

                            rowHeight = Math.max(rowHeight, (lines * currentRowHeight));
                        }
                    }
                });

                // bug 40737: restrict to maximum row height allowed in the UI
                // TODO: use real maximum according to file format: 409pt in OOXML, 1000mm in ODF
                rowHeight = Math.min(rowHeight, maxHeight);

                // do not generate operations, if the row height was optimal before, and the new height is inside the tolerance
                if (!rowDesc.merged.customHeight && (Math.abs(rowHeight - rowDesc.merged.height) < heightTolerance)) {
                    return;
                }

                // insert the optimal row height into the cache
                var lastInterval = resultIntervals.last();
                if (lastInterval && (lastInterval.last + 1 === rowDesc.index) && (lastInterval.fillData.attrs.height === rowHeight)) {
                    lastInterval.last += 1;
                } else {
                    lastInterval = new Interval(rowDesc.index);
                    lastInterval.fillData = { attrs: { height: rowHeight, customHeight: false } };
                    resultIntervals.push(lastInterval);
                }
            }, { delay: 'immediate', infoString: 'ViewFuncMixin: setOptimalRowHeight', app: app });

            // create the operations for all rows
            return promise.then(function () {
                return rowCollection.generateIntervalOperations(generator, resultIntervals);
            });
        }

        /**
         * Generates row operations to update the optimal row height in the
         * specified ranges, based on the content of cells.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single range address.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function updateOptimalRowHeight(generator, ranges) {

            // the merged and sorted row intervals
            var rowIntervals = RangeArray.get(ranges).rowIntervals().merge();

            // restrict to visible area (performance)
            // TODO: iterate all row intervals, not single rows, important when updating entire column
            rowIntervals = self.getVisibleIntervals(rowIntervals, false);

            // set update mode (do not modify rows with custom height)
            return setOptimalRowHeight(generator, rowIntervals, { update: true, changed: ranges });
        }

        /**
         * Performs additional checks whether the contents of the passed cell
         * ranges can be edited, regardless of their 'unlocked' attribute, and
         * the lock state of the active sheet. Used as helper method by various
         * public methods of this class that all support the same options.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @options {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.lockTables='none']
         *      Specifies how to check table ranges covering the passed cell
         *      range addresses. See method ViewFuncMixin.ensureLockedRanges2()
         *      for more details.
         *  @param {String} [options.lockMatrixes='none']
         *      Specifies how to check matrix formulas covering the passed cell
         *      ranges. See method ViewFuncMixin.ensureLockedRanges2() for more
         *      details.
         *
         * @returns {String|Null}
         *  The value null, if the contents of the passed cell ranges can be
         *  edited; otherwise one of the following error codes:
         *  - 'table:change': If the option 'lockTables' has been set to
         *      'full', and the passed cell ranges overlap with a table range
         *      in the active sheet.
         *  - 'table:header:change': If the option 'lockTables' has been set to
         *      'header', and the passed cell ranges overlap with the header
         *      cells of a table range in the active sheet.
         *  - 'formula:matrix:change': If the option 'lockMatrixes' has been
         *      set, and the passed cell ranges overlap with a matrix formula
         *      in the active sheet.
         */
        function ensureEditableContents(ranges, options) {

            // check the table ranges according to the passed mode
            switch (Utils.getStringOption(options, 'lockTables', 'none')) {

                case 'full':
                    // auto-filter always editable (getAllTables() does not return the auto-filter)
                    if (tableCollection.getAllTables().some(function (tableModel) {
                        return ranges.overlaps(tableModel.getRange());
                    })) {
                        return 'table:change';
                    }
                    break;

                case 'header':
                    // auto-filter always editable (getAllTables() does not return the auto-filter)
                    if (tableCollection.getAllTables().some(function (tableModel) {
                        var headerRange = tableModel.getHeaderRange();
                        return headerRange && ranges.overlaps(headerRange);
                    })) {
                        return 'table:header:change';
                    }
                    break;
            }

            // DOCS-196: check the matrix formulas in the active sheet
            switch (Utils.getStringOption(options, 'lockMatrixes', 'none')) {

                case 'full':
                    if (cellCollection.rangesOverlapAnyMatrix(ranges)) {
                        return 'formula:matrix:change';
                    }
                    break;

                case 'partial':
                    if (cellCollection.rangesOverlapAnyMatrix(ranges, { partial: true })) {
                        return 'formula:matrix:change';
                    }
                    break;
            }

            // the passed ranges are considered to be editable
            return null;
        }

        /**
         * A little helper function for various public methods that return a
         * promise after checking various conditions in the active sheet.
         *
         * @param {Null|String} errorCode
         *  The value null to indicate a successful check; or an error code as
         *  string, intended to be passed to SpreadsheetView.yellMessage() or
         *  SpreadsheetView.yellOnFailure().
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.sync=false]
         *      If set to true, this method will return the passed error code
         *      directly instead of a promise.
         *
         * @returns {Promise|Null|String}
         *  A resolved promise, if the passed error code is null, or a promise
         *  that will be rejected with an object with 'cause' property set to
         *  the passed error code. If the option 'sync' has been set to true,
         *  the error code will be returned directly instead of a promise.
         */
        function createEnsureResult(errorCode, options) {

            // if 'sync' option is set, return the result directly
            if (Utils.getBooleanOption(options, 'sync', false)) { return errorCode; }

            // create the resulting (resolved or rejected) promise
            return errorCode ? SheetUtils.makeRejected(errorCode) : self.createResolvedPromise(null);
        }

        /**
         * Generates and applies the operations, and undo operations, to insert
         * the specified cell contents, and to update the optimal row height of
         * the affected cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the cell has been changed, or
         *  that will be rejected on any error.
         */
        function setCellContents(address, contents, options) {

            // whether to update the optimal row heights after changing the cells
            var updateRows = Utils.getBooleanOption(options, 'updateRows', false);
            // the cell range to be selected while applying the operations
            var selectRange = Utils.getOption(options, 'selectRange', null);

            // generate and apply the operations
            return activeSheetModel.createAndApplyOperations(function (generator) {

                // try to fill the cell
                var promise = cellCollection.generateCellContentOperations(generator, address, contents);

                // set optimal row height afterwards (for changed rows only)
                if (updateRows) {
                    promise = promise.then(function (changedRanges) {
                        return updateOptimalRowHeight(generator, changedRanges);
                    });
                }

                // select the range passed with the options
                if (selectRange) {
                    self.selectRange(selectRange);
                }

                return promise;
            }, { storeSelection: true });
        }

        /**
         * Generates and applies the operations, and undo operations, to set
         * the contents of a cell to the passed edit text (tries to convert to
         * an appropriate data type, or to calculate a formula string).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function parseCellContents(address, parseText, options) {

            // parse the passed text to a cell contents object
            var contents = cellCollection.parseCellValue(parseText, 'val', address);

            // if specified, ignore a recognized URL for a cell hyperlink
            if (Utils.getBooleanOption(options, 'skipHyperlink', false)) {
                delete contents.url;
            }

            // create and apply the cell operation, update optimal row height
            return setCellContents(address, contents, options);
        }

        /**
         * Generates and applies the operations, and undo operations to fill
         * all cell ranges with the same value and formatting, and to update
         * the optimal row height of the affected cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the cell has been changed, or
         *  that will be rejected on any error.
         */
        function fillRangeContents(ranges, contents, options) {

            // whether to update the optimal row heights after changing the cells
            var updateRows = Utils.getBooleanOption(options, 'updateRows', false);

            // generate and apply the operations
            return activeSheetModel.createAndApplyOperations(function (generator) {

                // try to fill the cell ranges
                var promise = cellCollection.generateFillOperations(generator, ranges, contents, options);

                // set optimal row height afterwards (for changed rows only)
                if (updateRows) {
                    promise = promise.then(function (changedRanges) {
                        return updateOptimalRowHeight(generator, changedRanges);
                    });
                }

                return promise;
            }, { storeSelection: true });
        }

        /**
         * Initializes all sheet-dependent class members according to the
         * current active sheet.
         */
        function changeActiveSheetHandler(event, sheet, sheetModel) {

            // store model instance and index of the new active sheet
            activeSheetModel = sheetModel;
            activeSheet = sheet;

            // get collections of the new active sheet
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();
            cellCollection = sheetModel.getCellCollection();
            hyperlinkCollection = sheetModel.getHyperlinkCollection();
            tableCollection = sheetModel.getTableCollection();
            drawingCollection = sheetModel.getDrawingCollection();

            // forced update of selection settings (do not try to use cached data from old sheet)
            updateSelectionSettings(true);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the subtotal results in the current selection.
         *
         * @returns {SubtotalResult}
         *  The subtotal results in the current selection.
         */
        this.getSubtotalResult = function () {
            return selectedSubtotals;
        };

        /**
         * Returns whether the active sheet is locked. In locked sheets it is
         * not possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether the active sheet is locked.
         */
        this.isSheetLocked = function () {
            return activeSheetModel.isLocked();
        };

        /**
         * Returns a resolved or rejected promise according to the locked state
         * of the sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.errorCode='sheet:locked']
         *      The message code specifying the warning message shown when the
         *      sheet is locked. See method SpreadsheetView.yellMessage() for
         *      details.
         *  @param {Boolean} [options.sync=false]
         *      If set to true, this method will be executed synchronously, and
         *      the return value will be null or an error code instead of a
         *      promise.
         *
         * @returns {jQuery.Promise|Null|String}
         *  A promise that will be resolved, if the active sheet is not locked;
         *  or that will be rejected with an object with 'cause' property set
         *  to one of the following error codes:
         *  - 'readonly': The entire document is in read-only state.
         *  - passed error code or 'sheet:locked': The active sheet is locked.
         *  If the option 'sync' has been set to true, the return value will be
         *  null if the sheet is not locked, otherwise one of the error codes
         *  mentioned above as string.
         */
        this.ensureUnlockedSheet = function (options) {

            // prerequisite is global document edit mode
            var errorCode = docModel.getEditMode() ? null : 'readonly';

            // return the specified error code, if the sheet is locked
            if (!errorCode && this.isSheetLocked()) {
                errorCode = Utils.getStringOption(options, 'errorCode', 'sheet:locked');
            }

            // if 'sync' option is set, return the result directly
            return createEnsureResult(errorCode, options);
        };

        /**
         * Returns a resolved or rejected promise according to the locked state
         * of the cells in the passed cell ranges. A cell is in locked state,
         * if its 'unlocked' attribute is not set, AND if the active sheet is
         * locked; or if the cell is part of another locked sheet structure,
         * according to the passed options.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @options {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.errorCode='cells:locked']
         *      The message code specifying the warning message shown when the
         *      ranges are locked. See method SpreadsheetView.yellMessage() for
         *      details.
         *  @param {String} [options.lockTables='none']
         *      Specifies how to check a table range covering the cell ranges
         *      (the auto-filter, however, is never considered to be locked):
         *      - 'none' (default): Table ranges are not locked per se.
         *      - 'full': The entire table range is locked.
         *      - 'header': Only the header cells of the table are locked.
         *  @param {String} [options.lockMatrixes='none']
         *      Specifies how to check the matrix formulas covering the passed
         *      cell ranges:
         *      - 'none' (default): Matrix formulas are not locked per se.
         *      - 'full': All matrix formulas are locked.
         *      - 'partial': Only matrix formulas covered partially are locked.
         *  @param {Boolean} [options.sync=false]
         *      If set to true, this method will be executed synchronously, and
         *      the return value will be null or an error code instead of a
         *      promise.
         *
         * @returns {jQuery.Promise}
         *  A resolved promise, if the active sheet is not locked, or if the
         *  cell ranges are not locked; otherwise (i.e. active sheet, AND the
         *  cell ranges are locked) a rejected promise with a specific error
         *  code. If the option 'sync' has been set to true, the return value
         *  will be null if the cell ranges are editable, otherwise an error
         *  code as string.
         */
        this.ensureUnlockedRanges = function (ranges, options) {

            // prerequisite is global document edit mode
            var errorCode = docModel.getEditMode() ? null : 'readonly';

            // try to find a locked cell in a locked sheet
            if (!errorCode && this.isSheetLocked()) {

                // shortcut: use cached settings of the active cell, otherwise search in cell collection
                ranges = RangeArray.get(ranges);
                var isActiveCell = (ranges.length === 1) && ranges.first().single() && ranges.first().startsAt(activeAddress);
                var isLocked = isActiveCell ? !activeAttributeSet.cell.unlocked : cellCollection.findCellWithAttributes(ranges, { cell: { unlocked: false } });

                // extract the actual error code from the passed options
                if (isLocked) { errorCode = Utils.getStringOption(options, 'errorCode', 'cells:locked'); }
            }

            // bug 39869: do not allow to change table ranges (according to passed options)
            if (!errorCode) { errorCode = ensureEditableContents(ranges, options); }

            // if 'sync' option is set, return the result directly
            return createEnsureResult(errorCode, options);
        };

        /**
         * Returns a resolved or rejected promise according to the locked state
         * of the cell ranges in the current selection. A cell is in locked
         * state, if its 'unlocked' attribute is not set, AND if the active
         * sheet is locked; or if it is part of another locked sheet structure,
         * according to the passed options.
         *
         * @options {Object} [options]
         *  Optional parameters. See description of the public method
         *  ViewFuncMixin.ensureUnlockedRanges() for details.
         *
         * @returns {jQuery.Promise|Null|String}
         *  A resolved promise, if the active sheet is not locked, or if the
         *  selected cell ranges are not locked; otherwise (i.e. active sheet,
         *  AND selected ranges are locked) a rejected promise with a specific
         *  error code. If the option 'sync' has been set to true, the return
         *  value will be null if the selected ranges are editable, otherwise
         *  an error code as string.
         */
        this.ensureUnlockedSelection = function (options) {
            return this.ensureUnlockedRanges(this.getSelectedRanges(), options);
        };

        /**
         * Returns a resolved or rejected promise according to the locked state
         * of the active cell in the current selection. The active cell is in
         * locked state, if its 'unlocked' attribute is not set, AND if the
         * active sheet is locked; or if the cell is part of another locked
         * sheet structure, according to the passed options.
         *
         * @options {Object} [options]
         *  Optional parameters. See description of the public method
         *  ViewFuncMixin.ensureUnlockedRanges() for details.
         *
         * @returns {jQuery.Promise|Null|String}
         *  A resolved promise, if the active sheet is not locked, or if the
         *  active cell in a locked sheet is not locked; otherwise (i.e. active
         *  sheet, AND active cell are locked) a rejected promise with a
         *  specific error code. If the option 'sync' has been set to true, the
         *  return value will be null if the active cell is editable, otherwise
         *  an error code as string.
         */
        this.ensureUnlockedActiveCell = function (options) {
            return this.ensureUnlockedRanges(new Range(activeAddress), options);
        };

        // sheet operations ---------------------------------------------------

        /**
         * Returns whether the passed sheet type is supported by this
         * application.
         *
         * @param {String} sheetType
         *  A sheet type identifier, as used in the 'insertSheet' operation.
         *
         * @returns {Boolean}
         *  Whether the passed sheet type is supported by this application.
         */
        this.isSheetTypeSupported = function (sheetType) {
            return (/^(worksheet)$/).test(sheetType);
        };

        /**
         * Returns the name of the active sheet.
         *
         * @returns {String}
         *  The name of the active sheet.
         */
        this.getSheetName = function () {
            return docModel.getSheetName(activeSheet);
        };

        /**
         * Returns the merged formatting attributes of the active sheet.
         *
         * @param {Boolean} [direct=false]
         *  If set to true, the returned attribute set will be a reference to
         *  the original map stored in the sheet moel instance, which MUST NOT
         *  be modified! By default, a deep clone of the attribute set will be
         *  returned that can be freely modified.
         *
         * @returns {Object}
         *  The merged attribute set of the active sheet.
         */
        this.getSheetAttributeSet = function (direct) {
            return activeSheetModel.getMergedAttributeSet(direct);
        };

        /**
         * Inserts a new sheet in the spreadsheet document, and activates it in
         * the view. In case of an error, shows a notification message.
         *
         * @returns {Boolean}
         *  Whether the new sheet has been inserted successfully.
         */
        this.insertSheet = function () {

            // server-side configuration contains the maximum number of sheets per document
            if (docModel.getSheetCount() >= Config.MAX_SHEET_COUNT) {
                return $.Deferred().reject();
            }

            // generate and apply the 'insertSheet' operation
            var promise = docModel.createAndApplyOperations(function (generator) {
                var sheetName = docModel.generateUnusedSheetName();
                return docModel.generateInsertSheetOperations(generator, activeSheet + 1, sheetName);
            }, { storeSelection: self.activateNextSheet.bind(self) });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Deletes the active sheet from the spreadsheet document, and
         * activates another sheet in the view. Does nothing, if the document
         * contains a single sheet only or if the user doesn't really want
         * to delete the sheet without undo after being asked for.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the active sheet has been
         *  deleted, or that will be rejected otherwise.
         */
        this.deleteSheet = function () {

            // do not delete the last visible sheet
            if (this.getVisibleSheetCount() <= 1) {
                return $.Deferred().reject();
            }

            // ask to proceed (deleting sheets cannot be undone)
            var promise = this.showQueryDialog(gt('Delete Sheet'), gt('The following delete sheet operation cannot be undone. Proceed operation?'));

            // generate and apply the 'deleteSheet' operation after confirmation
            promise = promise.then(function () {
                return docModel.createAndApplyOperations(function (generator) {
                    return docModel.generateDeleteSheetOperations(generator, activeSheet);
                }, { storeSelection: true, undoMode: 'clear' });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Moves the specified sheet in the document to a new position. In
         * difference to most other methods of this class, this method does NOT
         * work on the active sheet, but on an arbitrary sheet in the document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be moved.
         *
         * @param {Number} to
         *  The new zero-based index of the sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheet has been moved, or
         *  that will be rejected otherwise.
         */
        this.moveSheet = function (sheet, to) {

            // nothing to do, if the sheet does not move
            if (sheet === to) { return $.when(); }

            // generate and apply the 'moveSheet' operation
            var promise = docModel.createAndApplyOperations(function (generator) {
                return docModel.generateMoveSheetOperations(generator, sheet, to);
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Shows the 'Reorder Sheets' dialog and creates and applies the
         * respective 'moveSheet' operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the dialog has been closed,
         *  and the sheets have been reordered.
         */
        this.showReorderSheetsDialog = function () {

            // leave cell edit mode before showing the dialog
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // show the 'Reorder Sheets' dialog
            var promise = new Dialogs.SheetOrderDialog(this).show();

            // generate all 'moveSheet' operations
            promise = promise.then(function (moveActions) {
                // TODO: optimize the move actions
                return docModel.createAndApplyOperations(function (generator) {

                    // generate all move operations, and the undo operations
                    var promise2 = self.iterateArraySliced(moveActions, function (moveData) {
                        return docModel.generateMoveSheetOperations(generator, moveData.from, moveData.to);
                    }, { infoString: 'ViewFuncMixin: showReorderSheetsDialog', app: app });

                    // reverse the undo operations to correctly undo multiple moves
                    return promise2.done(function () { generator.reverseOperations({ undo: true }); });
                });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Copies the active sheet and inserts the copy with the provided sheet
         * name behind the active sheet. In case of an error, shows a warning
         * message.
         *
         * @param {String} sheetName
         *  The name of the new sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheet has been copied, or
         *  that will be rejected otherwise.
         */
        this.copySheet = function (sheetName) {

            // leave cell edit mode before copying the sheet
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // generate and apply the 'copySheet' operation
            var promise = docModel.createAndApplyOperations(function (generator) {
                return docModel.generateCopySheetOperations(generator, activeSheet, activeSheet + 1, sheetName);
            }, { storeSelection: self.activateNextSheet.bind(self) });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Shows a dialog that allows to enter a name of a new sheet copied
         * from the active sheet. The dialog will be kept open until a valid
         * sheet name has been entered.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved with the new
         *  sheet name, after the sheet has been copied successfully; or will
         *  be rejected, if the dialog has been canceled, or if the document
         *  has switched to read-only mode.
         */
        this.showCopySheetDialog = function () {

            // leave cell edit mode before copying the sheet
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // create the dialog, initialize with an unused sheet name
            var sheetName = docModel.generateUnusedSheetName();
            var dialog = new Dialogs.SheetNameDialog(this, sheetName, this.copySheet.bind(this), {
                title: gt('Copy Sheet'),
                okLabel: gt('Copy')
            });

            // show the dialog, copying the sheet will be done in the callback function
            // (the dialog repeats showing itself until a valid sheet name has been entered)
            return dialog.show();
        };

        /**
         * Renames the active sheet in the spreadsheet document, if possible.
         * In case of an error, shows a notification message.
         *
         * @param {String} sheetName
         *  The new name of the active sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheet has been renamed, or
         *  that will be rejected otherwise.
         */
        this.renameSheet = function (sheetName) {

            // leave cell edit mode before renaming the sheet (sheet name may be used in formulas)
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // generate and apply the 'renameSheet' operation
            var promise = docModel.createAndApplyOperations(function (generator) {
                return docModel.generateRenameSheetOperations(generator, activeSheet, sheetName);
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Shows a dialog that allows to enter a new name for the active sheet.
         * The dialog will be kept open until a valid sheet name has been
         * entered.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved with the new
         *  sheet name, after the sheet has been renamed successfully; or will
         *  rejected, if the dialog has been canceled, or if the document has
         *  switched to read-only mode.
         */
        this.showRenameSheetDialog = function () {

            // leave cell edit mode before renaming the sheet (sheet name may be used in formulas)
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // create the dialog, initialize with the name of the active sheet
            var dialog = new Dialogs.SheetNameDialog(this, this.getSheetName(), this.renameSheet.bind(this), {
                title: gt('Rename Sheet'),
                okLabel: gt('Rename')
            });

            // show the dialog, renaming the sheet will be done in the callback function
            // (the dialog repeats showing itself until a valid sheet name has been entered)
            return dialog.show();
        };

        /**
         * Changes the formatting attributes of the active sheet.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set with the new formatting attributes
         *  for the active sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheet has been changed, or
         *  that will be rejected otherwise.
         */
        this.changeSheet = function (attributeSet) {

            // leave cell edit mode before hiding the sheet
            if (_.isObject(attributeSet.sheet) && (attributeSet.sheet.visible === false)) {
                if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }
            }

            // generate and apply the operations
            var promise = activeSheetModel.createAndApplyOperations(function (generator) {
                return activeSheetModel.generateChangeSheetOperations(generator, attributeSet);
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Makes all hidden sheets in the view visible.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if all sheet have been made
         *  visible, or that will be rejected otherwise.
         */
        this.showAllSheets = function () {

            // the attribute set to be inserted into the operations
            var attributeSet = { sheet: { visible: true } };

            // generate the 'changeSheet' operations for all hidden sheets
            var promise = docModel.createAndApplyOperations(function (generator) {
                return docModel.generateOperationsForAllSheets(generator, function (sheetModel) {
                    return sheetModel.generateChangeSheetOperations(generator, attributeSet);
                }, { visible: false });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Unhide the given sheets.
         *
         * @param {Array<Number>} sheets the sheet indices for unhide
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheets are unhidden, or
         *  that will be rejected otherwise.
         */
        this.unhideSheets = function (sheets) {
            if (_.isEmpty(sheets)) {
                return $.when();
            }

            var attributeSet = { sheet: { visible: true } };
            // generate the 'changeSheet' operations for all hidden sheets
            var promise = docModel.createAndApplyOperations(function (generator) {
                return docModel.generateOperationsForAllSheets(generator, function (sheetModel) {
                    return sheets.indexOf(sheetModel.getIndex()) >= 0 ? sheetModel.generateChangeSheetOperations(generator, attributeSet) : $.when();
                }, { visible: false });
            });

            return promise;
        };

        /**
         * Shows a dialog that allows to unhide sheets.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved with the sheets
         *  to unhide, after the sheets has been unhide successfully; or will
         *  rejected, if the dialog has been canceled, or if the document has
         *  switched to read-only mode.
         */
        this.showUnhideSheetsDialog = function () {

            // create the dialog
            var dialog = new Dialogs.UnhideSheetsDialog(this, this.unhideSheets.bind(this));

            return dialog.show();

        };

        // sheet view operations ----------------------------------------------

        /**
         * Returns the value of a specific view attribute of the active sheet.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @returns {Any}
         *  A clone of the attribute value.
         */
        this.getSheetViewAttribute = function (name) {
            return activeSheetModel.getViewAttribute(name);
        };

        /**
         * Changes specific view attributes of the active sheet.
         *
         * @param {Object} attributes
         *  A map with all view attributes to be changed.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setSheetViewAttributes = function (attributes) {
            activeSheetModel.setViewAttributes(attributes);
            return this;
        };

        /**
         * Changes a specific view attribute of the active sheet.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @param {Any} value
         *  The new value for the view attribute.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setSheetViewAttribute = function (name, value) {
            activeSheetModel.setViewAttribute(name, value);
            return this;
        };

        /**
         * Returns the current zoom factor of the active sheet.
         *
         * @returns {Number}
         *  The current zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         */
        this.getZoom = function () {
            return activeSheetModel.getViewAttribute('zoom');
        };

        /**
         * Changes the current zoom factor of all sheets in the document.
         *
         * @param {Number} zoom
         *  The new zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setZoom = function (zoom) {
            activeSheetModel.setViewAttribute('zoom', zoom);
            return this;
        };

        /**
         * Decreases the zoom factor of the active sheet in the document.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.decZoom = function () {

            // current zoom factor
            var currZoom = this.getZoom();
            // find last entry in ZOOM_FACTORS with a factor less than current zoom
            var prevZoom = Utils.findLast(PaneUtils.ZOOM_FACTORS, function (zoom) { return zoom < currZoom; });

            // set new zoom at active sheet
            if (_.isNumber(prevZoom)) { this.setZoom(prevZoom); }
            return this;
        };

        /**
         * Increases the zoom factor of the active sheet in the document.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.incZoom = function () {

            // current zoom factor
            var currZoom = this.getZoom();
            // find first entry in ZOOM_FACTORS with a factor greater than current zoom
            var nextZoom = _.find(PaneUtils.ZOOM_FACTORS, function (zoom) { return zoom > currZoom; });

            // set new zoom at active sheet
            if (_.isNumber(nextZoom)) { this.setZoom(nextZoom); }
            return this;
        };

        /**
         * Generates and sends operations for all view attributes that have
         * been changed while the document was open.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.sendChangedViewAttributes = function () {

            // Only send changed view attributes when user has edit rights, and has modified
            // the document locally, or if 'important' sheet view settings have been changed.
            // This is needed to prevent creating new versions of the document after selecting
            // cells or scrolling around, but without editing anything.
            if (!docModel.getEditMode()) { return this; }

            // whether the document is modified, and view settings can be sent
            var modified = app.isLocallyModified();

            // detect if any sheet contains 'important' changed view attributes
            if (!modified) {
                modified = IteratorUtils.some(docModel.createSheetIterator(), function (sheetModel) {
                    return sheetModel.hasChangedViewAttributes();
                });
            }

            // nothing to do (e.g. only selection has been changed, which will not be saved then)
            if (!modified) { return this; }

            // generate and apply the operations for the document, and all affected sheets
            docModel.createAndApplyOperations(function (generator) {

                // create attribute operation for the active sheet index
                if (activeSheet !== docModel.getDocumentAttribute('activeSheet')) {
                    generator.generateDocAttrsOperation({ activeSheet: activeSheet });
                }

                // create operations for the changed view attributes of all sheets
                return docModel.generateOperationsForAllSheets(generator, function (sheetModel) {
                    var sheetAttributes = sheetModel.getChangedViewAttributes();
                    if (!_.isEmpty(sheetAttributes)) {
                        return sheetModel.generateChangeSheetOperations(generator, { sheet: sheetAttributes });
                    }
                });
            }, { undoMode: 'skip' }); // do not create an undo action for view settings
            return this;
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the active sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the active sheet.
         */
        this.hasSplit = function () {
            return activeSheetModel.hasSplit();
        };

        /**
         * Returns the current view split mode of the active sheet.
         *
         * @returns {String}
         *  The current view split mode. The empty string indicates that the
         *  view is not split at all. Otherwise, one of the following values
         *  will be returned:
         *  - 'split': Dynamic split mode (movable split lines).
         *  - 'frozen': Frozen split mode (a fixed number of columns and/or
         *      rows are frozen, split lines are not movable). When leaving
         *      frozen split mode, the split lines will be hidden.
         *  - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
         *      mode). When leaving frozen split mode, the split lines remain
         *      visible and become movable (dynamic 'split' mode).
         */
        this.getSplitMode = function () {
            return activeSheetModel.getSplitMode();
        };

        /**
         * Returns whether the view is currently split, and dynamic split mode
         * is activated. Returns also true, if the split view is currently
         * frozen, but will thaw to dynamic split mode.
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (split modes 'split' or
         *  'frozenSplit').
         */
        this.hasDynamicSplit = function () {
            return activeSheetModel.hasDynamicSplit();
        };

        /**
         * Returns whether the view is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (split modes 'frozen' or
         *  'frozenSplit').
         */
        this.hasFrozenSplit = function () {
            return activeSheetModel.hasFrozenSplit();
        };

        /**
         * Toggles the dynamic view split mode according to the current view
         * split mode, and the current selection. Enabling dynamic split mode
         * in a frozen split view will thaw the frozen panes. Otherwise,
         * enabling the dynamic split mode will split the view at the position
         * left of and above the active cell. Disabling the dynamic split mode
         * will remove the split also if the view is in frozen split mode.
         *
         * @param {Boolean} state
         *  The new state of the dynamic view split mode.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (state) {

            // whether the view is currently split
            var hasSplit = this.hasSplit();
            // the position of the active cell in the sheet
            var cellRectangle = null;
            // the visible area of the active grid pane that will be split
            var visibleRectangle = null;
            // the new split positions
            var splitWidth = 0, splitHeight = 0;
            // additional scroll anchor attributes
            var attributes = null;

            // disable dynamic or frozen split mode
            if (!state && hasSplit) {
                activeSheetModel.clearSplit();

            // convert frozen split to dynamic split
            } else if (state && this.hasFrozenSplit()) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());

            // set dynamic split at active cell
            } else if (state && !hasSplit) {

                // additional scroll anchor attributes
                attributes = {};

                // get position of active cell and visible area of the grid pane
                cellRectangle = this.getCellRectangle(activeAddress, { expandMerged: true });
                visibleRectangle = this.getActiveGridPane().getVisibleRectangle();

                // do not split vertically, if the active cell is shown at the left border
                splitWidth = cellRectangle.left - visibleRectangle.left;
                if ((-cellRectangle.width < splitWidth) && (splitWidth < PaneUtils.MIN_PANE_SIZE)) {
                    splitWidth = 0;
                }

                // do not split horizontally, if the active cell is shown at the top border
                splitHeight = cellRectangle.top - visibleRectangle.top;
                if ((-cellRectangle.height < splitHeight) && (splitHeight < PaneUtils.MIN_PANE_SIZE)) {
                    splitHeight = 0;
                }

                // split in the middle of the grid pane, if active cell is outside the visible area
                if (((splitWidth === 0) && (splitHeight === 0)) ||
                    (splitWidth < 0) || (splitWidth > visibleRectangle.width - PaneUtils.MIN_PANE_SIZE) ||
                    (splitHeight < 0) || (splitHeight > visibleRectangle.height - PaneUtils.MIN_PANE_SIZE)
                ) {
                    splitWidth = Math.floor(visibleRectangle.width / 2);
                    splitHeight = Math.floor(visibleRectangle.height / 2);
                    attributes.anchorRight = colCollection.getScrollAnchorByOffset(visibleRectangle.left + splitWidth, { pixel: true });
                    attributes.anchorBottom = rowCollection.getScrollAnchorByOffset(visibleRectangle.top + splitHeight, { pixel: true });
                    attributes.activePane = 'topLeft';
                } else {
                    if (splitWidth > 0) { attributes.anchorRight = activeAddress[0]; }
                    if (splitHeight > 0) { attributes.anchorBottom = activeAddress[1]; }
                }

                // activate the dynamic split view
                splitWidth = activeSheetModel.convertPixelToHmm(splitWidth);
                splitHeight = activeSheetModel.convertPixelToHmm(splitHeight);
                activeSheetModel.setDynamicSplit(splitWidth, splitHeight, attributes);
            }

            return this;
        };

        /**
         * Toggles the frozen view split mode according to the current view
         * split mode, and the current selection. Enabling frozen split mode
         * in a dynamic split view will freeze the panes with their current
         * size (split mode 'frozenSplit'). Otherwise, enabling the frozen
         * split mode will split the view at the position left of and above the
         * active cell. Disabling the frozen split mode will either return to
         * dynamic split mode (if split mode was 'frozenSplit'), or will remove
         * the frozen split at all (if split mode was 'frozen').
         *
         * @param {Boolean} state
         *  The new state of the frozen view split mode.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (state) {

            // whether the view is currently split
            var hasSplit = this.hasSplit();

            // convert frozen split to dynamic split
            if (!state && hasSplit && (this.getSplitMode() === 'frozenSplit')) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());

            // disable split mode completely
            } else if (!state && hasSplit) {
                activeSheetModel.clearSplit();

            // convert dynamic split to frozen split
            } else if (state && hasSplit && !this.hasFrozenSplit()) {
                activeSheetModel.setFrozenSplit(activeSheetModel.getSplitColInterval(), activeSheetModel.getSplitRowInterval());

            // enable frozen split mode
            } else if (state && !hasSplit) {

                var // calculate frozen column interval (must result in at least one frozen column)
                    colAnchor = activeSheetModel.getViewAttribute('anchorRight'),
                    colInterval = new Interval(Math.round(colAnchor), activeAddress[0] - 1);
                if (colInterval.first > colInterval.last) { colInterval = null; }

                var // calculate frozen row interval (must result in at least one frozen row)
                    rowAnchor = activeSheetModel.getViewAttribute('anchorBottom'),
                    rowInterval = new Interval(Math.round(rowAnchor), activeAddress[1] - 1);
                if (rowInterval.first > rowInterval.last) { rowInterval = null; }

                // activate the frozen split view
                activeSheetModel.setFrozenSplit(colInterval, rowInterval);
            }

            return this;
        };

        /**
         * Freezes the passed number of columns and rows in the active sheet,
         * regardless of the current split settings.
         *
         * @param {Number} cols
         *  The number of columns to freeze. The value zero will not freeze any
         *  column.
         *
         * @param {Number} rows
         *  The number of rows to freeze. The value zero will not freeze any
         *  row.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setFixedFrozenSplit = function (cols, rows) {

            // the scroll anchors for the first visible column/row
            var colAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasColSplit() ? 'anchorLeft' : 'anchorRight');
            var rowAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasRowSplit() ? 'anchorTop' : 'anchorBottom');
            // the first visible column and row in the split view (reduce to be able to freeze the passed number of columns/rows)
            var col = Math.max(0, Math.min(Math.round(colAnchor), docModel.getMaxCol() - cols + 1));
            var row = Math.max(0, Math.min(Math.round(rowAnchor), docModel.getMaxRow() - rows + 1));

            // activate the frozen split view
            activeSheetModel.setFrozenSplit((cols > 0) ? new Interval(col, col + cols - 1) : null, (rows > 0) ? new Interval(row, row + rows - 1) : null);

            return this;
        };

        // column operations --------------------------------------------------

        /**
         * Returns the mixed formatting attributes of all columns in the active
         * sheet covered by the current selection.
         *
         * @returns {Object}
         *  The mixed attributes of all selected columns, as simple object. All
         *  attributes that could not be resolved unambiguously will be set to
         *  the value null.
         */
        this.getColumnAttributes = function () {
            var intervals = getSelectedIntervals(true);
            return colCollection.getMixedAttributes(intervals);
        };

        /**
         * Returns the settings of the column the active cell is currently
         * located in.
         *
         * @returns {ColRowDescriptor}
         *  The column descriptor of the active cell.
         */
        this.getActiveColumnDescriptor = function () {
            return colCollection.getEntry(activeAddress[0]);
        };

        /**
         * Returns whether the current selection contains hidden columns, or is
         * located next to hidden columns, that can be made visible.
         *
         * @returns {Boolean}
         *  Whether the current selection contains hidden columns.
         */
        this.canShowColumns = function () {
            return getSelectedHiddenIntervals(true).length > 0;
        };

        /**
         * Inserts new columns into the active sheet, according to the current
         * selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.insertColumns = function () {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var colRanges = docModel.makeColRanges(getSelectedIntervals(true));
                    return cellCollection.generateMoveCellsOperations(generator, colRanges, 'right');
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Deletes existing columns from the active sheet, according to the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.deleteColumns = function () {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var colRanges = docModel.makeColRanges(getSelectedIntervals(true));
                    return cellCollection.generateMoveCellsOperations(generator, colRanges, 'left');
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Changes the specified column attributes in the active sheet,
         * according to the current selection.
         *
         * @param {Object} attributes
         *  The (incomplete) column attributes, as simple object (NOT mapped as
         *  a 'column' sub-object).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeColumns = function (attributes) {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var intervals = getSelectedIntervals(true);
                    return colCollection.generateFillOperations(generator, intervals, { attrs: attributes });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Shows all hidden columns in the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.showColumns = function () {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var intervals = getSelectedHiddenIntervals(true);
                    return colCollection.generateFillOperations(generator, intervals, { attrs: { visible: true } });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Changes the width of columns in the active sheet, according to the
         * current selection.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.target]
         *      The target column to be changed. If specified, the column width
         *      will be set to that column only. If the current selection
         *      contains any ranges covering this column completely, the column
         *      width will be set to all columns contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setColumnWidth = function (width, options) {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {

                    // the target column to be modified
                    var targetCol = Utils.getIntegerOption(options, 'target');
                    // the column intervals to be modified
                    var intervals = getSelectedIntervals(true, { target: targetCol });
                    // hide columns if passed column width is zero; show hidden columns when width is changed
                    var attributes = (width === 0) ? { visible: false } : { visible: true, width: width, customWidth: true };

                    return colCollection.generateFillOperations(generator, intervals, { attrs: attributes });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Sets optimal column width based on the content of cells.
         *
         * @param {Number} [targetCol]
         *  The target column to be changed. If the current selection contains
         *  any ranges covering this column completely, the optimal column
         *  width will be set to all columns contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setOptimalColumnWidth = function (targetCol) {
            var intervals = getSelectedIntervals(true, { target: targetCol, visible: true });
            return this.yellOnFailure(setOptimalColumnWidth(intervals));
        };

        // row operations -----------------------------------------------------

        /**
         * Returns the mixed formatting attributes of all rows in the active
         * sheet covered by the current selection.
         *
         * @returns {Object}
         *  The mixed attributes of all selected rows, as simple object. All
         *  attributes that could not be resolved unambiguously will be set to
         *  the value null.
         */
        this.getRowAttributes = function () {
            var intervals = getSelectedIntervals(false);
            return rowCollection.getMixedAttributes(intervals);
        };

        /**
         * Returns the settings of the row the active cell is currently located
         * in.
         *
         * @returns {ColRowDescriptor}
         *  The row descriptor of the active cell.
         */
        this.getActiveRowDescriptor = function () {
            return rowCollection.getEntry(activeAddress[1]);
        };

        /**
         * Returns whether the current selection contains hidden rows, or is
         * located next to hidden rows, that can be made visible.
         *
         * @returns {Boolean}
         *  Whether the current selection contains hidden rows.
         */
        this.canShowRows = function () {
            return getSelectedHiddenIntervals(false).length > 0;
        };

        /**
         * Inserts new rows into the active sheet, according to the current
         * selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.insertRows = function () {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var rowRanges = docModel.makeRowRanges(getSelectedIntervals(false));
                    return cellCollection.generateMoveCellsOperations(generator, rowRanges, 'down');
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Deletes existing rows from the active sheet, according to the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.deleteRows = function () {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var rowRanges = docModel.makeRowRanges(getSelectedIntervals(false));
                    return cellCollection.generateMoveCellsOperations(generator, rowRanges, 'up');
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Changes the specified row attributes in the active sheet, according
         * to the current selection..
         *
         * @param {Object} attributes
         *  The (incomplete) row attributes, as simple object (NOT mapped as a
         *  'row' sub-object).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeRows = function (attributes) {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var intervals = getSelectedIntervals(false);
                    return rowCollection.generateFillOperations(generator, intervals, { attrs: attributes });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Shows all hidden rows in the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.showRows = function () {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var intervals = getSelectedHiddenIntervals(false);
                    return rowCollection.generateFillOperations(generator, intervals, { attrs: { visible: true } });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Changes the height of rows in the active sheet, according to the
         * current selection.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.target]
         *      The target row to be changed. If specified, the row height will
         *      be set to that row only. If the current selection contains any
         *      ranges covering this row completely, the row height will be set
         *      to all rows contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setRowHeight = function (height, options) {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {

                    // the target row to be modified
                    var targetRow = Utils.getIntegerOption(options, 'target');
                    // the row intervals to be modified
                    var intervals = getSelectedIntervals(false, { target: targetRow });
                    // hide rows if passed row height is zero; show hidden rows when height is changed
                    var attributes = (height === 0) ? { visible: false } : { visible: true, height: height, customHeight: true };

                    return rowCollection.generateFillOperations(generator, intervals, { attrs: attributes });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Sets optimal row height based on the content of cells.
         *
         * @param {Number} [targetRow]
         *  The target row to be changed. If the current selection contains any
         *  ranges covering this row completely, the optimal row height will be
         *  set to all rows contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setOptimalRowHeight = function (targetRow, options) {
            var parentGenerator = Utils.getOption(options, 'generator', null),
                intervals       = getSelectedIntervals(false, { target: targetRow, visible: true }),
                promise         = null;

            if (parentGenerator) {
                promise = setOptimalRowHeight(parentGenerator, intervals, options);
            } else {
                // generate and apply the row operations
                promise = activeSheetModel.createAndApplyOperations(function (generator) {
                    return setOptimalRowHeight(generator, intervals, options);
                }, { storeSelection: true });
            }

            // show error alert on failure
            return this.yellOnFailure(promise);
        };

        // cell operations ----------------------------------------------------

        /**
         * Returns the value of the active cell in the current selection.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The scalar value of the active cell in the current selection.
         */
        this.getActiveCellValue = function () {
            return activeCellValue;
        };

        /**
         * Returns the formula expression of the active cell in the current
         * selection, translated to the current UI language.
         *
         * @returns {Object|Null}
         *  The token array descriptor of the active cell in the current
         *  selection, if the cell is a formula cell; otherwise null. See
         *  method CellCollection.getTokenArray() for details.
         */
        this.getActiveTokenDesc = function () {
            return activeTokenDesc;
        };

        /**
         * Returns the display string of the active cell in the current
         * selection.
         *
         * @returns {String|Null}
         *  The default display string of the specified cell (regardless of the
         *  column width); or null, if the cell cannot display its value (e.g.
         *  due to an invalid number format).
         */
        this.getActiveCellDisplay = function () {
            return activeCellDisplay;
        };

        /**
         * Returns the merged formatting attributes of the active cell in the
         * current selection.
         *
         * @returns {Object}
         *  All formatting attributes of the active cell, as complete merged
         *  attribute set. If the cell in-place edit mode is currently active,
         *  the edit attributes added there will be included too.
         */
        this.getActiveAttributeSet = function () {

            // the merged attributes of the active cell
            var cellAttributeSet = _.copy(activeAttributeSet, true);
            // additional formatting attributes from cell edit mode
            var editAttributeSet = this.getEditAttributeSet();

            if (_.isObject(editAttributeSet)) {
                _.extend(cellAttributeSet.cell, editAttributeSet.cell);
                _.extend(cellAttributeSet.character, editAttributeSet.character);
            }

            return cellAttributeSet;
        };

        /**
         * Returns the parsed number format of the active cell in the current
         * selection.
         *
         * @returns {ParsedFormat}
         *  The parsed number format of the active cell.
         */
        this.getActiveParsedFormat = function () {
            return activeParsedFormat;
        };

        /**
         * Returns the URL of a hyperlink at the current active cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink attached to the active cell; or null, if the
         *  active cell does not contain a hyperlink.
         */
        this.getCellURL = function () {
            return hyperlinkCollection.getCellURL(activeAddress);
        };

        /**
         * Returns the URL of a hyperlink as returned by a formula (via the
         * function HYPERLINK) of the current active cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink as returned by a cell formula (via the
         *  function HYPERLINK) of the current active cell.
         */
        this.getFormulaURL = function () {
            return cellCollection.getFormulaURL(activeAddress);
        };

        /**
         * Returns the effective URL of a hyperlink at the current active cell.
         * If the cell contains a regular hyperlink, and a cell formula with a
         * HYPERLINK function, the regular hyperlink will be preferred.
         *
         * @returns {String|Null}
         *  The effective URL of a hyperlink at the active cell.
         */
        this.getEffectiveURL = function () {
            return cellCollection.getEffectiveURL(activeAddress);
        };

        /**
         * Changes the contents and/or formatting of the active cell, and
         * updates the view according to the new cell value and formatting.
         *
         * @param {Object} contents
         *  The new properties to be applied at the cell. See description of
         *  the method CellCollection.generateCellContentOperations() for more
         *  details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCellContents = function (contents) {

            // store character attributes while in-place cell edit mode is active
            if (this.isCellEditMode()) {
                if (_.isObject(contents.a)) { this.setEditAttributeSet(contents.a); }
                return $.when();
            }

            // whether the operation will modify the values of cells
            var changeValues = (contents.u === true) || ('v' in contents) || ('f' in contents);

            // apply operations (do nothing if active cell is locked)
            var lockMatrixes = changeValues ? 'partial' : 'none';
            var promise = this.ensureUnlockedActiveCell({ lockTables: 'header', lockMatrixes: lockMatrixes }).then(function () {
                return setCellContents(activeAddress, contents, { updateRows: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Sets the contents of the active cell to the passed edit text (tries
         * to convert to an appropriate data type, or to calculate a formula
         * string), and updates the view according to the new cell value and
         * formatting.
         *
         * @param {String} parseText
         *  The raw text to be parsed. May result in a number, a boolean value,
         *  an error code, or a formula expression whose result will be
         *  calculated. The empty string will remove the current cell value
         *  (will create a blank cell).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.parseCellContents = function (parseText) {

            // apply operations (do nothing if active cell is locked)
            var promise = this.ensureUnlockedActiveCell({ lockTables: 'header', lockMatrixes: 'partial' }).then(function () {
                return parseCellContents(activeAddress, parseText, { updateRows: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Generates the cell operations, and the undo operations, to change
         * the specified contents for a list of single cells in the active
         * sheet.
         *
         * @param {Array<Object>} contentsArray
         *  A list with the descriptors for all cells to be changed. Each array
         *  element MUST be an object with the following properties:
         *  - {Address} address
         *      The address of the cell to be modified.
         *  - {Object} contents
         *      The cell contents descriptor for the cell. See method
         *      CellCollection.generateCellContentOperations() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCellContentsArray = function (contentsArray) {

            // all cell addresses, as range array
            var addresses = AddressArray.map(contentsArray, function (entry) { return entry.address; });
            var ranges = RangeArray.mergeAddresses(addresses);

            // apply operations (do nothing if active cell is locked)
            var promise = this.ensureUnlockedRanges(ranges, { lockTables: 'header', lockMatrixes: 'partial' }).then(function () {

                // generate and apply the operations
                return activeSheetModel.createAndApplyOperations(function (generator) {

                    // try to fill the cell
                    var promise2 = cellCollection.generateContentsArrayOperations(generator, contentsArray);

                    // set optimal row height afterwards (for changed rows only)
                    promise2 = promise2.then(function (changedRanges) {
                        return updateOptimalRowHeight(generator, changedRanges);
                    });

                    return promise2;
                }, { storeSelection: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Changes the contents and/or formatting of all cells in the current
         * selection, and updates the view according to the new cell values and
         * formatting.
         *
         * @param {Object} contents
         *  The new properties to be applied at all selected cells. See method
         *  CellCollection.generateCellContentOperations() for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.fillRangeContents = function (contents) {

            // store character attributes while in-place cell edit mode is active
            if (this.isCellEditMode()) {
                if (_.isObject(contents.a)) { this.setEditAttributeSet(contents.a); }
                return $.when();
            }

            // whether the operation will modify the values of cells
            var changeValues = (contents.u === true) || ('v' in contents) || ('f' in contents);

            // do nothing if current selection contains locked cells (do not allow to change table header texts)
            var promise = this.ensureUnlockedSelection(changeValues ? { lockTables: 'header', lockMatrixes: 'partial' } : null).then(function () {
                return fillRangeContents(self.getSelectedRanges(), contents, { updateRows: true, refAddress: activeAddress });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Removes the values, formulas, and formatting of all cells in the
         * current selection (deletes the cell definitions from the document
         * model), and updates the view.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.clearRangeContents = function () {

            // do nothing if current selection contains locked cells (do not allow to delete table header texts)
            var promise = this.ensureUnlockedSelection({ lockTables: 'header', lockMatrixes: 'partial' }).then(function () {
                return fillRangeContents(self.getSelectedRanges(), { u: true }, { updateRows: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Inserts a matrix formula into the current selection.
         *
         * @param {Object} contents
         *  The new properties to be applied at all selected cells. See method
         *  CellCollection.generateCellContentOperations() for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setMatrixContents = function (contents) {

            // if matrix edit mode has been left without a formula expression, behave like normal fill mode
            if (!contents.f || !contents.result) { return this.fillRangeContents(contents); }

            // invalid formula result (e.g. circular reference, oversized matrix) results in #N/A
            var matrix = (contents.v instanceof ErrorCode) ? new Matrix([[contents.v]]) : contents.v;
            // if the active cell covers an existing matrix formula, use its bounding range regardless of selection
            var matrixRange = cellCollection.getMatrixRange(activeAddress);
            // whether to update an existing matrix (prevent auto-expansion of 1x1 matrixes)
            var updateMatrix = matrixRange !== null;
            // fall-back to the active range of the selection (ignore multi-selection)
            if (!matrixRange) { matrixRange = this.getActiveRange(); }

            // automatically expand single-cell selection to size of result matrix
            if (!updateMatrix && matrixRange.single()) {
                matrixRange.end[0] = matrixRange.start[0] + matrix.cols() - 1;
                matrixRange.end[1] = matrixRange.start[1] + matrix.rows() - 1;
            }

            // check the resulting matrix range:
            // - do not allow to insert oversized matrixes
            // - do not insert matrix formulas over merged ranges
            var matrixDim = Dimension.createFromRange(matrixRange);
            var errorCode = null;
            if (!Matrix.isValidDim(matrixDim)) {
                errorCode = 'formula:matrix:size';
            } else if (mergeCollection.rangesOverlapAnyMergedRange(matrixRange)) {
                errorCode = 'formula:matrix:merged';
            }

            // do not insert matrix formulas into table ranges, do not allow to change another existing matrix partially
            if (!errorCode) {
                var lockMatrixes = updateMatrix ? 'none' : 'partial';
                errorCode = this.ensureUnlockedRanges(matrixRange, { lockTables: 'full', lockMatrixes: lockMatrixes, sync: true });
            }

            // generate and apply the 'changeCells' operation for the matrix formula
            var promise = createEnsureResult(errorCode).then(function () {

                // build the cell contents matrix
                var cellMatrix = _.times(matrixDim.rows, function (row) {
                    return {
                        c: _.times(matrixDim.cols, function (col) {
                            var cellData = { v: Scalar.getCellValue(matrix.get(row, col)), f: null };
                            // add the formula and bounding range of the matrix to the anchor cell
                            if ((row === 0) && (col === 0)) {
                                cellData.f = contents.f;
                                cellData.mr = matrixRange;
                            }
                            return cellData;
                        })
                    };
                });

                return setCellContents(matrixRange.start, cellMatrix, { updateRows: true, selectRange: matrixRange });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Changes a single attribute of type 'cell' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the cell attribute.
         *
         * @param {Any} value
         *  The new value of the cell attribute. If set to the value null, all
         *  attributes set explicitly will be removed from the cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCellAttribute = function (name, value) {
            return this.fillRangeContents({ a: { cell: Utils.makeSimpleObject(name, value) } });
        };

        /**
         * Changes a single attribute of type 'character' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the character attribute.
         *
         * @param {Any} value
         *  The new value of the character attribute. If set to the value null,
         *  the current explicit attribute will be removed from the cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCharacterAttribute = function (name, value) {
            return this.fillRangeContents({ a: { character: Utils.makeSimpleObject(name, value) } });
        };

        /**
         * Returns the number format category of the active cell.
         *
         * @returns {String}
         *  The current number format category.
         */
        this.getNumberFormatCategory = function () {
            return activeParsedFormat.category;
        };

        /**
         * Sets selected number format category for the current active cell in
         * the current selection, and implicitly set the first format code of
         * the predefined template of this category to the cell.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setNumberFormatCategory = function (category) {
            var formatCode = numberFormatter.getDefaultCode(category);
            return (formatCode !== null) ? this.fillRangeContents({ format: formatCode }) : $.when();
        };

        /**
         * Returns the number format code of the active cell.
         *
         * @returns {String}
         *  The format code of the active cell, as string.
         */
        this.getNumberFormat = function () {
            return activeParsedFormat.formatCode;
        };

        /**
         * Returns the format code of the active cell.
         *
         * @param {Number|String} format
         *  The identifier of a number format as integer, or a format code as
         *  string, to be set to the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setNumberFormat = function (format) {

            // do nothing if custom fall-back button is clicked
            if (format === Labels.CUSTOM_FORMAT_VALUE) { return $.when(); }

            return this.fillRangeContents({ format: format });
        };

        /**
         * Returns the border mode of the current selection.
         *
         * @returns {Object}
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         */
        this.getBorderMode = function () {
            return MixedBorder.getBorderMode(selectedMixedBorders);
        };

        /**
         * Changes the visibility of the borders in all ranges of the current
         * selection.
         *
         * @param {Object} borderMode
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setBorderMode = function (borderMode) {

            // wait for the background loop that collects the mixed borders of the selection
            var promise = mixedBordersDef.then(function () {

                // convert border mode to real border attributes
                var borderAttributes = MixedBorder.getBorderAttributes(borderMode, selectedMixedBorders, DEFAULT_SINGLE_BORDER);

                // do nothing if current selection contains locked cells
                var promise2 = self.ensureUnlockedSelection().then(function () {
                    return activeSheetModel.createAndApplyOperations(function (generator) {
                        return cellCollection.generateBorderOperations(generator, self.getSelectedRanges(), borderAttributes);
                    }, { storeSelection: true });
                });

                // bug 34021: immediately update the cached mixed borders of the selection for fast GUI feedback
                promise2.done(function () {
                    _.extend(selectedMixedBorders, borderAttributes);
                });

                return promise2;
            });

            // show warning alerts if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Returns the mixed border attributes of the current selection.
         *
         * @returns {Object}
         *  An attribute map containing the mixed borders of all available
         *  border attributes ('borderTop', 'borderLeft', etc.), as well as the
         *  pseudo attributes 'borderInsideHor' and 'borderInsideVert' for
         *  inner borders in the current selection. Each border attribute value
         *  contains the properties 'style', 'width', and 'color'. If the
         *  border property is equal in all affected cells, it will be set as
         *  property value in the border object, otherwise the property will be
         *  set to the value null.
         */
        this.getBorderAttributes = function () {
            return _.copy(selectedMixedBorders, true);
        };

        /**
         * Changes the specified border properties (line style, line color, or
         * line width) of all visible cell borders in the current selection.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the current
         *  selection. Omitted border properties will not be changed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeVisibleBorders = function (border) {

            // do nothing if current selection contains locked cells
            var promise = this.ensureUnlockedSelection().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    return cellCollection.generateVisibleBorderOperations(generator, self.getSelectedRanges(), border);
                }, { storeSelection: true });
            });

            // bug 34021: immediately update the cached mixed borders of the selection for fast GUI feedback
            promise.done(function () {
                _.each(selectedMixedBorders, function (mixedBorder) {
                    if (MixedBorder.isVisibleBorder(mixedBorder)) {
                        _.extend(mixedBorder, border);
                    }
                });
            });

            // show warning messages
            return this.yellOnFailure(promise);
        };

        /**
         * Returns whether the format painter is currently active.
         *
         * @returns {Boolean}
         *  Whether the format painter is currently active.
         */
        this.isFormatPainterActive = function () {
            return this.isCustomSelectionMode('painter');
        };

        /**
         * Activates or deactivates the format painter.
         *
         * @param {Boolean} state
         *  Whether to activate or deactivate the format painter.
         *
         * @returns {Boolean}
         *  Whether the format painter has been activated.
         */
        this.activateFormatPainter = function (state) {

            // deactivate format painter: cancel custom selection mode
            if (!state) {
                this.cancelCustomSelectionMode();
                return false;
            }

            // fill the auto-style of the active cell into the range that will be selected
            var contents = { s: activeStyleId };
            // the promise representing custom selection mode
            var promise = null;

            // do not lock editable cells by copying the 'unlocked' attribute (in locked sheets only, as in Excel)
            if (this.isSheetLocked()) {
                contents.a = { cell: { unlocked: true } };
            }

            // start custom selection mode (wait for range selection)
            promise = this.enterCustomSelectionMode('painter', {
                selection: this.getActiveCellAsSelection(),
                statusLabel: Labels.FORMAT_PAINTER_LABEL
            });

            // select the target range (bug 35295: check that the cells are not locked)
            promise = promise.then(function (selection) {
                self.setCellSelection(selection);
                return self.ensureUnlockedSelection();
            });

            // copy the formatting attributes to the target range
            promise.done(function () {
                self.executeControllerItem('cell/fill', contents);
            });

            // return activated state (not the promise, which would block the GUI)
            return true;
        };

        /**
         * Fills or clears a cell range according to the position and contents
         * of the cell range currently selected (the 'auto-fill' feature).
         *
         * @param {String} border
         *  Which border of the selected range will be modified by the
         *  operation. Allowed values are 'left', 'right', 'top', or 'bottom'.
         *
         * @param {Number} count
         *  The number of columns/rows to extend the selected range with
         *  (positive values), or to shrink the selected range (negative
         *  values).
         *
         * @param {Boolean} invertType
         *  Swap the default expansion type according to the source data (copy
         *  versus auto-increment).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the auto-fill target range is
         *  editable, and the auto-fill operation has been applied. Otherwise,
         *  the promise will be rejected.
         */
        this.autoFill = function (border, count, options) {

            var // all cell ranges selected in the sheet
                ranges = this.getSelectedRanges(),
                // whether to expand/shrink the leading or trailing border
                leading = /^(left|top)$/.test(border),
                // whether to expand/delete columns or rows
                columns = /^(left|right)$/.test(border),
                // the target range for auto-fill
                targetRange = ranges.first().clone(),
                // the promise of the entire auto-fill operations
                promise = null;

            // for safety: do nothing if called in a multi-selection
            // (no error message, should not happen as the GUI does not offer auto-fill)
            if (ranges.length !== 1) {
                return $.Deferred().reject();
            }

            // fill cells outside the selected range
            if (count > 0) {

                // build the target range (adjacent to the selected range)
                if (leading) {
                    targetRange.setStart(ranges.first().getStart(columns) - count, columns);
                    targetRange.setEnd(ranges.first().getStart(columns) - 1, columns);
                } else {
                    targetRange.setStart(ranges.first().getEnd(columns) + 1, columns);
                    targetRange.setEnd(ranges.first().getEnd(columns) + count, columns);
                }

                // only the target range must be editable (not the source cells)
                promise = this.ensureUnlockedRanges(targetRange, { lockTables: 'header', lockMatrixes: 'partial' });

                // apply auto-fill, and update optimal row heights
                promise = promise.then(function () {
                    return activeSheetModel.createAndApplyOperations(function (generator) {

                        var promise2 = cellCollection.generateAutoFillOperations(generator, ranges.first(), border, count, options);

                        promise2 = promise2.then(function () {
                            return updateOptimalRowHeight(generator, targetRange);
                        });

                        // expand the selected range
                        return promise2.always(function () {
                            self.changeActiveRange(ranges.first().boundary(targetRange));
                        });
                    }, { storeSelection: true });
                });

            // delete cells inside the selected range
            } else if (count < 0) {

                // build the target range (part of the selected range, may be the entire selected range)
                if (leading) {
                    targetRange.setEnd(targetRange.getStart(columns) - count - 1, columns);
                } else {
                    targetRange.setStart(targetRange.getEnd(columns) + count + 1, columns);
                }

                // the target range must be editable (not the entire selected range)
                promise = this.ensureUnlockedRanges(targetRange, { lockTables: 'header', lockMatrixes: 'partial' });

                // remove all values and formulas from the cell range (but not the formatting),
                // and update optimal row heights
                promise = promise.then(function () {
                    return fillRangeContents(targetRange, { v: null, f: null }, { updateRows: true });
                });

                // update selection
                promise.always(function () {
                    // do not change selection after deleting the entire range
                    if (ranges.first().equals(targetRange)) { return; }
                    // select the unmodified part of the original range
                    targetRange = ranges.first().clone();
                    if (leading) {
                        targetRange.start.move(-count, columns);
                    } else {
                        targetRange.end.move(count, columns);
                    }
                    self.changeActiveRange(targetRange);
                });

            // do nothing if no cells will be filled or cleared
            } else {
                promise = $.when();
            }

            return this.yellOnFailure(promise);
        };

        /**
         * Inserts one or more formulas calculating subtotal results into the,
         * or next to the current selection.
         *
         * @param {String} funcKey
         *  The resource key of the subtotal function to be inserted into the
         *  formulas. MUST be a function that takes an argument of reference
         *  type.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the formulas have been
         *  created successfully.
         */
        this.insertAutoFormula = function (funcKey) {

            // start cell edit mode, if a single cell is selected
            if (this.isSingleCellSelection()) {
                var sourceRange = cellCollection.findAutoFormulaRange(activeAddress);
                var formula = '=' + formulaGrammar.generateAutoFormula(docModel, funcKey, sourceRange, activeAddress);
                return self.enterCellEditMode({ text: formula, pos: -1 });
            }

            // create and apply the operations
            return activeSheetModel.createAndApplyOperations(function (generator) {
                // generate the 'changeCells' operations, and the undo operations
                var range = self.getActiveRange();
                return cellCollection.generateAutoFormulaOperations(generator, funcKey, range).done(function (targetRange) {
                    if (targetRange) { self.selectRange(targetRange); }
                });
            }, { storeSelection: true });
        };

        /**
         * Shows the 'Insert Function' dialog, and starts the cell in-place
         * edit mode with a formula containing the function selected in the
         * dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after a function has been selected
         *  in the dialog, and the cell edit mode has been started; or that
         *  will be rejected after canceling the dialog.
         */
        this.showInsertFunctionDialog = function () {

            // the active cell must be editable
            var promise = this.ensureUnlockedActiveCell({ lockTables: 'header' }).then(function () {

                // pick the formula string if available, otherwise the display text
                var value = activeTokenDesc ? activeTokenDesc.formula : activeCellDisplay;

                // try to extract a function name
                if (_.isString(value)) {
                    var result = (/^=?\s*([^(]+)/).exec(value);
                    value = result ? result[1] : null;
                }

                // show the dialog, wait for the result
                return new Dialogs.FunctionDialog(self, value).show();
            });

            // start cell edit mode, insert the selected function as formula into the active cell
            promise = promise.then(function (funcKey) {
                var formula = '=' + formulaGrammar.getFunctionName(funcKey) + '()';
                return self.enterCellEditMode({ text: formula, pos: -1 });
            });

            // show warning alerts if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Merges or unmerges all cells of the current selection.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the selection.
         *  - 'toggle': Will be mapped to 'merge', if the selection does not
         *      cover any merged ranges, otherwise will be mapped to 'unmerge'.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the ranges have been merged
         *  or unmerged successfully, or that will be rejected on any error.
         */
        this.mergeRanges = function (type) {

            // the selected ranges to be merged/unmerged
            var ranges = this.getSelectedRanges();

            // toggle merge state if specified
            if (type === 'toggle') {
                type = mergeCollection.rangesOverlapAnyMergedRange(ranges) ? 'unmerge' : 'merge';
            }

            // do nothing, if current selection contains locked cells
            // bug 39869: do not allow to merge any cells in a table range (except auto-filter)
            // DOCS-196: do not allow to merge matrix formulas
            var unmerge = type === 'unmerge';
            var promise = this.ensureUnlockedSelection(unmerge ? null : { lockTables: 'full', lockMatrixes: 'full' }).then(function () {

                // create and apply the operations
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    // generate the 'mergeCells' operations, and the undo operations
                    return mergeCollection.generateMergeCellsOperations(generator, ranges, type);
                }, { storeSelection: true });
            });

            // show warning alerts if necessary
            return this.yellOnFailure(promise);
        };

        // hyperlink operations -----------------------------------------------

        /**
         * Inserts new hyperlink ranges covering the selected cells, or the
         * active cell.
         *
         * @param {String} url
         *  The URL of the hyperlink to be inserted.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.activeCell=false]
         *      If set to true, the hyperlink will be inserted into the active
         *      cell of the selection only. By default, the entire selection
         *      will be filled.
         *  @param {Boolean} [options.createStyles=false]
         *      If set to true, the cells covered by the new hyperlink will be
         *      set to underline style, and to the current hyperlink text
         *      color.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the hyperlink ranges have
         *  been inserted successfully, ot that will be rejected on any error.
         */
        this.insertHyperlink = function (url, options) {

            // whether to create the hyperlink for the active cell only, instead for the entire selection
            var activeCell = Utils.getBooleanOption(options, 'activeCell', false);
            // whether to create the character styles (underline, text color) for the hyperlink
            var createStyles = Utils.getBooleanOption(options, 'createStyles', false);
            // the character attributes for the cells covered by the hyperlink
            var attributes = createStyles ? { character: { underline: true, color: docModel.createJSONColor(Color.HYPERLINK, 'text') } } : null;
            // the cell contents object
            var contents = { a: attributes, url: url };

            // fill the active cell, or the entire cell selection
            return activeCell ? this.setCellContents(contents) : this.fillRangeContents(contents);
        };

        /**
         * Deletes the hyperlink ranges covering the selected cells, or the
         * active cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.activeCell=false]
         *      If set to true, the hyperlinks will be removed from the active
         *      cell of the selection only. By default, the entire selection
         *      will be cleared.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the hyperlink ranges have
         *  been deleted successfully, ot that will be rejected on any error.
         */
        this.deleteHyperlink = function (options) {

            // whether to create the hyperlink for the active cell only, instead for the entire selection
            var activeCell = Utils.getBooleanOption(options, 'activeCell', false);
            // the cell contents object (remove hyperlinks, and the character formatting)
            var contents = { a: { character: { underline: null, color: null } }, url: '' };

            // fill the active cell, or the entire cell selection
            return activeCell ? this.setCellContents(contents) : this.fillRangeContents(contents);
        };

        /**
         * Opens the 'Edit Hyperlink' dialog, and applies the settings made in
         * the dialog to the current cell selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the dialog has been closed,
         *  and the hyperlink settings have been applied successfully; or that
         *  will be rejected after canceling the dialog, or on any error.
         */
        this.editHyperlink = function () {

            // try to leave cell edit mode
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // the literal string value of the cell, used as initial display string
            var display = (_.isString(activeCellValue) && !activeTokenDesc) ? activeCellValue : null;
            // the current hyperlink of the active cell
            var currUrl = this.getCellURL();
            // whether the cell contained a hyperlink before
            var hasHyperlink = typeof currUrl === 'string';

            // active cell must be editable (perform additional check for tables)
            var promise = this.ensureUnlockedActiveCell({ lockTables: 'header' }).then(function () {

                // if there is no real hyperlink, try the literal cell string
                if (!currUrl && display) {
                    currUrl = HyperlinkUtils.checkForHyperlink(display);
                }

                // do not duplicate equal URL and display string in the dialog
                if (currUrl === display) { display = null; }

                // show the 'Edit Hyperlink' dialog
                return new Dialogs.HyperlinkDialog(self, currUrl, display).show();
            });

            // apply hyperlink settings after closing the dialog
            promise = promise.then(function (result) {

                // missing result.url indicates to delete a hyperlink ('Remove' button)
                if (!result.url) {
                    return self.deleteHyperlink();
                }

                // collect multiple complex sheet operations in a single undo action
                return undoManager.enterUndoGroup(function () {

                    // add character formatting for a new hyperlink, and generate the operation
                    var promise2 = self.insertHyperlink(result.url, { createStyles: !hasHyperlink });

                    // create a text cell, if user has entered a display string in the dialog;
                    // or if the cell was blank, and a valid URL has been entered (do not change
                    // existing cells, e.g. numbers, formulas, etc.)
                    if (result.text) {
                        promise2 = promise2.then(function () {
                            return parseCellContents(activeAddress, result.text, { updateRows: true, skipHyperlink: true });
                        });
                    } else if (activeCellValue === null) {
                        promise2 = promise2.then(function () {
                            return setCellContents(activeAddress, { v: result.url }, { updateRows: true });
                        });
                    }

                    return promise2;
                });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        // defined name operations --------------------------------------------

        /**
         * Inserts a new defined name into the document.
         *
         * @param {String} label
         *  The label of the new defined name.
         *
         * @param {String} formula
         *  The formula expression to be bound to the new defined name.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  created successfully, or that will be rejected on any error.
         */
        this.insertName = function (label, formula) {

            // the descriptor for the formula expression
            var formulaDesc = { grammarId: 'ui', formula: formula, refSheet: activeSheet, refAddress: activeAddress };

            // create and apply the operations to create the defined name
            // TODO: support for sheet-local names
            var promise = docModel.createAndApplyOperations(function (generator) {
                return docModel.getNameCollection().generateInsertNameOperations(generator, label, formulaDesc);
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Shows a dialog that allows to create a new defined name for the
         * document. The dialog will be kept open until a valid label and
         * formula expression has been entered for the defined name.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved, after the
         *  defined name has been created successfully; or will be rejected, if
         *  the dialog has been canceled, or if the document has switched to
         *  read-only mode.
         */
        this.showInsertNameDialog = function () {

            // leave cell edit mode before creating a name
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // create a formula from the current selection (bug 40477: use start address of single merged range)
            var tokenArray = new TokenArray(docModel, 'name');
            if (this.isSingleCellSelection()) {
                tokenArray.appendRange(new Range(activeAddress), { sheet: activeSheet, abs: true });
            } else {
                tokenArray.appendRangeList(this.getSelectedRanges(), { sheet: activeSheet, abs: true });
            }
            var formula = tokenArray.getFormula('ui');

            // create the dialog with empty input fields
            var dialog = new Dialogs.DefinedNameDialog(this, '', formula, this.insertName.bind(this), {
                title: gt('Insert name'),
                okLabel: gt('Insert')
            });

            // show the dialog (inserting the defined name will be done by the callback passed to the dialog)
            return dialog.show();
        };

        /**
         * Changes the label or formula definition of an existing defined name.
         *
         * @param {String} label
         *  The label of the defined name to be changed.
         *
         * @param {String|Null} newLabel
         *  The new label for the defined name. If set to null, the label of
         *  the defined name will not be changed.
         *
         * @param {String|Null} newFormula
         *  The new formula expression to be bound to the defined name. If set
         *  to null, the formula expression of the defined name will not be
         *  changed (an invalid formula expression in the defined name will be
         *  retained).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  changed successfully, or that will be rejected on any error.
         */
        this.changeName = function (label, newLabel, newFormula) {

            // the descriptor for the new formula expression
            var formulaDesc = _.isString(newFormula) ? { grammarId: 'ui', formula: newFormula, refSheet: activeSheet, refAddress: activeAddress } : null;

            // create and apply the operations to change the defined name
            // TODO: support for sheet-local names
            var promise = docModel.createAndApplyOperations(function (generator) {
                return docModel.getNameCollection().generateChangeNameOperations(generator, label, newLabel, formulaDesc);
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Shows a dialog that allows to change an existing defined name in the
         * document. The dialog will be kept open until a valid label and
         * formula expression has been entered.
         *
         * @param {String} label
         *  The label of the defined name to be changed.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved, after the
         *  defined name has been changed successfully; or will be rejected, if
         *  the dialog has been canceled, or if the document has switched to
         *  read-only mode.
         */
        this.showChangeNameDialog = function (label) {

            // leave cell edit mode before changing a defined name
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // get the name model from the document collection (TODO: shet-local names)
            var nameModel = docModel.getNameCollection().getName(label);
            if (!nameModel) { return $.Deferred().reject(); }

            // resolve the formula expression relative to the active cell
            var formula = nameModel.getFormula('ui', activeAddress);

            // the callback function invoked from the dialog after pressing the OK button
            function actionHandler(newLabel, newFormula) {
                return self.changeName(label, (label !== newLabel) ? newLabel : null, (formula !== newFormula) ? newFormula : null);
            }

            // create the dialog with empty input fields
            var dialog = new Dialogs.DefinedNameDialog(this, label, formula, actionHandler, {
                title: gt('Change name'),
                okLabel: gt('Change')
            });

            // show the dialog (changing the defined name will be done by the callback passed to the dialog)
            return dialog.show();
        };

        /**
         * Deletes a defined name from the document.
         *
         * @param {String} label
         *  The label of the defined name to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  deleted successfully, or that will be rejected on any error.
         */
        this.deleteName = function (label) {

            // create and apply the operations to delete the defined name
            // TODO: support for sheet-local names
            var promise = docModel.createAndApplyOperations(function (generator) {
                return docModel.getNameCollection().generateDeleteNameOperations(generator, label);
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        // table/filter operations --------------------------------------------

        /**
         * Returns the model of the table range located at the active cell of
         * the current selection. If the active cell is not located inside a
         * table range, tries to return the model of the anonymous table range
         * representing the auto-filter of the active sheet.
         *
         * @returns {TableModel|Null}
         *  The model of the active table range; or null, if no table range has
         *  been found.
         */
        this.getActiveTable = function () {
            return tableCollection.findTable(activeAddress) || tableCollection.getAutoFilter();
        };

        /**
         * Returns whether the active cell is located in a real table range
         * (any table but the auto-filter).
         *
         * @returns {Boolean}
         *  Whether the active cell is located in a real table range.
         */
        this.isRealTableSelected = function () {
            var tableModel = this.getActiveTable();
            return _.isObject(tableModel) && !tableModel.isAutoFilter();
        };

        /**
         * Returns whether the active cell of the current selection is located
         * in a table range with activated filter/sorting. If the active cell
         * is not located in a table range, returns whether the active sheet
         * contains an activated auto-filter (regardless of the selection).
         *
         * @returns {Boolean}
         *  Whether a table range with activated filter is selected.
         */
        this.hasTableFilter = function () {
            var tableModel = this.getActiveTable();
            return _.isObject(tableModel) && tableModel.isFilterEnabled();
        };

        /**
         * Enables or disables the filter in the selected table range of the
         * active sheet. If the active cell of the current selection is not
         * located inside a table range, the auto-filter of the active sheet
         * will be toggled.
         *
         * @param {Boolean} state
         *  Whether to activate or deactivate the table filter.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the table filter has been
         *  toggled successfully, or that will be rejected on any error.
         */
        this.toggleTableFilter = function (state) {

            // no filter manipulation in locked sheets
            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {

                    // get the active (selected) table range
                    var tableModel = self.getActiveTable();

                    // toggle filter of a real table range (not auto-filter)
                    if (tableModel && !tableModel.isAutoFilter()) {
                        return tableModel.generateChangeTableOperations(generator, { table: { filtered: state } });
                    }

                    // concatenate optional asynchronous methods
                    var promise2 = Utils.invokeChainedAsync(

                        // remove existing auto-filter (regardless of passed state)
                        tableModel ? function () {
                            return tableModel.generateDeleteOperations(generator);
                        } : null,

                        // create a new table for the auto-filter
                        state ? function () {

                            // the current selection ranges
                            var ranges = self.getSelectedRanges();
                            // the expanded first selected range
                            var range = null;
                            // fixed attributes for the auto-filter
                            var attributes = { table: { headerRow: true, filtered: true } };

                            // auto-filter cannot be created on a multi selection
                            if (ranges.length !== 1) {
                                return SheetUtils.makeRejected('autofilter:multiselect');
                            }

                            // expand single cell to content range
                            range = activeSheetModel.getContentRangeForCell(ranges.first());

                            // bug 36606: restrict range to used area of the sheet (range may become null)
                            range = range.intersect(cellCollection.getUsedRange());

                            // bug 36227: selection must not be entirely blank
                            if (!range || cellCollection.areRangesBlank(range)) {
                                return SheetUtils.makeRejected('autofilter:blank');
                            }

                            // auto-filter cannot be created on a multi selection
                            return tableCollection.generateInsertTableOperations(generator, '', range, attributes);
                        } : null
                    );

                    // translate error codes for auto-filter
                    return promise2.then(null, function (result) {
                        result.cause = result.cause.replace(/^table:/, 'autofilter:');
                        return result;
                    });
                }, { storeSelection: true });
            });

            // show warning alert if necessary
            return self.yellOnFailure(promise);
        };

        /**
         * Returns whether the active table range can be refreshed (the active
         * sheet must contain an active table, and that table must contain any
         * active filter rules or sort options).
         *
         * @returns {Boolean}
         *  Whether the active table range can be refreshed.
         */
        this.canRefreshTable = function () {
            var tableModel = this.getActiveTable();
            return _.isObject(tableModel) && tableModel.isRefreshable();
        };

        /**
         * Refreshes the active table ranges (reapplies all filter rules and
         * sort options to the current cell contents in the table range).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the table has been refreshed
         *  successfully, or that will be rejected on any error.
         */
        this.refreshTable = function () {

            // refresh the table (no filter manipulation in locked sheets)
            var promise = this.ensureUnlockedSheet().then(function () {

                // get the active (selected) table range
                var tableModel = self.getActiveTable();
                if (!tableModel) { return $.Deferred().reject(); }

                // generate and apply the refresh operations
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    return tableModel.generateRefreshOperations(generator);
                }, { storeSelection: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Changes the formatting attributes of the specified column in a table
         * range of the active sheet.
         *
         * @param {String} tableName
         *  The name of the table to be modified. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto-filter of the sheet.
         *
         * @param {Number} tableCol
         *  The zero-based index of the table column to be modified, relative
         *  to the cell range covered by the table.
         *
         * @param {Object} attributes
         *  The attribute set with filter and sorting attributes to be set for
         *  the table column.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the filter has been applied
         *  successfully, or that will be rejected on any error.
         */
        this.changeTableColumn = function (tableName, tableCol, attributes) {

            // ensure unlocked sheet (no filter manipulation in locked sheets)
            var promise = this.ensureUnlockedSheet().then(function () {

                // get the specified table range
                var tableModel = tableCollection.getTable(tableName);
                if (!tableModel) { return $.Deferred().reject(); }

                return activeSheetModel.createAndApplyOperations(function (generator) {
                    return tableModel.generateChangeColumnOperations(generator, tableCol, attributes);
                }, { storeSelection: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        // drawing operations -------------------------------------------------

        /**
         * Inserts new drawing sobject into the active sheet, and selects them
         * afterwards.
         *
         * @param {Array<Object>} drawingData
         *  The type and formatting attributes of the drawing objects to be
         *  inserted. Each array element MUST be an object with the following
         *  properties:
         *  @param {String} element.type
         *      The type identifier of the drawing object.
         *  @param {Object} element.attrs
         *      Initial attributes for the drawing object, especially the
         *      anchor attributes specifying the physical position of the
         *      drawing object in the sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked everytime after an
         *  'insertDrawing' operation has been generated for a drawing object.
         *  Allows to create additional operations for the drawing objects,
         *  before all these operations will be applied at once. Receives the
         *  following parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation of the current drawing).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The effective document position of the current drawing object
         *      in the sheet, as inserted into the 'insertDrawing' operation.
         *  (4) {Object} data
         *      The data element from the array parameter 'drawingData' that is
         *      currently processed.
         *  The callback function may return a promise to defer processing of
         *  subsequent drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  created successfully, or that will be rejected on any error.
         */
        this.insertDrawings = function (drawingData, callback) {

            // check that the active sheet is not locked
            var promise = this.ensureUnlockedSheet({ errorCode: 'drawing:delete:locked' }).then(function () {

                // generate and apply the drawing operations
                return activeSheetModel.createAndApplyOperations(function (generator) {

                    // the position of the first new drawing object
                    var firstPosition = [drawingCollection.getModelCount()];
                    // the positions of all drawing objects to be selected later
                    var allPositions = [];

                    var promise = drawingCollection.generateInsertDrawingOperations(generator, firstPosition, drawingData, function (generator, sheet, position, data) {
                        allPositions.push(position);
                        if (_.isFunction(callback)) {
                            return callback.call(self, generator, sheet, position, data);
                        }
                    });

                    // select the drawing objects on success
                    return promise.done(function () { self.setDrawingSelection(allPositions); });

                }, { storeSelection: true });
            });

            // show warning messages if needed
            return this.yellOnFailure(promise);
        };

        /**
         * Inserts a new drawing object into the active sheet, and selects it
         * afterwards.
         *
         * @param {String} type
         *  The type identifier of the drawing object.
         *
         * @param {Object} attributes
         *  Initial attributes for the drawing object, especially the anchor
         *  attributes specifying the position of the drawing object in the
         *  active sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked after the 'insertDrawing'
         *  operation has been generated for the drawing object. Allows to
         *  create additional operations for the drawing object, before all
         *  these operations will be applied at once. Receives the following
         *  parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The document position of the drawing object in the sheet, as
         *      inserted into the 'insertDrawing' operation.
         *  The callback function may return a promise to be able to run
         *  asynchronous code.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the drawing object has been
         *  created successfully, or that will be rejected on any error.
         */
        this.insertDrawing = function (type, attributes, callback) {
            return this.insertDrawings([{ type: type, attrs: attributes }], function (generator, sheet, position) {
                if (_.isFunction(callback)) {
                    return callback.call(self, generator, sheet, position);
                }
            });
        };

        /**
         * Inserts new image objects into the active sheet, and selects them
         * afterwards.
         *
         * @param {Object|Array<Object>} imageDescriptors
         *  The descriptor of the image object, containing the properties 'url'
         *  and 'name', or an array of such image descriptors to be able to
         *  insert multiple images at once.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the image object has been
         *  created and inserted into the active sheet, or that will be
         *  rejected on any error.
         */
        this.insertImages = function (imageDescriptors) {

            // create the image DOM nodes for all images
            var promises = _.getArray(imageDescriptors).map(function (imageDesc) {

                // create the image DOM node, the promise waits for it to load the image data
                var promise = app.createImageNode(ImageUtils.getFileUrl(app, imageDesc.url));

                // fail with empty resolved promise to be able to use $.when() which would
                // otherwise abort immediately if ANY of the promises fails
                return promise.then(function (imgNode) {
                    return { node: imgNode[0], url: imageDesc.url, name: imageDesc.name };
                }, _.constant($.when()));
            });

            // wait for all images nodes, create and apply the 'insertDrawing' operations
            var promise = $.when.apply($, promises).then(function () {

                // $.when() passes the results as function arguments, filter out invalid images
                var results = _.filter(arguments, _.identity);
                if (results.length === 0) { return $.Deferred().reject(); }

                // all image nodes collected in a jQuery object
                var imgNodes = $();

                // the drawing attributes to be passed to insertDrawings()
                var drawingData = results.map(function (result) {

                    var // current width of the image, in 1/100 mm
                        width = Utils.convertLengthToHmm(result.node.width, 'px'),
                        // current height of the image, in 1/100 mm
                        height = Utils.convertLengthToHmm(result.node.height, 'px'),
                        // the generic drawing attributes (anchor position)
                        drawingAttrs = {
                            anchorType: 'oneCell',
                            startCol: activeAddress[0],
                            startRow: activeAddress[1],
                            name: result.name
                        },
                        // additional specific image attributes
                        imageAttrs = { imageUrl: result.url };

                    // restrict size to fixed limit
                    if (width > height) {
                        if (width > MAX_IMAGE_SIZE) {
                            height = Math.round(height * MAX_IMAGE_SIZE / width);
                            width = MAX_IMAGE_SIZE;
                        }
                    } else {
                        if (height > MAX_IMAGE_SIZE) {
                            width = Math.round(width * MAX_IMAGE_SIZE / height);
                            height = MAX_IMAGE_SIZE;
                        }
                    }
                    drawingAttrs.width = width;
                    drawingAttrs.height = height;

                    // collect all image nodes for clean-up
                    imgNodes = imgNodes.add(result.node);

                    // create the drawing descriptor for insertDrawings()
                    return { type: 'image', attrs: { drawing: drawingAttrs, image: imageAttrs } };
                });

                // destroy the temporary image nodes
                app.destroyImageNodes(imgNodes);

                // insert the image objects into the active sheet
                return self.insertDrawings(drawingData);
            });

            // on any error, return a promise rejected with an error code
            return promise.then(null, function () {
                return SheetUtils.makeRejected('image:insert');
            });
        };

        /**
        * Shows an insert image dialog from drive, local or URL dialog
        *
        * @param {String} dialogType
        *  The dialog type. Types: 'drive', 'local' or 'url'. Default is 'drive'.
        *
        *
        * @returns {jQuery.Promise}
        *  A promise that will be resolved or rejected after the dialog has
        *  been closed and the image has been inserted into the document.
        */
        this.showInsertImageDialog = function (dialogType) {

            // leave cell edit mode before showing the dialog
            if (!this.leaveCellEditMode()) { return $.Deferred().reject(); }

            // show the 'Insert Image' dialog
            var promise = ImageUtils.showInsertImageDialogByType(app, dialogType);

            // handle the dialog result
            promise = promise.then(function (imageDescriptor) {
                return self.insertImages(imageDescriptor);
            });

            // show warning messages if needed
            return this.yellOnFailure(promise);
        };

        /**
         * Deletes all drawings currently selected in the active sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  deleted successfully, or that will be rejected on any error.
         */
        this.deleteDrawings = function () {

            // rescue drawing selection
            var positions = self.getSelectedDrawings();

            // check that the active sheet is not locked
            var promise = this.ensureUnlockedSheet({ errorCode: 'drawing:delete:locked' }).then(function () {

                // bug 47948: check whether any unsupported (not undoable) drawing object is selected
                var unsupported = positions.some(function (position) {
                    var drawingModel = drawingCollection.findModel(position);
                    return drawingModel && drawingModel.isOrContainsType('undefined');
                });
                if (!unsupported) { return false; }

                // ask user whether to delete drawing objects that cannot be restored
                var dialog = new Dialogs.ModalQueryDialog(Labels.DELETE_CONTENTS_QUERY, { title: Labels.DELETE_CONTENTS_TITLE });

                // close dialog automatically after losing edit rights
                self.closeDialogOnReadOnlyMode(dialog);

                // show dialog, promise resolves with value true
                return dialog.show();
            });

            promise = promise.then(function (unsupported) {

                // back to cell selection
                self.removeDrawingSelection();

                // generate and apply the drawing operations
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    return drawingCollection.generateDeleteDrawingOperations(generator, positions);
                }, { storeSelection: true, undoMode: unsupported ? 'clear' : 'generate' });
            });

            // show warning messages if needed
            return this.yellOnFailure(promise);
        };

        /**
         * Changes the formatting attributes of all selected drawing objects,
         * or of the specified drawing objects.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set to be applied at the drawing objects.
         *
         * @param {Array<Array<Number>>} [positions]
         *  An array with positions of the drawing objects to be modified. If
         *  omitted, all selected drawing objects will be modified.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  modified successfully, or that will be rejected on any error.
         */
        this.changeDrawings = function (attributes, positions) {

            // check that the active sheet is not locked
            var promise = this.ensureUnlockedSheet({ errorCode: 'drawing:change:locked' }).then(function () {

                // fall-back to the selected drawing objects
                positions = positions || self.getSelectedDrawings();

                // generate and apply the drawing operations
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    return drawingCollection.generateChangeDrawingOperations(generator, positions, attributes);
                }, { storeSelection: true });
            });

            // show warning messages if needed
            return this.yellOnFailure(promise);
        };

        // debugging ----------------------------------------------------------

        /**
         * Returns whether formula cells will be rendered with special markers
         * (available in debug mode only).
         *
         * @returns {Boolean}
         *  Whether formula cells will be rendered with special markers.
         */
        this.isHighlightFormulasMode = function () {
            return docModel.getViewAttribute('highlightFormulas');
        };

        /**
         * Enables or disables rendering of special markers for formula cells
         * (available in debug mode only).
         *
         * @param {Boolean} enable
         *  Whether to enable or diable the formula highlighting mode.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setHighlightFormulasMode = function (enable) {
            docModel.setViewAttribute('highlightFormulas', enable);
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize selection settings, used until first real update
        activeAttributeSet = docModel.getCellStyles().getDefaultStyleAttributeSet();
        activeParsedFormat = numberFormatter.getParsedFormatForAttributes(activeAttributeSet.cell);

        // initialize sheet-dependent class members according to the active sheet
        this.on('change:activesheet', changeActiveSheetHandler);

        // refresh selection settings after changing the selection
        this.on('change:selection:prepare', function () { updateSelectionSettings(false); });

        // update selection settings after any operational changes in the document
        this.one('change:selection', function () {
            self.listenTo(docModel, 'operations:success', function () { updateSelectionSettings(true); });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = app = docModel = undoManager = numberFormatter = formulaGrammar = autoStyles = null;
            activeSheetModel = colCollection = rowCollection = mergeCollection = cellCollection = null;
            hyperlinkCollection = tableCollection = drawingCollection = null;
        });

    } // class ViewFuncMixin

    // exports ================================================================

    return ViewFuncMixin;

});
