/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/sheetnamedialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialogs',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Dialogs, gt) {

    'use strict';

    // class SheetNameDialog ==================================================

    /**
     * Shows a dialog that allows to enter a sheet name, and will invoke an
     * arbitrary callback function for the sheet name. The dialog will be
     * kept open until a valid sheet name has been entered.
     *
     * @constructor
     *
     * @extends ModalInputDialog
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view instance that has created this dialog.
     *
     * @param {String} initialName
     *  The initial sheet name to be shown when the dialog is opened.
     *
     * @param {Function} actionHandler
     *  The callback function invoked when the OK button of the dialog has
     *  been clicked (will be passed as 'validator' option to the dialog).
     *  Receives the current sheet name entered in the dialog, and MUST return
     *  a boolean value or a promise that states whether the operation using
     *  the sheet name has been finished successfully. In case of an error, the
     *  dialog will be kept open.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports most options of the base class
     *  ModalInputDialog, except 'value', 'maxLength', 'placeholder', and
     *  'validator'.
     */
    var SheetNameDialog = Dialogs.ModalInputDialog.extend({ constructor: function (docView, initialName, actionHandler, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        Dialogs.ModalInputDialog.call(this, Utils.extendOptions(initOptions, {
            value: initialName,
            maxLength: docView.getDocModel().getMaxSheetNameLength(),
            placeholder: gt('Enter sheet name'),
            validator: function () { return actionHandler.call(self, self.getText()); }
        }));

        // public methods -----------------------------------------------------

        /**
         * Shows the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved after the
         *  action handler callback function has been invoked successfully; or
         *  will be rejected, if the dialog has been canceled, or if the
         *  document has switched to read-only mode.
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).then(null, function (result) {
                // reopen the dialog, if the action handler has failed
                return (result === 'invalid') ? new SheetNameDialog(docView, self.getText(), actionHandler, initOptions).show() : result;
            });
        });

        // initialization -----------------------------------------------------

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

    } }); // class SheetNameDialog

    // exports ================================================================

    return SheetNameDialog;

});
