/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/definednamedialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialogs',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Dialogs, SheetUtils, gt) {

    'use strict';

    // class DefinedNameDialog ================================================

    /**
     * Shows a dialog that allows to create or modify a defined name, and will
     * invoke an arbitrary callback function for the defined name. The dialog
     * will be kept open until the settings for the defined name are valid.
     *
     * @constructor
     *
     * @extends ModalInputDialog
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view instance that has created this dialog.
     *
     * @param {String} label
     *  The initial label for the defined name to be shown when the dialog is
     *  opened.
     *
     * @param {String} formula
     *  The initial formula expression to be shown when the dialog is opened.
     *
     * @param {Function} actionHandler
     *  The callback function invoked when the OK button of the dialog has
     *  been clicked (will be passed as 'validator' option to the dialog).
     *  Receives the current label and formula entered in the dialog, and MUST
     *  return a boolean value or a promise that states whether the operation
     *  using that label and formula expression has been finished successfully.
     *  In case of an error, the dialog will be kept open with its current
     *  settings.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports most options of the base class
     *  ModalInputDialog, except 'value', 'maxLength', 'placeholder', and
     *  'validator'.
     */
    var DefinedNameDialog = Dialogs.ModalInputDialog.extend({ constructor: function (docView, label, formula, actionHandler, initOptions) {

        var // self reference
            self = this,

            // text input field for the formula expression
            formulaField = null;

        // base constructor ---------------------------------------------------

        Dialogs.ModalInputDialog.call(this, Utils.extendOptions(initOptions, {
            inputLabel: gt('Name'),
            value: label,
            maxLength: SheetUtils.MAX_NAME_LENGTH,
            placeholder: gt('Enter name'),
            validator: function () { return actionHandler.call(self, self.getText(), formulaField.val()); }
        }));

        // public methods -----------------------------------------------------

        /**
         * Shows the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved after the
         *  action handler callback function has been invoked successfully; or
         *  will be rejected, if the dialog has been canceled, or if the
         *  document has switched to read-only mode.
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).then(null, function (result) {
                // reopen the dialog, if the action handler has failed
                return (result === 'invalid') ? new DefinedNameDialog(docView, self.getText(), formulaField.val(), actionHandler, initOptions).show() : result;
            });
        });

        // initialization -----------------------------------------------------

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // create and insert the formula text field into the dialog body
        formulaField = $('<input tabindex="0" type="text" class="form-control">').attr('placeholder', gt('Enter formula')).val(formula);
        this.getContentNode().append($('<label>').text(gt('Formula')), formulaField);

    } }); // class DefinedNameDialog

    // exports ================================================================

    return DefinedNameDialog;

});
