/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/formatcodepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, SimpleMap, RadioList, Labels, gt) {

    'use strict';

    // class FormatCodePicker =================================================

    /**
     * A drop-down list control for selection of predefined number formats
     * depending to currently selected category in the FormatCategoryPicker.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {SpreadsheetView} docView
     *  The document view containing this instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioList.
     */
    var FormatCodePicker = RadioList.extend({ constructor: function (docView, initOptions) {

        // self reference
        var self = this;

        // the number formatter of the document
        var numberFormatter = docView.getDocModel().getNumberFormatter();

        // a map used to resolve alternative format codes to list entries in this control
        var formatCodeMap = new SimpleMap();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: Labels.MORE_LABEL,
            //#. special number format codes for different categories of number formats
            tooltip: gt.pgettext('format-code', 'Format codes')
        }, initOptions, {
            updateCaptionMode: 'none',
            itemMatcher: itemMatcher
        }));

        // private methods ----------------------------------------------------

        /**
         * Returns whether the passed (arbitrary) format codes matches with the
         * format code of a list item.
         */
        function itemMatcher(formatCode, itemCode) {
            return (typeof formatCode === 'string') && (formatCodeMap.get(formatCode.toLowerCase(), Labels.CUSTOM_FORMAT_VALUE) === itemCode);
        }

        function isCategoryCode(formatCode, category) {
            return (typeof formatCode === 'string') && (numberFormatter.getParsedFormat(formatCode).category === category);
        }

        function isCustomCode(formatCode) {
            return (typeof formatCode === 'string') && !formatCodeMap.has(formatCode.toLowerCase());
        }

        function createListEntry(predefFormat) {

            // the parsed number format
            var parsedFormat = predefFormat.parsedFormat;
            // the format section for negative numbers, needed to detect red codes
            var negativeSection = parsedFormat.numberSections[1];

            // create an option button for the primary format code
            self.createOptionButton(parsedFormat.formatCode, {
                section: parsedFormat.category,
                label: _.noI18n(numberFormatter.formatValue(parsedFormat, predefFormat.previewValue)),
                labelStyle: (negativeSection && negativeSection.colorName) ? ('color:#' + negativeSection.rgbColor + ';') : null,
                visible: function (formatCode) { return isCategoryCode(formatCode, parsedFormat.category); }
            });

            // register all format codes in the reverse map
            predefFormat.formatCodes.forEach(function (formatCode) {
                formatCodeMap.insert(formatCode.toLowerCase(), parsedFormat.formatCode);
            });
        }

        // initialization -----------------------------------------------------

        // create all list entries
        Labels.FORMAT_CATEGORY_LIST.forEach(function (categoryData) {
            var predefFormats = numberFormatter.getPredefinedFormats(categoryData.name);
            if (predefFormats) { predefFormats.forEach(createListEntry); }
        });

        // create the 'Custom' entry
        this.createOptionButton(Labels.CUSTOM_FORMAT_VALUE, {
            section: 'custom',
            label: Labels.CUSTOM_FORMAT_LABEL,
            visible: isCustomCode
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = initOptions = numberFormatter = null;
        });

    } }); // class FormatCodePicker

    // exports ================================================================

    return FormatCodePicker;

});
