/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/formatcategorygroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/control/radiogroup',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, LocaleData, RadioGroup, Labels, gt) {

    'use strict';

    // class FormatCategoryGroup ==============================================

    /**
     * A button group with selected number format categories.
     *
     * @constructor
     *
     * @extends RadioGroup
     *
     * @param {SpreadsheetView} docView
     *  The document view containing this instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioGroup.
     */
    function FormatCategoryGroup(docView, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, Utils.extendOptions({
            toggleValue: 'standard'
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Creates an option button for this button group.
         *
         * @param {String} category
         *  The internal number of the number format category, used as value
         *  for the option button.
         *
         * @param {String} symbol
         *  The short label for the option button.
         */
        function createOptionButton(category, symbol) {
            var categoryData = _.findWhere(Labels.FORMAT_CATEGORY_LIST, { name: category });
            self.createOptionButton(category, {
                label: symbol,
                tooltip: categoryData.label,
                dropDownVersion: { label: categoryData.label }
            });
        }

        // initialization -----------------------------------------------------

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().addClass('format-category-group').attr({
            'aria-label': gt('Number formats'),
            'aria-labelledby': 'dummy'
        });

        // initialize the option buttons
        createOptionButton('currency', _.noI18n(LocaleData.CURRENCY));
        createOptionButton('percent', _.noI18n('%'));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = initOptions = null;
        });

    } // class FormatCategoryGroup

    // exports ================================================================

    // derive this class from class RadioGroup
    return RadioGroup.extend({ constructor: FormatCategoryGroup });

});
