/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/spreadsheet/model/numberformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/presetformattable',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, SimpleMap, LocaleData, Parser, Formatter, AppObjectMixin, Operations, SheetUtils, PresetFormatTable, FormulaUtils) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var MathUtils = FormulaUtils.Math;
    var Complex = FormulaUtils.Complex;

    // shortcuts to mathematical functions
    var isZero = MathUtils.isZero;

    // identifiers of predefined formats that can be overridden by user-defined formats
    var REPLACEABLE_FORMAT_ID_SET = Utils.makeSet(PresetFormatTable.CURRENCY_FORMAT_IDS);

    // the JSON data for predefined number formats loaded from the resources
    var PREDEFINED_FORMATS_JSON = null;

    // load the localized number format definitions
    var resourcePromise = LocaleData.loadResource('io.ox/office/spreadsheet/resource/numberformats').done(function (jsonData) {
        PREDEFINED_FORMATS_JSON = _.isObject(jsonData) ? jsonData : {};
    });

    // private global functions ===============================================

    function detectNumType(string) {

        function createResult(type, val, options) {
            return { type: type, value: val, options: options };
        }

        var numberResult = Parser.parseLeadingNumber(string, { sign: true, group: true });
        if (numberResult) {

            // text consists entirely of the floating-point number; return either scientific number format, or a simple number
            if (numberResult.remaining.length === 0) {
                return createResult('number', numberResult.number, {
                    scientific: numberResult.scientific,
                    dec: numberResult.dec,
                    grouped: numberResult.grouped
                });
            }

            // check if there is a fraction in the remaining string
            var fractionResult  = /^\s*(\d+\/\d+)(.*)/.exec(numberResult.remaining),
                remaining       = null,
                number          = null,
                isFraction      = false;

            // if a fraction was found
            if (fractionResult) {
                remaining = fractionResult[2];

                var fraction = fractionResult[1],
                    split    = Utils.trimString(fraction).split('/'),
                    dividend = parseInt(split[0], 10),
                    divisor  = parseInt(split[1], 10);

                if (divisor > 0) {
                    // calculate the absolute number incl. fraction
                    number = (Math.abs(numberResult.number) + (dividend / divisor));
                    // make number negative, if necessary
                    if (numberResult.sign === '-') { number = -number; }

                    // when nothing is remaining, return
                    if (remaining.length === 0) {
                        return createResult('fraction', number, {});

                    // otherwise, set 'isFraction' and try to parse the remaining string
                    } else {
                        isFraction = true;
                    }
                }

            // if no fraction was found
            } else {
                remaining = numberResult.remaining;
                number    = numberResult.number;
            }

            // text consists of a floating-point number with trailing percent sign, white-space allowed;
            // return standard percentage number format with or without decimal places
            if (/^\s*%$/.test(remaining)) {
                return createResult('percent', number / 100, {
                    scientific: numberResult.scientific,
                    dec: (numberResult.dec || isFraction),
                    grouped: numberResult.grouped
                });
            }

            // check if there is a currency sign at the end of the string
            if (Parser.checkFollowingCurrency(remaining)) {
                return createResult('currency', number, {
                    scientific: numberResult.scientific,
                    dec: (numberResult.dec || isFraction),
                    grouped: numberResult.grouped
                });
            }
        }

        return null;
    }

    // class NumberFormat =====================================================

    /**
     * Representation of a user-defined number format.
     *
     * @constructor
     *
     * @property {String} code
     *  The number format code.
     *
     * @property {Boolean} persistent
     *  Whether the number format is persistent, i.e. whether it has been
     *  created by an 'insertNumberFormat' document operation (see property
     *  'transient' for more details).
     *
     * @property {Number} transient
     *  The number of temporary usages of the number format without defining it
     *  with an 'insertNumberFormat' document operation, e.g. in the attribute
     *  set of a conditional formatting rule which contains both attributes
     *  'formatId' and 'formatCode'.
     */
    function NumberFormatDescriptor(code) {

        this.code = code;
        this.persistent = false;
        this.transient = 0;

    } // class NumberFormatDescriptor

    // class NumberFormatter ==================================================

    /**
     * A number formatter for a spreadsheet document. Supports the format code
     * syntax used in document operations, and allows to deal with all data
     * types of a spreadsheet document: numbers, strings, Boolean values, and
     * error codes.
     *
     * @constructor
     *
     * @extends Formatter
     * @extends AppObjectMixin
     *
     * @param {SpreadsheetModel} docModel
     *  The document model containing this instance.
     */
    var NumberFormatter = Formatter.extend({ constructor: function (docModel) {

        // self reference
        var self = this;

        // special behavior for different file formats
        var fileFormat = docModel.getApp().getFileFormat();

        // access to localized formula labels
        var formulaGrammar = docModel.getFormulaGrammar('ui');

        // the predefined format codes of the UI locale
        var presetTable = PresetFormatTable.create(LocaleData.LOCALE);

        // the user-defined format codes, and the overridden preset codes, mapped by identifiers
        var formatTable = new SimpleMap();

        // the number format identifiers, mapped by format codes
        var reverseTable = new SimpleMap();

        // the first unused number format identifier (as used in OOXML)
        var firstFreeId = PresetFormatTable.FIRST_USER_FORMAT_ID;

        // predefined number formats, as arrays mapped by categories
        var predefFormatsMap = new SimpleMap();

        // the format code of the standard number format
        var standardCode = presetTable.getFormatCode(PresetFormatTable.GENERAL_ID);

        // the parsed standard number format
        var standardFormat = Parser.parseFormatCode(fileFormat, 'op', standardCode);

        // base constructors --------------------------------------------------

        Formatter.call(this, {
            nullDate: Date.UTC(1899, 11, 30),
            negativeDates: fileFormat === 'odf',
            leapYearBug: fileFormat === 'ooxml',
            generalLength: SheetUtils.MAX_LENGTH_STANDARD_CELL
        });
        AppObjectMixin.call(this, docModel.getApp());

        // private methods ----------------------------------------------------

        /**
         * Creates a new entry in the number format table.
         *
         * @param {Number} formatId
         *  The identifier for the new number format.
         *
         * @param {String} formatCode
         *  The format code to be associated with the specified identifier.
         *
         * @returns {NumberFormatDescriptor}
         *  The descriptor object representing the new number format.
         */
        function createNumberFormat(formatId, formatCode) {

            // create and insert a new descriptor for the number format
            var formatDesc = new NumberFormatDescriptor(formatCode);
            formatTable.insert(formatId, formatDesc);

            // update the first free identifier for user-defined formats
            while (formatTable.has(firstFreeId)) { firstFreeId += 1; }

            // return the new format descriptor
            return formatDesc;
        }

        /**
         * Removes an existing entry from the number format table.
         *
         * @param {Number} formatId
         *  The identifier for the number format to be removed.
         *
         * @returns {NumberFormatDescriptor|Null}
         *  The descriptor object representing the removed number format, if
         *  existing; otherwise null.
         */
        function removeNumberFormat(formatId) {

            // remove the format descriptor from the map
            var formatDesc = formatTable.remove(formatId, null);
            if (!formatDesc) { return null; }

            // update the first free identifier for user-defined formats
            if ((formatId >= PresetFormatTable.FIRST_USER_FORMAT_ID) && (formatId <= firstFreeId)) {
                firstFreeId = formatId;
                while ((firstFreeId > PresetFormatTable.FIRST_USER_FORMAT_ID) && !formatTable.has(firstFreeId - 1)) {
                    firstFreeId -= 1;
                }
            }

            // remove the format code from the reverse table
            reverseTable.remove(formatDesc.code);
            return formatDesc;
        }

        /**
         * Extracts the attributes 'formatId' and 'formatCode' from the passed
         * cell attributes, and invokes the callback function if both
         * attributes exist.
         */
        function updateTransientFormat(cellAttrs, updateHandler) {

            // nothing to do for non-OOXML files
            if (fileFormat !== 'ooxml') { return; }

            // extract format identifier and the explicit format code from the attributes
            var formatId = Utils.getIntegerOption(cellAttrs, 'formatId', null);
            var formatCode = Utils.getStringOption(cellAttrs, 'formatCode', null);
            if ((formatId === null) || (formatCode === null)) { return; }

            // invoke the callback function with identifier and format code
            updateHandler(formatId, formatCode, formatTable.get(formatId, null));
        }

        function generateCurrencyFormatCode(symbol, options) {
            /*
            var withDec         = '0.00',
                withoutDec      = '0',
                withGroup       = '#,##',
                withoutGroup    = '',
                symbolCode      = (symbol === LocaleData.ISO_CURRENCY) ? ('[$' + symbol + ']') : symbol,
                formatCode      = options.leading ? (symbolCode + ' ') : '';

            formatCode += options.grouped ? withGroup : withoutGroup;
            formatCode += options.dec ? withDec : withoutDec;

            if (!options.leading) { formatCode += ' ' + symbolCode; }

            return formatCode;
            */

            // bug #49613: now simply get the predefined-codes until ...
            //             TODO: we cleaned up this whole piece of code
            var preDefined = self.getPredefinedFormats('currency');
            return (options.dec) ? preDefined[3].formatCodes[0] : preDefined[2].formatCodes[0];
        }

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'insertNumberFormat'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertNumberFormat' document operation.
         */
        this.applyInsertOperation = function (context) {

            // format identifier must not be negative
            var formatId = context.getInt('id');
            context.ensure(formatId >= 0, 'unexpected negative format identifier');

            // do not replace predefined number formats (except for currency formats), but do not fail in this case
            if ((formatId < PresetFormatTable.FIRST_USER_FORMAT_ID) && !(formatId in REPLACEABLE_FORMAT_ID_SET)) {
                Utils.warn('NumberFormatter.applyInsertOperation(): attempted to replace predefined number format ' + formatId);
                return;
            }

            // get the new format code from the operation (bug 46211: may be an empty string)
            var formatCode = context.getStr('code', true);

            // insert the format code into the internal map, or update an existing map entry;
            // remove the old existing user-defiend format code from the reverse map
            var formatDesc = formatTable.get(formatId, null);
            if (formatDesc) {
                reverseTable.remove(formatDesc.code);
            } else {
                formatDesc = createNumberFormat(formatId, formatCode);
            }

            // mark the number format to be persistent, insert the format code into the reverse map
            formatDesc.persistent = true;
            reverseTable.insert(formatCode, formatId);
        };

        /**
         * Callback handler for the document operation 'deleteNumberFormat'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteNumberFormat' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // format identifier must refer to a user-defined number format
            var formatId = context.getInt('id');
            context.ensure(formatId >= PresetFormatTable.FIRST_USER_FORMAT_ID, 'invalid format identifier');

            // reset the persistent flag, but do not remove the number format, if it is
            // still in use, e.g. in the attribute set of a conditional formatting rule
            var formatDesc = formatTable.get(formatId, null);
            context.ensure(formatDesc && formatDesc.persistent, 'invalid format identifier');
            if (formatDesc.transient <= 0) {
                formatTable.remove(formatId);
            } else {
                formatDesc.persistent = false;
            }

            // update the first free identifier for user-defined formats
            if (formatId <= firstFreeId) {
                firstFreeId = formatId;
                while ((firstFreeId > PresetFormatTable.FIRST_USER_FORMAT_ID) && !formatTable.has(firstFreeId - 1)) {
                    firstFreeId -= 1;
                }
            }
        };

        /**
         * Registers a transient number format, i.e. a number format that is
         * used in attribute sets without having created it before with an
         * 'insertNumberFormat' document operation.
         *
         * @param {Object|Null} cellAttrs
         *  An incomplete attribute map containing cell attributes. If the
         *  parameter is an object that contains the attibutes 'formatId' AND
         *  'formatCode', a transient number format will be registered.
         */
        this.registerTransientFormat = function (cellAttrs) {
            updateTransientFormat(cellAttrs, function (formatId, formatCode, formatDesc) {

                // create a new entry in the format table on demand (existing format code should not change, but do not fail)
                if (!formatDesc) {
                    formatDesc = createNumberFormat(formatId, formatCode);
                } else if (formatDesc.code !== formatCode) {
                    Utils.warn('NumberFormatter.registerTransientFormat(): format code mismatch: id: ' + formatId + ', old code: ' + formatDesc.code + ', new code: ' + formatCode);
                }

                // update the transient counter
                formatDesc.transient += 1;
            });
        };

        /**
         * Unregisters a transient number format that has been registered with
         * the method NumberFormatter.registerTransientFormat() before.
         *
         * @param {Object|Null} cellAttrs
         *  An incomplete attribute map containing cell attributes. If the
         *  parameter is an object that contains the attibutes 'formatId' AND
         *  'formatCode', a transient number format will be unregistered.
         */
        this.unregisterTransientFormat = function (cellAttrs) {
            updateTransientFormat(cellAttrs, function (formatId, formatCode, formatDesc) {

                // get the descriptor for the number format from the map (number format should exist, but do not fail)
                if (!formatDesc) {
                    Utils.warn('NumberFormatter.unregisterTransientFormat(): missing format code: id: ' + formatId);
                    return;
                }

                // update the transient counter
                formatDesc.transient -= 1;

                // remove the number format from the map only, if it is not persistent
                if (!formatDesc.persistent && (formatDesc.transient <= 0)) {
                    removeNumberFormat(formatId);
                }
            });
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the predefined format code for the specified identifier of a
         * number format, ignoring any user-defined number formats that are
         * overriding the preset format codes.
         *
         * @param {Number} formatId
         *  The identifier of a preset format code.
         *
         * @returns {String|Null}
         *  The predefined format code for the specified format identifier, if
         *  available; otherwise null.
         */
        this.getPresetCode = function (formatId) {
            return presetTable.getFormatCode(formatId);
        };

        /**
         * Returns the format code for the standard number format.
         *
         * @returns {String}
         *  The format code for the standard number format.
         */
        this.getStandardCode = function () {
            return standardCode;
        };

        /**
         * Returns a currency format code for the native currency of the
         * current UI language.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.int=false]
         *      Whether to return a format code for an integral currency
         *      (true), or a format code for a currency with two decimal places
         *      (false).
         *  @param {Boolean} [options.red=false]
         *      Specifies whether negative currencies will be formatted with
         *      red text color.
         *  @param {Boolean} [options.blind=false]
         *      Whether to replace the currency symbol in the format code with
         *      white-space (true), or to use the actual currency symbol
         *      (false).
         *
         * @returns {String}
         *  A currency format code for the native currency of the current UI
         *  language.
         */
        this.getCurrencyCode = function (options) {
            var int = Utils.getBooleanOption(options, 'int', false);
            var red = Utils.getBooleanOption(options, 'red', false);
            var blind = Utils.getBooleanOption(options, 'blind', false);
            var formatId = PresetFormatTable.getCurrencyId(int, red, blind);
            return presetTable.getFormatCode(formatId);
        };

        /**
         * Returns a format code for times in the current UI language.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean|Null} [options.hours24=null]
         *      Specifies how to show the hours of the time. If set to true,
         *      hours will be shown as number from 0 to 23. If set to false,
         *      hours will be shown as number from 1 to 12, and the format code
         *      will include the AM/PM token. If set to null or omitted, the
         *      hour mode will be selected as preferred by the current UI
         *      language.
         *  @param {Boolean} [options.seconds=false]
         *      Whether the format code includes the seconds of the time.
         *
         * @returns {String}
         *  A format code for times in the current UI language.
         */
        this.getTimeCode = function (options) {
            var hours24 = Utils.getBooleanOption(options, 'hours24', null);
            var seconds = Utils.getBooleanOption(options, 'seconds', false);
            var formatId = PresetFormatTable.getTimeId(hours24, seconds);
            return presetTable.getFormatCode(formatId);
        };

        /**
         * Returns the predefined number formats for a specific number format
         * category.
         *
         * @param {String} category
         *  The identifier of a number format category.
         *
         * @returns {Array<Object>|Null}
         *  An array of descriptor objects for all predefined number formats
         *  for the specified category; or null if no number formats exist.
         *  Each descriptor in the array contains the following properties:
         *  - {ParsedFormat} parsedFormat
         *      The parsed number format.
         *  - {Array<String>} formatCodes
         *      All format codes that are considered to represent the current
         *      predefined number format.
         *  - {Number|String} previewValue
         *      The recommended value to be used to represent the number format
         *      in the GUI.
         */
        this.getPredefinedFormats = function (category) {
            return predefFormatsMap.get(category, null);
        };

        /**
         * Returns the default code for the specified number format category.
         *
         * @param {String} category
         *  The identifier of a number format category.
         *
         * @returns {String|Null}
         *  The format code for the specified number format category, if
         *  available; otherwise null.
         */
        this.getDefaultCode = function (category) {

            // get a predefined format code from the map if available
            var predefFormats = this.getPredefinedFormats(category);
            if (predefFormats) { return predefFormats[0].parsedFormat.formatCode; }

            // return the format code for standard category
            return (category === 'standard') ? standardCode : null;
        };

        /**
         * Converts the specified number format to the effective format code.
         *
         * @param {Number|String} format
         *  The identifier of a number format as integer, or a format code as
         *  string.
         *
         * @returns {String|Null}
         *  If a format code (as string) has been passed, it will be returned
         *  immediately. The identifier of a number format (as integer) will be
         *  converted to its user-defined or preset format code, if available.
         *  Otherwise, null will be returned.
         */
        this.resolveFormatCode = function (format) {

            // immediately return passed format code
            if (typeof format === 'string') { return format; }

            // try to resolve user-defined format code, fall-back to predefined format code
            var formatDesc = formatTable.get(format, null);
            return formatDesc ? formatDesc.code : presetTable.getFormatCode(format);
        };

        /**
         * Converts the specified number format to the identifier of an
         * existing predefined or user-defined number format.
         *
         * @param {Number|String} format
         *  The identifier of a number format as integer, or a format code as
         *  string.
         *
         * @returns {Number|Null}
         *  If a format identifier (as integer) has been passed, it will be
         *  returned immediately. A format code (as string) will be resolved to
         *  a user-defined or preset number format, and its identifier will be
         *  returned, if available. Otherwise, null will be returned.
         */
        this.resolveFormatId = function (format) {

            // immediately return passed format identifiers
            if (typeof format === 'number') { return format; }

            // try to resolve identifier of a user-defined number format
            var formatId = reverseTable.get(format, null);
            if (formatId !== null) { return formatId; }

            // try to resolve a predefined number format, but DO NOT return the identifier
            // if the predefined format has been overridden by a user-defined number format
            var presetId = presetTable.getFormatId(format);
            return ((presetId !== null) && !formatTable.has(presetId)) ? presetId : null;
        };

        /**
         * Returns the parsed number format descriptor for the standard number
         * format.
         *
         * @returns {ParsedFormat}
         *  The parsed number format descriptor for the standard number format.
         */
        this.getStandardFormat = function () {
            return standardFormat;
        };

        /**
         * Returns the parsed number format descriptor for the specified number
         * format.
         *
         * @param {Number|String} format
         *  The identifier of a number format as integer, or a format code as
         *  string.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.grammarId='op']
         *      The identifier of a format grammar. See description of the
         *      static method Parser.parseFormatCode() for details.
         *
         * @returns {ParsedFormat}
         *  The parsed number format descriptor of a number format code.
         */
        this.getParsedFormat = function (format, options) {
            var grammarId = Utils.getStringOption(options, 'grammarId', 'op');
            var formatCode = this.resolveFormatCode(format);
            return (formatCode !== null) ? Parser.parseFormatCode(fileFormat, grammarId, formatCode) : standardFormat;
        };

        /**
         * Returns the parsed number format descriptor for the specified cell
         * formatting attributes.
         *
         * @param {Object} cellAttrs
         *  The merged (complete) cell attribute map, expected to contain the
         *  'formatId' attribute, and the 'formatCode' attribute.
         *
         * @returns {ParsedFormat}
         *  The parsed number format descriptor for the passed cell attributes.
         */
        this.getParsedFormatForAttributes = function (cellAttrs) {

            switch (fileFormat) {
                // OOXML: use the format identifier only
                case 'ooxml': return this.getParsedFormat(cellAttrs.formatId);
                // ODF: use the format code only
                case 'odf': return this.getParsedFormat(cellAttrs.formatCode);
            }

            Utils.error('NumberFormatter.parseAttributesFormat(): cannot resolve attributes');
            return standardFormat;
        };

        /**
         * Tries to convert the passed text to a scalar cell value (either a
         * floating-point number, a boolean value, an error code, or a string).
         *
         * @param {String} text
         *  The string to be converted to a scalar cell value.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.blank=false]
         *      If set to true, the empty string will be treated as blank cell,
         *      and the property 'value' of the result will be set to null.
         *  @param {Boolean} [options.keepApos=false]
         *      If set to true, a leading apostrophe in the passed text (used
         *      to protect a string from conversion to other data types) will
         *      be kept in the property 'value' of the result. by default, the
         *      leading apostrophe will be removed from the string.
         *
         * @returns {Object}
         *  A result descriptor with the resulting value, and a preset format
         *  identifier representing the original formatted text passed to this
         *  method, with the following properties:
         *  - {Number|String|Boolean|ErrorCode|Null} value
         *      The resulting value with correct data type.
         *  - {Number|String} format
         *      The identifier of the best matching preset format code, or an
         *      explicit format code as string, if the passed text has been
         *      converted to a floating-point number in a specific significant
         *      format; or 0 (standard number format) for simple floating-point
         *      numbers, or any other value type. May not fit exactly to the
         *      text, e.g. if the text represents a number in scientific
         *      notation, the identifier of the standard format code for
         *      scientific numbers '0.00E+00' will always be returned.
         */
        this.parseFormattedValue = function (text, options) {

            // creates a result object from the passed value, and optional built-in format code identifier
            function createResult(value, format) {
                return { value: value, format: format || 0 };
            }

            // special handling for the empty string
            if (text === '') {
                return createResult(Utils.getBooleanOption(options, 'blank', false) ? null : text);
            }

            // try to parse boolean literals
            var boolValue = formulaGrammar.getBooleanValue(text);
            if (_.isBoolean(boolValue)) { return createResult(boolValue); }

            // try to parse error code literals
            var errorCode = formulaGrammar.getErrorCode(text);
            if (errorCode) { return createResult(errorCode); }

            // try to parse a floating-point number in different formats (local scope returns null if not successful)
            var parseResult = (function () {

                // ignore leading/trailing whitespace when trying to parse numbers (but not for booleans or errors!)
                var trimmed = Parser.parseBrackets(Utils.trimString(text));

                // try to parse a leading floating-point number (decimal or scientific), TODO: group separator
                var numberResult = detectNumType(trimmed);
                if (numberResult) {

                    var flags = numberResult.options;

                    switch (numberResult.type) {
                        case 'fraction':
                            return createResult(numberResult.value, PresetFormatTable.getFractionId(false));
                        case 'percent':
                            return createResult(numberResult.value, PresetFormatTable.getPercentId(!flags.dec));
                        case 'currency':
                            var currency = (trimmed.indexOf(LocaleData.ISO_CURRENCY) >= 0) ? LocaleData.ISO_CURRENCY : LocaleData.CURRENCY;
                            return createResult(numberResult.value, generateCurrencyFormatCode(currency, _.extend(flags, { leading: false })));
                    }

                    // number contains exponent: use scientific format (always with 2 decimal places)
                    return createResult(numberResult.value, flags.scientific ? PresetFormatTable.getScientificId(true) : flags.grouped ? PresetFormatTable.getDecimalId(!flags.dec, true) : 0);
                }

                // try to detect a currency string with leading currency symbol
                // (for following currency symbol detection look further above)
                var currencyResult = Parser.parseLeadingCurrency(trimmed);
                if (currencyResult) {
                    numberResult = detectNumType(Parser.parseBrackets(Utils.trimString(currencyResult[1])));
                    if (numberResult) {
                        var formatCode = generateCurrencyFormatCode(currencyResult[2], _.extend(numberResult.options, { leading: true }));
                        return createResult(numberResult.value, formatCode);
                    }
                }

                var percentResult = Parser.parseLeadingPercent(trimmed);
                if (percentResult) {
                    numberResult = detectNumType(Parser.parseBrackets(Utils.trimString(percentResult[1])));
                    if (numberResult) {
                        return createResult((numberResult.value / 100), PresetFormatTable.getPercentId(!numberResult.options.dec));
                    }
                }

                // try to parse a leading date, time, or combined date/time or time/date
                var dateResult = null;
                var timeResult = null;

                // try date or date/time (separated with whitespace); then time or time/date
                if ((dateResult = this.parseLeadingDate(trimmed))) {
                    // trailing text (if existing) must be a valid time, otherwise return immediately
                    trimmed = dateResult.remaining;
                    if ((trimmed.length > 0) && !(timeResult = this.parseLeadingTime(Utils.trimString(trimmed), { complete: true }))) {
                        return null;
                    }
                } else if ((timeResult = this.parseLeadingTime(trimmed))) {
                    // trailing text (if existing) must be a valid date, otherwise return immediately
                    trimmed = timeResult.remaining;
                    if ((trimmed.length > 0) && !(/^\s/.test(trimmed) && (dateResult = this.parseLeadingDate(Utils.trimString(trimmed), { complete: true })))) {
                        return null;
                    }
                }

                // convert parsed date/time components to the serial number, and number format (always use the standard date/time format)
                if (dateResult && timeResult) {
                    // Always use format code 22 consisting of the standard date format, followed by an hour/minute time format.
                    return createResult(this.convertDateToNumber(dateResult.date) + timeResult.serial, 22);
                }

                // convert parsed date components (without time) to the serial number, and number format
                if (dateResult) {
                    // Find an identifier of a built-in date format code:
                    // - no year given (day/month only): format code 16 consists of day and month (e.g. 'DD.MM') matching the standard date format,
                    // - no day given (month/year only): format code 17 consists of month and year (e.g. 'MM.YYYY') matching the standard date format,
                    // - otherwise (day, month, year): format code 14 is the standard date format.
                    return createResult(this.convertDateToNumber(dateResult.date), (dateResult.Y === null) ? 16 : (dateResult.D === null) ? 17 : 14);
                }

                // convert parsed time components (without date) to the serial number, and number format
                if (timeResult) {
                    // Find an identifier of a built-in time format code:
                    // - milliseconds given: format code 47 contains a minute/second time format with one decimal place for tenth of seconds,
                    // - time is greater than or equal to 24 hours: format code 46 contains total hour code, e.g. '[h]:mm:ss',
                    // - otherwise, use a standard time format.
                    return createResult(timeResult.serial, (timeResult.ms !== null) ? 47 : (timeResult.serial >= 1) ? 46 : PresetFormatTable.getTimeId(!timeResult.ampm, timeResult.s !== null));
                }

                return null;

            }.call(this));

            // return a parsed floating-point number; everything else is a regular string
            return parseResult ? parseResult : createResult(Utils.getBooleanOption(options, 'keepApos', false) ? text : text.replace(/^'/, ''));
        };

        /**
         * Tries to convert the passed text to a scalar cell value (either a
         * floating-point number, a boolean value, an error code, or a string).
         * This method returns the converted value directly without additional
         * number format information, in difference to the public method
         * NumberFormatter.parseFormattedValue().
         *
         * @param {String} text
         *  The string to be converted to a scalar cell value.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method NumberFormatter.parseFormattedValue().
         *
         * @returns {Number|Boolean|String|ErrorCode|Null}
         *  The resulting value with correct data type.
         */
        this.parseValue = function (text, options) {
            return this.parseFormattedValue(text, options).value;
        };

        /**
         * Converts the passed scalar cell value to a string, formatted with
         * the specified number format.
         *
         * @param {ParsedFormat} parsedFormat
         *  The parsed format code that will be used to format the passed value
         *  as string.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The scalar cell value to be formatted.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all parameters also supported by of
         *  the base class method.
         *
         * @returns {String|Null}
         *  The formatted value; or null, if the passed number format code is
         *  invalid (e.g. syntax error in the format code), or if the value
         *  cannot be formatted with the format code (e.g. a number is too
         *  large to be formatted as date/time).
         */
        this.formatValue = _.wrap(this.formatValue, function (formatValueFunc, parsedFormat, value, options) {

            // blank cells appear always empty
            if (value === null) {
                return '';
            }

            // error codes will never be formatted
            if (value instanceof ErrorCode) {
                return formulaGrammar.getErrorName(value);
            }

            // format boolean values like strings (with text format section)
            if (typeof value === 'boolean') {
                value = formulaGrammar.getBooleanName(value);
            }

            // use base class method to format numbers and strings
            return formatValueFunc.call(this, parsedFormat, value, options);
        });

        /**
         * Converts the passed complex number to a string. The real and
         * imaginary coefficients will be formatted with the standard number
         * format using the passed maximum number of digits.
         *
         * @param {Complex} complex
         *  The complex number to be converted to a string.
         *
         * @param {Number} maxNumLength
         *  The maximum number of characters allowed for the absolute part of
         *  a coefficient, including the decimal separator and the complete
         *  exponent in scientific notation, but without the sign character.
         *  MUST be positive.
         *
         * @returns {String}
         *  The converted complex number.
         */
        this.convertComplexToString = function (complex, maxNumLength) {

            // the real coefficient, as string
            var real = this.formatStandardNumber(complex.real, maxNumLength);
            // the imaginary coefficient, as string
            var imag = this.formatStandardNumber(complex.imag, maxNumLength);
            // fall-back to unit 'i'
            var unit = complex.unit || 'i';

            // leave out imaginary part if zero
            if (complex.imag === 0) { return real; }

            // leave out real part if missing; do not add single '1' for imaginary part
            return ((complex.real === 0) ? '' : real) + (((complex.real !== 0) && (complex.imag > 0)) ? '+' : '') + (/^-?1$/.test(imag) ? '' : imag) + unit;
        };

        /**
         * Converts the passed text to a complex number.
         *
         * @param {String} text
         *  The string to be converted to a complex number.
         *
         * @returns {Complex|Null}
         *  The complex number represented by the passed string; or null, if
         *  the string cannot be parsed to a complex number.
         */
        this.convertStringToComplex = function (text) {

            // the matches of a regular expression
            var matches = null;
            // the parse result for a floating-point number from formatter
            var parseResult = null;
            // the parsed real coefficient
            var real = 0;

            // do not accept empty strings
            if (text.length === 0) { return null; }

            // string may be a single imaginary unit without coefficients: i, +i, -i (same for j)
            if ((matches = /^([-+]?)([ij])$/.exec(text))) {
                return new Complex(0, (matches[1] === '-') ? -1 : 1, matches[2]);
            }

            // pull leading floating-point number from the string
            if (!(parseResult = Parser.parseLeadingNumber(text, { sign: true }))) {
                return null;
            }

            // check for simple floating-point number without imaginary coefficient: a, +a, -a
            real = parseResult.number;
            text = parseResult.remaining;
            if (text === '') {
                return new Complex(real, 0);
            }

            // check for imaginary number without real coefficient: bi, +bi, -bi
            if ((text === 'i') || (text === 'j')) {
                return new Complex(0, real, text);
            }

            // check for following imaginary unit without coefficients, but with sign: a+i, a-i
            if ((matches = /^([-+])([ij])$/.exec(text))) {
                return new Complex(real, (matches[1] === '-') ? -1 : 1, matches[2]);
            }

            // pull trailing floating-point number from the string: a+bi, a-bi (sign is required here, something like 'abi' is not valid)
            if (!(parseResult = Parser.parseLeadingNumber(text, { sign: true })) || (parseResult.sign.length === 0)) {
                return null;
            }

            // remaining text must be the imaginary unit
            text = parseResult.remaining;
            if ((text === 'i') || (text === 'j')) {
                return new Complex(real, parseResult.number, text);
            }

            return null;
        };

        /**
         * Tries to convert the passed scalar value to a floating-point number.
         *
         * @param {Any} value
         *  A scalar value to be converted to a number. Dates will be converted
         *  to their serial numbers. Strings that represent a valid number
         *  (according to the current GUI language) will be converted to the
         *  number. The boolean value FALSE will be converted to 0, the boolean
         *  value TRUE will be converted to 1. The special value null
         *  (representing an empty cell) will be converted to 0. Other values
         *  (error codes, other strings, infinite numbers) cannot be converted
         *  to a number and result in returning the value null.
         *
         * @param {Boolean} [floor=false]
         *  If set to true, the number will be rounded down to the next integer
         *  (negative numbers will be rounded down too, i.e. away from zero!).
         *
         * @returns {Number|Null}
         *  The floating-point number, if available; otherwise null.
         */
        this.convertScalarToNumber = function (value, floor) {

            // convert strings and booleans to numbers
            if (_.isString(value)) {
                value = this.parseValue(value);
            } else if (_.isBoolean(value)) {
                value = value ? 1 : 0;
            } else if (value instanceof Date) {
                value = this.convertDateToNumber(value); // returns null for invalid dates
            } else if (value === null) {
                value = 0;
            }

            // the resulting value must be a finite floating-point number
            return ((typeof value === 'number') && isFinite(value)) ? (floor ? Math.floor(value) : value) : null;
        };

        /**
         * Tries to convert the passed scalar value to a UTC date.
         *
         * @param {Any} value
         *  A scalar value to be converted to a date. Strings that represent a
         *  valid number (according to the current GUI language) will be
         *  converted to a date representing that serial number. The boolean
         *  value FALSE will be converted to the null date of this formatter,
         *  the boolean value TRUE will be converted to the day following the
         *  null date. The special value null (representing an empty cell) will
         *  be converted to the null date too. Other values (error codes, other
         *  strings, infinite numbers) cannot be converted to a date and result
         *  in returning the value null.
         *
         * @param {Boolean} [floor=false]
         *  If set to true, the time components of the resulting date will be
         *  removed (the time will be set to midnight).
         *
         * @returns {Date|Null}
         *  The UTC date, if available; otherwise null.
         */
        this.convertScalarToDate = function (value, floor) {

            // convert scalar values to a date via conversion to a number
            if (!(value instanceof Date)) {
                // convert the value to a number (may return null)
                value = this.convertScalarToNumber(value, floor);
                if (value === null) { return null; }
                // convert the number to a date object (may return null)
                value = this.convertNumberToDate(value);
            }

            // the resulting value must be a valid Date object
            return ((value instanceof Date) && this.isValidDate(value)) ? value : null;
        };

        /**
         * Tries to convert the passed scalar value to a string.
         *
         * @param {Any} value
         *  A scalar value to be converted to a string. Numbers and dates will
         *  be converted to decimal or scientific notation (according to the
         *  current GUI language), boolean values will be converted to their
         *  localized text representation. The special value null (representing
         *  an empty cell) will be converted to the empty string. Other values
         *  (error codes, infinite numbers) cannot be converted to a string and
         *  result in returning the value null.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of a
         *  floating-point number converted to a string, including the decimal
         *  separator and the complete exponent in scientific notation, but
         *  without a leading minus sign. MUST be positive.
         *
         * @returns {String|Null}
         *  The string representation of the value, if available; otherwise
         *  null.
         */
        this.convertScalarToString = function (value, maxLength) {

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertScalarToNumber(value);
                if (value === null) { return null; } // do not convert invalid date to empty string
            }

            // convert numbers and booleans to strings
            if ((typeof value === 'number') && isFinite(value)) {
                value = this.formatStandardNumber(value, maxLength);
            } else if (_.isBoolean(value)) {
                value = formulaGrammar.getBooleanName(value); // always localized
            } else if (value === null) {
                value = '';
            }

            // the resulting value must be a string
            return _.isString(value) ? value : null;
        };

        /**
         * Tries to convert the passed scalar value to a boolean value.
         *
         * @param {Any} value
         *  A scalar value to be converted to a boolean. Floating-point numbers
         *  will be converted to TRUE if not zero, otherwise FALSE. The null
         *  date will be converted to FALSE, all other dates will be converted
         *  to TRUE (if they are valid according to the configuration of this
         *  formatter). Strings containing the exact translated name of the
         *  TRUE or FALSE values (case-insensitive) will be converted to the
         *  respective boolean value. The special value null (representing an
         *  empty cell) will be converted to FALSE. Other values (error codes,
         *  other strings, infinite numbers) cannot be converted to a boolean
         *  and result in returning the value null.
         *
         * @returns {Boolean|Null}
         *  The boolean value, if available; otherwise null.
         */
        this.convertScalarToBoolean = function (value) {

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertScalarToNumber(value);
                if (value === 0) { return null; } // do not convert invalid date to false
            }

            // convert numbers and strings to booleans
            if ((typeof value === 'number') && isFinite(value)) {
                value = !isZero(value);
            } else if (_.isString(value)) {
                // always localized, returns null on failure
                value = formulaGrammar.getBooleanValue(value);
            } else if (value === null) {
                value = false;
            }

            // the resulting value must be a boolean
            return _.isBoolean(value) ? value : null;
        };

        // operation generators -----------------------------------------------

        /**
         * Returns the cell attributes for a number format to be inserted into
         * an auto-style or a style sheet. If required, an operation may be
         * generated that defines a new user-defined number format, according
         * to the current file format.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operation.
         *
         * @param {Number|String} format
         *  The identifier of a number format as integer, or a format code as
         *  string.
         *
         * @returns {Object}
         *  The resulting cell attributes.
         */
        this.generateNumberFormatOperations = function (generator, format) {

            // the effective format code
            var formatCode = this.resolveFormatCode(format);
            // the effective identifier
            var formatId = this.resolveFormatId(format);
            // the resulting formatting attributes (always insert the format code)
            var cellAttrs = { formatCode: formatCode };

            switch (fileFormat) {

                // OOXML: check existence of the format code, create a user-defined format code on demand
                case 'ooxml':

                    // no number format found (neither predefined nor user-defined): create a new number format entry
                    if (formatId === null) { formatId = firstFreeId; }

                    // do not create explicit number format for predefined formats, but create an operation for existing
                    // user-defined but transient format codes, created e.g. from a conditional formatting rule
                    if (formatId >= PresetFormatTable.FIRST_USER_FORMAT_ID) {
                        var formatDesc = formatTable.get(formatId, null);
                        if (!formatDesc || !formatDesc.persistent) {
                            generator.generateOperation(Operations.INSERT_NUMBER_FORMAT, { id: formatId, code: formatCode });
                            generator.generateOperation(Operations.DELETE_NUMBER_FORMAT, { id: formatId }, { undo: true });
                        }
                    }
                    cellAttrs.formatId = formatId;
                    break;

                // ODF: do not create any additional operations, always use the number format property,
                // but add the preset format identifiers for further usage in the export filters
                // bug 49610: clear the 'formatId' attribute for user-defined format codes
                case 'odf':
                    var isPredef = (formatId !== null) && (formatId < PresetFormatTable.FIRST_USER_FORMAT_ID);
                    cellAttrs.formatId = isPredef ? formatId : null;
                    break;
            }

            return cellAttrs;
        };

        // initialization -----------------------------------------------------

        // generate all predefined number formats, grouped by category
        (function () {

            // preview value for decimal number formats
            var PREVIEW_NUMBER_DECIMAL = -1234.321;

            // returns a new version of the passed simple format code (single section) with red text color
            function getRedCode(formatCode, red) {
                return red ? (formatCode + ';[RED]-' + formatCode) : formatCode;
            }

            // creates a map entry for the passed format code(s)
            function createEntry(formatCodes, previewValue) {

                // convert parameter to an array
                formatCodes = _.getArray(formatCodes);

                // parse the first format code in the array (native format code for that entry)
                var parsedFormat = self.getParsedFormat(formatCodes[0]);
                var category = parsedFormat.category;
                var predefFormats = predefFormatsMap.getOrCreate(category, function () { return []; });
                predefFormats.push({ parsedFormat: parsedFormat, formatCodes: formatCodes, previewValue: previewValue });
            }

            // creates a format code for decimal number formats
            function createNumberEntry(int, group, red) {
                var formatId = PresetFormatTable.getDecimalId(int, group);
                var formatCode = presetTable.getFormatCode(formatId);
                createEntry(getRedCode(formatCode, red), PREVIEW_NUMBER_DECIMAL);
            }

            // creates a format code for currency number formats
//            function createCurrencyEntry(int, red) {
//                var formatId = PresetFormatTable.getCurrencyId(int, red, false);
//                var formatCode = presetTable.getFormatCode(formatId);
//                createEntry(formatCode, PREVIEW_NUMBER_DECIMAL);
//            }

            // creates a format code for date/time formats
            function createDateTimeEntry(seconds) {
                var formatCode = presetTable.getFormatCode(PresetFormatTable.SYSTEM_DATETIME_ID);
                createEntry(seconds ? formatCode.replace(/:mm/, ':mm:ss') : formatCode, 36525 + 49066 / 86400);
            }

            // creates a format code for percentage formats
            function createPercentEntry(int, red) {
                var formatCode = presetTable.getFormatCode(PresetFormatTable.getPercentId(int));
                createEntry(getRedCode(formatCode, red), -0.1234);
            }

            // creates a format code for scientific formats
            function createScientificEntry(long) {
                var formatCode = presetTable.getFormatCode(PresetFormatTable.getScientificId(long));
                // bug 35169: accept escaped white-space as matching format code
                createEntry([formatCode, formatCode.replace(' ', '\\ ')], 1234);
            }

            // creates a format code for fraction formats
            function createFractionEntry(digits, number) {
                var formatCode = PresetFormatTable.getFractionCode(digits);
                // bug 35169: accept escaped white-space as matching format code
                createEntry([formatCode, formatCode.replace(' ', '\\ ')], number);
            }

            // creates all format codes for a category loaded from resource data
            function createResourceEntries(category, previewValue) {
                var formatCodes = PREDEFINED_FORMATS_JSON[category];
                if (_.isArray(formatCodes)) {
                    formatCodes.forEach(function (formatCode) {
                        createEntry(formatCode, previewValue);
                    });
                }
            }

            // create all entries for decimal numbers
            createNumberEntry(true,  false, false);
            createNumberEntry(false, false, false);
            createNumberEntry(true,  false, true);
            createNumberEntry(false, false, true);
            createNumberEntry(true,  true,  false);
            createNumberEntry(false, true,  false);
            createNumberEntry(true,  true,  true);
            createNumberEntry(false, true,  true);

            // create all entries for currencies
// TODO
//            createCurrencyEntry(true,  false);
//            createCurrencyEntry(false, false);
//            createCurrencyEntry(true,  true);
//            createCurrencyEntry(false, true);
            createResourceEntries('currency', PREVIEW_NUMBER_DECIMAL);

            // create all entries for date formats
            // TODO
            createResourceEntries('date', 36404);

            // create all entries for time formats
            // TODO
            createResourceEntries('time', 49066 / 86400);

            // create all list entries for date/time formats
            createDateTimeEntry(false);
            createDateTimeEntry(true);

            // create all entries for percentages
            createPercentEntry(true,  false);
            createPercentEntry(true,  true);
            createPercentEntry(false, false);
            createPercentEntry(false, true);

            // create all entries for scientific numbers
            createScientificEntry(true);
            createScientificEntry(false);

            // create all entries for fractions
            createFractionEntry(1, 2.2);
            createFractionEntry(2, 2.44);
            createFractionEntry(3, 2.888);

            // create the entry for the default text format
            createEntry('@', '@');
        }());

        // change the null date via document operation
        this.listenTo(docModel, 'change:attributes', function (event, newAttrs) {
            var matches = /^(\d{4})-(\d{2})-(\d{2})$/.exec(newAttrs.nullDate);
            if (matches) {
                var nullDate = Date.UTC(parseInt(matches[1], 10), parseInt(matches[2], 10) - 1, parseInt(matches[3], 10));
                self.configure({ nullDate: nullDate });
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = formulaGrammar = null;
            presetTable = formatTable = reverseTable = null;
        });

    } }); // class NumberFormatter

    // exports ================================================================

    return resourcePromise.then(_.constant(NumberFormatter));

});
