/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/hyperlinkcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/typedarray',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, TypedArray, TimerMixin, ModelObject, Operations, SheetUtils) {

    'use strict';

    var RangeArray = SheetUtils.RangeArray;
    var Range = SheetUtils.Range;
    var MoveDescriptor = SheetUtils.MoveDescriptor;

    // class HyperlinkModel ===================================================

    /**
     * Simple model representation of a cell range with hyperlink URL.
     *
     * @constructor
     *
     * @property {Range} range
     *  The address of the cell range the URL is attached to.
     *
     * @property {String} url
     *  The URL attached to the cell range.
     */
    function HyperlinkModel(range, url) {
        this.range = range.clone();
        this.url = url;

    } // class HyperlinkModel

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this model instance.
     *
     * @returns {HyperlinkModel}
     *  A clone of this model instance.
     */
    HyperlinkModel.prototype.clone = function () {
        return new HyperlinkModel(this.range, this.url);
    };

    // class HyperlinkModelArray ==============================================

    /**
     * A typed array of HyperlinkModel instances.
     *
     * @constructor
     */
    var HyperlinkModelArray = TypedArray.create(HyperlinkModel);

    // class HyperlinkCollection ==============================================

    /**
     * Collects information about all hyperlinks of a single sheet in a
     * spreadsheet document.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */

    function HyperlinkCollection(sheetModel) {

        // self reference
        var self = this;

        // all hyperlink ranges
        var linkModels = new HyperlinkModelArray();

        var app = sheetModel.getApp();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, sheetModel.getDocModel());
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        // generates the operations to add a new hyperlink range
        function generateInsertHyperlinkOperation(generator, range, url) {
            generator.generateRangeOperation(Operations.INSERT_HYPERLINK, range, { url: url });
            generator.generateRangeOperation(Operations.DELETE_HYPERLINK, range, null, { undo: true });
        }

        // generates the operations to remove a hyperlink range
        function generateDeleteHyperlinkOperation(generator, hyperlink) {
            generator.generateRangeOperation(Operations.DELETE_HYPERLINK, hyperlink.range);
            generator.generateRangeOperation(Operations.INSERT_HYPERLINK, hyperlink.range, { url: hyperlink.url }, { undo: true });
        }

        // generates the restore operation for 'moveCells' (only undo)
        function generateRestoreHyperlinkOperation(generator, hyperlink) {
            generator.generateRangeOperation(Operations.INSERT_HYPERLINK, hyperlink.range, { url: hyperlink.url }, { undo: true });
        }

        /**
         * Returns specific hyperlink ranges from this collection.
         *
         * @param  {Range} range
         *    The range which should be scanned for hyperlink-ranges
         *
         * @param  {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.type='all']
         *      Specifies hhow the hyperlink ranges must match the given range:
         *      - 'all' (default):
         *      'all contain overlap exact'
         *
         * @return {HyperlinkModelArray}
         *  An array with all found hyperlink ranges.
         */
        function findHyperlinks(range, options) {

            var type = Utils.getStringOption(options, 'type', 'all');

            return linkModels.filter(function (linkModel) {
                var linkRange = linkModel.range;
                switch (type) {
                    case 'all':     return range.overlaps(linkRange);
                    case 'contain': return range.contains(linkRange);
                    case 'overlap': return range.overlaps(linkRange) && !range.contains(linkRange);
                    case 'exact':   return range.equals(linkRange);
                }
            });
        }

        // generates all needed operations to split a hyperlink-range
        // (uses add- and delete-methods to generate the operations)
        function splitExistingRanges(generator, range, link) {

            // DELETE old big HyperlinkRange (incl. undo)
            generateDeleteHyperlinkOperation(generator, link);

            // ADD remaining parts of HyperlinkRanges (if exists)
            new RangeArray(link.range).difference(range).forEach(function (ran) {
                generateInsertHyperlinkOperation(generator, ran, link.url);
            });
        }

        /**
         * Transforms the specified hyperlink range.
         *
         * @param {Range} range
         *  The hyperlink range to be transformed.
         *
         * @param {Array<MoveDescriptor>} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  passed hyperlink range.
         *
         * @param {Boolean} [reverse=false]
         *  If set to true, the move descriptors will be processed in reversed
         *  order, and the opposite move operation will be used to transform
         *  the hyperlink range.
         *
         * @returns {Range|Null}
         *  The transformed hyperlink range if available, otherwise null.
         */
        function transformHyperlinkRange(range, moveDescs, reverse) {
            // transform the passed range without expanding the end of the range
            return MoveDescriptor.transformRange(range, moveDescs, { reverse: reverse });
        }

        /**
         * Recalculates the position of all hyperlink ranges, after cells have
         * been moved (including inserted/deleted columns or rows) in the
         * sheet.
         */
        function moveCellsHandler(event, moveDesc) {
            linkModels = linkModels.filter(function (linkModel) {
                linkModel.range = transformHyperlinkRange(linkModel.range, [moveDesc]);
                return linkModel.range;
            });
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { linkModels: linkModels };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * hyperlinks from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {HyperlinkCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the contents of the source collection
            linkModels = cloneData.linkModels.clone(true);
        };

        /**
         * Callback handler for the document operation 'insertHyperlink' that
         * attaches a URL to a cell range.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertHyperlink' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyInsertHyperlinkOperation = function (context) {

            // the new hyperlink model
            var newModel = new HyperlinkModel(context.getRange(), context.getStr('url'));

            // remove all hyperlink ranges completely covered by the new range
            linkModels = linkModels.reject(function (linkModel) {
                return newModel.range.contains(linkModel.range);
            });

            // insert the new hyperlink range
            linkModels.push(newModel);

            // TODO: notify all change listeners?
        };

        /**
         * Callback handler for the document operation 'deleteHyperlink' that
         * removes all covered hyperlink ranges.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteHyperlink' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyDeleteHyperlinkOperation = function (context) {

            // the range to be cleared
            var range = context.getRange();

            // remove all hyperlink ranges overlapping with the range
            linkModels = linkModels.reject(function (linkModel) {
                return linkModel.range.equals(range);
            });

            // TODO: notify all change listeners?
        };

        // public methods -----------------------------------------------------

        /**
         * Returns wether the given range contains at least one
         * hyperlinkrange
         *
         * @param  {Range} range
         *   The range which should be scanned
         *
         * @return {Boolean}
         *   Wether the range contains at least on hyperlink
         */
        this.rangeOverlapsHyperlinks = function (range) {
            return linkModels.some(function (model) {
                return range.overlaps(model.range);
            });
        };

        /**
         * Returns the URL of a hyperlink at the specified address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink attached to the specified cell; or null, if
         *  the cell does not contain a hyperlink.
         */
        this.getCellURL = function (address) {
            var linkModel = linkModels.find(function (model) { return model.range.containsAddress(address); });
            return linkModel ? linkModel.url : null;
        };

        // generator methods --------------------------------------------------

        /**
         * Generates the operations and undo operations to insert or delete
         * hyperlinks for the specified cell ranges.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address to
         *  generate the operations for.
         *
         * @param {String} url
         *  The URL of the hyperlink to be attached to the cell range. If set
         *  to the empty string, existing hyperlinks will be removed instead.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateHyperlinkOperations = function (generator, ranges, url) {

            // whether to generate 'deleteHyperlink' operations
            var deleteHyperlinks = url.length === 0;

            // remove ranges that are completely covered by other ranges
            ranges = RangeArray.get(ranges).filterCovered();

            // create the operations to insert or delete hyperlinks
            return this.iterateArraySliced(ranges, function (range) {
                if (deleteHyperlinks) {

                    // if only a single cell is selected, search the matching HyperlinkRange
                    if (range.single() && self.rangeOverlapsHyperlinks(range)) {
                        range = findHyperlinks(range)[0].range;
                    }

                    // if the selected range matches one or more HyperlinkRanges
                    if (self.rangeOverlapsHyperlinks(range)) {
                        findHyperlinks(range).forEach(function (link) {
                            splitExistingRanges(generator, range, link);
                        });
                    }

                } else {

                    var covered = findHyperlinks(range, { type: 'contain' });
                    var overlaped = findHyperlinks(range, { type: 'overlap' });

                    // delete all complete covered hyperlinks
                    covered.forEach(function (link) {
                        generateDeleteHyperlinkOperation(generator, link);
                    });

                    // split all partly overlaped hyperlinks
                    overlaped.forEach(function (link) {
                        splitExistingRanges(generator, range, link);
                    });

                    // add new hyperlink
                    generateInsertHyperlinkOperation(generator, range, url);
                }
            }, { delay: 'immediate', infoString: 'HyperlinkCollection: generateHyperlinkOperations', app: app });
        };

        /**
         * Generates the operations and undo operations to insert or delete
         * hyperlinks for autofill-action
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Range} ranges
         *  A single cell range address to generate the operations from.
         *
         * @param {Range} targetRange
         *  A single cell range address to generate the operations for.
         *
         * @param {Boolean} columns
         *  Whether to use columns (true), or rows (false).
         *
         * @param {Boolean} reverse
         *  If set to true, the autofill values will be genereated in reversed
         *  order
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateAutoFillOperations = function (generator, range, targetRange, columns, reverse) {

            // source hyperlink which must be handled
            var sourceHyperlinks = HyperlinkModelArray.map(findHyperlinks(range), function (linkModel) {
                return new HyperlinkModel(linkModel.range.intersect(range), linkModel.url);
            });

            // links, which doesn't lean next to the source hyperlink
            var independentHyperlinks = new HyperlinkModelArray();
            // links, which are ajar directly to the source hyperlink
            var ajarHyperlinks = new HyperlinkModelArray();
            // new hyperlinks which must be added (resulting from the "independentHyperlinks")
            var singleHyperlinks = new HyperlinkModelArray();

            // Next Step: Check whether the hyperlinks cover a full side of the sourceRange (top -> bottom, left -> right)
            sourceHyperlinks.forEach(function (linkModel) {

                var range_start = range.getStart(!columns),
                    range_end   = range.getEnd(!columns),
                    link_start  = linkModel.range.getStart(!columns),
                    link_end    = linkModel.range.getEnd(!columns);

                var targetArray = (range_start === link_start && range_end === link_end) ? ajarHyperlinks : independentHyperlinks;
                targetArray.push(linkModel);
            });

            // prepare new independent hyperlink(s)
            var promise = self.iterateArraySliced(independentHyperlinks, function (linkModel) {

                var rest    = targetRange.size(!columns) % range.size(!columns),
                    i       = Math.ceil(targetRange.size(!columns) / range.size(!columns)),
                    multi   = reverse ? -1 : 1;

                for (; i > 0; i -= 1) {
                    var newRange = linkModel.range.clone();
                    newRange.moveBoth(i * range.size(!columns) * multi, !columns);
                    singleHyperlinks.push(new HyperlinkModel(newRange, linkModel.url));
                }

                if (rest > 0) {
                    var start_i         = singleHyperlinks[0].range.getStart(!columns),
                        end_i           = singleHyperlinks[0].range.getEnd(!columns),
                        target_end_i    = targetRange.getEnd(!columns),
                        target_start_i  = targetRange.getStart(!columns);

                    // remove entry, if it's outside of the target-range
                    if (reverse ? (end_i < target_start_i) : (start_i > target_end_i)) {
                        singleHyperlinks.shift();

                    // cut entry, if it's partly out of the target-range
                    } else {
                        if (reverse) {
                            singleHyperlinks[0].range.setStart(target_start_i, !columns);
                        } else {
                            singleHyperlinks[0].range.setEnd(target_end_i, !columns);
                        }
                    }
                }
            }, { delay: 'immediate', infoString: 'HyperlinkCollection: generateAutoFillOperations 1', app: app });

            // Next Step: Create new ajar Hyperlink(s)
            promise = promise.then(function () {
                return self.iterateArraySliced(ajarHyperlinks, function (linkModel) {

                    var linkRange = linkModel.range;

                    var col_i_start = columns ? linkRange.start[0] : targetRange.start[0],
                        row_i_start = columns ? targetRange.start[1] : linkRange.start[1],

                        col_i_end = columns ? linkRange.end[0] : targetRange.end[0],
                        row_i_end = columns ? targetRange.end[1] : linkRange.end[1];

                    linkRange.setStart(col_i_start, true);
                    linkRange.setStart(row_i_start, false);

                    linkRange.setEnd(col_i_end, true);
                    linkRange.setEnd(row_i_end, false);

                    return sheetModel.getCellCollection().generateFillOperations(generator, linkRange, { url: linkModel.url });
                }, { delay: 'immediate', infoString: 'HyperlinkCollection: generateAutoFillOperations 2', app: app });
            });

            // add new (independent) single hyperlink(s)
            promise = promise.then(function () {
                return self.iterateArraySliced(singleHyperlinks, function (linkModel) {
                    return sheetModel.getCellCollection().generateFillOperations(generator, linkModel.range, { url: linkModel.url });
                }, { delay: 'immediate', infoString: 'HyperlinkCollection: generateAutoFillOperations 3', app: app });
            });

            return promise;
        };

        /**
         * Generates the undo operations to restore the hyperlink in this
         * collection that would not be restored automatically with the reverse
         * operation of the passed move descriptor.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<MoveDescriptor>} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  hyperlink in this collection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMoveCellsOperations = function (generator, moveDescs) {

            // collect all hyperlinks that cannot be restored by applying the reversed move operation
            var restoreLinks = new HyperlinkModelArray();
            var promise = this.iterateArraySliced(linkModels, function (linkModel) {

                // transform the hyperlinks back and forth to decide whether it can be restored implicitly
                var transformRange = transformHyperlinkRange(linkModel.range, moveDescs);
                var restoredRange = transformRange ? transformHyperlinkRange(transformRange, moveDescs, true) : null;

                // collect all ranges that cannot be restored implicitly
                if (!restoredRange || restoredRange.differs(linkModel.range)) {
                    restoreLinks.push(linkModel);
                }
            }, { delay: 'immediate', infoString: 'HyperlinkCollection: generateMoveCellsOperations 1', app: app });

            // add all hyperlinks to the undo generator that need to be restored
            promise = promise.then(function () {
                return self.iterateArraySliced(restoreLinks, function (linkModel) {
                    generateRestoreHyperlinkOperation(generator, linkModel);
                }, { delay: 'immediate', infoString: 'HyperlinkCollection: generateMoveCellsOperations 2', app: app });
            });

            return promise;
        };

        /**
         * Generates the operations and undo operations to merge cells which
         * covers or overlaps hyperlink ranges
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address to
         *  generate the operations for.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMergeCellsOperations = function (generator, ranges) {

            var cellCollection = sheetModel.getCellCollection();

            return this.iterateArraySliced(RangeArray.get(ranges), function (range) {

                var determinationAddress = cellCollection.findFirstCell(range, { type: 'value', covered: true });
                var url = ''; // empty 'url' deletes hyperlinks

                if (determinationAddress && cellCollection.getEffectiveURL(determinationAddress)) {
                    url = cellCollection.getEffectiveURL(determinationAddress);
                } else {
                    var linkModel = findHyperlinks(new Range(range.start), { type: 'all' }).first();
                    if (linkModel) { url = linkModel.url; }
                }

                return self.generateHyperlinkOperations(generator, range, url);
            }, { delay: 'immediate', infoString: 'HyperlinkCollection: generateMergeCellsOperations', app: app });
        };

        // initialization -----------------------------------------------------

        // update hyperlinks after moving cells, or inserting/deleting columns or rows
        this.listenTo(sheetModel, 'move:cells', moveCellsHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = linkModels = app = null;
        });

    } // class HyperlinkCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: HyperlinkCollection });

});
