/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/parser/referencegrammar', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, SheetUtils, FormulaUtils) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var CellRef = FormulaUtils.CellRef;
    var SheetRef = FormulaUtils.SheetRef;

    // RE pattern for an absolute marker (an optional dollar sign) in A1 notation.
    var A1_ABS_PATTERN = '\\$?';

    // RE pattern for a simple column name in A1 notation.
    var A1_COL_PATTERN = '[a-z]+';

    // RE pattern for a simple row name in A1 notation.
    var A1_ROW_PATTERN = '\\d+';

    // RE capturing group for an absolute marker (an optional dollar sign) in A1 notation.
    // - Group 1: The dollar sign, if existing, otherwise an empty match.
    var A1_ABS_GROUP = '(' + A1_ABS_PATTERN + ')';

    // RE capturing group for a simple column name in A1 notation.
    // - Group 1: The column name.
    var A1_COL_GROUP = '(' + A1_COL_PATTERN + ')';

    // RE capturing group for a simple row name in A1 notation.
    // - Group 1: The row name.
    var A1_ROW_GROUP = '(' + A1_ROW_PATTERN + ')';

    // RE capturing group for an absolute index in R1C1 notation, e.g. the numbers in R1C1.
    // - Group 1: The absolute index.
    var RC_ABS_INDEX_GROUP = '(\\d+)';

    // RE capturing group for a relative index in R1C1 notation, e.g. the numbers in R[-1]C[1].
    // Matches the empty brackets, as in R[]C[1], and the empty string, as in RC[1].
    // - Group 1: The relative index without brackets (may be the empty string).
    var RC_REL_INDEX_GROUP = '(?:\\[(-?\\d+)?\\])?';

    // RE capturing groups for an index in R1C1 notation, e.g. the numbers in R1C1 or R[-1]C[1].
    // - Group 1: The absolute index (empty for a relative index).
    // - Group 2: The relative index, without brackets.
    var RC_INDEX_GROUPS = '(?:' + RC_ABS_INDEX_GROUP + '|' + RC_REL_INDEX_GROUP + ')';

    // character ranges (without brackets) for the leading character of a name or identifier
    var NAME_LEADING_CHARS = 'A-Z_a-z\\xA1\\xA4\\xA7-\\xA8\\xAA\\xAF-\\xBA\\xBC-\\u02B8\\u02BB-\\u02C1\\u02C7\\u02C9-\\u02CB\\u02CD\\u02D0-\\u02D1\\u02D8-\\u02DB\\u02DD\\u02E0-\\u02E4\\u02EE\\u0370-\\u0373\\u0376-\\u0377\\u037A-\\u037D\\u0386\\u0388-\\u038A\\u038C\\u038E-\\u03A1\\u03A3-\\u03F5\\u03F7-\\u0481\\u048A-\\u0523\\u0531-\\u0556\\u0559\\u0561-\\u0587\\u05D0-\\u05EA\\u05F0-\\u05F2\\u0621-\\u063F\\u0641-\\u064A\\u066E-\\u066F\\u0671-\\u06D3\\u06D5\\u06E5-\\u06E6\\u06EE-\\u06EF\\u06FA-\\u06FC\\u06FF\\u0710\\u0712-\\u072F\\u074D-\\u07A5\\u07B1\\u07CA-\\u07EA\\u07F4-\\u07F5\\u07FA\\u0904-\\u0939\\u093D\\u0950\\u0958-\\u0961\\u0971-\\u0972\\u097B-\\u097F\\u0985-\\u098C\\u098F-\\u0990\\u0993-\\u09A8\\u09AA-\\u09B0\\u09B2\\u09B6-\\u09B9\\u09BD\\u09CE\\u09DC-\\u09DD\\u09DF-\\u09E1\\u09F0-\\u09F1\\u0A05-\\u0A0A\\u0A0F-\\u0A10\\u0A13-\\u0A28\\u0A2A-\\u0A30\\u0A32-\\u0A33\\u0A35-\\u0A36\\u0A38-\\u0A39\\u0A59-\\u0A5C\\u0A5E\\u0A72-\\u0A74\\u0A85-\\u0A8D\\u0A8F-\\u0A91\\u0A93-\\u0AA8\\u0AAA-\\u0AB0\\u0AB2-\\u0AB3\\u0AB5-\\u0AB9\\u0ABD\\u0AD0\\u0AE0-\\u0AE1\\u0B05-\\u0B0C\\u0B0F-\\u0B10\\u0B13-\\u0B28\\u0B2A-\\u0B30\\u0B32-\\u0B33\\u0B35-\\u0B39\\u0B3D\\u0B5C-\\u0B5D\\u0B5F-\\u0B61\\u0B71\\u0B83\\u0B85-\\u0B8A\\u0B8E-\\u0B90\\u0B92-\\u0B95\\u0B99-\\u0B9A\\u0B9C\\u0B9E-\\u0B9F\\u0BA3-\\u0BA4\\u0BA8-\\u0BAA\\u0BAE-\\u0BB9\\u0BD0\\u0C05-\\u0C0C\\u0C0E-\\u0C10\\u0C12-\\u0C28\\u0C2A-\\u0C33\\u0C35-\\u0C39\\u0C3D\\u0C58-\\u0C59\\u0C60-\\u0C61\\u0C85-\\u0C8C\\u0C8E-\\u0C90\\u0C92-\\u0CA8\\u0CAA-\\u0CB3\\u0CB5-\\u0CB9\\u0CBD\\u0CDE\\u0CE0-\\u0CE1\\u0D05-\\u0D0C\\u0D0E-\\u0D10\\u0D12-\\u0D28\\u0D2A-\\u0D39\\u0D3D\\u0D60-\\u0D61\\u0D7A-\\u0D7F\\u0D85-\\u0D96\\u0D9A-\\u0DB1\\u0DB3-\\u0DBB\\u0DBD\\u0DC0-\\u0DC6\\u0E01-\\u0E3A\\u0E40-\\u0E4E\\u0E81-\\u0E82\\u0E84\\u0E87-\\u0E88\\u0E8A\\u0E8D\\u0E94-\\u0E97\\u0E99-\\u0E9F\\u0EA1-\\u0EA3\\u0EA5\\u0EA7\\u0EAA-\\u0EAB\\u0EAD-\\u0EB0\\u0EB2-\\u0EB3\\u0EBD\\u0EC0-\\u0EC4\\u0EC6\\u0EDC-\\u0EDD\\u0F00\\u0F40-\\u0F47\\u0F49-\\u0F6C\\u0F88-\\u0F8B\\u1000-\\u102A\\u103F\\u1050-\\u1055\\u105A-\\u105D\\u1061\\u1065-\\u1066\\u106E-\\u1070\\u1075-\\u1081\\u108E\\u10A0-\\u10C5\\u10D0-\\u10FA\\u10FC\\u1100-\\u1159\\u115F-\\u11A2\\u11A8-\\u11F9\\u1200-\\u1248\\u124A-\\u124D\\u1250-\\u1256\\u1258\\u125A-\\u125D\\u1260-\\u1288\\u128A-\\u128D\\u1290-\\u12B0\\u12B2-\\u12B5\\u12B8-\\u12BE\\u12C0\\u12C2-\\u12C5\\u12C8-\\u12D6\\u12D8-\\u1310\\u1312-\\u1315\\u1318-\\u135A\\u1380-\\u138F\\u13A0-\\u13F4\\u1401-\\u166C\\u166F-\\u1676\\u1681-\\u169A\\u16A0-\\u16EA\\u16EE-\\u16F0\\u1700-\\u170C\\u170E-\\u1711\\u1720-\\u1731\\u1740-\\u1751\\u1760-\\u176C\\u176E-\\u1770\\u1780-\\u17B3\\u17D7\\u17DC\\u1820-\\u1877\\u1880-\\u18A8\\u18AA\\u1900-\\u191C\\u1950-\\u196D\\u1970-\\u1974\\u1980-\\u19A9\\u19C1-\\u19C7\\u1A00-\\u1A16\\u1B05-\\u1B33\\u1B45-\\u1B4B\\u1B83-\\u1BA0\\u1BAE-\\u1BAF\\u1C00-\\u1C23\\u1C4D-\\u1C4F\\u1C5A-\\u1C7D\\u1D00-\\u1DBF\\u1E00-\\u1F15\\u1F18-\\u1F1D\\u1F20-\\u1F45\\u1F48-\\u1F4D\\u1F50-\\u1F57\\u1F59\\u1F5B\\u1F5D\\u1F5F-\\u1F7D\\u1F80-\\u1FB4\\u1FB6-\\u1FBC\\u1FBE\\u1FC2-\\u1FC4\\u1FC6-\\u1FCC\\u1FD0-\\u1FD3\\u1FD6-\\u1FDB\\u1FE0-\\u1FEC\\u1FF2-\\u1FF4\\u1FF6-\\u1FFC\\u2010\\u2013-\\u2016\\u2018\\u201C-\\u201D\\u2020-\\u2021\\u2025-\\u2027\\u2030\\u2032-\\u2033\\u2035\\u203B\\u2071\\u2074\\u207F\\u2081-\\u2084\\u2090-\\u2094\\u2102-\\u2103\\u2105\\u2107\\u2109-\\u2113\\u2115-\\u2116\\u2119-\\u211D\\u2121-\\u2122\\u2124\\u2126\\u2128\\u212A-\\u212D\\u212F-\\u2139\\u213C-\\u213F\\u2145-\\u2149\\u214E\\u2153-\\u2154\\u215B-\\u215E\\u2160-\\u2188\\u2190-\\u2199\\u21D2\\u21D4\\u2200\\u2202-\\u2203\\u2207-\\u2208\\u220B\\u220F\\u2211\\u2215\\u221A\\u221D-\\u2220\\u2223\\u2225\\u2227-\\u222C\\u222E\\u2234-\\u2237\\u223C-\\u223D\\u2248\\u224C\\u2252\\u2260-\\u2261\\u2264-\\u2267\\u226A-\\u226B\\u226E-\\u226F\\u2282-\\u2283\\u2286-\\u2287\\u2295\\u2299\\u22A5\\u22BF\\u2312\\u2460-\\u24B5\\u24D0-\\u24E9\\u2500-\\u254B\\u2550-\\u2574\\u2581-\\u258F\\u2592-\\u2595\\u25A0-\\u25A1\\u25A3-\\u25A9\\u25B2-\\u25B3\\u25B6-\\u25B7\\u25BC-\\u25BD\\u25C0-\\u25C1\\u25C6-\\u25C8\\u25CB\\u25CE-\\u25D1\\u25E2-\\u25E5\\u25EF\\u2605-\\u2606\\u2609\\u260E-\\u260F\\u261C\\u261E\\u2640\\u2642\\u2660-\\u2661\\u2663-\\u2665\\u2667-\\u266A\\u266C-\\u266D\\u266F\\u2C00-\\u2C2E\\u2C30-\\u2C5E\\u2C60-\\u2C6F\\u2C71-\\u2C7D\\u2C80-\\u2CE4\\u2D00-\\u2D25\\u2D30-\\u2D65\\u2D6F\\u2D80-\\u2D96\\u2DA0-\\u2DA6\\u2DA8-\\u2DAE\\u2DB0-\\u2DB6\\u2DB8-\\u2DBE\\u2DC0-\\u2DC6\\u2DC8-\\u2DCE\\u2DD0-\\u2DD6\\u2DD8-\\u2DDE\\u3000-\\u3003\\u3005-\\u3017\\u301D-\\u301F\\u3021-\\u3029\\u3031-\\u3035\\u3038-\\u303C\\u3041-\\u3096\\u309B-\\u309F\\u30A1-\\u30FF\\u3105-\\u312D\\u3131-\\u318E\\u31A0-\\u31B7\\u31F0-\\u321C\\u3220-\\u3229\\u3231-\\u3232\\u3239\\u3260-\\u327B\\u327F\\u32A3-\\u32A8\\u3303\\u330D\\u3314\\u3318\\u3322-\\u3323\\u3326-\\u3327\\u332B\\u3336\\u333B\\u3349-\\u334A\\u334D\\u3351\\u3357\\u337B-\\u337E\\u3380-\\u3384\\u3388-\\u33CA\\u33CD-\\u33D3\\u33D5-\\u33D6\\u33D8\\u33DB-\\u33DD\\u3400-\\u4DB5\\u4E00-\\u9FC3\\uA000-\\uA48C\\uA500-\\uA60C\\uA610-\\uA61F\\uA62A-\\uA62B\\uA640-\\uA65F\\uA662-\\uA66E\\uA680-\\uA697\\uA722-\\uA787\\uA78B-\\uA78C\\uA7FB-\\uA801\\uA803-\\uA805\\uA807-\\uA80A\\uA80C-\\uA822\\uA840-\\uA873\\uA882-\\uA8B3\\uA90A-\\uA925\\uA930-\\uA946\\uAA00-\\uAA28\\uAA40-\\uAA42\\uAA44-\\uAA4B\\uAC00-\\uD7A3\\uE000-\\uF848\\uF900-\\uFA2D\\uFA30-\\uFA6A\\uFA70-\\uFAD9\\uFB00-\\uFB06\\uFB13-\\uFB17\\uFB1D\\uFB1F-\\uFB28\\uFB2A-\\uFB36\\uFB38-\\uFB3C\\uFB3E\\uFB40-\\uFB41\\uFB43-\\uFB44\\uFB46-\\uFBB1\\uFBD3-\\uFD3D\\uFD50-\\uFD8F\\uFD92-\\uFDC7\\uFDF0-\\uFDFB\\uFE30-\\uFE31\\uFE33-\\uFE44\\uFE49-\\uFE52\\uFE54-\\uFE57\\uFE59-\\uFE66\\uFE68-\\uFE6B\\uFE70-\\uFE74\\uFE76-\\uFEFC\\uFF01-\\uFF5E\\uFF61-\\uFFBE\\uFFC2-\\uFFC7\\uFFCA-\\uFFCF\\uFFD2-\\uFFD7\\uFFDA-\\uFFDC\\uFFE0-\\uFFE6';

    // character ranges for the inner character of a name or identifier
    var NAME_INNER_CHARS = '\\d\\?A-Z_a-z\\xA1\\xA4\\xA7-\\xA8\\xAA\\xAF-\\xBA\\xBC-\\u034E\\u0350-\\u0377\\u037A-\\u037D\\u0384-\\u0386\\u0388-\\u038A\\u038C\\u038E-\\u03A1\\u03A3-\\u0523\\u0531-\\u0556\\u0559\\u0561-\\u0587\\u0591-\\u05BD\\u05BF\\u05C1-\\u05C2\\u05C4-\\u05C5\\u05C7\\u05D0-\\u05EA\\u05F0-\\u05F2\\u0600-\\u0603\\u0606-\\u0608\\u060B\\u060E-\\u061A\\u061F\\u0621-\\u063F\\u0641-\\u065E\\u0660-\\u0669\\u066E-\\u06D3\\u06D5-\\u06FF\\u070F-\\u074A\\u074D-\\u07B1\\u07C0-\\u07F6\\u07FA\\u0901-\\u0939\\u093C-\\u094D\\u0950-\\u0954\\u0958-\\u0963\\u0966-\\u096F\\u0971-\\u0972\\u097B-\\u097F\\u0981-\\u0983\\u0985-\\u098C\\u098F-\\u0990\\u0993-\\u09A8\\u09AA-\\u09B0\\u09B2\\u09B6-\\u09B9\\u09BC-\\u09C4\\u09C7-\\u09C8\\u09CB-\\u09CE\\u09D7\\u09DC-\\u09DD\\u09DF-\\u09E3\\u09E6-\\u09FA\\u0A01-\\u0A03\\u0A05-\\u0A0A\\u0A0F-\\u0A10\\u0A13-\\u0A28\\u0A2A-\\u0A30\\u0A32-\\u0A33\\u0A35-\\u0A36\\u0A38-\\u0A39\\u0A3C\\u0A3E-\\u0A42\\u0A47-\\u0A48\\u0A4B-\\u0A4D\\u0A51\\u0A59-\\u0A5C\\u0A5E\\u0A66-\\u0A75\\u0A81-\\u0A83\\u0A85-\\u0A8D\\u0A8F-\\u0A91\\u0A93-\\u0AA8\\u0AAA-\\u0AB0\\u0AB2-\\u0AB3\\u0AB5-\\u0AB9\\u0ABC-\\u0AC5\\u0AC7-\\u0AC9\\u0ACB-\\u0ACD\\u0AD0\\u0AE0-\\u0AE3\\u0AE6-\\u0AEF\\u0AF1\\u0B01-\\u0B03\\u0B05-\\u0B0C\\u0B0F-\\u0B10\\u0B13-\\u0B28\\u0B2A-\\u0B30\\u0B32-\\u0B33\\u0B35-\\u0B39\\u0B3C-\\u0B44\\u0B47-\\u0B48\\u0B4B-\\u0B4D\\u0B56-\\u0B57\\u0B5C-\\u0B5D\\u0B5F-\\u0B63\\u0B66-\\u0B71\\u0B82-\\u0B83\\u0B85-\\u0B8A\\u0B8E-\\u0B90\\u0B92-\\u0B95\\u0B99-\\u0B9A\\u0B9C\\u0B9E-\\u0B9F\\u0BA3-\\u0BA4\\u0BA8-\\u0BAA\\u0BAE-\\u0BB9\\u0BBE-\\u0BC2\\u0BC6-\\u0BC8\\u0BCA-\\u0BCD\\u0BD0\\u0BD7\\u0BE6-\\u0BFA\\u0C01-\\u0C03\\u0C05-\\u0C0C\\u0C0E-\\u0C10\\u0C12-\\u0C28\\u0C2A-\\u0C33\\u0C35-\\u0C39\\u0C3D-\\u0C44\\u0C46-\\u0C48\\u0C4A-\\u0C4D\\u0C55-\\u0C56\\u0C58-\\u0C59\\u0C60-\\u0C63\\u0C66-\\u0C6F\\u0C78-\\u0C7F\\u0C82-\\u0C83\\u0C85-\\u0C8C\\u0C8E-\\u0C90\\u0C92-\\u0CA8\\u0CAA-\\u0CB3\\u0CB5-\\u0CB9\\u0CBC-\\u0CC4\\u0CC6-\\u0CC8\\u0CCA-\\u0CCD\\u0CD5-\\u0CD6\\u0CDE\\u0CE0-\\u0CE3\\u0CE6-\\u0CEF\\u0CF1-\\u0CF2\\u0D02-\\u0D03\\u0D05-\\u0D0C\\u0D0E-\\u0D10\\u0D12-\\u0D28\\u0D2A-\\u0D39\\u0D3D-\\u0D44\\u0D46-\\u0D48\\u0D4A-\\u0D4D\\u0D57\\u0D60-\\u0D63\\u0D66-\\u0D75\\u0D79-\\u0D7F\\u0D82-\\u0D83\\u0D85-\\u0D96\\u0D9A-\\u0DB1\\u0DB3-\\u0DBB\\u0DBD\\u0DC0-\\u0DC6\\u0DCA\\u0DCF-\\u0DD4\\u0DD6\\u0DD8-\\u0DDF\\u0DF2-\\u0DF3\\u0E01-\\u0E3A\\u0E3F-\\u0E4E\\u0E50-\\u0E59\\u0E81-\\u0E82\\u0E84\\u0E87-\\u0E88\\u0E8A\\u0E8D\\u0E94-\\u0E97\\u0E99-\\u0E9F\\u0EA1-\\u0EA3\\u0EA5\\u0EA7\\u0EAA-\\u0EAB\\u0EAD-\\u0EB9\\u0EBB-\\u0EBD\\u0EC0-\\u0EC4\\u0EC6\\u0EC8-\\u0ECB\\u0ECD\\u0ED0-\\u0ED9\\u0EDC-\\u0EDD\\u0F00-\\u0F03\\u0F13-\\u0F39\\u0F3E-\\u0F47\\u0F49-\\u0F6C\\u0F71-\\u0F84\\u0F86-\\u0F8B\\u0F90-\\u0F97\\u0F99-\\u0FBC\\u0FBE-\\u0FCC\\u0FCE-\\u0FCF\\u1000-\\u1049\\u1050-\\u1099\\u109E-\\u10C5\\u10D0-\\u10FA\\u10FC\\u1100-\\u1159\\u115F-\\u11A2\\u11A8-\\u11F9\\u1200-\\u1248\\u124A-\\u124D\\u1250-\\u1256\\u1258\\u125A-\\u125D\\u1260-\\u1288\\u128A-\\u128D\\u1290-\\u12B0\\u12B2-\\u12B5\\u12B8-\\u12BE\\u12C0\\u12C2-\\u12C5\\u12C8-\\u12D6\\u12D8-\\u1310\\u1312-\\u1315\\u1318-\\u135A\\u135F-\\u1360\\u1369-\\u137C\\u1380-\\u1399\\u13A0-\\u13F4\\u1401-\\u166C\\u166F-\\u1676\\u1680-\\u169A\\u16A0-\\u16EA\\u16EE-\\u16F0\\u1700-\\u170C\\u170E-\\u1714\\u1720-\\u1734\\u1740-\\u1753\\u1760-\\u176C\\u176E-\\u1770\\u1772-\\u1773\\u1780-\\u17D3\\u17D7\\u17DB-\\u17DD\\u17E0-\\u17E9\\u17F0-\\u17F9\\u180E\\u1810-\\u1819\\u1820-\\u1877\\u1880-\\u18AA\\u1900-\\u191C\\u1920-\\u192B\\u1930-\\u193B\\u1940\\u1946-\\u196D\\u1970-\\u1974\\u1980-\\u19A9\\u19B0-\\u19C9\\u19D0-\\u19D9\\u19E0-\\u1A1B\\u1B00-\\u1B4B\\u1B50-\\u1B59\\u1B61-\\u1B7C\\u1B80-\\u1BAA\\u1BAE-\\u1BB9\\u1C00-\\u1C37\\u1C40-\\u1C49\\u1C4D-\\u1C7D\\u1D00-\\u1DE6\\u1DFE-\\u1F15\\u1F18-\\u1F1D\\u1F20-\\u1F45\\u1F48-\\u1F4D\\u1F50-\\u1F57\\u1F59\\u1F5B\\u1F5D\\u1F5F-\\u1F7D\\u1F80-\\u1FB4\\u1FB6-\\u1FC4\\u1FC6-\\u1FD3\\u1FD6-\\u1FDB\\u1FDD-\\u1FEF\\u1FF2-\\u1FF4\\u1FF6-\\u1FFE\\u2000-\\u200B\\u2010\\u2013-\\u2016\\u2018\\u201C-\\u201D\\u2020-\\u2021\\u2025-\\u2029\\u202F-\\u2030\\u2032-\\u2033\\u2035\\u203B\\u2044\\u2052\\u205F\\u2070-\\u2071\\u2074-\\u207C\\u207F-\\u208C\\u2090-\\u2094\\u20A0-\\u20B5\\u20D0-\\u20F0\\u2100-\\u214F\\u2153-\\u2188\\u2190-\\u2328\\u232B-\\u23E7\\u2400-\\u2426\\u2440-\\u244A\\u2460-\\u269D\\u26A0-\\u26BC\\u26C0-\\u26C3\\u2701-\\u2704\\u2706-\\u2709\\u270C-\\u2727\\u2729-\\u274B\\u274D\\u274F-\\u2752\\u2756\\u2758-\\u275E\\u2761-\\u2767\\u2776-\\u2794\\u2798-\\u27AF\\u27B1-\\u27BE\\u27C0-\\u27C4\\u27C7-\\u27CA\\u27CC\\u27D0-\\u27E5\\u27F0-\\u2982\\u2999-\\u29D7\\u29DC-\\u29FB\\u29FE-\\u2B4C\\u2B50-\\u2B54\\u2C00-\\u2C2E\\u2C30-\\u2C5E\\u2C60-\\u2C6F\\u2C71-\\u2C7D\\u2C80-\\u2CEA\\u2CFD\\u2D00-\\u2D25\\u2D30-\\u2D65\\u2D6F\\u2D80-\\u2D96\\u2DA0-\\u2DA6\\u2DA8-\\u2DAE\\u2DB0-\\u2DB6\\u2DB8-\\u2DBE\\u2DC0-\\u2DC6\\u2DC8-\\u2DCE\\u2DD0-\\u2DD6\\u2DD8-\\u2DDE\\u2DE0-\\u2DFF\\u2E2F\\u2E80-\\u2E99\\u2E9B-\\u2EF3\\u2F00-\\u2FD5\\u2FF0-\\u2FFB\\u3000-\\u3017\\u301D-\\u302F\\u3031-\\u303C\\u303E-\\u303F\\u3041-\\u3096\\u3099-\\u309F\\u30A1-\\u30FF\\u3105-\\u312D\\u3131-\\u318E\\u3192-\\u31B7\\u31C0-\\u31E3\\u31F0-\\u321E\\u3220-\\u3243\\u3250-\\u32FE\\u3300-\\u4DB5\\u4DC0-\\u9FC3\\uA000-\\uA48C\\uA490-\\uA4C6\\uA500-\\uA60C\\uA610-\\uA62B\\uA640-\\uA65F\\uA662-\\uA672\\uA67C-\\uA67D\\uA67F-\\uA697\\uA700-\\uA78C\\uA7FB-\\uA82B\\uA840-\\uA873\\uA880-\\uA8C4\\uA8D0-\\uA8D9\\uA900-\\uA92E\\uA930-\\uA953\\uAA00-\\uAA36\\uAA40-\\uAA4D\\uAA50-\\uAA59\\uAC00-\\uD7A3\\uD800-\\uD87F\\uDB80-\\uFA2D\\uFA30-\\uFA6A\\uFA70-\\uFAD9\\uFB00-\\uFB06\\uFB13-\\uFB17\\uFB1D-\\uFB36\\uFB38-\\uFB3C\\uFB3E\\uFB40-\\uFB41\\uFB43-\\uFB44\\uFB46-\\uFBB1\\uFBD3-\\uFD3D\\uFD50-\\uFD8F\\uFD92-\\uFDC7\\uFDF0-\\uFDFD\\uFE20-\\uFE26\\uFE30-\\uFE31\\uFE33-\\uFE44\\uFE49-\\uFE52\\uFE54-\\uFE57\\uFE59-\\uFE66\\uFE68-\\uFE6B\\uFE70-\\uFE74\\uFE76-\\uFEFC\\uFF01-\\uFF5E\\uFF61-\\uFFBE\\uFFC2-\\uFFC7\\uFFCA-\\uFFCF\\uFFD2-\\uFFD7\\uFFDA-\\uFFDC\\uFFE0-\\uFFE6\\uFFE8-\\uFFEE';

    // character ranges (without brackets) for a whitespace or other non-printable character
    var WHITESPACE_CHARS = '\\s\\x00-\\x1f\\x80-\\x9f';

    // class SheetConfigBase ==================================================

    /**
     * Base class for configurations of different sheet reference syntaxes used
     * in formula expressions.
     *
     * @constructor
     *
     * @property {String} REF_ERROR
     *  The actual string representation of the #REF! error code (may ba the
     *  translated error code).
     *
     * @property {Object<String, String>} PATTERNS
     *  A collection of basic patterns needed to build regular expressions used
     *  to parse sheet references in formula expressions.
     *
     * @property {Object<String, RegExp>} RE
     *  A collection of basic regular expressions needed to parse sheet
     *  references in formula expressions.
     */
    var SheetConfigBase = _.makeExtendable(function (refError, odf) {

        // the name of the #REF! error code
        this.REF_ERROR = refError;

        // basic RE patterns for sheet names
        var patterns = this.PATTERNS = {};

        // RE character class for the characters of a simple external file name (without apostrophes).
        patterns.FILE_SIMPLE_CHAR_CLASS = '[\\w.\\\\\\xa1-\\u2027\\u202a-\\uffff]';

        // RE character class for the leading character of a simple sheet name (without apostrophes).
        patterns.SHEET_LEADING_CHAR_CLASS = '[a-z_\\xa1-\\u2027\\u202a-\\uffff]';

        // RE character class for other characters of a simple sheet name (without apostrophes).
        patterns.SHEET_SIMPLE_CHAR_CLASS = '[\\w' + (odf ? '' : '.') + '\\xa1-\\u2027\\u202a-\\uffff]';

        // RE character class for any character of a complex sheet name (enclosed in apostrophes).
        patterns.SHEET_COMPLEX_CHAR_CLASS = '[^\\x00-\\x1f\\x80-\\x9f\\[\\]\'*?:/\\\\]';

        // RE character class for the leading character of a name or identifier (OOXML: allow leading backslash).
        patterns.NAME_LEADING_CHAR_CLASS = '[' + (odf ? '' : '\\\\') + NAME_LEADING_CHARS + ']';

        // RE character class for other characters of a name or identifier (OOXML: allow dots and backslash).
        patterns.NAME_INNER_CHAR_CLASS = '[' + (odf ? '' : '.\\\\') + NAME_INNER_CHARS + ']';

        // RE character class for other characters of a function name (ODF: allow periods in difference to defined names).
        patterns.FUNC_INNER_CHAR_CLASS = '[' + (odf ? '.' : '.\\\\') + NAME_INNER_CHARS + ']';

        // RE pattern for the #REF! error code.
        // - Group 1: The name of the #REF! error code.
        patterns.REF_ERROR = '(' + _.escapeRegExp(refError) + ')';

        // RE look-ahead pattern to exclude an opening parenthesis and an exclamation mark
        // after another pattern. All valid inner characters of names must be excluded too,
        // otherwise the previous groups would match less characters than they should.
        patterns.TERMINATE_REF = '(?!\\(|!|' + patterns.NAME_INNER_CHAR_CLASS + ')';

        // RE pattern for a simple external file name (enclosed in brackets, without apostrophes).
        // - Group 1: The name of the external file (the part inside the brackets).
        patterns.FILE_SIMPLE_NAME = '(?:\\[(' + patterns.FILE_SIMPLE_CHAR_CLASS + '+)\\])';

        // RE pattern for a simple sheet name.
        // - Group 1: The sheet name.
        patterns.SIMPLE_NAME = '(' + patterns.SHEET_LEADING_CHAR_CLASS + patterns.SHEET_SIMPLE_CHAR_CLASS + '*)';

        // RE pattern for a complex sheet name (with embedded double-apostrophes, without surrounding apostrophes).
        // - Group 1: The sheet name.
        patterns.COMPLEX_NAME = '(' + patterns.SHEET_COMPLEX_CHAR_CLASS + '+(?:(?:\'\')+' + patterns.SHEET_COMPLEX_CHAR_CLASS + '+)*)';

        // RE pattern for a complex sheet name (with embedded double-apostrophes, with surrounding apostrophes).
        // - Group 1: The sheet name (without the surrounding apostrophes).
        patterns.COMPLEX_NAME_QUOTED = '\'' + patterns.COMPLEX_NAME + '\'';

        // RE pattern for a single sheet name (without address separator), e.g. Sheet1 in the formula =Sheet1!A1,
        // or Sheet#2 in the formula ='Sheet#2'!A1.
        // - Group 1: The name of the sheet (simple sheet name).
        // - Group 2: The name of the sheet (complex sheet name without surrounding apostrophes).
        // Either group 1 or group 2 will match the sheet name, the other group will be empty.
        patterns.SINGLE_SHEET_NAME = '(?:' + patterns.SIMPLE_NAME + '|' + patterns.COMPLEX_NAME_QUOTED + ')';

        // RE pattern for a single sheet name (without address separator), e.g. Sheet1 in the formula =Sheet1!A1,
        // or Sheet#2 in the formula ='Sheet#2'!A1; or a broken sheet reference after deleting the referenced sheet,
        // e.g. #REF! in the formula =#REF!!A1.
        // - Group 1: The name of the sheet (simple sheet name).
        // - Group 2: The name of the sheet (complex sheet name without surrounding apostrophes).
        // - Group 3: The #REF! error code as sheet name.
        // Either group 1 or group 2 will match the sheet name, or group 3 will contain the error code, the other
        // groups will be empty.
        patterns.SINGLE_SHEET_NAME_OR_ERR = '(?:' + patterns.SINGLE_SHEET_NAME + '|' + patterns.REF_ERROR + ')';

        // RE pattern for a complete name or identifier.
        // - Group 1: The entire identifier.
        patterns.DEFINED_NAME = '(' + patterns.NAME_LEADING_CHAR_CLASS + patterns.NAME_INNER_CHAR_CLASS + '*)';

        // RE pattern for a complete function name.
        // - Group 1: The entire function name.
        patterns.FUNC_NAME = '(' + patterns.NAME_LEADING_CHAR_CLASS + patterns.FUNC_INNER_CHAR_CLASS + '*)';

        // basic regular expressions for sheet names
        var re = this.RE = {};

        // Matches a simple sheet name, or a range of simple sheet names, that do not need to be enclosed in apostrophes.
        // Supports an optional leading reference to an external file (also simple name only), enclosed in brackets.
        // Examples: 'Sheet1', 'Sheet1:Sheet2', '[file1.xlsx]Sheet1', '[path\to\file1.xlsx]Sheet1:Sheet2'.
        // - Group 1: The name of the external file (the part inside the brackets), may be undefined.
        // - Group 2: The first sheet name.
        // - Group 3: The second sheet name, may be undefined.
        re.SIMPLE_EXT_SHEET_NAME = new RegExp('^' + patterns.FILE_SIMPLE_NAME + '?' + patterns.SIMPLE_NAME + '(?::' + patterns.SIMPLE_NAME + ')?$', 'i');

    }); // class SheetConfigBase

    // class SheetConfigOOX ===================================================

    /**
     * Configuration settings and helper methods for sheet references in OOXML
     * based grammars.
     *
     * @constructor
     *
     * @extends SheetConfigBase
     */
    var SheetConfigOOX = SheetConfigBase.extend({ constructor: function (refError) {

        // base constructor
        SheetConfigBase.call(this, refError, false);

        // add OOMXL specific RE patterns
        var patterns = this.PATTERNS;

        // RE pattern for a single sheet name (without address separator), e.g. Sheet1 in the formula =Sheet1!A1.
        // - Group 1: The name of the sheet (simple sheet name).
        // - Group 2: The name of the sheet (complex sheet name).
        // Either group 1 or group 2 will match the sheet name, the other group will be empty.
        patterns.SHEET_REF = patterns.SINGLE_SHEET_NAME;

        // RE pattern for a single sheet name (without address separator), e.g. Sheet1 in the formula =Sheet1!A1,
        // or a broken sheet reference after deleting the referenced sheet, e.g. #REF! in the formula =#REF!!A1.
        // - Group 1: The name of the sheet (simple sheet name).
        // - Group 2: The name of the sheet (complex sheet name).
        // - Group 3: The #REF! error code as sheet name.
        // Either group 1 or group 2 will match the sheet name, or group 3 will contain the error code, the other
        // groups will be empty.
        patterns.SHEET_OR_ERR_REF = patterns.SINGLE_SHEET_NAME_OR_ERR;

        // RE pattern for a range of sheet names (without address separator), e.g. Sheet1:Sheet2 in the formula
        // =SUM(Sheet1:Sheet2!A1).
        // - Group 1: The name of the first sheet (simple sheet names).
        // - Group 2: The name of the second sheet (simple sheet names).
        // - Group 3: The name of the first sheet (complex sheet names).
        // - Group 4: The name of the second sheet (complex sheet names).
        // Either group 1 or group 3 will match the first sheet name, the other group will be empty.
        // Either group 2 or group 4 will match the second sheet name, the other group will be empty.
        patterns.SHEET_RANGE_REF = '(?:' + patterns.SIMPLE_NAME + ':' + patterns.SIMPLE_NAME + '|\'' + patterns.COMPLEX_NAME + ':' + patterns.COMPLEX_NAME + '\')';

    } }); // class SheetConfigOOX

    // static methods ---------------------------------------------------------

    SheetConfigOOX.create = _.memoize(function (refError) {
        return new SheetConfigOOX(refError);
    });

    // public methods ---------------------------------------------------------

    /**
     * Returns a sheet reference structure for the passed matches of a regular
     * expression.
     *
     * @param {SpreadsheetModel} docModel
     *  The model of the spreadsheet document used to resolve the sheet names.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the sheet reference in the passed matches.
     *
     * @returns {SheetRef}
     *  A sheet reference structure for the passed matches.
     */
    SheetConfigOOX.prototype.createSheetRef = function (docModel, matches, index) {
        return SheetRef.create(docModel, matches[index], matches[index + 1], matches[index + 2], true);
    };

    // class SheetConfigODF ===================================================

    /**
     * Configuration settings and helper methods for sheet references in ODF
     * based grammars.
     *
     * @constructor
     *
     * @extends SheetConfigBase
     */
    var SheetConfigODF = SheetConfigBase.extend({ constructor: function (refError) {

        // base constructor
        SheetConfigBase.call(this, refError, true);

        // add ODF specific RE patterns
        var patterns = this.PATTERNS;

        // RE pattern for a single sheet name (with optional absolute marker, without address separator), e.g.
        // $Sheet1 in the formula =$Sheet1!A1.
        // - Group 1: The absolute marker.
        // - Group 2: The name of the sheet (simple sheet name).
        // - Group 3: The name of the sheet (complex sheet name).
        // Either group 2 or group 3 will match the sheet name, the other group will be empty.
        patterns.SHEET_REF = A1_ABS_GROUP + patterns.SINGLE_SHEET_NAME;

        // RE pattern for a single sheet name (with optional absolute marker, without address separator), e.g.
        // $Sheet1 in the formula =$Sheet1!A1, or a broken sheet reference after deleting the referenced sheet,
        // e.g. $#REF! in the formula =$#REF!!A1.
        // - Group 1: The absolute marker.
        // - Group 2: The name of the sheet (simple sheet name).
        // - Group 3: The name of the sheet (complex sheet name).
        // - Group 4: The #REF! error code as sheet name.
        // Either group 2 or group 3 will match the sheet name, or group 4 will contain the error code, the other
        // groups will be empty.
        patterns.SHEET_OR_ERR_REF = A1_ABS_GROUP + patterns.SINGLE_SHEET_NAME_OR_ERR;

        // RE pattern for a range of sheet names (with optional absolute markers, without address separator), e.g.
        // Sheet1:$Sheet2 in the formula =SUM(Sheet1:$Sheet2!A1).
        // - Group 1: The absolute marker of the first sheet.
        // - Group 2: The name of the first sheet (simple sheet name).
        // - Group 3: The name of the first sheet (complex sheet name).
        // - Group 4: The absolute marker of the second sheet.
        // - Group 5: The name of the second sheet (simple sheet name).
        // - Group 6: The name of the second sheet (complex sheet name).
        // Either group 2 or group 3 will match the first sheet name, the other group will be empty.
        // Either group 5 or group 6 will match the second sheet name, the other group will be empty.
        patterns.SHEET_RANGE_REF = patterns.SHEET_REF + ':' + patterns.SHEET_REF;

    } }); // class SheetConfigODF

    // static methods ---------------------------------------------------------

    SheetConfigODF.create = _.memoize(function (refError) {
        return new SheetConfigODF(refError);
    });

    // public methods ---------------------------------------------------------

    /**
     * Returns a sheet reference structure for the passed matches of a regular
     * expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the sheet reference in the passed matches.
     *
     * @returns {SheetRef}
     *  A sheet reference structure for the passed matches.
     */
    SheetConfigODF.prototype.createSheetRef = function (docModel, matches, index) {
        return SheetRef.create(docModel, matches[index + 1], matches[index + 2], matches[index + 3], !!matches[index]);
    };

    // class AddressConfigBase ================================================

    /**
     * Base class for configurations of different address syntaxes used  in
     * formula expressions.
     *
     * @constructor
     *
     * @property {Object<String, String>} PATTERNS
     *  A collection of basic patterns needed to build regular expressions used
     *  to parse references in formula expressions.
     */
    var AddressConfigBase = _.makeExtendable(function () {

        // the RE patterns for the address references
        this.PATTERNS = {};

    }); // class AddressConfigBase

    // static methods ---------------------------------------------------------

    /**
     * Creates a hash key for a specific address reference syntax.
     */
    AddressConfigBase.getCacheKey = function (prefixChars) {
        return prefixChars ? ('rc=' + prefixChars) : 'a1';
    };

    /**
     * Returns an existing address configuration singleton from the internal
     * cache, if available; otherwise creates a new instance.
     *
     * @param {String|Null} prefixChars
     *  If set to a string, R1C1 notation will be used by the new configuration
     *  instance. Otherwise, A1 notation will be used.
     *
     * @returns {AddressConfigBase}
     *  A configuration singleton for the passed parameters.
     */
    AddressConfigBase.create = _.memoize(function (prefixChars) {
        return prefixChars ? new AddressConfigRC(prefixChars) : new AddressConfigA1();
    }, AddressConfigBase.getCacheKey);

    // class AddressConfigA1 ==================================================

    /**
     * Configuration settings and helper methods for cell references in A1
     * notation.
     *
     * @constructor
     *
     * @extends AddressConfigBase
     */
    var AddressConfigA1 = AddressConfigBase.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        AddressConfigBase.call(this);

        // initialization -----------------------------------------------------

        // create the matching patterns for A1 notation
        var patterns = this.PATTERNS;

        // RE pattern for a single column reference (absolute or relative), e.g. $C.
        // - Group 1: The absolute marker.
        // - Group 2: The column name.
        var COL_PATTERN = A1_ABS_GROUP + A1_COL_GROUP;

        // RE pattern for a single row reference (absolute or relative), e.g. $1.
        // - Group 1: The absolute marker.
        // - Group 2: The row name.
        var ROW_PATTERN = A1_ABS_GROUP + A1_ROW_GROUP;

        // RE pattern for a single cell reference (absolute or relative), e.g. $C$1.
        // - Group 1: The absolute marker of the column.
        // - Group 2: The column name.
        // - Group 3: The absolute marker of the row.
        // - Group 4: The row name.
        patterns.CELL_REF = COL_PATTERN + ROW_PATTERN;

        // RE pattern for a cell range reference (absolute or relative), e.g. $C$1:$D$2.
        // - Groups 1 to 4: The components of the first cell reference.
        // - Groups 5 to 8: The components of the second cell reference.
        patterns.RANGE_REF = patterns.CELL_REF + ':' + patterns.CELL_REF;

        // RE pattern for a column range reference (absolute or relative), e.g. $C:$D.
        // - Groups 1 to 2: The components of the first column.
        // - Groups 3 to 4: The components of the second column.
        patterns.COLS_REF = COL_PATTERN + ':' + COL_PATTERN;

        // RE pattern for a row range reference (absolute or relative), e.g. $1:$2.
        // - Groups 1 to 2: The components of the first row.
        // - Groups 3 to 4: The components of the second row.
        patterns.ROWS_REF = ROW_PATTERN + ':' + ROW_PATTERN;

        // RE pattern for a single cell reference without absolute markers, e.g. C1.
        // - Group 1: The column name.
        // - Group 2: The row name.
        patterns.RL_CELL_REF = A1_COL_GROUP + A1_ROW_GROUP;

        // RE pattern for a single cell range reference without absolute markers, e.g. C1:D2.
        // - Group 1: The first column name.
        // - Group 2: The first row name.
        // - Group 3: The second column name.
        // - Group 4: The second row name.
        patterns.RL_RANGE_REF = patterns.RL_CELL_REF + ':' + patterns.RL_CELL_REF;

        // RE pattern for a column range reference without absolute markers, e.g. C:D.
        // - Group 1: The name of the first column.
        // - Group 2: The name of the second column.
        patterns.RL_COLS_REF = A1_COL_GROUP + ':' + A1_COL_GROUP;

        // RE pattern for a row range reference without absolute markers, e.g. 1:2.
        // - Group 1: The name of the first row.
        // - Group 2: The name of the second row.
        patterns.RL_ROWS_REF = A1_ROW_GROUP + ':' + A1_ROW_GROUP;

    } }); // class AddressConfigA1

    // public methods ---------------------------------------------------------

    /**
     * Extracts a column index from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The array index of the column index in the passed matches.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @returns {Number}
     *  The column for the passed matches, if it is valid; otherwise -1.
     */
    AddressConfigA1.prototype.getCol = function (matches, index, maxCol) {
        var col = Address.parseCol(matches[index]);
        return (col <= maxCol) ? col : -1;
    };

    /**
     * Extracts a row index from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The array index of the row index in the passed matches.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @returns {Number}
     *  The row for the passed matches, if it is valid; otherwise -1.
     */
    AddressConfigA1.prototype.getRow = function (matches, index, maxRow) {
        var row = Address.parseRow(matches[index]);
        return (row <= maxRow) ? row : -1;
    };

    /**
     * Extracts a cell address from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the cell address in the passed matches. The
     *  matches are expected to contain two elements for the column name, and
     *  the row name, in this order.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @returns {Address|Null}
     *  A cell address for the passed matches, if the column and row indexes
     *  are valid; otherwise null.
     */
    AddressConfigA1.prototype.getAddress = function (matches, index, maxCol, maxRow) {
        var col = this.getCol(matches, index, maxCol);
        var row = this.getRow(matches, index + 1, maxRow);
        return ((col >= 0) && (row >= 0)) ? new Address(col, row) : null;
    };

    /**
     * Returns a cell reference structure for the passed matches of a regular
     * expression in A1 notation.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the cell address in the passed matches. The
     *  matches are expected to contain four elements for the absolute column
     *  marker, the column index, the absolute row marker, and the row index,
     *  in this order.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the column and
     *  row indexes are valid; otherwise null.
     */
    AddressConfigA1.prototype.createCellRef = function (matches, index, maxCol, maxRow) {
        var cellRef = new CellRef(0, 0, true, true);
        if (!cellRef.parseCol(matches[index + 1], matches[index], maxCol)) { return null; }
        if (!cellRef.parseRow(matches[index + 3], matches[index + 2], maxRow)) { return null; }
        return cellRef;
    };

    /**
     * Returns a cell reference structure for a column from the passed matches
     * of a regular expression in A1 notation.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the column index in the passed matches. The
     *  matches are expected to contain two elements for the absolute column
     *  marker, and the column index, in this order.
     *
     * @param {Number} fixedRow
     *  The fixed row index to be inserted into the cell reference.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the column index
     *  is valid; otherwise null.
     */
    AddressConfigA1.prototype.createColRef = function (matches, index, fixedRow, maxCol) {
        var cellRef = new CellRef(0, fixedRow, true, true);
        return cellRef.parseCol(matches[index + 1], matches[index], maxCol) ? cellRef : null;
    };

    /**
     * Returns a cell reference structure for a row from the passed matches of
     * a regular expression in A1 notation.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the row index in the passed matches. The
     *  matches are expected to contain two elements for the absolute row
     *  marker, and the row index, in this order.
     *
     * @param {Number} fixedCol
     *  The fixed column index to be inserted into the cell reference.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the row index is
     *  valid; otherwise null.
     */
    AddressConfigA1.prototype.createRowRef = function (matches, index, fixedCol, maxRow) {
        var cellRef = new CellRef(fixedCol, 0, true, true);
        return cellRef.parseRow(matches[index + 1], matches[index], maxRow) ? cellRef : null;
    };

    // class AddressConfigRC ==================================================

    /**
     * Configuration settings and helper methods for cell references in R1C1
     * notation.
     *
     * @constructor
     *
     * @extends AddressConfigBase
     *
     * @param {String} prefixChars
     *  The prefix characters for row and column references.
     */
    var AddressConfigRC = AddressConfigBase.extend({ constructor: function (prefixChars) {

        // base constructor ---------------------------------------------------

        AddressConfigBase.call(this);

        // initialization -----------------------------------------------------

        // create the matching patterns for R1C1 notation
        var patterns = this.PATTERNS;

        // RE pattern for a single column reference (absolute or relative), e.g. C[1].
        // - Groups 1: The absolute column index (empty for a relative column index).
        // - Groups 2: The relative column index, without brackets.
        patterns.COL_REF = prefixChars[1] + RC_INDEX_GROUPS;

        // RE pattern for a column range reference (absolute or relative), e.g. C[-1]:C[1].
        // - Groups 1 to 2: The components of the first column.
        // - Groups 3 to 4: The components of the second column.
        patterns.COLS_REF = patterns.COL_REF + ':' + patterns.COL_REF;

        // RE pattern for a single row reference (absolute or relative), e.g. R[1].
        // - Groups 1: The absolute row index (empty for a relative row index).
        // - Groups 2: The relative row index, without brackets.
        patterns.ROW_REF = prefixChars[0] + RC_INDEX_GROUPS;

        // RE pattern for a row range reference (absolute or relative), e.g. R[-1]:R[1].
        // - Groups 1 to 2: The components of the first row.
        // - Groups 3 to 4: The components of the second row.
        patterns.ROWS_REF = patterns.ROW_REF + ':' + patterns.ROW_REF;

        // RE pattern for a single cell reference (absolute or relative), e.g. R[-1]C[1].
        // - Group 1: The absolute row index (empty for a relative row index).
        // - Group 2: The relative row index, without brackets.
        // - Group 3: The absolute column index (empty for a relative column index).
        // - Group 4: The relative column index, without brackets.
        patterns.CELL_REF = patterns.ROW_REF + patterns.COL_REF;

        // RE pattern for a cell range reference (absolute or relative), e.g. R[-1]C[-1]:R[1]C[1].
        // - Groups 1 to 4: The components of the first cell reference.
        // - Groups 5 to 8: The components of the second cell reference.
        patterns.RANGE_REF = patterns.CELL_REF + ':' + patterns.CELL_REF;

        // RE pattern for a single absolute column reference e.g. C1.
        // - Group 1: The absolute column index.
        patterns.RL_COL_REF = prefixChars[1] + RC_ABS_INDEX_GROUP;

        // RE pattern for an absolute column range reference, e.g. C1:C3.
        // - Group 1: The index of the first column.
        // - Group 2: The index of the second column.
        patterns.RL_COLS_REF = patterns.RL_COL_REF + ':' + patterns.RL_COL_REF;

        // RE pattern for a single absolute row reference e.g. R1.
        // - Group 1: The absolute row index.
        patterns.RL_ROW_REF = prefixChars[0] + RC_ABS_INDEX_GROUP;

        // RE pattern for an absolute row range reference, e.g. R1:R3.
        // - Group 1: The index of the first row.
        // - Group 2: The index of the second row.
        patterns.RL_ROWS_REF = patterns.RL_ROW_REF + ':' + patterns.RL_ROW_REF;

        // RE pattern for an absolute single cell reference, e.g. R1C3.
        // - Group 1: The row index.
        // - Group 2: The column index.
        patterns.RL_CELL_REF = patterns.RL_ROW_REF + patterns.RL_COL_REF;

        // RE pattern for an absolute cell range reference, e.g. R1C1:R3C3.
        // - Group 1: The index of the first row.
        // - Group 2: The index of the first column.
        // - Group 3: The index of the second row.
        // - Group 4: The index of the second column.
        patterns.RL_RANGE_REF = patterns.RL_CELL_REF + ':' + patterns.RL_CELL_REF;

    } }); // class AddressConfigRC

    // public methods ---------------------------------------------------------

    /**
     * Extracts a column index from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The array index of the column index in the passed matches.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @returns {Number}
     *  The column for the passed matches, if it is valid; otherwise -1.
     */
    AddressConfigRC.prototype.getCol = function (matches, index, maxCol) {
        var col = parseInt(matches[index], 10) - 1;
        return (col <= maxCol) ? col : -1;
    };

    /**
     * Extracts a row index from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The array index of the row index in the passed matches.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @returns {Number}
     *  The row for the passed matches, if it is valid; otherwise -1.
     */
    AddressConfigRC.prototype.getRow = function (matches, index, maxRow) {
        var row = parseInt(matches[index], 10) - 1;
        return (row <= maxRow) ? row : -1;
    };

    /**
     * Extracts a cell address from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the cell address in the passed matches. The
     *  matches are expected to contain two elements for the row index, and the
     *  column index, in this order.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @returns {Address|Null}
     *  A cell address for the passed matches, if the column and row indexes
     *  are valid; otherwise null.
     */
    AddressConfigRC.prototype.getAddress = function (matches, index, maxCol, maxRow) {
        var col = this.getCol(matches, index + 1, maxCol);
        var row = this.getRow(matches, index, maxRow);
        return ((col >= 0) && (row >= 0)) ? new Address(col, row) : null;
    };

    /**
     * Returns a cell reference structure for the passed matches of a regular
     * expression in R1C1 notation.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the cell address in the passed matches. The
     *  matches are expected to contain four elements for the absolute row
     *  index, the relative row index, the absolute column index, and the
     *  relative column index, in this order.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @param {Address} refAddress
     *  The address of the reference cell used to resolve column/row offsets.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the column/row indexes at the sheet boundaries when
     *  resolving a column/row offset in R1C1 notation.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the column and
     *  row indexes are valid; otherwise null.
     */
    AddressConfigRC.prototype.createCellRef = function (matches, index, maxCol, maxRow, refAddress, wrapReferences) {
        var cellRef = new CellRef(0, 0, true, true);
        if (!cellRef.parseColRC(matches[index + 2], matches[index + 3], maxCol, refAddress[0], wrapReferences)) { return null; }
        if (!cellRef.parseRowRC(matches[index], matches[index + 1], maxRow, refAddress[1], wrapReferences)) { return null; }
        return cellRef;
    };

    /**
     * Returns a cell reference structure for a column from the passed matches
     * of a regular expression in R1C1 notation.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the column index in the passed matches. The
     *  matches are expected to contain two elements for the absolute column
     *  index, and the relative column index, in this order.
     *
     * @param {Number} fixedRow
     *  The fixed row index to be inserted into the cell reference.
     *
     * @param {Number} maxCol
     *  The largest valid column index that will be accepted by this method.
     *
     * @param {Number} refCol
     *  The index of the reference column needed to resolve column offsets.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the column index at the sheet boundaries when resolving
     *  a column offset in R1C1 notation.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the column index
     *  is valid; otherwise null.
     */
    AddressConfigRC.prototype.createColRef = function (matches, index, fixedRow, maxCol, refCol, wrapReferences) {
        var cellRef = new CellRef(0, fixedRow, true, true);
        return cellRef.parseColRC(matches[index], matches[index + 1], maxCol, refCol, wrapReferences) ? cellRef : null;
    };

    /**
     * Returns a cell reference structure for a row from the passed matches of
     * a regular expression in R1C1 notation.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the row index in the passed matches. The
     *  matches are expected to contain two elements for the absolute row
     *  index, and the relative row index, in this order.
     *
     * @param {Number} fixedCol
     *  The fixed column index to be inserted into the cell reference.
     *
     * @param {Number} maxRow
     *  The largest valid row index that will be accepted by this method.
     *
     * @param {Number} refCol
     *  The index of the reference row needed to resolve row offsets.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the row index at the sheet boundaries when resolving a
     *  row offset in R1C1 notation.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the row index is
     *  valid; otherwise null.
     */
    AddressConfigRC.prototype.createRowRef = function (matches, index, fixedCol, maxRow, refRow, wrapReferences) {
        var cellRef = new CellRef(fixedCol, 0, true, true);
        return cellRef.parseRowRC(matches[index], matches[index + 1], maxRow, refRow, wrapReferences) ? cellRef : null;
    };

    // class ReferenceGrammarBase =============================================

    /**
     * Base class for configurations of different reference syntaxes used in
     * formula expressions.
     *
     * @constructor
     *
     * @property {SheetConfigBase} sheetConfig
     *  The sheet reference syntax configuration passed to the constructor.
     *
     * @property {AddressConfigBase} addrConfig
     *  The address reference syntax configuration, according to the reference
     *  style specified in the constructor.
     *
     * @property {Object<String, RegExp>} RE
     *  The regular expressions to match different kinds of references in
     *  formulas.
     */
    var ReferenceGrammarBase = _.makeExtendable(function (sheetConfig, prefixChars) {

        // the sheet reference syntax configuration
        this.sheetConfig = sheetConfig;

        // the cell reference syntax configuration
        this.addrConfig = AddressConfigBase.create(prefixChars);

        // the regular expressions to match different kinds of references in formulas
        var re = this.RE = {};

        // RangeList: A sequence of white-space characters as separator between two ranges.
        re.RL_SEP_WS = /^\s+/;

        // RangeList: A single separator character other than white-space between two ranges.
        re.RL_SEP_OTHER = /^[,;]/;

    }); // class ReferenceGrammarBase

    // public methods ---------------------------------------------------------

    /**
     * Returns a sheet reference structure for the passed matches of a regular
     * expression.
     *
     * @param {SpreadsheetModel} docModel
     *  The model of the spreadsheet document used to resolve the sheet names.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the sheet reference in the passed matches.
     *
     * @returns {SheetRef}
     *  A sheet reference structure for the passed matches.
     */
    ReferenceGrammarBase.prototype.createSheetRef = function (docModel, matches, index) {
        return this.sheetConfig.createSheetRef(docModel, matches, index);
    };

    /**
     * Returns an object with two cell reference structures for a cell range
     * address from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first cell address in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second cell address in the passed matches.
     *  If omitted, the matches for a single cell address will be converted to
     *  the result.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address (or null for single cell addresses), if the
     *  column and row indexes are all valid; otherwise null.
     */
    ReferenceGrammarBase.prototype.createCellRangeRefs = function (docModel, refAddress, wrapReferences, matches, index1, index2) {

        // get the sheet limits of the document
        var maxCol = docModel.getMaxCol();
        var maxRow = docModel.getMaxRow();

        // parse the first cell reference, exit on error
        var cell1Ref = this.addrConfig.createCellRef(matches, index1, maxCol, maxRow, refAddress, wrapReferences);
        if (!cell1Ref) { return null; }

        // indicate single cell addresses by setting property 'r2' to null
        if (typeof index2 !== 'number') { return { r1: cell1Ref, r2: null }; }

        // parse the second cell reference, exit on error
        var cell2Ref = this.addrConfig.createCellRef(matches, index2, maxCol, maxRow, refAddress, wrapReferences);
        return cell2Ref ? { r1: cell1Ref, r2: cell2Ref } : null;
    };

    /**
     * Returns an object with two cell reference structures for a column range
     * address from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first column index in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second column index in the passed matches.
     *  If omitted, the matches for a single column reference will be converted
     *  to the result.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address, if the column indexes are all valid;
     *  otherwise null.
     */
    ReferenceGrammarBase.prototype.createColRangeRefs = function (docModel, refAddress, wrapReferences, matches, index1, index2) {

        // get the sheet limits of the document
        var maxCol = docModel.getMaxCol();
        var maxRow = docModel.getMaxRow();

        // parse the first column reference, exit on error
        var cell1Ref = this.addrConfig.createColRef(matches, index1, 0, maxCol, refAddress[0], wrapReferences);
        if (!cell1Ref) { return null; }

        // add end address to single column references
        if (typeof index2 !== 'number') {
            return { r1: cell1Ref, r2: new CellRef(cell1Ref.col, maxRow, cell1Ref.absCol, true) };
        }

        // parse the second column reference, exit on error
        var cell2Ref = this.addrConfig.createColRef(matches, index2, maxRow, maxCol, refAddress[0], wrapReferences);
        return cell2Ref ? { r1: cell1Ref, r2: cell2Ref } : null;
    };

    /**
     * Returns an object with two cell reference structures for a row range
     * address from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first row index in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second row index in the passed matches. If
     *  omitted, the matches for a single column reference will be converted to
     *  the result.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address, if the row indexes are all valid;
     *  otherwise null.
     */
    ReferenceGrammarBase.prototype.createRowRangeRefs = function (docModel, refAddress, wrapReferences, matches, index1, index2) {

        // get the sheet limits of the document
        var maxCol = docModel.getMaxCol();
        var maxRow = docModel.getMaxRow();

        // parse the first row reference, exit on error
        var cell1Ref = this.addrConfig.createRowRef(matches, index1, 0, maxRow, refAddress[1], wrapReferences);
        if (!cell1Ref) { return null; }

        // add end address to single row references
        if (typeof index2 !== 'number') {
            return { r1: cell1Ref, r2: new CellRef(maxCol, cell1Ref.row, true, cell1Ref.absRow) };
        }

        // parse the second row reference, exit on error
        var cell2Ref = this.addrConfig.createRowRef(matches, index2, maxCol, maxRow, refAddress[1], wrapReferences);
        return cell2Ref ? { r1: cell1Ref, r2: cell2Ref } : null;
    };

    // class ReferenceGrammarSheetRangeBase ===================================

    /**
     * Base class for configurations of different reference syntaxes used in
     * formula expressions. This base class is used for references with sheet
     * ranges in front of cell ranges, separated by an exclamation mark, e.g.
     * Sheet1:Sheet2!A1:A1 (in difference to reference syntaxes with
     * independent sheet/cell references, e.g. in the OpenFormula reference
     * [Sheet1.A1:Sheet2.A1]).
     *
     * @constructor
     *
     * @extends ReferenceGrammarBase
     */
    var ReferenceGrammarOOXBase = ReferenceGrammarBase.extend({ constructor: function (sheetConfig, prefixChars) {

        // base constructor ---------------------------------------------------

        ReferenceGrammarBase.call(this, sheetConfig, prefixChars);

        // initialization -----------------------------------------------------

        // the RE patterns for sheet references
        var sheetPatterns = this.sheetConfig.PATTERNS;
        // the RE patterns for cell/range references
        var addrPatterns = this.addrConfig.PATTERNS;
        // the regular expressions to match different kinds of references in formulas
        var re = this.RE;

        // A local cell reference, e.g. A1 or $A$1, but without trailing parenthesis,
        // e.g. F2(), and without trailing sheet separator, e.g. F2!A1.
        re.CELL_REF = new RegExp('^' + addrPatterns.CELL_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A cell reference with single sheet, e.g. Sheet1!$A$1, but without trailing parenthesis,
        // e.g. Module1!F2() which is a call of the macro function F2() contained in Module1.
        re.CELL_3D_REF = new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + addrPatterns.CELL_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A cell reference in multiple sheets, e.g. Sheet1:Sheet2!$A$1.
        re.CELL_CUBE_REF = new RegExp('^' + sheetPatterns.SHEET_RANGE_REF + '!' + addrPatterns.CELL_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A local range reference, e.g. A1:B2 or $A$1:$B$2, but without trailing parenthesis, e.g. A1:F2(),
        // which is the cell address A1, and the result of the function call F2(), connected with a range operator.
        re.RANGE_REF = new RegExp('^' + addrPatterns.RANGE_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A range reference with single sheet, e.g. Sheet1!$A$1:$B$2, but without trailing parenthesis,
        // e.g. Sheet1!A1:F2().
        re.RANGE_3D_REF = new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + addrPatterns.RANGE_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A range reference in multiple sheets, e.g. Sheet1:Sheet2!$A$1:$A$1.
        re.RANGE_CUBE_REF = new RegExp('^' + sheetPatterns.SHEET_RANGE_REF + '!' + addrPatterns.RANGE_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A local single column reference, e.g. C[1] (only used in R1C1 notation).
        re.COL_REF = addrPatterns.COL_REF ? new RegExp('^' + addrPatterns.COL_REF + sheetPatterns.TERMINATE_REF, 'i') : /^(?!)/;

        // A single column reference with single sheet, e.g. Sheet1!C[1] (only used in R1C1 notation).
        re.COL_3D_REF = addrPatterns.COL_REF ? new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + addrPatterns.COL_REF + sheetPatterns.TERMINATE_REF, 'i') : /^(?!)/;

        // A single column reference in multiple sheets, e.g. Sheet1:Sheet2!C[1] (only used in R1C1 notation).
        re.COL_CUBE_REF = addrPatterns.COL_REF ? new RegExp('^' + sheetPatterns.SHEET_RANGE_REF + '!' + addrPatterns.COL_REF + sheetPatterns.TERMINATE_REF, 'i') : /^(?!)/;

        // A local column reference, e.g. A:B or $C:$C (always with colon).
        re.COLS_REF = new RegExp('^' + addrPatterns.COLS_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A column reference with single sheet, e.g. Sheet1!$C:$C (always with colon).
        re.COLS_3D_REF = new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + addrPatterns.COLS_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A column reference in multiple sheets, e.g. Sheet1:Sheet2!$C:$C (always with colon).
        re.COLS_CUBE_REF = new RegExp('^' + sheetPatterns.SHEET_RANGE_REF + '!' + addrPatterns.COLS_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A local single row reference, e.g. R[1] (only used in R1C1 notation).
        re.ROW_REF = addrPatterns.ROW_REF ? new RegExp('^' + addrPatterns.ROW_REF + sheetPatterns.TERMINATE_REF, 'i') : /^(?!)/;

        // A single row reference with single sheet, e.g. Sheet1!R[1] (only used in R1C1 notation).
        re.ROW_3D_REF = addrPatterns.ROW_REF ? new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + addrPatterns.ROW_REF + sheetPatterns.TERMINATE_REF, 'i') : /^(?!)/;

        // A single row reference in multiple sheets, e.g. Sheet1:Sheet2!R[1] (only used in R1C1 notation).
        re.ROW_CUBE_REF = addrPatterns.ROW_REF ? new RegExp('^' + sheetPatterns.SHEET_RANGE_REF + '!' + addrPatterns.ROW_REF + sheetPatterns.TERMINATE_REF, 'i') : /^(?!)/;

        // A local row reference, e.g. 1:2 or $3:$3 (always with colon).
        re.ROWS_REF = new RegExp('^' + addrPatterns.ROWS_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A row reference with single sheet, e.g. Sheet1!$3:$3 (always with colon).
        re.ROWS_3D_REF = new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + addrPatterns.ROWS_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A row reference in multiple sheets, e.g. Sheet1:Sheet2!$3:$3 (always with colon).
        re.ROWS_CUBE_REF = new RegExp('^' + sheetPatterns.SHEET_RANGE_REF + '!' + addrPatterns.ROWS_REF + sheetPatterns.TERMINATE_REF, 'i');

        // A reference error with single sheet reference, e.g. Sheet1!#REF!.
        re.ERROR_3D_REF = new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + sheetPatterns.REF_ERROR, 'i');

        // A reference error in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!#REF!.
        re.ERROR_CUBE_REF = new RegExp('^' + sheetPatterns.SHEET_RANGE_REF + '!' + sheetPatterns.REF_ERROR, 'i');

        // A local function name (names followed by an opening parenthesis), e.g. SUM().
        re.FUNCTION_REF = new RegExp('^' + sheetPatterns.FUNC_NAME + '(?=\\()', 'i');

        // A function name (names followed by an opening parenthesis) with sheet, e.g. Module1!macro().
        re.FUNCTION_3D_REF = new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + sheetPatterns.FUNC_NAME + '(?=\\()', 'i');

        // Defined names without sheet reference, e.g. my_name.
        re.NAME_REF = new RegExp('^' + sheetPatterns.DEFINED_NAME + sheetPatterns.TERMINATE_REF, 'i');

        // Defined names with sheet reference, e.g. Sheet1!my_name.
        re.NAME_3D_REF = new RegExp('^' + sheetPatterns.SHEET_OR_ERR_REF + '!' + sheetPatterns.DEFINED_NAME + sheetPatterns.TERMINATE_REF, 'i');

        // RangeList: A cell address, e.g. A1.
        re.RL_CELL_REF = new RegExp('^' + addrPatterns.RL_CELL_REF, 'i');

        // RangeList: A cell address with sheet, e.g. Sheet1!A1.
        re.RL_CELL_3D_REF = new RegExp('^' + sheetPatterns.SINGLE_SHEET_NAME + '!' + addrPatterns.RL_CELL_REF, 'i');

        // RangeList: A range address, e.g. A1:B2.
        re.RL_RANGE_REF = new RegExp('^' + addrPatterns.RL_RANGE_REF, 'i');

        // RangeList: A range address with sheet, e.g. Sheet1!A1:B2.
        re.RL_RANGE_3D_REF = new RegExp('^' + sheetPatterns.SINGLE_SHEET_NAME + '!' + addrPatterns.RL_RANGE_REF, 'i');

        // RangeList: A single column, e.g. C1 (only used in R1C1 notation).
        re.RL_COL_REF = addrPatterns.RL_COL_REF ? new RegExp('^' + addrPatterns.RL_COL_REF, 'i') : /^(?!)/;

        // RangeList: A single column with sheet, e.g. Sheet1!C1 (only used in R1C1 notation).
        re.RL_COL_3D_REF = addrPatterns.RL_COL_REF ? new RegExp('^' + sheetPatterns.SINGLE_SHEET_NAME + '!' + addrPatterns.RL_COL_REF, 'i') : /^(?!)/;

        // RangeList: A column interval, e.g. A:B (always with colon).
        re.RL_COLS_REF = new RegExp('^' + addrPatterns.RL_COLS_REF, 'i');

        // RangeList: A column interval with sheet, e.g. Sheet1!A:B (always with colon).
        re.RL_COLS_3D_REF = new RegExp('^' + sheetPatterns.SINGLE_SHEET_NAME + '!' + addrPatterns.RL_COLS_REF, 'i');

        // RangeList: A single row, e.g. R1 (only used in R1C1 notation).
        re.RL_ROW_REF = addrPatterns.RL_ROW_REF ? new RegExp('^' + addrPatterns.RL_ROW_REF, 'i') : /^(?!)/;

        // RangeList: A single row with sheet, e.g. Sheet1!R1 (only used in R1C1 notation).
        re.RL_ROW_3D_REF = addrPatterns.RL_ROW_REF ? new RegExp('^' + sheetPatterns.SINGLE_SHEET_NAME + '!' + addrPatterns.RL_ROW_REF, 'i') : /^(?!)/;

        // RangeList: A row interval, e.g. 1:2 (always with colon).
        re.RL_ROWS_REF = new RegExp('^' + addrPatterns.RL_ROWS_REF, 'i');

        // RangeList: A row interval with sheet, e.g. Sheet1!1:2 (always with colon).
        re.RL_ROWS_3D_REF = new RegExp('^' + sheetPatterns.SINGLE_SHEET_NAME + '!' + addrPatterns.RL_ROWS_REF, 'i');

    } }); // class ReferenceGrammarSheetRangeBase

    // class ReferenceGrammarOOX ==============================================

    /**
     * Configuration settings and helper methods for references as used in the
     * OOXML file format (exclamation mark as sheet name separator, sheets are
     * always implicitly absolute, apostrophes enclose an entire range of sheet
     * names, e.g.: ='Sheet1:Sheet 2'!A1:B2).
     *
     * @constructor
     *
     * @extends ReferenceGrammarOOXBase
     *
     * @param {String} refError
     *  The name of the #REF! error code, according to the formula language.
     *
     * @param {String|Null} prefixChars
     *  If set to a string, R1C1 notation will be used by the new configuration
     *  instance. Otherwise, A1 notation will be used.
     */
    var ReferenceGrammarOOX = ReferenceGrammarOOXBase.extend({ constructor: function (refError, prefixChars) {

        // configuration for sheet references
        var sheetConfig = SheetConfigOOX.create(refError);

        // base constructor
        ReferenceGrammarOOXBase.call(this, sheetConfig, prefixChars);

    } }); // class ReferenceGrammarOOX

    // static methods ---------------------------------------------------------

    ReferenceGrammarOOX.create = _.memoize(function (refError, prefixChars) {
        return new ReferenceGrammarOOX(refError, prefixChars);
    }, function (refError, prefixChars) {
        return refError + '!' + AddressConfigBase.getCacheKey(prefixChars);
    });

    // public methods ---------------------------------------------------------

    /**
     * Returns an object with sheet reference structures for a single sheet
     * name, or for a range of sheet names from the passed matches of a regular
     * expression.
     *
     * @param {SpreadsheetModel} docModel
     *  The model of the spreadsheet document used to resolve the sheet names.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the sheet reference in the passed matches.
     *
     * @param {Boolean} [range=false]
     *  If set to true, the passed matches represent a range of sheet names.
     *
     * @returns {Object}
     *  An object with properties 'r1' and 'r2' set to the sheet reference
     *  structures for the passed matches (property 'r2' will be null for a
     *  single sheet name).
     */
    ReferenceGrammarOOX.prototype.createSheetRefs = function (docModel, matches, index, range) {

        // sheet range: RE matches contain [simple1, simple2, complex1, complex2] in this order
        if (range) {
            var sheet1Ref = SheetRef.create(docModel, matches[index], matches[index + 2], null, true);
            var sheet2Ref = SheetRef.create(docModel, matches[index + 1], matches[index + 3], null, true);
            return { r1: sheet1Ref, r2: sheet2Ref };
        }

        // single sheet: RE matches contain [simple, complex, error] in this order
        return { r1: this.sheetConfig.createSheetRef(docModel, matches, index), r2: null };
    };

    // class ReferenceGrammarOF ===============================================

    /**
     * Configuration settings and helper methods for references as used in the
     * OpenFormula specification (entire reference enclosed in brackets,
     * references in A1 notation, row or column index can be #REF! errors,
     * period as sheet name separator, sheets can be absolute or relative,
     * apostrophes enclose single sheet names in a range of sheets, e.g.:
     * =[Sheet1.A1:$'Sheet 2'.B#REF!]).
     *
     * @constructor
     *
     * @extends ReferenceGrammarBase
     *
     * @param {String} refError
     *  The name of the #REF! error code, according to the formula language.
     */
    var ReferenceGrammarOF = ReferenceGrammarBase.extend({ constructor: function (refError) {

        // configuration for sheet references
        var sheetConfig = SheetConfigODF.create(refError);

        // base constructor ---------------------------------------------------

        ReferenceGrammarBase.call(this, sheetConfig, null);

        // initialization -----------------------------------------------------

        // RE patterns for ODF sheet references
        var sheetPatterns = sheetConfig.PATTERNS;
        // the RE patterns for cell/range references
        var addrPatterns = this.addrConfig.PATTERNS;

        // RE pattern for the #REF! error code.
        var REF_ERROR_PATTERN = _.escapeRegExp(refError);

        // RE pattern for a column name (absolute or relative, #REF! error allowed).
        // - Group 1: The absolute marker.
        // - Group 2: The column name, or the #REF! error code.
        var COL_PATTERN = A1_ABS_GROUP + '(' + A1_COL_PATTERN + '|' + REF_ERROR_PATTERN + ')';

        // RE pattern for a row name (absolute or relative, #REF! error allowed).
        // - Group 1: The absolute marker.
        // - Group 2: The row name, or the #REF! error code.
        var ROW_PATTERN = A1_ABS_GROUP + '(' + A1_ROW_PATTERN + '|' + REF_ERROR_PATTERN + ')';

        // RE pattern for a cell address (with leading period, #REF! errors for column or row allowed).
        // - Group 1: The absolute marker for the column.
        // - Group 2: The name of the column, or the #REF! error code.
        // - Group 3: The absolute marker for the row.
        // - Group 4: The name of the row, or the #REF! error code.
        var CELL_PATTERN = '\\.' + COL_PATTERN + ROW_PATTERN;

        // RE pattern for a cell address with sheet name (#REF! errors for column or row allowed).
        // - Groups 1 to 4: The conponents of the sheet reference.
        // - Groups 5 to 8: The components of the cell reference.
        var CELL_3D_PATTERN = sheetPatterns.SHEET_OR_ERR_REF + CELL_PATTERN;

        // RangeList: A cell address with sheet without brackets, e.g. Sheet1.A1.
        var RL_CELL_3D_PATTERN = sheetPatterns.SINGLE_SHEET_NAME + '\\.' + addrPatterns.RL_CELL_REF;

        // the regular expressions to match different kinds of references in formulas
        var re = this.RE;

        // A local cell reference, e.g. [.A1], [.$A$1], or [.A$#REF!].
        // - Groups 1 to 4: The components of the cell reference.
        re.CELL_REF = new RegExp('^\\[' + CELL_PATTERN + '\\]', 'i');

        // A cell reference with sheet in OpenFormula syntax, e.g. [Sheet1.A1], [$Sheet1.$A$1], or [#REF!.A$#REF!].
        // - Groups 1 to 4: The components of the sheet reference.
        // - Groups 5 to 8: The components of the cell reference.
        re.CELL_3D_REF = new RegExp('^\\[' + CELL_3D_PATTERN + '\\]', 'i');

        // A local cell range reference in OpenFormula syntax, e.g. [.A1:.$A$1], or [.A#REF!:.B2].
        // - Groups 1 to 4: The components of the first cell reference.
        // - Groups 5 to 8: The components of the second cell reference.
        re.RANGE_REF = new RegExp('^\\[' + CELL_PATTERN + ':' + CELL_PATTERN + '\\]', 'i');

        // A cell range reference with sheet in OpenFormula syntax, e.g. [Sheet1.A1:.$A$1], or [$Sheet2.A#REF!:.B2].
        // - Groups 1 to 4: The components of the sheet reference.
        // - Groups 5 to 8: The components of the first cell reference.
        // - Groups 9 to 12: The components of the second cell reference.
        re.RANGE_3D_REF = new RegExp('^\\[' + CELL_3D_PATTERN + ':' + CELL_PATTERN + '\\]', 'i');

        // A cell range reference in multiple sheets in OpenFormula syntax, e.g. [Sheet1.A1:$Sheet2.$A$1],
        // or [$Sheet2.A#REF!:#REF!.#REF!B2].
        // - Groups 1 to 4: The components of the first sheet reference.
        // - Groups 5 to 8: The components of the first cell reference.
        // - Groups 9 to 12: The components of the second sheet reference.
        // - Groups 13 to 16: The components of the second cell reference.
        re.RANGE_CUBE_REF = new RegExp('^\\[' + CELL_3D_PATTERN + ':' + CELL_3D_PATTERN + '\\]', 'i');

        // A local function name (names followed by an opening parenthesis with optional whitespace), e.g. SUM ().
        re.FUNCTION_REF = new RegExp('^' + sheetPatterns.FUNC_NAME + '(?:[' + WHITESPACE_CHARS + ']*)(?=\\()', 'i');

        // Defined names without sheet reference, e.g. my_name.
        re.NAME_REF = new RegExp('^' + sheetPatterns.DEFINED_NAME + sheetPatterns.TERMINATE_REF, 'i');

        // RangeList: A cell address without dots and brackets, e.g. A1.
        re.RL_CELL_REF = new RegExp('^' + addrPatterns.RL_CELL_REF, 'i');

        // RangeList: A cell address with sheet without brackets, e.g. Sheet1.A1.
        re.RL_CELL_3D_REF = new RegExp('^' + RL_CELL_3D_PATTERN, 'i');

        // RangeList: A range address without dots and brackets, e.g. A1:B2.
        re.RL_RANGE_REF = new RegExp('^' + addrPatterns.RL_RANGE_REF, 'i');

        // RangeList: A range address with sheet without brackets, e.g. Sheet1.A1:A1.
        re.RL_RANGE_3D_REF = new RegExp('^' + sheetPatterns.SINGLE_SHEET_NAME + '\\.' + addrPatterns.RL_RANGE_REF, 'i');

        // RangeList: A range address in multiple sheets without brackets, e.g. Sheet1.A1:Sheet2.A1.
        re.RL_RANGE_CUBE_REF = new RegExp('^' + RL_CELL_3D_PATTERN + ':' + RL_CELL_3D_PATTERN, 'i');

    } }); // class ReferenceGrammarOF

    // static methods ---------------------------------------------------------

    ReferenceGrammarOF.create = _.memoize(function (refError) {
        return new ReferenceGrammarOF(refError);
    });

    // public methods ---------------------------------------------------------

    /**
     * Returns an object with sheet reference structures for a single sheet
     * name, or for a range of sheet names from the passed matches of a regular
     * expression.
     *
     * @param {SpreadsheetModel} docModel
     *  The model of the spreadsheet document used to resolve the sheet names.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first sheet reference in the passed
     *  matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second sheet reference in the passed
     *  matches. If omitted, the matches for a single sheet name will be
     *  converted to the result.
     *
     * @returns {Object}
     *  An object with properties 'r1' and 'r2' set to the sheet reference
     *  structures for the passed matches (property 'r2' will be null for a
     *  single sheet name).
     */
    ReferenceGrammarOF.prototype.createSheetRefs = function (docModel, matches, index1, index2) {
        var sheetRange = typeof index2 === 'number';
        var sheet1Ref = this.sheetConfig.createSheetRef(docModel, matches, index1);
        var sheet2Ref = sheetRange ? this.sheetConfig.createSheetRef(docModel, matches, index2) : null;
        return { r1: sheet1Ref, r2: sheet2Ref };
    };

    // class ReferenceGrammarODFUI ============================================

    /**
     * Configuration settings and helper methods for references as used in the
     * user interface for ODF documents (similar to OOXML style, but with ODF
     * specific extensions: exclamation mark as sheet name separator, sheets
     * can be absolute or relative, apostrophes enclose single sheet names in a
     * range of sheets, e.g.: =Sheet1:$'Sheet 2'!A1:B2).
     *
     * @constructor
     *
     * @extends ReferenceGrammarOOXBase
     *
     * @param {String} refError
     *  The name of the #REF! error code, according to the formula language.
     *
     * @param {String|Null} prefixChars
     *  If set to a string, R1C1 notation will be used by the new configuration
     *  instance. Otherwise, A1 notation will be used.
     */
    var ReferenceGrammarODFUI = ReferenceGrammarOOXBase.extend({ constructor: function (refError, prefixChars) {

        // configuration for sheet references
        var sheetConfig = SheetConfigODF.create(refError);

        // base constructor
        ReferenceGrammarOOXBase.call(this, sheetConfig, prefixChars);

    } }); // class ReferenceGrammarODFUI

    // static methods ---------------------------------------------------------

    ReferenceGrammarODFUI.create = _.memoize(function (refError, prefixChars) {
        return new ReferenceGrammarODFUI(refError, prefixChars);
    }, function (refError, prefixChars) {
        return refError + '!' + AddressConfigBase.getCacheKey(prefixChars);
    });

    // public methods ---------------------------------------------------------

    /**
     * Returns an object with sheet reference structures for a single sheet
     * name, or for a range of sheet names from the passed matches of a regular
     * expression.
     *
     * @param {SpreadsheetModel} docModel
     *  The model of the spreadsheet document used to resolve the sheet names.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the sheet reference in the passed matches.
     *
     * @param {Boolean} [range=false]
     *  If set to true, the passed matches represent a range of sheet names.
     *
     * @returns {Object}
     *  An object with properties 'r1' and 'r2' set to the sheet reference
     *  structures for the passed matches (property 'r2' will be null for a
     *  single sheet name).
     */
    ReferenceGrammarODFUI.prototype.createSheetRefs = function (docModel, matches, index, range) {

        // sheet range: RE matches contain [abs1, simple1, complex1, abs2, simple2, complex2] in this order
        if (range) {
            var sheet1Ref = SheetRef.create(docModel, matches[index + 1], matches[index + 2], null, !!matches[index]);
            var sheet2Ref = SheetRef.create(docModel, matches[index + 4], matches[index + 5], null, !!matches[index + 3]);
            return { r1: sheet1Ref, r2: sheet2Ref };
        }

        // single sheet: RE matches contain [abs, simple, complex, error] in this order
        return { r1: this.sheetConfig.createSheetRef(docModel, matches, index), r2: null };
    };

    // static class ReferenceGrammar ==========================================

    var ReferenceGrammar = {};

    // static methods ---------------------------------------------------------

    /**
     * Returns an existing reference grammar singleton from the internal cache,
     * if available; otherwise creates a new instance.
     *
     * @param {FormulaResource} formulaResource
     *  The collection of file-format dependent formula resources to create the
     *  reference grammar for.
     *
     * @param {Boolean} localized
     *  Whether to create a reference grammar for operations (false), or for
     *  the user interface (true).
     *
     * @param {Boolean} rcStyle
     *  Whether to create a reference grammar for A1 notation (false), or for
     *  R1C1 notation (true).
     *
     * @returns {ReferenceGrammarBase}
     *  A reference grammar singleton for the passed parameters.
     */
    ReferenceGrammar.create = function (formulaResource, localized, rcStyle) {

        // the native or translated name of the #REF! error code
        var refError = formulaResource.getErrorCollection().getName(ErrorCode.REF.key, localized);
        // prefix characters for R1C1 notation; or null for A1 notation
        var prefixChars = rcStyle ? formulaResource.getRCPrefixChars(localized) : null;

        // create a new configuration instance on demand
        switch (formulaResource.getFileFormat()) {
            case 'ooxml':
                return ReferenceGrammarOOX.create(refError, localized ? prefixChars : null);
            case 'odf':
                return localized ? ReferenceGrammarODFUI.create(refError, prefixChars) : ReferenceGrammarOF.create(refError);
        }
        Utils.error('ReferenceGrammar.create(): invalid file format: "' + formulaResource.getFileFormat() + '"');
    };

    // exports ================================================================

    return ReferenceGrammar;

});
