/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/parser/formulaparser', [
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/parser/expressionparser',
    'io.ox/office/spreadsheet/model/formula/parser/rangelistparser'
], function (SimpleMap, ModelObject, FormulaUtils, ExpressionParser, RangeListParser) {

    'use strict';

    // class FormulaParser ====================================================

    /**
     * Parses formula expressions to arrays of formula tokens.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    var FormulaParser = ModelObject.extend({ constructor: function (docModel) {

        // all existing expression parsers, mapped by grammar identifiers
        var exprParserMap = new SimpleMap();

        // all existing range list parsers, mapped by grammar identifiers
        var listParserMap = new SimpleMap();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Gets the expression parser for the specified formula grammar.
         */
        function getExpressionParser(grammarId) {
            return exprParserMap.getOrCreate(grammarId, function () {
                return ExpressionParser.create(docModel, grammarId);
            });
        }

        /**
         * Gets the range list parser for the specified formula grammar.
         */
        function getRangeListParser(grammarId) {
            return listParserMap.getOrCreate(grammarId, function () {
                return RangeListParser.create(docModel, grammarId);
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Converts the passed formula expression to an array of formula
         * tokens.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar to be used to parse the
         *  formula expression. See SpreadsheetDocument.getFormulaGrammar() for
         *  more details.
         *
         * @param {String} formula
         *  The formula string, without the leading equality sign.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.autoCorrect=false]
         *      If set to true, an incomplete formula will be auto-corrected,
         *      e.g. missing closing parentheses at the end will be added.
         *  @param {Address} [options.refAddress]
         *      The address of the original reference cell the passed formula
         *      expression is related to. Will be used for example to resolve
         *      relative cell references in R1C1 notation. If omitted, address
         *      A1 will be used as reference address.
         *  @param {Number} [options.refSheet]
         *      The zero-based index of the sheet the passed formula expression
         *      is related to.
         *  @param {Boolean} [options.extendSheet=false]
         *      If set to true, and the option 'refSheet' has been set, all
         *      cell references without a sheet reference, and all references
         *      to existing sheet-local names will be extended with a sheet
         *      reference pointing to the sheet contained in this option.
         *      Example: The formula expression 'A1+global+local' extended to
         *      the first sheet may become 'Sheet1!A1+global+Sheet1!local'.
         *  @param {Boolean} [options.unqualifiedTables=false]
         *      If set to true, and the options 'refAddress' and 'refSheet'
         *      have been set, the parser accepts unqualified structured table
         *      references, and implicitly resolves them to the table range
         *      that contains the specified reference cell.
         *
         * @returns {Array<TokenDescriptor>}
         *  An array of descriptors for all parsed formula tokens.
         */
        this.parseFormula = FormulaUtils.profileMethod('FormulaParser.parseFormula()', function (grammarId, formula, options) {
            FormulaUtils.log('grammar="' + grammarId + '", formula="' + formula + '"');
            return getExpressionParser(grammarId).parseFormula(formula, options);
        });

        /**
         * Parses the passed range expression to a cell range address. The
         * range is expected to be in A1 notation.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar used to parse the range. See
         *  SpreadsheetDocument.getFormulaGrammar() for more details.
         *
         * @param {String} formula
         *  The text to be parsed to a cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipSheets=false]
         *      If set to true, the parser will accept (and ignore!) arbitrary
         *      sheet names in front of the cell range in the expression. By
         *      default, the expression must not contain sheet names.
         *
         * @returns {Range|Null}
         *  The cell range address parsed from the specified range expression;
         *  or null on any parser error.
         */
        this.parseRange = FormulaUtils.profileMethod('FormulaParser.parseRange()', function (grammarId, formula, options) {
            FormulaUtils.log('formula="' + formula + '"');
            return getRangeListParser(grammarId).parseRange(formula, options);
        });

        /**
         * Converts the passed cell range address to its text representation.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar used to generate the text
         *  representation. See SpreadsheetDocument.getFormulaGrammar() for
         *  more details.
         *
         * @param {Range} range
         *  The cell range address to be converted.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.sheetName]
         *      A sheet name that will be inserted for the cell range address
         *      according to the formula grammar represented by this instance.
         *
         * @returns {String}
         *  The text representation of the passed cell range address.
         */
        this.formatRange = function (grammarId, range, options) {
            return getRangeListParser(grammarId).formatRange(range, options);
        };

        /**
         * Parses the passed range list expression to an array of cell range
         * addresses. The range list is expected to contain range addresses in
         * A1 notation, separated by white-space characters.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar used to parse the range list.
         *  See SpreadsheetDocument.getFormulaGrammar() for more details.
         *
         * @param {String} formula
         *  The text to be parsed to an array of cell range addresses.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.extSep=false]
         *      If set to true, the parser will accept single commas, or single
         *      semicolons as list separators. Nevertheless, the entire range
         *      list expression must not use different separators.
         *  @param {Boolean} [options.skipSheets=false]
         *      If set to true, the parser will accept (and ignore!) arbitrary
         *      sheet names in front of the cell ranges in the expression. By
         *      default, the expression must not contain sheet names.
         *
         * @returns {RangeArray|Null}
         *  The array of cell range addresses parsed from the specified range
         *  list expression; or null on any parser error.
         */
        this.parseRangeList = FormulaUtils.profileMethod('FormulaParser.parseRangeList()', function (grammarId, formula, options) {
            FormulaUtils.log('formula="' + formula + '"');
            return getRangeListParser(grammarId).parseRangeList(formula, options);
        });

        /**
         * Converts the passed array of cell range addressses to their text
         * representation as used in range lists.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar used to generate the text
         *  representation. See SpreadsheetDocument.getFormulaGrammar() for
         *  more details.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.sep=' ']
         *      The separator character that will be inserted between the cell
         *      range addresses.
         *  @param {String} [options.sheetName]
         *      A sheet name that will be inserted for each cell range address
         *      according to the formula grammar represented by this instance.
         *
         * @returns {String}
         *  The text representation of the passed cell range addresses.
         */
        this.formatRangeList = function (grammarId, ranges, options) {
            return getRangeListParser(grammarId).formatRangeList(ranges, options);
        };

        /**
         * Parses the passed range list expression to an array of cell range
         * addresses with sheet indexes. The range list is expected to contain
         * range addresses in A1 notation, separated by white-space characters.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar used to parse the range list.
         *  See SpreadsheetDocument.getFormulaGrammar() for more details.
         *
         * @param {String} formula
         *  The text to be parsed to an array of cell range addresses.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.extSep=false]
         *      If set to true, the parser will accept single commas, or single
         *      semicolons as list separators. Nevertheless, the entire range
         *      list expression must not use different separators.
         *  @param {Number} [options.refSheet]
         *      The zero-based index of the reference sheet that will be added
         *      to cell ranges that do not contain sheet names in the passed
         *      expression. If omitted, range addresses without sheet names
         *      will not be accepted.
         *
         * @returns {Range3DArray|Null}
         *  The array of cell range addresses with sheet indexes parsed from
         *  the specified range list expression; or null on any parser error.
         */
        this.parseRange3DList = FormulaUtils.profileMethod('FormulaParser.parseRange3DList()', function (grammarId, formula, options) {
            FormulaUtils.log('formula="' + formula + '"');
            return getRangeListParser(grammarId).parseRange3DList(formula, options);
        });

        /**
         * Converts the passed array of cell range addressses with sheet
         * indexes to their text representation as used in range lists.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar used to generate the text
         *  representation. See SpreadsheetDocument.getFormulaGrammar() for
         *  more details.
         *
         * @param {Range3DArray|Range3D} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.sep=' ']
         *      The separator character that will be inserted between the cell
         *      range addresses.
         *
         * @returns {String}
         *  The text representation of the passed cell range addresses.
         */
        this.formatRange3DList = function (grammarId, ranges, options) {
            return getRangeListParser(grammarId).formatRange3DList(ranges, options);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            exprParserMap.forEach(function (exprParser) { exprParser.destroy(); });
            listParserMap.forEach(function (listParser) { listParser.destroy(); });
            docModel = exprParserMap = listParserMap = null;
        });

    } }); // class FormulaParser

    // exports ================================================================

    return FormulaParser;

});
