/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/interpret/formulacontext', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/interpret/operand',
    'io.ox/office/spreadsheet/model/formula/interpret/filteraggregator'
], function (Utils, IteratorUtils, Config, SheetUtils, FormulaUtils, Operand, FilterAggregator) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;
    var MathUtils = FormulaUtils.Math;
    var Scalar = FormulaUtils.Scalar;
    var Complex = FormulaUtils.Complex;
    var Matrix = FormulaUtils.Matrix;
    var Dimension = FormulaUtils.Dimension;

    // shortcuts to mathematical functions
    var isZero = MathUtils.isZero;

    // whether to disable timeout detection
    var DEBUG_FORMULAS = Config.getDebugUrlFlag('spreadsheet:debug-formulas');

    // private global functions ===============================================

    /**
     * Throws the first error code found in the passed matrix. If no error code
     * has been found, the passed matrix will be returned.
     *
     * @param {Matrix} matrix
     *  A matrix containing any scalar values used in formulas.
     *
     * @returns {Matrix}
     *  The passed matrix, if it does not contain an error code.
     *
     * @throws {ErrorCode}
     *  The first error code contained in the passed matrix.
     */
    function checkMatrixForErrorCode(matrix) {
        matrix.forEach(FormulaUtils.throwErrorCode);
        return matrix;
    }

    /**
     * Checks that all elements in the passed matrix are passing the specified
     * truth test. If any matrix element fails, the #VALUE! error code will be
     * thrown. If any matrix element is an error code, it will be thrown
     * instead. If all matrix elements are valid, the passed matrix will be
     * returned.
     *
     * @param {Matrix} matrix
     *  A matrix containing any scalar values used in formulas.
     *
     * @param {Function} callback
     *  The predicate callback function invoked for every matrix element.
     *
     * @returns {Matrix}
     *  The passed matrix, if all its elements are valid according to the test.
     *
     * @throws {ErrorCode}
     *  The first error code contained in the passed matrix, or the #VALUE!
     *  error code, if a matrix element does not pass the truth test.
     */
    function checkMatrixForDataType(matrix, callback) {
        matrix.forEach(function (elem) {
            FormulaUtils.throwErrorCode(elem);
            if (!callback(elem)) { throw ErrorCode.VALUE; }
        });
        return matrix;
    }

    /**
     * Checks that the all cell range addresses in the passed array refer to
     * the same single sheet.
     *
     * @returns {Range3DArray}
     *  The range array passed to this function, if it is valid.
     *
     * @throws {ErrorCode}
     *  - The #NULL! error code, if the range array is empty.
     *  - The #VALUE! error code, if the ranges refer to different sheets.
     */
    function checkSingleSheetRanges(ranges) {
        if (ranges.empty()) { throw ErrorCode.NULL; }
        if (!ranges.singleSheet()) { throw ErrorCode.VALUE; }
        return ranges;
    }

    /**
     * Returns the type of the passed operand.
     */
    function getOperandType(operand) {
        if (operand instanceof Operand) { return operand.getType(); }
        if (operand instanceof Matrix) { return 'mat'; }
        if ((operand instanceof Range3D) || (operand instanceof Range3DArray)) { return 'ref'; }
        return 'val';
    }

    /**
     * Returns the size of the passed operand.
     */
    function getOperandSize(operand) {
        var value = (operand instanceof Operand) ? operand.getRawValue() : operand;
        if (value instanceof Matrix) { return value.size(); }
        if ((value instanceof Range3D) || (value instanceof Range3DArray)) { return value.cells(); }
        return 1; // scalar value
    }

    /**
     * Returns the conversion mode from the passed options, according to the
     * type of the formula operand.
     */
    function getConvertMode(operand, options) {
        return Utils.getStringOption(options, getOperandType(operand) + 'Mode', 'convert');
    }

    /**
     * Creates an iterator that filters all null values provided by the passed
     * iterator, and adds the sequence index to the result.
     */
    function createValueIterator(iterator) {
        var index = 0;
        return IteratorUtils.createTransformIterator(iterator, function (value, result) {
            if (value === null) { return null; }
            result.index = index;
            index += 1;
            return result;
        });
    }

    /**
     * A static filter matcher callback function that matches non-blank cells
     * (any scalar value but null).
     *
     * @param {Any} value
     *  A scalar value to be matched by a data filter.
     *
     * @returns {Boolean}
     *  Whether the passed value is not null.
     */
    function scalarMatcher(value) {
        return value !== null;
    }

    /**
     * A static filter matcher callback function that matches blank cells
     * (represented by the value null).
     *
     * @param {Any} value
     *  A scalar value to be matched by a data filter.
     *
     * @returns {Boolean}
     *  Whether the passed value is null.
     */
    function blankMatcher(value) {
        return value === null;
    }

    /**
     * A static filter matcher callback function that matches blank cells
     * (represented by the value null), and cells with empty strings (may be
     * returned by formula cells).
     *
     * @param {Any} value
     *  A scalar value to be matched by a data filter.
     *
     * @returns {Boolean}
     *  Whether the passed value is null or an empty string.
     */
    function emptyMatcher(value) {
        return (value === '') || (value === null);
    }

    /**
     * A static filter matcher callback function that does not match anything.
     *
     * @returns {Boolean}
     *  The boolean value false.
     */
    var nothingMatcher = _.constant(false);

    // class FormulaContext ===================================================

    /**
     * An instance of this class serves as calling context object for the
     * implementations of operators and functions, so that the symbol 'this'
     * inside these implementations provides various useful helper methods.
     *
     * @constructor
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Number} [initOptions.refSheet]
     *      The index of the sheet the interpreted formula is related to. Used
     *      to resolve reference tokens without sheet reference. If omitted,
     *      reference tokens without sheet indexes will result in #REF! errors.
     *  @param {Address} [initOptions.targetAddress]
     *      The address of the target reference cell in the specified reference
     *      sheet the formula is related to.If omitted, cell A1 will be used as
     *      target reference cell.
     *  @param {Boolean} [initOptions.detectCircular=false]
     *      If set to true, trying to dereference a cell reference covering the
     *      target address leads to a circular dependeny error.
     *  @param {Boolean} [options.recalcDirty=false]
     *      If set to true, all dirty cell formulas resolved by cell reference
     *      operands will be recalculated recursively using the dependency
     *      manager of the document.
     *
     * @property {SpreadsheetApplication} app
     *  The spreadsheet application instance containing the document model with
     *  the formula currently calculated.
     *
     * @property {SpreadsheetModel} docModel
     *  The document model containing the formula currently calculated.
     *
     * @property {String} fileFormat
     *  The file format identifier. May be used to implement different behavior
     *  for functions and operators.
     *
     * @property {FormulaResource} formulaResource
     *  The formula resource containing localized configuration for formulas.
     *
     * @property {FormulaGrammar} formulaGrammar
     *  The formula grammar configuration for the current UI language, the
     *  current file format of the document, and the current reference style of
     *  the document (A1 or R1C1).
     *
     * @property {NumberFormatter} numberFormatter
     *  The number formatter of the document model.
     */
    function FormulaContext(docModel, initOptions) {

        // private properties -------------------------------------------------

        // the formula interpreter of the document, for further helper methods
        this._formulaInterpreter = docModel.getFormulaInterpreter();

        // the formula dependency manager of the document
        this._dependencyManager = docModel.getDependencyManager();

        // the reference sheet
        this._refSheet = Utils.getIntegerOption(initOptions, 'refSheet', null);

        // the source reference address (anchor of matrix formulas)
        this._refAddress = Utils.getOption(initOptions, 'targetAddress', Address.A1).clone();

        // the target reference address (address of current target element in a matrix formula)
        this._targetAddress = this._refAddress.clone();

        // whether to check the target address for circular reference errors
        this._detectCircular = Utils.getBooleanOption(initOptions, 'detectCircular', false);

        // whether to recalculate all dirty cell formulas resolved by cell references
        this._recalcDirty = Utils.getOption(initOptions, 'recalcDirty', false);

        // standard number format
        this._standardFormat = docModel.getNumberFormatter().getStandardFormat();

        // maximum length of a string, maximum valid string index (one-based)
        this._MAX_STRING_LEN = (docModel.getApp().getFileFormat() === 'odf') ? 65535 : 32767;

        // maximum time stamp after which interpreting a formula will fail (prevent freezing browser)
        this._END_EVAL_TIME = _.now() + FormulaUtils.MAX_EVAL_TIME;

        // operand resolvers of ancestor operators/functions
        this._resolverStack = [];

        // provides access to the array of operands of the current function/operator
        this._operandResolver = null;

        // current context type for an operator
        this._contextType = 'any';

        // dimension of the result matrix (in matrix context)
        this._matDim = null;

        // current row index while calculating the results of a matrix
        this._matRow = 0;

        // current column index while calculating the results of a matrix
        this._matCol = 0;

        // public properties --------------------------------------------------

        // the application instance
        this.app = docModel.getApp();

        // the spreadsheet document model
        this.docModel = docModel;

        // the file format identifier for different behavior of functions and operators
        this.fileFormat = this.app.getFileFormat();

        // the formula resources
        this.formulaResource = docModel.getFormulaResource();

        // the formula grammar for the UI language
        this.formulaGrammar = docModel.getFormulaGrammar('ui');

        // the number formatter
        this.numberFormatter = docModel.getNumberFormatter();

    } // class FormulaContext

    // private methods --------------------------------------------------------

    /**
     * Returns the scalar value from the passed operand, according to the
     * current element position of a matrix formula.
     *
     * @param {Operand} operand
     *  The operand to be converted to a scalar value.
     *
     * @returns {Any}
     *  The scalar value, if the operand can be resolved.
     *
     * @throws {ErrorCode}
     *  An internal error code carried by the operand.
     */
    FormulaContext.prototype._getScalar = function (operand) {
        return (this._contextType === 'mat') ? operand.getScalar(this._matRow, this._matCol) : operand.getScalar(null, null);
    };

    /**
     * Returns the value of the specified cell in the spreadsheet document.
     * Depending on the 'recalcDirty' flag passed to the constructor, the cell
     * value will be fetched directly from the cell collection, or will be
     * queried from the dependency manager which leads to recursive calculation
     * of pending dirty formulas.
     *
     * @param {SheetModel} sheetModel
     *  The model of the sheet containing the cell whose value is queried.
     *
     * @param {Address} address
     *  The address of the cell whose value is queried.
     *
     * @returns {Any}
     *  The value of the specified cell.
     */
    FormulaContext.prototype._getCellValue = function (sheetModel, address) {

        // default case: resolve current cell value from cell collection
        if (!this._recalcDirty) {
            return sheetModel.getCellCollection().getValue(address);
        }

        // resolve cell value via dependency manager by recalculating dirty formulas recursively
        var valueDesc = this._dependencyManager.getCellValue(sheetModel, address);
        // adjust own evaluation time limit (do not count the evaluation time of the dependent formulas)
        this._END_EVAL_TIME += valueDesc.time;
        // return the scalar value
        return valueDesc.value;
    };

    /**
     * Throws a CIRCULAR_ERROR error code, if the passed cell range address
     * contains the reference cell of this formula context.
     *
     * @param {Range3D} range
     *  The cell range address to be checked.
     *
     * @returns {Range3D}
     *  The passed cell range address, if it does not contain the reference
     *  cell.
     *
     * @throws {ErrorCode}
     *  The CIRCULAR_ERROR error code, if the passed cell range address
     *  contains the reference cell of this formula context.
     */
    FormulaContext.prototype._checkCircularReference = function (range) {
        if (this._detectCircular && _.isNumber(this._refSheet) && range.containsSheet(this._refSheet) && range.containsAddress(this._refAddress)) {
            throw FormulaUtils.CIRCULAR_ERROR;
        }
        return range;
    };

    /**
     * Creates an iterator that visits all scalar values contained in the
     * passed operand, matrix, or reference.
     *
     * @param {Any} operand
     *  The operand (scalar value of any supported data type, including null
     *  and error codes; or instances of the classes Matrix, Range3D, or
     *  Range3DArray; or a plain JS array of scalar values) whose contents will
     *  be visited.
     *
     * @param {Function} converter
     *  A callback function invoked for each scalar value contained in the
     *  operand. Will be called in the context of this instance. Receives the
     *  value as first parameter, and must return the converted scalar value.
     *  May throw a formula error code.
     *
     * @param {Object} [options]
     *  Optional parameters. See FormulaContext.createScalarIterator() for
     *  details.
     *
     * @returns {Object}
     *  An iterator that visits all scalar values contained in the passed
     *  operand. See FormulaContext.createScalarIterator() for details.
     */
    FormulaContext.prototype._createOperandIterator = function (operand, converter, options) {

        // self reference for local functions
        var self = this;
        // whether to visit or skip the empty function parameter
        var emptyParam = Utils.getBooleanOption(options, 'emptyParam', false);
        // whether to visit empty cells in references
        var emptyCells = Utils.getBooleanOption(options, 'emptyCells', false);
        // whether to pass error codes to the iterator
        var acceptErrors = Utils.getBooleanOption(options, 'acceptErrors', false);
        // whether to collect and provide the number formats of the operand
        var collectFormats = Utils.getOption(options, 'collectFormats', false);
        // whether to visit visible rows only
        var skipHiddenRows = Utils.getBooleanOption(options, 'skipHiddenRows', false);
        // whether to skip hidden rows in a sheet with active filter
        var skipFilteredRows = Utils.getBooleanOption(options, 'skipFilteredRows', false);
        // counter for time check in large loops
        var evalCounter = 0;

        function createResult(value, offset, width, col, row, format) {
            // throw error codes, unless error codes will be accepted
            if (!acceptErrors) { FormulaUtils.throwErrorCode(value); }
            // check total evaluation time (throws if formula evaluation takes too long)
            if ((evalCounter += 1) % 50 === 0) { self.checkEvalTime(); }
            // convert the passed value (may throw an error code)
            value = converter.call(self, value);
            // fall-back to standard number format
            if (!format && collectFormats) { format = self._standardFormat; }
            // return the iterator result object
            return { value: value, format: format, offset: offset + width * row + col, col: col, row: row };
        }

        // creates an iterator that visits a scalar value (skip empty parameters unless specified otherwise)
        function createScalarIterator(value) {
            return (emptyParam || (value !== null)) ?
                IteratorUtils.createSingleIterator(createResult(value, 0, 1, 0, 0, null)) :
                IteratorUtils.ALWAYS_DONE_ITERATOR;
        }

        // creates an iterator that visits all elements of a matrix
        function createMatrixIterator(matrix) {
            var width = matrix.cols();
            return IteratorUtils.createTransformIterator(matrix.iterator(), function (value, result) {
                return createResult(value, 0, width, result.col, result.row, null);
            });
        }

        // creates an iterator that visits the cells in the passed cell range address
        function createRangeIterator(sheetModel, range, offset) {

            // start position of the iteration range, and its width (needed for relative indexes)
            var col = range.start[0], row = range.start[1], width = range.cols();
            // the cell collection of the visited sheet
            var cellCollection = sheetModel.getCellCollection();
            // whether to skip cells in hidden rows
            var visibleMode = (skipHiddenRows || (skipFilteredRows && sheetModel.getTableCollection().hasFilteredTables())) ? 'rows' : false;
            // iterate all content cells in the cell collection (skip empty cells, unless the option is set)
            var addressIt = cellCollection.createAddressIterator(range, { type: emptyCells ? 'all' : 'value', covered: true, visible: visibleMode });
            // create an iterator that creates the expected result objects (cell value, and relative column/row offsets)
            return IteratorUtils.createTransformIterator(addressIt, function (address) {
                var parsedFormat = collectFormats ? cellCollection.getParsedFormat(address) : null;
                return createResult(self._getCellValue(sheetModel, address), offset, width, address[0] - col, address[1] - row, parsedFormat);
            });
        }

        // creates an iterator that visits the cells in the passed 3D cell range address
        function createRange3DIterator(range3d, offset) {

            // convert the cell range to an instance of class Range
            var range2d = range3d.toRange();
            // number of cells per single sheet
            var cellCount = range2d.cells();

            // visit all worksheets covered by the 3D cell range
            var iterator = self.docModel.createSheetIntervalIterator(range3d.sheet1, range3d.sheet2, { types: 'worksheet' });
            return IteratorUtils.createNestedIterator(iterator, function (sheetModel) {
                var thisOffset = offset;
                offset += cellCount;
                return createRangeIterator(sheetModel, range2d, thisOffset);
            });
        }

        // creates an iterator that visits the cells in a range, or an array of ranges
        function createReferenceIterator(ranges) {

            if (Utils.getBooleanOption(options, 'complexRef', false)) {
                var arrayIt = Range3DArray.iterator(ranges);
                var offset = 0;
                return IteratorUtils.createNestedIterator(arrayIt, function (range3d) {
                    self._checkCircularReference(range3d);
                    var thisOffset = offset;
                    offset += range3d.cells();
                    return createRange3DIterator(range3d, thisOffset);
                });
            }

            return createRange3DIterator(self.convertToValueRange(ranges), 0);
        }

        // process a real operand (instance of class Operand)
        if (operand instanceof Operand) {
            var value = operand.getRawValue();
            switch (operand.getType()) {
                case 'val': return createScalarIterator(value);
                case 'mat': return createMatrixIterator(value);
                case 'ref': return createReferenceIterator(value);
            }
            FormulaUtils.throwInternal('FormulaContext._createOperandIterator(): invalid operand type');
        }

        // passed operand can also be a matrix, a range address, or an array of range addresses
        if (operand instanceof Matrix) { return createMatrixIterator(operand); }
        if ((operand instanceof Range3D) || (operand instanceof Range3DArray)) { return createReferenceIterator(operand); }

        // scalar value: create a single iterator; missing operand: return an empty iterator
        return _.isUndefined(operand) ? IteratorUtils.ALWAYS_DONE_ITERATOR : createScalarIterator(operand);
    };

    /**
     * Invokes a callback function for an operand iterator.
     *
     * @returns {Number}
     *  The number of values visited by the iterator.
     */
    FormulaContext.prototype._iterateValues = function (iterator, callback) {
        var count = 0;
        IteratorUtils.forEach(iterator, function (value, result) {
            callback.call(this, value, count, result.offset, result.col, result.row, result.format);
            count += 1;
        }, this);
        return count;
    };

    /**
     * Implementation helper for aggregators over multiple operands.
     */
    FormulaContext.prototype._aggregateValues = function (operands, initial, iterate, aggregate, finalize, options) {

        // the intermediate result
        var resultValue = initial;
        // the total counter for the visited values across multiple operands
        var valueCount = 0;
        // the total counter for the offset across multiple operands
        var offsetCount = 0;
        // whether to collect and aggregate the number formats of the operands
        var collectFormats = Utils.getBooleanOption(options, 'collectFormats', true);
        // the intermediate parsed number format
        var resultFormat = collectFormats ? this._standardFormat : null;

        // process all passed operands ('operands' may be an instance of Arguments)
        _.each(operands, function (operand) {
            valueCount += iterate.call(this, operand, function (value, index, offset, col, row, format) {
                var currIndex = valueCount + index;
                var currOffset = offsetCount + offset;
                resultValue = aggregate.call(this, resultValue, value, currIndex, currOffset);
                if (resultFormat) { resultFormat = FormulaUtils.combineParsedFormats(resultFormat, format); }
            }, options);
            offsetCount += getOperandSize(operand);
        }, this);

        // resolve the final result, immediately throw error codes
        resultValue = FormulaUtils.throwErrorCode(finalize.call(this, resultValue, valueCount, offsetCount));

        // return an operator instance, if format categories have been aggregated
        return resultFormat ? this.createOperand(resultValue, { format: resultFormat }) : resultValue;
    };

    // protected methods ------------------------------------------------------

    /**
     * Sets the specified element position inside a matrix formula.
     *
     * @param {Number} matRow
     *  The row index in the matrix context.
     *
     * @param {Number} matCol
     *  The column index in the matrix context.
     *
     * @returns {FormulaContext}
     *  A reference to this instance.
     */
    FormulaContext.prototype.setMatrixPosition = function (matRow, matCol) {
        this._targetAddress[0] += (matCol - this._matCol);
        this._targetAddress[1] += (matRow - this._matRow);
        this._matRow = matRow;
        this._matCol = matCol;
        return this;
    };

    /**
     * Registers the operands of the next function or operator to be resolved.
     *
     * @attention
     *  This method will be invoked by the formula interpreter to be able to
     *  reuse this context instance for all functions and operators contained
     *  in a complete formula.
     *
     * @param {OperandResolver} resolver
     *  A resolver that provides access to the operands of the function or
     *  operator that will be processed next.
     *
     * @param {String} contextType
     *  The context type used to evaluate an operator.
     *
     * @returns {FormulaContext}
     *  A reference to this instance.
     */
    FormulaContext.prototype.pushOperandResolver = function (resolver, contextType, matrixDim) {
        this._resolverStack.push({ resolver: this._operandResolver, contextType: this._contextType, matDim: this._matDim, matRow: this._matRow, matCol: this._matCol });
        this._operandResolver = resolver;
        this._contextType = contextType;
        this._matDim = matrixDim;
        return this.setMatrixPosition(null, null);
    };

    /**
     * Removes the last operand resolver from the inetrnal stack that has been
     * registered with the method FormulaContext.pushOperandResolver().
     *
     * @returns {FormulaContext}
     *  A reference to this instance.
     */
    FormulaContext.prototype.popOperandResolver = function () {
        var entry = this._resolverStack.pop();
        this._operandResolver = entry.resolver;
        this._contextType = entry.contextType;
        this._matDim = entry.matDim;
        return this.setMatrixPosition(entry.matRow, entry.matCol);
    };

    // data type conversion ---------------------------------------------------

    /**
     * Returns the context type that will be used to evaluate the current
     * function or operator.
     *
     * @returns {String}
     *  The current context type.
     */
    FormulaContext.prototype.getContextType = function () {
        return this._contextType;
    };

    /**
     * Tries to convert the passed scalar value to a floating-point number.
     *
     * @param {Any} value
     *  A scalar value to be converted to a number. Dates will be converted to
     *  numbers. Strings that represent a valid number (according to the
     *  current GUI language) will be converted to the number. The boolean
     *  value FALSE will be converted to 0, the boolean value TRUE will be
     *  converted to 1. The special value null (representing an empty cell)
     *  will be converted to 0.
     *
     * @param {Boolean} [floor=false]
     *  If set to true, the number will be rounded down to the next integer
     *  (negative numbers will be rounded down too, i.e. away from zero!).
     *
     * @returns {Number}
     *  The floating-point number, if available.
     *
     * @throws {ErrorCode}
     *  If the passed value is an error code, it will be thrown. If the value
     *  cannot be converted to a floating-point number, a #VALUE! error will be
     *  thrown.
     */
    FormulaContext.prototype.convertToNumber = function (value, floor) {

        // immediately throw error codes
        FormulaUtils.throwErrorCode(value);

        // convert the scalar value to a number
        value = this.numberFormatter.convertScalarToNumber(value, floor);

        // throw #VALUE! error code, if conversion has failed
        if (value === null) { throw ErrorCode.VALUE; }
        return value;
    };

    /**
     * Tries to convert the passed scalar value to a date.
     *
     * @param {Any} value
     *  A scalar value to be converted to a date. Strings that represent a
     *  valid number (according to the current GUI language) will be converted
     *  to a date representing that number. The boolean value FALSE will be
     *  converted to the null date of the number formatter, the boolean value
     *  TRUE will be converted to the day following the null date. The special
     *  value null (representing an empty cell) will be converted to the null
     *  date too.
     *
     * @param {Boolean} [floor=false]
     *  If set to true, the time components of the resulting date will be
     *  removed (the time will be set to midnight).
     *
     * @returns {Date}
     *  The UTC date, if available.
     *
     * @throws {ErrorCode}
     *  If the passed value is an error code, it will be thrown. If the value
     *  cannot be converted to a date value, a #VALUE! error will be thrown.
     */
    FormulaContext.prototype.convertToDate = function (value, floor) {

        // immediately throw error codes
        FormulaUtils.throwErrorCode(value);

        // convert the scalar value to a date
        value = this.numberFormatter.convertScalarToDate(value, floor);

        // throw #VALUE! error code, if conversion has failed
        if (value === null) { throw ErrorCode.VALUE; }
        return value;
    };

    /**
     * Tries to convert the passed scalar value to a string.
     *
     * @param {Any} value
     *  A scalar value to be converted to a string. Numbers and dates will be
     *  converted to decimal or scientific notation (according to the current
     *  GUI language), boolean values will be converted to their localized text
     *  representation (formulas will always be calculated using the current
     *  GUI language). The special value null (representing an empty cell) will
     *  be converted to the empty string.
     *
     * @returns {String}
     *  The string representation of the value, if available.
     *
     * @throws {ErrorCode}
     *  If the passed value is an error code, it will be thrown. All other
     *  values will be converted to strings.
     */
    FormulaContext.prototype.convertToString = function (value) {

        // immediately throw error codes
        FormulaUtils.throwErrorCode(value);

        // convert the scalar value to a string
        value = this.numberFormatter.convertScalarToString(value, SheetUtils.MAX_LENGTH_STANDARD_FORMULA);

        // throw #VALUE! error code, if conversion has failed
        if (value === null) { throw ErrorCode.VALUE; }
        return value;
    };

    /**
     * Tries to convert the passed scalar value to a boolean value.
     *
     * @param {Any} value
     *  A scalar value to be converted to a boolean. Floating-point numbers
     *  will be converted to TRUE if not zero, otherwise FALSE. The null date
     *  will be converted to FALSE, all other dates will be converted to TRUE.
     *  Strings containing the exact translated name of the TRUE or FALSE
     *  values (case-insensitive) will be converted to the respective boolean
     *  value (formulas will always be calculated using the current GUI
     *  language). The special value null (representing an empty cell) will be
     *  converted to FALSE.
     *
     * @returns {Boolean}
     *  The boolean value, if available.
     *
     * @throws {ErrorCode}
     *  If the passed value is an error code, it will be thrown. If the value
     *  cannot be converted to a boolean, a #VALUE! error will be thrown.
     */
    FormulaContext.prototype.convertToBoolean = function (value) {

        // immediately throw error codes
        FormulaUtils.throwErrorCode(value);

        // convert the scalar value to a boolean
        value = this.numberFormatter.convertScalarToBoolean(value);

        // throw #VALUE! error code, if conversion has failed
        if (value === null) { throw ErrorCode.VALUE; }
        return value;
    };

    /**
     * Tries to convert the passed scalar value to a complex number.
     *
     * @param {Any} value
     *  A scalar value to be converted to a complex number. Floating-point
     *  numbers will be converted to complex numbers with the imaginary
     *  coefficient set to zero. Strings must represent a valid complex number
     *  (optional leading signed real coefficient, optional trailing signed
     *  imaginary coefficient, followed by the lower-case character 'i' or
     *  'j'). The coefficients must containing the decimal separator of the
     *  current GUI language. The special value null (representing an empty
     *  cell) will be converted to the complex number 0.
     *
     * @returns {Complex}
     *  The complex number, if available.
     *
     * @throws {ErrorCode}
     *  If the passed value is an error code, it will be thrown. Boolean values
     *  will cause to throw a #VALUE! error. If the string cannot be converted
     *  to a complex number, a #NUM! error will be thrown (not the #VALUE!
     *  error code as thrown by the other conversion methods).
     */
    FormulaContext.prototype.convertToComplex = function (value) {

        // immediately throw error codes
        FormulaUtils.throwErrorCode(value);

        // boolean values result in #VALUE! (but not invalid strings)
        if (_.isBoolean(value)) { throw ErrorCode.VALUE; }

        // treat date objects as plain unformatted numbers
        if (value instanceof Date) {
            value = this.convertToNumber(value);
        }

        // convert numbers and strings to complex numbers
        if (_.isNumber(value) && _.isFinite(value)) {
            value = new Complex(value, 0);
        } else if (_.isString(value)) {
            value = this.numberFormatter.convertStringToComplex(value);
        } else if (value === null) {
            // empty cells are treated as complex number 0 (in difference to empty strings)
            value = new Complex(0, 0);
        }

        // the resulting value must be a complex number
        if (value instanceof Complex) { return value; }
        // invalid strings will result in #NUM! instead of #VALUE!
        throw ErrorCode.NUM;
    };

    /**
     * Extracts a single cell range from the passed array of cell range
     * addresses. The range array must contain exactly one cell range address.
     * Unless specified via an option, the cell range must refer to a single
     * sheet.
     *
     * @param {Range3DArray|Range3D} ranges
     *  An array of cell range addresses, or a single cell range address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.multiSheet=false]
     *      If set to true, the cell range address may refer to multiple
     *      sheets.
     *  @param {ErrorCode} [options.errorCode=ErrorCode.REF]
     *      A specific error code that will be thrown if the passed range array
     *      contains more than one cell range address. If omitted, the #REF!
     *      error code will be used.
     *
     * @returns {Range3D}
     *  The cell range address, if available.
     *
     * @throws {ErrorCode}
     *  - The #NULL! error code, if the range array is empty.
     *  - The #REF! or another error code (see option 'errorCode'), if the
     *  range array consists of more than one element.
     *  - The #VALUE! error code, if the option 'multiSheet' has not been set,
     *      and a single cell range address is available but refers to multiple
     *      sheets.
     */
    FormulaContext.prototype.convertToRange = function (ranges, options) {

        // convert parameter to an array
        ranges = Range3DArray.get(ranges);

        // must not be empty
        if (ranges.empty()) { throw ErrorCode.NULL; }

        // must be a single range in a single sheet
        if (ranges.length > 1) {
            throw Utils.getOption(options, 'errorCode', ErrorCode.REF);
        }

        // must point to a single sheet, unless multiple sheets are allowed (always throw the #VALUE! error code)
        if (!Utils.getBooleanOption(options, 'multiSheet', false) && !ranges[0].singleSheet()) {
            throw ErrorCode.VALUE;
        }

        return ranges[0];
    };

    /**
     * Extracts a single cell range from the passed array of cell range
     * addresses, intended to be resolved to the cell contents in that range.
     * The range array must contain exactly one cell range address pointing to
     * a single sheet, and the reference cell of this formula context must not
     * be part of that range.
     *
     * @param {Range3DArray|Range3D} ranges
     *  An array of cell range addresses, or a single cell range address.
     *
     * @returns {Range3D}
     *  The resulting cell range address, if available. The sheet indexes in
     *  that range will be equal.
     *
     * @throws {ErrorCode}
     *  - The #NULL! error code, if the range array is empty.
     *  - The #VALUE! error code, if the range array contains more than one
     *      element, or if the only range refers to multiple sheets.
     *  - The CIRCULAR_ERROR error code, if a single range is available, but
     *      the reference cell is part of that range (a circular reference).
     */
    FormulaContext.prototype.convertToValueRange = function (ranges) {
        // resolve to a single range (throw #VALUE! on multiple ranges), detect circular references
        return this._checkCircularReference(this.convertToRange(ranges, { errorCode: ErrorCode.VALUE }));
    };

    /**
     * Converts the specified operand according to the passed data type
     * specifier as used in the operator/function signature.
     *
     * @param {Operand} operand
     *  The zero-based index of an operand.
     *
     * @param {String} typeSpec
     *  The type specifier used to decide how to convert the operand value.
     *
     * @returns {Any}
     *  The resulting operand value, if available, according to the passed type
     *  specifier.
     *
     * @throws {ErrorCode}
     *  Any error that occured while trying to convert the operand value.
     */
    FormulaContext.prototype.convertOperandToValue = function (operand, typeSpec) {
        switch (typeSpec) {
            case 'any':
            case 'any:lazy':
                return operand;
            case 'val':
                return FormulaUtils.throwErrorCode(this._getScalar(operand));
            case 'val:num':
                return this.convertToNumber(this._getScalar(operand));
            case 'val:int':
                return this.convertToNumber(this._getScalar(operand), true);
            case 'val:date':
                return this.convertToDate(this._getScalar(operand));
            case 'val:day':
                return this.convertToDate(this._getScalar(operand), true);
            case 'val:str':
                return this.convertToString(this._getScalar(operand));
            case 'val:bool':
                return this.convertToBoolean(this._getScalar(operand));
            case 'val:comp':
                return this.convertToComplex(this._getScalar(operand));
            case 'val:any':
                return this._getScalar(operand);
            case 'mat':
                return checkMatrixForErrorCode(operand.getMatrix());
            case 'mat:num':
                // no conversion from other data types in matrix parameters!
                return checkMatrixForDataType(operand.getMatrix(), _.isNumber);
            case 'mat:str':
                // no conversion from other data types in matrix parameters!
                return checkMatrixForDataType(operand.getMatrix(), _.isString);
            case 'mat:bool':
                // no conversion from other data types in matrix parameters!
                return checkMatrixForDataType(operand.getMatrix(), _.isBoolean);
            case 'mat:any':
                return operand.getMatrix();
            case 'ref':
                return operand.getRanges();
            case 'ref:sheet':
                return checkSingleSheetRanges(operand.getRanges());
            case 'ref:single':
                return this.convertToRange(operand.getRanges());
            case 'ref:multi':
                return this.convertToRange(operand.getRanges(), { multiSheet: true });
            case 'ref:val':
                return this.convertToValueRange(operand.getRanges());
        }
        FormulaUtils.throwInternal('FormulaContext.convertOperandToValue(): unknown type specifier: "' + typeSpec + '"');
    };

    /**
     * Returns whether the passed scalar values are considered to be equal
     * according to the current file format. For ODF documents, the character
     * case of strings matters. Provided to help implementing the comparison
     * operators.
     *
     * @param {Number|String|Boolean|Null} value1
     *  The first value to be compared.
     *
     * @param {Number|String|Boolean|Null} value2
     *  The second value to be compared.
     *
     * @returns {Boolean}
     *  Whether the passed scalar values are considered to be equal.
     */
    FormulaContext.prototype.equalScalars = function (value1, value2) {
        return this._formulaInterpreter.equalScalars(value1, value2);
    };

    /**
     * Compares the passed scalar values according to the current file format.
     * For ODF documents, the character case of strings matters. Provided to
     * help implementing the comparison operators.
     *
     * @param {Number|String|Boolean|Null} value1
     *  The first value to be compared.
     *
     * @param {Number|String|Boolean|Null} value2
     *  The second value to be compared.
     *
     * @returns {Number}
     *  A negative value, if value1 is less than value2; or a positive value,
     *  if value1 is greater than value2; or zero, if both values are of the
     *  same type and are equal. See Scalar.compare() for more details.
     */
    FormulaContext.prototype.compareScalars = function (value1, value2) {
        return this._formulaInterpreter.compareScalars(value1, value2);
    };

    /**
     * Converts the passed scalar value to be stored in an operand. A date will
     * be converted to a floating-point number, if possible. If the value is
     * INF or NaN, the #NUM! error code will be returned instead. If the number
     * is too close to zero (using a denormalized binary representation
     * internally), zero will be returned. If a string is too long, a #VALUE!
     * error code will be returned. Complex numbers will be converted to their
     * string representation, unless one of the coefficients is INF or NaN
     * which will result in the #NUM! error too.
     *
     * @param {Any} value
     *  The value to be validated.
     *
     * @returns {Any}
     *  The validated value, ready to be inserted into a formula operand.
     */
    FormulaContext.prototype.validateValue = function (value) {

        // convert date object to floating-point number
        if (value instanceof Date) {
            value = this.numberFormatter.convertDateToNumber(value);
            return _.isNumber(value) ? value : ErrorCode.NUM;
        }

        // convert complex numbers to strings
        if (value instanceof Complex) {
            // convert INF or NaN to #NUM! error
            if (!_.isFinite(value.real) || !_.isFinite(value.imag)) { return ErrorCode.NUM; }
            // convert very small (denormalized) numbers to zero
            if (isZero(value.real)) { value.real = 0; }
            if (isZero(value.imag)) { value.imag = 0; }
            // convert to string
            value = this.numberFormatter.convertComplexToString(value, SheetUtils.MAX_LENGTH_STANDARD_FORMULA);
        }

        // validate floating-point numbers
        if (_.isNumber(value)) {
            // convert INF or NaN to #NUM! error
            if (!_.isFinite(value)) { return ErrorCode.NUM; }
            // convert very small (denormalized) numbers to zero
            return isZero(value) ? 0 : value;
        }

        // validate strings
        if (_.isString(value)) {
            // check maximum length of string (depends on file format)
            return (value.length <= this._MAX_STRING_LEN) ? value : ErrorCode.VALUE;
        }

        // no conversion for other values
        return value;
    };

    /**
     * Checks the passed string length used in string functions. Throws the
     * #VALUE! error code, if the index is less than zero, or larger than the
     * maximum length of string results supported by the formula engine.
     *
     * @param {Number} length
     *  The string length to be checked.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the passed string length is invalid.
     */
    FormulaContext.prototype.checkStringLength = function (length) {
        if ((length < 0) || (length > this._MAX_STRING_LEN)) { throw ErrorCode.VALUE; }
    };

    /**
     * Checks the passed character index used in string functions. Throws the
     * #VALUE! error code, if the index is less than one, or larger than the
     * maximum length of string results supported by the formula engine.
     *
     * @param {Number} index
     *  The character index to be checked.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the passed character index is invalid.
     */
    FormulaContext.prototype.checkStringIndex = function (index) {
        if ((index < 1) || (index > this._MAX_STRING_LEN)) { throw ErrorCode.VALUE; }
    };

    /**
     * Concatenates the passed strings, and validates the length of the
     * resulting string.
     *
     * @param {String} ...
     *  The strings to be concatenated.
     *
     * @returns {String}
     *  The concatenation of the passed strings.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the length of the resulting string would be
     *  invalid.
     */
    FormulaContext.prototype.concatStrings = function () {
        var result = _.reduce(arguments, function (str1, str2) { return str1 + str2; }, '');
        this.checkStringLength(result.length);
        return result;
    };

    /**
     * Repeats the passed string, and validates the length of the resulting
     * string.
     *
     * @param {String} str
     *  The string to be repeated.
     *
     * @param {Number} repeat
     *  The number of repetitions.
     *
     * @returns {String}
     *  The resulting repeated string.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the length of the resulting string would be
     *  invalid.
     */
    FormulaContext.prototype.repeatString = function (str, repeat) {
        this.checkStringLength(str.length * repeat);
        return (str.length === 0) ? '' : Utils.repeatString(str, repeat);
    };

    /**
     * Checks whether the passed cell reference structure is valid.
     *
     * @param {CellRef} cellRef
     *  The cell reference structure to be checked.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the passed cell reference is not valid.
     */
    FormulaContext.prototype.checkCellRef = function (cellRef) {
        if (!this.docModel.isValidAddress(cellRef.toAddress())) { throw ErrorCode.VALUE; }
    };

    /**
     * Checks whether the passed cell range addresses have equal dimensions
     * (width and height, ignoring the sheet indexes of 3D ranges).
     *
     * @param {Range|Range3D} range1
     *  The first cell range to be checked.
     *
     * @param {Range|Range3D} range2
     *  The second cell range to be checked.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the dimensions (width and height) of the
     *  passed cell range addresses are different.
     */
    FormulaContext.prototype.checkRangeDims = function (range1, range2) {
        if (!Range.prototype.equalSize.call(range1, range2)) { throw ErrorCode.VALUE; }
    };

    /**
     * Checks the elapsed time since this formula context instance has been
     * created. If the duration exceeds the limit set in the constant
     * FormulaUtils.MAX_EVAL_TIME, an UNSUPPORTED error will be thrown.
     *
     * @throws {ErrorCode}
     * The UNSUPPORTED error code, if the duration exceeds the limit set in the
     * constant FormulaUtils.MAX_EVAL_TIME.
     */
    FormulaContext.prototype.checkEvalTime = DEBUG_FORMULAS ? _.noop : function () {
        if (_.now() > this._END_EVAL_TIME) { throw FormulaUtils.TIMEOUT_ERROR; }
    };

    // operand access ---------------------------------------------------------

    /**
     * Returns the number of existing operands passed to the operator or
     * function.
     *
     * @returns {Number}
     *  The number of existing operands.
     */
    FormulaContext.prototype.getOperandCount = function () {
        return this._operandResolver.size();
    };

    /**
     * Returns the specified operand passed to the operator or function.
     *
     * @param {Number} index
     *  The zero-based index of an operand.
     *
     * @param {String} [typeSpec='any']
     *  The type specifier used to decide how to convert the operand value. If
     *  omitted, or set to 'any', the operand will be returned as-is (as
     *  instance of the class Operand).
     *
     * @returns {Any}
     *  The specified operand, if available, according to the passed type
     *  specifier.
     *
     * @throws {ErrorCode}
     *  The #N/A error code, if the operand does not exist, or any other error
     *  code occurred during conversion of the operand value.
     */
    FormulaContext.prototype.getOperand = function (index, typeSpec) {
        var operand = this._operandResolver.resolve(index);
        if (!operand) { throw ErrorCode.NA; }
        return typeSpec ? this.convertOperandToValue(operand, typeSpec) : operand;
    };

    /**
     * Returns the raw value of the specified operand passed to the operator or
     * function.
     *
     * @param {Number} index
     *  The zero-based index of an operand.
     *
     * @returns {Any}
     *  The raw value of the specified operand, if available, including literal
     *  error codes. See method Operand.getRawValue() for details.
     *
     * @throws {ErrorCode}
     *  The #N/A error code, if the operand does not exist.
     */
    FormulaContext.prototype.getOperandValue = function (index) {
        return this.getOperand(index).getRawValue();
    };

    /**
     * Returns whether the specified operand is missing (the optional
     * parameters at the end of a function; e.g. the third parameter in the
     * formula =SUM(1,2) is missing).
     *
     * @param {Number} index
     *  The zero-based index of an operand.
     *
     * @returns {Boolean}
     *  Whether the specified operand is missing. Returns false for existing
     *  but empty operands, e.g. in the formula =SUM(1,,2).
     */
    FormulaContext.prototype.isMissingOperand = function (index) {
        return index >= this._operandResolver.size();
    };

    /**
     * Returns whether the specified operand exists, but is empty (e.g. the
     * second operand in the formula =SUM(1,,2) is empty).
     *
     * @param {Number} index
     *  The zero-based index of an operand.
     *
     * @returns {Boolean}
     *  Whether the specified operand exists but is empty. Returns false for
     *  missing operands (optional parameters at the end of the function).
     *  Returns false for references to empty cells too (the operand is a
     *  reference, and not empty).
     */
    FormulaContext.prototype.isEmptyOperand = function (index) {
        var operand = this._operandResolver.resolve(index);
        return !!operand && operand.isEmpty();
    };

    /**
     * Returns whether the specified operand is missing (the optional
     * parameters at the end of a function; e.g. the third parameter in the
     * formula =SUM(1,2) is missing), or empty (e.g. the second operand in the
     * formula =SUM(1,,2) is empty).
     *
     * @param {Number} index
     *  The zero-based index of an operand.
     *
     * @returns {Boolean}
     *  Whether the specified operand is missing or empty.
     */
    FormulaContext.prototype.isMissingOrEmptyOperand = function (index) {
        return this.isMissingOperand(index) || this.isEmptyOperand(index);
    };

    /**
     * Returns the specified operands passed to the operator or function as
     * array.
     *
     * @param {Number} first
     *  The zero-based index of the first operand to be returned.
     *
     * @param {Number} [length]
     *  The number of operands to be returned. Must be a positive integer. If
     *  this value would exceed the number of available operands, a #VALUE!
     *  error code will be thrown. If omitted, all remaining existing operands
     *  will be returned.
     *
     * @returns {Array<Operand>}
     *  The specified operand and its successors, if existing. This array will
     *  never be empty.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the specified operands do not exist.
     */
    FormulaContext.prototype.getOperands = function (first, length) {
        var resolver = this._operandResolver;
        var count = resolver.size() - first;
        if (length) { count = Math.min(count, length); }
        if (count < 1) { throw ErrorCode.VALUE; }
        return _.times(count, function (index) {
            return resolver.resolve(first + index);
        });
    };

    /**
     * Visits the existing operands of the processed operator or function.
     *
     * @param {Number} first
     *  Specifies the zero-based index of the first operand to be visited.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each operand. Receives
     *  the following parameters:
     *  (1) {Operand} operand
     *      The original operand instance.
     *  (2) {Number} index
     *      The zero-based index of the operand.
     *  The callback function will be invoked in the context of this instance.
     *
     * @returns {OperandsMixin}
     *  A reference to this instance.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the specified operand does not exist.
     */
    FormulaContext.prototype.iterateOperands = function (first, callback) {
        this.getOperands(first).forEach(function (operand, index) {
            callback.call(this, operand, first + index);
        }, this);
        return this;
    };

    /**
     * Creates a new instance of the class Operand. An operand is an object
     * that carries formula result values of any data type.
     *
     * @param {Any} value
     *  The value to be stored as operand. Can be any value supported by the
     *  constructor of the class Operand.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options that are supported by the
     *  constructor of the class Operand.
     *
     * @returns {Operand}
     *  A new operand with the passed value and options.
     */
    FormulaContext.prototype.createOperand = function (value, options) {
        return new Operand(this, value, options);
    };

    // document access --------------------------------------------------------

    /**
     * Returns whether the passed sheet index is equal to the reference sheet
     * the interpreted formula is located in.
     *
     * @param {Number} sheet
     *  A zero-based sheet index to be tested against the reference sheet.
     *
     * @returns {Boolean}
     *  Whether the passed sheet index is equal to the reference sheet.
     */
    FormulaContext.prototype.isRefSheet = function (sheet) {
        var refSheet = this._refSheet;
        return (refSheet !== null) && (refSheet >= 0) && (refSheet === sheet);
    };

    /**
     * Returns the index of the reference sheet the interpreted formula is
     * located in.
     *
     * @returns {Number}
     *  The zero-based index of the reference sheet.
     *
     * @throws {ErrorCode}
     *  The #REF! error code, if the formula is being interpreted without a
     *  reference sheet.
     */
    FormulaContext.prototype.getRefSheet = function () {
        if (typeof this._refSheet !== 'number') { throw ErrorCode.REF; }
        return this._refSheet;
    };

    /**
     * Returns the model of the reference sheet the interpreted formula is
     * located in.
     *
     * @returns {SheetModel}
     *  The model instance of the reference sheet.
     *
     * @throws {ErrorCode}
     *  The #REF! error code, if the formula is being interpreted without a
     *  reference sheet.
     */
    FormulaContext.prototype.getRefSheetModel = function () {
        return this.docModel.getSheetModel(this.getRefSheet());
    };

    /**
     * Returns the address of the source reference cell the interpreted formula
     * is located in. In a matrix formula, this is the address of its anchor
     * cell.
     *
     * @returns {Address}
     *  The address of the source reference cell.
     */
    FormulaContext.prototype.getRefAddress = function () {
        return this._refAddress.clone();
    };

    /**
     * Returns whether the passed cell address is equal to the address of the
     * target reference cell the interpreted formula is located in.
     *
     * @param {Address} address
     *  A cell address to be tested against the target reference address.
     *
     * @returns {Boolean}
     *  Whether the passed cell address is equal to the address of the target
     *  reference cell.
     */
    FormulaContext.prototype.isTargetAddress = function (address) {
        return this._targetAddress.equals(address);
    };

    /**
     * Returns the address of the target reference cell the interpreted formula
     * is located in. May differ from the source reference address in matrix
     * formulas while calculating the single elements of a result matrix.
     *
     * @returns {Address}
     *  The address of the target reference cell.
     */
    FormulaContext.prototype.getTargetAddress = function () {
        return this._targetAddress.clone();
    };

    /**
     * Returns the address of the cell in the passed cell range, according to
     * the current element position in matrix context.
     *
     * @param {Range} range
     *  The address of a cell range.
     *
     * @returns {Address}
     *  The address of the cell in the passed range, according to the current
     *  element position in matrix context.
     *
     * @throws {ErrorCode}
     *  The #N/A error code, if the current element position exceeds the size
     *  of the passed cell range address.
     */
    FormulaContext.prototype.getMatrixAddress = function (range) {
        return Matrix.getAddress(range, this._matRow, this._matCol);
    };

    /**
     * Returns the current value (or formula result) of the specified cell.
     *
     * @param {Number} sheet
     *  The zero-based index of the sheet.
     *
     * @param {Address} address
     *  The address of the cell in the specified sheet.
     *
     * @returns {Number|String|Boolean|ErrorCode|Null}
     *  The current value (or formula result) of the specified cell.
     *
     * @throws {ErrorCode}
     *  The #REF! error code, if the passed sheet index is invalid.
     */
    FormulaContext.prototype.getCellValue = function (sheet, address) {

        // the sheet model containing the cell (safety check: #REF! error if sheet is invalid)
        var sheetModel = this.docModel.getSheetModel(sheet);
        if (!sheetModel) { throw ErrorCode.REF; }

        // return typed value from cell collection
        return this._getCellValue(sheetModel, address);
    };

    /**
     * Returns the current values (or formula results) of all cells in the
     * specified cell range as matrix.
     *
     * @param {Number} sheet
     *  The zero-based index of the sheet.
     *
     * @param {Range} range
     *  The address of the cell range in the specified sheet.
     *
     * @returns {Matrix}
     *  The matrix containing all values in the specified cell range.
     *
     * @throws {ErrorCode}
     *  - The #REF! error code, if the passed sheet index is invalid.
     *  - UNSUPPORTED_ERROR, if the resulting matrix would be too large.
     */
    FormulaContext.prototype.getCellMatrix = function (sheet, range) {

        // the sheet model containing the cell range (safety check: #REF! error if sheet is invalid)
        var sheetModel = this.docModel.getSheetModel(sheet);
        if (!sheetModel) { throw ErrorCode.REF; }

        // dimension of the resulting matrix
        var matrixDim = Dimension.createFromRange(range);
        FormulaUtils.ensureMatrixDim(matrixDim);

        // build an empty matrix
        var matrix = Matrix.create(matrixDim.rows, matrixDim.cols, null);

        // fill existing cell values
        var iterator = sheetModel.getCellCollection().createAddressIterator(range, { type: 'value', covered: true });
        IteratorUtils.forEach(iterator, function (address) {
            matrix.set(address[1] - range.start[1], address[0] - range.start[0], this._getCellValue(sheetModel, address));
        }, this);

        return matrix;
    };

    /**
     * Returns the localized formula expression of the specified cell.
     *
     * @param {Number} sheet
     *  The zero-based index of the sheet.
     *
     * @param {Address} address
     *  The address of the cell in the specified sheet.
     *
     * @returns {Object|Null}
     *  A result object, if the cell contains a formula; otherwise null. The
     *  result object will contain the following properties:
     *  - {String} formula
     *      The localized formula expression of the specified cell.
     *  - {Boolean} matrix
     *      Whether the cell is part of a matrix formula.
     *
     * @throws {ErrorCode}
     *  The #REF! error code, if the passed sheet index is invalid.
     */
    FormulaContext.prototype.getCellFormula = function (sheet, address) {

        // the sheet model containing the cell (safety check: #REF! error if passed sheet index is invalid)
        var sheetModel = this.docModel.getSheetModel(sheet);
        if (!sheetModel) { throw ErrorCode.REF; }

        // get formula data from cell collection; return null if formula is not available
        var tokenDesc = sheetModel.getCellCollection().getTokenArray(address, { fullMatrix: true, grammarId: 'ui' });
        if (!tokenDesc) { return null; }

        // return the formula expression, add the matrix flag
        return { formula: tokenDesc.formula, matrix: !!tokenDesc.matrixRange };
    };

    // iterator helpers -------------------------------------------------------

    /**
     * Creates an iterator that visits the values provided by the passed value
     * iterators simultaneously, and stops at all offsets with at least one
     * available value.
     *
     * @param {Array<Object>} iterators
     *  An array of value iterators as created by the methods of this class.
     *  The result objects of the iterators MUST provide a property 'offset'
     *  specifying the position of the result value in the respective operand.
     *
     * @returns {Object}
     *  An iterator that visits all values contained in the passed iterators.
     *  The iterator will stop at all offsets with at least one available
     *  non-blank value. The values covered by the other iterators may
     *  represent blank cells.
     */
    FormulaContext.prototype.createParallelIteratorSome = function (iterators) {
        function indexer(value, result) { return result.offset; }
        return IteratorUtils.createParallelIterator(iterators, indexer);
    };

    /**
     * Creates an iterator that visits the values provided by the passed value
     * iterators simultaneously, and stops only at offsets where all iterators
     * provide a value.
     *
     * @param {Array<Object>} iterators
     *  An array of value iterators as created by the methods of this class.
     *  The result objects of the iterators MUST provide a property 'offset'
     *  specifying the position of the result value in the respective operand.
     *
     * @returns {Object}
     *  An iterator that visits all values contained in the passed iterators.
     *  The iterator will only stop at offsets where all iterators provide a
     *  value. If one of the iterators skips an offset, this offset will not be
     *  visited at all.
     */
    FormulaContext.prototype.createParallelIteratorEvery = function (iterators) {

        // create a parallel iterator that stops if any of the source iterators stops
        var iterator = this.createParallelIteratorSome(iterators);

        // restrict the iterator to offsets where ALL source iterators have stopped,
        // create the result object with column/row indexes
        return IteratorUtils.createTransformIterator(iterator, function (values, result) {
            if (result.complete) {
                var firstResult = result.results[0];
                return { value: values, offset: result.offset, col: firstResult.col, row: firstResult.row, results: result.results };
            }
        });
    };

    // scalar iterators -------------------------------------------------------

    /**
     * Creates an iterator that visits all scalar values contained in the
     * passed operand, matrix, or reference.
     *
     * @param {Any} operand
     *  The operand (scalar value of any supported data type, including null
     *  and error codes; or instances of the classes Matrix, Range3D, or
     *  Range3DArray; or a plain JS array of scalar values) whose contents will
     *  be visited.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.emptyParam=false]
     *      If set to true, the iterator will visit the empty function
     *      parameter (represented by the value null). By default, empty
     *      parameters will be skipped silently.
     *  @param {Boolean} [options.emptyCells=false]
     *      If set to true, the iterator will visit empty cells in a cell
     *      reference operand (represented by the value null). By default,
     *      empty cells will be skipped silently. Please use with caution, as
     *      this may become a serious performance bottleneck for large cell
     *      ranges.
     *  @param {Boolean} [options.acceptErrors=false]
     *      If set to true, the iterator will visit error codes too. By
     *      default, error codes will be thrown immediately.
     *  @param {Boolean} [options.complexRef=false]
     *      If set to true, a reference operand may be complex. Complex
     *      references may contain multiple cell ranges, and each of the cell
     *      ranges may refer to multiple sheets. By default, a reference must
     *      refer to a single range in a single sheet.
     *  @param {Boolean} [options.collectFormats=false]
     *      If set to true, the number format categories of cells referred by
     *      the passed reference operand will be resolved and provided by the
     *      iterator.
     *  @param {Boolean|String} [options.skipHiddenRows=false]
     *      If set to true, cells in all hidden rows will be skipped. Cells in
     *      visible rows and hidden columns will be visited.
     *  @param {Boolean|String} [options.skipFilteredRows=false]
     *      If set to true, cells in all hidden rows will be skipped, if the
     *      sheet contains at least one active filter range (either a table
     *      range, or an auto-filter). Cells in visible rows and hidden columns
     *      will be visited.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, all values in the operand have been visited. No
     *      more values are available; this result object does not contain any
     *      other properties!
     *  - {Number|String|Boolean|ErrorCode|Null} value
     *      The current scalar value.
     *  - {Number} offset
     *      The relative position of the number in the operand. For a scalar
     *      operand, this offset will be zero; for matrixes, this will be the
     *      index of the matrix element (counted by rows); and for cell
     *      references, this will be the index of the cell in all cells covered
     *      by the cell reference (counted across all cell ranges contained in
     *      the reference operand).
     *  - {Number} col
     *      The relative column index of the number in the operand. For a
     *      scalar operand, this will be zero; and for matrixes and cell
     *      references, this is the column index in the matrix or current cell
     *      range.
     *  - {Number} row
     *      The relative row index of the number in the operand. For a scalar
     *      operand, this will be zero; and for matrixes and cell references,
     *      this is the row index in the matrix or current cell range.
     *  - {ParsedFormat|Null} format
     *      The parsed number format of the cell referred by a reference
     *      operand, if the option 'collectFormats' has been set; or the value
     *      null for any other operands.
     *  The method next() will throw any error code contained in the passed
     *  operand, or that will be thrown while converting the operand values to
     *  numbers.
     */
    FormulaContext.prototype.createScalarIterator = function (operand, options) {
        return this._createOperandIterator(operand, _.identity, options);
    };

    /**
     * Visits all scalar values contained in the passed operand, by creating a
     * scalar iterator, and invoking the callback function for all values it
     * provides. See description of FormulaContext.createScalarIterator() for
     * more details.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) to be visited.
     *
     * @param {Function} callback
     *  The callback function invoked for each scalar value contained in the
     *  operand. Will be called in the context of this instance. Receives the
     *  following parameters:
     *  (1) {Any} value
     *      The current scalar value.
     *  (2) {Number} index
     *      The zero-based sequence index of the scalar value (a counter for
     *      the scalar values that ignores all skipped blank cells in the
     *      operand).
     *  (3) {Number} offset
     *      The relative position of the value in the operand.
     *  (4) {Number} col
     *      The relative column index of the value in the operand.
     *  (5) {Number} row
     *      The relative row index of the value in the operand.
     *  (6) {ParsedFormat|Null} format
     *      The parsed number format of the cell referred by a reference
     *      operand, if the option 'collectFormats' has been set; or the value
     *      null for any other operands.
     *
     * @param {Object} [options]
     *  Optional parameters. See FormulaContext.createScalarIterator() for
     *  details.
     *
     * @returns {Number}
     *  The total count of the visited scalar values.
     *
     * @throws {ErrorCode}
     *  Any error code contained in the passed operand.
     */
    FormulaContext.prototype.iterateScalars = function (operand, callback, options) {
        var iterator = this.createScalarIterator(operand, options);
        return this._iterateValues(iterator, callback);
    };

    /**
     * Returns all scalar values contained in the passed operand as a plain
     * array. Internally, method FormulaContext.iterateScalars() will be used
     * to collect the values.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) whose contents will be returned as array.
     *
     * @param {Object} [options]
     *  Optional parameters. See FormulaContext.createScalarIterator() for
     *  details.
     *
     * @returns {Array<Any>}
     *  The scalar values contained in the passed operand.
     *
     * @throws {ErrorCode}
     *  Any error code contained in the passed operand (unless error codes will
     *  be accepted as values according to the passed options).
     */
    FormulaContext.prototype.getScalarsAsArray = function (operand, options) {
        return IteratorUtils.toArray(this.createScalarIterator(operand, options));
    };

    // number iterators -------------------------------------------------------

    /**
     * Creates an iterator that visits all floating-point numbers contained in
     * the passed operand. Provides the value of a scalar operand, all elements
     * of a matrix operand, and the cell contents referred by a reference
     * operand. In difference to simple value parameters where strings and
     * boolean values contained in matrixes and cell ranges will always be
     * converted to numbers if possible, the iterator will implement different
     * conversion strategies as required by the respective functions.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) whose contents will be visited.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.valMode='convert']
     *      Conversion strategy for operands of type value:
     *      - 'convert': Strings will be converted to numbers, if possible.
     *          Boolean values will be converted to numbers. Strings that
     *          represent a boolean value result in a #VALUE! error code
     *          though.
     *      - 'rconvert' (restricted conversion): Strings will be converted to
     *          numbers, if possible. Boolean values and strings that represent
     *          a boolean value result in a #VALUE! error code though.
     *      - 'exact': Accepts numbers only. All strings, boolean values, and
     *          the empty function parameter result in a #VALUE! error code.
     *      - 'skip': All strings and all boolean values will be skipped
     *          silently, also strings that could be converted to numbers.
     *      - 'zero': All strings will be treated as the number zero, also
     *          strings that could be converted to numbers. Boolean values will
     *          be converted to numbers.
     *  @param {String} [options.matMode='convert']
     *      Conversion strategy for the elements in matrixes. See option
     *      'valMode' for details.
     *  @param {String} [options.refMode='convert']
     *      Conversion strategy for the contents of cells referenced by the
     *      operand. See option 'valMode' for details.
     *  @param {Boolean} [options.emptyParam=false]
     *      If set to true, the iterator will visit the empty function
     *      parameter (converted to zero). By default, empty parameters will be
     *      skipped silently.
     *  @param {Boolean} [options.complexRef=false]
     *      If set to true, a reference operand may be complex. Complex
     *      references may contain multiple cell ranges, and each of the cell
     *      ranges may refer to multiple sheets. By default, a reference must
     *      refer to a single range in a single sheet.
     *  @param {Boolean} [options.skipErrors=false]
     *      If set to true, error codes passed as scalar value, contained in
     *      matrixes, or contained in references, will be skipped silently,
     *      instead of being thrown.
     *  @param {Boolean} [options.collectFormats=false]
     *      If set to true, the number format categories of cells referred by
     *      the passed reference operand will be resolved and provided by the
     *      iterator.
     *  @param {Boolean} [options.dates=false]
     *      If set to true, the numbers will be converted to date objects. If
     *      conversion of a value to a date fails, the #VALUE! error code will
     *      be thrown. If the option 'floor' has been set to true, the time
     *      components of the resulting dates will be removed (the time will be
     *      set to midnight).
     *  @param {Boolean} [options.floor=false]
     *      If set to true, the numbers will be rounded down to the next
     *      integer (negative numbers will be rounded down too, i.e. away from
     *      zero!).
     *  @param {Boolean|String} [options.visible=false]
     *      Specifies how to handle hidden columns and rows in cell ranges
     *      referred by reference operands (does not have any effect for scalar
     *      and matrix operands):
     *      - true (boolean): Only visible cells (cells in visible columns, AND
     *          visible rows) will be visited.
     *      - 'columns': Cells in hidden columns will be skipped, but cells in
     *          visible columns and hidden rows will be visited.
     *      - 'rows': Cells in hidden rows will be skipped, but cells in
     *          visible rows and hidden columns will be visited.
     *      - false (boolean, default): By default, all visible and hidden
     *          cells will be covered.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties (see description of the method
     *  FormulaContext.createScalarIterator() for more details):
     *  - {Boolean} done
     *      If set to true, all numbers in the operand have been visited. No
     *      more numbers are available; this result object does not contain any
     *      other properties!
     *  - {Number|Date} value
     *      The current floating-point number, or a Date instance, if the
     *      option 'dates' has been set.
     *  - {Number} index
     *      The zero-based sequence index of the number (a counter for the
     *      numbers that ignores all other skipped values in the operand).
     *  - {Number} offset
     *      The relative position of the number in the operand.
     *  - {Number} col
     *      The relative column index of the number in the operand.
     *  - {Number} row
     *      The relative row index of the number in the operand.
     *  - {ParsedFormat|Null} format
     *      The parsed number format of the cell referred by a reference
     *      operand, if the option 'collectFormats' has been set; or the value
     *      null for any other operands.
     *  The method next() will throw any error code contained in the passed
     *  operand, or that will be thrown while converting the operand values to
     *  numbers.
     */
    FormulaContext.prototype.createNumberIterator = function (operand, options) {

        // conversion strategy
        var convertMode = getConvertMode(operand, options);
        // whether to skip error codes silently
        var skipErrors = Utils.getBooleanOption(options, 'skipErrors', false);
        // whether to round numbers down to integers
        var floorMode = Utils.getBooleanOption(options, 'floor', false);
        // date mode (convert all numbers to date values)
        var dateMode = Utils.getBooleanOption(options, 'dates', false);

        // create an iterator for all values in the operand, try to convert to numbers
        var iterator = this._createOperandIterator(operand, function (value) {

            // silently skip error codes
            if (skipErrors && (value instanceof ErrorCode)) { return null; }

            // convert current value according to conversion strategy
            switch (convertMode) {
                case 'convert':
                    // convert value to number (may still fail for strings)
                    value = this.convertToNumber(value, floorMode);
                    break;
                case 'rconvert':
                    // throw #VALUE! error code on booleans
                    if (_.isBoolean(value)) { throw ErrorCode.VALUE; }
                    // convert value to number (may still fail for strings)
                    value = this.convertToNumber(value, floorMode);
                    break;
                case 'exact':
                    // throw #VALUE! error code on booleans, strings, and empty function parameters
                    if (_.isString(value) || _.isBoolean(value) || (value === null)) { throw ErrorCode.VALUE; }
                    // convert dates to numbers
                    value = this.convertToNumber(value, floorMode);
                    break;
                case 'skip':
                    // skip strings and boolean values
                    if (!_.isNumber(value)) { value = null; } else if (floorMode) { value = Math.floor(value); }
                    break;
                case 'zero':
                    // replace all strings with zero
                    value = _.isString(value) ? 0 : this.convertToNumber(value, floorMode);
                    break;
                default:
                    FormulaUtils.throwInternal('FormulaContext.createNumberIterator(): unknown conversion mode: "' + convertMode + '"');
            }

            // convert numbers to dates if specified
            return (dateMode && _.isNumber(value)) ? this.convertToDate(value) : value;

        }, _.extend({}, options, { emptyCells: false, acceptErrors: skipErrors }));

        // create the resulting iterator that filters all null values and adds the sequence index
        return createValueIterator(iterator);
    };

    /**
     * Visits all floating-point numbers contained in the passed operand, by
     * creating a number iterator, and invoking the callback function for all
     * numbers it provides. See FormulaContext.createNumberIterator() for more
     * details.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) whose contents will be visited.
     *
     * @param {Function} callback
     *  The callback function invoked for each floating-point number contained
     *  in the operand. Will be called in the context of this instance.
     *  Receives the following parameters:
     *  (1) {Number|Date} number
     *      The current floating-point number, or a Date instance, if the
     *      option 'dates' has been set.
     *  (2) {Number} index
     *      The zero-based sequence index of the number (a counter for the
     *      numbers that ignores all other skipped values in the operand).
     *  (3) {Number} offset
     *      The relative position of the number in the operand.
     *  (4) {Number} col
     *      The relative column index of the number in the operand.
     *  (5) {Number} row
     *      The relative row index of the number in the operand.
     *  (6) {ParsedFormat|Null} format
     *      The parsed number format of the cell referred by a reference
     *      operand, if the option 'collectFormats' has been set; or the value
     *      null for any other operands.
     *
     * @param {Object} [options]
     *  Optional parameters passed to the number iterator used internally. See
     *  method FormulaContext.createNumberIterator() for details.
     *
     * @returns {Number}
     *  The total count of the visited numbers.
     *
     * @throws {ErrorCode}
     *  Any error code contained in the passed operand, or thrown while
     *  converting values to numbers.
     */
    FormulaContext.prototype.iterateNumbers = function (operand, callback, options) {
        var iterator = this.createNumberIterator(operand, options);
        return this._iterateValues(iterator, callback);
    };

    /**
     * Returns all floating-point numbers contained in the passed operand as a
     * plain array. Internally, the method FormulaContext.createNumberIterator()
     * will be used to collect the numbers.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) whose contents will be returned as array.
     *
     * @param {Object} [options]
     *  Optional parameters. See FormulaContext.createNumberIterator() for
     *  details.
     *
     * @returns {Array<Number>}
     *  The floating-point numbers contained in the passed operand.
     *
     * @throws {ErrorCode}
     *  Any error code contained in the passed operand, or thrown while
     *  converting values to numbers.
     */
    FormulaContext.prototype.getNumbersAsArray = function (operand, options) {
        return IteratorUtils.toArray(this.createNumberIterator(operand, options));
    };

    /**
     * Reduces all numbers (and other values convertible to numbers) of the
     * passed operands.
     *
     * @param {Array<Any>|Arguments} operands
     *  The operands containing numbers to be aggregated. The elements of the
     *  array can be instances of the classes Operand, Matrix, Range3D,
     *  Range3DArray, or may be any scalar value used in formulas.
     *
     * @param {Any} initial
     *  The initial intermediate result, passed to the first invocation of the
     *  aggregation callback function.
     *
     * @param {Function} aggregate
     *  The aggregation callback function. Receives the following parameters:
     *  (1) {Any} result
     *      The current intermediate result. On first invocation, this is the
     *      value passed in the 'initial' parameter.
     *  (2) {Number|Date} number
     *      The new number to be combined with the intermediate result, or a
     *      Date instance, if the option 'dates' has been set.
     *  (3) {Number} index
     *      The zero-based sequence index of the number (a counter for the
     *      numbers that ignores all other skipped values in the operands).
     *  (4) {Number} offset
     *      The relative position of the number in all operands (including all
     *      skipped empty parameters, and blank cells).
     *  Must return the new intermediate result which will be passed to the
     *  next invocation of this callback function, or may throw an error code.
     *  Will be called in the context of this instance.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the intermediate
     *  result (returned by the last invocation of the aggregation callback
     *  function) to the actual function result. Receives the following
     *  parameters:
     *  (1) {Any} result
     *      The last intermediate result to be converted to the result of the
     *      implemented function.
     *  (2) {Number} count
     *      The total count of all visited numbers.
     *  (3) {Number} size
     *      The total size of all operands, including all skipped values.
     *  Must return the final result of the aggregation, or an error code
     *  object. The type of the final result does not matter, but must match
     *  the type specification of the implemented funtion. Alternatively, the
     *  finalizer may throw an error code. Will be called in the context of
     *  this instance.
     *
     * @param {Object} [options]
     *  Optional parameters passed to the number iterator used internally. See
     *  method FormulaContext.createNumberIterator() for details. The beaviour
     *  of the option 'collectFormats' changes as following:
     *  @param {Boolean} [options.collectFormats=false]
     *      If set to true, the number formats of all visited numbers will be
     *      aggregated. If a resuting number format could be determined, the
     *      return value of this method will be an instance of Operand that
     *      contains the final parsed number format.
     *
     * @returns {Any}
     *  The result of the spreadsheet function.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the function result has failed.
     */
    FormulaContext.prototype.aggregateNumbers = function (operands, initial, aggregate, finalize, options) {
        return this._aggregateValues(operands, initial, this.iterateNumbers, aggregate, finalize, options);
    };

    /**
     * Reduces all numbers (and other values convertible to numbers) of the
     * passed operands. All numbers will be collected in a plain JavaScript
     * array first, and will be passed at once to the provided callback.
     *
     * @param {Array<Any>|Arguments} operands
     *  The operands containing numbers to be aggregated. The elements of the
     *  array can be instances of the classes Operand, Matrix, Range3D,
     *  Range3DArray, or may be any scalar value used in formulas.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the collected numbers
     *  to the actual function result. Receives the following parameters:
     *  (1) {Array<Number>} numbers
     *      The collected numbers, as plain JavaScript array.
     *  (2) {Number} sum
     *      The sum of all collected numbers, for convenience.
     *  (3) {Number} count
     *      The count of all collected numbers (the length of the 'numbers'
     *      array), for convenience.
     *  Must return the final result of the aggregation, or an error code
     *  object. The type of the final result does not matter, but must match
     *  the type specification of the implemented funtion. Alternatively, the
     *  finalizer may throw an error code. Will be called in the context of
     *  this instance.
     *
     * @param {Object} [options]
     *  Parameters passed to the number iterator used internally. See method
     *  FormulaContext.createNumberIterator() for details.
     *
     * @returns {Any}
     *  The result of the spreadsheet function.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the function result has failed.
     */
    FormulaContext.prototype.aggregateNumbersAsArray = function (operands, finalize, options) {

        // all numbers collected in an array
        var numbers = [];

        // stores the new number in the array, and updates the sum
        function aggregate(sum, number) {
            numbers.push(number);
            return sum + number;
        }

        // invokes the passed 'finalize' callback with the collected numbers
        function finalizeImpl(sum, count) {
            return finalize.call(this, numbers, sum, count);
        }

        // resolve the final result, throw error codes
        return this.aggregateNumbers(operands, 0, aggregate, finalizeImpl, options);
    };

    /**
     * Reduces all numbers (and other values convertible to numbers) of the
     * passed operands. The numbers of the passed operands will be visited
     * simultaneously, according to their relative position in the operand.
     *
     * @param {Array<Any>|Arguments} operands
     *  The operands containing numbers to be aggregated. The elements of the
     *  array can be instances of the classes Operand, Matrix, Range3D,
     *  Range3DArray, or may be any scalar value used in formulas. The operands
     *  array MUST NOT be empty. If the operand contains a complex reference
     *  (multiple ranges, or a range referring to multiple sheets), the #VALUE!
     *  error code will be thrown.
     *
     * @param {Any} initial
     *  The initial intermediate result, passed to the first invocation of the
     *  aggregation callback function.
     *
     * @param {Function} aggregate
     *  The aggregation callback function. Receives the following parameters:
     *  (1) {Any} result
     *      The current intermediate result. On first invocation, this is the
     *      value passed in the 'initial' parameter.
     *  (2) {Array<Number>|Array<Date>} tuple
     *      The numbers received from all operands to be combined with the
     *      intermediate result, as array; or an array of Date instance, if the
     *      option 'dates' has been set.
     *  (3) {Number} index
     *      The zero-based sequence index of the numbers (a counter for the
     *      numbers that ignores all other skipped values in the operands).
     *  Must return the new intermediate result which will be passed to the
     *  next invocation of this callback function, or may throw an error code.
     *  Will be called in the context of this instance.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the intermediate
     *  result (returned by the last invocation of the aggregation callback
     *  function) to the actual function result. Receives the following
     *  parameters:
     *  (1) {Any} result
     *      The last intermediate result to be converted to the result of the
     *      implemented function.
     *  (2) {Number} count
     *      The count of all visited number tuples.
     *  Must return the final result of the aggregation, or an error code
     *  object. The type of the final result does not matter, but must match
     *  the type specification of the implemented funtion. Alternatively, the
     *  finalizer may throw an error code. Will be called in the context of
     *  this instance.
     *
     * @param {Object} [options]
     *  Parameters passed to the number iterators used internally. See method
     *  FormulaContext.createNumberIterator() for details. The following
     *  additional options are supported:
     *  @param {Boolean} [options.checkSize='exact']
     *      Specifies how to check the size of the passed operands. The
     *      following values are supported:
     *      - 'exact' (default):
     *          The dimensions of all operands (width and height) must be
     *          equal.
     *      - 'size':
     *          The size of all operands (number of scalar values) must be
     *          equal, regardless of the exact width and height. Example: a 2x2
     *          cell range and a 1x4 matrix are compatible operands.
     *      - 'none':
     *          The size of the operands will not be checked.
     *  @param {ErrorCode} [options.sizeMismatchError=ErrorCode.VALUE]
     *      The error code that will be thrown if the passed operands do not
     *      have the same size.
     *
     * @returns {Any}
     *  The result of the spreadsheet function.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the function result has failed.
     */
    FormulaContext.prototype.aggregateNumbersParallel = function (operands, initial, aggregate, finalize, options) {

        // convert parameter to plain array of Operand instances (needed to get the operand sizes)
        operands = _.map(operands, function (operand) {
            return (operand instanceof Operand) ? operand : this.createOperand(operand);
        }, this);

        // the size checking mode
        var checkSize = Utils.getStringOption(options, 'checkSize', 'exact');

        // in parallel mode, all operands must have equal size
        if ((operands.length > 1) && (checkSize !== 'none')) {
            var exact = checkSize !== 'size';
            var dim0 = operands[0].getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
            var size = dim0.size();
            operands.forEach(function (operand, index) {
                if (index === 0) { return; }
                var dim = operand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
                if (exact ? !dim0.equals(dim) : (size !== dim.size())) {
                    throw Utils.getOption(options, 'sizeMismatchError', ErrorCode.VALUE);
                }
            });
        }

        // create the number iterators for every operand
        var iterators = operands.map(function (operand) {
            return this.createNumberIterator(operand, options);
        }, this);

        // the intermediate result
        var result = initial;
        // the count of all visited number tuples in all operands
        var count = 0;
        // create the parallel iterator that visits all tuples consisting of numbers only
        var iterator = this.createParallelIteratorEvery(iterators);

        // visit all number tuples and invoke the aggregation callback function
        IteratorUtils.forEach(iterator, function (tuple) {
            result = aggregate.call(this, result, tuple, count);
            count += 1;
        });

        // resolve the final result, throw error codes
        return FormulaUtils.throwErrorCode(finalize.call(this, result, count));
    };

    /**
     * Reduces all numbers (and other values convertible to numbers) of the
     * passed operands. The numbers of the passed operands will be visited
     * simultaneously, according to their relative position in the operand, and
     * they will be collected in plain JavaScript arrays first, and will be
     * passed at once to the provided callback.
     *
     * @param {Array<Any>|Arguments} operands
     *  The operands containing numbers to be aggregated. The elements of the
     *  array can be instances of the classes Operand, Matrix, Range3D,
     *  Range3DArray, or may be any scalar value used in formulas. The operands
     *  array MUST NOT be empty. If the operand contains a complex reference
     *  (multiple ranges, or a range referring to multiple sheets), the #VALUE!
     *  error code will be thrown.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the collected numbers
     *  to the actual function result. Receives the following parameters:
     *  (1) {Array<Array<Numbers>>} tuples
     *      The collected number tuples, as two-dimensional JavaScript array.
     *      Each array element is a tuple with as many numbers as operands have
     *      been passed to this method. The first array element of the inner
     *      arrays corresponds to the first operand, and so on.
     *  (2) {Array<Number>} sums
     *      The sums of all collected numbers, for convenience. The first array
     *      element is the sum of all numbers of the first operand, and so on.
     *  (3) {Number} count
     *      The count of all collected numbers per operand (the length of the
     *      'numbers' array), for convenience.
     *  Must return the final result of the aggregation, or an error code
     *  object. The type of the final result does not matter, but must match
     *  the type specification of the implemented funtion. Alternatively, the
     *  finalizer may throw an error code. Will be called in the context of
     *  this instance.
     *
     * @param {Object} [options]
     *  Parameters passed to the number iterators used internally. See method
     *  FormulaContext.createNumberIterator() for details. Additionally, this
     *  method supports all options that are supported by the method
     *  FormulaContext.aggregateNumbersParallel().
     *
     * @returns {Any}
     *  The result of the spreadsheet function.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the function result has failed.
     */
    FormulaContext.prototype.aggregateNumbersParallelAsArrays = function (operands, finalize, options) {

        // numbers of all operands, collected in an array of arrays
        var tuples = [];
        // the initial value passed to the aggregator (the sums of all operands)
        var initial = _.times(operands.length, _.constant(0));

        // stores the new numbers in the arrays, and updates the sums
        function aggregate(sums, tuple) {
            tuples.push(tuple);
            tuple.forEach(function (number, index) { sums[index] += number; });
            return sums;
        }

        // invokes the passed 'finalize' callback with the collected numbers
        function finalizeImpl(sums, count) {
            return finalize.call(this, tuples, sums, count);
        }

        // resolve the final result, throw error codes
        return this.aggregateNumbersParallel(operands, initial, aggregate, finalizeImpl, options);
    };

    // string iterators -------------------------------------------------------

    /**
     * Creates an iterator that visits all strings contained in the passed
     * operand. Provides the value of a scalar operand, all elements of a
     * matrix operand, and the cell contents referred by a reference operand.
     * In difference to simple value parameters where numbers and boolean
     * values contained in matrixes and cell ranges will always be converted to
     * strings if possible, this method implements different conversion
     * strategies as required by the respective functions.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) whose contents will be visited.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.valMode='convert']
     *      Conversion strategy for operands of type value:
     *      - 'convert': Numbers and boolean values will be converted to
     *          strings.
     *      - 'skip': All numbers and boolean values will be skipped silently.
     *  @param {String} [options.matMode='convert']
     *      Conversion strategy for the elements in matrixes. See option
     *      'valMode' for details.
     *  @param {String} [options.refMode='convert']
     *      Conversion strategy for the contents of cells referenced by the
     *      operand. See option 'valMode' for details.
     *  @param {Boolean} [options.emptyParam=false]
     *      If set to true, the iterator will visit the empty function
     *      parameter (converted to the empty string). By default, empty
     *      parameters will be skipped silently.
     *  @param {Boolean} [options.complexRef=false]
     *      If set to true, a reference operand may be complex. Complex
     *      references may contain multiple cell ranges, and each of the cell
     *      ranges may refer to multiple sheets. By default, a reference must
     *      refer to a single range in a single sheet.
     *  @param {Boolean} [options.collectFormats=false]
     *      If set to true, the number format categories of cells referred by
     *      the passed reference operand will be resolved and provided by the
     *      iterator.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties (see description of the method
     *  FormulaContext.createScalarIterator() for more details):
     *  - {Boolean} done
     *      If set to true, all strings in the operand have been visited. No
     *      more strings are available; this result object does not contain any
     *      other properties!
     *  - {String} value
     *      The current string value.
     *  - {Number} index
     *      The zero-based sequence index of the string (a counter for the
     *      strings that ignores all other skipped values in the operand).
     *  - {Number} offset
     *      The relative position of the value in the operand.
     *  - {Number} col
     *      The relative column index of the value in the operand.
     *  - {Number} row
     *      The relative row index of the value in the operand.
     *  - {ParsedFormat|Null} format
     *      The parsed number format of the cell referred by a reference
     *      operand, if the option 'collectFormats' has been set; or the value
     *      null for any other operands.
     *  The method next() will throw any error code contained in the passed
     *  operand, or that will be thrown while converting the operand values to
     *  strings.
     */
    FormulaContext.prototype.createStringIterator = function (operand, options) {

        // conversion strategy
        var convertMode = getConvertMode(operand, options);

        // create an iterator for all values in the operand, try to convert to strings
        var iterator = this._createOperandIterator(operand, function (value) {
            switch (convertMode) {
                case 'convert':
                    // convert value to string
                    return this.convertToString(value);
                case 'skip':
                    // skip numbers and boolean values
                    return (typeof value === 'string') ? value : null;
                default:
                    FormulaUtils.throwInternal('FormulaContext.createStringIterator(): unknown conversion mode: "' + convertMode + '"');
            }
        }, _.extend({}, options, { emptyCells: false, acceptErrors: false }));

        // create the resulting iterator that filters all null values and adds the sequence index
        return createValueIterator(iterator);
    };

    /**
     * Visits all strings contained in the passed operand, by creating a string
     * iterator, and invoking the callback function for all strings it
     * provides. See method FormulaContext.createStringIterator() for more
     * details.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) whose contents will be visited.
     *
     * @param {Function} callback
     *  The callback function invoked for each string contained in the operand.
     *  Will be called in the context of this instance. Receives the following
     *  parameters:
     *  (1) {String} value
     *      The current string value.
     *  (2) {Number} index
     *      The zero-based sequence index of the string (a counter for the
     *      string that ignores all other skipped values in the operand).
     *  (3) {Number} offset
     *      The relative position of the value in the operand.
     *  (4) {Number} col
     *      The relative column index of the value in the operand.
     *  (5) {Number} row
     *      The relative row index of the value in the operand.
     *  (6) {ParsedFormat|Null} format
     *      The parsed number format of the cell referred by a reference
     *      operand, if the option 'collectFormats' has been set; or the value
     *      null for any other operands.
     *
     * @param {Object} [options]
     *  Parameters passed to the string iterator used internally. See method
     *  FormulaContext.createStringIterator() for details.
     *
     * @returns {Number}
     *  The total count of the visited strings.
     *
     * @throws {ErrorCode}
     *  Any error code contained in the passed operand, or thrown while
     *  converting values to strings.
     */
    FormulaContext.prototype.iterateStrings = function (operand, callback, options) {
        var iterator = this.createStringIterator(operand, options);
        return this._iterateValues(iterator, callback);
    };

    /**
     * Reduces all strings (and other values convertible to strings) of the
     * passed operands.
     *
     * @param {Array<Any>|Arguments} operands
     *  The operands containing strings to be aggregated. The elements of the
     *  array can be instances of the classes Operand, Matrix, Range3D,
     *  Range3DArray, or may be any scalar value used in formulas.
     *
     * @param {Any} initial
     *  The initial intermediate result, passed to the first invocation of the
     *  aggregation callback function.
     *
     * @param {Function} aggregate
     *  The aggregation callback function. Receives the following parameters:
     *  (1) {Any} result
     *      The current intermediate result. On first invocation, this is the
     *      value passed in the 'initial' parameter.
     *  (2) {String} value
     *      The new string to be combined with the intermediate result.
     *  (3) {Number} index
     *      The zero-based sequence index of the string (a counter for the
     *      strings that ignores all other skipped values in the operands).
     *  (4) {Number} offset
     *      The relative position of the string in all operands (including all
     *      skipped empty parameters, and blank cells).
     *  Must return the new intermediate result which will be passed to the
     *  next invocation of this callback function, or may throw an error code.
     *  Will be called in the context of this instance.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the intermediate
     *  result (returned by the last invocation of the aggregation callback
     *  function) to the actual function result. Receives the following
     *  parameters:
     *  (1) {Any} result
     *      The last intermediate result to be converted to the result of the
     *      implemented function.
     *  (2) {Number} count
     *      The total count of all visited strings.
     *  (3) {Number} size
     *      The total size of all operands, including all skipped values.
     *  Must return the final result of the aggregation, or an error code
     *  object. The type of the final result does not matter, but must match
     *  the type specification of the implemented funtion. Alternatively, may
     *  throw an error code. Will be called in the context of this instance.
     *
     * @param {Object} [options]
     *  Parameters passed to the string iterator used internally. See method
     *  FormulaContext.createStringIterator() for details. The beaviour of the
     *  option 'collectFormats' changes as following:
     *  @param {Boolean} [options.collectFormats=false]
     *      If set to true, the number formats of all visited strings will be
     *      aggregated. If a resuting number format could be determined, the
     *      return value of this method will be an instance of Operand that
     *      contains the final parsed number format.
     *
     * @returns {Any}
     *  The result of the spreadsheet function.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the function result has failed.
     */
    FormulaContext.prototype.aggregateStrings = function (operands, initial, aggregate, finalize, options) {
        return this._aggregateValues(operands, initial, this.iterateStrings, aggregate, finalize, options);
    };

    // boolean iterators --------------------------------------------------

    /**
     * Creates an iterator that visits all boolean values contained in the
     * passed operand. Provides the value of a scalar operand, all elements of
     * a matrix operand, and the cell contents referred by a reference operand.
     * In difference to simple value parameters where numbers and strings
     * contained in matrixes and cell ranges will always be converted to
     * boolean values if possible, this method implements different conversion
     * strategies as required by the respective functions.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) whose contents will be visited.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.valMode='convert']
     *      Conversion strategy for operands of type value:
     *      - 'convert': Strings will be converted to boolean values, if
     *          possible. Numbers will be converted to boolean values. Strings
     *          that represent a number result in a #VALUE! error code though.
     *      - 'skip': All strings will be skipped silently, also strings that
     *          could be converted to boolean values. Numbers will be converted
     *          though.
     *  @param {String} [options.matMode='convert']
     *      Conversion strategy for the elements in matrixes. See option
     *      'valMode' for details.
     *  @param {String} [options.refMode='convert']
     *      Conversion strategy for the contents of cells referenced by the
     *      operand. See option 'valMode' for details.
     *  @param {Boolean} [options.emptyParam=false]
     *      If set to true, the iterator will visit the empty function
     *      parameter (converted to false). By default, empty parameters will
     *      be skipped silently.
     *  @param {Boolean} [options.complexRef=false]
     *      If set to true, a reference operand may be complex. Complex
     *      references may contain multiple cell ranges, and each of the cell
     *      ranges may refer to multiple sheets. By default, a reference must
     *      refer to a single range in a single sheet.
     *  @param {Boolean} [options.collectFormats=false]
     *      If set to true, the number format categories of cells referred by
     *      the passed reference operand will be resolved and provided by the
     *      iterator.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties (see description of the method
     *  FormulaContext.createScalarIterator() for more details):
     *  - {Boolean} done
     *      If set to true, all boolean values in the operand have been
     *      visited. No more values are available; this result object does not
     *      contain any other properties!
     *  - {Boolean} value
     *      The current boolean value.
     *  - {Number} index
     *      The zero-based sequence index of the boolean value (a counter for
     *      the boolean values that ignores all other skipped values in the
     *      operand).
     *  - {Number} offset
     *      The relative position of the value in the operand.
     *  - {Number} col
     *      The relative column index of the value in the operand.
     *  - {Number} row
     *      The relative row index of the value in the operand.
     *  - {ParsedFormat|Null} format
     *      The parsed number format of the cell referred by a reference
     *      operand, if the option 'collectFormats' has been set; or the value
     *      null for any other operands.
     *  The method next() will throw any error code contained in the passed
     *  operand, or that will be thrown while converting the operand values to
     *  booleans.
     */
    FormulaContext.prototype.createBooleanIterator = function (operand, options) {

        // conversion strategy
        var convertMode = getConvertMode(operand, options);

        // create an iterator for all values in the operand, try to convert to booleans
        var iterator = this._createOperandIterator(operand, function (value) {
            switch (convertMode) {
                case 'convert':
                    // convert value to boolean (may still fail for strings)
                    return this.convertToBoolean(value);
                case 'skip':
                    // skip strings (but not numbers)
                    return _.isString(value) ? null : this.convertToBoolean(value);
                default:
                    FormulaUtils.throwInternal('FormulaContext.createBooleanIterator(): unknown conversion mode: "' + convertMode + '"');
            }
        }, _.extend({}, options, { emptyCells: false, acceptErrors: false }));

        // create the resulting iterator that filters all null values and adds the sequence index
        return createValueIterator(iterator);
    };

    /**
     * Visits all boolean values contained in the passed operand, by creating a
     * boolean iterator, and invoking the callback function for all boolean
     * values it provides. See FormulaContext.createBooleanIterator() for more
     * details.
     *
     * @param {Any} operand
     *  The operand (instance of classes Operand, Matrix, Range3D, or
     *  Range3DArray; or a scalar value of any supported data type, including
     *  null and error codes) whose contents will be visited.
     *
     * @param {Function} callback
     *  The callback function invoked for each boolean value contained in the
     *  operand. Will be called in the context of this instance. Receives the
     *  following parameters:
     *  (1) {Boolean} value
     *      The current boolean value.
     *  (2) {Number} index
     *      The zero-based sequence index of the boolean value (a counter for
     *      the boolean value that ignores all other skipped values in the
     *      operand).
     *  (3) {Number} offset
     *      The relative position of the value in the operand.
     *  (4) {Number} col
     *      The relative column index of the value in the operand.
     *  (5) {Number} row
     *      The relative row index of the value in the operand.
     *  (6) {ParsedFormat|Null} format
     *      The parsed number format of the cell referred by a reference
     *      operand, if the option 'collectFormats' has been set; or the value
     *      null for any other operands.
     *
     * @param {Object} [options]
     *  Parameters passed to the boolean value iterator used internally. See
     *  method FormulaContext.createBooleanIterator() for details.
     *
     * @returns {Number}
     *  The total count of the visited boolean values.
     *
     * @throws {ErrorCode}
     *  Any error code contained in the passed operand, or thrown while
     *  converting values to booleans.
     */
    FormulaContext.prototype.iterateBooleans = function (operand, callback, options) {
        var iterator = this.createBooleanIterator(operand, options);
        return this._iterateValues(iterator, callback);
    };

    /**
     * Reduces all boolean values (and other values convertible to boolean
     * values) of the passed operands.
     *
     * @param {Array<Any>|Arguments} operands
     *  The operands containing boolean values to be aggregated. The elements
     *  of the array can be instances of the classes Operand, Matrix, Range3D,
     *  Range3DArray, or may be any scalar value used in formulas.
     *
     * @param {Any} initial
     *  The initial intermediate result, passed to the first invocation of the
     *  aggregation callback function.
     *
     * @param {Function} aggregate
     *  The aggregation callback function. Receives the following parameters:
     *  (1) {Any} result
     *      The current intermediate result. On first invocation, this is the
     *      value passed in the 'initial' parameter.
     *  (2) {Boolean} value
     *      The new boolean value to be combined with the intermediate result.
     *  (3) {Number} index
     *      The zero-based sequence index of the boolean value (a counter for
     *      the boolean values that ignores all other skipped values in the
     *      operands).
     *  (4) {Number} offset
     *      The relative position of the boolean value in all operands
     *      (including all skipped empty parameters, and blank cells).
     *  Must return the new intermediate result which will be passed to the
     *  next invocation of this callback function, or may throw an error code.
     *  Will be called in the context of this instance.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the intermediate
     *  result (returned by the last invocation of the aggregation callback
     *  function) to the actual function result. Receives the following
     *  parameters:
     *  (1) {Any} result
     *      The last intermediate result to be converted to the result of the
     *      implemented function.
     *  (2) {Number} count
     *      The total count of all visited boolean values.
     *  (3) {Number} size
     *      The total size of all operands, including all skipped values.
     *  Must return the final result of the aggregation, or an error code
     *  object. The type of the final result does not matter, but must match
     *  the type specification of the implemented funtion. Alternatively, may
     *  throw an error code. Will be called in the context of this instance.
     *
     * @param {Object} [options]
     *  Parameters passed to the boolean value iterator used internally. See
     *  method FormulaContext.createBooleanIterator() for details. The beaviour
     *  of the option 'collectFormats' changes as following:
     *  @param {Boolean} [options.collectFormats=false]
     *      If set to true, the number formats of all visited boolean values
     *      will be aggregated. If a resuting number format could be
     *      determined, the return value of this method will be an instance of
     *      Operand that contains the final parsed number format.
     *
     * @returns {Any}
     *  The result of the spreadsheet function.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the function result has failed.
     */
    FormulaContext.prototype.aggregateBooleans = function (operands, initial, aggregate, finalize, options) {
        return this._aggregateValues(operands, initial, this.iterateBooleans, aggregate, finalize, options);
    };

    // matrix helpers ---------------------------------------------------------

    /**
     * Returns whether the current context type is 'mat' (matrix context), and
     * the dimension of the result matrix is known (the public method
     * FormulaContext.getMatrixDim() will return a dimension).
     *
     * @returns {Boolean}
     *  Whether the current context type is 'mat' (matrix context), and the
     *  dimension of the result matrix is known.
     */
    FormulaContext.prototype.isMatrixMode = function () {
        return (this._contextType === 'mat') && !!this._matDim;
    };

    /**
     * Returns the precalculated dimension of the matrix result calculated for
     * the current operator/function in matrix context.
     *
     * @returns {Dimension|Null}
     *  The precalculated dimension of the matrix result; or null, if no matrix
     *  result will be calculated.
     */
    FormulaContext.prototype.getMatrixDim = function () {
        return this._matDim;
    };

    /**
     * Builds a matrix result with the result values of the specified element
     * aggregation callback function.
     *
     * @param {Dimension} matrixDim
     *  The dimension of the matrix result.
     *
     * @param {Function} aggregate
     *  The aggregation callback function. Will be invoked for every element of
     *  the result matrix. Receives the following parameters:
     *  (1) {Number} row
     *      The row index of the current matrix element to be calculated.
     *  (2) {Number} col
     *      The column index of the current matrix element to be calculated.
     *  Must return the scalar value of the matrix element, or may throw an
     *  error code. Will be called in the context of this instance.
     *
     * @returns {Matrix}
     *  The result matrix built from the return values of the aggregation
     *  callback function.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the result matrix has failed.
     */
    FormulaContext.prototype.aggregateMatrix = function (matrixDim, aggregate) {

        // build the result matrix (in matrix context)
        try {
            this.pushOperandResolver(this._operandResolver, 'mat', matrixDim);
            return Matrix.generate(matrixDim.rows, matrixDim.cols, function (matRow, matCol) {
                // check maximum evaluation time for each matrix element to prevent freezing browser
                this.checkEvalTime();
                this.setMatrixPosition(matRow, matCol);
                try {
                    return aggregate.call(this, matRow, matCol);
                } catch (error) {
                    if (error instanceof ErrorCode) { return error; }
                    throw error; // rethrow internal errors
                }
            }, this);
        } finally {
            this.popOperandResolver();
        }
    };

    // data filters -----------------------------------------------------------

    /**
     * Returns the number of blank cells in the passed cell ranges.
     *
     * @param {Range3DArray|Ranges3D} ranges
     *  The addresses of the cell ranges, or a single cell range address, in
     *  which to count the blank cells.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.emptyStr=false]
     *      If set to true, empty strings (e.g. returned by a formula) will be
     *      counted as blank cells too.
     *
     * @returns {Number}
     *  The number of blank cells in the passed cell ranges.
     */
    FormulaContext.prototype.countBlankCells = function (ranges, options) {

        // whether empty strings (formula results) are counted as blank
        var emptyStr = Utils.getBooleanOption(options, 'emptyStr', false);
        // the number of non-blank cells
        var scalarCount = 0;

        // the callback function according to the empty-string mode (created outside
        // the loop to prevent to check the emptyStr flag in each invocation)
        var counterFunc = emptyStr ?
            function (value) { if (value !== '') { scalarCount += 1; } } :
            function () { scalarCount += 1; };

        // count all non-blank cells in the reference
        this.iterateScalars(ranges, counterFunc, {
            acceptErrors: true, // count all error codes
            complexRef: true // accept multiple ranges in the array, and multi-sheet ranges
        });

        // return number of remaining blank cells
        return ranges.cells() - scalarCount;
    };

    /**
     * Converts the passed simple filter pattern to a case-insensitive regular
     * expression object.
     *
     * @param {String} pattern
     *  The simple filter pattern. An asterisk character in the string matches
     *  any sequence of characters (also the empty sequence), and a question
     *  mark matches an arbitrary single character. If an asterisk or question
     *  mark is preceded by a tilde character, they lose their special meaning
     *  and will be matched literally.
     *
     * @param {Boolean} [complete=false]
     *  If set to true, the regular expression created by this method will only
     *  match complete strings. By default, any substring of a string will be
     *  matched.
     *
     * @returns {RegExp}
     *  A case-insensitive regular expression that represents the passed simple
     *  filter pattern.
     */
    FormulaContext.prototype.convertPatternToRegExp = function (pattern, complete) {

        // convert passed search text to regular expression pattern matching the text literally
        pattern = _.escapeRegExp(pattern)
            // replace all occurrences of asterisk (without leading tilde) to '.*' RE pattern (e.g. 'a\*b' => 'a.*b')
            .replace(/(^|[^~])\\\*/g, '$1.*')
            // replace all occurrences of question mark (without leading tilde) to '.' RE pattern (e.g. 'a\?b' => 'a.b')
            .replace(/(^|[^~])\\\?/g, '$1.')
            // remove tilde characters before (already escaped) asterisks and question marks (e.g. 'a~\*b' => 'a\*b')
            .replace(/~(\\[*?])/g, '$1');

        // optionally, let the regular expression only match complete strings
        if (complete) { pattern = '^' + pattern + '$'; }

        // create the case-insensitive regular expression
        return new RegExp(pattern, 'i');
    };

    /**
     * Creates a filter matcher predicate function for the specified filter
     * criterion.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} criterion
     *  A scalar value representing the filter criterion. Numbers, boolean
     *  values, and error codes will match the corresponding values, and will
     *  match strings that can be converted to these values. The value null
     *  represents a reference to an empty cell, and will match the number zero
     *  only (but not the empty string, and not empty cells). Strings may
     *  contain leading comparison operators, or may specify filter patterns
     *  with asterisk characters or question marks. Otherwise, string criteria
     *  will be converted to numbers, boolean values, or error codes, if
     *  possible, and will match the corresponding values.
     *
     * @returns {Function}
     *  A predicate function that takes a single parameter (a scalar value of
     *  any type that can occur in a formula), and returns a boolean whether
     *  the passed scalar value matches the filter criterion.
     */
    FormulaContext.prototype.createFilterMatcher = function (criterion) {

        // the number formatter of the document
        var numberFormatter = this.numberFormatter;
        // the filter operator (empty string for criterion without operator)
        var op = null;
        // the filter operand to be compared with the values passed to the matcher
        var comp = null;
        // whether to test for equality (needs some special handling)
        var equality = null;

        // parse the filter operator and the filter operand from a string criterion
        if (typeof criterion === 'string') {
            var matches = /^(<=|>=|<>|<|>|=)?(.*)$/.exec(criterion);
            op = matches[1] || '';
            // leading apostrophe in operand counts as regular character
            comp = numberFormatter.parseValue(matches[2], { keepApos: true });
        } else {
            // all other data types cause filtering by equality
            op = '';
            comp = criterion;
        }

        // testing for equality converts strings in source data to other data types
        equality = (op === '') || (op === '=');

        // empty filter operand (empty parameter, or reference to empty cell) matches zero
        // numbers (and strings that can be parsed to zero number) in source data, but not
        // empty strings, FALSE, or empty cells
        if (comp === null) {
            return function zeroMatcher(value) {
                return ((typeof value === 'string') ? numberFormatter.parseValue(value) : value) === 0;
            };
        }

        // special handling for the empty string as filter operand
        if (comp === '') {
            switch (op) {
                // empty string without filter operator matches empty strings AND empty cells
                case '': return emptyMatcher;
                // single equality operator without operand matches blank cells, but not empty strings
                case '=': return blankMatcher;
                // single inequality operator without operand matches anything but blank cells
                case '<>': return scalarMatcher;
                // all other comparison operators without operand do not match anything
                default: return nothingMatcher;
            }
        }

        // the data type identifier of the filter operand
        var compType = Scalar.getType(comp);

        // equality or inequality with pattern matching
        if ((compType === Scalar.Type.STRING) && /[*?]/.test(comp) && (equality || (op === '<>'))) {

            // the regular expression for the pattern (match complete strings only)
            var regExp = this.convertPatternToRegExp(comp, true);

            // create the pattern matcher predicate function (pattern filters work on strings only)
            return function patternMatcher(value) {
                return equality === ((typeof value === 'string') && regExp.test(value));
            };
        }

        // strings are always matched case-insensitively
        if (compType === Scalar.Type.STRING) { comp = comp.toUpperCase(); }
        // try to convert string values to numbers when testing equality with a number
        var convertToNumber = equality && (compType === Scalar.Type.NUMBER);

        // create the comparison matcher predicate function (pattern filters work on strings only)
        return function comparisonMatcher(value) {

            // The equality operator converts strings in source data to numbers if possible. Strings will
            // not be converted to numbers if the comparison operand is not a number (too expensive); in
            // this case the comparison operand cannot be converted to a number (has been tried above),
            // thus a string value looking like a number will not be equal to the string comparison operand.
            // BUT: Strings will NOT be converted to booleans or error codes, and the inequality operator
            // does not convert anything at all.
            if (convertToNumber && (typeof value === 'string')) {
                var converted = numberFormatter.parseValue(value);
                if (typeof converted === 'number') { value = converted; }
            }

            // compare the passed value with the filter operand (resulting NaN means incompatible types)
            var canCompare = (value !== null) && (compType === Scalar.getType(value));
            var rel = canCompare ? Scalar.compare(value, comp) : Number.NaN;

            // return whether the value matches, according to the filter operand
            switch (op) {
                case '':
                case '=':  return isFinite(rel) && (rel === 0);
                case '<>': return !isFinite(rel) || (rel !== 0); // incompatible types match the inequality operator
                case '<':  return isFinite(rel) && (rel < 0);
                case '<=': return isFinite(rel) && (rel <= 0);
                case '>':  return isFinite(rel) && (rel > 0);
                case '>=': return isFinite(rel) && (rel >= 0);
            }

            FormulaUtils.throwInternal('FormulaContext.comparisonMatcher(): invalid operator');
        };
    };

    /**
     * Implementation helper for COUNTIF and COUNTIFS. Counts all entries in
     * the cell ranges that match the filter criteria.
     *
     * @returns {Number}
     *  The number of cells matched by the specified criteria.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the function result has failed.
     */
    FormulaContext.prototype.countFilteredCells = function () {

        // wrapper for the filter ranges, and matcher predicate functions
        var aggregator = new FilterAggregator(this, 0);

        // count all matching entries in the filter ranges
        return aggregator.countMatches();
    };

    /**
     * Reduces all numbers that match a specific filter criterion to the result
     * of a spreadsheet function.
     *
     * @param {String} funcType
     *  The type category of the function to be implemented. Must be one of the
     *  following values:
     *  - 'single': The function uses a single filter criterion, passed as the
     *      leading parameters 'filterRange' and 'criterion' to the function,
     *      e.g. SUMIF or AVERAGEIF. The data range will be passed as optional
     *      third parameter to the function. If missing, the filter range will
     *      be used instead.
     *  - 'multi': The function takes a mandatory leading data range, followed
     *      by an arbitrary number of parameter pais representing the filter
     *      criteria, e.g. SUMIFS, AVERAGEIFS, MINIFS, or MAXIFS.
     *
     * @param {Number} initial
     *  The initial intermediate result, passed to the first invocation of the
     *  aggregation callback function.
     *
     * @param {Function} aggregate
     *  The aggregation callback function. Receives the following parameters:
     *  (1) {Number} result
     *      The current intermediate result. On first invocation, this is the
     *      number passed in the 'initial' parameter.
     *  (2) {Number} number
     *      The new number to be combined with the intermediate result.
     *  Must return the new intermediate result which will be passed to the
     *  next invocation of this callback function, or may throw an error code.
     *  Will be called in the context of this instance.
     *
     * @param {Function} [finalize]
     *  A callback function invoked at the end to convert the intermediate
     *  result (returned by the last invocation of the aggregation callback
     *  function) to the actual function result. Receives the following
     *  parameters:
     *  (1) {Number} result
     *      The last intermediate result to be converted to the result of the
     *      implemented function.
     *  (2) {Number} count
     *      The count of all numbers the aggregation callback function has been
     *      invoked for.
     *  Must return the final numeric result of the spreadsheet function, or an
     *  error code. Alternatively, may throw an error code. Will be called in
     *  the context of this instance.
     *
     * @returns {Number|Matrix}
     *  The result of the spreadsheet function.
     *
     * @throws {ErrorCode}
     *  An error code if calculating the function result has failed.
     */
    FormulaContext.prototype.aggregateFilteredCells = function (funcType, initial, aggregate, finalize) {

        // single functions (SUMIF etc.): data range follows as optional third parameter
        // multi functions (SUMIFS etc.): data range is the mandatory first parameter
        var multiMode = funcType === 'multi';
        // wrapper for the filter ranges, and matcher predicate functions
        var aggregator = new FilterAggregator(this, multiMode ? 1 : 0);
        // the first filter range (needed to compare range size with data range)
        var filterRange = aggregator.getFilterRange(0);
        // the data range
        var dataRange = multiMode ? this.getOperand(0, 'ref:val') : this.isMissingOperand(2) ? null : this.getOperand(2, 'ref:val');

        // single mode (SUMIF etc.): ignore original size of the data range, set it to the size of the source range
        if (!multiMode) {
            if (dataRange) {
                dataRange.end[0] = Math.min(dataRange.start[0] + filterRange.cols() - 1, this.docModel.getMaxCol());
                dataRange.end[1] = Math.min(dataRange.start[1] + filterRange.rows() - 1, this.docModel.getMaxRow());
                // shorten the source range, if the data range has been shortened
                filterRange.end[0] = filterRange.start[0] + dataRange.cols() - 1;
                filterRange.end[1] = filterRange.start[1] + dataRange.rows() - 1;
            } else {
                // no data range passed: aggregate filter range directly
                dataRange = filterRange.clone();
            }
        }

        // the data range must have the same size as the filter ranges
        return aggregator.aggregateNumbers(dataRange, initial, aggregate, finalize);
    };

    // exports ================================================================

    return FormulaContext;

});
