/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/operators', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (SheetUtils, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all unary and binary operators used in
     * spreadsheet formulas.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of operator descriptor objects.
     *
     *************************************************************************/

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Range3DArray = SheetUtils.Range3DArray;
    var MathUtils = FormulaUtils.Math;

    // exports ================================================================

    return {

        // arithmetic operators -----------------------------------------------

        // binary addition operator, or unary plus operator
        add: {
            name: '+',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            format: 'combine',
            signature: 'val val:num', // unary plus operator passes all values (no conversion to number)
            resolve: function (value1, value2) {
                return this.isMissingOperand(1) ? value1 : (this.convertToNumber(value1) + value2);
            }
        },

        // binary subtraction operator, or unary minus operator
        sub: {
            name: '-',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            format: 'combine',
            signature: 'val:num val:num', // unary minus operator always converts to numbers
            resolve: function (value1, value2) {
                return this.isMissingOperand(1) ? (-value1) : (value1 - value2);
            }
        },

        // binary multiplication operator
        mul: {
            name: '*',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: MathUtils.mul
        },

        // binary division operator
        div: {
            name: '/',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: MathUtils.div
        },

        // binary power operator
        pow: {
            name: '^',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: MathUtils.POW
        },

        // unary percent operator
        pct: {
            name: '%',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) { return number / 100; }
        },

        // string operators ---------------------------------------------------

        // binary string concatenation operator
        con: {
            name: '&',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:str',
            resolve: function (str1, str2) {
                // concatenate the URL parts of both operands (fall-back to the regular strings if either URL is missing)
                var url1 = this.getOperand(0).getURL();
                var url2 = this.getOperand(1).getURL();
                var url = ((url1 !== null) || (url2 !== null)) ? (((url1 === null) ? str1 : url1) + ((url2 === null) ? str2 : url2)) : null;
                return this.createOperand(str1 + str2, { url: url });
            }
        },

        // comparison operators -----------------------------------------------

        // binary less-than operator
        lt: {
            name: '<',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareScalars(value1, value2) < 0;
            }
        },

        // binary less-than-or-equal operator
        le: {
            name: '<=',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareScalars(value1, value2) <= 0;
            }
        },

        // binary greater-than operator
        gt: {
            name: '>',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareScalars(value1, value2) > 0;
            }
        },

        // binary greater-than-or-equal operator
        ge: {
            name: '>=',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareScalars(value1, value2) >= 0;
            }
        },

        // binary equals operator
        eq: {
            name: '=',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.equalScalars(value1, value2);
            }
        },

        // binary equals-not operator
        ne: {
            name: '<>',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return !this.equalScalars(value1, value2);
            }
        },

        // reference operators ------------------------------------------------

        // binary list reference operator
        list: {
            name: { ooxml: ',', odf: '~' },
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref|deps:pass ref|deps:pass',
            resolve: function (ranges1, ranges2) {
                return ranges1.concat(ranges2);
            }
        },

        // binary intersection reference operator
        isect: {
            name: { ooxml: ' ', odf: '!' },
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            recalc: 'always', // resulting reference may not cause circular dependencies although parameters would do
            signature: 'ref:sheet|deps:skip ref:sheet|deps:skip',
            resolve: function (ranges1, ranges2) {

                // ranges in both operands must refer to the same sheet
                if (ranges1.first().sheet1 !== ranges2.first().sheet1) {
                    throw ErrorCode.VALUE;
                }

                // calculate the intersection of each range pair from ranges1 and ranges2,
                // process all outer ranges separately to be able to check resulting list
                // size inside the loop, otherwise this loop may create a MAX^2 list
                var resultRanges = new Range3DArray();
                ranges1.forEach(function (range1) {
                    var ranges = ranges2.intersect(range1);
                    if (resultRanges.length + ranges.length > FormulaUtils.MAX_REF_LIST_SIZE) {
                        throw FormulaUtils.UNSUPPORTED_ERROR;
                    }
                    resultRanges.append(ranges);
                });
                return resultRanges;
            }
        },

        // binary range reference operator
        range: {
            name: ':',
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            recalc: 'always', // resulting reference may cover cells not covered by the parameters
            signature: 'ref:sheet|deps:skip ref:sheet|deps:skip',
            resolve: function (ranges1, ranges2) {

                // ranges in both operands must refer to the same sheet
                if (ranges1.first().sheet1 !== ranges2.first().sheet1) {
                    throw ErrorCode.VALUE;
                }

                // build the bounding ranges from all ranges
                return ranges1.concat(ranges2).boundary();
            }
        }
    };

});
