/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/engineeringfuncs', [
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/utils/errorcode'
], function (FormulaUtils, ErrorCode) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions that fit more or less
     * into the function category 'engineering'.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // convenience shortcuts
    var MathUtils = FormulaUtils.Math;

    // mathematical constants
    var PI = Math.PI;
    var PI_2 = PI / 2;
    var PI_4 = PI / 4;
    var EULER_GAMMA = 0.5772156649015328606;

    // shortcuts to built-in Math functions
    var floor = Math.floor;
    var abs = Math.abs;
    var pow = Math.pow;
    var sqrt = Math.sqrt;
    var exp = Math.exp;
    var log = Math.log;
    var sin = Math.sin;
    var cos = Math.cos;

    // maximum number of iterations to prevent endless loops etc.
    var MAX_ITERATIONS = 50000;

    // minimum difference between iteration steps
    var EPSILON = 1e-15;

    // private global functions ===============================================

    function besselI(value, order) {

        if (order < 0) { throw ErrorCode.NUM; }

        var result = 0,
            maxIteration = 2000,
            xHalf = value / 2,
            k = 0,
            term = 1;

        for (k = 1; k <= order; ++k) {
            term = term / k * xHalf;
        }
        result = term;
        if (term !== 0) {
            k = 1;
            do {
                term = term * xHalf / k * xHalf / (k + order);
                result += term;
                k += 1;
            } while ((abs(term) > abs(result) * EPSILON) && (k < maxIteration));
        }
        return result;
    }

    function besselJ(value, order) {

        if (order < 0) { throw ErrorCode.NUM; }

        var sign,
            estimateIteration,
            asymptoticPossible,
            epsilon = 1e-15,
            hasfound = false,
            k = 0,
            u,
            mBar, gBar, gBarDeltaU,
            g = 0,
            deltaU = 0,
            bar = -1,
            twoDivPI = 2 / PI;

        function fmod(v1, v2) {
            return v1 - floor(v1 / v2) * v2;
        }

        if (value === 0) {
            return (order === 0) ? 1 : 0;
        }
        sign = (value % 2 === 1 && value < 0) ? -1 : 1;
        value = abs(value);

        estimateIteration = value * 1.5 + order;
        asymptoticPossible = pow(value, 0.4) > order;
        if (estimateIteration > MAX_ITERATIONS) {
            if (asymptoticPossible) {
                return sign * sqrt(twoDivPI / value) * cos(value - order * PI_2 - PI_4);
            }
            throw ErrorCode.NUM;
        }

        if (value === 0) {
            u = 1;
            gBarDeltaU = 0;
            gBar = -2 / value;
            deltaU = gBarDeltaU / gBar;
            u += deltaU;
            g = -1 / gBar;
            bar *= g;
            k = 2;
        } else {
            u = 0;
            for (k = 1; k <= order - 1; k += 1) {
                mBar = 2 * fmod(k - 1, 2) * bar;
                gBarDeltaU = -g * deltaU - mBar * u; // alpha_k = 0
                gBar = mBar - 2 * k / value + g;
                deltaU = gBarDeltaU / gBar;
                u += deltaU;
                g = -1 / gBar;
                bar *= g;
            }
            // Step alpha_N = 1
            mBar = 2 * fmod(k - 1, 2) * bar;
            gBarDeltaU = bar - g * deltaU - mBar * u; // alpha_k = 1
            gBar = mBar - 2 * k / value + g;
            deltaU = gBarDeltaU / gBar;
            u += deltaU;
            g = -1 / gBar;
            bar *= g;
            k += 1;
        }
        // Loop until desired accuracy, always alpha_k = 0
        do {
            mBar = 2 * fmod(k - 1, 2) * bar;
            gBarDeltaU = -g * deltaU - mBar * u;
            gBar = mBar - 2 * k / value + g;
            deltaU = gBarDeltaU / gBar;
            u *= deltaU;
            g = -1 / gBar;
            bar *= g;
            hasfound = abs(deltaU) <= abs(u) * epsilon;
            k += 1;
        } while (!hasfound && k <= MAX_ITERATIONS);

        if (hasfound) {
            return u * sign;
        }

        throw ErrorCode.VALUE;
    }

    function besselK0(value) {

        var result, num2, y;

        if (value <= 2) {
            num2 = value * 0.5;
            y = num2 * num2;

            result = -log(num2) * besselI(value, 0) +
                    (-EULER_GAMMA + y * (0.4227842 + y * (0.23069756 + y * (0.0348859 +
                        y * (0.00262698 + y * (0.0001075 + y * 0.74e-5))))));
        } else {
            y = 2 / value;

            result = exp(-value) / sqrt(value) * (1.25331414 + y * (-0.07832358 +
                    y * (0.02189568 + y * (-0.01062446 + y * (0.00587872 +
                        y * (-0.00251540 + y * 0.00053208))))));
        }
        return result;
    }

    function besselK1(value) {

        var result, num2, y;

        if (value <= 2) {
            num2 = value / 2;
            y = num2 * num2;

            result = log(num2) * besselI(value, 1) +
                    (1 + y * (0.15443144 + y * (-0.67278579 + y * (-0.18156897 + y * (-0.1919402e-1 +
                        y * (-0.00110404 + y * -0.4686e-4)))))) / value;
        } else {
            y = 2 / value;

            result = exp(-value) / sqrt(value) * (1.25331414 + y * (0.23498619 +
                    y * (-0.0365562 + y * (0.01504268 + y * (-0.00780353 +
                    y * (0.00325614 + y * -0.00068245))))));
        }

        return result;
    }

    function besselK(value, order) {

        if (order < 0) { throw ErrorCode.NUM; }

        var result = 0,
            tox, bkm, bkp, bk;

        switch (order) {
            case 0:
                result = besselK0(value);
                break;
            case 1:
                result = besselK1(value);
                break;
            default:
                tox = 2 / value;
                bkm = besselK0(value);
                bk = besselK1(value);

                for (var n = 1; n < order; n++) {
                    bkp = bkm + n * tox * bk;
                    bkm = bk;
                    bk = bkp;
                }

                result = bk;
        }
        return result;
    }

    function besselY0(value) {

        var alpha = log(value / 2) + EULER_GAMMA,
            u,
            k = 1,
            mBar = 0,
            gBarDeltaU = 0,
            gBar = -2 / value,
            deltaU = gBarDeltaU / gBar,
            g = -1 / gBar,
            fBar = -1 * g,
            signAlpha = 1,
            km1mod2,
            hasFound = false;

        if (value <= 0) {
            throw ErrorCode.VALUE;
        }
        if (value > 5e+6) {
            return sqrt(1 / PI / value) * (sin(value) - cos(value));
        }
        u = alpha;
        k += 1;
        do {
            km1mod2 = (k - 1) % 2;
            mBar = 2 * km1mod2 * fBar;
            if (km1mod2 === 0) {
                alpha = 0;
            } else {
                alpha = signAlpha * (4 / k);
                signAlpha = -signAlpha;
            }
            gBarDeltaU = fBar * alpha - g * deltaU - mBar * u;
            gBar = mBar - 2 * k / value + g;
            deltaU = gBarDeltaU / gBar;
            u += deltaU;
            g = -1 / gBar;
            fBar *= g;
            hasFound = abs(deltaU) <= abs(u) * EPSILON;
            k += 1;
        }
        while (!hasFound && k < MAX_ITERATIONS);
        if (hasFound) {
            return u / PI_2;
        }
        throw ErrorCode.VALUE;
    }

    function besselY1(value) {

        var alpha = 1 / value,
            fBar = -1,
            g = 0,
            u = alpha,
            k = 1,
            mBar = 0,
            gBarDeltaU,
            gBar = -2 / value,
            deltaU,
            signAlpha = -1,
            km1mod2,
            q,
            hasFound = false;

        if (value <= 0) {
            throw ErrorCode.VALUE;
        }
        if (value > 5e6) {
            return -sqrt(1 / PI / value) * (sin(value) + cos(value));
        }
        alpha = 1 - EULER_GAMMA - log(value / 2);
        gBarDeltaU = -alpha;
        deltaU = gBarDeltaU / gBar;
        u += deltaU;
        g = -1 / gBar;
        fBar *= g;

        k += 1;
        do {
            km1mod2 = (k - 1) % 2;
            mBar = 2 * km1mod2 * fBar;
            q = (k - 1) / 2;
            if (km1mod2 === 0) {
                alpha = signAlpha * (1 / q + 1 / (q + 1));
                signAlpha = -signAlpha;
            } else {
                alpha = 0;
            }
            gBarDeltaU = fBar * alpha - g * deltaU - mBar * u;
            gBar = mBar - 2 * k / value + g;
            deltaU = gBarDeltaU / gBar;
            u += deltaU;
            g = -1 / gBar;
            fBar *= g;
            hasFound = abs(deltaU) <= abs(u) * EPSILON;
            k += 1;
        }
        while (!hasFound && k < MAX_ITERATIONS);
        if (hasFound) {
            return -u / PI_2;
        }
        throw ErrorCode.VALUE;
    }

    function besselY(value, order) {

        if (order < 0) { throw ErrorCode.NUM; }

        var result,
            byp,
            tox = 2 / value,
            bym,
            by;

        switch (order) {
            case 0:
                result = besselY0(value);
                break;
            case 1:
                result = besselY1(value);
                break;
            default:
                bym = besselY0(value);
                by = besselY1(value);

                for (var n = 1; n < order; n += 1) {
                    byp = n * tox * by - bym;
                    bym = by;
                    by = byp;
                }
                result = by;
        }

        return result;
    }

    /**
     * Throws the #NUM! error code, if the passed number is not supported by
     * the bit manipulation functions.
     *
     * @param {Number} number
     *  The input number to be tested.
     *
     * @returns {Number}
     *  The passed number, if it is valid, for convenience.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number is negative, or is greater
     *  than or equal to 2^48.
     */
    function ensureBitNumber(number) {
        if ((number < 0) || (number > 0xFFFFFFFFFFFF)) { throw ErrorCode.NUM; }
        return number;
    }

    /**
     * Shifts the passed number by the specified number of bits to the left.
     *
     * @param {Number} number
     *  The number to be shifted.
     *
     * @param {Number} bits
     *  The number of bits to shift. A negative value will shift the number to
     *  the right.
     *
     * @returns {Number}
     *  The shifted number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number or bit count is invalid.
     */
    function shiftBitsLeft(number, bits) {
        ensureBitNumber(number);
        if (Math.abs(bits) > 53) { throw ErrorCode.NUM; }
        // weird Excel behavior: BITLSHIFT(1,48) is #NUM!; but BITLSHIFT(1,[49...53]) is zero
        return (Math.abs(bits) > 48) ? 0 : ensureBitNumber(Math.floor(number * pow(2, bits)));
    }

    /**
     * Combines the bit patterns of the passed numbers. This is needed to
     * work-around the limitation of JavaScript's built-in bit operators to
     * 32-bit integers.
     *
     * @param {Number} number1
     *  The first number to be combined with the second number.
     *
     * @param {Number} number2
     *  The second number to be combined with the first number.
     *
     * @param {Function} bitOp
     *  The bit operator implementation. Receives two 32-bit integer numbers.
     *
     * @returns {Number}
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if one of the passed numbers is invalid.
     */
    function combineBits(number1, number2, bitOp) {
        ensureBitNumber(number1);
        ensureBitNumber(number2);
        var resultH = bitOp(number1 / 0x1000000, number2 / 0x1000000);
        var resultL = bitOp(number1 % 0x1000000, number2 % 0x1000000);
        return resultH * 0x1000000 + resultL;
    }

    // exports ================================================================

    return {

        BESSELI: {
            category: 'engineering',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: besselI
        },

        BESSELJ: {
            category: 'engineering',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: besselJ
        },

        BESSELK: {
            category: 'engineering',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: besselK
        },

        BESSELY: {
            category: 'engineering',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: besselY
        },

        BITAND: {
            category: 'engineering',
            name: { ooxml: '_xlfn.BITAND' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: function (number1, number2) {
                return combineBits(number1, number2, function (i1, i2) { return i1 & i2; });
            }
        },

        BITLSHIFT: {
            category: 'engineering',
            name: { ooxml: '_xlfn.BITLSHIFT' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: shiftBitsLeft
        },

        BITOR: {
            category: 'engineering',
            name: { ooxml: '_xlfn.BITOR' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: function (number1, number2) {
                return combineBits(number1, number2, function (i1, i2) { return i1 | i2; });
            }
        },

        BITRSHIFT: {
            category: 'engineering',
            name: { ooxml: '_xlfn.BITRSHIFT' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: function (number, bits) {
                return shiftBitsLeft(number, -bits);
            }
        },

        BITXOR: {
            category: 'engineering',
            name: { ooxml: '_xlfn.BITXOR' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: function (number1, number2) {
                return combineBits(number1, number2, function (i1, i2) { return i1 ^ i2; });
            }
        },

        DELTA: {
            category: 'engineering',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: function (number, step) {
                if (_.isUndefined(step)) { step = 0; }
                return (number === step) ? 1 : 0;
            }
        },

        ERF: {
            category: 'engineering',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: (function () {

                // implementation of binary ERF() without error checking
                function erf(a, b) {
                    // equal values always result in zero (do not calculate erf(a) explicitly)
                    if (a === b) { return 0; }
                    // no second parameter: calculate erf(a)
                    var erf_a = MathUtils.erf(a);
                    if (!_.isNumber(b)) { return erf_a; }
                    // absolute value of parameters are equal: do not calculate erf(b)
                    return (a === -b) ? (-2 * erf_a) : (MathUtils.erf(b) - erf_a);
                }

                return {
                    ooxml: function (a, b) {
                        // legacy ERF() function: negative numbers result in #NUM! error
                        if ((a < 0) || (_.isNumber(b) && (b < 0))) { throw ErrorCode.NUM; }
                        return erf(a, b);
                    },
                    odf: erf
                };
            }())

        },

        'ERF.PRECISE': {
            category: 'engineering',
            name: { ooxml: '_xlfn.ERF.PRECISE', odf: 'COM.MICROSOFT.ERF.PRECISE' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: MathUtils.erf
        },

        ERFC: {
            category: 'engineering',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: {
                ooxml: function (a) {
                    // legacy ERFC() function: negative numbers result in #NUM! error
                    if (a < 0) { throw ErrorCode.NUM; }
                    return MathUtils.erfc(a);
                },
                odf: MathUtils.erfc
            }
        },

        'ERFC.PRECISE': {
            category: 'engineering',
            name: { ooxml: '_xlfn.ERFC.PRECISE', odf: 'COM.MICROSOFT.ERFC.PRECISE' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: MathUtils.erfc
        },

        GESTEP: {
            category: 'engineering',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: function (number, step) {
                if (_.isUndefined(step)) { step = 0; }
                return number >= step ? 1 : 0;
            }
        }
    };

});
