/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/sheetdescriptor', [
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils'
], function (SimpleMap, SheetUtils, DependencyUtils) {

    'use strict';

    // convenience shortcuts
    var AddressSet = SheetUtils.AddressSet;

    // class SheetDescriptor ==================================================

    /**
     * This descriptor object collects all dependency settings for a sheet in
     * a spreadsheet document.
     *
     * @constructor
     *
     * @property {SheetModel} sheetModel
     *  The model of the sheet represented by this descriptor object.
     *
     * @property {SimpleMap<FormulaDescriptor>} formulaMap
     *  The formula descriptors (instances of class FormulaDescriptor) for all
     *  known token arrays of the sheet.
     *
     * @property {AddressSet} cellFormulaSet
     *  The formula descriptors (instances of class FormulaDescriptor) for all
     *  known cell formulas of the sheet, as property 'formula' in the cell
     *  address objects of an address set.
     *
     * @property {SimpleMap<SimpleMap<FormulaDescriptor>>} ruleFormulaMaps
     *  The descriptors of all formulas attached to a condition of a formatting
     *  rule. The elements of this map are maps of formula descriptors. The
     *  inner maps are keyed by the UIDs of the parent rule model. The formula
     *  descriptors are keyed by the complete formula keys of the token arrays
     *  (the UID of the rule model, combined with the internal keys of the
     *  token arrays used inside the rule model).
     *
     * @property {SimpleMap<FormulaDescriptor>} missingNamesMap
     *  The formula descriptors for all known token arrays that contain
     *  references to non-existing defined names. These formulas are considered
     *  to be dirty, if a defined name has been inserted, or an existing defind
     *  name has changed its label, which may result in validating a #NAME?
     *  error. The property is a map with the labels of missing names as key,
     *  and stores a formula map (keyed by formula keys) per missing name.
     *
     * @property {SimpleMap<FormulaDescriptor>} recalcAlwaysMap
     *  The formula descriptors for all known token arrays with recalculation
     *  mode 'always'. These formulas will always be recalculated whenever
     *  something in the document has changed, regardless of their
     *  dependencies.
     *
     * @property {SimpleMap<FormulaDescriptor>} recalcOnceMap
     *  The formula descriptors for all known token arrays with recalculation
     *  mode 'once'. These formulas will be recalculated once after the
     *  document has been imported, regardless of their dependencies.
     *
     * @property {SimpleMap<Number>} missingFuncMap
     *  The resource keys of all unimplemented functions that occur in any
     *  formula of the sheet, and the number of occurences.
     */
    function SheetDescriptor(sheetModel) {

        this.sheetModel = sheetModel;

        this.formulaMap = new SimpleMap();
        this.cellFormulaSet = new AddressSet();
        this.ruleFormulaMaps = new SimpleMap();
        this.missingNamesMap = new SimpleMap();
        this.recalcAlwaysMap = new SimpleMap();
        this.recalcOnceMap = new SimpleMap();
        this.missingFuncMap = new SimpleMap();

    } // class SheetDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Inserts the passed formula descriptor into the formula caches of this
     * sheet descriptor.
     *
     * @param {String} formulaKey
     *  The formula key of the formula descriptor, used as map key.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The formula descriptor to be inserted into the formula caches.
     */
    SheetDescriptor.prototype.insertFormula = function (formulaKey, formulaDesc) {

        // insert the formula descriptor into the own maps
        this.formulaMap.insert(formulaKey, formulaDesc);
        switch (formulaDesc.recalc) {
            case 'always': this.recalcAlwaysMap.insert(formulaKey, formulaDesc); break;
            case 'once':   this.recalcOnceMap.insert(formulaKey, formulaDesc);   break;
        }

        // insert the formula into the type-specific containers
        if (formulaDesc.cellAddress) {
            var address = formulaDesc.cellAddress.clone();
            this.cellFormulaSet.insert(address).formula = formulaDesc;
        } else if (formulaDesc.ruleModel) {
            var parentUid = formulaDesc.ruleModel.getUid();
            this.ruleFormulaMaps.getOrConstruct(parentUid, SimpleMap).insert(formulaKey, formulaDesc);
        }

        // map the formula descriptor by missing name labels
        SimpleMap.forEachKey(formulaDesc.missingNames, function (key) {
            var formulaMap = this.missingNamesMap.getOrConstruct(key, SimpleMap);
            formulaMap.insert(formulaKey, formulaDesc);
        }, this);

        // register all unimplemented functions
        SimpleMap.forEachKey(formulaDesc.missingFuncs, function (key) {
            this.missingFuncMap.update(key, 0, function (count) { return count + 1; });
        }, this);

        DependencyUtils.withLogging(function () {
            DependencyUtils.log('new dependencies for ' + DependencyUtils.getFormulaLabel(formulaKey) + ': ' + formulaDesc.references);
        });
    };

    /**
     * Deletes the specified formula descriptor from the formula caches of this
     * sheet descriptor.
     *
     * @param {String} formulaKey
     *  The formula key of the formula descriptor to be deleted.
     *
     * @returns {FormulaDescriptor|Null}
     *  The formula descriptor deleted from this sheet descriptor, or null.
     */
    SheetDescriptor.prototype.removeFormula = function (formulaKey) {

        // find and remove an existing descriptor registered for this sheet
        var formulaDesc = this.formulaMap.remove(formulaKey);
        if (!formulaDesc) { return null; }

        // remove the formula descriptor from the own maps
        this.recalcAlwaysMap.remove(formulaKey);
        this.recalcOnceMap.remove(formulaKey);
        SimpleMap.forEachKey(formulaDesc.missingNames, function (key) {
            this.missingNamesMap.with(key, function (formulaMap) {
                formulaMap.remove(formulaKey);
                if (formulaMap.empty()) { this.missingNamesMap.remove(key); }
            }, this);
        }, this);

        // remove the formula from the type-specific containers
        if (formulaDesc.cellAddress) {
            this.cellFormulaSet.remove(formulaDesc.cellAddress);
        } else if (formulaDesc.ruleModel) {
            var parentUid = formulaDesc.ruleModel.getUid();
            this.ruleFormulaMaps.with(parentUid, function (formulaMap) {
                formulaMap.remove(formulaKey);
                if (formulaMap.empty()) {
                    this.ruleFormulaMaps.remove(parentUid);
                }
            }, this);
        }

        // unregister all unimplemented functions
        SimpleMap.forEachKey(formulaDesc.missingFuncs, function (key) {
            var count = this.missingFuncMap.update(key, 0, function (count) { return count - 1; });
            if (count <= 0) { this.missingFuncMap.remove(key); }
        }, this);

        DependencyUtils.log('removed dependencies for ' + DependencyUtils.getFormulaLabel(formulaKey));
        return formulaDesc;
    };

    // exports ================================================================

    return SheetDescriptor;

});
